﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights  reserved.
// </copyright> 
//------------------------------------------------------------------------------ 


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common;

using SpaceBuilder.Club;
using SpaceBuilder.Caching;

namespace SpaceBuilder.Event
{
    /// <summary>
    /// 活动权限
    /// </summary>
    public static class EventPermission
    {

        /// <overloads>验证用户是否具有某种权限</overloads>
        /// <summary>
        /// 验证是否有某种权限
        /// </summary>
        /// <param name="eventID">活动实体ID</param>
        /// <param name="permission">权限实体</param>
        /// <param name="user">当前用户</param>
        /// <returns>是否通过验证</returns>
        public static bool Validate(int eventID, User user, string permissionItemKey)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
                return false;
            return Validate(eventThread, user, permissionItemKey);
        }

        /// <overloads>验证用户是否具有某种权限</overloads>
        /// <summary>
        /// 验证是否有某种权限
        /// </summary>
        /// <param name="eventThread">活动实体</param>
        /// <param name="permission">权限实体</param>
        /// <param name="user">当前用户</param>
        /// <returns>是否通过验证</returns>
        public static bool Validate(EventThread eventThread, User user, string permissionItemKey)
        {
            string exceptionType = ExceptionTypes.Instance().AccessDenied();
            return Validate(eventThread, user, permissionItemKey, ref exceptionType);
        }

        /// <overloads>核实是否有某种权限</overloads>
        /// <summary>
        /// 核实是否有某种权限
        /// </summary>
        /// <param name="section">功能模块实体</param>
        /// <param name="permission">权限实体</param>
        /// <param name="user">当前用户</param>
        /// <param name="exceptionType">抛出异常类型</param>
        /// <returns>是否通过核实</returns>
        public static bool Validate(EventThread eventThread, User user, string permissionItemKey, ref string exceptionType)
        {
            //管理员
            if ((user != null) && (user.IsAdministrator || user.IsContentAdministrator))
                return true;

            PermissionCollection userPermissionCollection = GetUserPermissionCollectionForEvent(eventThread, user);

            #region 活动自有规则
            //私有活动，不允许匿名用户查看
            if (user == null && !eventThread.IsPublic)
            {
                return false;
            }
            if (eventThread != null && user != null)
            {
                //活动发起人
                if (eventThread.UserID == user.UserID)
                    return true;

                bool isMember = EventMembers.IsMember(user.UserID, eventThread.EventID, true);
                bool isManager = EventMembers.IsManager(user.UserID, eventThread.EventID, true);
                //如果活动是私有的,并且当前用户不是活动成员
                if (!eventThread.IsPublic && !isMember)
                {
                    exceptionType = ExceptionTypes.Instance().AccessDenied();
                    return false;
                }
                //邀请朋友
                if (permissionItemKey == PermissionItemKeys.Instance().InviteFriendJoinEvent())
                {
                    if (!userPermissionCollection.ContainsKey(permissionItemKey) || userPermissionCollection[permissionItemKey] == null)
                        return false;
                    if (userPermissionCollection[permissionItemKey].AllowType == AllowTypes.Refuse)
                        return false;
                    if (!isMember)
                        return false;
                    if (isManager)
                        return true;
                    return eventThread.AllowMemberInviteFriend;
                }
                //上传照片
                if (permissionItemKey == PermissionItemKeys.Instance().UploadEventPhoto())
                {
                    if (!userPermissionCollection.ContainsKey(permissionItemKey) || userPermissionCollection[permissionItemKey] == null)
                        return false;
                    if (userPermissionCollection[permissionItemKey].AllowType == AllowTypes.Refuse)
                        return false;
                    if (!isMember)
                        return false;
                    if (isManager)
                        return true;
                    return eventThread.AllowMemberUploadPhoto;
                }
            }

            #endregion

            //系统规则
            if (userPermissionCollection.Validate(permissionItemKey))
            {
                //检查站点设置是否启用了匿名发帖
                if (user == null && permissionItemKey == PermissionItemKeys.Instance().CreateEventPost() && !SiteSettingsManager.GetSiteSettings().EnableAnonymousPosting)
                    return false;

                return true;
            }

            exceptionType = ExceptionTypes.Instance().GeneralAccessDenied();
            return false;
        }

        /// <summary>
        /// 获取用户在Event中的权限集合
        /// </summary>
        private static PermissionCollection GetUserPermissionCollectionForEvent(EventThread eventThread, User user)
        {
            string cacheKey = string.Format("UserPermission:{0}-{1}-EventID:{2}", user != null ? user.UserID : 0, ApplicationIDs.Instance().Event(), eventThread != null ? eventThread.EventID : 0);
            PermissionCollection permissionCollectionForEvent = CacheRepository.Get(cacheKey, false) as PermissionCollection;

            if (permissionCollectionForEvent == null)
            {
                //1.获取通用的活动权限集合
                PermissionCollection userPermissionCollection = Permissions.ResolveUserPermission(user, ApplicationIDs.Instance().Event());
                permissionCollectionForEvent = userPermissionCollection.Clone();

                if (user != null && eventThread != null)
                {
                    EventMember eventMember = EventMembers.GetEventMember(user.UserID, eventThread.EventID, true);

                    if (eventMember != null)
                    {
                        Role eventMemberRoleObject = null;
                        if (eventMember.IsApproved && !eventMember.IsManager)
                            eventMemberRoleObject = Roles.GetRole(UserRoleNames.Instance().EventMember(), false);
                        else if (eventMember.IsManager && user.UserID != eventThread.UserID)
                            eventMemberRoleObject = Roles.GetRole(UserRoleNames.Instance().EventManager(), false);
                        else if (user.UserID == eventThread.UserID)
                            eventMemberRoleObject = Roles.GetRole(UserRoleNames.Instance().Owner(), false);

                        if (eventMemberRoleObject != null)
                        {
                            PermissionCollection eventMemberRolePermissionCollection = Permissions.GetPermissionItemRolesInUserRole(eventMemberRoleObject.RoleID, ApplicationIDs.Instance().Event());
                            permissionCollectionForEvent.Merge(eventMemberRolePermissionCollection);
                        }
                    }
                }

                CacheRepository.Add(cacheKey, permissionCollectionForEvent, CachingExpirationTypes.UsualObjectCollection);
            }

            return permissionCollectionForEvent;
        }

        /// <summary>
        /// 验证活动管理员的设置权限
        /// </summary>
        public static bool ValidateSetPermission(EventThread eventThread, User user, string permissionItemKey)
        {
            PermissionCollection userPermissionCollection = GetUserPermissionCollectionForEvent(eventThread, user);
            if (!userPermissionCollection.ContainsKey(permissionItemKey) || userPermissionCollection[permissionItemKey] == null)
                return false;
            if (userPermissionCollection[permissionItemKey].AllowType == AllowTypes.Refuse)
                return false;
            return true;
        }

        /// <summary>
        /// 验证用户对活动留言的管理权限
        /// </summary>
        /// <param name="eventPost">活动留言</param>
        /// <param name="user">被验证的用户</param>
        public static bool ValidateEventPost(EventPost eventPost, User user)
        {
            if (user == null)
                return false;

            if (user.IsContentAdministrator)
                return true;

            bool isManager = EventMembers.IsManager(user.UserID, eventPost.EventID, false);
            if (isManager)
                return true;
            if (user.UserID == eventPost.UserID)
                return true;
            return false;
        }
        /// <summary>
        /// 验证用户对活动照片的权限
        /// </summary>
        public static bool ValidateEventPhoto(EventPhoto eventPhoto, User user)
        {

            if (user == null)
                return false;

            if (user.IsContentAdministrator)
                return true;

            bool isManager = EventMembers.IsManager(user.UserID, eventPhoto.EventID, false);
            if (isManager)
                return true;
            if (user.UserID == eventPhoto.UserID)
                return true;

            return false;
        }

        /// <summary>
        /// 验证用户报名权限
        /// </summary>
        public static bool ValidateSignUp(EventThread eventThread, User user)
        {
            string statusMessageContent = string.Empty;
            return ValidateSignUp(eventThread, user, ref statusMessageContent);
        }
        /// <summary>
        /// 验证用户对活动照片的权限
        /// </summary>
        public static bool ValidateSignUp(EventThread eventThread, User user, ref string statusMessageContent)
        {
            if (eventThread == null)
                return false;

            //活动是否正在进行
            if (eventThread.EventStatus != EventStatuses.Published)
                return false;

            //是否已超过活动最大限制人数
            if (eventThread.LimitCount > 0 && eventThread.SurplusCount <= 0)
            {
                statusMessageContent = ResourceManager.GetString("Message_SignUpUserAlreadyFullNoSignUp", Globals.GetCurrentUserLanguage(), ApplicationIDs.Instance().Event());
                return false;
            }
            if (user != null)
            {
                EventMember member = EventMembers.GetEventMember(user.UserID, eventThread.EventID, true);
                //已经报过名
                if (member != null)
                {
                    statusMessageContent = ResourceManager.GetString("Message_AlreadySignUpNotAgainSignUp", Globals.GetCurrentUserLanguage(), ApplicationIDs.Instance().Event());
                    return false;
                }
            }

            //是否已超过报名截止时间
            if (DateTime.Now > eventThread.SignEndTime)
            {
                statusMessageContent = ResourceManager.GetString("Message_PastEventSignUpEndTimeNoSignUp", Globals.GetCurrentUserLanguage(), ApplicationIDs.Instance().Event());
                return false;
            }

            //如果群组活动并设置了只允许群组成员参加的情况
            if (eventThread.IsClubEvent && eventThread.IsOnlyAllowClubMemberSignUp)
            {
                statusMessageContent = ResourceManager.GetString("Message_EventAdministratorNotAllowDirectSignUp", Globals.GetCurrentUserLanguage(), ApplicationIDs.Instance().Event());
                return ClubMembers.IsClubMember(eventThread.ClubID, user.UserID);
            }
            //匿名用户根据活动是否允许匿名用户报名确定
            if (user == null)
            {
                statusMessageContent = ResourceManager.GetString("Message_EventAdministratorNotAllowDirectSignUp", Globals.GetCurrentUserLanguage(), ApplicationIDs.Instance().Event());
                return eventThread.IsAllowAnonymousUserSignUp;
            }
            return true;
        }

        internal static void RemoveEventPermissionCache(int userID, int eventID)
        {
            string cacheKey = string.Format("UserPermission:{0}-{1}-EventID:{2}", userID, ApplicationIDs.Instance().Event(), eventID);
            CacheRepository.Remove(cacheKey);
        }
    }
}