﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights  reserved.
// </copyright> 
//------------------------------------------------------------------------------ 


using System;
using System.Collections.Generic;
using System.Text;
using System.Web;
using System.IO;

using SpaceBuilder.Common;
using SpaceBuilder.Event;

using SpaceBuilder.Utils;
using SpaceBuilder.Common.Handlers;
using SpaceBuilder.Web;

namespace SpaceBuilder.Event.Handlers
{
    /// <summary>
    /// 显示活动照片
    /// </summary>
    public class EventPhotoHandler : DownloadFileHandlerBase
    {
        public override void ProcessRequest(HttpContext context)
        {
            //防盗链检测
            if (SPBConfig.Instance().EnableAntiLeech && !Globals.IsAllowableReferrer(context.Request))
            {
                context.Response.Redirect(Globals.GetFullUrlInResourceSite("~/Themes/Shared/Styles/Images/anti-Leech.gif"), true);
                return;
            }

            int attachmentID = context.Request.QueryString.GetInt("AttachmentID", 0);
            if (attachmentID <= 0)
            {
                WebUtils.Return404(context);
                return;
            }

            //EventPhotoTypes photoType = (EventPhotoTypes)context.Request.QueryString.GetInt("PhotoType", (int)EventPhotoTypes.Details);
            EventPhotoTypes photoType = (EventPhotoTypes)Enum.Parse(typeof(EventPhotoTypes), context.Request.QueryString.Get("PhotoType"));

            EventPhoto photo = EventPhotos.GetEventPhoto(attachmentID, true);
            if (photo == null)
            {
                WebUtils.Return404(context);
                return;
            }

            User currentUser = Globals.GetCurrentUser();
            if (photo.AuditingStatus == AuditingStatuses.Fail && (currentUser == null || (!currentUser.IsAdministrator && !currentUser.IsContentAdministrator)))
            {
                WebUtils.Return403(context);
                return;
            }

            IFile photoFile = EventAttachmentManager.Instance().GetResizedPhoto(photo, photoType);
            if (photoFile == null)
            {
                WebUtils.Return404(context);
                return;
            }

            if (!CanUserView(photo, Globals.GetCurrentUser()) && photoType != EventPhotoTypes.Thumbnail)
            {
                WebUtils.Return404(context);
                return;
            }

            DateTime lastModified = photoFile.LastModified.ToUniversalTime();
            if (IsCacheOK(context, lastModified))
            {
                WebUtils.Return304(context);
                return;
            }
            else
            {
                SetResponsesDetails(context, photo.ContentType, photo.FriendlyFileName, lastModified);
                FileSystemFile fileSystemFile = photoFile as FileSystemFile;
                if (fileSystemFile != null && (!fileSystemFile.FullLocalPath.StartsWith(@"\")))
                {
                    // Send files stored on UNC paths explicitly to avoid a bug with TransmitFile.
                    context.Response.TransmitFile(fileSystemFile.FullLocalPath);
                }
                else
                {
                    context.Response.AddHeader("Content-Length", photoFile.ContentLength.ToString("0"));
                    context.Response.Buffer = false;
                    context.Response.BufferOutput = false;
                    Stream stream = null;

                    try
                    {
                        stream = photoFile.OpenReadStream();
                        if (stream == null)
                        {
                            WebUtils.Return404(context);
                            return;
                        }

                        int bufferLength = photoFile.ContentLength <= DownloadFileHandlerBase.BufferLength ? photoFile.ContentLength : DownloadFileHandlerBase.BufferLength;
                        byte[] buffer = new byte[bufferLength];
                        int readedSize;
                        while ((readedSize = stream.Read(buffer, 0, bufferLength)) > 0)
                        {
                            if (!context.Response.IsClientConnected)
                                break;

                            context.Response.OutputStream.Write(buffer, 0, readedSize);
                            context.Response.OutputStream.Flush();
                        }

                        context.Response.OutputStream.Flush();
                        context.Response.Flush();
                    }
                    catch (Exception ex)
                    {
                        context.Response.Redirect(SiteUrls.Instance().Error(ex.ToString()), true);
                        return;
                    }
                    finally
                    {
                        if (stream != null)
                            stream.Close();
                    }
                }
            }

            context.Response.Cache.VaryByParams["AttachmentID"] = true;
            context.Response.Cache.VaryByParams["PhotoType"] = true;
            context.Response.End();
        }

        /// <summary>
        /// 用户是否有浏览权限
        /// </summary>   
        public static bool CanUserView(EventPhoto photo, User user)
        {
            EventThread thread = EventThreads.GetEvent(photo.EventID, false);
            return EventPermission.Validate(thread, user, PermissionItemKeys.Instance().ViewEvent());
        }

    }
}
