﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;

using SpaceBuilder.Web;
using SpaceBuilder.File;
using SpaceBuilder.Common;
using SpaceBuilder.Common.Controllers;
using SpaceBuilder.LuceneSearch;
using System.Web.Routing;

using SpaceBuilder.Utils;
using System.Web.UI;

namespace SpaceBuilder.File.Controllers
{
    [HttpCompress]
    public class ChannelFileController : ChannelControllerBase
    {
        private int appid = ApplicationIDs.Instance().File();

        /// <summary>
        /// 总站中文件首页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Home()
        {
            Header.AddTitle(string.Format("{1}{0}{2}", Globals.BrowserTitleSeparator, GetResourceString("Title_File", appid), Globals.SiteName));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_File", ApplicationIDs.Instance().File()), string.Empty);
            List<SiteCategory> siteCategories = SiteCategories.Instance(ApplicationIDs.Instance().File()).GetRoots(false);
            ViewData["siteCategories"] = siteCategories;
            ViewData["ExperiencePeople"] = FileSections.GetTopSectionsSortBy(FileSectionSortBy.Points, 5);

            return View("Pages/Files/Home.aspx", "Masters/File.master");
        }
        #region   显示文件
        /// <summary>
        /// 文件评论列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_FileComments(int threadID, int? pageIndex)
        {
            FileThread file = FileThreads.GetThread(threadID, false);
            FileSection currentFileSection = FileSections.GetSection(file.OwnerUserID, false);
            bool isManager = false;
            if (CurrentUser != null && (CurrentUser.IsContentAdministrator || file.OwnerUserID == CurrentUser.UserID))
            {
                isManager = true;
                ViewData["isManager"] = true;
            }
            if (file == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", appid), threadID)));
            }

            PagingDataSet<FileComment> comments = FileComments.GetCommentsOfThread(threadID, pageIndex ?? 1, isManager);
            ViewData["file"] = file;
            ViewData["currentUser"] = CurrentUser;
            ViewData["currentFileSection"] = currentFileSection;
            ViewData["isManager"] = isManager;
            UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
            ViewData["rememberMyName"] = userCookie.GetRememberMeInfor_name();
            ViewData["rememberMyUrl"] = userCookie.GetRememberMeInfor_url();

            return View("Controls/Files/FileComments.ascx", comments);
        }



        #endregion

        /// <summary>
        /// 显示具体文件
        /// </summary>
        /// <param name="userDomainName">当前空间用户名</param>
        /// <param name="postID">显示文件的PostID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShowFile(int threadID)
        {
            FileThread currentFile = FileThreads.GetThread(threadID, false);
            if (currentFile == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", appid), threadID)));
            }
            bool hasManagePermission = false;
            if (CurrentUser != null && (CurrentUser.IsContentAdministrator || currentFile.OwnerUserID == CurrentUser.UserID))
            {
                hasManagePermission = true;
                currentFile = FileThreads.GetThread(threadID, true);

                ViewData["hasManagePermission"] = true;
            }
            FileSection currentFileSection = FileSections.GetSection(currentFile.OwnerUserID, hasManagePermission);
            if (currentFileSection == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", appid), threadID)));


            ShowCategoryBreadCrumb(currentFile.SiteCategoryID);
            BreadCrumb.AddBreadCrumbItem("正文", string.Empty);



            #region 处理审核
            int auditingStatusValue = (int)Applications.GetApplication(ApplicationIDs.Instance().File()).AuditingStatusForPublicDisplay;
            if ((int)currentFile.AuditingStatus < auditingStatusValue)
            {
                if (currentFile.AuditingStatus == AuditingStatuses.Fail)
                    ViewData["errorMessage"] = GetResourceString("Message_FileNotCheckPass", appid);
                else
                    ViewData["errorMessage"] = GetResourceString("Message_FileChecking", appid);
                ViewData["errorAuditingStatus"] = true;
            }
            #endregion

            #region 处理隐私
            PrivacyStatuses? errorPrivacyStatus = null;
            switch (currentFile.PrivacyStatus)
            {
                case PrivacyStatuses.NeedPassword:
                    UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
                    ViewData["passwordRight"] = userCookie.AuthorizeFileThreadID(threadID);
                    break;
                case PrivacyStatuses.OnlyFriend:
                    //仅朋友可见
                    if (!Friends.IsFriend(CurrentUser != null ? CurrentUser.UserID : 0, currentFile.OwnerUserID))
                        errorPrivacyStatus = PrivacyStatuses.OnlyFriend;
                    break;
                case PrivacyStatuses.Privacy:
                    // 私密
                    errorPrivacyStatus = PrivacyStatuses.Privacy;
                    break;
                case PrivacyStatuses.Public:
                    // 公开
                    break;
                default:
                    break;
            }
            ViewData["errorPrivacyStatus"] = errorPrivacyStatus;
            #endregion


            ViewData["CurrentUser"] = CurrentUser;

            #region 文件及FileGallery的计数
            CounterFactory.IncreaseCount<FileThreadCounter>(IncreasableCountTypes.WebView, threadID);
            CounterFactory.IncreaseCount<FileSectionCounter>(IncreasableCountTypes.WebView, currentFileSection.OwnerUserID);
            //AddWebCount<FileThreadCounter>(currentFile.ThreadID);
            //AddWebCount<FileSectionCounter>(currentFileSection.OwnerUserID);
            #endregion

            #region 设置标题
            base.SetPageTitle(currentFile.Subject);
            #endregion

            //#region 设置meta标签
            //SetPageMeta(currentFile);
            //#endregion

            #region 设置Meta标签
            string metaDescription = string.Empty;
            metaDescription = string.Format("{0}：{1}", currentFile.Subject, StringUtils.Trim(currentFile.GetBody(false), 150));
            Header.AddMetaKeywords(currentFile.Subject);
            Header.AddMetaDescription(metaDescription);
            #endregion

            IList<Rating> ratings = FileThreads.GetRatings(threadID, 30);
            ViewData["ratings"] = ratings;

            IList<FileDownLoadRecord> downLoadRecords = FileThreads.GetTopRecordsOfThread(threadID, 18);
            ViewData["downLoadRecords"] = downLoadRecords;

            return View("Pages/Files/ShowFile.aspx", "Masters/File.master", currentFile);

        }

        public ActionResult updateFileAuditingStatuses(int threadID)
        {
            List<int> threadIds = new List<int>();
            threadIds.Add(threadID);
            FileThreads.UpdateAuditingStatus(threadIds, AuditingStatuses.Success);
            return RedirectToAction("ShowFile", new RouteValueDictionary { { "threadID", threadID } });
        }

        /// <summary>
        /// 设置为精华
        /// </summary>
        public ActionResult SetEssential(int threadID)
        {
            FileThread file = FileThreads.GetThread(threadID, true);
            if (file == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", appid), threadID)));
            }
            if (CurrentUser == null || !CurrentUser.IsContentAdministrator)
            {
                return new EmptyResult();
                //throw new ArgumentException(GetResourceString("Exception_AdminSetHighLimit"));
            }

            file.IsEssential = true;
            FileThreads.SetEssential(file);

            return new EmptyResult();
        }

        /// <summary>
        /// 取消精华设置
        /// </summary>
        public ActionResult CancelEssential(int threadID)
        {

            FileThread file = FileThreads.GetThread(threadID, true);
            if (file == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", appid), threadID)));
            }
            //throw new ArgumentException(string.Format("文件：{0}不存在", threadID));
            if (CurrentUser == null || !CurrentUser.IsContentAdministrator)
            {
                return new EmptyResult();
                // throw new ArgumentException(GetResourceString("Description_AdminSetEssence"));
            }

            file.IsEssential = false;
            FileThreads.SetEssential(file);

            return new EmptyResult();
        }

        /// <summary>
        /// 设置为置顶
        /// </summary>
        public ActionResult SetTop(int threadID)
        {
            FileThread file = FileThreads.GetThread(threadID, true);
            if (file == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", appid), threadID)));
            }
            //throw new ArgumentException(string.Format("文件：{0}不存在", threadID));
            if (CurrentUser == null || !CurrentUser.IsContentAdministrator)
            {
                return new EmptyResult();
                //throw new ArgumentException(GetResourceString("Description_AdminSetEssence"));
            }

            file.SpecialOrder = 1;
            FileThreads.SetSticky(file);

            return new EmptyResult();
        }

        /// <summary>
        /// 取消置顶
        /// </summary>
        public ActionResult CancelTop(int threadID)
        {
            FileThread file = FileThreads.GetThread(threadID, true);
            if (file == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", appid), threadID)));
            }
            if (CurrentUser == null || !CurrentUser.IsContentAdministrator)
            {
                return new EmptyResult();
            }

            file.SpecialOrder = 0;
            FileThreads.SetSticky(file);

            return new EmptyResult();
        }

        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateFileComment(int threadID, int? pageIndex)
        {
            FileThread currentFile = FileThreads.GetThread(threadID, false);

            FileSection currentFileSection = FileSections.GetSection(currentFile.OwnerUserID, false);

            if (currentFile == null)
                throw new ArgumentException(string.Format(GetResourceString("Exception_Pattern_FileNotFound", appid)));

            if (!currentFileSection.EnableNewComments(CurrentUser))
                return new EmptyResult();

            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                if (verifyCode != null)
                    verifyCode = verifyCode.Trim();
                int parentID = Request.Form.GetInt("parentID", 0);
                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
                    if (CurrentUser == null)
                    {
                        TempData["url"] = Request.Form.GetString("url", string.Empty);
                        TempData["author"] = Request.Form.GetString("author", string.Empty);
                    }
                    TempData["parentID"] = parentID;
                    TempData["commentBody"] = Request.Form.GetString("commentBody", string.Empty);
                    return RedirectToAction("Control_FileComments", new RouteValueDictionary { { "threadID", threadID }, { "pageIndex", pageIndex } });
                }
            }

            FileComment comment = new FileComment();
            comment.Body = Request.Form.GetString("commentBody", string.Empty);
            comment.IsApproved = !currentFileSection.ModerateComments;
            comment.ThreadID = currentFile.ThreadID;
            comment.OwnerUserID = currentFile.OwnerUserID;

            comment.UserID = CurrentUser != null ? CurrentUser.UserID : 0;
            comment.PostDate = DateTime.Now;
            comment.ParentID = Request.Form.GetInt("parentID", 0);

            if (comment.ParentID > 0)
            {

                FileComment parentComment = FileComments.GetComment(comment.ParentID, null, true);
                if (parentComment != null)
                {
                    comment.ParentAuthor = parentComment.Author;
                    comment.ParentBody = parentComment.Body;
                    comment.ParentUserID = parentComment.UserID;
                    comment.ParentPostDate = parentComment.PostDate;
                    if (comment.Body.StartsWith(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author), StringComparison.CurrentCultureIgnoreCase))
                    {
                        if (comment.Body.Length > string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author).Length)
                        {
                            comment.Body = comment.Body.Substring(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author).Length);
                        }
                        else
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = "请输入回复内容";
                            TempData["commentBody"] = comment.Body;
                            TempData["parentID"] = comment.ParentID;
                            return RedirectToAction("Control_FileComments", new RouteValueDictionary { { "threadID", threadID }, { "pageIndex", pageIndex } });
                        }
                    }
                }
            }

            if (CurrentUser == null)
            {
                comment.Author = Request.Form.GetString("author", GetResourceString("Common_AnonymousDisplayName")).Trim();
                comment.UserID = 0;

                UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
                if (Request.Form.Get<int>("rememberMe", -1) > 0)
                {
                    userCookie.SaveRememberMeInfor_name(WebUtils.HtmlEncode(comment.Author));
                    // userCookie.SaveRememberMeInfor_url(WebUtils.HtmlEncode(comment.TitleUrl));
                }
                else
                {
                    userCookie.SaveRememberMeInfor_name(string.Empty);
                    // userCookie.SaveRememberMeInfor_url(string.Empty);
                }
            }
            else
            {
                comment.Author = CurrentUser.DisplayName;
                comment.UserID = CurrentUser.UserID;
                comment.TitleUrl = SiteUrls.Instance().UserDomainHome(CurrentUser);
            }

            try
            {
                #region 检查发帖时间间隔,防灌水
                Globals.CheckPostDate();
                #endregion

                comment = FileComments.CreateComment(comment);
            }
            catch (SPBException e)
            {
                ExceptionMessage exceptionMessage = ResourceManager.GetExceptionMessage(e.ExceptionType, ApplicationIDs.Instance().File());
                if (exceptionMessage != null)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = exceptionMessage.Title + "，" + exceptionMessage.Body;
                }
                return Control_FileComments(threadID, pageIndex);
            }

            if (comment.PostID > 0)
            {
                if (comment.AuditingStatus == AuditingStatuses.Success)
                {
                    if (comment.IsApproved)
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                        TempData["StatusMessageContent"] = GetResourceString("Message_SucceedDeclareOneself", appid);
                    }
                    else
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                        TempData["StatusMessageContent"] = GetResourceString("Message_CommentSucceedNeedCheckPassCanDisplay", appid);
                    }
                }
                else
                {
                    switch (comment.AuditingStatus)
                    {
                        case AuditingStatuses.Again:
                        case AuditingStatuses.Pending:
                            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                            TempData["StatusMessageContent"] = GetResourceString("Message_CommentSucceedNeedAdministratorsCheckPassCanDisplay", appid);
                            break;
                        case AuditingStatuses.Fail:
                            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                            TempData["StatusMessageContent"] = GetResourceString("Message_CommentSucceedNeedAdministratorsCheckPassCanDisplay", appid);
                            break;
                        default:
                            break;
                    }
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_CommentLostContactAdministrators", appid);
            }

            return Control_FileComments(threadID, pageIndex);
        }

        /// <summary>
        /// 删除单条文件评论
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="filePostID">要删除评论所属文件的ID</param>
        /// <param name="commentPostID">要删除评论的ID</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteFileComment(int fileThreadID, int commentPostID, int? pageIndex)
        {
            if (CurrentUser == null)
                throw new ArgumentException(GetResourceString("Exception_WithoutPurview", appid));
            FileComment fileComment = FileComments.GetComment(commentPostID, null, true);

            User currentDomainUser = Users.GetUser(fileComment.OwnerUserID, false);

            if (fileComment == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_Pattern_FileCommentNotFound", appid), commentPostID)));
            }
            if (CurrentUser.UserID == fileComment.OwnerUserID || CurrentUser.IsContentAdministrator)
                FileComments.DeleteComment(commentPostID);
            else
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());


            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSuccess");

            return RedirectToAction("Control_FileComments", new RouteValueDictionary { { "threadID", fileThreadID }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 设置文件评论状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult SetFileCommentApproveStatus(int threadID, int postID, bool isApproved, int? pageIndex)
        {
            FileComment fileComment = FileComments.GetComment(postID, null, true);
            User currentDomainUser = Users.GetUser(fileComment.OwnerUserID, false);

            if (fileComment == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_Pattern_FileCommentNotFound", appid), postID)));
            }
            if (CurrentUser.UserID != fileComment.OwnerUserID && CurrentUser.UserID != fileComment.OwnerUserID && !CurrentUser.IsContentAdministrator)
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }
            FileComments.ApproveComment(currentDomainUser.UserID, postID, isApproved);
            return Control_FileComments(threadID, pageIndex);
        }

        #region FileThread

        /// <summary>
        /// 总站中文件排行
        /// </summary>
        /// <param name="sortBy">排序类型</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult FileRanks(FileThreadSortBy? rankBy, int? categoryID, string tagName, int? pageIndex)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, GetResourceString("Title_FileRank", appid), GetResourceString("Title_File", appid), Globals.SiteName));
            BreadCrumb.AddBreadCrumbItem(string.Format(GetResourceString("Title_FileRank", appid)), string.Empty);
            ViewData["rankBy"] = rankBy;
            return View("Pages/Files/FileRanks.aspx", "Masters/File.master");
        }

        /// <summary>
        /// 站点首页 
        /// </summary>
        /// <param name="pageSize"></param>
        /// <returns></returns>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListStickyFilesBlock(int displayCount)
        {
            IList<FileThread> fileThreads = FileThreads.GetTopThreadsSortBy(FileThreadSortBy.StickyAndPostDate, null, null, displayCount);
            bool isShow = false;
            if (Applications.IsEnabled(ApplicationIDs.Instance().File()))
                isShow = true;
            ViewData["isShow"] = isShow;

            return PartialView("Controls/Files/ListStickyFilesBlock.ascx", fileThreads);
        }

        /// <summary>
        /// 显示文件模块
        /// </summary>
        public ActionResult Control_ListFilesBlock(string viewName, string moduleTitle, FileThreadSortBy? sortBy, int? siteCagegoryID, bool? isEssential, int displayCount)
        {
            IList<FileThread> fileThreads = FileThreads.GetTopThreadsSortBy(sortBy ?? FileThreadSortBy.HitTimes, siteCagegoryID, isEssential, displayCount);
            ViewData["moduleTitle"] = moduleTitle;
            ViewData["sortBy"] = sortBy;
            ViewData["siteCagegoryID"] = siteCagegoryID;
            return PartialView(string.Format("Controls/Files/{0}.ascx", viewName), fileThreads);
        }

        /// <summary>
        /// 用户上传的其它文件
        /// </summary>
        public ActionResult Control_OtherFiles(User user, int count)
        {
            PagingDataSet<FileThread> pds = FileThreads.GetThreads(user.UserID, string.Empty, null, 1);

            int i = 1;
            List<FileThread> fileThreads = new List<FileThread>();
            foreach (FileThread file in pds.Records)
            {
                i++;
                fileThreads.Add(file);
                if (i > count)
                    break;
            }

            ViewData["user"] = user;
            return PartialView("Controls/Files/OtherFiles.ascx", fileThreads);
        }

        #endregion

        #region FileSiteTag

        /// <summary>
        /// 标签地图
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult FileTagMap()
        {
            Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, GetResourceString("Title_HotTag"), GetResourceString("Title_File", appid), Globals.SiteName));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_File", appid), SiteUrls.Instance().ChannelFileHome());
            BreadCrumb.AddBreadCrumbItem(string.Format("标签地图"), string.Empty);

            Dictionary<string, KeyValuePair<int, SiteTag>> siteTags = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetTopSiteTagsSortBy(SiteTagsSortBy.ItemCount, 300);

            return View("Pages/Files/FileTagMap.aspx", "Masters/File.master", siteTags);
        }


        /// <summary>
        /// 频道标签列表显示
        /// </summary>
        /// <param name="pageSize">显示标签数量</param>
        /// <param name="tagType">标签类型</param>
        /// <param name="sortBy">按文章数或用户数排序</param>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListFileSiteTags(int pageSize, SiteTagsSortBy? sortBy)
        {
            Dictionary<string, KeyValuePair<int, SiteTag>> pds = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetTopSiteTagsSortBy(sortBy ?? SiteTagsSortBy.ItemCount, pageSize);
            return PartialView("Controls/Files/ListFileSiteTags.ascx", pds);
        }

        /// <summary>
        /// 频道标签列表显示
        /// </summary>
        /// <param name="pageSize">显示标签数量</param>
        /// <param name="tagType">标签类型</param>
        /// <param name="sortBy">按文章数或用户数排序</param>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListFileSiteCategary(int topNumber, int siteCategaryID)
        {
            IList<FileThread> fileThreads = FileThreads.GetTopThreadsSortBy(FileThreadSortBy.HitTimes, siteCategaryID, null, topNumber);
            return PartialView("Controls/Files/ListFileSiteCategary.ascx", fileThreads);
        }

        #endregion

        #region Control Actions

        /// <summary>
        /// 总站文件中的推荐文件
        /// </summary>
        /// <param name="commendTypeID">推荐类别</param>
        /// <returns></returns>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_RecommendFileBlock(int commendTypeID, int? displayCount)
        {
            //获取推荐类型实体集合
            IList<RecommendedItem> pds = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().File(), commendTypeID, displayCount ?? 9);
            //List<FileThread> fileThreads = new List<FileThread>();
            //foreach (RecommendedItem item in pds)
            //{
            //    FileThread thread = FileThreads.GetThread(item.ItemID, false);
            //    fileThreads.Add(thread);
            //}
            //ViewData["fileThreads"] = fileThreads;
            ViewData["RecommendationType"] = RecommendationTypes.GetRecommendationType(commendTypeID, false);
            return PartialView("Controls/Files/RecommendFileBlock.ascx", pds);
        }

        /// <summary>
        /// 文件列表
        /// </summary>
        /// <param name="pageSize">显示数量</param>
        /// <param name="sortBy">排序方式</param>
        /// <param name="sortOrder">升序还是降序</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListFileThreads(int pageSize, FileThreadSortBy? sortBy, int? categoryID, string tagName, int? pageIndex)
        {
            PagingDataSet<FileThread> pds = null;
            if (!string.IsNullOrEmpty(tagName))
            {
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
                ViewData["tagName"] = tagName;
            }
            ViewData["sortBy"] = sortBy;
            if (categoryID.HasValue && categoryID.Value > 0)
                ViewData["categoryID"] = categoryID;
            pds = FileThreads.GetPagingThreadsSortBy(sortBy ?? FileThreadSortBy.HitTimes, categoryID, tagName, null, pageSize, pageIndex ?? 1);
            return PartialView("Controls/Files/ListFileThreads.ascx", pds);
        }

        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_tab_Files(int pageSize, FileThreadSortBy? sortBy)
        {
            IList<FileThread> threads = FileThreads.GetTopThreadsSortBy(sortBy ?? FileThreadSortBy.HitTimes, null, null, pageSize);
            ViewData["sortBy"] = sortBy;
            return PartialView("Controls/Files/tab_Files.ascx", threads);
        }

        #endregion

        #region 隐私密码

        /// <summary>
        /// 验证隐私密码
        /// </summary>
        [HttpGet]
        public ActionResult Control_PrivacyPassword(int threadID)
        {
            return PartialView("Controls/Files/PrivacyPassword.ascx");
        }
        /// <summary>
        /// 验证隐私密码
        /// </summary>
        [HttpPost]
        public ActionResult AuthorizePrivacyPassword(int threadID)
        {
            FileThread thread = FileThreads.GetThread(threadID, null, true);
            if (thread == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format("找不到CategoryID为{0}的文章", threadID));
                return RedirectToAction("Control_PrivacyPassword", CurrentRouteValueDictionary);
            }

            string password = Request.Form.GetString("password", string.Empty).Trim();
            bool isSuccess = FileThreads.ValidPrivacyPassword(password, thread.Password);
            if (isSuccess)
            {
                UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);

                userCookie.SaveAuthorizedFileThreadIDs(threadID, password);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "通过验证");
            }
            else
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "密码错误");
            }
            return RedirectToAction("Control_PrivacyPassword", CurrentRouteValueDictionary);
        }

        #endregion

        /// <summary>
        /// 生成文件类别面包屑
        /// </summary>
        private void ShowCategoryBreadCrumb(int siteCategoryID)
        {
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_File", appid), SiteUrls.Instance().ChannelFileHome());
            Stack<SiteCategory> stack = new Stack<SiteCategory>();
            SiteCategory currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().File()).Get(siteCategoryID, false);
            while (currentCategory != null)
            {
                stack.Push(currentCategory);
                currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().File()).Get(currentCategory.ParentID, false);
            }
            while (stack.Count > 0)
            {
                currentCategory = stack.Pop();
                BreadCrumb.AddBreadCrumbItem(currentCategory.GetMultilingualCategoryName(CurrentUserLanguage), SPBUrlHelper.Action("Search", "ChannelFile", new RouteValueDictionary { { "SiteCategoryID", currentCategory.CategoryID } }));
            }
        }

        #region Rating

        /// <summary>
        /// 对文件进行星级评价
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Rate()
        {
            //文件ThreadID
            int itemID = Request.QueryString.GetInt("itemID", -1);

            //评价的星级
            int value = Request.QueryString.GetInt("value", -1);


            ContentResult result = new ContentResult();
            result.ContentType = "text/html";
            if (!this.HttpContext.User.Identity.IsAuthenticated)
            {
                result.Content = GetResourceString("Message_PleaceLogin");
            }
            else
            {
                if (CurrentUser != null && itemID > 0 && value > 0)
                {
                    Rating rating = new Rating();
                    rating.UserID = CurrentUser.UserID;
                    rating.Author = CurrentUser.DisplayName;
                    rating.ItemID = itemID;
                    rating.Rate = value;
                    FileThreads.Rate(rating);
                    result.Content = GetResourceString("Message_ThanksForRating");
                }
                else
                {
                    result.Content = GetResourceString("Message_ParameterIncorrect");
                }
            }
            return result;
        }
        /// <summary>
        /// 显示评价人列表
        /// </summary>
        public ActionResult Control_ListRatingUsers(int itemID, int? topNumber)
        {
            IList<Rating> ratings = FileThreads.GetRatings(itemID, topNumber ?? 30);
            return View("Controls/Common/ListRatingUsers.ascx", ratings);
        }

        #endregion

        #region FileAttachment
        /// <summary>
        /// 管理博客主题附件
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_UploadFileFeaturedImage(int? threadID, string queueCompleteCallback)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            UploadAttachmentSettings settings = new UploadAttachmentSettings();
            FileFeaturedImageSettings attachmentSettings = (FileFeaturedImageSettings)FileSectionConfiguration.Instance().FeaturedImageSettings;
            string uploadPhotoTypes = attachmentSettings.Extensions;
            StringBuilder newUploadPhotoTypes = new StringBuilder();
            string[] types = uploadPhotoTypes.Split(',');
            foreach (string type in types)
            {
                if (!string.IsNullOrEmpty(type))
                    newUploadPhotoTypes.AppendFormat("*.{0};", type);
            }
            settings.AssociateID = threadID ?? 0;
            settings.FileTypes = newUploadPhotoTypes.ToString();
            settings.UploadUrl = Globals.GetFullUrlInFileHandlerSite(FileFeaturedImageManager.Instance().StorageProviderName, WebUtils.ResolveUrl("~/Services/File/UploadFeaturedImages.ashx"));
            settings.FlashUrl = Globals.GetFullUrlInFileHandlerSite(FileFeaturedImageManager.Instance().StorageProviderName, WebUtils.ResolveUrl("~/Utility/UploadFiles/swfupload.swf"));
            settings.FileQueryLimit = 8;
            settings.FileSizeLimit = attachmentSettings.MaxAttachmentSize.ToString();
            settings.QueueCompleteCallback = queueCompleteCallback;
            settings.IsImage = true;

            ViewData["UploadAttachmentSettings"] = settings;
            return View("Controls/UploadAttachmentInEditer.ascx", settings);
        }

        /// <summary>
        /// 管理博客主题附件
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ManageThreadAttachments(int? threadID)
        {
            if (CurrentUser == null)
                return new EmptyResult();
            FileSection currentFile = FileSections.GetSection(CurrentUser.UserID, false);
            if (currentFile == null)
                return new EmptyResult();
            ManageThreadAttachmentsModel model = new ManageThreadAttachmentsModel();
            model.IsImage = true;

            if (threadID.HasValue && threadID.Value > 0)
            {
                FileThread fileThread = FileThreads.GetThread(threadID.Value, null, true);
                if (fileThread == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_Pattern_VisitFileArticleNoFound", ApplicationIDs.Instance().File())));
                }

                IList<Attachment> attachments = FileFeaturedImageManager.Instance().GetAttachmentsByAssociateID(threadID.Value, true);
                model.Attachments = attachments;
                model.UserFeaturedImageAttachmentID = fileThread.UserFeaturedImageAttachmentID;
            }
            else
            {
                IList<Attachment> attachments = FileFeaturedImageManager.Instance().GetTemporaryAttachmentsOfUser(currentFile.OwnerUserID);
                model.Attachments = attachments;
            }

            model.GetAttachmentUrl = SiteUrls.Instance().FileFeaturedImageUrl;
            model.DeleteAttachmentUrl = SiteUrls.Instance().DeleteFileAttachmentUrl;
            model.ShowAttachmentThumbnail = SPBHtmlHelper.Instance().ShowFileFeaturedImageThumbnail;
            return PartialView("Controls/ManageThreadAttachmentsInEditer.ascx", model);
        }

        public ActionResult UserContentSelector_UpdateFileAttachments(int? threadID)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            return PartialView("Controls/Files/UserContentSelector_UpdateFileAttachments.ascx");
        }

        #region Manage

        /// <summary>
        /// 删除文件附件
        /// </summary>
        public ActionResult DeleteFileAttachment(int attachmentID)
        {

            Attachment attachment = FileFeaturedImageManager.Instance().GetAttachment(attachmentID, true);
            if (attachment == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "找不到附件");
                return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID } });
            }

            if (CurrentUser == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "请先登录");
                return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID } });
            }

            if (attachment.AssociateID > 0)
            {
                FileThread fileThread = FileThreads.GetThread(attachment.AssociateID, null, true);
                if (fileThread == null)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "找不到附件所属的文件");
                    return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID } });
                }
            }

            User attachmentOwner = Users.GetUser(attachment.UserID, false);
            if (attachmentOwner == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "找不到附件的上传者");
                return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID } });
            }

            if (CurrentUser.UserID != attachment.UserID && !CurrentUser.IsContentAdministrator)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "没有权限，只有上传者和网站管理员才能删除附件");
                return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID } });
            }

            FileFeaturedImageManager.Instance().DeleteAttachment(attachmentID);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "删除成功");
            return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID } });
        }

        #endregion

        #endregion


        #region 用户内容选择器

        /// <summary>
        /// 在用户内容选择器中显示用户的照片
        /// </summary>
        /// <param name="categoryID">类别ID</param>
        public ActionResult UserContentSelector_ListFiles(int? categoryID, int? pageIndex)
        {
            IList<FileThread> fileThreads = null;

            User currentUser = Globals.GetCurrentUser();
            if (currentUser != null && currentUser != null)
            {

                fileThreads = FileThreads.GetThreads(currentUser.UserID, string.Empty, categoryID, 1).Records;
            }
            else
            {
                fileThreads = new List<FileThread>();
            }

            List<UserCategory> userCategories = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().File()).GetUserCategories(currentUser.UserID, false);

            ViewData["userCategories"] = userCategories;
            return PartialView("Controls/UserContentSelector_ListFiles.ascx", fileThreads);
        }

        #endregion

        #region FullTextSearch

        /// <summary>
        /// 文件全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Search(int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_FileSearch", appid));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_FileSearch", appid), string.Empty);
            FileThreadFullTextQuery query = new FileThreadFullTextQuery();
            query.ConvertFromQueryString(Request);
            query.PageSize = 10;
            PagingDataSet<FileThread> pds = null;

            Stack<SiteCategory> categoryStack = null;
            List<SiteCategory> siteCategoties = GetChildCategories(query.SiteCategoryID, out categoryStack);
            ViewData["CurrentCategory"] = SiteCategories.Instance(ApplicationIDs.Instance().File()).Get(query.SiteCategoryID, false);

            if (pageIndex.HasValue)
                query.PageIndex = pageIndex.Value;


            if (!string.IsNullOrEmpty(query.Keyword) || query.SearchScopeUserID > 0)
            {
                SearchResultDataSet<FileThread> results = FileSearchManager.Instance().Search(query);
                if (results != null)
                {
                    pds = new PagingDataSet<FileThread>();
                    pds.Records = results.Records;
                    pds.PageSize = results.PageSize;
                    pds.PageIndex = results.PageIndex;
                    pds.TotalRecords = results.TotalRecords;
                }
                ViewData["IsFullTextSearch"] = true;
            }
            else
                pds = FileThreads.GetPagingThreadsSortBy(FileThreadSortBy.StickyAndPostDate, query.SiteCategoryID, query.TagName, null, query.PageSize, pageIndex ?? 1);

            ViewData["categoryStack"] = categoryStack;
            ViewData["siteCategoties"] = siteCategoties;
            ViewData["Query"] = query;
            return View("Pages/Files/Search.aspx", "Masters/File.master", pds);
        }

        /// 文件全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Search()
        {
            RouteValueDictionary parameters = new RouteValueDictionary();
            parameters["Keyword"] = Request.Form.GetString("Keyword", string.Empty);

            int searchScopeUserID = Request.Form.GetInt("searchScopeUserID", -1);
            if (searchScopeUserID > 0)
                parameters["searchScopeUserID"] = searchScopeUserID.ToString();


            return RedirectToAction("Search", parameters);
        }


        /// <summary>
        /// 获取当前类别的子类别，如果是最后一级，则获取同辈类别
        /// </summary>
        /// <param name="siteCategoryID">当前类别</param>
        private List<SiteCategory> GetChildCategories(int? currentCagoryID, out Stack<SiteCategory> categoryStack)
        {
            categoryStack = new Stack<SiteCategory>();
            SiteCategory currentCategory = SiteCategories.Instance(appid).Get(currentCagoryID ?? 0, false);

            List<SiteCategory> siteCategoties = null;

            if (currentCategory == null)
                siteCategoties = SiteCategories.Instance(ApplicationIDs.Instance().File()).GetRoots(false);
            else if (currentCategory.ChildCount > 0)
                siteCategoties = currentCategory.Childs;
            else
            {
                if (currentCategory.ParentID <= 0)
                    siteCategoties = SiteCategories.Instance(ApplicationIDs.Instance().File()).GetRoots(false);
                else
                    siteCategoties = SiteCategories.Instance(ApplicationIDs.Instance().File()).Get(currentCategory.ParentID, false).Childs;
            }

            while (currentCategory != null)
            {
                if (currentCategory.Childs.Count > 0)
                    categoryStack.Push(currentCategory);
                currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().File()).Get(currentCategory.ParentID, false);
            }
            return siteCategoties;
        }

        #endregion

    }
    /// <summary>
    /// 频道中的文件子菜单
    /// </summary>
    public enum ChannelFileSubMenu
    {
        /// <summary>
        /// 文件首页
        /// </summary>
        Home = 1,
        /// <summary>
        /// 搜索文件
        /// </summary>
        Search = 2,
        /// <summary>
        /// 我的文件
        /// </summary>
        MyFiles = 3,
        /// <summary>
        /// 我的文件
        /// </summary>
        FileRanks = 4
    }
}
