﻿//------------------------------------------------------------------------------
//<copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights reserved.
//</copyright> 
//------------------------------------------------------------------------------ 

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Collections;
using SpaceBuilder.Web;
using SpaceBuilder.Common;
using SpaceBuilder.File;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;


using SpaceBuilder.Utils;
using System.Web.Routing;

namespace SpaceBuilder.File.Controllers
{
    /// <summary>
    /// 个人空间中文件相关
    /// </summary>
    [UserDomainGuideInstallApplication(ApplicationID = 113)]
    [HttpCompress]
    public class FileController : UserDomainControllerBase
    {
        private int fileApplicationID = ApplicationIDs.Instance().File();
        /// <summary>
        /// 个人空间文件首页
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult Home(string userDomainName)
        {
            #region 文件及FileGallery的计数
            AddWebCount<FileSectionCounter>(GetFileSection(userDomainName).OwnerUserID);
            #endregion

            #region 设置标题
            SetPageTitle(userDomainName, string.Empty);
            #endregion

            return View("Pages/Files/Home.aspx", "Masters/File.master");
        }

        /// <summary>
        /// 文件全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Search(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            RouteValueDictionary parameters = new RouteValueDictionary();
            parameters["Keyword"] = Request.Form.GetString("Keyword", string.Empty);
            parameters["SearchScopeUserID"] = currentDomainUser.UserID.ToString();

            return RedirectToAction("Search", "ChannelFile", parameters);
        }

        /// <summary>
        /// 下载
        /// </summary>  
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserVisitAndUserDomainHitTime]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult DownloadFiles(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_DownloadFileInfor", ApplicationIDs.Instance().File()));
            return View("Pages/Files/ListFilesByDownload.aspx", "Masters/File.master");
        }
        /// <summary>
        /// 下载记录
        /// </summary>      
        public ActionResult Control_ListFilesByDownload(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<FileDownLoadRecord> fileDownRecord = FileThreads.GetDownLoadRecordsOfUser(currentDomainUser.UserID, pageIndex ?? 1);
            Dictionary<int, FileThread> filesDictionary = new Dictionary<int, FileThread>();
            foreach (FileDownLoadRecord filethreadRecord in fileDownRecord.Records)
            {
                FileThread filethread = FileThreads.GetThread(filethreadRecord.ThreadID, false);
                if (filethread != null)
                    filesDictionary[filethread.ThreadID] = filethread;
            }
            ViewData["filesDictionary"] = filesDictionary;
            return PartialView("Controls/Files/ListFilesByDownload.ascx", fileDownRecord);
        }
        public ActionResult Control_FileCountBlock(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            FileSection userdate = FileSections.GetSection(currentDomainUser.UserID, true);
            return PartialView("Controls/Files/FileCountBlock.ascx", userdate);
        }

        #region FileThread

        #region Show


        /// <summary>
        /// 显示某类别（文件夹）下的所有文件
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="categoryID">类别ID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListFilesByUserCategory(string userDomainName, int categoryID)
        {
            #region 文件及FileGallery的计数
            AddWebCount<FileSectionCounter>(GetFileSection(userDomainName).OwnerUserID);
            #endregion

            #region 设置标题

            UserCategory currentCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().File()).GetUserCategory(categoryID, false);
            if (currentCategory != null)
            {
                SetPageTitle(userDomainName, currentCategory.CategoryName);
            }

            #endregion

            return View("Pages/Files/ListFilesByCategory.aspx", "Masters/File.master");
        }

        /// <summary>
        /// 显示某类别（文件夹）下的所有文件 在要显示
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="categoryID">类别ID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult SummaryFilesByUserCategory(string userDomainName, int categoryID)
        {
            #region 文件及FileGallery的计数
            AddWebCount<FileSectionCounter>(GetFileSection(userDomainName).OwnerUserID);
            #endregion

            #region 设置标题

            UserCategory currentCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().File()).GetUserCategory(categoryID, false);
            if (currentCategory != null)
            {
                SetPageTitle(userDomainName, currentCategory.CategoryName);
            }

            #endregion

            return View("Pages/Files/SummaryFilesByCategory.aspx", "Masters/File.master");

        }

        /// <summary>
        /// 根据标签名称显示某个标签下的所有文件列表
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名那个</param>
        /// <param name="tagName">标签名称</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListFilesByUserTag(string userDomainName, string tagName)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            #region 文件及FileGallery的计数
            AddWebCount<FileSectionCounter>(GetFileSection(userDomainName).OwnerUserID);
            #endregion

            UserTag userTag = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTag(tagName, GetFileSection(userDomainName).OwnerUserID, true);

            IList<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTags(GetFileSection(userDomainName).OwnerUserID, true).ToList();

            if (userTag == null || userTags == null || !userTags.Contains(userTag))
                return Redirect(SiteUrls.Instance().Error404("标签" + tagName + "不存在或已被删除！"));

            #region 设置标题
            SetPageTitle(userDomainName, tagName);
            #endregion

            return View("Pages/Files/ListFilesByUserTag.aspx", "Masters/File.master");
        }


        /// <summary>
        /// 根据标签名称显示某个标签下的所有文件列表  摘要显示
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名那个</param>
        /// <param name="tagName">标签名称</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult SummaryFilesByUserTag(string userDomainName, string tagName)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");

            UserTag userTag = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTag(tagName, GetFileSection(userDomainName).OwnerUserID, true);

            IList<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTags(GetFileSection(userDomainName).OwnerUserID, true).ToList();

            if (userTag == null || userTags == null || !userTags.Contains(userTag))
                return Redirect(SiteUrls.Instance().Error404("标签" + tagName + "不存在或已被删除！"));


            #region 文件及FileGallery的计数
            AddWebCount<FileSectionCounter>(GetFileSection(userDomainName).OwnerUserID);
            #endregion

            #region 设置标题
            SetPageTitle(userDomainName, tagName);
            #endregion

            return View("Pages/Files/SummaryFilesByUserTag.aspx", "Masters/File.master");
        }

        /// <summary>
        /// 显示文件列表
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="pageIndex">当前页码</param>
        /// <param name="categoryID">所属类别ID</param>
        /// <param name="tagName">所属标签名称</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        public ActionResult Control_SummaryFiles(string userDomainName, int? pageIndex, int? categoryID, string tagName)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            bool isManager = false;
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                isManager = true;
            //string listFilesUrl = SPBUrlHelper.Action("ListFiles", "File", new RouteValueDictionary { { "userDomainName", userDomainName } });
            string listFilesUrl = SPBUrlHelper.Action("Control_ListFiles", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", categoryID }, { "tagName", WebUtils.UrlEncode(tagName) }, { "pageIndex", pageIndex } });
            string title = string.Empty;

            if (categoryID != null && (int)categoryID > 0)
            {
                UserCategory currentCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().File()).GetUserCategory((int)categoryID, false);
                if (currentCategory != null)
                {
                    title = string.Format(GetResourceString("Common_Category") + "：{0}", currentCategory.CategoryName);
                    //listFilesUrl = SPBUrlHelper.Action("ListFilesByUserCategory", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", categoryID } });
                    listFilesUrl = SPBUrlHelper.Action("Control_ListFiles", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", categoryID }, { "tagName", WebUtils.UrlEncode(tagName) }, { "pageIndex", pageIndex } });
                }
            }

            if (!string.IsNullOrEmpty(tagName))
            {
                title = string.Format(GetResourceString("Common_UserTag") + "：{0}", tagName);
                //listFilesUrl = SPBUrlHelper.Action("ListFilesByUserTag", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "tagName", WebUtils.UrlEncode(tagName) } });
                listFilesUrl = SPBUrlHelper.Action("Control_ListFiles", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", categoryID }, { "tagName", WebUtils.UrlEncode(tagName) }, { "pageIndex", pageIndex } });
            }


            PagingDataSet<FileThread> threads = new PagingDataSet<FileThread>();
            if (isManager)
                threads = FileThreads.GetMyThreads(currentDomainUser.UserID, tagName, categoryID, pageIndex ?? 1);
            else
                threads = FileThreads.GetThreads(currentDomainUser.UserID, tagName, categoryID, pageIndex ?? 1);
            bool isFriend = false;
            isFriend = CurrentUser != null && Friends.IsFriend(CurrentUser.UserID, currentDomainUser.UserID);
            ViewData["listFilesUrl"] = listFilesUrl;
            ViewData["currentDomainUserID"] = currentDomainUser.UserID;
            ViewData["isFriend"] = isFriend;
            ViewData["title"] = title;
            ViewData["isManager"] = isManager;
            ViewData["tagName"] = tagName;
            ViewData["userCategoryID"] = categoryID;
            return PartialView("Controls/Files/SummaryFiles.ascx", threads);
        }

        /// <summary>
        /// 显示我的朋友的最新文件
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListFriendsFileThreads(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (!pageIndex.HasValue)
                pageIndex = 0;
            PagingDataSet<FileSection> friendsFileSections = FileSections.GetFirendRecentFileThreads(currentDomainUser.UserID, pageIndex.Value, 20);

            return PartialView("Controls/Files/ListFriendsFileThreads.ascx", friendsFileSections);
        }

        /// <summary>
        /// 列表显示朋友的文件
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListFriendsFileThreads(string userDomainName, int? pageIndex)
        {
            //User currentDomainUser = GetCurrentDomainUser(userDomainName);
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_FriendsFile", fileApplicationID));
            #endregion
            return View("Pages/Files/ListFriendsFileThreads.aspx", "Masters/File.master", string.Empty);
        }

        /// <summary>
        /// 显示输入密码框
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_EditPassWord(string userDomainName)
        {
            ViewData["actionUrl"] = SPBUrlHelper.Action("SetFileThreadsPrivacyStatus", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "setFileThreadsPirvacyStatus", (int)PrivacyStatuses.NeedPassword } });

            return PartialView("Controls/Common/EditPassWord.ascx");
        }

        /// <summary>
        /// 管理文件
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ManageFileThreads(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_FilesManage", ApplicationIDs.Instance().File()));

            return this.View("Pages/Files/ManageFiles.aspx", "Masters/File.master");
        }


        /// <summary>
        /// 管理文件
        /// </summary>
        public ActionResult Control_ManageFileThreads(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            bool hasPermission = HasManagePermission(userDomainName);
            PagingDataSet<FileThread> blogThreads = FileThreads.GetMyThreads(currentDomainUser.UserID, string.Empty, null, pageIndex ?? 1);
            ViewData["hasPermission"] = hasPermission;
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTags(currentDomainUser.UserID, true);
            List<UserCategory> userCategoryList = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().File()).GetUserCategories(currentDomainUser.UserID, true);
            List<SiteCategory> allSiteCategoryList = SiteCategories.Instance(ApplicationIDs.Instance().File()).GetAll(false);
            ViewData["userTags"] = userTags;
            ViewData["userCategoryList"] = userCategoryList;
            ViewData["allSiteCategoryList"] = allSiteCategoryList;
            return PartialView("Controls/Files/ManageFiles.ascx", blogThreads);
        }


        #endregion

        #region Manage


        public ActionResult UploadFilesWihtJqueryHandler(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            int uploadUserID = currentDomainUser.UserID;
            string subject = Request.Form.GetString("subject", string.Empty);
            string body = Request.Form.GetString("body", string.Empty);

            for (int i = 0; i < Request.Files.Count; i++)
            {
                System.Web.HttpPostedFileBase hpf = Request.Files[i] as System.Web.HttpPostedFileBase;
                if (hpf.ContentLength == 0)
                    continue;

                FileThread fileThread = new FileThread();
                fileThread.OwnerUserID = currentDomainUser.UserID;
                List<string> selectedTagNames = new List<string>();
                int siteCategoryID = Request.Form.GetInt("blogSiteCategoryDDL", 0);
                int userCategoryID = Request.Form.GetInt("userCategory", 0);
                string userTags = Request.Form.GetString("userTags", string.Empty);
                string passWord = Request.Form.GetString("passWord", string.Empty);
                int points = Request.Form.GetInt("price", 0);
                List<UserTag> selectedUserTags = new List<UserTag>();

                if (fileThread.OwnerUserID > 0 && Users.GetUser(fileThread.OwnerUserID, false) != null)
                {
                    fileThread.Author = currentDomainUser.DisplayName;
                    fileThread.SiteCategoryID = siteCategoryID;
                    fileThread.UserCategoryID = userCategoryID;

                    if (!string.IsNullOrEmpty(userTags))
                    {
                        string[] tagNames = (userTags).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                        if (tagNames != null && tagNames.Length > 0)
                        {
                            foreach (string tagName in tagNames)
                            {
                                if (!string.IsNullOrEmpty(tagName.Trim()))
                                {
                                    UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTag(tagName.Trim(), fileThread.OwnerUserID, true);
                                    if (tag == null)
                                    {
                                        UserTag newTag = new UserTag();
                                        newTag.UserID = fileThread.OwnerUserID;
                                        newTag.TagName = StringUtils.Trim(tagName, 8, string.Empty);
                                        newTag.TagName = Formatter.CleanTagName(newTag.TagName);
                                        newTag.TagName = Formatter.FormatSingleLinePlainTextForStorage(newTag.TagName);
                                        if (UserTagManager.Instance(UserTagItemKeys.Instance().File()).CreateUserTag(newTag))
                                        {
                                            selectedUserTags.Add(newTag);
                                            selectedTagNames.Add(newTag.TagName);
                                        }
                                    }
                                    else
                                    {
                                        selectedTagNames.Add(tag.TagName);
                                        selectedUserTags.Add(tag);
                                    }
                                }
                            }
                        }
                    }

                    string sfsd = string.Empty;

                    //fileThread.IsPublic = isPublic;
                    fileThread.Password = passWord;
                    fileThread.Price = points;

                    //创建的时候使有各种属性！
                    Attachment attachment = new Attachment(FileAttachmentManager.Instance().StorageProviderName, hpf.FileName);
                    attachment.ContentSize = hpf.ContentLength;
                    attachment.ContentType = hpf.ContentType;
                    attachment.DateCreated = DateTime.Now;
                    fileThread.ContentType = attachment.ContentType;
                    fileThread.FileName = attachment.FileName;
                    fileThread.Attachment = attachment;
                    fileThread.PrivacyStatus = PrivacyStatuses.Privacy;

                    FileThreads.CreateThread(fileThread, selectedUserTags, hpf.InputStream);
                    if (selectedTagNames.Count > 0)
                        UserTagManager.Instance(UserTagItemKeys.Instance().File()).UpdateUserTagsOfItem(fileThread.ThreadID, fileThread.OwnerUserID, selectedTagNames);
                }


            }
            return new EmptyResult();
        }

        /// <summary>
        /// 上传文件
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="postID">需要编辑的文件的PostID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        //[UserTypeAuthorize(UserType = UserTypes.PersonUser)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult UploadFiles(string userDomainName, int? threadID)
        {
            User CurrentDomainUser = GetCurrentDomainUser(userDomainName);
            FileSection currentFileSection = GetFileSection(userDomainName);
            FileThread currentFileThread = null;
            if (threadID != null && threadID > 0)
            {
                currentFileThread = FileThreads.GetThread((int)threadID, true);
                if (currentFileThread == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", fileApplicationID), threadID)));
                }
                ValidateItemOwner(currentFileThread.OwnerUserID);
                ViewData["SelectedUserTags"] = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTagsOfItem(currentFileThread.ThreadID, HasManagePermission(CurrentDomainUser));
            }
            else
            {
                currentFileThread = new FileThread();
            }

            #region 设置标题
            if (currentFileThread.ThreadID > 0)
                SetPageTitle(userDomainName, string.Format(GetResourceString("Title_Pattern_Edit", fileApplicationID), currentFileThread.Subject));
            else
                SetPageTitle(userDomainName, GetResourceString("Title_CreateFile", fileApplicationID));

            #endregion

            ViewData["restrictIllustration"] = string.Format(GetResourceString("Description_Pattern_AllowUploadType", fileApplicationID), FileAttachmentManager.Instance().Settings.Extensions, FileAttachmentManager.Instance().Settings.MaxAttachmentSize);

            return View("Pages/Files/UploadFiles.aspx", "Masters/File.master", currentFileThread);
        }

        /// <summary>
        /// 编辑文件
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="threadID">需要编辑的文件的threadID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserTypeAuthorize(UserType = UserTypes.PersonUser)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult EditFile(string userDomainName, int? threadID)
        {
            User CurrentDomainUser = GetCurrentDomainUser(userDomainName);
            FileSection currentFileSection = GetFileSection(userDomainName);
            FileThread currentFileThread = null;
            if (threadID != null && threadID > 0)
            {
                currentFileThread = FileThreads.GetThread((int)threadID, true);
                if (currentFileThread == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", fileApplicationID), threadID)));
                }
                ValidateItemOwner(currentFileThread.OwnerUserID);
                ViewData["SelectedUserTags"] = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTagsOfItem(currentFileThread.ThreadID, HasManagePermission(CurrentDomainUser));
            }
            else if (TempData["currentFileThread"] != null)
            {
                currentFileThread = TempData.Get<FileThread>("currentFileThread", new FileThread());
            }
            if (currentFileThread == null)
            {
                currentFileThread = new FileThread();
            }

            #region 设置标题
            if (currentFileThread.ThreadID > 0)
                SetPageTitle(userDomainName, string.Format(GetResourceString("Title_Pattern_Edit", fileApplicationID), currentFileThread.Subject));
            else
                SetPageTitle(userDomainName, GetResourceString("Title_CreateFile", fileApplicationID));

            #endregion

            ViewData["restrictIllustration"] = string.Format(GetResourceString("Description_Pattern_AllowUploadType", fileApplicationID), FileAttachmentManager.Instance().Settings.Extensions, FileAttachmentManager.Instance().Settings.MaxAttachmentSize);

            return View("Pages/Files/EditFile.aspx", "Masters/File.master", currentFileThread);
        }


        /// <summary>
        /// 更新文件（普通模式）
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="postID">需要更新的文件的PostID</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserTypeAuthorize(UserType = UserTypes.PersonUser)]
        public ActionResult OrdinaryUpdate(string userDomainName, int? threadID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            FileSection currentFileSection = GetFileSection(userDomainName);

            FileThread currentFileThread = null;
            if (threadID.HasValue && threadID.Value > 0)
            {
                currentFileThread = FileThreads.GetThread(threadID.Value, true);
                ValidateItemOwner(currentFileThread.OwnerUserID);
            }
            else
            {
                currentFileThread = new FileThread();
                ValidateItemOwner(currentDomainUser.UserID);
            }

            bool fileChanged = false;
            System.Web.HttpPostedFileBase hpf = null;

            currentFileThread.Subject = Request.Form.GetString("subject", string.Empty).Trim();
            if (string.IsNullOrEmpty(currentFileThread.Subject))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请填写文件名称";
            }
            else
            {
                int oldUserCategoryID = currentFileThread.UserCategoryID;
                currentFileThread.BodyForDataTransmission = Request.Form.GetString("description", string.Empty);
                currentFileThread.PrivacyStatus = (PrivacyStatuses)Request.Form.GetInt("setFilePirvacyStatus", (int)PrivacyStatuses.Public);
                currentFileThread.Password = Request.Form.GetString("setFilePirvacyStatus_Password", string.Empty);
                currentFileThread.OnlyAllowRegisteredUserDownload = Request.Form.GetBool("onlyAllowRegisteredUserDownload", FileSectionConfiguration.Instance().OnlyAllowRegisteredUserDownload);
                currentFileThread.UserCategoryID = Request.Form.GetInt("fileUserCategoryDDL", 0);
                if (FileSectionConfiguration.Instance().EableUserEntrySiteCategory)
                    currentFileThread.SiteCategoryID = Request.Form.GetInt("fileSiteCategoryDDL", 0);
                currentFileThread.Price = Request.Form.GetInt("price", 0);
                currentFileThread.UserFeaturedImageAttachmentID = Request.Form.GetInt("featurePicAttachmentID", 0);
                TempData["currentFileThread"] = currentFileThread;
                if (!(threadID > 0))
                {
                    currentFileThread.OwnerUserID = currentDomainUser.UserID;
                    currentFileThread.Author = currentDomainUser.DisplayName;
                }

                if (Request.Files.Count > 0)
                {
                    hpf = Request.Files[0] as System.Web.HttpPostedFileBase;

                    if (hpf.ContentLength == 0)
                    {
                        if (!(currentFileThread.ThreadID > 0))
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = "上传文件大小不能为0字节。";
                            return RedirectToAction("EditFile", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", currentFileThread.ThreadID } });
                        }
                    }
                    else
                    {
                        if (hpf.ContentLength > FileAttachmentManager.Instance().Settings.MaxAttachmentSize * 1024)
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = string.Format("上传文件大小不能超过{0}K。", FileAttachmentManager.Instance().Settings.MaxAttachmentSize);
                            return RedirectToAction("EditFile", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", currentFileThread.ThreadID } });
                        }
                        if (hpf.ContentLength / 1024 > (currentDomainUser.DatabaseQuota - currentDomainUser.DatabaseQuotaUsed))
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = "你的空间容量不足";
                            return RedirectToAction("EditFile", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", currentFileThread.ThreadID } });
                        }
                        if (!string.IsNullOrEmpty(hpf.FileName))
                        {
                            currentFileThread.FileName = hpf.FileName;
                            string fileTypeString = hpf.FileName.Substring(hpf.FileName.LastIndexOf(".") + 1);
                            string uploadPhotoTypes = FileAttachmentManager.Instance().Settings.Extensions;
                            StringBuilder newUploadPhotoTypes = new StringBuilder();
                            string[] types = uploadPhotoTypes.Split(',');
                            bool isPermitted = false;
                            foreach (string type in types)
                            {
                                if (!string.IsNullOrEmpty(type))
                                {
                                    if (string.Compare(type, fileTypeString, true) == 0)
                                    {
                                        isPermitted = true;
                                    }
                                }
                            }
                            if (!isPermitted)
                            {
                                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                                TempData["StatusMessageContent"] = string.Format("允许上传文件类型有：{0}。", FileAttachmentManager.Instance().Settings.Extensions);
                                return RedirectToAction("EditFile", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", currentFileThread.ThreadID } });
                            }
                        }
                        string newfileName = hpf.FileName.Substring(hpf.FileName.LastIndexOf("\\") + 1);
                        if (!(currentFileThread.ThreadID > 0) || (currentFileThread != null && !(newfileName == currentFileThread.Attachment.FileName && hpf.ContentLength == (int)currentFileThread.Attachment.ContentSize)))
                        {
                            Attachment attachment = new Attachment(FileAttachmentManager.Instance().StorageProviderName, null);
                            attachment.ContentSize = hpf.ContentLength;
                            attachment.ContentType = hpf.ContentType;
                            attachment.DateCreated = DateTime.Now;
                            attachment.FriendlyFileName = newfileName;
                            currentFileThread.Attachment = attachment;
                            fileChanged = true;
                        }
                    }
                }

                List<string> selectedTagNames = new List<string>();
                List<UserTag> selectedTags = new List<UserTag>();
                if (!string.IsNullOrEmpty(Convert.ToString(Request.Form.Get<string>("userTags"))))
                {
                    string[] tagNames = (Convert.ToString(Request.Form.Get<string>("userTags"))).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                    if (tagNames != null && tagNames.Length > 0)
                    {
                        foreach (string tagName in tagNames)
                        {
                            if (!string.IsNullOrEmpty(tagName.Trim()))
                            {
                                UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTag(tagName.Trim(), currentFileThread.OwnerUserID, true);
                                if (tag == null)
                                {
                                    UserTag newTag = new UserTag();
                                    newTag.UserID = currentFileThread.OwnerUserID;
                                    newTag.TagName = StringUtils.Trim(tagName, 8, string.Empty);
                                    newTag.TagName = Formatter.CleanTagName(newTag.TagName);
                                    newTag.TagName = Formatter.FormatSingleLinePlainTextForStorage(newTag.TagName);

                                    if (UserTagManager.Instance(UserTagItemKeys.Instance().File()).CreateUserTag(newTag))
                                    {
                                        selectedTags.Add(newTag);
                                        selectedTagNames.Add(newTag.TagName);
                                    }
                                }
                                else
                                {
                                    selectedTagNames.Add(tag.TagName);
                                    selectedTags.Add(tag);
                                }
                            }
                        }
                    }
                }

                TempData["StatusMessageContent"] = GetResourceString("Message_FileUpdateSucceed", fileApplicationID);

                if (selectedTagNames.Count > 0)
                    UserTagManager.Instance(UserTagItemKeys.Instance().File()).UpdateUserTagsOfItem(currentFileThread.ThreadID, currentFileThread.OwnerUserID, selectedTagNames);
                else
                    UserTagManager.Instance(UserTagItemKeys.Instance().File()).DeleteUserTagsOfItem(currentFileThread.ThreadID);

                if (currentFileThread.ThreadID <= 0 && hpf != null)
                {
                    FileThreads.CreateThread(currentFileThread, selectedTags, hpf.InputStream);
                    TempData["StatusMessageContent"] = "文件创建成功！";
                }
                else if (fileChanged && hpf != null)
                {
                    FileThreads.UpdateThread(currentFileThread, hpf.InputStream, oldUserCategoryID, selectedTagNames);
                    TempData["StatusMessageContent"] = "文件更新成功！";
                }
                else
                {
                    FileThreads.UpdateThread(currentFileThread, oldUserCategoryID, selectedTagNames);
                    TempData["StatusMessageContent"] = "文件更新成功！";
                }

                TempData["StatusMessageType"] = (int)StatusMessageType.Success;


            }
            TempData["currentFileThread"] = null;
            return Redirect(SiteUrls.Instance().ShowFile(currentFileThread.ThreadID));
        }

        /// <summary>
        /// 更新文件
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="postID">需要更新的文件的PostID</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        //[UserTypeAuthorize(UserType = UserTypes.PersonUser)]
        public ActionResult UpdateFile(string userDomainName, int? threadID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            FileSection currentFileSection = GetFileSection(userDomainName);
            FileThread currentFileThread = null;
            if (threadID.HasValue && threadID.Value > 0)
                currentFileThread = FileThreads.GetThread(threadID.Value, true);
            else
            {
                IList<int> threadIDs = FileThreads.GetTopThreadIDsOfUser(currentDomainUser.UserID, 1);
                if (threadIDs != null && threadIDs.Count > 0)
                    currentFileThread = FileThreads.GetThread(threadIDs.First(), true);
                //转换临时附件
                if (currentFileThread != null)
                    FileFeaturedImageManager.Instance().ToggleTemporaryAttachments(currentFileThread.ThreadID, currentFileThread.OwnerUserID, null);
            }
            if (currentFileThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", fileApplicationID), threadID)));
            }

            ValidateItemOwner(currentFileThread.OwnerUserID);

            currentFileThread.Subject = Request.Form.GetString("subject", string.Empty).Trim();
            if (string.IsNullOrEmpty(currentFileThread.Subject))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_FillInFileName", fileApplicationID);
                return Redirect(SiteUrls.Instance().ShowFile(currentFileThread.ThreadID));
            }

            int oldUserCategoryID = currentFileThread.UserCategoryID;
            currentFileThread.BodyForDataTransmission = Request.Form.GetString("description", string.Empty);
            currentFileThread.PrivacyStatus = (PrivacyStatuses)Request.Form.GetInt("setFilePirvacyStatus", (int)PrivacyStatuses.Public);
            currentFileThread.Password = Request.Form.GetString("setFilePirvacyStatus_Password", string.Empty);
            currentFileThread.OnlyAllowRegisteredUserDownload = Request.Form.GetBool("onlyAllowRegisteredUserDownload", FileSectionConfiguration.Instance().OnlyAllowRegisteredUserDownload);
            currentFileThread.UserCategoryID = Request.Form.GetInt("fileUserCategoryDDL", 0);
            if (FileSectionConfiguration.Instance().EableUserEntrySiteCategory)
                currentFileThread.SiteCategoryID = Request.Form.GetInt("fileSiteCategoryDDL", 0);
            currentFileThread.Price = Request.Form.GetInt("price", 0);
            currentFileThread.UserFeaturedImageAttachmentID = Request.Form.GetInt("featurePicAttachmentID", 0);

            if (!threadID.HasValue || threadID.Value <= 0)
            {
                currentFileThread.OwnerUserID = currentDomainUser.UserID;
                currentFileThread.Author = currentDomainUser.DisplayName;
            }
            List<string> selectedTagNames = new List<string>();
            List<UserTag> selectedTags = new List<UserTag>();
            if (!string.IsNullOrEmpty(Convert.ToString(Request.Form.Get<string>("userTags"))))
            {
                string[] tagNames = (Convert.ToString(Request.Form.Get<string>("userTags"))).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                if (tagNames != null && tagNames.Length > 0)
                {
                    foreach (string tagName in tagNames)
                    {
                        if (!string.IsNullOrEmpty(tagName.Trim()))
                        {
                            UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTag(tagName.Trim(), currentFileThread.OwnerUserID, true);
                            if (tag == null)
                            {
                                UserTag newTag = new UserTag();
                                newTag.UserID = currentFileThread.OwnerUserID;
                                newTag.TagName = StringUtils.Trim(tagName, 8, string.Empty);
                                newTag.TagName = Formatter.CleanTagName(newTag.TagName);
                                newTag.TagName = Formatter.FormatSingleLinePlainTextForStorage(newTag.TagName);

                                if (UserTagManager.Instance(UserTagItemKeys.Instance().File()).CreateUserTag(newTag))
                                {
                                    selectedTags.Add(newTag);
                                    selectedTagNames.Add(newTag.TagName);
                                }
                            }
                            else
                            {
                                selectedTagNames.Add(tag.TagName);
                                selectedTags.Add(tag);
                            }
                        }
                    }
                }
            }

            FileThreads.UpdateThread(currentFileThread, oldUserCategoryID, selectedTagNames);
            TempData["StatusMessageContent"] = GetResourceString("Message_FileUpdateSucceed", fileApplicationID);

            if (selectedTagNames.Count > 0)
                UserTagManager.Instance(UserTagItemKeys.Instance().File()).UpdateUserTagsOfItem(currentFileThread.ThreadID, currentFileThread.OwnerUserID, selectedTagNames);
            else
                UserTagManager.Instance(UserTagItemKeys.Instance().File()).DeleteUserTagsOfItem(currentFileThread.ThreadID);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;

            return Redirect(SiteUrls.Instance().ShowFile(currentFileThread.ThreadID));
        }

        /// <summary>
        /// 批量更新文件
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        //[UserTypeAuthorize(UserType = UserTypes.PersonUser)]
        public ActionResult UpdateFiles(string userDomainName, string fileIDs)
        {
            string[] editeIDsString = fileIDs.Split(',');
            List<int> editIDs = new List<int>();
            foreach (string editIDstring in editeIDsString)
            {
                if (!string.IsNullOrEmpty(editIDstring))
                    editIDs.Add(Convert.ToInt32(editIDstring));
            }

            foreach (int fileID in editIDs)
            {
                FileThread file = FileThreads.GetThread(fileID, true);
                if (file != null)
                {
                    ValidateItemOwner(file.OwnerUserID);
                    file.Subject = Request.Form.GetString(string.Format("subject{0}", fileID), file.FileName).Trim();
                    file.BodyForDataTransmission = Request.Form.GetString(string.Format("body{0}", fileID), string.Empty);
                    file.BodyForDataTransmission = Formatter.FormatMultiLinePlainTextForStorage(file.BodyForDataTransmission, true, false);

                    FileThreads.UpdateThread(file, file.UserCategoryID, file.UserTags);

                }
            }

            return RedirectToAction("Home", "File", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 删除单个文件
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="postID">文件的PostID</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteFile(string userDomainName, int threadID, int? pageIndex, bool isSummaryFiles)
        {

            FileSection currentFileSection = GetFileSection(userDomainName);
            FileThread currentFile = FileThreads.GetThread(threadID, true);
            if (currentFile == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", fileApplicationID), threadID)));
            }
            ValidateItemOwner(currentFile.OwnerUserID);
            FileThreads.DeleteThread(currentFile);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_FileDeleteSucceed", fileApplicationID));
            if (Request.QueryString.GetBool("inShowFile", false))
                return RedirectToAction("Home", "ChannelFile");
            else
                if (isSummaryFiles)
                    return RedirectToAction("Control_SummaryFiles", CurrentRouteValueDictionary);

                else
                    return RedirectToAction("Control_ListFiles", CurrentRouteValueDictionary);
        }
        /// <summary>
        /// 批量删除文件
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="categoryID"></param>
        /// <param name="tagName"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserTypeAuthorize(UserType = UserTypes.PersonUser)]
        public ActionResult DeleteFileThreads(string userDomainName, int? categoryID, string tagName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string[] setPostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("ThreadCheckBoxGroup", string.Empty)))
            {
                setPostIDs = Request.Form.GetString("ThreadCheckBoxGroup", string.Empty).Split(',');
                if (setPostIDs != null && setPostIDs.Length > 0)
                {
                    foreach (string postID in setPostIDs)
                    {
                        int id = int.Parse(postID);
                        FileThread file = FileThreads.GetThread(id, true);
                        if (file != null)
                        {
                            ValidateItemOwner(file.OwnerUserID);
                            FileThreads.DeleteThread(file);
                        }
                    }
                }
            }
            return Control_ManageFileThreads(userDomainName, Request.QueryString.GetInt("pageIndex", 1));
        }

        /// <summary>
        /// 文件管理
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListFiles(string userDomainName, int? pageIndex)
        {
            #region 设置标题
            SetPageTitle(userDomainName, GetResourceString("Title_FileList", fileApplicationID));
            #endregion

            return View("Pages/Files/ListFiles.aspx", "Masters/File.master");
        }

        /// <summary>
        /// 文件管理控件
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ListFiles(string userDomainName, int? categoryID, string tagName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            //string summaryUrl = SPBUrlHelper.Action("Home", "file", new RouteValueDictionary { { "userDomainName", userDomainName } });
            string summaryUrl = SPBUrlHelper.Action("Control_SummaryFiles", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex ?? 1 }, { "categoryID", categoryID }, { "tagName", WebUtils.UrlEncode(tagName) } });
            string title = string.Empty;

            if (categoryID.HasValue && categoryID.Value > 0)
            {
                UserCategory fileUserCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().File()).GetUserCategory(categoryID.Value, false);
                if (fileUserCategory == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserCategoryNotFound", fileApplicationID)));
                }
                title = string.Format(GetResourceString("Common_Category") + "：{0}", fileUserCategory.CategoryName);
                //summaryUrl = SPBUrlHelper.Action("SummaryFilesByUserCategory", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", categoryID } });
                summaryUrl = SPBUrlHelper.Action("Control_SummaryFiles", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex ?? 1 }, { "categoryID", categoryID }, { "tagName", WebUtils.UrlEncode(tagName) } });


            }
            if (!string.IsNullOrEmpty(tagName))
            {
                UserTag fileUserTag = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTag(tagName, currentDomainUser.UserID, false);
                if (fileUserTag == null)
                    return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_UserTagFileNotFound", fileApplicationID), tagName)));

                title = string.Format(GetResourceString("Common_UserTag") + "：{0}", tagName);
                //summaryUrl = SPBUrlHelper.Action("SummaryFilesByUserTag", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "tagName", WebUtils.UrlEncode(tagName) } });
                summaryUrl = SPBUrlHelper.Action("Control_SummaryFiles", "File", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex ?? 1 }, { "categoryID", categoryID }, { "tagName", WebUtils.UrlEncode(tagName) } });
            }

            PagingDataSet<FileThread> threads = null;
            bool isManager = false;
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
            {
                isManager = true;
                threads = FileThreads.GetMyThreads(currentDomainUser.UserID, tagName, categoryID, pageIndex ?? 1);
            }
            else
                threads = FileThreads.GetThreads(currentDomainUser.UserID, tagName, categoryID, pageIndex ?? 1);

            ViewData["isManager"] = isManager;
            ViewData["tagName"] = tagName;
            ViewData["categoryID"] = categoryID;
            ViewData["currentDomainUserID"] = currentDomainUser.UserID;
            ViewData["summaryUrl"] = summaryUrl;
            ViewData["title"] = title;

            return PartialView("Controls/Files/ListFiles.ascx", threads);
        }



        /// <summary>
        /// 设置为精华
        /// </summary>
        public ActionResult SetEssential(int threadID)
        {
            FileThread file = FileThreads.GetThread(threadID, true);
            if (file == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", fileApplicationID), threadID)));
            }
            if (CurrentUser == null || !CurrentUser.IsContentAdministrator)
            {
                return new EmptyResult();
                //throw new ArgumentException(GetResourceString("Exception_AdminSetHighLimit"));
            }

            file.IsEssential = true;
            FileThreads.SetEssential(file);

            return new EmptyResult();
        }

        /// <summary>
        /// 取消精华设置
        /// </summary>
        public ActionResult CancelEssential(int threadID)
        {

            FileThread file = FileThreads.GetThread(threadID, true);
            if (file == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", fileApplicationID), threadID)));
            }
            //throw new ArgumentException(string.Format("文件：{0}不存在", threadID));
            if (CurrentUser == null || !CurrentUser.IsContentAdministrator)
            {
                return new EmptyResult();
                //throw new ArgumentException(GetResourceString("Description_AdminSetEssence"));
            }

            file.IsEssential = false;
            FileThreads.UpdateThread(file, file.UserCategoryID, null);

            return new EmptyResult();
        }

        /// <summary>
        /// 设置为置顶
        /// </summary>
        public ActionResult SetTop(int threadID)
        {
            FileThread file = FileThreads.GetThread(threadID, true);
            if (file == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", fileApplicationID), threadID)));
            }
            //throw new ArgumentException(string.Format("文件：{0}不存在", threadID));
            if (CurrentUser == null || !CurrentUser.IsContentAdministrator)
            {
                return new EmptyResult();
                //throw new ArgumentException(GetResourceString("Description_AdminSetEssence"));
            }

            file.SpecialOrder = setSpecialOrder;
            FileThreads.SetSticky(file);

            return new EmptyResult();
        }

        /// <summary>
        /// 取消置顶
        /// </summary>
        public ActionResult CancelTop(int threadID)
        {
            FileThread file = FileThreads.GetThread(threadID, true);
            if (file == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FileNotFound", fileApplicationID), threadID)));
            }
            //throw new ArgumentException(string.Format("文件：{0}不存在", threadID));
            if (CurrentUser == null || !CurrentUser.IsContentAdministrator)
            {
                return new EmptyResult();
                //throw new ArgumentException(GetResourceString("Description_AdminSetEssence"));
            }

            file.SpecialOrder = 0;
            FileThreads.UpdateThread(file, file.UserCategoryID, file.UserTags);

            return new EmptyResult();
        }


        /// <summary>
        /// 批量设置文件隐私设置
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetFileThreadsPrivacyStatus(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string[] setPostIDs = null;
            PrivacyStatuses setPrivacyStatus = (PrivacyStatuses)Request.QueryString.GetInt("setFileThreadsPirvacyStatus", 0);

            if (!string.IsNullOrEmpty(Request.Form.GetString("ThreadCheckBoxGroup", string.Empty)))
            {
                setPostIDs = Request.Form.GetString("ThreadCheckBoxGroup", string.Empty).Split(',');
                string setPassword = Request.Form.GetString("ThreadsPirvacyStatus_Password", string.Empty);

                if (setPostIDs != null && setPostIDs.Length > 0)
                {
                    List<int> setFileThreadIDs = new List<int>();
                    foreach (string postID in setPostIDs)
                    {
                        int id = int.Parse(postID);
                        FileThread fileThread = FileThreads.GetThread(id, true);
                        if (fileThread != null)
                        {
                            ValidateItemOwner(fileThread.OwnerUserID);
                            setFileThreadIDs.Add(id);
                        }
                    }
                    FileThreads.UpdatePrivacyStatus(setFileThreadIDs, setPrivacyStatus, setPassword, currentDomainUser.UserID);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "修改成功！";
                }
            }
            if (setPrivacyStatus == PrivacyStatuses.NeedPassword)
                return RedirectToAction("Control_EditPassWord");
            else
                return RedirectToAction("Control_ManageFileThreads", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 批量设置标签
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetFileThreadsTags(string userDomainName)
        {
            User fileOwner = GetCurrentDomainUser(userDomainName);
            string[] setPostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("ThreadCheckBoxGroup", string.Empty)))
            {
                //设置标签
                string selectedTagNames = Request.Form.GetString("setUserTagss", string.Empty);

                List<string> tagNames = new List<string>();
                if (!string.IsNullOrEmpty(selectedTagNames))
                {

                    string[] selectdTagNamesArry = selectedTagNames.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                    if (selectdTagNamesArry.Length > 0)
                    {
                        foreach (string tagname in selectdTagNamesArry)
                        {
                            if (!string.IsNullOrEmpty(tagname.Trim()))
                            {
                                UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTag(tagname, fileOwner.UserID, true);
                                if (tag != null) //设置本标签
                                {
                                    tagNames.Add(tag.TagName);
                                }
                                else //创建并设置标签
                                {
                                    UserTag userTag = new UserTag();
                                    userTag.UserID = fileOwner.UserID;
                                    userTag.TagName = StringUtils.Trim(tagname, 8, string.Empty);
                                    userTag.TagName = Formatter.CleanTagName(userTag.TagName);
                                    userTag.TagName = Formatter.FormatSingleLinePlainTextForStorage(userTag.TagName);

                                    bool isCreated = UserTagManager.Instance(UserTagItemKeys.Instance().File()).CreateUserTag(userTag);
                                    if (isCreated)
                                        tagNames.Add(userTag.TagName);
                                }
                            }
                        }
                    }
                }
                if (tagNames.Count > 0)
                {
                    setPostIDs = Request.Form.GetString("ThreadCheckBoxGroup", string.Empty).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                    if (setPostIDs != null && setPostIDs.Length > 0)
                    {
                        List<int> setFileThreadIDs = new List<int>();
                        foreach (string postID in setPostIDs)
                        {
                            int id = int.Parse(postID);
                            FileThread file = FileThreads.GetThread(id, true);
                            if (file != null)
                            {
                                UserTagManager.Instance(UserTagItemKeys.Instance().File()).UpdateUserTagsOfItem(file.ThreadID, file.OwnerUserID, tagNames);
                            }
                        }
                    }
                }
            }
            return RedirectToAction("Control_ManageFileThreads", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 批量设置用户分类
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetFileThreadsUserCategory(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string[] setPostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("ThreadCheckBoxGroup", string.Empty)))
            {
                setPostIDs = Request.Form.GetString("ThreadCheckBoxGroup", string.Empty).Split(',');
                if (setPostIDs != null && setPostIDs.Length > 0)
                {
                    int setUserCategoryID = Request.QueryString.GetInt("userCategoryID", 0);
                    List<int> setFileThreadIDs = new List<int>();
                    foreach (string postID in setPostIDs)
                    {
                        int id = int.Parse(postID);
                        FileThread file = FileThreads.GetThread(id, true);
                        if (file != null)
                        {
                            //ValidateItemOwner(blogThread.OwnerUserID);
                            setFileThreadIDs.Add(id);
                        }
                    }
                    FileThreads.UpdateThreadsUserCategory(setFileThreadIDs, setUserCategoryID, currentDomainUser.UserID);
                }
            }

            return RedirectToAction("Control_ManageFileThreads", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 批量设置站点分类
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetFileThreadsSiteCategory(string userDomainName)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            string[] setPostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("ThreadCheckBoxGroup", string.Empty)))
            {
                int setSiteCategoryID = Request.QueryString.GetInt("siteCategoryID", 0);
                setPostIDs = Request.Form.GetString("ThreadCheckBoxGroup", string.Empty).Split(',');
                if (setPostIDs != null && setPostIDs.Length > 0)
                {
                    List<int> setFileThreadIDs = new List<int>();
                    foreach (string postID in setPostIDs)
                    {
                        int id = int.Parse(postID);
                        FileThread file = FileThreads.GetThread(id, true);
                        if (file != null)
                        {
                            //ValidateItemOwner(blogThread.OwnerUserID);
                            setFileThreadIDs.Add(id);
                        }
                    }
                    FileThreads.UpdateThreadsSiteCategory(setFileThreadIDs, setSiteCategoryID);
                }
            }
            string tagName = Request.QueryString.GetString("tagName", string.Empty);
            int categoryID = Request.QueryString.GetInt("categoryID", 0);
            return RedirectToAction("Control_ManageFileThreads", CurrentRouteValueDictionary);
        }

        #endregion

        #endregion

        #region FileComment



        #region Manage



        /// <summary>
        /// 文件评论管理
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ManageFileComments(string userDomainName, int? pageIndex)
        {
            #region 设置标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManageFileComment", fileApplicationID));
            #endregion

            return View("Pages/Files/ManageFileComments.aspx", "Masters/File.master");
        }



        /// <summary>
        /// 根据评论状态显示评论列表
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="approvalStatus">选择状态</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult ManageFileCommentsByApprovalStatus(string userDomainName, int? pageIndex)
        {
            #region 设置标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManageFileComment", fileApplicationID));
            #endregion

            int getApprovalStatus = int.Parse(Request.Form.GetString("approveDropDownList", string.Empty));
            ApprovalStatuses approvalStatus = ApprovalStatuses.IsApproved;
            if (getApprovalStatus == 0)
                approvalStatus = ApprovalStatuses.NotApprove;
            else if (getApprovalStatus == 1)
                approvalStatus = ApprovalStatuses.IsApproved;
            else
                return Control_ManageFileComments(userDomainName, 0, null);

            return Control_ManageFileComments(userDomainName, pageIndex, approvalStatus);
        }

        /// <summary>
        /// 设置文件评论是否通过批准
        /// </summary>
        /// <param name="userDomainName">当前用户用户空间</param>
        /// <param name="isApproved">选中的评论设置为通过验证（true），选中的评论设置为未通过验证（false）</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetFileCommentsApproveStatus(string userDomainName, bool isApproved)
        {
            #region 设置标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManageFileComment", fileApplicationID));
            #endregion

            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            int getApprovalStatus = Convert.ToInt32(Request.Form["approveDropDownList"]);
            ApprovalStatuses? approvalStatus = null;
            if (getApprovalStatus == 0)
                approvalStatus = ApprovalStatuses.NotApprove;
            else if (getApprovalStatus == 1)
                approvalStatus = ApprovalStatuses.IsApproved;

            string approveCommentsString = Request.Form.GetString("fileCommentsCheckBoxGroup", string.Empty);
            if (!string.IsNullOrEmpty(approveCommentsString))
            {
                string[] approveIDStrings = approveCommentsString.Split(',');
                if (approveIDStrings != null && approveIDStrings.Length > 0)
                {
                    List<int> approveIDs = new List<int>();
                    foreach (string IDString in approveIDStrings)
                    {
                        Int32 approveID = int.Parse(IDString);
                        if (approveID > 0)
                            approveIDs.Add(approveID);
                    }
                    FileComments.ApproveComments(currentDomainUser.UserID, approveIDs, isApproved);
                }
            }
            return Control_ManageFileComments(userDomainName, null, approvalStatus);
        }



        /// <summary>
        /// 评论删除文件评论
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteFileComments(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            int getApprovalStatus = int.Parse(Request.Form.GetString("approveDropDownList", string.Empty));
            ApprovalStatuses? approvalStatus = null;
            if (getApprovalStatus == 0)
                approvalStatus = ApprovalStatuses.NotApprove;
            else if (getApprovalStatus == 1)
                approvalStatus = ApprovalStatuses.IsApproved;

            string deleteCommentsString = Request.Form.GetString("fileCommentsCheckBoxGroup", string.Empty);
            if (!string.IsNullOrEmpty(deleteCommentsString))
            {
                string[] approveIDStrings = deleteCommentsString.Split(',');
                if (approveIDStrings != null && approveIDStrings.Length > 0)
                {
                    List<int> approveIDs = new List<int>();
                    foreach (string IDString in approveIDStrings)
                    {
                        Int32 approveID = int.Parse(IDString);
                        if (approveID > 0)
                            approveIDs.Add(approveID);
                    }

                    FileComments.DeleteComments(approveIDs);
                }
            }

            return Control_ManageFileComments(userDomainName, null, approvalStatus);
        }


        /// <summary>
        /// 文件评论管理
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_ManageFileComments(string userDomainName, int? pageIndex, ApprovalStatuses? approvalStatus)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            PagingDataSet<FileComment> pds = FileComments.GetMyFileComments(currentDomainUser.UserID, approvalStatus, pageIndex ?? 1);

            ViewData["approvalStatus"] = approvalStatus;
            return PartialView("Controls/Files/ManageFileComments.ascx", pds);
        }



        #endregion

        #endregion

        #region FileSecion



        /// <summary>
        /// 更新文件设置
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UpdateFileSettings(string userDomainName)
        {
            FileSection currentFileSection = GetFileSection(userDomainName);
            currentFileSection.SectionName = Formatter.FormatSingleLinePlainTextForStorage(Request.Form.GetString("name", string.Empty)).Trim();
            if (string.IsNullOrEmpty(currentFileSection.SectionName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
            }
            else
            {
                currentFileSection.EnableComments = Request.Form.GetBool("enableComments", true);
                currentFileSection.EnableAnonymousPosting = Request.Form.GetBool("enableAnonymousPosting", true);
                currentFileSection.ModerateComments = Request.Form.GetBool("moderateComments", false);

                FileSections.Update(currentFileSection);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;

            }
            return RedirectToAction("ManageFileSettings", "File", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        #endregion

        #region FileCategory

        #region Show

        /// <summary>
        /// 管理文件文件夹（文件类别）
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ManageFileCategories(string userDomainName)
        {
            #region 设置标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManageFolder", fileApplicationID));
            #endregion

            return View("Pages/Files/ManageFileCategories.aspx", "Masters/File.master");
        }

        /// <summary>
        /// 文件夹类别
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_ListFileUserCategories(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            bool isManager = false;
            if (CurrentUser != null && (CurrentUser.IsContentAdministrator || currentDomainUser.UserID == CurrentUser.UserID))
            {
                isManager = true;
            }
            List<UserCategory> userCategories = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().File()).GetUserCategories(currentDomainUser.UserID, isManager);
            ViewData["userDomainName"] = userDomainName;
            ViewData["isManager"] = isManager;
            return PartialView("Controls/Files/ListFileUserCategories.ascx", userCategories);
        }

        /// <summary>
        /// 文件夹下拉列表
        /// </summary>
        //[AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_FileCategoryDropdownList(string userDomainName, int? currentCategoryID, string controlName, string cssClass)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            List<UserCategory> userCategories = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().File()).GetUserCategories(currentDomainUser.UserID, true);
            if (currentCategoryID != null && currentCategoryID > 0)
            {
                ViewData["currentCategoryID"] = (int)currentCategoryID;
            }
            else
                ViewData["currentCategoryID"] = 0;
            ViewData["controlName"] = controlName;
            ViewData["cssClass"] = cssClass;
            return PartialView("Controls/Files/FileCategoryDropdownList.ascx", userCategories);
        }

        #endregion

        #endregion

        #region FileUserTag

        /// <summary>
        /// 文件用户标签
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_ListFileUserTags(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_Pattern_UserNotFound", fileApplicationID), userDomainName)));
            }
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTags(currentDomainUser.UserID, HasManagePermission(userDomainName));
            bool isManager = false;
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                isManager = true;
            ViewData["isManager"] = isManager;
            return PartialView("Controls/Files/ListFileUserTags.ascx", userTags);
        }

        /// <summary>
        /// 管理文件用户标签
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ManageFileUserTags(string userDomainName)
        {

            #region 设置标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManageFileTag", fileApplicationID));
            #endregion

            return View("Pages/Files/ManageFileUserTags.aspx", "Masters/File.master");
        }

        #endregion

        #region 辅助方法

        private int setSpecialOrder = 1;

        /// <summary>
        /// 获取当前空间文件模块
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        private FileSection GetFileSection(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            FileSection fileSection = FileSections.GetSection(currentDomainUser.UserID, false);
            if (fileSection == null)
            {
                Response.Redirect(SiteUrls.Instance().Error404(String.Format(GetResourceString("Message_Pattern_FileModuleNotFound", fileApplicationID), userDomainName)));
            }
            //throw new ArgumentException(String.Format("{0}的文件模块不存在", userDomainName));
            return fileSection;
        }

        /// <summary>
        /// 设置页面标题
        /// </summary>
        protected override void SetPageTitle(string userDomainName, string text)
        {
            //base.SetPageTitle(userDomainName, text);
            FileSection currentFileSection = GetFileSection(userDomainName);
            if (currentFileSection == null)
                return;
            if (string.IsNullOrEmpty(text))
                Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, currentFileSection.SectionName, GetCurrentDomainUser(userDomainName).DisplayName, Globals.SiteName));
            else
                Header.AddTitle(string.Format("{1}{0}{2}{0}{3}{0}{4}", Globals.BrowserTitleSeparator, text, currentFileSection.SectionName, GetCurrentDomainUser(userDomainName).DisplayName, Globals.SiteName));
        }

        /// <summary>
        /// 设置文件详细显示页面的meta标签
        /// </summary>
        private void SetPageMeta(FileThread file)
        {
            string metaKeyWords = string.Empty;
            string metaDescription = string.Empty;

            //标签
            if (file.UserTags != null && file.UserTags.Count > 0)
            {
                foreach (string userTag in file.UserTags)
                {
                    if (userTag != null)
                    {
                        metaKeyWords = string.Format("{1},{0}", metaKeyWords, userTag);
                    }
                }
            }

            //标题和用户名
            metaKeyWords = metaKeyWords + file.Subject;
            metaKeyWords = string.Format("{0},{1}", metaKeyWords, file.Author);

            //站点分类和用户分类
            //if (file.SiteCategory != null)
            //    metaKeyWords = string.Format("{0},{1}",metaKeyWords,file.SiteCategory.CategoryName);
            //if (file.UserCategory != null)
            //    metaKeyWords = string.Format("{0},{1}",metaKeyWords,file.UserCategory.CategoryName);

            metaDescription = string.Format("{0}：{1}", file.Subject, HtmlUtils.TrimHtml(file.GetBody(true), 150));

            Header.AddMetaDescription(metaDescription);
            Header.AddMetaKeywords(metaKeyWords);
        }


        #endregion

        #region 更改文件审核状态

        public ActionResult updateFileAuditingStatuses(int threadID, string userDomainName)
        {
            List<int> threadIds = new List<int>();
            threadIds.Add(threadID);
            FileThreads.UpdateAuditingStatus(threadIds, AuditingStatuses.Success);
            return RedirectToAction("ShowFile", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", threadID } });
        }

        #endregion

    }

    /// <summary>
    /// 空间文件二级菜单
    /// </summary>
    public enum FileUserDomainSubMenus
    {
        /// <summary>
        /// 我的文件
        /// </summary>
        myFile = 1,

        /// <summary>
        /// 朋友的文件
        /// </summary>
        friendFile = 10,

        /// <summary>
        /// 站点中的文件
        /// </summary>
        siteFile = 20,

        /// <summary>
        /// 我下载过的文件
        /// </summary>
        downloadFile = 30
    }
}
