﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common;
using System.Web.Mvc;
using SpaceBuilder.File;
using SpaceBuilder.Common.Controllers;
using SpaceBuilder.Web;
using System.Web.Routing;

namespace SpaceBuilder.File.Controllers
{
    [ManageAuthorize]
    [HttpCompress]
    public class ManageFileController : ManageControllerBase
    {
        #region 文件管理

        /// <summary>
        /// 置顶使用
        /// </summary>
        private int topSpecialOrder = 1;

        /// <summary>
        /// 获取文件列表
        /// </summary>
        /// <param name="Statuses">分类 如：已通过审核</param>
        /// <param name="pageIndex">页码</param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ListManageFiles(AuditingStatusesForDisplay? status)
        {
            SetPageTitle("文件管理");

            status = status ?? AuditingStatusesForDisplay.Pending;
            ViewData["status"] = status;
            int intStatus = Request.QueryString.GetInt("status", -1);
            if (intStatus > 0)
                ViewData["status"] = (AuditingStatusesForDisplay)intStatus;


            return View("Pages/File/ManageFiles.aspx", "Masters/File.master");
        }

        /// <summary>
        /// 文件列表用户控件列表 
        /// </summary>
        /// <param name="Statuses">分类 如：已通过审核</param>
        /// <param name="pageIndex">页码</param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ListFiles(AuditingStatusesForDisplay? status, int? pageIndex)
        {
            int ownerUserID = Request.QueryString.GetInt("OwnerUserID", -1);
            string subjectKeywords = Request.QueryString.GetString("SubjectKeywords", string.Empty).Trim();
            status = status ?? AuditingStatusesForDisplay.Pending;

            PagingDataSet<FileThread> pds = FileThreads.GetThreadsForAdmin(17, pageIndex ?? 1, status, ownerUserID, subjectKeywords);
            return PartialView("Controls/File/ListFiles.ascx", pds);
        }

        /// <summary>
        /// 删除单个文件
        /// </summary>
        /// <param name="Statuses">分类 如：已通过审核</param>
        /// <param name="pageIndex">页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteFile(AuditingStatusesForDisplay? status, int? pageIndex)
        {
            int threadID = Request.QueryString.GetInt("threadID", -1);
            if (threadID < 0)
                throw new ArgumentException("文件threadID错误！");
            FileThread currentFile = FileThreads.GetThread(threadID, true);
            if (currentFile == null)
                throw new ArgumentException(string.Format("{0}文件不存在", threadID));
            FileThreads.DeleteThread(currentFile);
            return Redirect(Request.UrlReferrer.ToString());
        }

        /// <summary>
        /// 批量删除文件
        /// </summary>
        /// <param name="returnUrl">返回的url</param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteFiles(string returnUrl)
        {
            string selectedThreadIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedThreadIDsString))
            {
                string[] selectedThreadIDsStringArray = selectedThreadIDsString.Split(',');
                if (selectedThreadIDsStringArray != null && selectedThreadIDsStringArray.Length > 0)
                {
                    int threadID;
                    foreach (string Id in selectedThreadIDsStringArray)
                    {

                        if (int.TryParse(Id, out threadID))
                        {
                            if (threadID > 0)
                            {
                                FileThread currentFile = FileThreads.GetThread(threadID, true);
                                FileThreads.DeleteThread(currentFile);
                            }
                        }

                    }
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "文件批量删除成功";
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要删除的文件";
            }
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 设置为精华
        /// </summary>
        public ActionResult SetEssential(string returnUrl)
        {
            if (!CurrentUser.IsContentAdministrator)
            {
                throw new ArgumentException(GetResourceString("Exception_AdminSetHighLimit"));
            }
            string selectedThreadIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedThreadIDsString))
            {
                string[] selectedThreadIDsStringArray = selectedThreadIDsString.Split(',');
                if (selectedThreadIDsStringArray != null && selectedThreadIDsStringArray.Length > 0)
                {
                    int threadID;
                    List<int> ListThreadID = new List<int>();
                    foreach (string Id in selectedThreadIDsStringArray)
                    {
                        try
                        {
                            threadID = int.Parse(Id);
                            ListThreadID.Add(threadID);
                        }
                        catch { }
                    }


                    FileThreads.SetEssential(ListThreadID, true);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "文件批量设置精华成功";

                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要操作的文件";
            }
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 取消精华设置
        /// </summary>
        public ActionResult CancelEssential(string returnUrl)
        {

            if (!CurrentUser.IsContentAdministrator)
            {
                throw new ArgumentException("只有管理员才有取消精华的权限！");
            }
            string selectedThreadIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedThreadIDsString))
            {
                string[] selectedThreadIDsStringArray = selectedThreadIDsString.Split(',');
                if (selectedThreadIDsStringArray != null && selectedThreadIDsStringArray.Length > 0)
                {
                    int threadID;
                    List<int> ListThreadID = new List<int>();
                    foreach (string Id in selectedThreadIDsStringArray)
                    {
                        try
                        {
                            threadID = int.Parse(Id);
                            ListThreadID.Add(threadID);
                        }
                        catch { }
                    }

                    FileThreads.SetEssential(ListThreadID, false);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "文件批量取消精华成功";

                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要操作的文件";
            }
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 设为置顶
        /// </summary>
        public ActionResult SetTop(string returnUrl)
        {
            string selectedThreadIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedThreadIDsString))
            {
                string[] selectedThreadIDsStringArray = selectedThreadIDsString.Split(',');
                if (selectedThreadIDsStringArray != null && selectedThreadIDsStringArray.Length > 0)
                {
                    int threadID;
                    List<int> ListThreadID = new List<int>();
                    foreach (string Id in selectedThreadIDsStringArray)
                    {
                        try
                        {
                            threadID = int.Parse(Id);
                            ListThreadID.Add(threadID);
                        }
                        catch { }
                    }

                    FileThreads.SetSticky(ListThreadID, topSpecialOrder);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "文件批量设置置顶成功";

                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要操作的文件";
            }
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 取消置顶
        /// </summary>
        public ActionResult CancelTop(string returnUrl)
        {
            string selectedThreadIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedThreadIDsString))
            {
                string[] selectedThreadIDsStringArray = selectedThreadIDsString.Split(',');
                if (selectedThreadIDsStringArray != null && selectedThreadIDsStringArray.Length > 0)
                {
                    int threadID;
                    List<int> ListThreadID = new List<int>();
                    foreach (string Id in selectedThreadIDsStringArray)
                    {
                        try
                        {
                            threadID = int.Parse(Id);
                            ListThreadID.Add(threadID);
                        }
                        catch { }
                    }

                    FileThreads.SetSticky(ListThreadID, 0);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "文件批量取消置顶成功";
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要操作的文件";
            }
            return Redirect(returnUrl);
        }


        /// <summary>
        /// 批量更新文件状态
        /// </summary>
        /// <param name="returnUrl">返回url</param>
        /// <param name="auditingStatus">AuditingStatuses枚举</param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateFileAuditingStatus(string returnUrl, AuditingStatuses auditingStatus)
        {
            string selectedThreadIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedThreadIDsString))
            {
                string[] selectedThreadIDsStringArray = selectedThreadIDsString.Split(',');
                if (selectedThreadIDsStringArray != null && selectedThreadIDsStringArray.Length > 0)
                {
                    int threadID;
                    List<int> ListThreadID = new List<int>();
                    foreach (string Id in selectedThreadIDsStringArray)
                    {
                        try
                        {
                            threadID = int.Parse(Id);
                            ListThreadID.Add(threadID);
                        }
                        catch { }
                    }

                    FileThreads.UpdateAuditingStatus(ListThreadID, auditingStatus);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "文件批量操作成功";

                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要操作的文件";
            }
            return Redirect(returnUrl);
        }

        #endregion

        #region 文件评论管理
        /// <summary>
        ///获取文件评论列表 
        /// </summary>
        /// <param name="Statuses">分类 如：已通过审核</param>
        /// <param name="pageIndex">页码</param>
        /// <returns></returns>
        public ActionResult ListManageComments(AuditingStatusesForDisplay? status, int? pageIndex)
        {
            SetPageTitle("文件管理");
            ViewData["status"] = AuditingStatusesForDisplay.Pending;
            if (status.HasValue)
                ViewData["status"] = (AuditingStatusesForDisplay)status;

            int intStatus = Request.QueryString.GetInt("status", -1);
            if (intStatus > 0)
                ViewData["status"] = (AuditingStatusesForDisplay)intStatus;

            return View("Pages/File/ManageFileComments.aspx", "Masters/File.master");
        }

        /// <summary>
        /// 获取用户控件的评论列表
        /// </summary>
        /// <param name="Statuses">分类 如：已通过审核</param>
        /// <param name="pageIndex">页码</param>
        /// <returns></returns>
        public ActionResult Control_ListComments(AuditingStatusesForDisplay? status, int? pageIndex)
        {
            int userID = Request.QueryString.GetInt("CommentUserID", -1);
            string bodyKeywords = Request.QueryString.GetString("CommentKeyWords", string.Empty).Trim();

            PagingDataSet<FileComment> pds = FileComments.GetCommentsForAdmin(17, pageIndex ?? 1, status ?? AuditingStatusesForDisplay.Success, userID, bodyKeywords);
            return PartialView("Controls/File/ListComments.ascx", pds);
        }

        /// <summary>
        /// 批量更新评论的状态
        /// </summary>
        /// <param name="returnUrl">返回的地址</param>
        /// <param name="auditingStatus">状态 AuditingStatuses 枚举</param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateCommentAuditingStatus(string returnUrl, AuditingStatuses auditingStatus)
        {
            string selectedPostIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedPostIDsString))
            {
                string[] selectedPostIDsStringArray = selectedPostIDsString.Split(',');
                if (selectedPostIDsStringArray != null && selectedPostIDsStringArray.Length > 0)
                {
                    int postID;
                    List<int> ListPostID = new List<int>();
                    string[] ids;
                    foreach (string Id in selectedPostIDsStringArray)
                    {
                        try
                        {
                            ids = Id.Split(':');
                            postID = int.Parse(ids[0]);
                            ListPostID.Add(postID);
                        }
                        catch { }
                    }


                    FileComments.UpdateAuditingStatus(ListPostID, auditingStatus);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "评论批量操作成功";

                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要操作的评论";
            }
            return Redirect(returnUrl);
        }

        /// <summary>
        ///批量删除文件评论 
        /// </summary>
        /// <param name="returnUrl">返回的url</param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteComments(string returnUrl)
        {
            string selectedPostIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedPostIDsString))
            {
                string[] selectedPostIDsStringArray = selectedPostIDsString.Split(',');
                if (selectedPostIDsStringArray != null && selectedPostIDsStringArray.Length > 0)
                {
                    string[] IDs;
                    int postID;
                    int ownerID;
                    foreach (string postownerIDs in selectedPostIDsStringArray)
                    {
                        IDs = postownerIDs.Split(':');
                        if (int.TryParse(IDs[0], out postID) && int.TryParse(IDs[1], out ownerID))
                            if (postID > 0 && ownerID > 0)
                                FileComments.DeleteComment(postID);
                    }
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "评论批量删除成功";
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要删除的评论";
            }
            return Redirect(returnUrl);
        }


        #endregion

        #region 站点类别
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult FileCategories()
        {
            SetPageTitle("文件管理");
            ViewData["SiteCategoryDepth"] = FileSectionConfiguration.Instance().SiteCategoryLevel - 1;
            return View("Pages/File/ManageFileCategories.aspx", "Masters/File.master");
        }

        /// <summary>
        /// 设置站点类别
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Control_SetSiteCategories()
        {
            string selectedThreadIDsString = Request["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedThreadIDsString))
            {
                ViewData["ListThreadID"] = selectedThreadIDsString;
            }
            else
            {
                ViewData["StatusMessageType"] = (int)StatusMessageType.Error;
                ViewData["StatusMessageContent"] = "请选中要操作的文件";
                return PartialView("Controls/File/SetSiteCategories.ascx");
            }

            List<SiteCategory> allCategories = SiteCategories.Instance(ApplicationIDs.Instance().File()).GetAll(true);
            return PartialView("Controls/File/SetSiteCategories.ascx", allCategories);
        }

        /// <summary>
        /// 更新站点类别
        /// </summary>
        /// <param name="selectedThreadIDsString"></param>
        /// <returns></returns>
        public ActionResult Control_UpdateSiteCategories(string selectedThreadIDsString)
        {
            string[] selectedThreadIDsStringArray = selectedThreadIDsString.Split(',');
            if (selectedThreadIDsStringArray != null && selectedThreadIDsStringArray.Length > 0)
            {
                int threadID;
                List<int> ListThreadID = new List<int>();
                foreach (string Id in selectedThreadIDsStringArray)
                {
                    try
                    {
                        threadID = int.Parse(Id);
                        ListThreadID.Add(threadID);
                    }
                    catch { }
                }
                int CategoryID = Request.Form.GetInt("toCategoryID", -1);
                if (CategoryID > 0)
                {
                    FileThreads.UpdateThreadsSiteCategory(ListThreadID, CategoryID);
                    ViewData["StatusMessageType"] = (int)StatusMessageType.Success;
                    ViewData["StatusMessageContent"] = "设置站点类别批量操作成功";
                }
                else
                {
                    ViewData["StatusMessageType"] = (int)StatusMessageType.Hint;
                    ViewData["StatusMessageContent"] = "设置站点类别操作失败，请选择一个站点类别";
                }

            }
            ViewData["ListThreadID"] = selectedThreadIDsString;
            List<SiteCategory> allCategories = SiteCategories.Instance(ApplicationIDs.Instance().File()).GetAll(true);
            return PartialView("Controls/File/SetSiteCategories.ascx", allCategories);
        }
        #endregion

        #region 标签管理
        /// <summary>
        /// 标签管理
        /// </summary>
        /// <returns></returns>
        public ActionResult ManageSiteTags()
        {
            return View("Pages/File/ManageSiteTags.aspx", "Masters/File.master");
        }
        #endregion

        /// <summary>
        /// 获取文件统计数据
        /// </summary>      
        public JsonResult GetManageableCounts()
        {
            Dictionary<FileManageableCounts, int> dictionary = FileThreads.GetManageableCounts();
            Application fileApplication = Applications.GetApplication(ApplicationIDs.Instance().File());
            List<ManageableCount> list = new List<ManageableCount>();
            list.Add(new ManageableCount { Name = "新增文件", Value = fileApplication.CountInPast24Hours });
            if (dictionary.ContainsKey(FileManageableCounts.PendingThreadCount))
                list.Add(new ManageableCount { Name = "待审核文件", Value = dictionary[FileManageableCounts.PendingThreadCount], Url = SPBUrlHelper.Action("ListManageFiles", "ManageFile", new RouteValueDictionary { { "status", AuditingStatuses.Pending } }) });
            if (dictionary.ContainsKey(FileManageableCounts.AgainThreadCount))
                list.Add(new ManageableCount { Name = "需再审核文件", Value = dictionary[FileManageableCounts.AgainThreadCount], Url = SPBUrlHelper.Action("ListManageFiles", "ManageFile", new RouteValueDictionary { { "status", AuditingStatuses.Again } }) });
            if (dictionary.ContainsKey(FileManageableCounts.PendingCommentCount))
                list.Add(new ManageableCount { Name = "待审核评论", Value = dictionary[FileManageableCounts.PendingCommentCount], Url = SPBUrlHelper.Action("ListManageComments", "ManageFile", new RouteValueDictionary { { "status", AuditingStatuses.Pending } }) });

            list.Add(new ManageableCount { Name = "总文件", Value = fileApplication.TotalCount });
            return Json(list, JsonRequestBehavior.AllowGet);
        }

    }

    #region FileGallerySubMenu

    public enum FileGallerySubMenu
    {
        /// <summary>
        /// 文件管理
        /// </summary>
        ManageFiles = 2601,

        /// <summary>
        /// 文件评论管理
        /// </summary>
        ManageFileComments = 2606,

        /// <summary>
        ///文件的站点类别 
        /// </summary>
        ManageFileCategories = 2609,
        /// <summary>
        /// 标签管理
        /// </summary>
        ManageSiteTags = 2603,
    }
    #endregion
}
