﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using SpaceBuilder.Common;

using SpaceBuilder.Utils;
using SpaceBuilder.Web;
using SpaceBuilder.Common.Mail;

namespace SpaceBuilder.File.Modules
{
    /// <summary>
    /// 处理与文件相关的动态
    /// </summary>
    public class DisposeActivityForFile : IFileModule
    {
        #region IFileModule 成员

        /// <summary>
        /// 注册相关事件处理程序
        /// </summary>
        public void Init(FileEventManager fem, System.Xml.Linq.XElement node)
        {
            fem.AfterFileThreadChange += new FileThreadEventHandler(fem_AfterFileThreadChange);
            fem.AfterFileCommentChange += new FilePostEventHandler(fem_AfterFileCommentChange);
            fem.AfterFileDownLoadRecordChange += new DownloadRecordEventHandler(fem_AfterFileDownLoadRecordChange);
        }

        void fem_AfterFileCommentChange(FileComment fileComment, GlobalEventArgs e)
        {
            if (e.State == ObjectState.Create)
            {
                User user = Users.GetUser(fileComment.UserID, false);
                if (user == null)
                    return;

                FileThread fileThread = FileThreads.GetThread(fileComment.ThreadID, false);
                if (fileThread == null)
                    return;

                bool activityIsPublic = false;

                string itemKey = ActivityItemKeys.Instance().NewFileComment();
                ActivityItem activityItem = Activities.GetActivityItem(itemKey);

                if (activityItem == null)
                    return;

                User owner = Users.GetUser(fileThread.OwnerUserID, false);
                if (owner == null)
                    return;

                if ((fileThread.PrivacyStatus == PrivacyStatuses.Public || fileThread.PrivacyStatus == PrivacyStatuses.OnlyFriend) && fileComment.IsApproved)
                    activityIsPublic = true;

                #region 处理个人动态

                UserActivity userActivity = new UserActivity();
                userActivity.ApplicationID = activityItem.ApplicationID;
                userActivity.ActivityItemID = activityItem.ItemID;
                userActivity.LeadingActorUserID = user.UserID;
                userActivity.LeadingActor = user.DisplayName;
                userActivity.Subject = string.Format(ResourceManager.GetString("Title_Pattern_Activity_NewFileComment", user.Language, ApplicationIDs.Instance().File()), SPBHtmlHelper.Instance().UserDisplayNameLink(user,true), GetFileSubjectWithHyperlink(owner, fileThread));
                userActivity.Body = HtmlUtils.TrimHtml(fileComment.Body, 250);
                userActivity.IsPublic = activityIsPublic;

                userActivity.RelativeItemID = fileComment.PostID;

                Activities.CreateUserActivity(userActivity);

                #endregion

                #region 处理朋友动态
                if (activityIsPublic)
                {
                    UserActivitySettings userActivitySettings = Activities.GetUserActivitySettings(user.UserID, false);
                    //判断用户是否允许发送给 朋友动态项目
                    if (userActivitySettings.IsAllowedInActivityItemsOfSending(itemKey))
                    {
                        FriendActivity friendActivity = new FriendActivity();
                        friendActivity.ApplicationID = activityItem.ApplicationID;
                        friendActivity.ActivityItemID = activityItem.ItemID;
                        friendActivity.LeadingActorUserID = user.UserID;
                        friendActivity.LeadingActor = user.DisplayName;
                        friendActivity.Subject = userActivity.Subject;
                        friendActivity.Body = HtmlUtils.TrimHtml(fileComment.Body, 250);

                        friendActivity.RelativeItemID = fileComment.PostID;

                        List<int> receiverUserIDs = Activities.GetFriendActivityReceiverUserIDsFromFriends(user, itemKey);
                        Activities.CreateFriendActivity(friendActivity, receiverUserIDs);
                    }
                }
                #endregion

            }
        }

        void fem_AfterFileThreadChange(FileThread fileThread, GlobalEventArgs e)
        {
            if (e.State == ObjectState.Create)
            {
                User user = Users.GetUser(fileThread.OwnerUserID, false);
                if (user == null )
                    return;

                bool activityIsPublic = false;

                string itemKey = ActivityItemKeys.Instance().NewFile();
                ActivityItem activityItem = Activities.GetActivityItem(itemKey);

                if (fileThread.PrivacyStatus == PrivacyStatuses.OnlyFriend || fileThread.PrivacyStatus == PrivacyStatuses.Public)
                    activityIsPublic = true;

                #region 处理个人动态

                UserActivity userActivity = new UserActivity();
                userActivity.ApplicationID = activityItem.ApplicationID;
                userActivity.ActivityItemID = activityItem.ItemID;
                userActivity.LeadingActorUserID = user.UserID;
                userActivity.LeadingActor = user.DisplayName;
                userActivity.Subject = string.Format(ResourceManager.GetString("Title_Pattern_Activity_NewFile", user.Language, ApplicationIDs.Instance().File()), SPBHtmlHelper.Instance().UserDisplayNameLink(user,true), GetFileSubjectWithHyperlink(user, fileThread));
                userActivity.Body = HtmlUtils.TrimHtml(fileThread.BodyForDataTransmission, 250);
                userActivity.IsPublic = activityIsPublic;

                userActivity.RelativeItemID = fileThread.ThreadID;
                userActivity.RelativeItemName = fileThread.Subject;

                Activities.CreateUserActivity(userActivity);

                #endregion

                #region 处理朋友动态
                if (activityIsPublic)
                {
                    UserActivitySettings userActivitySettings = Activities.GetUserActivitySettings(user.UserID, false);
                    //判断用户是否允许发送给 朋友动态项目
                    if (userActivitySettings.IsAllowedInActivityItemsOfSending(itemKey))
                    {
                        FriendActivity friendActivity = new FriendActivity();
                        friendActivity.ApplicationID = activityItem.ApplicationID;
                        friendActivity.ActivityItemID = activityItem.ItemID;
                        friendActivity.LeadingActorUserID = user.UserID;
                        friendActivity.LeadingActor = user.DisplayName;

                        friendActivity.Subject = userActivity.Subject;
                        friendActivity.Body = HtmlUtils.TrimHtml(fileThread.BodyForDataTransmission, 250);

                        friendActivity.RelativeItemID = fileThread.ThreadID;
                        friendActivity.RelativeItemName = fileThread.Subject;

                        List<int> receiverUserIDs = Activities.GetFriendActivityReceiverUserIDsFromFriends(user, itemKey);
                        Activities.CreateFriendActivity(friendActivity, receiverUserIDs);
                    }
                }
                #endregion
            }
        }

        void fem_AfterFileDownLoadRecordChange(FileDownLoadRecord record)
        {
            User user = Users.GetUser(record.UserID, false);
            if (user == null )
                return;

            FileThread fileThread = FileThreads.GetThread(record.ThreadID, false);
            if (fileThread == null)
                return;

            bool activityIsPublic = false;

            string itemKey = ActivityItemKeys.Instance().DownloadFile();
            ActivityItem activityItem = Activities.GetActivityItem(itemKey);

            if (fileThread.PrivacyStatus == PrivacyStatuses.OnlyFriend || fileThread.PrivacyStatus == PrivacyStatuses.Public)
                activityIsPublic = true;

            #region 处理个人动态

            UserActivity userActivity = new UserActivity();
            userActivity.ApplicationID = activityItem.ApplicationID;
            userActivity.ActivityItemID = activityItem.ItemID;
            userActivity.LeadingActorUserID = user.UserID;
            userActivity.LeadingActor = user.DisplayName;
            userActivity.Subject = string.Format(ResourceManager.GetString("Title_Pattern_Activity_DownloadFile", user.Language, ApplicationIDs.Instance().File()), SPBHtmlHelper.Instance().UserDisplayNameLink(user,true), GetFileSubjectWithHyperlink(user, fileThread));
            userActivity.IsPublic = activityIsPublic;

            userActivity.RelativeItemID = fileThread.ThreadID;
            userActivity.RelativeItemName = fileThread.Subject;

            Activities.CreateUserActivity(userActivity);

            #endregion

            #region 处理朋友动态
            if (activityIsPublic)
            {
                UserActivitySettings userActivitySettings = Activities.GetUserActivitySettings(user.UserID, false);
                //判断用户是否允许发送给 朋友动态项目
                if (userActivitySettings.IsAllowedInActivityItemsOfSending(itemKey))
                {
                    FriendActivity friendActivity = new FriendActivity();
                    friendActivity.ApplicationID = activityItem.ApplicationID;
                    friendActivity.ActivityItemID = activityItem.ItemID;
                    friendActivity.LeadingActorUserID = user.UserID;
                    friendActivity.LeadingActor = user.DisplayName;

                    friendActivity.Subject = userActivity.Subject;

                    friendActivity.RelativeItemID = fileThread.ThreadID;
                    friendActivity.RelativeItemName = fileThread.Subject;

                    List<int> receiverUserIDs = Activities.GetFriendActivityReceiverUserIDsFromFriends(user, itemKey);
                    Activities.CreateFriendActivity(friendActivity, receiverUserIDs);
                }
            }
            #endregion
        }


        #endregion

        /// <summary>
        /// 获取带链接的博客文章或评论的html代码
        /// </summary>
        private static string GetFileSubjectWithHyperlink(User author, FileThread fileThread)
        {
            return string.Format("<a {0} href=\"{1}\">{2}</a>", HyperlinkTargetOfUserStory, SiteUrls.Instance().ShowFile(fileThread.ThreadID), fileThread.Subject);
        }

        private static readonly string HyperlinkTargetOfUserStory = SiteSettingsManager.GetSiteSettings().OpenNewWindowInUserActivity ? "target=\"_blank\"" : string.Empty;
    }
}
