﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Lucene.Net.Search;
using Lucene.Net.Index;

using SpaceBuilder.File;
using Lucene.Net.QueryParsers;
using Lucene.Net.Documents;
using Lucene.Net.Analysis.Standard;
using Lucene.Net.Analysis;

using SpaceBuilder.Common;
using SpaceBuilder.Utils;

namespace SpaceBuilder.LuceneSearch
{
    /// <summary>
    /// 文件全文检索
    /// </summary>
    public class FileSearchManager : SearchManagerBase<FileThread>
    {
        private static readonly string FileIndexFileDirectory = "File";
        private static volatile FileSearchManager _self = null;
        private static readonly object lockObject = new object();

        private FileSearchManager(string IndexFileDirectory) : base(IndexFileDirectory) { }

        public static FileSearchManager Instance()
        {
            if (_self == null)
            {
                lock (lockObject)
                {
                    if (_self == null)
                    {
                        _self = new FileSearchManager(FileIndexFileDirectory);
                    }
                }
            }

            return _self;
        }

        /// <summary>
        /// 搜索
        /// </summary>
        /// <param name="query"></param>
        /// <returns></returns>
        public SearchResultDataSet<FileThread> Search(FileThreadFullTextQuery query)
        {
            //索引文件不存在时，返回null
            if (!IsIndexFilesExists)
                return new SearchResultDataSet<FileThread>();

            BooleanQuery currentQuery = new BooleanQuery();
            BooleanQuery queryForFilter = new BooleanQuery();

            if (query.SearchScopeUserID > 0)
            {
                Term userIDTerm = new Term(FileIndexFields.UserID, query.SearchScopeUserID.ToString());
                Query userIDQuery = new TermQuery(userIDTerm);
                queryForFilter.Add(userIDQuery, BooleanClause.Occur.MUST);
            }

            if (query.SiteCategoryID > 0)
            {
                if (query.IncludeSiteCategoryDescendant)
                {
                    QueryParser siteCategoryIDQueryParser = new QueryParser(CurrentLuceneVersion,FileIndexFields.SiteCategoryID, new WhitespaceAnalyzer());

                    List<SiteCategory> childSiteCategories = SiteCategories.Instance(ApplicationIDs.Instance().File()).GetAllChilds(query.SiteCategoryID,false);
                    StringBuilder siteCategoryIDs = new StringBuilder();
                    siteCategoryIDs.Append(query.SiteCategoryID.ToString());
                    if (childSiteCategories != null)
                    {
                        foreach (SiteCategory siteCategory in childSiteCategories)
                        {
                            siteCategoryIDs.Append(" " + siteCategory.CategoryID);
                        }
                    }

                    Query siteCategoryIDQuery = siteCategoryIDQueryParser.Parse(siteCategoryIDs.ToString());
                    currentQuery.Add(siteCategoryIDQuery, BooleanClause.Occur.MUST);
                }
                else
                {
                    Term siteCategoryIDTerm = new Term(FileIndexFields.SiteCategoryID, query.SiteCategoryID.ToString());
                    Query siteCategoryIDQuery = new TermQuery(siteCategoryIDTerm);
                    currentQuery.Add(siteCategoryIDQuery, BooleanClause.Occur.MUST);
                }
            }

            if (query.IsPublicFilter)
            {
                Term isPublicTerm = new Term(FileIndexFields.PrivacyStatus, ((int)PrivacyStatuses.Public).ToString());
                Query isPublicQuery = new TermQuery(isPublicTerm);
                queryForFilter.Add(isPublicQuery, BooleanClause.Occur.MUST);
            }
            
            if (!string.IsNullOrEmpty(query.Keyword))
            {
                Query postKeywordQuery = null;
                query.Keyword = StringUtilsForLucene.LuceneKeywordsScrubber(query.Keyword.ToLower());
                string keywordSegments = SegmentForQueryParser(query.Keyword);

                if (!string.IsNullOrEmpty(query.Keyword))
                {
                    string[] searchFieldsForKeyword = new string[4];
                    searchFieldsForKeyword[0] = FileIndexFields.Subject;
                    searchFieldsForKeyword[1] = FileIndexFields.Body;
                    searchFieldsForKeyword[2] = FileIndexFields.Tags;
                    searchFieldsForKeyword[3] = FileIndexFields.Author;

                    MultiFieldQueryParser keywordParser = new MultiFieldQueryParser(CurrentLuceneVersion, searchFieldsForKeyword, GetChineseAnalyzerOfUnTokenized());
                    keywordParser.SetLowercaseExpandedTerms(true);
                    keywordParser.SetDefaultOperator(QueryParser.OR_OPERATOR);
                    postKeywordQuery = keywordParser.Parse(keywordSegments);
                    currentQuery.Add(postKeywordQuery, BooleanClause.Occur.MUST);
                }
            }

            if (!string.IsNullOrEmpty(query.TagName))
            {
                string[] tagSegments = SegmentForPhraseQuery(query.TagName);
                if (tagSegments != null && tagSegments.Length > 0)
                {
                    PhraseQuery tagQuery = new PhraseQuery();
                    foreach (var tagSegment in tagSegments)
                        tagQuery.Add(new Term(FileIndexFields.Tags, tagSegment));

                    tagQuery.SetBoost((float)Math.Pow(3, 5));
                    currentQuery.Add(tagQuery, BooleanClause.Occur.MUST);
                }
            }

            SortField[] sortFields;
            switch (query.SortBy)
            {
                case FullTextQueryFileThreadSortBy.DateCreated:
                    sortFields = new SortField[] { new SortField(FileIndexFields.ThreadID, SortField.INT, true) };
                    break;
                default:
                    sortFields = new SortField[] { SortField.FIELD_SCORE };
                    break;
            }

            Filter filter = null;
            if (queryForFilter.Clauses().Count > 0)
                filter = new QueryWrapperFilter(queryForFilter);

            SearchResultDataSet<FileThread> pds = Search(currentQuery, filter,sortFields, query.PageIndex, query.PageSize);
            foreach (var item in pds.Records)
            {
                item.Subject = HighlighterForKeyWord(item.Subject, query.Keyword);
                item.Summary = HighlighterForKeyWord(item.Summary, query.Keyword);
            }
            return pds;
        }

        /// <summary>
        /// 初始化索引
        /// </summary>
        public override void InitializeIndex(string indexPath)
        {
            if (!System.IO.Directory.Exists(indexPath))
            {
                try
                {
                    System.IO.Directory.CreateDirectory(indexPath);
                }
                catch
                {
                    throw new ApplicationException(string.Format("create Directory '{0}' failed", PhysicalIndexDirectory));
                }
            }

            #region 索引文件
            int indexPageSize = 2000;
            bool createIndexFile = true;
            PagingDataSet<FileThread> pds = FileThreads.GetThreadsForAdmin(indexPageSize, 1, null, -1, string.Empty);
            double tIndex = Convert.ToDouble(pds.TotalRecords) / Convert.ToDouble(indexPageSize);
            int indexPageIndex = (int)Math.Ceiling(tIndex);

            if (pds.TotalRecords > 0)
            {
                //分多次进行索引
                for (int i = 1; i <= indexPageIndex; i++)
                {
                    if (i != 1)
                        pds = FileThreads.GetThreadsForAdmin(indexPageSize, i, null, -1, string.Empty);
                    Insert(pds.Records, indexPath, createIndexFile);
                    if (createIndexFile)
                        createIndexFile = false;
                }
            }

            #endregion
        }

        /// <summary>
        /// Document转化成FileThread
        /// </summary>
        protected override FileThread ConvertDocumentToObj(Document doc)
        {
            FileThread fileThread = new FileThread();
            int userID;
            int.TryParse(doc.Get(FileIndexFields.UserID), out userID);
            fileThread.OwnerUserID = userID;

            fileThread.Author = doc.Get(FileIndexFields.Author);

            int threadID;
            int.TryParse(doc.Get(FileIndexFields.ThreadID), out threadID);
            fileThread.ThreadID = threadID;

            fileThread.Subject = doc.Get(FileIndexFields.Subject);
            fileThread.Summary = doc.Get(FileIndexFields.Body);

            long contentSize;
            long.TryParse(doc.Get(FileIndexFields.ContentSize), out contentSize);
            fileThread.ContentSize = contentSize;

            int userCategoryID;
            int.TryParse(doc.Get(FileIndexFields.UserCategoryID), out userCategoryID);
            fileThread.UserCategoryID = userCategoryID;

            int siteCategoryID;
            int.TryParse(doc.Get(FileIndexFields.SiteCategoryID), out siteCategoryID);
            fileThread.SiteCategoryID = siteCategoryID;

            int privacyStatusInt;
            int.TryParse(doc.Get(FileIndexFields.PrivacyStatus), out privacyStatusInt);
            fileThread.PrivacyStatus = (PrivacyStatuses)privacyStatusInt;

            int auditingStatusValue = (int)AuditingStatuses.Success;
            int.TryParse(doc.Get(FileIndexFields.AuditingStatus), out auditingStatusValue);
            fileThread.AuditingStatus = (AuditingStatuses)auditingStatusValue;

            try
            {
                fileThread.PostDate = DateTools.StringToDate(doc.Get(FileIndexFields.PostDate));
            }
            catch { }

            return fileThread;
        }

        /// <summary>
        /// FileThread 转化成Document
        /// </summary>
        protected override Document ConvertObjToDocument(FileThread thread)
        {
            if (thread == null)
                return null;

            //待审核及未通过审核不允许加入索引
            if (thread.AuditingStatus == AuditingStatuses.Pending || thread.AuditingStatus == AuditingStatuses.Fail)
                return null; 

            Document doc = new Document();
            Field field;

            field = new Field(FileIndexFields.ContentSize, thread.ContentSize.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(FileIndexFields.UserID, thread.OwnerUserID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(FileIndexFields.ThreadID, thread.ThreadID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(FileIndexFields.Author, thread.Author.ToLower(), Field.Store.YES, Field.Index.NOT_ANALYZED);            
            doc.Add(field);

            field = new Field(FileIndexFields.Subject, thread.Subject.ToLower(), Field.Store.YES, Field.Index.ANALYZED, Field.TermVector.YES);
            field.SetBoost(2.0F);
            doc.Add(field);

            field = new Field(FileIndexFields.Body, HtmlUtils.StripAllTags(thread.GetBody(false).ToLower()), Field.Store.YES, Field.Index.ANALYZED, Field.TermVector.YES);
            doc.Add(field);

            field = new Field(FileIndexFields.UserCategoryID, thread.UserCategoryID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(FileIndexFields.SiteCategoryID, thread.SiteCategoryID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            if (thread.UserTags != null)
            {
                foreach (var tag in thread.UserTags)
                {
                    field = new Field(FileIndexFields.Tags, tag.ToLower(), Field.Store.YES, Field.Index.ANALYZED);
                    field.SetBoost(2.0F);
                    doc.Add(field);
                }
            }

            field = new Field(FileIndexFields.PrivacyStatus, ((int)thread.PrivacyStatus).ToString().ToLower(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(FileIndexFields.AuditingStatus, ((int)thread.AuditingStatus).ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(FileIndexFields.PostDate, DateTools.DateToString(thread.PostDate, DateTools.Resolution.DAY), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            return doc;
        }
    }
}
