//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;

using System.Collections;
using System.IO;
using System.Linq;

using SpaceBuilder.Caching;
using SpaceBuilder.Common;
using SpaceBuilder.Utils;


namespace SpaceBuilder.File
{
    /// <summary>
    /// ļ߼
    /// </summary>
    public static class FileComments
    {

        #region Create/Update/Delete

        /// <summary>
        /// ļʵ
        /// </summary>
        /// <param name="comment">Ҫļʵ</param>
        /// <returns>ļʵ</returns>
        public static FileComment CreateComment(FileComment comment)
        {
            UserPointDirection? direction = null;

            #region AuditingStatus

            if (Auditings.NeedAuditing(comment.UserID, AuditingItemKeys.Instance().FileComment(), AuditingStrictDegrees.Create))
                comment.AuditingStatus = AuditingStatuses.Pending;
            else
            {
                comment.AuditingStatus = AuditingStatuses.Success;
                direction = UserPointDirection.Input;
            }

            #endregion

            FileEvents.BeforeFilePostChange(comment, ObjectState.Create);

            FileDataProvider.Instance().CreateComment(comment);

            #region »

            //״̬ǷԶʾ
            if (Auditings.IsInAllowScope(comment.AuditingStatus, ApplicationIDs.Instance().File()))
            {
                string cacheKeyOfFileCommentsOfThread = GetCacheKey_FileCommentsOfThread(comment.ThreadID);
                BlockPagingID blockPagingID = CacheRepository.Get(cacheKeyOfFileCommentsOfThread, true) as BlockPagingID;
                if (blockPagingID != null)
                {
                    blockPagingID.InsertIntoFirst(comment.PostID);
                    CacheRepository.AddOrReplace(cacheKeyOfFileCommentsOfThread, blockPagingID, CachingExpirationTypes.ObjectCollection);
                }
            }
            CacheRepository.Add(GetCacheKey_SingleFileComment(comment.PostID), comment, CachingExpirationTypes.SingleObject);
            #endregion

            FileEvents.AfterFilePostChange(comment, ObjectState.Create);

            //
            if (direction.HasValue)
                FileEvents.AfterPointChangeOfFileComment(comment, ObjectState.Create, direction.Value);

            return comment;
        }

        /// <summary>
        /// ׼ļظ
        /// </summary>
        /// <param name="ownerUserID">ļUserID</param>
        /// <param name="postID">׼ļID</param>
        /// <param name="isApproved">Ƿ׼</param>
        public static void ApproveComments(int ownerUserID, List<int> postIDs, bool isApproved)
        {
            FileDataProvider.Instance().ApproveComments(postIDs, isApproved);

            #region »
            foreach (int postID in postIDs)
            {
                FileComment fileComment = GetComment(postID, null, true);
                if (fileComment != null)
                {
                    fileComment.IsApproved = isApproved;
                    CacheRepository.AddOrReplace(GetCacheKey_SingleFileComment(fileComment.PostID), fileComment, CachingExpirationTypes.SingleObject);
                }
            }
            if (ownerUserID > 0)
            {
                //׼״̬
                BlockPagingID blockPagingIDForIsApproved = GetMyFileCommentsBlockID(ownerUserID, ApprovalStatuses.IsApproved);
                if (blockPagingIDForIsApproved != null)
                {
                    if (isApproved)
                        blockPagingIDForIsApproved.UnionIDs(postIDs);
                    else
                        blockPagingIDForIsApproved.ExceptIDs(postIDs);
                    CacheRepository.AddOrReplace(GetCacheKey_MyFileComments(ownerUserID, ApprovalStatuses.IsApproved), blockPagingIDForIsApproved, CachingExpirationTypes.ObjectCollection);
                }
                //δ׼״̬
                BlockPagingID blockPagingIDForNotApprove = GetMyFileCommentsBlockID(ownerUserID, ApprovalStatuses.NotApprove);
                if (blockPagingIDForNotApprove != null)
                {
                    if (!isApproved)
                        blockPagingIDForNotApprove.UnionIDs(postIDs);
                    else
                        blockPagingIDForNotApprove.ExceptIDs(postIDs);

                    CacheRepository.AddOrReplace(GetCacheKey_MyFileComments(ownerUserID, ApprovalStatuses.NotApprove), blockPagingIDForNotApprove, CachingExpirationTypes.ObjectCollection);
                }
            }
            #endregion

        }

        public static void ApproveComment(int ownerUserID, int postID, bool isApproved)
        {
            List<int> postIDs = new List<int>();
            postIDs.Add(postID);
            ApproveComments(ownerUserID, postIDs, isApproved);
        }


        /// <summary>
        /// ׼(̨ûʹ)
        /// </summary>
        public static void ApproveCommentForAdmin(List<int> postIDs, bool isApproved)
        {
            FileDataProvider.Instance().ApproveComments(postIDs, isApproved);
        }

        /// <summary>
        /// ļ״̬
        /// </summary>
        public static void UpdateAuditingStatus(List<int> postIDs, AuditingStatuses auditingStatus)
        {
            FileDataProvider.Instance().UpdateCommentAuditingStatus(postIDs, auditingStatus);
            #region »
            FileComment comment = null;
            foreach (int postID in postIDs)
            {
                comment = GetComment(postID, null, true);
                if (comment != null)
                {
                    //
                    if (comment.AuditingStatus == AuditingStatuses.Again || comment.AuditingStatus == AuditingStatuses.Success)
                        FileEvents.AfterPointChangeOfFileComment(comment, ObjectState.Delete, UserPointDirection.Output);

                    //ۻ
                    comment.AuditingStatus = auditingStatus;
                    CacheRepository.AddOrReplace(GetCacheKey_SingleFileComment(comment.PostID), comment, CachingExpirationTypes.SingleObject);
                }
            }
            #endregion
        }

        /// <summary>
        /// ɾļ
        /// </summary>
        /// <param name="postID">Ҫɾļ</param>
        public static void DeleteComment(int postID)
        {
            FileComment comment = GetComment(postID, null, true);
            if (comment != null)
            {
                FileEvents.BeforeFilePostChange(comment, ObjectState.Delete);
                FileDataProvider.Instance().DeleteComment(postID);
                FileEvents.AfterFilePostChange(comment, ObjectState.Delete);

                //
                if (comment.AuditingStatus == AuditingStatuses.Again || comment.AuditingStatus == AuditingStatuses.Success)
                    FileEvents.AfterPointChangeOfFileComment(comment, ObjectState.Delete, UserPointDirection.Output);

                #region 
                comment.IsDeletedInDataBase = true;
                CacheRepository.MarkDeletion(GetCacheKey_SingleFileComment(comment.PostID), comment, CachingExpirationTypes.SingleObject);
                #endregion
            }
        }

        /// <summary>
        /// ɾļ
        /// </summary>
        /// <param name="ownerUserID">ļUserID</param>
        /// <param name="postIDs">ļID</param>
        public static void DeleteComments(List<int> postIDs)
        {
            foreach (int postID in postIDs)
            {
                DeleteComment(postID);
            }
        }

        #endregion


        #region Get & Gets


        /// <summary>
        /// PostIDȡļ
        /// </summary>
        /// <param name="isPublicFilter">Ƿֻȡ</param>
        public static FileComment GetComment(int postID, bool? isApproved, bool isRealTime)
        {
            string cacheKey = GetCacheKey_SingleFileComment(postID);
            FileComment comment = CacheRepository.Get(cacheKey, isRealTime) as FileComment;

            if (comment == null)
            {
                comment = FileDataProvider.Instance().GetFileComment(postID);
                CacheRepository.Add(cacheKey, comment, CachingExpirationTypes.SingleObject);
            }

            if (comment == null || comment.IsDeletedInDataBase)
                return null;

            if (isApproved.HasValue && (comment.IsApproved != isApproved.Value))
                return null;
            else
                return comment;
        }

        /// <summary>
        /// ļ۲ѯȡļʵ弯
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="pageIndex"></param>
        /// <returns>PagingDataSet&lt;FileComment&gt;ڷҳʾ</returns>
        public static PagingDataSet<FileComment> GetCommentsOfThread(int threadID, int pageIndex, bool entityIsRealTime)
        {
            int pageSize = FileSectionConfiguration.Instance().CommentsPerPage;
            int cacheablePageCount = FileSectionConfiguration.Instance().CacheablePageCount;

            List<int> postIDs;
            BlockPagingID blockPagingID = null;
            if (pageIndex <= cacheablePageCount)
            {
                string cacheKey = GetCacheKey_FileCommentsOfThread(threadID);
                blockPagingID = CacheRepository.Get(cacheKey, true) as BlockPagingID;
                if (blockPagingID == null)
                {
                    blockPagingID = FileDataProvider.Instance().GetFileCommentIDsOfThread(Applications.GetApplication(ApplicationIDs.Instance().File()).AuditingStatusForPublicDisplay, FileSectionConfiguration.Instance().PrimaryMaxRecords, pageSize * cacheablePageCount, 1, threadID);
                    CacheRepository.Add(cacheKey, blockPagingID, CachingExpirationTypes.ObjectCollection);
                }
                postIDs = blockPagingID.GetPagingIDs(pageSize, pageIndex);
            }
            else
            {
                blockPagingID = FileDataProvider.Instance().GetFileCommentIDsOfThread(Applications.GetApplication(ApplicationIDs.Instance().File()).AuditingStatusForPublicDisplay, FileSectionConfiguration.Instance().PrimaryMaxRecords, pageSize, pageIndex, threadID);
                postIDs = blockPagingID.GetAll();
            }

            IList<FileComment> fileComments = PopulateFileCommentsFromIDs(postIDs, entityIsRealTime);
            PagingDataSet<FileComment> pagingFileComments = new PagingDataSet<FileComment>(fileComments);
            pagingFileComments.PageIndex = pageIndex;
            pagingFileComments.PageSize = pageSize;
            pagingFileComments.TotalRecords = blockPagingID.TotalRecords;
            return pagingFileComments;
        }


        /// <summary>
        /// ȡļʵ弯
        /// </summary>
        /// <remarks>
        /// Ҫʱ
        /// </remarks>
        /// <returns></returns>
        public static PagingDataSet<FileComment> GetMyFileComments(int ownerUserID, ApprovalStatuses? approvalStatus, int pageIndex)
        {
            int cacheablePageCount = FileSectionConfiguration.Instance().CacheablePageCount;
            int pageSize = FileSectionConfiguration.Instance().CommentsPerPage;
            List<int> commentIDs;
            BlockPagingID blockPagingID = null;
            if (pageIndex <= cacheablePageCount)
            {
                blockPagingID = GetMyFileCommentsBlockID(ownerUserID, approvalStatus);
                commentIDs = blockPagingID.GetPagingIDs(pageSize, pageIndex);
            }
            else
            {
                blockPagingID = FileDataProvider.Instance().GetFileCommentIDs(ownerUserID, approvalStatus, Applications.GetApplication(ApplicationIDs.Instance().File()).AuditingStatusForPublicDisplay, true, FileSectionConfiguration.Instance().PrimaryMaxRecords, pageSize, pageIndex);
                commentIDs = blockPagingID.GetAll();
            }

            IList<FileComment> fileComments = PopulateFileCommentsFromIDs(commentIDs, true);
            PagingDataSet<FileComment> pagingFileComments = new PagingDataSet<FileComment>(fileComments);
            pagingFileComments.PageIndex = pageIndex;
            pagingFileComments.PageSize = pageSize;
            pagingFileComments.TotalRecords = blockPagingID.TotalRecords;
            return pagingFileComments;
        }


        /// <summary>
        /// ļ۲ѯȡļʵ弯(Աں̨ʹ)
        /// </summary>
        /// <param name="pageSize"></param>
        /// <param name="pageIndex"></param>
        /// <returns>PagingDataSet&lt;FilePost&gt;ڷҳʾ</returns>
        public static PagingDataSet<FileComment> GetCommentsForAdmin(int pageSize, int pageIndex, AuditingStatusesForDisplay? auditingStatusForDisplay, int? userID, string bodyKeywords)
        {
            //if (string.IsNullOrEmpty(bodyKeywords))
            //    bodyKeywords = StringUtils.StripSQLInjection(bodyKeywords);

            BlockPagingID blockPagingID = FileDataProvider.Instance().GetFileCommentIDsForAdmin(FileSectionConfiguration.Instance().PrimaryMaxRecords, pageSize, pageIndex, auditingStatusForDisplay, userID, bodyKeywords);

            IList<FileComment> fileComments = PopulateFileCommentsFromIDs(blockPagingID.GetAll(), true);
            PagingDataSet<FileComment> pagingFileComments = new PagingDataSet<FileComment>(fileComments);
            pagingFileComments.PageIndex = pageIndex;
            pagingFileComments.PageSize = pageSize;
            pagingFileComments.TotalRecords = blockPagingID.TotalRecords;
            return pagingFileComments;
        }

        /// <summary>
        /// ȡȡļID
        /// </summary>
        /// <param name="ownerUserID"></param>
        /// <param name="approvalStatus"></param>
        /// <returns></returns>
        private static BlockPagingID GetMyFileCommentsBlockID(int ownerUserID, ApprovalStatuses? approvalStatus)
        {
            int cacheablePageCount = FileSectionConfiguration.Instance().CacheablePageCount;
            int pageSize = FileSectionConfiguration.Instance().CommentsPerPage;
            string cacheKey = GetCacheKey_MyFileComments(ownerUserID, approvalStatus);
            BlockPagingID blockPagingID = CacheRepository.Get(cacheKey, true) as BlockPagingID;
            if (blockPagingID == null)
            {
                blockPagingID = FileDataProvider.Instance().GetFileCommentIDs(ownerUserID, approvalStatus, Applications.GetApplication(ApplicationIDs.Instance().File()).AuditingStatusForPublicDisplay, true, FileSectionConfiguration.Instance().PrimaryMaxRecords, pageSize * cacheablePageCount, 1);
                CacheRepository.Add(cacheKey, blockPagingID, CachingExpirationTypes.ObjectCollection);
            }
            return blockPagingID;
        }

        #endregion


        #region HelpMethds


        /// <summary>
        /// ͨFileCommentIDIList<FileThread>
        /// </summary>
        /// <param name="fileCommentIDs"></param>
        /// <returns></returns>
        private static IList<FileComment> PopulateFileCommentsFromIDs(List<int> fileCommentIDs, bool isRealTime)
        {
            IList<FileComment> fileComments = new List<FileComment>();

            foreach (int fileThreadID in fileCommentIDs)
            {
                FileComment fileComment = GetComment(fileThreadID, null, isRealTime);
                if (fileComment != null)
                    fileComments.Add(fileComment);
            }
            return fileComments;
        }

        /// <summary>
        /// ȡcommentCacheKey
        /// </summary>
        /// <param name="postID"></param>
        /// <returns></returns>
        private static string GetCacheKey_SingleFileComment(int postID)
        {
            return "FileComment:" + postID;
        }
        /// <summary>
        /// ȡļcommentбCacheKey
        /// </summary>
        /// <param name="threadID"></param>
        /// <returns></returns>
        private static string GetCacheKey_FileCommentsOfThread(int threadID)
        {
            return "FileCommentsOfThread:" + threadID;
        }

        /// <summary>
        /// ȡҵļcommentбCacheKey
        /// </summary>
        /// <param name="threadID"></param>
        /// <returns></returns>
        private static string GetCacheKey_MyFileComments(int ownerUserID, ApprovalStatuses? approvalStatus)
        {
            return CacheKeyGenerator.Generate("MyFileComments", new Dictionary<string, string> { { "O", ownerUserID.ToString() }, { "A", approvalStatus.ToString() } });
        }
        #endregion

    }
}

