﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using SpaceBuilder.Common;
using System.IO;
using SpaceBuilder.Utils;
using SpaceBuilder.Caching;

namespace SpaceBuilder.File
{
    /// <summary>
    /// 文件附件 业务逻辑类
    /// </summary>
    public class FileFeaturedImageManager : AttachmentManager
    {
        #region Instance
        private static volatile FileFeaturedImageManager _instance = null;
        private static readonly object lockObject = new object();

        public static FileFeaturedImageManager Instance()
        {
            if (_instance == null)
            {
                lock (lockObject)
                {
                    if (_instance == null)
                    {
                        _instance = new FileFeaturedImageManager();
                    }
                }
            }

            return _instance;
        }

        private FileFeaturedImageManager()
        {
        }
        #endregion

        /// <remarks>
        /// 数据提供者
        /// </remarks>
        public override AttachmentDataProvider DataProvider
        {
            get { return FileFeaturedImageDataProvider.Instance(); }
        }

        /// <summary>
        /// 文件存储ProviderName
        /// </summary>
        public override string StorageProviderName
        {
            get { return "FileFeaturedImageStorage"; }
        }

        public override AttachmentSettings Settings
        {
            get { return FileSectionConfiguration.Instance().FeaturedImageSettings; }
        }

        /// <remarks>
        /// CacheKey
        /// </remarks>
        public override string CacheKeyPrefix
        {
            get { return "FileFeaturedImage"; }
        }
        
        public override bool RequireDataBaseStorage
        {
            get { return true; }
        }

        /// <summary>
        /// 创建附件
        /// </summary>
        /// <param name="attachment"></param>
        /// <param name="contentStream"></param>
        /// <param name="enableWatermark"></param>
        public override void CreateAttachment(Attachment attachment, Stream contentStream)
        {
            base.CreateAttachment(attachment, contentStream);

            if (attachment.IsImage)
            {
                //预生成 标题图
                GetResizedPhoto(attachment, FileFeaturedImageTypes.Title);
            }
        }

        /// <summary>
        /// 获取直连URL
        /// </summary>
        /// <param name="attachment"></param>
        /// <param name="imageType"></param>
        /// <returns></returns>
        public string GetDirectlyUrl(Attachment attachment, FileFeaturedImageTypes imageType)
        {
            FileStorageProvider fileStorageProvider = FileStorageProvider.Instance(this.StorageProviderName);
            string fileName;
            if (imageType == FileFeaturedImageTypes.Original)
            {
                fileName = attachment.FileName;
            }
            else
            {
                ImageSettings imageSettings = null;
                FileSectionConfiguration fileConfiguration = FileSectionConfiguration.Instance();
                switch (imageType)
                {
                    case FileFeaturedImageTypes.Title:
                        imageSettings = fileConfiguration.TitleImageSettings;
                        break;
                    case FileFeaturedImageTypes.Slide:
                        imageSettings = fileConfiguration.SlideImageSettings;
                        break;
                }
                fileName = GetResizedFilename(attachment.FileName, imageSettings);
            }

            return fileStorageProvider.GetDirectlyUrl(attachment.GetRelativePath(), fileName, attachment.DateCreated);
        }


        #region Get Resized Photo

        /// <summary>
        /// 获取不同尺寸的图片(如果不存在则创建)
        /// </summary>
        /// <param name="photo">图片</param>
        /// <param name="GalleryImageType">图片类型</param>
        public IFile GetResizedPhoto(Attachment photo, FileFeaturedImageTypes imageType)
        {
            if (imageType == FileFeaturedImageTypes.Original)
                return photo.File;

            ImageSettings imageSettings = null;
            FileSectionConfiguration blogConfiguration = FileSectionConfiguration.Instance();
            switch (imageType)
            {
                case FileFeaturedImageTypes.Title:
                    imageSettings = blogConfiguration.TitleImageSettings;
                    break;
                case FileFeaturedImageTypes.Slide:
                    imageSettings = blogConfiguration.SlideImageSettings;
                    break;
            }

            return GetResizedPhoto(photo, imageSettings);
        }


        /// <summary>
        /// 获取缩放后的图片(如果不存在则创建)
        /// </summary>
        /// <param name="photo">图片</param>
        /// <param name="imageSettings">图片设置</param>
        private IFile GetResizedPhoto(Attachment attachment, ImageSettings imageSettings)
        {
            if (attachment == null || !attachment.IsImage)
                return null;

            if (imageSettings == null)
                return null;

            string relativePath = attachment.GetRelativePath();
            string filename = GetResizedFilename(attachment.FileName, imageSettings);

            IFile file = FileStorageProvider.Instance(StorageProviderName).GetFile(relativePath, filename);
            if (file == null)
            {
                using (Stream originalStream = attachment.OpenReadOriginalStream())
                {
                    if (originalStream != null)
                    {
                        string fileExtension;
                        Stream resizedStream = ImageUtils.HandleImage(originalStream, imageSettings, out fileExtension);
                        if (resizedStream != null)
                        {
                            file = FileStorageProvider.Instance(StorageProviderName).AddOrUpdateFile(relativePath, filename, resizedStream);
                            resizedStream.Close();
                            resizedStream.Dispose();
                        }
                        originalStream.Close();
                    }
                }
            }
            return file;
        }

        /// <summary>
        /// 获取缩放后的图片文件名称
        /// </summary>
        /// <param name="filename">上传原始附件文件名称</param>
        /// <param name="imageSettings">图片设置</param>
        private string GetResizedFilename(string filename, ImageSettings imageSettings)
        {
            return string.Format("{0}-{1}x{2}{3}", filename, imageSettings.Width, imageSettings.Height, Path.GetExtension(filename));
        }

        #endregion

    }
}
