﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights  reserved.
// </copyright> 
//------------------------------------------------------------------------------ 


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Utils;
using System.IO;
using SpaceBuilder.Common;

using SpaceBuilder.Caching;
using System.Text.RegularExpressions;


namespace SpaceBuilder.File
{
    /// <summary>
    /// FileThread 业务逻辑类
    /// </summary>
    public class FileThreads
    {

        #region Create/Update/Delete

        /// <summary>
        /// 创建文件
        /// </summary>
        public static void CreateThread(FileThread fileThread, IList<UserTag> userTags, Stream contentStream)
        {
            UserPointDirection? direction = null;

            #region 设置AuditingStatus

            if (Auditings.NeedAuditing(fileThread.OwnerUserID, AuditingItemKeys.Instance().FileThread(), AuditingStrictDegrees.Create))
                fileThread.AuditingStatus = AuditingStatuses.Pending;
            else
            {
                fileThread.AuditingStatus = AuditingStatuses.Success;
                direction = UserPointDirection.Input;
            }
            #endregion

            //附件重命名 在持久化到数据库和文件系统之前
            fileThread.FileName = fileThread.Attachment.FileName;

            FileEvents.BeforeFileThreadChange(fileThread, ObjectState.Create);

            FileDataProvider.Instance().CreateThread(fileThread);

            if (fileThread.ThreadID > 0)
            {
                //TODO:暂时 ---- 为了确保下面的CreateAttachment没有问题！
                fileThread.Attachment.AssociateID = fileThread.ThreadID;

                FileAttachmentManager.Instance().CreateAttachment(fileThread.Attachment, contentStream);

                if (contentStream != null)
                {
                    contentStream.Close();
                    contentStream.Dispose();
                }

                #region 更新缓存


                //更新类别列表和全部列表缓存
                KeyValuePair<string, List<string>> cacheKeyItem1 = new KeyValuePair<string, List<string>>("U", new List<string>() { fileThread.OwnerUserID.ToString() });
                KeyValuePair<string, List<string>> cacheKeyItem2 = new KeyValuePair<string, List<string>>("T", new List<string>() { string.Empty });
                KeyValuePair<string, List<string>> cacheKeyItem3 = new KeyValuePair<string, List<string>>("C", new List<string>() { fileThread.UserCategoryID.ToString(), string.Empty });
                List<string> myFileThreadIDsCacheKeys = CacheKeyGenerator.Generate("MyFileThreadIDs", cacheKeyItem1, cacheKeyItem2, cacheKeyItem3);


                //更新 拥有者标签相关列表
                if (userTags != null && userTags.Count > 0)
                {

                    cacheKeyItem2 = new KeyValuePair<string, List<string>>("T", userTags.Select(n => n.TagName).ToList());
                    cacheKeyItem3 = new KeyValuePair<string, List<string>>("C", new List<string>() { string.Empty });
                    List<string> myFileThreadIDsCacheKeys_UserTag = CacheKeyGenerator.Generate("MyFileThreadIDs", cacheKeyItem1, cacheKeyItem2, cacheKeyItem3);

                    myFileThreadIDsCacheKeys.AddRange(myFileThreadIDsCacheKeys_UserTag);
                }

                foreach (var myFileThreadIDsCacheKey in myFileThreadIDsCacheKeys)
                {
                    BlockPagingID blockPagingID = CacheRepository.Get(myFileThreadIDsCacheKey, true) as BlockPagingID;
                    if (blockPagingID != null)
                    {
                        blockPagingID.InsertIntoFirst(fileThread.ThreadID);
                        CacheRepository.AddOrReplace(myFileThreadIDsCacheKey, blockPagingID, CachingExpirationTypes.ObjectCollection);
                    }
                }
                //将单个实体加入缓存
                CacheRepository.Add(GetCacheKey_SingleFileThread(fileThread.ThreadID), fileThread, CachingExpirationTypes.SingleObject);

                //更新实体内容缓存
                CacheRepository.Add(GetCacheKey_FileThreadBody(fileThread.ThreadID), fileThread.BodyForDataTransmission, CachingExpirationTypes.UsualSingleObject);

                #endregion

                FileEvents.AfterFileThreadChange(fileThread, ObjectState.Create);

                //处理积分
                if (direction.HasValue)
                    FileEvents.AfterPointChangeOfFileThread(fileThread, ObjectState.Create, direction.Value);

                //重新统计FileSection
                CounterFactory.ReCount<FileSectionCounter>(fileThread.OwnerUserID);
            }
        }

        /// <overloads>更新文件</overloads>
        /// <summary>
        /// 更新文件
        /// </summary>
        /// <param name="fileThread">要更新的文件</param>
        /// <param name="postAttachmentID">文件临时ID</param>
        public static void UpdateThread(FileThread fileThread, Stream contentStream, int oldUserCategoryID, IList<string> newUserTags)
        {
            #region 更新附件
            FileAttachmentManager.Instance().CreateAttachment(fileThread.Attachment, contentStream);
            #endregion

            UpdateThread(fileThread, oldUserCategoryID, newUserTags);
        }

        /// <summary>
        /// 更新文件
        /// </summary>
        /// <param name="fileThread">需要更新的文件</param>
        public static void UpdateThread(FileThread fileThread, int oldUserCategoryID, IList<string> newUserTags)
        {

            #region 设置AuditingStatus
            if (Auditings.NeedAuditing(fileThread.OwnerUserID, AuditingItemKeys.Instance().FileThread(), AuditingStrictDegrees.Update))
                fileThread.AuditingStatus = AuditingStatuses.Again;
            #endregion

            FileEvents.BeforeFileThreadChange(fileThread, ObjectState.Update);

            //强制设置标题图
            if (fileThread.UserFeaturedImageAttachmentID <= 0 && FileSectionConfiguration.Instance().ForceSetTitleImage)
            {
                IList<Attachment> attachmentsOfThread = FileFeaturedImageManager.Instance().GetAttachmentsByAssociateID(fileThread.ThreadID, true);
                if (attachmentsOfThread != null && attachmentsOfThread.Count(n => n.IsImage) > 0)
                {
                    fileThread.UserFeaturedImageAttachmentID = attachmentsOfThread.First(n => n.IsImage).AttachmentID;
                }
            }

            // Update it in the database
            FileDataProvider.Instance().UpdateThread(fileThread);

            #region 更新缓存

            //更新标签
            List<string> oldTags = UserTagManager.Instance(UserTagItemKeys.Instance().File()).GetUserTagsOfItem(fileThread.ThreadID, true) ?? new List<string>();
            if (newUserTags == null)
                newUserTags = new List<string>();

            //取出相交的TagName
            IList<string> newTagNames = newUserTags;
            List<string> oldTagNames = oldTags;
            List<string> intersectTagNames = newTagNames.Intersect(oldTagNames).ToList();

            //去除相交的集合 剩下的就是改变的
            List<string> newTags_change = newTagNames.Except(intersectTagNames).ToList();
            List<string> oldTags_change = oldTagNames.Except(intersectTagNames).ToList();

            if (oldTags_change.Count > 0)
            {
                KeyValuePair<string, List<string>> cacheKeyItem1 = new KeyValuePair<string, List<string>>("U", new List<string>() { fileThread.OwnerUserID.ToString() });
                KeyValuePair<string, List<string>> cacheKeyItem2 = new KeyValuePair<string, List<string>>("T", oldTags_change);
                KeyValuePair<string, List<string>> cacheKeyItem3 = new KeyValuePair<string, List<string>>("C", new List<string>() { string.Empty });
                List<string> myFileThreadIDsCacheKeys = CacheKeyGenerator.Generate("MyFileThreadIDs", cacheKeyItem1, cacheKeyItem2, cacheKeyItem3);
                foreach (var myFileThreadIDsCacheKey in myFileThreadIDsCacheKeys)
                {
                    BlockPagingID blockPagingID = CacheRepository.Get(myFileThreadIDsCacheKey, true) as BlockPagingID;
                    if (blockPagingID != null)
                    {
                        blockPagingID.Remove(fileThread.ThreadID);
                        CacheRepository.AddOrReplace(myFileThreadIDsCacheKey, blockPagingID, CachingExpirationTypes.ObjectCollection);
                    }
                }
            }
            if (newTags_change.Count > 0)
            {
                KeyValuePair<string, List<string>> cacheKeyItem1 = new KeyValuePair<string, List<string>>("U", new List<string>() { fileThread.OwnerUserID.ToString() });
                KeyValuePair<string, List<string>> cacheKeyItem2 = new KeyValuePair<string, List<string>>("T", newTags_change);
                KeyValuePair<string, List<string>> cacheKeyItem3 = new KeyValuePair<string, List<string>>("C", new List<string>() { string.Empty });
                List<string> myFileThreadIDsCacheKeys = CacheKeyGenerator.Generate("MyFileThreadIDs", cacheKeyItem1, cacheKeyItem2, cacheKeyItem3);
                foreach (var myFileThreadIDsCacheKey in myFileThreadIDsCacheKeys)
                {
                    BlockPagingID blockPagingID = CacheRepository.Get(myFileThreadIDsCacheKey, true) as BlockPagingID;
                    if (blockPagingID != null)
                    {
                        blockPagingID.InsertIntoFirst(fileThread.ThreadID);
                        CacheRepository.AddOrReplace(myFileThreadIDsCacheKey, blockPagingID, CachingExpirationTypes.ObjectCollection);
                    }
                }
            }

            //更新类别缓存
            if (oldUserCategoryID != fileThread.UserCategoryID)
            {
                if (fileThread.UserCategoryID > 0)
                {
                    string newMyFileThreadIDsCacheKey = GetCacheKey_MyFileThreadIDs(fileThread.OwnerUserID, string.Empty, fileThread.UserCategoryID);

                    //更新新的类别
                    BlockPagingID blockPagingID = CacheRepository.Get(newMyFileThreadIDsCacheKey, true) as BlockPagingID;
                    if (blockPagingID != null)
                    {
                        blockPagingID.InsertIntoFirst(fileThread.ThreadID);
                        CacheRepository.AddOrReplace(newMyFileThreadIDsCacheKey, blockPagingID, CachingExpirationTypes.ObjectCollection);
                    }
                }

                if (oldUserCategoryID > 0)
                {
                    string oldMyFileThreadIDsCacheKey = GetCacheKey_MyFileThreadIDs(fileThread.OwnerUserID, string.Empty, oldUserCategoryID);
                    //更新旧的类别
                    BlockPagingID blockPagingID = CacheRepository.Get(oldMyFileThreadIDsCacheKey, true) as BlockPagingID;
                    if (blockPagingID != null)
                    {
                        blockPagingID.Remove(fileThread.ThreadID);
                        CacheRepository.AddOrReplace(oldMyFileThreadIDsCacheKey, blockPagingID, CachingExpirationTypes.ObjectCollection);
                    }
                }
            }

            //更新单个实体缓存
            CacheRepository.AddOrReplace(GetCacheKey_SingleFileThread(fileThread.ThreadID), fileThread, CachingExpirationTypes.SingleObject);

            CacheRepository.AddOrReplace(GetCacheKey_FileThreadBody(fileThread.ThreadID), fileThread.BodyForDataTransmission, CachingExpirationTypes.UsualSingleObject);

            #endregion

            FileEvents.AfterFileThreadChange(fileThread, ObjectState.Update);

        }
        #region  Top & Essential

        /// <summary>
        /// 设为精华
        /// </summary>
        public static void SetEssential(FileThread fileThread)
        {
            FileDataProvider.Instance().UpdateThread(fileThread);
            #region 更新缓存
            CacheRepository.AddOrReplace(GetCacheKey_SingleFileThread(fileThread.ThreadID), fileThread, CachingExpirationTypes.SingleObject);
            #endregion
            FileEvents.AfterFileThreadSetEssential(fileThread, ObjectState.Update);
        }

        /// <summary>
        /// 置顶
        /// </summary>
        public static void SetSticky(FileThread fileThread)
        {
            FileDataProvider.Instance().UpdateThread(fileThread);
            #region 更新缓存
            CacheRepository.AddOrReplace(GetCacheKey_SingleFileThread(fileThread.ThreadID), fileThread, CachingExpirationTypes.SingleObject);
            #endregion
            FileEvents.AfterFileThreadSetSticky(fileThread, ObjectState.Update);

        }

        /// <summary>
        /// 批量置顶/取消置顶
        /// </summary>
        public static void SetSticky(List<int> threadIDs, int specialOrder)
        {
            FileDataProvider.Instance().SetSticky(threadIDs, specialOrder);
            FileThread fileThread = null;
            foreach (int threadID in threadIDs)
            {
                fileThread = FileThreads.GetThread(threadID, null, true);
                if (fileThread != null)
                {
                    fileThread.SpecialOrder = specialOrder;
                    CacheRepository.AddOrReplace(GetCacheKey_SingleFileThread(fileThread.ThreadID), fileThread, CachingExpirationTypes.SingleObject);
                }
                FileEvents.AfterFileThreadSetSticky(fileThread, ObjectState.Update);
            }
        }

        /// <summary>
        /// 批量设置精华/取消精华
        /// </summary>
        public static void SetEssential(List<int> threadIDs, bool setEssential)
        {
            FileDataProvider.Instance().SetEssential(threadIDs, setEssential);
            FileThread fileThread = null;
            foreach (int threadID in threadIDs)
            {
                fileThread = FileThreads.GetThread(threadID, null, true);
                if (fileThread != null)
                {
                    fileThread.IsEssential = setEssential;
                    CacheRepository.AddOrReplace(GetCacheKey_SingleFileThread(fileThread.ThreadID), fileThread, CachingExpirationTypes.SingleObject);
                }
                FileEvents.AfterFileThreadSetEssential(fileThread, ObjectState.Update);
            }
        }

        #endregion

        /// <summary>
        /// 文件设置站点类别
        /// </summary>
        public static void UpdateThreadsSiteCategory(List<int> threadIDs, int siteCategoryID)
        {
            FileDataProvider.Instance().UpdateThreadsSiteCategory(threadIDs, siteCategoryID);

            foreach (int threadID in threadIDs)
            {
                FileThread fileThread = GetThread(threadID, null, true);
                if (fileThread != null)
                {
                    fileThread.SiteCategoryID = siteCategoryID;

                    ////执行文件更新事件
                    //FileEvents.AfterFileThreadChange(fileThread, ObjectState.Update);

                    //更新缓存
                    CacheRepository.AddOrReplace(GetCacheKey_SingleFileThread(fileThread.ThreadID), fileThread, CachingExpirationTypes.SingleObject);
                }


            }
        }

        /// <summary>
        /// 批量设置用户所属分类
        /// </summary>
        public static void UpdateThreadsUserCategory(List<int> threadIDs, int newUserCategoryID, int ownerUserID)
        {
            if (threadIDs != null && threadIDs.Count > 0)
            {
                FileDataProvider.Instance().UpdateThreadsUserCategory(threadIDs, newUserCategoryID);

                #region 更新缓存

                string newMyFileThreadIDsCacheKey = GetCacheKey_MyFileThreadIDs(ownerUserID, string.Empty, newUserCategoryID);
                BlockPagingID newBlockPagingID = CacheRepository.Get(newMyFileThreadIDsCacheKey, true) as BlockPagingID;

                foreach (int threadID in threadIDs)
                {
                    FileThread fileThread = GetThread(threadID, null, true);
                    if (fileThread != null)
                    {
                        //如果新的类别和旧的类别相同 不做处理
                        if (fileThread.UserCategoryID == newUserCategoryID)
                            continue;

                        if (newUserCategoryID > 0 && newBlockPagingID != null)
                            newBlockPagingID.InsertIntoFirst(fileThread.ThreadID);

                        if (fileThread.UserCategoryID > 0)
                        {
                            //将id从旧的列表移除
                            string oldMyFileThreadIDsCacheKey = GetCacheKey_MyFileThreadIDs(ownerUserID, string.Empty, fileThread.UserCategoryID);
                            BlockPagingID oldBlockPagingID = CacheRepository.Get(oldMyFileThreadIDsCacheKey, true) as BlockPagingID;
                            if (oldBlockPagingID != null)
                            {
                                oldBlockPagingID.Remove(fileThread.ThreadID);
                                CacheRepository.AddOrReplace(oldMyFileThreadIDsCacheKey, oldBlockPagingID, CachingExpirationTypes.ObjectCollection);
                            }
                        }

                        fileThread.UserCategoryID = newUserCategoryID;

                        //更新缓存
                        CacheRepository.AddOrReplace(GetCacheKey_SingleFileThread(fileThread.ThreadID), fileThread, CachingExpirationTypes.SingleObject);
                    }
                }

                CacheRepository.AddOrReplace(newMyFileThreadIDsCacheKey, newBlockPagingID, CachingExpirationTypes.ObjectCollection);
                #endregion
            }

        }

        /// <summary>
        /// 删除文件
        /// </summary>
        /// <param name="fileThread">要删除的文件</param>
        public static void DeleteThread(FileThread fileThread)
        {
            //删除文件的标题图
            FileFeaturedImageManager.Instance().DeleteAttachmentsByAssociateID(fileThread.ThreadID);

            //删除文件附件
            FileAttachmentManager.Instance().DeleteAttachmentsByAssociateID(fileThread.ThreadID);

            FileEvents.BeforeFileThreadChange(fileThread, ObjectState.Delete);

            FileDataProvider.Instance().DeleteThread(fileThread.ThreadID);

            FileEvents.AfterFileThreadChange(fileThread, ObjectState.Delete);

            //删除与用户标签的关联[在存储过程spb_FileThread_Delete 中处理]
            //UserTagManager.Instance(UserTagItemKeys.Instance().File()).DeleteUserTagsOfItem(fileThread.ThreadID);

            //下载记录在存储过程中处理

            #region 更新缓存
            //标记删除
            fileThread.IsDeletedInDataBase = true;
            //更新缓存
            CacheRepository.MarkDeletion(GetCacheKey_SingleFileThread(fileThread.ThreadID), fileThread, CachingExpirationTypes.SingleObject);

            #endregion

            //处理积分
            if (fileThread.AuditingStatus == AuditingStatuses.Again || fileThread.AuditingStatus == AuditingStatuses.Success)
                FileEvents.AfterPointChangeOfFileThread(fileThread, ObjectState.Delete, UserPointDirection.Output);

            //重新统计FileSection
            CounterFactory.ReCount<FileSectionCounter>(fileThread.OwnerUserID);
        }

        /// <summary>
        /// 设置文件审核状态
        /// </summary>
        public static void UpdateAuditingStatus(List<int> threadIDs, AuditingStatuses auditingStatus)
        {
            FileDataProvider.Instance().UpdateThreadAuditingStatus(threadIDs, auditingStatus);

            #region 更新缓存
            foreach (int threadID in threadIDs)
            {
                FileThread thread = GetThread(threadID, null, true);
                if (thread != null)
                {

                    UserPointDirection? userPointDirection = Auditings.ResolvePointChangeStatus(thread.AuditingStatus, auditingStatus);
                    thread.AuditingStatus = auditingStatus;
                    CacheRepository.AddOrReplace(GetCacheKey_SingleFileThread(thread.ThreadID), thread, CachingExpirationTypes.SingleObject);

                    if (userPointDirection.HasValue)
                        FileEvents.AfterPointChangeOfFileThread(thread, ObjectState.Update, userPointDirection.Value);
                }
            }
            #endregion


        }

        /// <summary>
        /// 更新隐私设置
        /// </summary>
        public static void UpdatePrivacyStatus(List<int> threadIDs, PrivacyStatuses privacyStatus, string password, int userID)
        {
            if (privacyStatus == PrivacyStatuses.NeedPassword)
            {
                FileThread fileThread = null;
                foreach (int threadID in threadIDs)
                {
                    fileThread = FileThreads.GetThread(threadID, null, true);
                    if (fileThread != null)
                    {
                        fileThread.Password = password;
                        fileThread.PrivacyStatus = privacyStatus;
                        FileThreads.UpdateThread(fileThread, fileThread.UserCategoryID, null);
                    }
                }
            }
            else
            {
                FileDataProvider.Instance().UpdateThreadPrivacyStatus(threadIDs, privacyStatus);
                FileThread thread = null;
                foreach (int threadID in threadIDs)
                {
                    thread = GetThread(threadID, null, true);
                    if (thread != null)
                    {
                        thread.PrivacyStatus = privacyStatus;
                        CacheRepository.AddOrReplace(GetCacheKey_SingleFileThread(thread.ThreadID), thread, CachingExpirationTypes.SingleObject);
                    }
                }
            }
        }


        #endregion

        #region siteCategory

        /// <summary>
        /// 更换文件所属站点分类
        /// </summary>
        public static void UpdateFileSiteCategories(int formSiteCategoryID, int toSiteCategoryID)
        {
            FileDataProvider.Instance().UpdateFileSiteCategories(formSiteCategoryID, toSiteCategoryID);

            //未处理缓存
        }

        #endregion

        #region Get Gets
        /// <summary>
        /// 获取用户上传的前N条文件ID集合
        /// </summary>
        public static IList<int> GetTopThreadIDsOfUser(int userID, int count)
        {
            return FileDataProvider.Instance().GetFileThreadIDs(userID, string.Empty, null, null, null, true, 500, count, 1).GetAll();
        }

        public static FileThread GetThread(int threadID, bool isRealTime)
        {
            return GetThread(threadID, null, isRealTime);
        }

        /// <summary>
        /// 根据ThreadID获取文件，使用缓存
        /// </summary>
        /// <param name="ThreadID">要获取的文件的ThreadID</param>
        /// <param name="privacyStatus">是否只显示公开的文件</param>
        /// <returns>文件实体</returns>
        public static FileThread GetThread(int ThreadID, PrivacyStatuses? privacyStatus, bool isRealTime)
        {
            string cacheKey = GetCacheKey_SingleFileThread(ThreadID);
            FileThread fileThread = CacheRepository.Get(cacheKey, isRealTime) as FileThread;
            if (fileThread == null)
            {
                fileThread = FileDataProvider.Instance().GetThread(ThreadID);

                CacheRepository.Add(cacheKey, fileThread, CachingExpirationTypes.SingleObject);
            }

            if (fileThread == null || fileThread.IsDeletedInDataBase)
                return null;

            if (privacyStatus.HasValue)
            {
                if (fileThread.PrivacyStatus == privacyStatus.Value)
                    return fileThread;
                else
                    return null;
            }
            else
            {
                return fileThread;
            }
        }


        /// <summary>
        /// 获取非主流排序方式的前topNumber条文件
        /// </summary>
        /// <remarks>
        /// 主要用于频道首页
        /// </remarks>
        /// <param name="sortBy"></param>
        /// <param name="topNumber"></param>
        /// <returns></returns>
        public static IList<FileThread> GetTopThreadsSortBy(FileThreadSortBy sortBy, int? siteCategoryID, bool? isEssential, int topNumber)
        {
            BlockPagingID blockPagingID = GetThreadIDsSortBy(sortBy, siteCategoryID, string.Empty, isEssential);
            return PopulateFileThreadsFromIDs(blockPagingID.GetTopIDs(topNumber), PrivacyStatuses.Public, false);
        }

        /// <summary>
        /// 分页获取非主流排序方式的文件
        /// </summary>
        /// <remarks>
        /// 主要用于频道排行
        /// </remarks>
        /// <param name="sortBy"></param>
        /// <param name="siteCategoryID"></param>
        /// <param name="pageSize"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public static PagingDataSet<FileThread> GetPagingThreadsSortBy(FileThreadSortBy sortBy, int? siteCategoryID, string tagName, bool? isEssential, int pageSize, int pageIndex)
        {
            BlockPagingID blockPagingID = GetThreadIDsSortBy(sortBy, siteCategoryID, tagName, isEssential);
            IList<FileThread> fileTheads = PopulateFileThreadsFromIDs(blockPagingID.GetPagingIDs(pageSize, pageIndex), PrivacyStatuses.Public, false);

            PagingDataSet<FileThread> pagingFileTheads = new PagingDataSet<FileThread>(fileTheads);
            pagingFileTheads.PageIndex = pageIndex;
            pagingFileTheads.PageSize = pageSize;
            pagingFileTheads.TotalRecords = blockPagingID.Count;

            return pagingFileTheads;
        }

        /// <summary>
        /// 获取非主流排序方式的FileThreadID集合
        /// </summary>
        /// <param name="sortBy">排序方式</param>
        private static BlockPagingID GetThreadIDsSortBy(FileThreadSortBy sortBy, int? categoryID, string tagName, bool? isEssential)
        {
            string cacheKey = string.Format("FileThreadIDsSortBy::S-{0}:C-{1}:T-{2}:Es-{3}", (int)sortBy, categoryID, tagName, isEssential);
            BlockPagingID blockPagingID = CacheRepository.Get(cacheKey, false) as BlockPagingID;

            if (blockPagingID == null)
            {
                Application fileApplication = Applications.GetApplication(ApplicationIDs.Instance().File());
                blockPagingID = FileDataProvider.Instance().GetFileThreadIDsSortBy(sortBy, categoryID, tagName, isEssential, FileSectionConfiguration.Instance().SecondaryMaxRecords, PrivacyStatusesForDisplay.Public, fileApplication.AuditingStatusForPublicDisplay);
                CacheRepository.Add(cacheKey, blockPagingID, CachingExpirationTypes.UsualObjectCollection);
            }

            return blockPagingID;
        }

        /// <summary>
        /// 获取文件用户分页（自己查看时）缓存需要及时更新
        /// </summary>
        /// <param name="ownerUserID"></param>
        /// <param name="tagName"></param>
        /// <param name="userCategoryID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public static PagingDataSet<FileThread> GetMyThreads(int ownerUserID, string tagName, int? userCategoryID, int pageIndex)
        {
            int cacheablePageCount = FileSectionConfiguration.Instance().CacheablePageCount;
            int pageSize = FileSectionConfiguration.Instance().ThreadsPerPage;
            List<int> threadIDs;
            BlockPagingID blockPagingID = null;
            if (pageIndex <= cacheablePageCount)
            {
                string cacheKey = GetCacheKey_MyFileThreadIDs(ownerUserID, tagName, userCategoryID);
                blockPagingID = CacheRepository.Get(cacheKey, true) as BlockPagingID;
                if (blockPagingID == null)
                {
                    blockPagingID = FileDataProvider.Instance().GetFileThreadIDs(ownerUserID, tagName, userCategoryID, null, null, true, FileSectionConfiguration.Instance().PrimaryMaxRecords, cacheablePageCount * pageSize, 1);
                    CacheRepository.Add(cacheKey, blockPagingID, CachingExpirationTypes.ObjectCollection);
                }
                threadIDs = blockPagingID.GetPagingIDs(pageSize, pageIndex);
            }
            else
            {
                blockPagingID = FileDataProvider.Instance().GetFileThreadIDs(ownerUserID, tagName, userCategoryID, null, null, true, FileSectionConfiguration.Instance().PrimaryMaxRecords, pageSize, pageIndex);
                threadIDs = blockPagingID.GetAll();
            }

            IList<FileThread> fileTheads = PopulateFileThreadsFromIDs(threadIDs, null, true);
            PagingDataSet<FileThread> pagingfileTheads = new PagingDataSet<FileThread>(fileTheads);
            pagingfileTheads.PageIndex = pageIndex;
            pagingfileTheads.PageSize = pageSize;
            pagingfileTheads.TotalRecords = blockPagingID.TotalRecords;
            return pagingfileTheads;

        }

        /// <summary>
        ///   获取文件并用于分页(他人查看时)
        /// </summary>
        /// <param name="UserID"></param>
        /// <param name="tagName"></param>
        /// <param name="userCategoryID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public static PagingDataSet<FileThread> GetThreads(int ownerUserID, string tagName, int? userCategoryID, int pageIndex)
        {
            int cacheablePageCount = FileSectionConfiguration.Instance().CacheablePageCount;
            int pageSize = FileSectionConfiguration.Instance().ThreadsPerPage;
            AuditingStatusesForDisplay auditingStatusForDisplay;

            List<int> threadIDs;
            BlockPagingID blockPagingID = null;
            if (pageIndex <= cacheablePageCount)
            {
                string cacheKey = CacheKeyGenerator.Generate("UserFileThreadIDs", new Dictionary<string, string>() { { "U", ownerUserID.ToString() }, { "T", tagName }, { "C", userCategoryID.ToString() } });
                blockPagingID = CacheRepository.Get(cacheKey, false) as BlockPagingID;
                if (blockPagingID == null)
                {
                    auditingStatusForDisplay = Applications.GetApplication(ApplicationIDs.Instance().File()).AuditingStatusForPublicDisplay;

                    blockPagingID = FileDataProvider.Instance().GetFileThreadIDs(ownerUserID, tagName, userCategoryID, SpaceBuilder.Common.PrivacyStatusesForDisplay.NeedPassword_GreaterThanOrEqual, auditingStatusForDisplay, false, FileSectionConfiguration.Instance().PrimaryMaxRecords, cacheablePageCount * pageSize, 1);
                    CacheRepository.Add(cacheKey, blockPagingID, CachingExpirationTypes.ObjectCollection);
                }
                threadIDs = blockPagingID.GetPagingIDs(pageSize, pageIndex);
            }
            else
            {
                auditingStatusForDisplay = Applications.GetApplication(ApplicationIDs.Instance().File()).AuditingStatusForPublicDisplay;
                blockPagingID = FileDataProvider.Instance().GetFileThreadIDs(ownerUserID, tagName, userCategoryID, SpaceBuilder.Common.PrivacyStatusesForDisplay.NeedPassword_GreaterThanOrEqual, auditingStatusForDisplay, false, FileSectionConfiguration.Instance().PrimaryMaxRecords, pageSize, pageIndex);
                threadIDs = blockPagingID.GetAll();
            }

            IList<FileThread> fileTheads = PopulateFileThreadsFromIDs(threadIDs, null, false);
            PagingDataSet<FileThread> pagingfileTheads = new PagingDataSet<FileThread>(fileTheads);
            pagingfileTheads.PageIndex = pageIndex;
            pagingfileTheads.PageSize = pageSize;
            pagingfileTheads.TotalRecords = blockPagingID.TotalRecords;
            return pagingfileTheads;
        }


        /// <summary>
        /// 分页频道中标签下的文件
        /// </summary>
        /// <param name="tagName"></param>
        /// <param name="pageSize"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public static PagingDataSet<FileThread> GetPagingThreadsInSiteTag(string tagName, int pageSize, int pageIndex)
        {
            BlockPagingID blockPagingID = GetThreadIDsSortBy(FileThreadSortBy.StickyAndPostDate, null, tagName, null);
            IList<FileThread> fileTheads = PopulateFileThreadsFromIDs(blockPagingID.GetPagingIDs(pageSize, pageIndex), PrivacyStatuses.Public, false);
            PagingDataSet<FileThread> pagingFileTheads = new PagingDataSet<FileThread>(fileTheads);
            pagingFileTheads.PageIndex = pageIndex;
            pagingFileTheads.PageSize = pageSize;
            pagingFileTheads.TotalRecords = blockPagingID.TotalRecords;
            return pagingFileTheads;
        }

        /// <summary>
        /// 获取文件并用于分页（后台管理员使用）
        /// </summary>
        /// <param name="pageSize"></param>
        /// <param name="pageIndex"></param>
        /// <param name="auditingStatusForDisplay"></param>
        /// <param name="ownerUserID"></param>
        /// <param name="subjectKeywords"></param>
        /// <returns></returns>
        public static PagingDataSet<FileThread> GetThreadsForAdmin(int pageSize, int pageIndex, AuditingStatusesForDisplay? auditingStatusForDisplay, int? ownerUserID, string subjectKeywords)
        {
            BlockPagingID blockPagingID = FileDataProvider.Instance().GetFileThreadIDsForAdmin(FileSectionConfiguration.Instance().PrimaryMaxRecords, pageSize, pageIndex, auditingStatusForDisplay, ownerUserID, subjectKeywords);

            IList<FileThread> fileTheads = PopulateFileThreadsFromIDs(blockPagingID.GetAll(), null, true);
            PagingDataSet<FileThread> pagingFileTheads = new PagingDataSet<FileThread>(fileTheads);
            pagingFileTheads.PageIndex = pageIndex;
            pagingFileTheads.PageSize = pageSize;
            pagingFileTheads.TotalRecords = blockPagingID.TotalRecords;
            return pagingFileTheads;
        }

        /// <summary>
        /// 从缓存中获取BlogThread内容
        /// </summary>
        internal static string GetBody(int threadID, bool isRealTime)
        {
            string cacheKey = GetCacheKey_FileThreadBody(threadID);

            string body = CacheRepository.Get(cacheKey, isRealTime) as string;
            if (body == null)
            {
                body = FileDataProvider.Instance().GetFileThreadBody(threadID);
                CacheRepository.Add(cacheKey, body, CachingExpirationTypes.UsualSingleObject);
            }
            return body;
        }

        ///// <summary>
        ///// 从本地缓存获取文件内容
        ///// </summary>
        ///// <returns></returns>
        //internal static string GetThreadBodyFromLocalCache(int threadID)
        //{
        //    string cacheKey = GetCacheKey_FileThreadBody(threadID,false);
        //    string body = CommonCacheRepository.Get(cacheKey,false) as string;
        //    if (body == null)
        //    {
        //        body = FileDataProvider.Instance().GetFileThreadBody(threadID);
        //        CommonCacheRepository.Add(cacheKey, body, CachingExpirationTypes.UsualSingleObject);
        //    }
        //    return body;
        //}

        ///// <summary>
        ///// 通过分布式缓存获取文件内容
        ///// </summary>
        ///// <returns></returns>
        //internal static string GetThreadBodyFromDistributedCache(int threadID)
        //{
        //    if (!SPBConfig.Instance().EnableDistributedCache)
        //        return null;

        //    string cacheKey = GetCacheKey_FileThreadBody(threadID, false);
        //    string body = CommonCacheRepository.Get(cacheKey,true) as string;
        //    if (body == null)
        //    {
        //        body = FileDataProvider.Instance().GetFileThreadBody(threadID);
        //        CommonCacheRepository.Add(cacheKey, body, CachingExpirationTypes.UsualSingleObject);
        //    }
        //    return body;
        //}
        #endregion

        #region Rating

        /// <summary>
        /// 评价
        /// </summary>
        /// <param name="rating">评价对象</param>
        public static void Rate(Rating rating)
        {
            FileDataProvider.Instance().Rate(rating);

            //执行评价事件
            ServiceEvents.Rate(rating, ApplicationIDs.Instance().File());

            #region 清除缓存

            string cacheKey = GetCacheKey_FileRatings(rating.ItemID);
            List<Rating> ratings = CacheRepository.Get(cacheKey, true) as List<Rating>;
            if (ratings != null)
            {
                //评价是允许多次评价的 以最后一次评价为准 所以要先移除掉
                ratings.RemoveAll(n => n.UserID == rating.UserID && n.ItemID == rating.ItemID);
                ratings.Add(rating);
                CacheRepository.AddOrReplace(cacheKey, ratings, CachingExpirationTypes.ObjectCollection);
            }

            #endregion
        }

        /// <summary>
        /// 获取主题的所有评价
        /// </summary>
        public static List<Rating> GetRatings(int itemID, int topNumber)
        {
            string cacheKey = GetCacheKey_FileRatings(itemID);

            List<Rating> ratings = CacheRepository.Get(cacheKey, false) as List<Rating>;
            if (ratings == null)
            {
                ratings = FileDataProvider.Instance().GetRatings(itemID, topNumber);
                CacheRepository.Add(cacheKey, ratings, CachingExpirationTypes.ObjectCollection);
            }
            return ratings;
        }

        #endregion

        #region Valid

        /// <summary>
        /// 
        /// </summary>
        /// <param name="password"></param>
        /// <param name="storedPassword"></param>
        /// <returns></returns>
        public static bool ValidPrivacyPassword(string password, string storedPassword)
        {
            string passwordSalt = FileDataProvider.Instance().GenerateSalt();
            if (string.IsNullOrEmpty(storedPassword))
                return false;
            else
            {
                return FileDataProvider.Instance().CheckPassword(password, storedPassword, UserPasswordFormats.Clear, passwordSalt);
            }
        }

        #endregion


        #region DownloadRecord

        /// <summary>
        /// 创建下载记录
        /// </summary>
        public static FileDownLoadRecord CreateDownLoadRecord(User currentUser, int threadID, string fromUrl)
        {
            FileThread thread = GetThread(threadID, true);
            if (thread == null || thread.Price < 0)
                return null;
            FileDownLoadRecord record = new FileDownLoadRecord();
            record.UserID = currentUser == null ? 0 : currentUser.UserID;
            record.UserDisplayName = currentUser == null ? "匿名用户" : currentUser.DisplayName;
            record.ThreadID = threadID;
            record.Price = thread.Price;
            record.UserHostAddress = WebUtils.IPAddress;
            if (!string.IsNullOrEmpty(fromUrl))
                record.FromUrl = fromUrl;
            FileDataProvider.Instance().CreateDownLoadRecord(record);

            if (record.RecordID > 0 && record.UserID > 0)
            {
                FileEvents.AfterFileDownLoadRecordChange(record);
                #region 更新缓存
                string cachekey = GetCacheKey_TopRecordsOfThread(threadID);
                List<int> list = CacheRepository.Get(cachekey, false) as List<int>;
                if (list != null)
                    list.Insert(0, record.RecordID);
                string cacheKeyOfFavoriteItemIDFavoriteIDs = GetCacheKey_ThreadIDRecordIDsOfUser(record.UserID);
                Dictionary<int, int> dictionary = GetThreadIDRecordIDsOfUser(record.UserID);
                if (!dictionary.ContainsKey(record.ThreadID))
                    dictionary[record.ThreadID] = record.RecordID;

                CacheRepository.AddOrReplace(cacheKeyOfFavoriteItemIDFavoriteIDs, dictionary, CachingExpirationTypes.ObjectCollection);

                CacheRepository.Add(GetCacheKey_FileDownLoadRecord(record.RecordID), record, CachingExpirationTypes.SingleObject);
                #endregion
            }
            return record;
        }

        /// <summary>
        /// 根据用户ID批量删除下载记录
        /// </summary>
        public static void DeleteDownloadRecordsByUserID(int userID)
        {
            FileDataProvider.Instance().DeleteDownloadRecordsByUserID(userID);
        }

        /// <summary>
        /// 是否有权限下载需要花积分的文件
        /// </summary>
        public static bool ValidateDownloadFile(User currentUser, int threadID)
        {
            FileThread thread = FileThreads.GetThread(threadID, true);
            if (thread == null)
                return false;

            if (currentUser != null && (currentUser.IsContentAdministrator || currentUser.IsAdministrator || currentUser.UserID == thread.OwnerUserID))
                return true;

            //处理仅允许注册用户下载
            if (thread.OnlyAllowRegisteredUserDownload && currentUser == null)
                return false;

            //处理隐私状态
            if (currentUser != null && thread.PrivacyStatus == PrivacyStatuses.OnlyFriend && !Friends.IsFriend(currentUser.UserID, thread.OwnerUserID))
                return false;
            if (thread.PrivacyStatus == PrivacyStatuses.Privacy)
                return false;

            //处理售价
            if (thread.Price <= 0)
                return true;
            if (currentUser != null && FileThreads.IsDownloaded(currentUser.UserID, threadID))
                return true;
            if (currentUser != null && thread.Price > currentUser.TradePoints)
                return false;
            return true;
        }

        /// <summary>
        /// 获取FileDownLoadRecord
        /// </summary>    
        public static FileDownLoadRecord GetFileDownLoadRecord(int recordID)
        {
            string cacheKey = GetCacheKey_FileDownLoadRecord(recordID);
            FileDownLoadRecord record = CacheRepository.Get(cacheKey, false) as FileDownLoadRecord;

            if (record == null)
            {
                record = FileDataProvider.Instance().GetFileDownLoadRecord(recordID);
                CacheRepository.Add(cacheKey, record, CachingExpirationTypes.UsualSingleObject);
            }
            return record == null ? null : (record.IsDeletedInDataBase ? null : record);
        }
        /// <summary>
        /// 根据用户和文件来获取文件下载记录
        /// </summary>
        public static FileDownLoadRecord GetFileDownLoadRecord(int userID, int threadID)
        {
            Dictionary<int, int> dictionary = GetThreadIDRecordIDsOfUser(userID);
            if (dictionary != null && dictionary.ContainsKey(threadID))
            {
                return GetFileDownLoadRecord(dictionary[threadID]);
            }
            return null;
        }
        /// <summary>
        /// 是否已经下载过
        /// </summary>
        public static bool IsDownloaded(int userID, int itemID)
        {
            Dictionary<int, int> dictionary = GetThreadIDRecordIDsOfUser(userID);
            if (dictionary != null)
                return dictionary.ContainsKey(itemID);
            return false;
        }

        /// <summary>
        /// 获取前几条下载记录
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="isRealTime"></param>
        /// <returns></returns>
        public static IList<FileDownLoadRecord> GetTopRecordsOfThread(int threadID, int topNumber)
        {
            string cachekey = GetCacheKey_TopRecordsOfThread(threadID);
            List<int> list = CacheRepository.Get(cachekey, false) as List<int>;
            if (list == null)
            {
                list = FileDataProvider.Instance().GetTopRecordIDsOfThread(threadID, topNumber);
                CacheRepository.Add(cachekey, list, CachingExpirationTypes.ObjectCollection);
            }
            return PopulateFileDownLoadRecordsFromIDs(list);
        }

        /// <summary>
        /// 获取某个用户的下载记录字典
        /// Key=ItemID
        /// Value=FileDownLoadRecordID
        /// </summary>
        private static Dictionary<int, int> GetThreadIDRecordIDsOfUser(int userID)
        {
            string cachekey = GetCacheKey_ThreadIDRecordIDsOfUser(userID);
            Dictionary<int, int> dictionary = CacheRepository.Get(cachekey, true) as Dictionary<int, int>;
            if (dictionary == null)
            {
                dictionary = FileDataProvider.Instance().GetThreadIDDownLoadRecordIDsOfUser(userID);
                CacheRepository.Add(cachekey, dictionary, CachingExpirationTypes.ObjectCollection);
            }
            return dictionary;
        }

        /// <summary>
        /// 获取某个用户的所有下载记录列表
        /// </summary>
        public static IList<FileDownLoadRecord> GetDownLoadRecordsOfUser(int userID)
        {
            Dictionary<int, int> dictionary = GetThreadIDRecordIDsOfUser(userID);
            return PopulateFileDownLoadRecordsFromIDs(dictionary.Values.ToList());
        }

        /// <summary>
        /// 获取用户的下载记录分页集合
        /// </summary>
        public static PagingDataSet<FileDownLoadRecord> GetDownLoadRecordsOfUser(int userID, int pageIndex)
        {
            int cacheablePageCount = FileSectionConfiguration.Instance().CacheablePageCount;
            int pageSize = FileSectionConfiguration.Instance().ThreadsPerPage;

            List<int> recordIDs;
            BlockPagingID blockPagingID = null;
            if (pageIndex <= cacheablePageCount)
            {
                string cacheKey = GetCacheKey_DownLoadRecordsOfUser(userID);
                blockPagingID = CacheRepository.Get(cacheKey, false) as BlockPagingID;
                if (blockPagingID == null)
                {
                    blockPagingID = FileDataProvider.Instance().GetFileDownLoadRecords(FileSectionConfiguration.Instance().SecondaryMaxRecords, pageSize * cacheablePageCount, 1, userID);
                    CacheRepository.Add(cacheKey, blockPagingID, CachingExpirationTypes.ObjectCollection);
                }
                recordIDs = blockPagingID.GetPagingIDs(pageSize, pageIndex);
            }
            else
            {
                blockPagingID = FileDataProvider.Instance().GetFileDownLoadRecords(FileSectionConfiguration.Instance().SecondaryMaxRecords, pageSize, pageIndex, userID);
                recordIDs = blockPagingID.GetAll();
            }

            IList<FileDownLoadRecord> favorites = PopulateFileDownLoadRecordsFromIDs(recordIDs);
            PagingDataSet<FileDownLoadRecord> pagingFileDownLoadRecords = new PagingDataSet<FileDownLoadRecord>(favorites);
            pagingFileDownLoadRecords.PageIndex = pageIndex;
            pagingFileDownLoadRecords.PageSize = pageSize;
            pagingFileDownLoadRecords.TotalRecords = blockPagingID.TotalRecords;
            return pagingFileDownLoadRecords;
        }

        /// <summary>
        /// 通过favoriteIDs生成IList<FileDownLoadRecord>
        /// </summary>
        private static IList<FileDownLoadRecord> PopulateFileDownLoadRecordsFromIDs(List<int> recordIDs)
        {
            IList<FileDownLoadRecord> records = new List<FileDownLoadRecord>();

            foreach (int recordID in recordIDs)
            {
                FileDownLoadRecord record = GetFileDownLoadRecord(recordID);
                if (record != null)
                    records.Add(record);
            }
            return records;
        }

        /// <summary>
        /// 获取某个人下载列表的CacheKey
        /// </summary>
        private static string GetCacheKey_ThreadIDRecordIDsOfUser(int userID)
        {
            return string.Format("FileDownLoadRecordDic::U-{0}", userID);
        }

        /// <summary>
        /// 某个人的下载列表分页缓存CacheKey
        /// </summary>
        private static string GetCacheKey_DownLoadRecordsOfUser(int userID)
        {
            return string.Format("DownLoadRecordsOfUser::U-{0}", userID);
        }

        /// <summary>
        /// 获取文件下载记录CacheKey
        /// </summary>
        private static string GetCacheKey_FileDownLoadRecord(int recordID)
        {
            return string.Format("FileDownLoadRecord-{0}", recordID);
        }

        /// <summary>
        /// 获取文件的最新的下载记录CacheKey
        /// </summary>
        private static string GetCacheKey_TopRecordsOfThread(int threadID)
        {
            return string.Format("TopRecordsOfThread-{0}", threadID);
        }

        #endregion


        #region  delete User & Statistics

        /// <summary>
        /// 删除用户时删除与文件相关的内容
        /// </summary>
        /// <param name="userID">删除的UserID</param>
        public static void DeleteUser(int userID)
        {
            //删除评论
            BlockPagingID commentBlockPagingID = FileDataProvider.Instance().GetFileCommentIDsForAdmin(ValueHelper.GetSqlMaxInt(), ValueHelper.GetSqlMaxInt(), 1, null, userID, string.Empty);
            foreach (int commentID in commentBlockPagingID.GetAll())
            {
                FileComment comment = FileComments.GetComment(commentID, null, true);
                if (comment == null)
                    continue;
                FileComments.DeleteComment(commentID);
            }
            //删除文件
            BlockPagingID fileBlockPagingID = FileDataProvider.Instance().GetFileThreadIDs(userID, null, null, null, null, true, ValueHelper.GetSafeSqlInt(int.MaxValue), ValueHelper.GetSafeSqlInt(int.MaxValue), 1);

            foreach (int threadID in fileBlockPagingID.GetAll())
            {
                FileThread thread = GetThread(threadID, null, true);
                if (thread == null)
                    continue;
                FileThreads.DeleteThread(thread);

                //删除附件                
                FileAttachmentManager.Instance().DeleteAttachmentsByAssociateID(threadID);
            }

            //删除文件标题图附件
            FileFeaturedImageManager.Instance().DeleteAttachmentsByOwnerID(userID);

            //删除下载记录
            FileThreads.DeleteDownloadRecordsByUserID(userID);
            //删除用户标签
            UserTagManager userTagManager = UserTagManager.Instance(UserTagItemKeys.Instance().File());
            ICollection<UserTag> userTags = new List<UserTag>(userTagManager.GetUserTags(userID, true));
            if (userTags != null && userTags.Count > 0)
            {
                foreach (UserTag userTag in userTags)
                {
                    userTagManager.DeleteUserTag(userTag.TagName, userID);
                }
            }

            //删除用户分类
            UserCategoryManager userCategoryManager = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().File());
            List<UserCategory> userCategories = userCategoryManager.GetUserCategories(userID, true);
            if (userCategories != null && userCategories.Count > 0)
            {
                foreach (UserCategory userCategory in userCategories)
                {
                    userCategoryManager.DeleteUserCategory(userCategory.CategoryID, userID);
                }
            }

            //删除模块
            FileSection fileSection = FileSections.GetSection(userID, true);
            if (fileSection != null)
                FileSections.Delete(userID);
        }

        /// <summary>
        /// 删除用户并重设用户内容时处理与文件相关的内容
        /// </summary>
        public static void DeleteUserAndReassignContent(int userID, string reassignUserName)
        {
            FileDataProvider.Instance().DeleteUserAndReassignContent(userID, reassignUserName);

            //未处理缓存
        }

        /// <summary>
        /// 执行统计
        /// </summary>
        public static void ExecuteStatistics()
        {
            FileDataProvider.Instance().ExecuteStatistics();
        }

        /// <summary>
        /// 获取文件统计数据
        /// </summary>
        public static Dictionary<FileManageableCounts, int> GetManageableCounts()
        {
            string cacheKey = "FileManageableCounts";
            Dictionary<FileManageableCounts, int> dictionary = CacheRepository.Get(cacheKey, false) as Dictionary<FileManageableCounts, int>;
            if (dictionary == null)
            {
                dictionary = FileDataProvider.Instance().GetManageableCounts();
                CacheRepository.Add(cacheKey, dictionary, CachingExpirationTypes.ObjectCollection);
            }
            return dictionary;
        }


        #endregion

        #region Help Methods

        /// <summary>
        /// 通过FileThreadID生成IList<FileThread>
        /// </summary>
        /// <param name="fileThreadIDs"></param>
        /// <returns></returns>
        private static IList<FileThread> PopulateFileThreadsFromIDs(List<int> fileThreadIDs, PrivacyStatuses? privacyStatus, bool isRealTime)
        {
            IList<FileThread> fileThreads = new List<FileThread>();

            foreach (int fileThreadID in fileThreadIDs)
            {
                FileThread fileThread = GetThread(fileThreadID, privacyStatus, isRealTime);
                if (fileThread != null)
                    fileThreads.Add(fileThread);
            }
            return fileThreads;
        }

        /// <summary>
        /// 获取单个文件实体缓存
        /// </summary>
        /// <param name="threadID"></param>
        /// <returns></returns>
        private static string GetCacheKey_SingleFileThread(int threadID)
        {
            return "FileThread:" + threadID;
        }


        /// <summary>
        /// 获取我的FileThreadID列表
        /// </summary>
        /// <returns></returns>
        private static string GetCacheKey_MyFileThreadIDs(int ownerUserID, string tagName, int? userCategoryID)
        {
            return CacheKeyGenerator.Generate("MyFileThreadIDs", new Dictionary<string, string> { { "U", ownerUserID.ToString() }, { "T", tagName }, { "C", userCategoryID.ToString() } });
        }

        /// <summary>
        /// 获取文件类型的cacheKey
        /// </summary>
        /// <returns></returns>
        private static string GetCacheKey_FileThreadTypes()
        {
            return "File:AllFileThreadTypes";
        }

        /// <summary>
        /// 获取文件评价的cacheKEy
        /// </summary>
        /// <param name="itemID"></param>
        /// <returns></returns>
        private static string GetCacheKey_FileRatings(int itemID)
        {
            return "File_Ratings:" + itemID;
        }

        /// <summary>
        /// 获取文件实体内容缓存
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="isLocalCache"></param>
        /// <returns></returns>
        private static string GetCacheKey_FileThreadBody(int threadID)
        {
            return "FileThread:B:" + threadID;
        }

        #endregion

    }
}
