﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common;
using SpaceBuilder.Caching;
using System.ComponentModel;

using System.Xml;
using System.Xml.Linq;


namespace SpaceBuilder.File
{
    #region delegates

    /// <summary>
    /// 用于处理文件的委托
    /// </summary>
    /// <param name="FileThread">触发事件的文件</param>
    /// <param name="e">事件数据</param>
    public delegate void FileThreadEventHandler(FileThread fileThread, GlobalEventArgs e);

    /// <summary>
    /// 用于处理文件评论的委托
    /// </summary>
    /// <param name="FilePost">触发事件的文件评论</param>
    /// <param name="e">事件数据</param>
    public delegate void FilePostEventHandler(FileComment filePost, GlobalEventArgs e);

    /// <summary>
    /// 用于处理文件模块的委托
    /// </summary>
    /// <param name="fileSection">触发事件的文件模块</param>
    /// <param name="e">事件数据</param>
    public delegate void FileSectionEventHandler(FileSection fileSection, GlobalEventArgs e);

    /// <summary>
    /// 用于处理下载记录的委托
    /// </summary>
    /// <param name="record">触发事件的下载记录</param>
    /// <param name="e">事件数据</param>
    public delegate void DownloadRecordEventHandler(FileDownLoadRecord record);

    /// <summary>
    /// 用于处理文件审核状态变化的委托
    /// </summary>
    /// <param name="e">事件数据</param>
    /// <param name="userPointDirection">审核结果</param>
    public delegate void PointChangeOfFileThreadHandler(FileThread fileThread, GlobalEventArgs e, UserPointDirection userPointDirection);

    /// <summary>
    /// 用于处理文件评论审核状态变化的委托
    /// </summary>
    /// <param name="e">事件数据</param>
    /// <param name="userPointDirection">审核结果</param>
    public delegate void PointChangeOfFileCommentHandler(FileComment comment, GlobalEventArgs e, UserPointDirection userPointDirection);

    #endregion

    /// <summary>
    /// 文件事件管理器
    /// </summary>
    public class FileEventManager
    {

        #region private members
        private EventHandlerList Events = new EventHandlerList();
        private Dictionary<string, IFileModule> modules = new Dictionary<string, IFileModule>();
        #endregion

        #region Event Keys (static)

        private static object EventKey_BeforeFileThreadChange = new object();
        private static object EventKey_AfterFileThreadChange = new object();

        private static object EventKey_AfterFileThreadSetEssential = new object();
        private static object EventKey_AfterFileThreadSetSticky = new object();

        private static object EventKey_BeforeFilePostChange = new object();
        private static object EventKey_AfterFilePostChange = new object();

        private static object EventKey_BeforeFileSectionChange = new object();
        private static object EventKey_AfterFileSectionChange = new object();

        private static object EventKey_AfterFileDownLoadRecordChange = new object();

        private static object EventKey_AfterPointChangeOfFileThread = new object();
        private static object EventKey_AfterPointChangeOfFileComment = new object();

        #endregion

        private static volatile FileEventManager _self = null;
        private static readonly object lockObject = new object();

        private FileEventManager()
        { }


        /// <summary>
        /// 获取FileEventManager的实例
        /// </summary>
        internal static FileEventManager Instance()
        {
            if (_self == null)
            {
                lock (lockObject)
                {
                    if (_self == null)
                    {
                        _self = new FileEventManager();

                        XElement eventExtensionModulesElement = FileSectionConfiguration.Instance().ApplicationElement.Element("eventExtensionModules");
                        if (eventExtensionModulesElement != null)
                        {

                            foreach (XElement n in eventExtensionModulesElement.Elements())
                            {
                                if (n.NodeType != XmlNodeType.Comment)
                                {
                                    switch (n.Name.LocalName)
                                    {
                                        case "clear":
                                            _self.modules.Clear();
                                            break;

                                        case "remove":
                                            string nameRemove = n.Attribute("name").Value;
                                            if (_self.modules.ContainsKey(nameRemove))
                                                _self.modules.Remove(nameRemove);
                                            break;

                                        case "add":

                                            string name = n.Attribute("name").Value;
                                            string itype = n.Attribute("type").Value;

                                            Type type = Type.GetType(itype);

                                            if (type == null)
                                                throw new Exception(itype + " does not exist");

                                            IFileModule mod = Activator.CreateInstance(type) as IFileModule;

                                            if (mod == null)
                                                throw new Exception(itype + " does not implement IFileModule or is not configured correctly");

                                            mod.Init(_self, n);
                                            _self.modules[name] = mod;
                                            break;
                                    }
                                }
                            }
                        }
                    }
                }
            }
            return _self;
        }


        #region Events

        /// <summary>
        /// 添加、修改、删除文件，持久化以前触发的事件
        /// </summary>
        public event FileThreadEventHandler BeforeFileThreadChange
        {
            add { Events.AddHandler(EventKey_BeforeFileThreadChange, value); }
            remove { Events.RemoveHandler(EventKey_BeforeFileThreadChange, value); }
        }

        /// <summary>
        /// 添加、修改、删除文件，持久化以后触发的事件
        /// </summary>
        public event FileThreadEventHandler AfterFileThreadChange
        {
            add { Events.AddHandler(EventKey_AfterFileThreadChange, value); }
            remove { Events.RemoveHandler(EventKey_AfterFileThreadChange, value); }
        }

        /// <summary>
        /// 文件设为精华，持久化以后触发的事件
        /// </summary>
        public event FileThreadEventHandler AfterFileThreadSetEssential
        {
            add { Events.AddHandler(EventKey_AfterFileThreadSetEssential, value); }
            remove { Events.RemoveHandler(EventKey_AfterFileThreadSetEssential, value); }
        }

        /// <summary>
        /// 文件置顶，持久化以后触发的事件
        /// </summary>
        public event FileThreadEventHandler AfterFileThreadSetSticky
        {
            add { Events.AddHandler(EventKey_AfterFileThreadSetSticky, value); }
            remove { Events.RemoveHandler(EventKey_AfterFileThreadSetSticky, value); }
        }

        /// <summary>
        /// 添加、修改、删除文件评论，持久化以前触发的事件
        /// </summary>
        public event FilePostEventHandler BeforeFileCommentChange
        {
            add { Events.AddHandler(EventKey_BeforeFilePostChange, value); }
            remove { Events.RemoveHandler(EventKey_BeforeFilePostChange, value); }
        }

        /// <summary>
        /// 添加、修改、删除文件评论，持久化以后触发的事件
        /// </summary>
        public event FilePostEventHandler AfterFileCommentChange
        {
            add { Events.AddHandler(EventKey_AfterFilePostChange, value); }
            remove { Events.RemoveHandler(EventKey_AfterFilePostChange, value); }
        }

        /// <summary>
        /// 添加 修改 删除文件模块，持久化之前触发事件
        /// </summary>
        public event FileSectionEventHandler BeforeFileSectionChange
        {
            add { Events.AddHandler(EventKey_BeforeFileSectionChange, value); }
            remove { Events.RemoveHandler(EventKey_BeforeFileSectionChange, value); }
        }

        /// <summary>
        /// 添加 修改 删除文件模块，持久化之后触发事件
        /// </summary>
        public event FileSectionEventHandler AfterFileSectionChange
        {
            add { Events.AddHandler(EventKey_AfterFileSectionChange, value); }
            remove { Events.RemoveHandler(EventKey_AfterFileSectionChange, value); }
        }

        /// <summary>
        /// 文件下载记录持久化之后触发事件
        /// </summary>
        public event DownloadRecordEventHandler AfterFileDownLoadRecordChange
        {
            add { Events.AddHandler(EventKey_AfterFileDownLoadRecordChange, value); }
            remove { Events.RemoveHandler(EventKey_AfterFileDownLoadRecordChange, value); }
        }

        /// <summary>
        /// 文件审核状态变化，持久化之后触发事件
        /// </summary>
        public event PointChangeOfFileThreadHandler AfterPointChangeOfFileThread
        {
            add { Events.AddHandler(EventKey_AfterPointChangeOfFileThread, value); }
            remove { Events.RemoveHandler(EventKey_AfterPointChangeOfFileThread, value); }
        }

        /// <summary>
        /// 文件评论审核状态变化，持久化之后触发事件
        /// </summary>
        public event PointChangeOfFileCommentHandler AfterPointChangeOfFileComment
        {
            add { Events.AddHandler(EventKey_AfterPointChangeOfFileComment, value); }
            remove { Events.RemoveHandler(EventKey_AfterPointChangeOfFileComment, value); }
        }

        #endregion

        #region Execute Events

        /// <summary>
        /// 添加、修改、删除文件，持久化以前调用
        /// </summary>
        internal void ExecuteBeforeFileThreadChangeEvents(FileThread FileThread, ObjectState state)
        {
            ExecuteFileThreadEvent(EventKey_BeforeFileThreadChange, FileThread, state);
        }

        /// <summary>
        /// 添加、修改、删除文件，持久化以后调用 
        /// </summary>
        internal void ExecuteAfterFileThreadChangeEvents(FileThread FileThread, ObjectState state)
        {
            ExecuteFileThreadEvent(EventKey_AfterFileThreadChange, FileThread, state);
        }

        /// <summary>
        /// 文件设置为精华，持久化以后调用 
        /// </summary>
        internal void ExecuteAfterFileThreadSetEssentialEvents(FileThread FileThread, ObjectState state)
        {
            ExecuteFileThreadEvent(EventKey_AfterFileThreadSetEssential, FileThread, state);
        }

        /// <summary>
        /// 文件置顶，持久化以后调用 
        /// </summary>
        internal void ExecuteAfterFileThreadSetStickyEvents(FileThread FileThread, ObjectState state)
        {
            ExecuteFileThreadEvent(EventKey_AfterFileThreadSetSticky, FileThread, state);
        }

        /// <summary>
        /// 执行文件相关事件
        /// </summary>
        protected void ExecuteFileThreadEvent(object EventKey, FileThread FileThread, ObjectState state)
        {
            FileThreadEventHandler handler = Events[EventKey] as FileThreadEventHandler;
            if (handler != null)
            {
                handler(FileThread, new GlobalEventArgs(state, ApplicationIDs.Instance().File()));
            }
        }


        /// <summary>
        /// 添加、修改、删除文件评论，持久化以前调用
        /// </summary>
        internal void ExecuteBeforeFileCommentChangeEvents(FileComment FilePost, ObjectState state)
        {
            ExecuteFileCommentEvent(EventKey_BeforeFilePostChange, FilePost, state);
        }

        /// <summary>
        /// 添加、修改、删除文件评论，持久化以后调用
        /// </summary>
        internal void ExecuteAfterFileCommentChangeEvents(FileComment FilePost, ObjectState state)
        {
            ExecuteFileCommentEvent(EventKey_AfterFilePostChange, FilePost, state);
        }

        /// <summary>
        /// 执行文件评论相关事件
        /// </summary>
        protected void ExecuteFileCommentEvent(object eventKey, FileComment FilePost, ObjectState state)
        {
            FilePostEventHandler handler = Events[eventKey] as FilePostEventHandler;
            if (handler != null)
            {
                handler(FilePost, new GlobalEventArgs(state, ApplicationIDs.Instance().File()));
            }
        }

        /// <summary>
        /// 添加、修改、删除文件模块，持久化之前调用
        /// </summary>
        internal void ExecuteBeforeFileSectionChangeEvents(FileSection fileSecion, ObjectState state)
        {
            ExcuteFileSectionEvent(EventKey_BeforeFileSectionChange, fileSecion, state);
        }

        /// <summary>
        /// 添加、修改、删除文件模块，持久化以后调用
        /// </summary>
        internal void ExecuteAfterFileSectionChangeEvents(FileSection fileSeciont, ObjectState state)
        {
            ExcuteFileSectionEvent(EventKey_AfterFileSectionChange, fileSeciont, state);
        }

        /// <summary>
        /// 执行文件相关事件
        /// </summary>
        protected void ExcuteFileSectionEvent(object eventKey, FileSection fileSetion, ObjectState state)
        {
            FileSectionEventHandler handler = Events[eventKey] as FileSectionEventHandler;
            if (handler != null)
            {
                handler(fileSetion, new GlobalEventArgs(state, ApplicationIDs.Instance().File()));
            }
        }
        /// <summary>
        /// 添加、修改、删除文件模块，持久化以后调用
        /// </summary>
        internal void ExecuteAfterFileDownLoadRecordChangeEvents(FileDownLoadRecord record)
        {
            DownloadRecordEventHandler handler = Events[EventKey_AfterFileDownLoadRecordChange] as DownloadRecordEventHandler;
            if (handler != null)
            {
                handler(record);
            }
        }

        /// <summary>
        /// 文件审核状态变化，持久化之后调用
        /// </summary>
        internal void ExecutePointChangeOfFileThreadEvents(FileThread fileThread, ObjectState state, UserPointDirection userPointDirection)
        {
            ExecutePointChangeOfFileThreadEvent(EventKey_AfterPointChangeOfFileThread, fileThread, state, userPointDirection);
        }

        /// <summary>
        /// 执行文件审核状态变化相关事件
        /// </summary>
        protected void ExecutePointChangeOfFileThreadEvent(object eventKey, FileThread fileThread, ObjectState state, UserPointDirection userPointDirection)
        {
            PointChangeOfFileThreadHandler handler = Events[eventKey] as PointChangeOfFileThreadHandler;
            if (handler != null)
            {
                handler(fileThread, new GlobalEventArgs(state, ApplicationIDs.Instance().File()), userPointDirection);
            }
        }

        /// <summary>
        /// 文件评论审核状态变化，持久化之后调用
        /// </summary>
        internal void ExecutePointChangeOfFileCommentEvents(FileComment comment, ObjectState state, UserPointDirection userPointDirection)
        {
            ExecutePointChangeOfFileCommentEvent(EventKey_AfterPointChangeOfFileComment, comment, state, userPointDirection);
        }

        /// <summary>
        /// 执行文件评论审核状态变化相关事件
        /// </summary>
        protected void ExecutePointChangeOfFileCommentEvent(object eventKey, FileComment comment, ObjectState state, UserPointDirection userPointDirection)
        {
            PointChangeOfFileCommentHandler handler = Events[eventKey] as PointChangeOfFileCommentHandler;
            if (handler != null)
            {
                handler(comment, new GlobalEventArgs(state, ApplicationIDs.Instance().File()), userPointDirection);
            }
        }

        #endregion
    }
}