﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.SqlClient;
using SpaceBuilder.Common;

using System.Data;
using SpaceBuilder.Utils;


namespace SpaceBuilder.File
{
    /// <summary>
    /// 文件数据SQL数据库提供者
    /// </summary>
    public sealed class SqlFileDataProvider : FileDataProvider
    {
        #region Member variables
        private string databaseOwner = "dbo";	// overwrite in web.config
        string writableConnectionString = null;
        string readOnlyConnectionString = null;
        #endregion

        #region Constructor
        /// <summary>
        /// 文件数据提供者构造函数
        /// </summary>
        /// <param name="databaseOwner">数据库用户名</param>
        /// <param name="connectionString">连接字符串</param>
        public SqlFileDataProvider(string databaseOwner, string writableConnectionString, string readOnlyConnectionString)
        {
            this.databaseOwner = databaseOwner;
            this.writableConnectionString = writableConnectionString;
            this.readOnlyConnectionString = readOnlyConnectionString;
        }
        #endregion

        #region helpers
        /// <summary>
        /// 获取可写数据库连接
        /// </summary>
        /// <returns></returns>
        private SqlConnection GetWritableConnection()
        {
            try
            {
                return new SqlConnection(writableConnectionString);
            }
            catch
            {
                throw new SPBException(ExceptionTypes.Instance().DataProvider(), string.Format("SQL Connection String {0} is invalid.", writableConnectionString));
            }
        }

        /// <summary>
        /// 获取只读数据库连接
        /// </summary>
        /// <returns></returns>
        private SqlConnection GetReadOnlyConnection()
        {
            try
            {
                return new SqlConnection(readOnlyConnectionString);
            }
            catch
            {
                throw new SPBException(ExceptionTypes.Instance().DataProvider(), string.Format("SQL Connection String {0} is invalid.", readOnlyConnectionString));
            }
        }

        #endregion

        #region FileSection

        /// <summary>
        /// 创建、更新FileSection
        /// </summary>
        public override FileSection CreateUpdateSection(FileSection section, DataProviderAction action)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileSection_CreateUpdateDelete", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@Action", SqlDbType.Int).Value = (int)action;

                myCommand.Parameters.Add("@OwnerUserID", SqlDbType.Int).Value = section.OwnerUserID;
                myCommand.Parameters.Add("@SectionName", SqlDbType.NVarChar, 512).Value = section.SectionName;
                myCommand.Parameters.Add("@Description", SqlDbType.NVarChar, 2000).Value = section.Description;
                myCommand.Parameters.Add("@IsActive", SqlDbType.Bit).Value = section.IsActive;
                myCommand.Parameters.Add("@EnableSearch", SqlDbType.Bit).Value = section.EnableSearch;

                SerializerData data = section.GetSerializerData();
                myCommand.Parameters.Add("@PropertyNames", SqlDbType.NText).Value = data.Keys;
                myCommand.Parameters.Add("@PropertyValues", SqlDbType.NText).Value = data.Values;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myConnection.Close();

                if (action == DataProviderAction.Create)
                {
                    section.DateCreated = DateTime.Now;
                }
                return section;
            }
        }

        /// <summary>
        /// 删除FileSection
        /// </summary>
        public override void DeleteSection(int ownerUserID)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileSection_CreateUpdateDelete", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@Action", SqlDbType.Int).Value = (int)DataProviderAction.Delete;
                myCommand.Parameters.Add("@OwnerUserID", SqlDbType.Int).Value = ownerUserID;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myConnection.Close();
            }
        }

        /// <summary>
        /// FileSection重新统计
        /// </summary>
        public override void ExecFileSectionResetStatistics(List<int> list)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileSection_ResetStatistics", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@OwnerUserID", SqlDbType.Int);

                myConnection.Open();
                foreach (var countObject in list)
                {
                    myCommand.Parameters["@OwnerUserID"].Value = countObject;
                    myCommand.ExecuteNonQuery();
                }
                myConnection.Close();
            }
        }


        /// <summary>
        /// 根据空间文件模块的SectionID获取空间文件模块
        /// </summary>
        public override FileSection GetSection(int ownerUserID)
        {
            using (SqlConnection myConnection = GetReadOnlyConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileSection_Get", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;
                myCommand.Parameters.Add("@OwnerUserID", SqlDbType.Int).Value = ownerUserID;

                FileSection fileSection = null;
                myConnection.Open();
                using (SqlDataReader dr = myCommand.ExecuteReader(CommandBehavior.CloseConnection))
                {
                    if (dr.Read())
                        fileSection = PopulateFileSectionFromIDataReader(dr);

                    dr.Close();
                }
                myConnection.Close();
                return fileSection;
            }
        }


        /// <summary>
        /// 获取非主流排序方式的文件主题ID集合
        /// </summary>
        /// <param name="sortBy">排序方式</param>
        /// <param name="maxRecords">最大记录限制</param>
        /// <param name="fileActiveStatus"></param>
        /// <param name="onlyShowHasContent"></param>
        /// <returns></returns>
        public override BlockPagingID GetFileSectionIDsSortBy(FileSectionSortBy sortBy, int maxRecords, FileActiveStatus? fileActiveStatus, bool onlyShowHasContent)
        {
            using (SqlConnection connection = GetReadOnlyConnection())
            {
                using (SqlCommand command = new SqlCommand(databaseOwner + ".spb_Common_GetPagingRecordsWithNotIn", connection))
                {
                    command.CommandType = CommandType.StoredProcedure;

                    string fromClause = string.Format("{0}.spb_FileSections S", databaseOwner);

                    StringBuilder whereClause = new StringBuilder();
                    if (fileActiveStatus.HasValue)
                    {
                        whereClause.AppendFormat("  and S.IsActive = {0} ", (int)fileActiveStatus);
                    }

                    if (onlyShowHasContent)
                    {
                        whereClause.Append(" and S.FileCount > 0 ");
                    }

                    string orderByClause;
                    switch (sortBy)
                    {
                        case FileSectionSortBy.Points:
                            orderByClause = "S.Points desc";
                            break;
                        case FileSectionSortBy.PostCount:
                            orderByClause = "S.FileCount desc";
                            break;
                        case FileSectionSortBy.CommentCount:
                            orderByClause = "S.CommentCount desc";
                            break;
                        case FileSectionSortBy.WeekHitTimes:
                            orderByClause = "S.StageHitTimes desc";
                            break;
                        case FileSectionSortBy.HitTimes:
                            orderByClause = "S.HitTimes desc";
                            break;
                        case FileSectionSortBy.DateCreated:
                        default:
                            orderByClause = "S.OwnerUserID desc";
                            break;
                    }

                    //移除开始的" and "
                    if (whereClause.Length > 0)
                        whereClause.Remove(0, 5);

                    command.Parameters.Add("@PageIndex", SqlDbType.Int).Value = 0;
                    command.Parameters.Add("@PageSize", SqlDbType.Int).Value = maxRecords;
                    command.Parameters.Add("@FromClause", SqlDbType.VarChar, 1024).Value = fromClause;
                    command.Parameters.Add("@SelectFields", SqlDbType.VarChar, 512).Value = "S.OwnerUserID";
                    command.Parameters.Add("@WhereClause", SqlDbType.NVarChar, 4000).Value = whereClause.ToString();
                    command.Parameters.Add("@OrderByClause", SqlDbType.VarChar, 256).Value = orderByClause;
                    command.Parameters.Add("@UniqueField", SqlDbType.VarChar, 512).Value = "S.OwnerUserID";
                    command.Parameters.Add("@ReturnRecordCount", SqlDbType.Bit).Value = false;  //一定不要返回记录总数
                    command.Parameters.Add("@MaxRecords", SqlDbType.Int).Value = maxRecords;
                    command.Parameters.Add("@TotalRecords", SqlDbType.Int).Direction = ParameterDirection.Output;

                    List<int> idList = new List<int>();

                    connection.Open();
                    using (SqlDataReader dr = command.ExecuteReader(CommandBehavior.CloseConnection))
                    {
                        while (dr.Read())
                        {
                            idList.Add(Convert.ToInt32(dr["OwnerUserID"]));
                        }
                        dr.Close();
                    }
                    connection.Close();

                    BlockPagingID blockPagingID = new BlockPagingID(idList);
                    return blockPagingID;
                }
            }
        }
        /// <summary>
        /// 获取好友的最新文件
        /// </summary>
        public override BlockPagingID GetFileSectionIDsOfFriends(int userID)
        {
            using (SqlConnection myConnection = GetReadOnlyConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileSections_GetSectionIDOfFriends", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;
                myCommand.Parameters.Add("@UserID", SqlDbType.Int).Value = userID;

                List<int> idList = new List<int>();
                myConnection.Open();
                using (SqlDataReader dr = myCommand.ExecuteReader())
                {
                    while (dr.Read())
                    {
                        idList.Add(Convert.ToInt32(dr["OwnerUserID"]));
                    }
                }
                myConnection.Close();
                BlockPagingID blockPagingID = new BlockPagingID(idList);

                return blockPagingID;
            }
        }

        #endregion

        #region FileThread

        /// <summary>
        /// 创建文件
        /// </summary>
        /// <param name="thread">要创建的文件实体</param>
        public override void CreateThread(FileThread thread)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThread_CreateUpdate", myConnection);

                string passwordSalt = GenerateSalt();
                myCommand.CommandType = CommandType.StoredProcedure;
                myCommand.Parameters.Add("@Action", SqlDbType.Int).Value = (int)DataProviderAction.Create;
                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int).Direction = ParameterDirection.Output;
                myCommand.Parameters.Add("@OwnerUserID", SqlDbType.Int).Value = thread.OwnerUserID;
                myCommand.Parameters.Add("@Author", SqlDbType.NVarChar, 64).Value = thread.Author;
                myCommand.Parameters.Add("@PrivacyStatus", SqlDbType.Int).Value = (int)thread.PrivacyStatus;
                myCommand.Parameters.Add("@Password", SqlDbType.NVarChar, 64).Value = thread.Password;
                myCommand.Parameters.Add("@Price", SqlDbType.Int).Value = thread.Price;
                myCommand.Parameters.Add("@IsLocked", SqlDbType.Bit).Value = thread.IsLocked;
                myCommand.Parameters.Add("@IsEssential", SqlDbType.Bit).Value = thread.IsEssential;
                myCommand.Parameters.Add("@SpecialOrder", SqlDbType.Int).Value = thread.SpecialOrder;
                myCommand.Parameters.Add("@StickyDate", SqlDbType.DateTime).Value = ValueHelper.GetSafeSqlDateTime(thread.StickyDate);
                myCommand.Parameters.Add("@Subject", SqlDbType.NVarChar, 256).Value = thread.Subject;
                //myCommand.Parameters.Add("@Body", SqlDbType.NText).Value = thread.BodyForDataTransmission;

                SqlParameter body = myCommand.Parameters.Add("@Body", SqlDbType.NText);
                if (string.IsNullOrEmpty(thread.BodyForDataTransmission))
                    body.Value = string.Empty;
                else
                    body.Value = thread.BodyForDataTransmission;

                myCommand.Parameters.Add("@SiteCategoryID", SqlDbType.Int).Value = thread.SiteCategoryID;
                myCommand.Parameters.Add("@UserCategoryID", SqlDbType.Int).Value = thread.UserCategoryID;
                myCommand.Parameters.Add("@AuditingStatus", SqlDbType.Int).Value = (int)thread.AuditingStatus;
                myCommand.Parameters.Add("@UserHostAddress", SqlDbType.NVarChar, 64).Value = thread.UserHostAddress;
                myCommand.Parameters.Add("@FileName", SqlDbType.NVarChar, 512).Value = thread.Attachment.FileName;
                myCommand.Parameters.Add("@ContentType", SqlDbType.NVarChar, 64).Value = thread.Attachment.ContentType;
                myCommand.Parameters.Add("@ContentSize", SqlDbType.Int).Value = thread.Attachment.ContentSize;
                myCommand.Parameters.Add("@Height", SqlDbType.Int).Value = thread.Attachment.Height;
                myCommand.Parameters.Add("@Width", SqlDbType.Int).Value = thread.Attachment.Width;
                myCommand.Parameters.Add("@IsRemote", SqlDbType.Int).Value = thread.Attachment.IsRemote;

                SerializerData data = thread.GetSerializerData();
                myCommand.Parameters.Add("@PropertyNames", SqlDbType.NText).Value = data.Keys;
                myCommand.Parameters.Add("@PropertyValues", SqlDbType.NText).Value = data.Values;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myConnection.Close();

                thread.PostDate = DateTime.Now;
                thread.ThreadID = Convert.ToInt32(myCommand.Parameters["@ThreadID"].Value);
            }
        }

        /// <summary>
        /// 更新文件
        /// </summary>
        /// <param name="thread">要更新的文件实体</param>
        public override void UpdateThread(FileThread thread)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThread_CreateUpdate", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;
                string passwordSalt = GenerateSalt();
                myCommand.Parameters.Add("@Action", SqlDbType.Int).Value = (int)DataProviderAction.Update;
                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int).Value = thread.ThreadID;
                myCommand.Parameters.Add("@Author", SqlDbType.NVarChar, 64).Value = thread.Author;
                myCommand.Parameters.Add("@OwnerUserID", SqlDbType.Int).Value = thread.OwnerUserID;
                myCommand.Parameters.Add("@PrivacyStatus", SqlDbType.Int).Value = (int)thread.PrivacyStatus;
                myCommand.Parameters.Add("@Password", SqlDbType.NVarChar, 64).Value = thread.Password;
                myCommand.Parameters.Add("@IsLocked", SqlDbType.Bit).Value = thread.IsLocked;
                myCommand.Parameters.Add("@IsEssential", SqlDbType.Bit).Value = thread.IsEssential;
                myCommand.Parameters.Add("@SpecialOrder", SqlDbType.Int).Value = thread.SpecialOrder;
                myCommand.Parameters.Add("@StickyDate", SqlDbType.DateTime).Value = ValueHelper.GetSafeSqlDateTime(thread.StickyDate);
                myCommand.Parameters.Add("@Subject", SqlDbType.NVarChar, 256).Value = thread.Subject;
                myCommand.Parameters.Add("@Price", SqlDbType.Int).Value = thread.Price;
                if (string.IsNullOrEmpty(thread.BodyForDataTransmission))
                    myCommand.Parameters.Add("@Body", SqlDbType.NText).Value = DBNull.Value;
                else
                    myCommand.Parameters.Add("@Body", SqlDbType.NText).Value = thread.BodyForDataTransmission;

                myCommand.Parameters.Add("@SiteCategoryID", SqlDbType.Int).Value = thread.SiteCategoryID;
                myCommand.Parameters.Add("@UserCategoryID", SqlDbType.Int).Value = thread.UserCategoryID;
                myCommand.Parameters.Add("@AuditingStatus", SqlDbType.Int).Value = (int)thread.AuditingStatus;
                myCommand.Parameters.Add("@UserHostAddress", SqlDbType.NVarChar, 64).Value = thread.UserHostAddress;
                myCommand.Parameters.Add("@FileName", SqlDbType.NVarChar, 512).Value = thread.Attachment.FileName;
                myCommand.Parameters.Add("@ContentType", SqlDbType.NVarChar, 64).Value = thread.Attachment.ContentType;
                myCommand.Parameters.Add("@ContentSize", SqlDbType.Int).Value = thread.Attachment.ContentSize;
                myCommand.Parameters.Add("@Height", SqlDbType.Int).Value = thread.Attachment.Height;
                myCommand.Parameters.Add("@Width", SqlDbType.Int).Value = thread.Attachment.Width;
                myCommand.Parameters.Add("@IsRemote", SqlDbType.Int).Value = thread.Attachment.IsRemote;

                SerializerData data = thread.GetSerializerData();
                myCommand.Parameters.Add("@PropertyNames", SqlDbType.NText).Value = data.Keys;
                myCommand.Parameters.Add("@PropertyValues", SqlDbType.NText).Value = data.Values;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myConnection.Close();
            }
        }

        /// <summary>
        /// 更新站点类别
        /// </summary>
        public override void UpdateThreadsSiteCategory(List<int> threadIDs, int siteCategoryID)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThread_UpdateSiteCategory", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int);
                myCommand.Parameters.Add("@SiteCategoryID", SqlDbType.Int).Value = siteCategoryID;

                myConnection.Open();
                foreach (int threadID in threadIDs)
                {
                    myCommand.Parameters["@ThreadID"].Value = threadID;
                    myCommand.ExecuteNonQuery();
                }
                myConnection.Close();
            }
        }

        /// <summary>
        /// 更新所属用户类别
        /// </summary>
        public override void UpdateThreadsUserCategory(List<int> threadIDs, int userCategoryID)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThread_UpdateUserCategory", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int);
                myCommand.Parameters.Add("@UserCategoryID", SqlDbType.Int).Value = userCategoryID;

                myConnection.Open();
                foreach (int threadID in threadIDs)
                {
                    myCommand.Parameters["@ThreadID"].Value = threadID;
                    myCommand.ExecuteNonQuery();
                }
                myConnection.Close();
            }
        }

        /// <summary>
        /// 删除文件
        /// </summary>
        /// <param name="postID">要删除文件的ThreadID</param>
        public override void DeleteThread(int threadID)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThread_Delete", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int).Value = threadID;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myConnection.Close();
            }
        }

        /// <summary>
        /// 批量删除文件
        /// </summary>
        public override void DeleteThreads(List<int> threadIDs)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThread_Delete", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int);

                myConnection.Open();
                foreach (int threadID in threadIDs)
                {
                    myCommand.Parameters["@ThreadID"].Value = threadID;
                    myCommand.ExecuteNonQuery();
                }
                myConnection.Close();
            }
        }

        /// <summary>
        /// 设置主题审核状态
        /// </summary>
        public override void UpdateThreadAuditingStatus(List<int> threadIDs, AuditingStatuses auditingStatus)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThread_UpdateAuditingStatus", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int);
                myCommand.Parameters.Add("@AuditingStatus", SqlDbType.Int).Value = (int)auditingStatus;

                myConnection.Open();
                foreach (int threadID in threadIDs)
                {
                    myCommand.Parameters["@ThreadID"].Value = threadID;
                    myCommand.ExecuteNonQuery();
                }
                myConnection.Close();
            }
        }

        /// <summary>
        /// 批量设置隐私状态
        /// </summary>
        public override void UpdateThreadPrivacyStatus(List<int> threadIDs, PrivacyStatuses privacyStatus)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThread_UpdatePrivacyStatus", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int);
                myCommand.Parameters.Add("@PrivacyStatus", SqlDbType.Int).Value = (int)privacyStatus;

                myConnection.Open();
                foreach (int threadID in threadIDs)
                {
                    myCommand.Parameters["@ThreadID"].Value = threadID;
                    myCommand.ExecuteNonQuery();
                }
                myConnection.Close();
            }
        }

        /// <summary>
        /// 根据文件的PostID获取文件实体
        /// </summary>
        public override FileThread GetThread(int threadID)
        {
            using (SqlConnection myConnection = GetReadOnlyConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThread_Get", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int).Value = threadID;

                FileThread thread = null;

                myConnection.Open();

                using (SqlDataReader dr = myCommand.ExecuteReader(CommandBehavior.CloseConnection))
                {
                    if (dr.Read())
                    {
                        thread = PopulateFileThreadFromIDataReader(dr);
                    }
                }

                myConnection.Close();

                return thread;
            }
        }

        /// <summary>
        /// 获取非主流排序方式的文件主题ID集合
        /// </summary>
        /// <param name="sortBy">排序方式</param>
        /// <param name="maxRecords">最大记录限制</param>
        /// <param name="privacyStatusesForDisplay">显示何种隐私状态的数据</param>
        /// <param name="auditingStatusesForDisplay">显示何种审核状态的数据</param>
        public override BlockPagingID GetFileThreadIDsSortBy(FileThreadSortBy sortBy, int? categoryID, string tagName, bool? isEssential, int maxRecords, PrivacyStatusesForDisplay? privacyStatusForDisplay, AuditingStatusesForDisplay? auditingStatusForDisplay)
        {
            using (SqlConnection connection = GetReadOnlyConnection())
            {
                using (SqlCommand command = new SqlCommand(databaseOwner + ".spb_Common_GetPagingRecordsWithNotIn", connection))
                {
                    tagName = StringUtils.StripSQLInjection(tagName);

                    command.CommandType = CommandType.StoredProcedure;

                    StringBuilder fromClause = new StringBuilder(string.Format("{0}.spb_FileThreads T", databaseOwner));

                    StringBuilder whereClause = new StringBuilder();

                    if (!string.IsNullOrEmpty(tagName))
                    {
                        //fromClause.AppendFormat("  join {0}.spb_FilesInUserTags IUT on IUT.ItemID = T.ThreadID inner join {0}.spb_FileUserTags UT on UT.TagName = IUT.TagName ", databaseOwner);
                        fromClause.AppendFormat("  join {0}.spb_FilesInUserTags IUT on IUT.ItemID = T.ThreadID", databaseOwner);
                        whereClause.AppendFormat(" and IUT.TagName = N'{0}' ", tagName);
                    }

                    if (isEssential.HasValue)
                    {
                        whereClause.AppendFormat(" and IsEssential={0} ", isEssential.Value ? 1 : 0);
                    }

                    if (privacyStatusForDisplay.HasValue)
                    {
                        switch (privacyStatusForDisplay.Value)
                        {
                            case PrivacyStatusesForDisplay.Privacy:
                            case PrivacyStatusesForDisplay.NeedPassword:
                            case PrivacyStatusesForDisplay.OnlyFriend:
                            case PrivacyStatusesForDisplay.Public:
                                whereClause.AppendFormat(" and PrivacyStatus={0} ", (int)privacyStatusForDisplay);
                                break;
                            case PrivacyStatusesForDisplay.NeedPassword_GreaterThanOrEqual:
                            case PrivacyStatusesForDisplay.OnlyFriend_GreaterThanOrEqual:
                                whereClause.AppendFormat(" and PrivacyStatus > {0} ", (int)privacyStatusForDisplay);
                                break;
                            default:
                                break;
                        }
                    }
                    if (auditingStatusForDisplay != null)
                    {
                        switch (auditingStatusForDisplay)
                        {
                            case AuditingStatusesForDisplay.Fail:
                            case AuditingStatusesForDisplay.Pending:
                            case AuditingStatusesForDisplay.Again:
                            case AuditingStatusesForDisplay.Success:
                                whereClause.AppendFormat(" and AuditingStatus={0} ", (int)auditingStatusForDisplay);
                                break;
                            case AuditingStatusesForDisplay.Pending_GreaterThanOrEqual:
                            case AuditingStatusesForDisplay.Again_GreaterThanOrEqual:
                                whereClause.AppendFormat(" and AuditingStatus > {0} ", (int)auditingStatusForDisplay);
                                break;
                            default:
                                break;
                        }
                    }

                    if (categoryID.HasValue && categoryID.Value > 0)
                    {
                        List<SiteCategory> childCategories = SiteCategories.Instance(ApplicationIDs.Instance().File()).GetAllChilds(categoryID.Value, false);
                        StringBuilder childCategoriesIDs = new StringBuilder();
                        childCategoriesIDs.Append(categoryID);
                        if (childCategories != null && childCategories.Count > 0)
                        {
                            foreach (SiteCategory siteCategory in childCategories)
                            {
                                childCategoriesIDs.AppendFormat(",{0}", siteCategory.CategoryID);
                            }
                        }
                        whereClause.AppendFormat(" and T.SiteCategoryID in ({0}) ", childCategoriesIDs.ToString());
                    }

                    string orderByClause;
                    switch (sortBy)
                    {
                        case FileThreadSortBy.StickyAndPostDate:
                            orderByClause = "SpecialOrder desc, ThreadID desc";
                            break;
                        case FileThreadSortBy.CommentCount:
                            orderByClause = "CommentCount desc";
                            break;
                        case FileThreadSortBy.HitTimes:
                            orderByClause = "HitTimes desc";
                            break;
                        case FileThreadSortBy.DownloadCount:
                            orderByClause = " DownloadCount desc ";
                            break;
                        case FileThreadSortBy.StageHitTimes:
                            orderByClause = " StageHitTimes desc ";
                            break;
                        case FileThreadSortBy.StageDownloadCount:
                            orderByClause = " StageDownloadCount desc ";
                            break;
                        case FileThreadSortBy.PostDate:
                        default:
                            orderByClause = "ThreadID desc";
                            break;
                    }

                    //移除开始的" and "
                    if (whereClause.Length > 0)
                        whereClause.Remove(0, 5);

                    command.Parameters.Add("@PageIndex", SqlDbType.Int).Value = 0;
                    command.Parameters.Add("@PageSize", SqlDbType.Int).Value = maxRecords;
                    command.Parameters.Add("@FromClause", SqlDbType.VarChar, 1024).Value = fromClause.ToString();
                    command.Parameters.Add("@SelectFields", SqlDbType.VarChar, 512).Value = "ThreadID";
                    command.Parameters.Add("@WhereClause", SqlDbType.NVarChar, 4000).Value = whereClause.ToString();
                    command.Parameters.Add("@OrderByClause", SqlDbType.VarChar, 256).Value = orderByClause;
                    command.Parameters.Add("@UniqueField", SqlDbType.VarChar, 512).Value = "ThreadID";
                    command.Parameters.Add("@ReturnRecordCount", SqlDbType.Bit).Value = false;  //一定不要返回记录总数
                    command.Parameters.Add("@MaxRecords", SqlDbType.Int).Value = maxRecords;
                    command.Parameters.Add("@TotalRecords", SqlDbType.Int).Direction = ParameterDirection.Output;

                    List<int> idList = new List<int>();

                    connection.Open();
                    using (SqlDataReader dr = command.ExecuteReader(CommandBehavior.CloseConnection))
                    {
                        while (dr.Read())
                        {
                            idList.Add(Convert.ToInt32(dr["ThreadID"]));
                        }
                        dr.Close();
                    }
                    connection.Close();

                    BlockPagingID blockPagingID = new BlockPagingID(idList);
                    return blockPagingID;
                }
            }
        }

        /// <summary>
        /// 根据Query获取文件主题ID集合
        /// </summary>
        /// <param name="query">文件主题查询对象</param>
        /// <returns>BlockPagingID</returns>
        public override BlockPagingID GetFileThreadIDs(int ownerUserID, string tagName, int? userCategoryID, PrivacyStatusesForDisplay? privacyStatusForDisplay, AuditingStatusesForDisplay? auditingStatusForDisplay, bool readFromWriteDB, int maxRecords, int pageSize, int pageIndex)
        {
            using (SqlConnection connection = readFromWriteDB ? GetWritableConnection() : GetReadOnlyConnection())
            {
                using (SqlCommand command = new SqlCommand(databaseOwner + ".spb_Common_GetPagingRecords", connection))
                {
                    command.CommandType = CommandType.StoredProcedure;

                    StringBuilder fromClause = new StringBuilder();
                    fromClause.AppendFormat("{0}.spb_FileThreads T with (nolock)", databaseOwner);

                    StringBuilder whereClause = new StringBuilder();
                    if (ownerUserID > 0)
                        whereClause.AppendFormat(" and T.OwnerUserID = {0} ", ownerUserID);

                    if (privacyStatusForDisplay.HasValue)
                    {
                        switch (privacyStatusForDisplay.Value)
                        {
                            case PrivacyStatusesForDisplay.Privacy:
                            case PrivacyStatusesForDisplay.NeedPassword:
                            case PrivacyStatusesForDisplay.OnlyFriend:
                            case PrivacyStatusesForDisplay.Public:
                                whereClause.AppendFormat(" and T.PrivacyStatus={0} ", (int)privacyStatusForDisplay);
                                break;
                            case PrivacyStatusesForDisplay.NeedPassword_GreaterThanOrEqual:
                            case PrivacyStatusesForDisplay.OnlyFriend_GreaterThanOrEqual:
                                whereClause.AppendFormat(" and T.PrivacyStatus > {0} ", (int)privacyStatusForDisplay);
                                break;
                            default:
                                break;
                        }
                    }
                    if (auditingStatusForDisplay != null)
                    {
                        switch (auditingStatusForDisplay)
                        {
                            case AuditingStatusesForDisplay.Fail:
                            case AuditingStatusesForDisplay.Pending:
                            case AuditingStatusesForDisplay.Again:
                            case AuditingStatusesForDisplay.Success:
                                whereClause.AppendFormat(" and AuditingStatus={0} ", (int)auditingStatusForDisplay);
                                break;
                            case AuditingStatusesForDisplay.Pending_GreaterThanOrEqual:
                            case AuditingStatusesForDisplay.Again_GreaterThanOrEqual:
                                whereClause.AppendFormat(" and AuditingStatus > {0} ", (int)auditingStatusForDisplay);
                                break;
                            default:
                                break;
                        }
                    }

                    if (!string.IsNullOrEmpty(tagName))
                    {
                        if (ownerUserID > 0)
                        {
                            UserTagManager userTagManager = UserTagManager.Instance(UserTagItemKeys.Instance().File());
                            UserTag userTag = userTagManager.GetUserTag(tagName, ownerUserID, true);
                            if (userTag == null)
                                return new BlockPagingID(new List<int>());
                            else
                            {
                                fromClause.AppendFormat("  join {0}.spb_FilesInUserTags IIUT on IIUT.ItemID = T.ThreadID ", databaseOwner);
                                whereClause.AppendFormat(" and IIUT.TagName = '{0}' ", userTag.TagName);
                            }
                        }
                        else
                        {
                            fromClause.AppendFormat("  join {0}.spb_FilesInUserTags IIUT on IIUT.ItemID = T.ThreadID inner join {0}.spb_FileUserTags UT on UT.TagName = IIUT.TagName ", databaseOwner);
                            whereClause.AppendFormat(" and UT.TagName = N'{0}' ", tagName);
                        }
                    }

                    //if (query.SiteCategoryID > 0)
                    //{
                    //    if (query.IncludeSiteCategoryDescendant)
                    //    {
                    //        List<SiteCategory> childCategories = SiteCategories.Instance(ApplicationIDs.Instance().File()).GetAllChilds(query.SiteCategoryID);
                    //        StringBuilder childCategoriesIDs = new StringBuilder();
                    //        childCategoriesIDs.Append(query.SiteCategoryID);
                    //        if (childCategories != null && childCategories.Count > 0)
                    //        {
                    //            foreach (SiteCategory siteCategory in childCategories)
                    //            {
                    //                childCategoriesIDs.AppendFormat(",{0}", siteCategory.CategoryID);
                    //            }
                    //        }
                    //        whereClause.AppendFormat(" and T.SiteCategoryID in ({0}) ", childCategoriesIDs.ToString());
                    //    }
                    //    else
                    //        whereClause.AppendFormat(" and T.SiteCategoryID={0} ", query.SiteCategoryID);
                    //}

                    if (userCategoryID.HasValue && userCategoryID.Value > 0)
                    {
                        whereClause.AppendFormat(" and T.UserCategoryID={0} ", userCategoryID.Value);
                    }

                    //移除开始的" and "
                    if (whereClause.Length > 0)
                        whereClause.Remove(0, 5);

                    command.Parameters.Add("@PageIndex", SqlDbType.Int).Value = pageIndex - 1;
                    command.Parameters.Add("@PageSize", SqlDbType.Int).Value = pageSize;
                    command.Parameters.Add("@FromClause", SqlDbType.VarChar, 1024).Value = fromClause.ToString();
                    command.Parameters.Add("@SelectFields", SqlDbType.VarChar, 512).Value = "T.ThreadID";
                    command.Parameters.Add("@WhereClause", SqlDbType.NVarChar, 4000).Value = whereClause.ToString();
                    command.Parameters.Add("@SortField", SqlDbType.VarChar, 256).Value = "T.ThreadID";
                    command.Parameters.Add("@SortFieldIsDesc", SqlDbType.Bit).Value = true;
                    command.Parameters.Add("@ReturnRecordCount", SqlDbType.Bit).Value = true;
                    command.Parameters.Add("@MaxRecords", SqlDbType.Int).Value = maxRecords;
                    command.Parameters.Add("@TotalRecords", SqlDbType.Int).Direction = ParameterDirection.Output;
                    command.Parameters.Add("@ReverseOrder", SqlDbType.Bit).Direction = ParameterDirection.Output;

                    List<int> idList = new List<int>();

                    connection.Open();
                    using (SqlDataReader dr = command.ExecuteReader(CommandBehavior.CloseConnection))
                    {
                        while (dr.Read())
                        {
                            idList.Add(Convert.ToInt32(dr["ThreadID"]));
                        }
                        dr.Close();
                    }

                    int totalRecords = Convert.ToInt32(command.Parameters["@TotalRecords"].Value);
                    bool reverseOrder = Convert.ToBoolean(command.Parameters["@ReverseOrder"].Value);
                    connection.Close();

                    if (reverseOrder)
                        idList.Reverse();

                    BlockPagingID blockPagingID = new BlockPagingID(idList, totalRecords);
                    return blockPagingID;
                }
            }
        }
        /// <summary>
        /// 根据Query获取文件主题ID集合(管理员后台使用)
        /// </summary>
        /// <param name="maxRecords"></param>
        /// <param name="pageSize"></param>
        /// <param name="pageIndex"></param>
        /// <param name="auditingStatusForDisplay"></param>
        /// <param name="ownerUserID"></param>
        /// <param name="subjectKeywords"></param>
        /// <returns></returns>
        public override BlockPagingID GetFileThreadIDsForAdmin(int maxRecords, int pageSize, int pageIndex, AuditingStatusesForDisplay? auditingStatusForDisplay, int? ownerUserID, string subjectKeywords)
        {
            using (SqlConnection connection = GetWritableConnection())
            {
                using (SqlCommand command = new SqlCommand(databaseOwner + ".spb_Common_GetPagingRecords", connection))
                {

                    command.CommandType = CommandType.StoredProcedure;

                    StringBuilder fromClause = new StringBuilder();
                    fromClause.AppendFormat("{0}.spb_FileThreads T with (nolock)", databaseOwner);

                    StringBuilder whereClause = new StringBuilder();
                    if (ownerUserID.HasValue && ownerUserID.Value > 0)
                        whereClause.AppendFormat(" and T.OwnerUserID = {0} ", ownerUserID.Value);

                    if (auditingStatusForDisplay != null)
                    {
                        switch (auditingStatusForDisplay)
                        {
                            case AuditingStatusesForDisplay.Fail:
                            case AuditingStatusesForDisplay.Pending:
                            case AuditingStatusesForDisplay.Again:
                            case AuditingStatusesForDisplay.Success:
                                whereClause.AppendFormat(" and AuditingStatus={0} ", (int)auditingStatusForDisplay);
                                break;
                            case AuditingStatusesForDisplay.Pending_GreaterThanOrEqual:
                            case AuditingStatusesForDisplay.Again_GreaterThanOrEqual:
                                whereClause.AppendFormat(" and AuditingStatus > {0} ", (int)auditingStatusForDisplay);
                                break;
                            default:
                                break;
                        }
                    }
                    subjectKeywords = StringUtils.StripSQLInjection(subjectKeywords);
                    if (!string.IsNullOrEmpty(subjectKeywords))
                    {
                        subjectKeywords = StringUtils.StripSQLInjection(subjectKeywords);
                        whereClause.AppendFormat(" and T.Subject like '%{0}%' ", subjectKeywords);
                    }

                    //移除开始的" and "
                    if (whereClause.Length > 0)
                        whereClause.Remove(0, 5);

                    command.Parameters.Add("@PageIndex", SqlDbType.Int).Value = pageIndex - 1;
                    command.Parameters.Add("@PageSize", SqlDbType.Int).Value = pageSize;
                    command.Parameters.Add("@FromClause", SqlDbType.VarChar, 1024).Value = fromClause.ToString();
                    command.Parameters.Add("@SelectFields", SqlDbType.VarChar, 512).Value = "T.ThreadID";
                    command.Parameters.Add("@WhereClause", SqlDbType.NVarChar, 4000).Value = whereClause.ToString();
                    command.Parameters.Add("@SortField", SqlDbType.VarChar, 256).Value = "T.ThreadID";
                    command.Parameters.Add("@SortFieldIsDesc", SqlDbType.Bit).Value = true;
                    command.Parameters.Add("@ReturnRecordCount", SqlDbType.Bit).Value = true;
                    command.Parameters.Add("@MaxRecords", SqlDbType.Int).Value = maxRecords;
                    command.Parameters.Add("@TotalRecords", SqlDbType.Int).Direction = ParameterDirection.Output;
                    command.Parameters.Add("@ReverseOrder", SqlDbType.Bit).Direction = ParameterDirection.Output;

                    List<int> idList = new List<int>();

                    connection.Open();
                    using (SqlDataReader dr = command.ExecuteReader(CommandBehavior.CloseConnection))
                    {
                        while (dr.Read())
                        {
                            idList.Add(Convert.ToInt32(dr["ThreadID"]));
                        }
                        dr.Close();
                    }

                    int totalRecords = Convert.ToInt32(command.Parameters["@TotalRecords"].Value);
                    bool reverseOrder = Convert.ToBoolean(command.Parameters["@ReverseOrder"].Value);
                    connection.Close();

                    if (reverseOrder)
                        idList.Reverse();

                    BlockPagingID blockPagingID = new BlockPagingID(idList, totalRecords);
                    return blockPagingID;
                }
            }
        }


        /// <summary>
        /// 获取FileThread的内容
        /// </summary>
        public override string GetFileThreadBody(int threadID)
        {
            using (SqlConnection myConnection = GetReadOnlyConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThread_Get", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;
                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int).Value = threadID;

                string threadBody = null;
                myConnection.Open();
                using (SqlDataReader dr = myCommand.ExecuteReader(CommandBehavior.CloseConnection))
                {
                    if (dr.Read())
                    {
                        threadBody = Convert.ToString(dr["Body"]);
                    }
                    dr.Close();
                }

                myConnection.Close();

                return threadBody;
            }
        }
        #endregion

        #region siteCategory

        /// <summary>
        /// 批量设置站点类别
        /// </summary>
        public override void UpdateFileSiteCategories(int fromSiteCategoryID, int toSiteCategoryID)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThread_UpdateSiteCategoryByCategoryID", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@fromSiteCategoryID", SqlDbType.Int).Value = fromSiteCategoryID;
                myCommand.Parameters.Add("@toSiteCategoryID", SqlDbType.Int).Value = toSiteCategoryID;

                myConnection.Open();

                myCommand.ExecuteNonQuery();

                myConnection.Close();
            }
        }

        #endregion

        #region FileComment

        /// <summary>
        /// 创建文件评论
        /// </summary>
        /// <param name="comment">要创建的文件</param>
        public override void CreateComment(FileComment comment)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileComment_CreateUpdate", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@Action", SqlDbType.Int).Value = (int)DataProviderAction.Create;
                myCommand.Parameters.Add("@PostID", SqlDbType.Int).Direction = ParameterDirection.Output;
                myCommand.Parameters.Add("@ParentID", SqlDbType.Int).Value = comment.ParentID;
                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int).Value = comment.ThreadID;
                myCommand.Parameters.Add("@OwnerUserID", SqlDbType.Int).Value = comment.OwnerUserID;
                myCommand.Parameters.Add("@UserID", SqlDbType.Int).Value = comment.UserID;
                myCommand.Parameters.Add("@Author", SqlDbType.NVarChar, 256).Value = comment.Author;
                myCommand.Parameters.Add("@Body", SqlDbType.NText).Value = comment.Body;

                SerializerData data = comment.GetSerializerData();
                myCommand.Parameters.Add("@PropertyNames", SqlDbType.NText).Value = data.Keys;
                myCommand.Parameters.Add("@PropertyValues", SqlDbType.NText).Value = data.Values;

                myCommand.Parameters.Add("@AuditingStatus", SqlDbType.Int).Value = comment.AuditingStatus;
                myCommand.Parameters.Add("@UserHostAddress", SqlDbType.NVarChar, 64).Value = comment.UserHostAddress;
                myCommand.Parameters.Add("@IsApproved", SqlDbType.Bit).Value = comment.IsApproved;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myConnection.Close();
                try
                {
                    comment.PostDate = DateTime.Now;
                    comment.PostID = Convert.ToInt32(myCommand.Parameters["@PostID"].Value);
                }
                catch { }
            }
        }

        /// <summary>
        /// 根据PostID获取当个文件评论
        /// </summary>
        public override FileComment GetFileComment(int postID)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileComment_Get", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@PostID", SqlDbType.Int).Value = postID;

                FileComment post = null;

                myConnection.Open();

                using (SqlDataReader dr = myCommand.ExecuteReader(CommandBehavior.CloseConnection | CommandBehavior.SingleRow))
                {
                    if (dr.Read())
                        post = PopulateFileCommentFromIDataReader(dr);

                    dr.Close();
                }

                myConnection.Close();

                return post;
            }
        }

        /// <summary>
        /// 获取文件文章的评论ID集合
        /// </summary>
        /// <param name="auditingStatusForDisplay"></param>
        /// <param name="maxRecords"></param>
        /// <param name="pageSize"></param>
        /// <param name="pageIndex"></param>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public override BlockPagingID GetFileCommentIDsOfThread(AuditingStatusesForDisplay auditingStatusForDisplay, int maxRecords, int pageSize, int pageIndex, int threadID)
        {
            using (SqlConnection connection = GetReadOnlyConnection())
            {
                using (SqlCommand command = new SqlCommand(databaseOwner + ".spb_Common_GetPagingRecords", connection))
                {
                    command.CommandType = CommandType.StoredProcedure;

                    StringBuilder fromClause = new StringBuilder();
                    fromClause.AppendFormat("{0}.spb_FileComments C with (nolock)", databaseOwner);

                    string whereClause = string.Format(" C.ThreadID = {0} ", threadID);

                    switch (auditingStatusForDisplay)
                    {
                        case AuditingStatusesForDisplay.Fail:
                        case AuditingStatusesForDisplay.Pending:
                        case AuditingStatusesForDisplay.Again:
                        case AuditingStatusesForDisplay.Success:
                            whereClause += string.Format(" and AuditingStatus={0} ", (int)auditingStatusForDisplay);
                            break;
                        case AuditingStatusesForDisplay.Pending_GreaterThanOrEqual:
                        case AuditingStatusesForDisplay.Again_GreaterThanOrEqual:
                            whereClause += string.Format(" and AuditingStatus > {0} ", (int)auditingStatusForDisplay);
                            break;
                        default:
                            break;
                    }


                    command.Parameters.Add("@PageIndex", SqlDbType.Int).Value = pageIndex - 1;
                    command.Parameters.Add("@PageSize", SqlDbType.Int).Value = pageSize;
                    command.Parameters.Add("@FromClause", SqlDbType.VarChar, 1024).Value = fromClause.ToString();
                    command.Parameters.Add("@SelectFields", SqlDbType.VarChar, 512).Value = "C.PostID";
                    command.Parameters.Add("@WhereClause", SqlDbType.NVarChar, 4000).Value = whereClause;
                    command.Parameters.Add("@SortField", SqlDbType.VarChar, 256).Value = "C.PostID";
                    command.Parameters.Add("@SortFieldIsDesc", SqlDbType.Bit).Value = true;
                    command.Parameters.Add("@ReturnRecordCount", SqlDbType.Bit).Value = true;
                    command.Parameters.Add("@MaxRecords", SqlDbType.Int).Value = maxRecords;
                    command.Parameters.Add("@TotalRecords", SqlDbType.Int).Direction = ParameterDirection.Output;
                    command.Parameters.Add("@ReverseOrder", SqlDbType.Bit).Direction = ParameterDirection.Output;

                    List<int> idList = new List<int>();

                    connection.Open();
                    using (SqlDataReader dr = command.ExecuteReader(CommandBehavior.CloseConnection))
                    {
                        while (dr.Read())
                        {
                            idList.Add(Convert.ToInt32(dr["PostID"]));
                        }
                        dr.Close();
                    }

                    int totalRecords = Convert.ToInt32(command.Parameters["@TotalRecords"].Value);
                    bool reverseOrder = Convert.ToBoolean(command.Parameters["@ReverseOrder"].Value);
                    connection.Close();

                    if (reverseOrder)
                        idList.Reverse();

                    BlockPagingID blockPagingID = new BlockPagingID(idList, totalRecords);
                    return blockPagingID;
                }
            }
        }


        /// <summary>
        /// 根据Query获取文件评论ID集合
        /// </summary>
        /// <param name="ownerUserID"></param>
        /// <param name="approvalStatus"></param>
        /// <param name="auditingStatusForDisplay"></param>
        /// <param name="readFromWriteDB"></param>
        /// <param name="maxRecords"></param>
        /// <param name="pageSize"></param>
        /// <param name="pageIndex"></param>
        /// <returns>BlockPagingID</returns>
        public override BlockPagingID GetFileCommentIDs(int ownerUserID, ApprovalStatuses? approvalStatus, AuditingStatusesForDisplay? auditingStatusForDisplay, bool readFromWriteDB, int maxRecords, int pageSize, int pageIndex)
        {
            using (SqlConnection connection = readFromWriteDB ? GetWritableConnection() : GetReadOnlyConnection())
            {
                using (SqlCommand command = new SqlCommand(databaseOwner + ".spb_Common_GetPagingRecords", connection))
                {
                    command.CommandType = CommandType.StoredProcedure;

                    StringBuilder fromClause = new StringBuilder();
                    fromClause.AppendFormat("{0}.spb_FileComments C with (nolock)", databaseOwner);

                    StringBuilder whereClause = new StringBuilder();
                    if (ownerUserID > 0)
                        whereClause.AppendFormat(" and C.OwnerUserID = {0} ", ownerUserID);

                    if (approvalStatus.HasValue)
                        whereClause.AppendFormat(" and C.IsApproved = {0} ", (int)approvalStatus.Value);

                    //if (query.UserID > 0)
                    //    whereClause.AppendFormat(" and C.UserID = {0} ", query.UserID);

                    if (auditingStatusForDisplay != null)
                    {
                        switch (auditingStatusForDisplay)
                        {
                            case AuditingStatusesForDisplay.Fail:
                            case AuditingStatusesForDisplay.Pending:
                            case AuditingStatusesForDisplay.Again:
                            case AuditingStatusesForDisplay.Success:
                                whereClause.AppendFormat(" and AuditingStatus={0} ", (int)auditingStatusForDisplay);
                                break;
                            case AuditingStatusesForDisplay.Pending_GreaterThanOrEqual:
                            case AuditingStatusesForDisplay.Again_GreaterThanOrEqual:
                                whereClause.AppendFormat(" and AuditingStatus > {0} ", (int)auditingStatusForDisplay);
                                break;
                            default:
                                break;
                        }
                    }

                    //移除开始的" and "
                    if (whereClause.Length > 0)
                        whereClause.Remove(0, 5);

                    command.Parameters.Add("@PageIndex", SqlDbType.Int).Value = pageIndex - 1;
                    command.Parameters.Add("@PageSize", SqlDbType.Int).Value = pageSize;
                    command.Parameters.Add("@FromClause", SqlDbType.VarChar, 1024).Value = fromClause.ToString();
                    command.Parameters.Add("@SelectFields", SqlDbType.VarChar, 512).Value = "C.PostID";
                    command.Parameters.Add("@WhereClause", SqlDbType.NVarChar, 4000).Value = whereClause.ToString();
                    command.Parameters.Add("@SortField", SqlDbType.VarChar, 256).Value = "C.PostID";
                    command.Parameters.Add("@SortFieldIsDesc", SqlDbType.Bit).Value = true;
                    command.Parameters.Add("@ReturnRecordCount", SqlDbType.Bit).Value = true;
                    command.Parameters.Add("@MaxRecords", SqlDbType.Int).Value = maxRecords;
                    command.Parameters.Add("@TotalRecords", SqlDbType.Int).Direction = ParameterDirection.Output;
                    command.Parameters.Add("@ReverseOrder", SqlDbType.Bit).Direction = ParameterDirection.Output;

                    List<int> idList = new List<int>();

                    connection.Open();
                    using (SqlDataReader dr = command.ExecuteReader(CommandBehavior.CloseConnection))
                    {
                        while (dr.Read())
                        {
                            idList.Add(Convert.ToInt32(dr["PostID"]));
                        }
                        dr.Close();
                    }

                    int totalRecords = Convert.ToInt32(command.Parameters["@TotalRecords"].Value);
                    bool reverseOrder = Convert.ToBoolean(command.Parameters["@ReverseOrder"].Value);
                    connection.Close();

                    if (reverseOrder)
                        idList.Reverse();

                    BlockPagingID blockPagingID = new BlockPagingID(idList, totalRecords);
                    return blockPagingID;
                }
            }
        }




        /// <summary>
        /// 根据Query获取文件评论ID集合(管理员后台使用)
        /// </summary>
        /// <param name="maxRecords"></param>
        /// <param name="pageSize"></param>
        /// <param name="pageIndex"></param>
        /// <param name="auditingStatusForDisplay"></param>
        /// <param name="userID"></param>
        /// <param name="bodyKeywords"></param>
        /// <returns></returns>
        public override BlockPagingID GetFileCommentIDsForAdmin(int maxRecords, int pageSize, int pageIndex, AuditingStatusesForDisplay? auditingStatusForDisplay, int? userID, string bodyKeywords)
        {
            using (SqlConnection connection = GetWritableConnection())
            {
                using (SqlCommand command = new SqlCommand(databaseOwner + ".spb_Common_GetPagingRecords", connection))
                {
                    bodyKeywords = StringUtils.StripSQLInjection(bodyKeywords);
                    command.CommandType = CommandType.StoredProcedure;

                    StringBuilder fromClause = new StringBuilder();
                    fromClause.AppendFormat("{0}.spb_FileComments C with (nolock)", databaseOwner);

                    StringBuilder whereClause = new StringBuilder();

                    if (userID.HasValue && userID.Value > 0)
                        whereClause.AppendFormat(" and C.UserID = {0} ", userID.Value);

                    if (auditingStatusForDisplay != null)
                    {
                        switch (auditingStatusForDisplay)
                        {
                            case AuditingStatusesForDisplay.Fail:
                            case AuditingStatusesForDisplay.Pending:
                            case AuditingStatusesForDisplay.Again:
                            case AuditingStatusesForDisplay.Success:
                                whereClause.AppendFormat(" and AuditingStatus={0} ", (int)auditingStatusForDisplay);
                                break;
                            case AuditingStatusesForDisplay.Pending_GreaterThanOrEqual:
                            case AuditingStatusesForDisplay.Again_GreaterThanOrEqual:
                                whereClause.AppendFormat(" and AuditingStatus > {0} ", (int)auditingStatusForDisplay);
                                break;
                            default:
                                break;
                        }
                    }

                    if (!string.IsNullOrEmpty(bodyKeywords))
                        whereClause.AppendFormat(" and C.Body like '%{0}%' ", bodyKeywords);

                    //移除开始的" and "
                    if (whereClause.Length > 0)
                        whereClause.Remove(0, 5);

                    command.Parameters.Add("@PageIndex", SqlDbType.Int).Value = pageIndex - 1;
                    command.Parameters.Add("@PageSize", SqlDbType.Int).Value = pageSize;
                    command.Parameters.Add("@FromClause", SqlDbType.VarChar, 1024).Value = fromClause.ToString();
                    command.Parameters.Add("@SelectFields", SqlDbType.VarChar, 512).Value = "C.PostID";
                    command.Parameters.Add("@WhereClause", SqlDbType.NVarChar, 4000).Value = whereClause.ToString();
                    command.Parameters.Add("@SortField", SqlDbType.VarChar, 256).Value = "C.PostID";
                    command.Parameters.Add("@SortFieldIsDesc", SqlDbType.Bit).Value = true;
                    command.Parameters.Add("@ReturnRecordCount", SqlDbType.Bit).Value = true;
                    command.Parameters.Add("@MaxRecords", SqlDbType.Int).Value = maxRecords;
                    command.Parameters.Add("@TotalRecords", SqlDbType.Int).Direction = ParameterDirection.Output;
                    command.Parameters.Add("@ReverseOrder", SqlDbType.Bit).Direction = ParameterDirection.Output;

                    List<int> idList = new List<int>();

                    connection.Open();
                    using (SqlDataReader dr = command.ExecuteReader(CommandBehavior.CloseConnection))
                    {
                        while (dr.Read())
                        {
                            idList.Add(Convert.ToInt32(dr["PostID"]));
                        }
                        dr.Close();
                    }

                    int totalRecords = Convert.ToInt32(command.Parameters["@TotalRecords"].Value);
                    bool reverseOrder = Convert.ToBoolean(command.Parameters["@ReverseOrder"].Value);
                    connection.Close();

                    if (reverseOrder)
                        idList.Reverse();

                    BlockPagingID blockPagingID = new BlockPagingID(idList, totalRecords);
                    return blockPagingID;
                }
            }
        }

        /// <summary>
        /// 批准FilePost
        /// </summary>      
        public override void ApproveComments(List<int> postIDs, bool isApproved)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileComment_Approve", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@PostID", SqlDbType.Int);
                myCommand.Parameters.Add("@IsApproved", SqlDbType.Bit);

                myConnection.Open();
                foreach (int postID in postIDs)
                {
                    myCommand.Parameters["@PostID"].Value = postID;
                    myCommand.Parameters["@IsApproved"].Value = isApproved;
                    myCommand.ExecuteNonQuery();
                }
                myConnection.Close();
            }
        }

        /// <summary>
        /// 设置评论审核状态
        /// </summary>
        public override void UpdateCommentAuditingStatus(List<int> postIDs, AuditingStatuses auditingStatus)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileComment_UpdateAuditingStatus", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@PostID", SqlDbType.Int);
                myCommand.Parameters.Add("@AuditingStatus", SqlDbType.Int).Value = (int)auditingStatus;

                myConnection.Open();
                foreach (int postID in postIDs)
                {
                    myCommand.Parameters["@PostID"].Value = postID;
                    myCommand.ExecuteNonQuery();
                }
                myConnection.Close();
            }
        }

        /// <summary>
        /// 删除文件帖子
        /// </summary>
        /// <param name="postID">要删除文件帖子的ID</param>
        public override void DeleteComment(int postID)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileComment_Delete", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@PostID", SqlDbType.Int).Value = postID;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myConnection.Close();
            }
        }

        #endregion

        #region FileDownLoadRecord


        /// <summary>
        /// 创建下载记录
        /// </summary>
        public override void CreateDownLoadRecord(FileDownLoadRecord record)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileDownLoadRecord_Create", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@RecordID", SqlDbType.Int).Direction = ParameterDirection.Output;
                myCommand.Parameters.Add("@UserID", SqlDbType.Int).Value = record.UserID;
                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int).Value = record.ThreadID;
                myCommand.Parameters.Add("@UserDisplayName", SqlDbType.NVarChar, 256).Value = record.UserDisplayName;
                myCommand.Parameters.Add("@Price", SqlDbType.Int).Value = record.Price;
                myCommand.Parameters.Add("@UserHostAddress", SqlDbType.NVarChar, 64).Value = record.UserHostAddress;
                myCommand.Parameters.Add("@FromUrl", SqlDbType.NVarChar, 2000).Value = record.FromUrl;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myCommand.Dispose();
                myConnection.Close();

                record.RecordID = (int)myCommand.Parameters["@RecordID"].Value;
                record.DateCreated = DateTime.Now;

            }
        }
        /// <summary>
        /// 删除用户的所有下载记录
        /// </summary>
        public override void DeleteDownloadRecordsByUserID(int userID)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileDownLoadRecords_DeleteByUserID", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@UserID", SqlDbType.Int).Value = userID;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myCommand.Dispose();
                myConnection.Close();
            }
        }


        /// <summary>
        /// 获取某个用户的下载记录字典
        /// Key=ThreadID
        /// Value=FileDownLoadRecordID
        /// </summary>
        public override Dictionary<int, int> GetThreadIDDownLoadRecordIDsOfUser(int userID)
        {
            using (SqlConnection myConnection = GetReadOnlyConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileDownLoadRecords_GetByUserID", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@UserID", SqlDbType.Int).Value = userID;

                Dictionary<int, int> downLoadRecord = new Dictionary<int, int>();
                myConnection.Open();
                using (SqlDataReader dr = myCommand.ExecuteReader(CommandBehavior.CloseConnection))
                {
                    while (dr.Read())
                    {
                        downLoadRecord[Convert.ToInt32(dr["ThreadID"])] = Convert.ToInt32(dr["RecordID"]);
                    }
                    dr.Close();
                }
                myCommand.Dispose();
                myConnection.Close();
                return downLoadRecord;
            }

        }

        /// <summary>
        /// 获取单条下载记录
        /// </summary>
        public override FileDownLoadRecord GetFileDownLoadRecord(int recordID)
        {
            using (SqlConnection myConnection = GetReadOnlyConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileDownLoadRecord_Get", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@RecordID", SqlDbType.Int).Value = recordID;
                myConnection.Open();
                FileDownLoadRecord downLodRecord = null;
                using (SqlDataReader dr = myCommand.ExecuteReader(CommandBehavior.CloseConnection))
                {

                    if (dr.Read())
                    {
                        downLodRecord = PopulateFileDownLoadRecordFromDataReader(dr);
                    }
                    dr.Close();
                }
                myCommand.Dispose();
                myConnection.Close();
                return downLodRecord;
            }

        }


        /// <summary>
        /// 获取某用户的下载记录分页集合
        /// </summary>
        public override BlockPagingID GetFileDownLoadRecords(int MaxRecords, int pageSize, int pageIndex, int userID)
        {
            using (SqlConnection connection = GetReadOnlyConnection())
            {
                using (SqlCommand command = new SqlCommand(databaseOwner + ".spb_Common_GetPagingRecords", connection))
                {
                    command.CommandType = CommandType.StoredProcedure;
                    string fromClause = string.Format(" {0}.spb_FileDownLoadRecords C with (nolock) ", databaseOwner);
                    string whereClause = string.Empty;
                    if (userID > 0)
                    {
                        whereClause = string.Format(" C.UserID={0} ", userID);
                    }

                    command.Parameters.Add("@PageIndex", SqlDbType.Int).Value = pageIndex - 1;
                    command.Parameters.Add("@PageSize", SqlDbType.Int).Value = pageSize;
                    command.Parameters.Add("@FromClause", SqlDbType.VarChar, 1024).Value = fromClause;
                    command.Parameters.Add("@SelectFields", SqlDbType.VarChar, 512).Value = "C.RecordID";
                    command.Parameters.Add("@WhereClause", SqlDbType.NVarChar, 4000).Value = whereClause;
                    command.Parameters.Add("@SortField", SqlDbType.VarChar, 256).Value = "C.RecordID";
                    command.Parameters.Add("@SortFieldIsDesc", SqlDbType.Bit).Value = true;
                    command.Parameters.Add("@ReturnRecordCount", SqlDbType.Bit).Value = true;
                    command.Parameters.Add("@MaxRecords", SqlDbType.Int).Value = MaxRecords;
                    command.Parameters.Add("@TotalRecords", SqlDbType.Int).Direction = ParameterDirection.Output;
                    command.Parameters.Add("@ReverseOrder", SqlDbType.Bit).Direction = ParameterDirection.Output;

                    List<int> recordIDList = new List<int>();

                    connection.Open();
                    using (SqlDataReader dr = command.ExecuteReader(CommandBehavior.CloseConnection))
                    {
                        while (dr.Read())
                        {
                            recordIDList.Add(Convert.ToInt32(dr["RecordID"]));
                        }
                        dr.Close();
                    }

                    int totalRecords = Convert.ToInt32(command.Parameters["@TotalRecords"].Value);
                    bool reverseOrder = Convert.ToBoolean(command.Parameters["@ReverseOrder"].Value);
                    connection.Close();

                    if (reverseOrder)
                        recordIDList.Reverse();

                    BlockPagingID blockPagingID = new BlockPagingID(recordIDList, totalRecords);
                    return blockPagingID;

                }
            }

        }



        /// <summary>
        /// 获取文件的前N条下载记录
        /// </summary>
        /// <returns>RecordID集合</returns>
        public override List<int> GetTopRecordIDsOfThread(int threadID, int topNumber)
        {
            using (SqlConnection myConnection = GetReadOnlyConnection())
            {
                SqlCommand myCommand = new SqlCommand();
                myCommand.Connection = myConnection;
                myCommand.CommandText = string.Format("select top {0} RecordID from {1}.spb_FileDownLoadRecords with (nolock) where threadID={2} and UserID > 0 order by datecreated desc", topNumber, databaseOwner, threadID);
                List<int> recordIDs = new List<int>();
                myConnection.Open();
                using (SqlDataReader dr = myCommand.ExecuteReader(CommandBehavior.CloseConnection))
                {
                    while (dr.Read())
                    {
                        recordIDs.Add(Convert.ToInt32(dr["RecordID"]));
                    }
                    dr.Close();
                }
                myCommand.Dispose();
                myConnection.Close();
                return recordIDs;
            }

        }


        #endregion

        #region Rating

        /// <summary>
        /// 评价
        /// </summary>        
        /// <remarks>
        /// 如果用户曾经评价过则更新评价结果
        /// </remarks>
        public override void Rate(Rating rating)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileRating_Rate", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ItemID", SqlDbType.Int).Value = rating.ItemID;
                myCommand.Parameters.Add("@UserID", SqlDbType.Int).Value = rating.UserID;
                myCommand.Parameters.Add("@Author", SqlDbType.NVarChar, 64).Value = rating.Author;
                myCommand.Parameters.Add("@Rate", SqlDbType.Int).Value = rating.Rate;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myConnection.Close();
            }
        }

        /// <summary>
        /// 获取针对ItemID的所有评价
        /// </summary>
        public override List<Rating> GetRatings(int itemID, int topNumber)
        {
            using (SqlConnection myConnection = GetReadOnlyConnection())
            {
                SqlCommand myCommand = new SqlCommand();
                myCommand.Connection = myConnection;
                myCommand.CommandText = string.Format("select top {0} * from {1}.spb_FileRatings with (nolock) where ThreadID = {2} ", topNumber, databaseOwner, itemID);
                List<Rating> ratings = new List<Rating>();
                myConnection.Open();
                using (SqlDataReader dr = myCommand.ExecuteReader(CommandBehavior.CloseConnection))
                {
                    while (dr.Read())
                    {
                        ratings.Add(PopulateFileRatingFromIDataReader(dr));
                    }
                    dr.Close();
                }
                myCommand.Dispose();
                myConnection.Close();
                return ratings;
            }
        }

        #endregion

        #region Top & Essential

        /// <summary>
        /// 批量置顶/取消置顶
        /// </summary>
        public override void SetSticky(List<int> threadIDs, int specialOrder)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThread_SetSticky", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int);
                myCommand.Parameters.Add("@SpecialOrder", SqlDbType.Int).Value = specialOrder;

                myConnection.Open();
                foreach (int threadID in threadIDs)
                {
                    myCommand.Parameters["@ThreadID"].Value = threadID;
                    myCommand.ExecuteNonQuery();
                }
                myConnection.Close();
            }
        }

        /// <summary>
        /// 批量设置精华/取消精华
        /// </summary>
        public override void SetEssential(List<int> threadIDs, bool isEssential)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThread_SetEssential", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int);
                if (isEssential)
                    myCommand.Parameters.Add("@IsEssential", SqlDbType.Bit).Value = true;
                else
                    myCommand.Parameters.Add("@IsEssential", SqlDbType.Bit).Value = false;

                myConnection.Open();
                foreach (int threadID in threadIDs)
                {
                    myCommand.Parameters["@ThreadID"].Value = threadID;
                    myCommand.ExecuteNonQuery();
                }
                myConnection.Close();
            }
        }

        #endregion

        #region  delete User & Statistics

        /// <summary>
        ///  删除用户并重设用户内容时处理与文件相关的内容 SQL实现
        /// </summary>
        public override void DeleteUserAndReassignContent(int userID, string reassignUserName)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_File_DeleteUserAndReassignContent", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@UserID", SqlDbType.Int).Value = userID;
                myCommand.Parameters.Add("@ReassignUserName", SqlDbType.NVarChar, 64).Value = reassignUserName;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myConnection.Close();
            }
        }

        /// <summary>
        /// 文件应用相关统计
        /// </summary>
        public override void ExecuteStatistics()
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_Task_File_Statistics", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myConnection.Close();
            }
        }
        /// <summary>
        /// 获取文件统计数据
        /// </summary>
        public override Dictionary<FileManageableCounts, int> GetManageableCounts()
        {
            using (SqlConnection myConnection = GetReadOnlyConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileManageableCounts_Get", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;
                myCommand.Parameters.Add("@PendingThreadCount", SqlDbType.Int).Direction = ParameterDirection.Output;
                myCommand.Parameters.Add("@AgainThreadCount", SqlDbType.Int).Direction = ParameterDirection.Output;
                myCommand.Parameters.Add("@PendingCommentCount", SqlDbType.Int).Direction = ParameterDirection.Output;

                Dictionary<FileManageableCounts, int> dictionary = new Dictionary<FileManageableCounts, int>();
                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myConnection.Close();

                dictionary[FileManageableCounts.PendingThreadCount] = Convert.ToInt32(myCommand.Parameters["@PendingThreadCount"].Value);
                dictionary[FileManageableCounts.AgainThreadCount] = Convert.ToInt32(myCommand.Parameters["@AgainThreadCount"].Value);
                dictionary[FileManageableCounts.PendingCommentCount] = Convert.ToInt32(myCommand.Parameters["@PendingCommentCount"].Value);

                return dictionary;
            }
        }

        #endregion

        #region FileSection点击计数

        /// <summary>
        /// FileSection点击计数更新到数据库
        /// </summary>
        /// <param name="countObjects">FileSection计数队列</param>
        public override void SaveFileSectionCounterQueue(Dictionary<int, int> countObjects)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileSectionView_Add", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ItemID", SqlDbType.Int);
                myCommand.Parameters.Add("@WebViewCount", SqlDbType.Int);

                myConnection.Open();
                foreach (var countObject in countObjects)
                {
                    myCommand.Parameters["@ItemID"].Value = countObject.Key;
                    myCommand.Parameters["@WebViewCount"].Value = countObject.Value;
                    myCommand.ExecuteNonQuery();
                }
                myConnection.Close();
            }
        }

        #endregion

        #region 文件点击计数

        /// <summary>
        /// 文件点击计数更新到数据库
        /// </summary>
        /// <param name="countObjects">文件计数队列</param>
        public override void SaveFileThreadCounterQueue(Dictionary<int, int> rssViewCounts, Dictionary<int, int> webViewCounts, Dictionary<int, int> downloadCountCounts)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileThreadView_Add", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ItemID", SqlDbType.Int);
                myCommand.Parameters.Add("@RssViewCount", SqlDbType.Int);
                myCommand.Parameters.Add("@WebViewCount", SqlDbType.Int);
                myCommand.Parameters.Add("@DownloadCount", SqlDbType.Int);

                myConnection.Open();
                foreach (var countObject in rssViewCounts)
                {
                    myCommand.Parameters["@ItemID"].Value = countObject.Key;
                    myCommand.Parameters["@RssViewCount"].Value = countObject.Value;
                    myCommand.ExecuteNonQuery();
                }
                myCommand.Parameters["@RssViewCount"].Value = 0;
                foreach (var countObject in webViewCounts)
                {
                    myCommand.Parameters["@ItemID"].Value = countObject.Key;
                    myCommand.Parameters["@WebViewCount"].Value = countObject.Value;
                    myCommand.ExecuteNonQuery();
                }
                myCommand.Parameters["@WebViewCount"].Value = 0;
                foreach (var countObject in downloadCountCounts)
                {
                    myCommand.Parameters["@ItemID"].Value = countObject.Key;
                    myCommand.Parameters["@DownloadCount"].Value = countObject.Value;
                    myCommand.ExecuteNonQuery();
                }

                myConnection.Close();
            }
        }

        #endregion

    }
}
