﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.SqlClient;
using SpaceBuilder.Common;
using System.Data;


namespace SpaceBuilder.File
{
    public class SqlFileFeaturedImageDataProvider : FileFeaturedImageDataProvider
    {
        #region Member variables

        private string databaseOwner = "dbo";	// overwrite in web.config
        string writableConnectionString = null;
        string readOnlyConnectionString = null;

        #endregion

        #region Constructor

        /// <summary>
        /// 审核数据提供者构造函数
        /// </summary>
        /// <param name="databaseOwner">数据库用户名</param>
        /// <param name="connectionString">连接字符串</param>
        public SqlFileFeaturedImageDataProvider(string databaseOwner, string writableConnectionString, string readOnlyConnectionString)
        {
            this.databaseOwner = databaseOwner;
            this.writableConnectionString = writableConnectionString;
            this.readOnlyConnectionString = readOnlyConnectionString;
        }
        #endregion

        #region helpers
        /// <summary>
        /// 获取可写数据库连接
        /// </summary>
        /// <returns></returns>
        private SqlConnection GetWritableConnection()
        {
            try
            {
                return new SqlConnection(writableConnectionString);
            }
            catch
            {
                throw new SPBException(ExceptionTypes.Instance().DataProvider(), string.Format("SQL Connection String {0} is invalid.", writableConnectionString));
            }
        }

        /// <summary>
        /// 获取只读数据库连接
        /// </summary>
        /// <returns></returns>
        private SqlConnection GetReadOnlyConnection()
        {
            try
            {
                return new SqlConnection(readOnlyConnectionString);
            }
            catch
            {
                throw new SPBException(ExceptionTypes.Instance().DataProvider(), string.Format("SQL Connection String {0} is invalid.", readOnlyConnectionString));
            }
        }
        #endregion

        /// <summary>
        /// 添加附件
        /// </summary>
        public override void CreateAttachment(Attachment attachment)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileAttachment_Create", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@AttachmentID", SqlDbType.Int).Direction = ParameterDirection.Output;
                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int).Value = attachment.AssociateID;
                myCommand.Parameters.Add("@UserID", SqlDbType.Int).Value = attachment.UserID;
                myCommand.Parameters.Add("@FileName", SqlDbType.NVarChar, 512).Value = attachment.FileName;
                myCommand.Parameters.Add("@FriendlyFileName", SqlDbType.NVarChar, 512).Value = attachment.FriendlyFileName;
                myCommand.Parameters.Add("@ContentType", SqlDbType.NVarChar, 64).Value = attachment.ContentType;
                myCommand.Parameters.Add("@ContentSize", SqlDbType.Int).Value = attachment.ContentSize;
                myCommand.Parameters.Add("@Height", SqlDbType.Int).Value = attachment.Height;
                myCommand.Parameters.Add("@Width", SqlDbType.Int).Value = attachment.Width;
                myCommand.Parameters.Add("@IsRemote", SqlDbType.Bit).Value = attachment.IsRemote;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myCommand.Dispose();
                myConnection.Close();

                attachment.AttachmentID = (int)myCommand.Parameters["@AttachmentID"].Value;
                attachment.DateCreated = DateTime.Now;
            }
        }

        /// <summary>
        /// 删除附件
        /// </summary>
        public override void DeleteAttachment(int attachmentID)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileAttachment_Delete", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@AttachmentID", SqlDbType.Int).Value = attachmentID;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myCommand.Dispose();
                myConnection.Close();
            }
        }

        /// <summary>
        /// 删除AssociateID相关的附件
        /// </summary>
        public override void DeleteAttachmentsByAssociateID(int associateID)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileAttachments_DeleteByThreadID", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int).Value = associateID;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myCommand.Dispose();
                myConnection.Close();
            }
        }

        /// <summary>
        /// 删除OwnerID相关的附件
        /// </summary>
        /// <remarks>
        /// 返回AssociateID集合
        /// </remarks>
        public override List<int> DeleteAttachmentsByOwnerID(int ownerID)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileAttachments_DeleteByOwnerUserID", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@OwnerUserID", SqlDbType.Int).Value = ownerID;
                List<int> list = new List<int>();

                myConnection.Open();
                using (SqlDataReader dr = myCommand.ExecuteReader(CommandBehavior.CloseConnection))
                {
                    while (dr.Read())
                    {
                        list.Add((int)dr["ThreadID"]);
                    }
                    dr.Close();
                }
                myCommand.Dispose();
                myConnection.Close();

                return list;
            }
        }

        /// <summary>
        /// 根据attachmentID获取附件
        /// </summary>
        public override Attachment GetAttachment(int attachmentID)
        {
            using (SqlConnection myConnection = GetReadOnlyConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileAttachment_Get", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@AttachmentID", SqlDbType.Int).Value = attachmentID;

                Attachment attachment = null;
                myConnection.Open();
                using (SqlDataReader dr = myCommand.ExecuteReader(CommandBehavior.CloseConnection))
                {
                    if (dr.Read())
                        attachment = PopulateFileAttachmentFromIDataReader(dr);
                    dr.Close();
                }
                myCommand.Dispose();
                myConnection.Close();

                return attachment;
            }
        }

        /// <summary>
        /// 根据AssociateID获取附件列表
        /// </summary>
        public override List<int> GetAttachmentsByAssociateID(int associateID)
        {
            using (SqlConnection myConnection = GetReadOnlyConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileAttachments_GetByThreadID", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@ThreadID", SqlDbType.Int).Value = associateID;

                List<int> attachmentIDs = new List<int>();
                myConnection.Open();
                using (SqlDataReader dr = myCommand.ExecuteReader(CommandBehavior.CloseConnection))
                {
                    while (dr.Read())
                    {
                        attachmentIDs.Add(Convert.ToInt32(dr["AttachmentID"]));
                    }
                    dr.Close();
                }
                myCommand.Dispose();
                myConnection.Close();

                return attachmentIDs;
            }
        }

        /// <summary>
        /// 通过UserID获取临时附件ID集合
        /// </summary>
        public override List<int> GetTemporaryAttachmentIDsOfOwner(int userID, int? ownerID)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileTemporaryAttachmentIDs_GetByUserID", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@UserID", SqlDbType.Int).Value = userID;

                List<int> attachments = new List<int>();
                myConnection.Open();
                using (SqlDataReader dr = myCommand.ExecuteReader(CommandBehavior.CloseConnection))
                {
                    while (dr.Read())
                    {
                        attachments.Add(Convert.ToInt32(dr["AttachmentID"]));
                    }
                    dr.Close();
                }
                myCommand.Dispose();
                myConnection.Close();

                return attachments;
            }
        }
        /// <summary>
        /// 更新附件的AssociateID
        /// </summary>
        public override void UpdateAssociateIDOfAttachment(int attachmentID, int associateID)
        {
            using (SqlConnection myConnection = GetWritableConnection())
            {
                SqlCommand myCommand = new SqlCommand(databaseOwner + ".spb_FileAttachment_UpdateAssociateID", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;

                myCommand.Parameters.Add("@AttachmentID", SqlDbType.Int).Value = attachmentID;
                myCommand.Parameters.Add("@AssociateID", SqlDbType.Int).Value = associateID;

                myConnection.Open();
                myCommand.ExecuteNonQuery();
                myCommand.Dispose();
                myConnection.Close();
            }
        }
    }
}