﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights  reserved.
// </copyright> 
//------------------------------------------------------------------------------ 


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common.Controllers;

using SpaceBuilder.Web;
using SpaceBuilder.Forum;
using SpaceBuilder.Common;

using System.Web.Mvc;
using System.IO;
using System.Collections;
using SpaceBuilder.Utils;
using System.Web.Routing;
using SpaceBuilder.LuceneSearch;
using SpaceBuilder.Club;
using System.Text.RegularExpressions;


namespace SpaceBuilder.Forum.Controllers
{
    [HttpCompress]
    public class ChannelForumController : ChannelControllerBase
    {
        private int forumApplicationID = ApplicationIDs.Instance().Forum();

        /// <summary>
        /// 论坛首页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Home()
        {
            #region PageTitle & BreadCrumb

            SetPageTitle(true, GetResourceString("Title_ForumIndex", forumApplicationID));

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ForumIndex", forumApplicationID), string.Empty);
            #endregion


            IList<ForumSectionGroup> forumGroups = ForumSectionGroups.GetSiteGroups(HasManagePermission());

            Dictionary<ForumSectionGroup, IList<ForumSection>> dict = new Dictionary<ForumSectionGroup, IList<ForumSection>>();
            foreach (var group in forumGroups)
            {
                IList<ForumSection> sections = ForumSections.GetIndentedSectionsOfGroup(group.GroupID, true);
                dict[group] = sections;
            }
            return this.View("Pages/Forums/Home.aspx", "Masters/Forum.master", dict);
        }

        /// <summary>
        /// 标签地图
        /// </summary>
        public ActionResult ForumsTagMap()
        {
            Header.AddTitle(GetResourceString("Title_TagMap"));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_TagMap"), string.Empty);
            Dictionary<string, KeyValuePair<int, ApplicationTag>> siteTags = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).GetTopApplicationTags(300, ApplicationTagsSortBy.ItemCount, null);
            return View("Pages/Forums/ForumsTagMap.aspx", "Masters/Forum.master", siteTags);
        }

        /// <summary>
        /// 频道标签列表显示
        /// </summary>     
        public ActionResult Control_ListForumsSiteTags(int topNumber, ApplicationTagsSortBy? sortBy)
        {
            Dictionary<string, KeyValuePair<int, ApplicationTag>> siteTags = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).GetTopApplicationTags(topNumber, sortBy ?? ApplicationTagsSortBy.ItemCount, null);
            return PartialView("Controls/Forums/ListForumsSiteTags.ascx", siteTags);
        }

        #region ForumThread

        /// <summary>
        /// 显示新建主题帖页面
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ChannelGuideInstallApplication(ApplicationID = 211)]
        public ActionResult NewForumThread(int? sectionID)
        {
            ForumThread thread = null;
            if (TempData["thread"] != null)
                thread = TempData.Get<ForumThread>("thread");
            else
                thread = new ForumThread();
            if (sectionID.HasValue)
            {
                thread.SectionID = sectionID.Value;

                #region 权限验证
                if (!ForumPermission.AccessCheck(thread.ForumSection, CurrentUser, PermissionItemKeys.Instance().CreateForumThread(), HttpContext, SiteUrls.Instance().Login(true)))
                    return new EmptyResult();
                #endregion

                #region PageTitle & BreadCrumb
                SetPageTitle(true, thread.ForumSection.SectionName);

                SetForumSectionBreadCrumb(thread.ForumSection.SectionID);
                BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_PublishThread", forumApplicationID), string.Empty);
                #endregion

            }
            return this.View("Pages/Forums/EditForumThread.aspx", "Masters/Forum.master", thread);
        }

        /// <summary>
        /// 创建主题帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [ChannelGuideInstallApplication(ApplicationID = 211)]
        public ActionResult CreateForumThread(int? sectionID)
        {

            ForumThread thread = new ForumThread();
            if (sectionID.HasValue && sectionID.Value > 0)
                thread.SectionID = sectionID.Value;
            else
                thread.SectionID = Request.Form.GetInt("sectionID", 0);
            ForumSection section = ForumSections.GetSection(thread.SectionID, true);
            if (section == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ThreadNotExist", forumApplicationID), thread.SectionID)));
            }
            thread.CategoryID = Request.Form.GetInt("categoryID", -1);
            thread.Price = Request.Form.GetInt("Price", 0);
            thread.UserID = CurrentUser.UserID;
            thread.Author = CurrentUser.DisplayName;
            thread.Subject = Request.Form.GetString("subject", string.Empty).Trim();
            thread.Body = Request.Form.GetString("body", string.Empty).Trim();
            thread.IsLocked = Request.Form.GetBool("isLocked", false);
            thread.IsUseSignature = Request.Form.GetBool("isUseSignature", false);
            thread.IsEssential = Request.Form.GetBool("isEssential", false);
            thread.IsSticky = Request.Form.GetBool("isSticky", false);
            int stickyDays = Request.Form.GetInt("stickyDays", 1);
            thread.IsHidden = Request.Form.GetBool("IsHidden", false);
            if (thread.IsSticky)
                thread.StickyDate = DateTime.Now.AddDays(stickyDays);

            if (thread.ForumSection != null && thread.ForumSection.ThreadCategoryStatus == ThreadCategoryStatuses.ForceEnabled && thread.CategoryID <= 0)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_BaseAdminSet", forumApplicationID));
                TempData["thread"] = thread;
                return RedirectToAction("NewForumThread", new RouteValueDictionary { { "sectionID", sectionID } });
            }
            if (string.IsNullOrEmpty(thread.Subject))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_TitleNotNull", forumApplicationID));
                TempData["thread"] = thread;
                return RedirectToAction("NewForumThread", new RouteValueDictionary { { "sectionID", sectionID } });
            }
            if (string.IsNullOrEmpty(thread.Body))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_ContentNotNull", forumApplicationID));
                TempData["thread"] = thread;
                return RedirectToAction("NewForumThread", new RouteValueDictionary { { "sectionID", sectionID } });
            }
            if (!ForumPermission.AccessCheck(section, CurrentUser, PermissionItemKeys.Instance().CreateForumThread(), HttpContext, SiteUrls.Instance().Login(true)))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Message_Pattern_SendTopicLost", forumApplicationID), section.SectionName));
                TempData["thread"] = thread;
                return RedirectToAction("NewForumThread", new RouteValueDictionary { { "sectionID", sectionID } });
            }

            List<string> tagNameList = new List<string>();
            string[] formattedTagNames = Request.Form.GetString("forumTags", string.Empty).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
            if (formattedTagNames != null && formattedTagNames.Length > 0)
            {
                ApplicationTagManager applicationTagManager = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum());
                foreach (string formattedTagName in formattedTagNames)
                {
                    if (applicationTagManager.GetApplicationTag(formattedTagName, true) == null)
                    {
                        ApplicationTag tag = new ApplicationTag();
                        tag.TagName = formattedTagName;
                        tag.UserID = CurrentUser.UserID;
                        tag.Author = CurrentUser.DisplayName;
                        tag.TagName = Formatter.CleanTagName(tag.TagName);
                        tag.TagName = Formatter.FormatSingleLinePlainTextForStorage(tag.TagName);
                        bool isCreateed = applicationTagManager.CreateApplicationTag(tag);
                        if (isCreateed)
                            tagNameList.Add(tag.TagName);
                    }
                    else
                    {
                        tagNameList.Add(ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).GetApplicationTag(formattedTagName, true).TagName);
                    }
                }
            }
            ForumThreads.CreateThread(thread);
            if (thread.ThreadID > 0)
            {
                ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).UpdateApplicationTagsOfItem(thread.ThreadID, tagNameList);

                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_IssueThreadTopicSucceed", forumApplicationID));
                return Redirect(SiteUrls.Instance().ShowForumThread(thread.ThreadID));
            }
            else
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_SendTopicLost", forumApplicationID));
                TempData["thread"] = thread;
                return RedirectToAction("NewForumThread", new RouteValueDictionary { { "sectionID", sectionID } });
            }
        }

        /// <summary>
        /// 显示编辑主题帖页面
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult EditForumThread(int threadID)
        {
            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ThreadTopicNotExist", forumApplicationID), threadID)));
            }


            string exceptionType = ExceptionTypes.Instance().UnknownError();
            if (CurrentUser.UserID != thread.UserID && !ForumPermission.Validate(thread.ForumSection, CurrentUser, PermissionItemKeys.Instance().EditForumPost(), ref exceptionType))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Message_Pattern_NotEditLimit", forumApplicationID), StringUtils.Trim(thread.Subject, 14)));
                return Redirect(SiteUrls.Instance().ShowForumThread(threadID));
            }

            #region PageTitle & BreadCrumb
            SetPageTitle(GetResourceString("Title_EditThread", forumApplicationID));

            SetForumSectionBreadCrumb(thread.SectionID);
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_EditThread", forumApplicationID), string.Empty);
            #endregion

            string tagNames = string.Empty;
            foreach (var tag in thread.Tags)
            {
                tagNames += tag + ",";
            }
            ViewData["SelectedUserTags"] = tagNames;
            return this.View("Pages/Forums/EditForumThread.aspx", "Masters/Forum.master", thread);
        }

        /// <summary>
        /// 更新主题帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateForumThread(int threadID)
        {
            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ThreadNotExist", forumApplicationID), threadID)));
            }
            ForumSection section = ForumSections.GetSection(thread.SectionID, true);
            string exceptionType = ExceptionTypes.Instance().UnknownError();
            if (CurrentUser.UserID != thread.UserID && !ForumPermission.Validate(section, CurrentUser, PermissionItemKeys.Instance().EditForumPost(), ref exceptionType))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Message_Pattern_UpdateThreadLost", forumApplicationID), section.SectionName));
                return Redirect(SiteUrls.Instance().ShowForumThread(threadID));
            }
            thread.CategoryID = Request.Form.GetInt("categoryID", 0);
            thread.Price = Request.Form.GetInt("Price", 0);
            thread.Subject = Request.Form.GetString("subject", string.Empty).Trim();
            thread.Body = Request.Form.GetString("body", string.Empty).Trim();
            thread.IsLocked = Request.Form.GetBool("isLocked", false);
            thread.IsUseSignature = Request.Form.GetBool("isUseSignature", false);
            thread.IsEssential = Request.Form.GetBool("isEssential", false);
            thread.IsSticky = Request.Form.GetBool("isSticky", false);
            int stickyDays = Request.Form.GetInt("stickyDays", 1);
            thread.IsHidden = Request.Form.GetBool("IsHidden", false);
            if (thread.IsSticky)
                thread.StickyDate = DateTime.Now.AddDays(stickyDays);

            if (thread.ForumSection != null && thread.ForumSection.ThreadCategoryStatus == ThreadCategoryStatuses.ForceEnabled && thread.CategoryID <= 0)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_BaseAdminSet", forumApplicationID));
                return RedirectToAction("EditForumThread", new RouteValueDictionary { { "threadID", threadID } });
            }

            if (string.IsNullOrEmpty(thread.Subject))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_TitleNotNull", forumApplicationID));
                return RedirectToAction("EditForumThread", new RouteValueDictionary { { "threadID", threadID } });
            }
            if (string.IsNullOrEmpty(thread.Body))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_ContentNotNull", forumApplicationID));
                return RedirectToAction("EditForumThread", new RouteValueDictionary { { "threadID", threadID } });
            }

            List<string> tagNameList = new List<string>();
            string[] formattedTagNames = Request.Form.GetString("forumTags", string.Empty).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
            if (formattedTagNames != null && formattedTagNames.Length > 0)
            {

                foreach (string formattedTagName in formattedTagNames)
                {
                    if (ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).GetApplicationTag(formattedTagName, true) == null)
                    {
                        ApplicationTag tag = new ApplicationTag();
                        tag.TagName = formattedTagName;
                        tag.UserID = CurrentUser.UserID;
                        tag.Author = CurrentUser.DisplayName;
                        tag.TagName = Formatter.CleanTagName(tag.TagName);
                        tag.TagName = Formatter.FormatSingleLinePlainTextForStorage(tag.TagName);

                        bool isCreated = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).CreateApplicationTag(tag);
                        if (isCreated)
                            tagNameList.Add(tag.TagName);
                    }
                    else
                    {
                        tagNameList.Add(ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).GetApplicationTag(formattedTagName, true).TagName);
                    }
                }
            }

            ForumThreads.UpdateThread(thread);
            ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).UpdateApplicationTagsOfItem(thread.ThreadID, tagNameList);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdateThreadTopicSucceed", forumApplicationID));
            return Redirect(SiteUrls.Instance().ShowForumThread(thread.ThreadID));
        }

        /// <summary>
        /// 删除单个主题帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteForumThread(int threadID)
        {

            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ThreadNotExist", forumApplicationID), threadID)));
            }

            ForumSection section = ForumSections.GetSection(thread.SectionID, false);
            string exceptionType = ExceptionTypes.Instance().UnknownError();
            if (CurrentUser.UserID != thread.UserID && !ForumPermission.Validate(section, CurrentUser, PermissionItemKeys.Instance().DeleteForumPost(), ref exceptionType))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Message_Pattern_NotDeleteLimit", forumApplicationID), StringUtils.Trim(thread.Subject, 14)));
                return Redirect(SiteUrls.Instance().ShowForumThread(thread.ThreadID));
            }
            int sectionID = thread.SectionID;
            ForumThreads.DeleteThread(threadID);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteThreadTopicSucceed", forumApplicationID));
            return Redirect(SiteUrls.Instance().ShowForumSection(sectionID));
        }

        /// <summary>
        /// 显示主题
        /// </summary>
        /// <param name="replyUserID">回帖人UserID(例如：只看楼主)</param>        
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShowForumThread(int threadID, int? replyUserID, bool? isDesc, int? pageIndex)
        {
            ForumThread thread = ForumThreads.GetThread(threadID, false);
            if (thread == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ThreadIDNotExist", forumApplicationID), threadID)));
            ForumSectionGroup group = ForumSectionGroups.GetGroup(thread.GroupID, false);
            if (group == null || group.PresentAreaID != PresentAreaIDs.Instance().Channel())
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ThreadIDNotExist", forumApplicationID), threadID)));

            if (!ForumPermission.AccessCheck(thread.ForumSection, CurrentUser, PermissionItemKeys.Instance().ViewForumThread(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            #region 处理审核

            int auditingStatusValue = (int)Applications.GetApplication(ApplicationIDs.Instance().Forum()).AuditingStatusForPublicDisplay;

            if ((int)thread.AuditingStatus < auditingStatusValue)
            {
                if (thread.AuditingStatus != AuditingStatuses.Success)
                    ViewData["errorMessage"] = GetResourceString("Message_TopicNotPass", forumApplicationID);

                ViewData["errorAuditingStatus"] = true;
            }

            #endregion

            #region PageTitle & BreadCrumb & meta &浏览计数
            SetPageTitle(true, thread.Subject);
            SetForumSectionBreadCrumb(thread.SectionID);
            BreadCrumb.AddBreadCrumbItem(thread.Subject, SiteUrls.Instance().ShowForumThread(thread.ThreadID));
            SetPageMeta(thread);
            //浏览计数           
            CounterFactory.IncreaseCount<ForumThreadCounter>(IncreasableCountTypes.WebView, threadID);
            #endregion


            bool hasPermission = CurrentUser == null || HasManagePermission() || ForumPermission.Validate(thread.ForumSection, CurrentUser, PermissionItemKeys.Instance().AdministerForum());

            PagingDataSet<ForumPost> pds = ForumPosts.GetPostsOfThread(threadID, replyUserID, isDesc ?? false, pageIndex ?? 1, hasPermission);
            ViewData["forumThread"] = thread;

            int applicationID = ApplicationIDs.Instance().Forum();
            if (CurrentUser != null)
            {
                ViewData["isNeedInstallApplication"] = CurrentUser.UserType == UserTypes.PersonUser && !Applications.IsInstalled(PresentAreaIDs.Instance().UserDomain(), CurrentUser.UserID, applicationID);
            }
            return this.View("Pages/Forums/ShowForumThread.aspx", "Masters/Forum.master", pds);
        }

        /// <summary>
        /// 用于楼层跳转
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="pageIndex"></param>
        /// <param name="floorNum"></param>
        /// <returns></returns>
        public ActionResult ForumElevator(int threadID, string floor)
        {
            ForumThread thread = ForumThreads.GetThread(threadID, false);
            if (thread == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ThreadIDNotExist", forumApplicationID), threadID)));

            int? pageIndex;
            int fNum = 0;
            int.TryParse(floor, out fNum);

            RouteValueDictionary dictionary = new RouteValueDictionary { { "threadID", threadID } };
            string url = SPBUrlHelper.Action("ShowForumThread", "ChannelForum", dictionary);

            if (fNum <= thread.ReplyCount && fNum > 0)
            {
                pageIndex = fNum / ForumConfiguration.Instance().PostsPerPage;

                if (fNum % ForumConfiguration.Instance().PostsPerPage > 0)
                    pageIndex++;

                if (pageIndex > 0)
                    dictionary.Add("pageIndex", pageIndex);
                url = SPBUrlHelper.Action("ShowForumThread", "ChannelForum", dictionary) + "#f_" + fNum;
            }

            return Redirect(url);
        }

        /// <summary>
        /// 更新单个主题的精华状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult SetEssential(int threadID, bool? isEssential)
        {
            if (isEssential == null)
                return new JsonResult();
            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
            {
                return new JsonResult();
            }

            ForumSection section = ForumSections.GetSection(thread.SectionID, false);
            if (!ForumPermission.AccessCheck(section, CurrentUser, PermissionItemKeys.Instance().SetForumThreadEssential(), HttpContext, SiteUrls.Instance().Login(true)))
                return new JsonResult();

            ForumThreads.SetEssential(threadID, isEssential.Value);

            return RedirectToAction("ShowForumThread", new RouteValueDictionary { { "threadID", threadID } });
        }

        /// <summary>
        /// 更新单个主题的锁定状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult SetLock(int threadID, bool? isLocked)
        {

            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_NotFindForumTopicID", forumApplicationID), threadID)));

            ForumSection section = ForumSections.GetSection(thread.SectionID, false);
            if (!ForumPermission.AccessCheck(section, CurrentUser, PermissionItemKeys.Instance().SetForumThreadEssential(), HttpContext, SiteUrls.Instance().Login(true)))
                return Redirect(SiteUrls.Instance().Error404("版块不存在！"));

            ForumThreads.SetLock(threadID, isLocked ?? false);

            return RedirectToAction("ShowForumThread", new RouteValueDictionary { { "threadID", threadID } });
        }

        /// <summary>
        /// 移动单个论坛主题
        /// </summary>
        public ActionResult Control_MoveForumThread(int threadID)
        {
            if (threadID <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_NotFindForumTopicID", forumApplicationID), threadID);
            }

            ViewData["threadIDs"] = threadID;

            return PartialView("Controls/Forums/MoveForumThread.ascx");
        }

        /// <summary>
        /// 设置置顶
        /// </summary>
        public ActionResult Control_SetSticky(int threadID)
        {

            if (threadID <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_NotFindForumTopicID", forumApplicationID), threadID);
            }

            ViewData["threadIDs"] = threadID;

            return PartialView("Controls/Forums/SetSticky.ascx");
        }

        /// <summary>
        /// 设置高亮
        /// </summary>
        public ActionResult Control_SetHighlight(int threadID, int sectionID)
        {

            if (threadID <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_NotFindForumTopicID", forumApplicationID), threadID);
            }

            ViewData["threadIDs"] = threadID;

            return PartialView("Controls/Forums/SetHighlight.ascx");
        }

        /// <summary>
        /// 设置主题类别
        /// </summary>
        public ActionResult Control_SetThreadCategory(int threadID, int sectionID)
        {
            ForumThread thread = ForumThreads.GetThread(threadID, false);
            if (thread == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_NotFindForumTopicID", forumApplicationID), threadID);
            }

            ForumSection section = ForumSections.GetSection(sectionID, false);

            if (section == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_NotFindForumSectionID", forumApplicationID), sectionID);
            }

            if (section.ThreadCategories == null || section.ThreadCategories.Count <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "没有可用的版块类别！";
            }

            ViewData["thread"] = thread;
            ViewData["section"] = section;

            return PartialView("Controls/Forums/SetThreadCategory.ascx");
        }

        /// <summary>
        /// 设置主题分类
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SetThreadCategory()
        {
            int threadID = Request.Form.GetInt("threadID", 0);


            ForumThread thread = ForumThreads.GetThread(threadID, true);
            ForumSection section = ForumSections.GetSection(thread.SectionID, false);
            if (!ForumPermission.AccessCheck(section, CurrentUser, PermissionItemKeys.Instance().SetForumThreadEssential(), HttpContext, SiteUrls.Instance().Login(true)))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("你没有相关操作权限！");
                return PartialView("Controls/Forums/SetThreadCategory.ascx");
            }

            if (thread == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("ID为{0}的主题未找到！", threadID);
                return PartialView("Controls/Forums/SetThreadCategory.ascx");
            }

            int categoryID = Request.Form.GetInt("categoryID", 0);

            if (categoryID <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("ID为{0}的类别不存在！", categoryID);
                return PartialView("Controls/Forums/SetThreadCategory.ascx");
            }

            thread.CategoryID = categoryID;
            ForumThreads.UpdateThread(thread);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "设置主题类别成功！";

            return PartialView("Controls/Forums/SetThreadCategory.ascx");

        }

        /// <summary>
        /// 移动主题
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult MoveForumThread(int threadID)
        {
            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ThreadNotExist", forumApplicationID), threadID)));
            }
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));

            ForumSection section = ForumSections.GetSection(thread.SectionID, false);
            if (!ForumPermission.AccessCheck(section, CurrentUser, PermissionItemKeys.Instance().MoveForumThread(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            int sectionID = Request.Form.GetInt("sectionID", 0);
            if (sectionID <= 0)
                return RedirectToAction("ShowMoveForumThread", new RouteValueDictionary { { "threadID", threadID } });
            ForumThreads.MoveThread(threadID, sectionID);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_MoveThreadSucceed", forumApplicationID));
            return Redirect(SiteUrls.Instance().ShowForumThread(threadID));
        }

        /// <summary>
        ///设置主题图章
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SetThreadStamp()
        {
            int threadID = Request.Form.GetInt("threadID", 0);
            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("主题不存在或已被删除！");
                return PartialView("Controls/Forums/SetForumStamp.ascx");
            }

            ForumSection section = ForumSections.GetSection(thread.SectionID, false);
            if (!ForumPermission.AccessCheck(section, CurrentUser, PermissionItemKeys.Instance().SetForumThreadEssential(), HttpContext, SiteUrls.Instance().Login(true)))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("你没有相关操作权限！");
                return PartialView("Controls/Forums/SetForumStamp.ascx");
            }

            int stampID = Request.Form.GetInt("stampID", 0);

            ForumThreads.SetStamp(thread.ThreadID, stampID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = string.Format("设置成功！");

            return PartialView("Controls/Forums/SetForumStamp.ascx");

        }

        /// <summary>
        /// 显示最新主题、热点主题、精华主题
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ListForumThreads(ForumThreadSortBy? sortBy, bool? isEssential)
        {
            #region PageTitle & BreadCrumb
            SetPageTitle(true, GetResourceString("Title_ForumRank", forumApplicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ForumRank", forumApplicationID), string.Empty);
            #endregion

            ViewData["sortBy"] = sortBy;
            return this.View("Pages/Forums/ListForumThreads.aspx", "Masters/Forum.master");
        }

        public ActionResult Control_ListForumThreads(ForumThreadSortBy? sortBy, bool? isEssential, int? pageIndex)
        {
            PagingDataSet<ForumThread> pds = ForumThreads.GetPagingThreadsSortBy(sortBy ?? ForumThreadSortBy.LastRepliedDate, null, PresentAreaIDs.Instance().Channel(), isEssential, 20, pageIndex ?? 1);

            ViewData["sortBy"] = sortBy.HasValue ? (int)sortBy : (int)ForumThreadSortBy.PostDate;
            ViewData["isEssential"] = isEssential;
            return PartialView("Controls/Forums/ListForumThreads.ascx", pds);
        }

        /// <summary>
        /// 论坛主题列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ListForumThreadsBlock(int displayCount)
        {
            IList<ForumThread> pds = ForumThreads.GetTopThreadsSortBy(ForumThreadSortBy.LastRepliedDate, null, PresentAreaIDs.Instance().Channel(), null, displayCount);
            return this.PartialView("Controls/Forums/ListForumThreadsBlock.ascx", pds);
        }

        /// <summary>
        /// 热门论坛主题
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ListHotForumThreadsBlock(int displayCount)
        {
            IList<ForumThread> forumThreads = ForumThreads.GetTopThreadsSortBy(ForumThreadSortBy.HitTimes, null, PresentAreaIDs.Instance().Channel(), null, displayCount);
            return this.PartialView("Controls/Forums/ListForumThreadsBlock.ascx", forumThreads);
        }

        /// <summary>
        /// 显示最新主题、热点主题、精华主题
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_SideThreadsBlock(ForumThreadSortBy? sortBy, bool? isEssential, int? maxResult, string title)
        {
            string modelTitle = GetResourceString("Title_NewThread", forumApplicationID);

            IList<ForumThread> listThreads = ForumThreads.GetTopThreadsSortBy(sortBy ?? ForumThreadSortBy.HitTimes, null, PresentAreaIDs.Instance().Channel(), isEssential, maxResult ?? 10);

            ViewData["ModelTitle"] = title;
            ViewData["sortBy"] = (int)sortBy;
            ViewData["isEssential"] = isEssential;

            return this.View("Controls/Forums/ListThreads.ascx", listThreads);
        }

        /// <summary>
        /// 主题盖章
        /// </summary>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public ActionResult Control_SetForumStamp(int threadID)
        {
            ForumThread thread = threadID < 0 ? null : ForumThreads.GetThread(threadID, false);
            if (thread == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_NotFindForumTopicID", forumApplicationID), threadID);
            }
            ViewData["forumThread"] = thread;
            ViewData["Stamps"] = StampManager.Instance().GetStamps();
            return this.PartialView("Controls/Forums/SetForumStamp.ascx");
        }

        #endregion

        #region FroumPosts

        /// <summary>
        /// 帖子审核管理时用于查看帖子详细内容
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShowForumPost(int postID)
        {
            ForumPost forumPost = ForumPosts.GetPost(postID, true);
            return PartialView("Controls/Forums/ShowForumPost.ascx", forumPost);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="auditingStatusesForDisplay"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public ActionResult Control_ListForumPosts(AuditingStatusesForDisplay? auditingStatusesForDisplay, int? pageIndex)
        {
            int forumSectionSselect = Request.QueryString.GetInt("forumSectionSselect", -1);
            IList<int> sectionIDs = new List<int>();
            if (forumSectionSselect != -1)
            {
                sectionIDs.Add(forumSectionSselect);
            }
            else if (CurrentUser != null && !CurrentUser.IsContentAdministrator)
                sectionIDs = ForumSections.CanModerateSectionIDs(CurrentUser.UserName);

            ViewData["auditingStatusesForDisplay"] = auditingStatusesForDisplay;
            ViewData["pageIndex"] = pageIndex;
            string KeyWords = Request.QueryString.GetString("KeyWords", string.Empty);
            int autoID = Request.QueryString.GetInt("autoID", 0);

            int selectStatuses = Request.QueryString.GetInt("SelectStatuses", -1);
            PagingDataSet<ForumPost> pds = ForumPosts.GetPostsForManage(KeyWords, null, autoID, sectionIDs, auditingStatusesForDisplay, 15, pageIndex ?? 1);
            return PartialView("Controls/Forums/ListForumPosts.ascx", pds);
        }

        public ActionResult Control_QuickReplyPost(string subject, int postID)
        {
            return PartialView("Controls/Forums/QuickReplyPost.ascx");
        }
        #endregion

        #region ForumSection


        /// <summary>
        /// 创建更新版块
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateUpdateForumSection(int sectionID)
        {
            ForumSection section = ForumSections.GetSection(sectionID, true);

            if (section == null)
                return Redirect(SiteUrls.Instance().Error("版块不存在或已被删除！"));

            if (CurrentUser == null || (!section.IsModerator(CurrentUser.UserID) && !CurrentUser.IsAdministrator && !CurrentUser.IsContentAdministrator))
                return Redirect(SiteUrls.Instance().Error404("页面不存在！"));


            section.SectionName = Request.Form.GetString("forumSectionName", string.Empty).Trim();
            section.Description = Request.Form.GetString("description", string.Empty).Trim();
            section.Rules = Request.Form.GetString("rules", string.Empty).Trim();
            section.EnableAnonymousPosting = Request.Form.GetBool("enableAnonymousPosting", true);
            section.ThreadCategoryStatus = (ThreadCategoryStatuses)Request.Form.GetInt("threadCategoryStatus", (int)section.ThreadCategoryStatus);
            System.Web.HttpPostedFileBase picFile = null;
            if (Request.Files != null && Request.Files.Count > 0)
                picFile = Request.Files[0];

            if (sectionID > 0)
            {
                try
                {
                    ForumSections.Update(section, picFile);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "更新版块操作已成功";
                }
                catch (Exception)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "更新版块失败，请重试";
                }
            }
            else
            {
                try
                {
                    ForumSections.Create(section, picFile);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "创建版块操作已成功";
                }
                catch (Exception)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "创建版块失败，请重试";
                }
            }

            return RedirectToAction("EditForumSection", new RouteValueDictionary { { "sectionID", sectionID } });
        }

        /// <summary>
        /// 编辑版块
        /// </summary>
        /// <param name="sectionID"></param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult EditForumSection(int sectionID)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));
            ForumSection section = ForumSections.GetSection(sectionID, true);
            if (section == null)
                return Redirect(SiteUrls.Instance().Error("版块不存在或已被删除！"));

            if ((!section.IsModerator(CurrentUser.UserID) && !CurrentUser.IsAdministrator && !CurrentUser.IsContentAdministrator))
                return Redirect(SiteUrls.Instance().Error("你没有权限管理该板块！"));

            SetPageTitle("编辑版块");

            return View("Pages/Forums/EditForumSection.aspx", "Masters/Forum.master", section);
        }

        #endregion

        #region ForumGroup

        #endregion

        #region Manage

        #region ForumThread

        /// <summary>
        /// 设置主题高亮
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Control_BatchSetHighlight()
        {
            string strThreadIDs = Request.Form.GetString("threadIDCheckBox", string.Empty);

            if (!string.IsNullOrEmpty(strThreadIDs))
                ViewData["threadIDs"] = strThreadIDs;
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选择要设置高亮的主题！";
            }

            return PartialView("Controls/Forums/SetHighlight.ascx");
        }

        /// <summary>
        /// 批量设置高亮
        /// </summary>
        /// <param name="sectionID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult BatchSetHighlight()
        {

            if (CurrentUser == null)
                return new EmptyResult();

            string[] setEssentialThreadIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("threadIDs", string.Empty)))
            {
                setEssentialThreadIDs = Request.Form.GetString("threadIDs", string.Empty).Split(',');
                if (setEssentialThreadIDs != null && setEssentialThreadIDs.Length > 0)
                {
                    StringBuilder highlightstyle = new StringBuilder();
                    if (Request.Form.GetBool("IsHighlight", false))
                    {
                        if (!string.IsNullOrEmpty(Request.Form.GetString("Fonts", string.Empty)))
                            highlightstyle.Append(Request.Form.GetString("Fonts", string.Empty));

                        if (!string.IsNullOrEmpty(Request.Form.GetString("FontSize", string.Empty)))
                            highlightstyle.Append(Request.Form.GetString("FontSize", string.Empty));

                        if (!string.IsNullOrEmpty(Request.Form.GetString("FontStyle", string.Empty)))
                        {
                            string[] fontStyles = Request.Form.GetString("FontStyle", string.Empty).Split(',');
                            foreach (string fontStyle in fontStyles)
                            {
                                highlightstyle.Append(fontStyle);
                            }
                        }

                        if (!string.IsNullOrEmpty(Request.Form.GetString("FontColor", string.Empty)))
                            highlightstyle.Append("color:" + Request.Form.GetString("FontColor", string.Empty) + ";");

                        if (!string.IsNullOrEmpty(Request.Form.GetString("BGColor", string.Empty)))
                            highlightstyle.Append("background-color:" + Request.Form.GetString("BGColor", string.Empty) + ";");
                    }

                    foreach (string threadID in setEssentialThreadIDs)
                    {
                        int id = 0;
                        int.TryParse(threadID, out id);

                        ForumThread thread = ForumThreads.GetThread(id, true);

                        if (thread != null)
                        {
                            if (!ForumPermission.AccessCheck(thread.ForumSection, CurrentUser, PermissionItemKeys.Instance().HighlightForumThread(), HttpContext, string.Empty))
                            {
                                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                                TempData["StatusMessageContent"] = "你没有该项操作的管理权限！";
                                return PartialView("Controls/Forums/SetHighlight.ascx");
                            }
                        }

                        if (Request.Form.GetBool("IsHighlight", false) && !string.IsNullOrEmpty(highlightstyle.ToString()))
                        {
                            thread.HighlightStyle = "style=\"" + highlightstyle.ToString() + "\"";

                            DateTime highlightDate = DateTime.Now.AddDays(1);

                            highlightDate = Request.Form.Get<DateTime>("HighlightDate", DateTime.Now);
                            thread.HighlightDate = highlightDate;
                        }
                        else
                            thread.HighlightStyle = string.Empty;

                        ForumThreads.UpdateThread(thread);

                    }
                }
            }

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "设置主题高亮成功！";

            return PartialView("Controls/Forums/SetHighlight.ascx");
        }

        /// <summary>
        /// 设置主题置顶
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Control_BatchSetSticky()
        {
            string strThreadIDs = Request.Form.GetString("threadIDCheckBox", string.Empty);

            if (!string.IsNullOrEmpty(strThreadIDs))
                ViewData["threadIDs"] = strThreadIDs;
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选择要置顶的帖子！";
            }

            return PartialView("Controls/Forums/SetSticky.ascx");
        }

        /// <summary>
        /// 移动论坛主题
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Control_MoveForumThreads()
        {
            string strThreadIDs = Request.Form.GetString("threadIDCheckBox", string.Empty);

            if (!string.IsNullOrEmpty(strThreadIDs))
                ViewData["threadIDs"] = strThreadIDs;
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选择要移动的主题！";
            }

            return PartialView("Controls/Forums/MoveForumThread.ascx");
        }

        /// <summary>
        /// 批量设置主题帖的精华状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult BatchSetEssential(bool isEssential)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            string[] setEssentialThreadIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("threadIDCheckBox", string.Empty)))
            {
                setEssentialThreadIDs = Request.Form.GetString("threadIDCheckBox", string.Empty).Split(',');
                if (setEssentialThreadIDs != null && setEssentialThreadIDs.Length > 0)
                {
                    foreach (string threadID in setEssentialThreadIDs)
                    {
                        int id = 0;
                        int.TryParse(threadID, out id);
                        ForumThread thread = ForumThreads.GetThread(id, true);

                        if (thread != null)
                        {
                            if (!ForumPermission.AccessCheck(thread.ForumSection, CurrentUser, PermissionItemKeys.Instance().SetForumThreadEssential(), HttpContext, string.Empty))
                                return Redirect(SiteUrls.Instance().Error("你没有该项操作的管理权限！"));

                            ForumThreads.SetEssential(thread.ThreadID, isEssential);
                        }
                    }
                }
            }

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "设置主题精华成功！";

            return RedirectToAction("ManageForumThreads", CurrentRouteValueDictionary);
        }

        #endregion

        #endregion

        #region Views



        /// <summary>
        /// 显示版块组
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShowForumGroup(BrowseMode? mode, int groupID)
        {

            ForumSectionGroup group = ForumSectionGroups.GetGroup(groupID, false);

            if (group == null || group.PresentAreaID != PresentAreaIDs.Instance().Channel())
                return Redirect(SiteUrls.Instance().Error404());


            #region PageTitle & BreadCrumb

            SetPageTitle(true, group.GroupName);

            BreadCrumb.AddBreadCrumbItem(group.GroupName, string.Empty);
            #endregion

            List<ForumSection> forumSectionsOfGroup = ForumSections.GetRootSectionsOfGroup(group.GroupID, true);

            ViewData["mode"] = mode.HasValue ? (int)mode : (int)BrowseMode.detail;

            return this.View("Pages/Forums/ShowForumGroup.aspx", "Masters/Forum.master", group);
        }

        /// <summary>
        /// 显示版块
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShowForumSection(int sectionID, int? categoryID, bool? isEssential, ForumDateScopes? scopes, ForumThreadSortBy? sortBy, int? pageIndex)
        {
            ForumSection section = ForumSections.GetSection(sectionID, false);

            if (section == null)
                return Redirect(SiteUrls.Instance().Error404("版块不存在！"));

            ForumSectionGroup group = ForumSectionGroups.GetGroup(section.GroupID, false);
            if (group == null || group.PresentAreaID != PresentAreaIDs.Instance().Channel())
                return Redirect(SiteUrls.Instance().Error404("版块不存在！"));

            #region PageTitle & BreadCrumb

            SetPageTitle(true, section.SectionName);
            SetForumSectionBreadCrumb(sectionID);

            #endregion

            bool hasPermission = false;
            if (CurrentUser != null && (section.Moderators.Where(n => n.UserID == CurrentUser.UserID).Count() > 0 || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
                hasPermission = true;

            isEssential = isEssential.HasValue && isEssential.Value ? isEssential : null;
            PagingDataSet<ForumThread> pds = ForumThreads.GetForumThreadsOfSection(sectionID, categoryID, scopes, sortBy ?? ForumThreadSortBy.LastRepliedDate, isEssential, pageIndex ?? 1, hasPermission);

            if (pageIndex == null || pageIndex.Value <= 1)
                ViewData["StickyThreads"] = ForumThreads.GetStickyThreads(sectionID, categoryID, false);

            ViewData["Section"] = section;
            ViewData["scopes"] = (int?)scopes;
            ViewData["sortBy"] = sortBy.HasValue ? (int)sortBy : (int)ForumThreadSortBy.LastRepliedDate;
            return this.View("Pages/Forums/ShowForumSection.aspx", "Masters/Forum.master", pds);
        }

        /// <summary>
        /// 主题管理
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ManageForumThreads(int? groupID, int? sectionID, string keywords, int? pageIndex)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));
            if (!CurrentUser.IsContentAdministrator && !ForumSections.IsModeratorOfAnySection(CurrentUser.UserName))
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());

            int userID = Request.QueryString.GetInt("userID", 0);

            if (!sectionID.HasValue)
            {
                if (!HasManagePermission())
                {
                    IList<int> sectionIDs = ForumSections.CanModerateSectionIDs(CurrentUser.UserName);
                    if (sectionIDs != null && sectionIDs.Count > 0)
                    {
                        ForumSection section = ForumSections.GetSection(sectionIDs[0], false);
                        groupID = section.GroupID;
                        sectionID = section.SectionID;
                    }
                }
            }
            #region PageTitle & BreadCrumb
            SetPageTitle(true, GetResourceString("Title_TopicAuditingManage", forumApplicationID));
            BreadCrumb.AddBreadCrumbItem("主题管理", string.Empty);
            #endregion

            PagingDataSet<ForumThread> pds = ForumThreads.GetThreadsForManage(keywords, groupID, sectionID, userID, PresentAreaIDs.Instance().Channel(), null, 40, pageIndex ?? 1);

            ViewData["section"] = ForumSections.GetSection(sectionID ?? 0, HasManagePermission());

            ViewData["sideSubMenu"] = (int)ForumSubMenus.ManageThread;

            return this.View("Pages/Forums/ManageForumThreads.aspx", "Masters/Forum.master", pds);
        }

        /// <summary>
        /// 帖子审核管理
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ManageForumPosts(AuditingStatusesForDisplay? auditingStatusesForDisplay, int? pageIndex)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));
            if (!CurrentUser.IsContentAdministrator && !ForumSections.IsModeratorOfAnySection(CurrentUser.UserName))
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());

            #region PageTitle & BreadCrumb
            SetPageTitle(true, GetResourceString("Title_TopicAuditingManage", forumApplicationID));

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Forum", forumApplicationID), SiteUrls.Instance().ChannelForumHome());
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_TopicAuditingManage", forumApplicationID), string.Empty);
            #endregion

            if (auditingStatusesForDisplay.HasValue)
                ViewData["auditingStatusesForDisplay"] = auditingStatusesForDisplay;
            int status = Request.QueryString.GetInt("SelectStatuses", -1);
            if (status > 0)
                ViewData["auditingStatusesForDisplay"] = (AuditingStatusesForDisplay)status;

            ViewData["pageIndex"] = pageIndex;
            return this.View("Pages/Forums/ManageForumPosts.aspx", "Masters/Forum.master");
        }

        /// <summary>
        /// 管理论坛板块
        /// </summary>      
        public ActionResult ManageForumSections()
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));
            if (!CurrentUser.IsContentAdministrator && !ForumSections.IsModeratorOfAnySection(CurrentUser.UserName))
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());

            #region PageTitle & BreadCrumb
            SetPageTitle(GetResourceString("Title_ManageForumModuel", ApplicationIDs.Instance().Forum()));

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Forum", forumApplicationID), SiteUrls.Instance().ChannelForumHome());
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_ManageSections", forumApplicationID), string.Empty);
            #endregion

            IList<ForumSection> forumSections = new List<ForumSection>();
            if (CurrentUser != null && (CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
            {
                IList<ForumSectionGroup> groups = ForumSectionGroups.GetSiteGroups(false);
                if (groups != null && groups.Count > 0)
                {
                    foreach (var group in groups)
                    {
                        IList<ForumSection> sections = ForumSections.GetIndentedSectionsOfGroupForAdmin(group.GroupID, true);
                        if (sections != null && sections.Count > 0)
                            forumSections = forumSections.Union(sections).ToList();
                    }
                }
            }
            else
                forumSections = ForumSections.CanModerateSections(CurrentUser.UserName);

            return View("Pages/Forums/ManageForumSections.aspx", "Masters/Forum.master", forumSections);
        }

        /// <summary>
        /// 显示版块的主题分类页
        /// </summary>    
        public ActionResult ManageForumCategories()
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));
            if (!CurrentUser.IsContentAdministrator && !ForumSections.IsModeratorOfAnySection(CurrentUser.UserName))
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());

            #region PageTitle & BreadCrumb
            SetPageTitle(GetResourceString("Label_ThreadCategory", ApplicationIDs.Instance().Forum()));

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Forum", forumApplicationID), SiteUrls.Instance().ChannelForumHome());
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_ManageForumCategories", forumApplicationID), string.Empty);
            #endregion
            string username = CurrentUser.UserName;
            return View("Pages/Forums/ManageForumCategories.aspx", "Masters/Forum.master");
        }



        /// <summary>
        /// 显示新建回复帖页面
        /// </summary>
        /// <param name="parentID">被回复的帖子PostID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [ChannelGuideInstallApplication(ApplicationID = 211)]
        public ActionResult NewForumPost(int? parentID, bool? quote)
        {
            User currentUser = Globals.GetCurrentUser();
            if (currentUser == null && !SiteSettingsManager.GetSiteSettings().EnableAnonymousPosting)
            {

                return Redirect(SiteUrls.Instance().ShowExceptionMessage(ExceptionTypes.Instance().PostReplyAccessDenied(), GetResourceString("Message_NotHavepurview", forumApplicationID)));
            }

            ForumPost parentPost = ForumPosts.GetPost(parentID ?? 0, false);
            if (parentPost == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_PostIDNotExist", forumApplicationID), parentID)));
            }

            ForumThread thread = ForumThreads.GetThread(parentPost.ThreadID, false);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ByReplyTopicNotFound", forumApplicationID), parentPost.ThreadID)));
            }


            if (thread.IsLocked)
                throw new SPBException(ExceptionTypes.Instance().PostLocked());

            if (SiteSettingsManager.GetSiteSettings().EnableAnonymousPosting && !ForumPermission.AccessCheck(thread.ForumSection, CurrentUser, PermissionItemKeys.Instance().ReplyForumThread(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            #region PageTitle & BreadCrumb
            SetPageTitle(true, thread.Subject);

            SetForumSectionBreadCrumb(thread.SectionID);
            BreadCrumb.AddBreadCrumbItem(thread.Subject, SiteUrls.Instance().ShowForumThread(thread.ThreadID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ReplyThread", forumApplicationID), string.Empty);
            #endregion

            ForumPost post = null;
            if (TempData["post"] != null)
                post = TempData.Get<ForumPost>("post");
            if (post == null)
                post = new ForumPost();
            post.ParentID = parentID ?? 0;
            post.SectionID = thread.SectionID;

            int FloorNumber = Request.QueryString.GetInt("FloorNumber", -1);
            string strFloor = string.Empty;
            if (FloorNumber == 0)
                strFloor = GetResourceString("Label_FloorHost", forumApplicationID);
            else if (FloorNumber > 0)
                strFloor = FloorNumber.ToString() + GetResourceString("Label_Floor", forumApplicationID);
            post.Subject = string.Format("{0}：{1} {2}", GetResourceString("Common_Reply"), strFloor, parentPost.Author);

            if (quote.HasValue && quote.Value)
                post.Body = string.Format("[quote user=\"{0}\"]{1}[/quote]", parentPost.Author, parentPost.TrimNoQuoteHtml(200));

            ViewData["section"] = thread.ForumSection;

            return this.View("Pages/Forums/EditForumPost.aspx", "Masters/Forum.master", post);
        }



        /// <summary>
        /// 显示编辑回复帖页面
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult EditForumPost(int postID)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));
            ForumPost post = ForumPosts.GetPost(postID, true);
            if (post == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_ReplyTopicPageNotExist", forumApplicationID))));
            }

            ForumSection section = ForumSections.GetSection(post.SectionID, false);

            if (post.UserID != CurrentUser.UserID &&
            !ForumPermission.AccessCheck(section, CurrentUser, PermissionItemKeys.Instance().EditForumPost(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();
            if (post == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_ReplyTopicPageNotExist", forumApplicationID))));
            }


            #region PageTitle & BreadCrumb
            SetPageTitle(true, post.Subject);

            SetForumSectionBreadCrumb(post.SectionID);
            BreadCrumb.AddBreadCrumbItem(post.Subject, SiteUrls.Instance().ShowForumThread(post.ThreadID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_EditTopic", forumApplicationID), string.Empty);
            #endregion
            ViewData["section"] = section;
            return this.View("Pages/Forums/EditForumPost.aspx", "Masters/Forum.master", post);
        }


        #region SiteTag


        #endregion

        #endregion

        #region Controls

        /// <summary>
        /// 显示某一楼层帖子
        /// </summary>
        public ActionResult Control_ForumPostFloor(int? replyUserID, bool? isDesc, int? postID)
        {
            ForumPost forumPost = ForumPosts.GetPost(postID ?? 0, true);
            if (forumPost == null)
                return new EmptyResult();
            ForumThread forumThread = ForumThreads.GetThread(forumPost.ThreadID, false);
            if (forumThread == null)
                return new EmptyResult();
            ForumSection forumSection = ForumSections.GetSection(forumPost.SectionID, false);
            if (forumSection == null)
                return new EmptyResult();
            ViewData["forumPost"] = forumPost;
            ViewData["forumSection"] = forumSection;
            ViewData["number"] = (forumThread.ReplyCount++) + 1;

            return View("Controls/Forums/ForumPostFloor.ascx");
        }

        /// <summary>
        /// 群组话题Block
        /// </summary>
        public ActionResult Control_ListClubForumBlock(ForumThreadSortBy? sortBy, int? displayCount)
        {
            IList<ForumThread> forumThreads = ForumThreads.GetTopThreadsSortBy(sortBy ?? ForumThreadSortBy.PostDate, null, PresentAreaIDs.Instance().ClubDomain(), null, displayCount ?? 10);
            return PartialView("Controls/Forums/ListClubForumBlock.ascx", forumThreads);
        }

        /// <summary>
        /// 显示快速导航
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_ShowForumNavigation()
        {
            Dictionary<ForumSectionGroup, List<ForumSection>> dicForumNavigation = new Dictionary<ForumSectionGroup, List<ForumSection>>();
            IList<ForumSectionGroup> forumGroups = ForumSectionGroups.GetSiteGroups(false);
            foreach (var group in forumGroups)
            {
                if (group == null)
                    continue;
                List<ForumSection> forumSectionsOfGroup = ForumSections.GetIndentedSectionsOfGroup(group.GroupID, true);
                dicForumNavigation.Add(group, forumSectionsOfGroup);
            }
            return this.View("Controls/Forums/ForumNavigation.ascx", dicForumNavigation);
        }

        /// <summary>
        /// 主题分类列表
        /// </summary>     
        public ActionResult Control_ListForumCategories(int sectionID)
        {
            List<ForumThreadCategory> categories = ForumThreadCategories.GetForumThreadCategories(sectionID, true);
            if (categories == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format("找不到SectionID为{0}的版块下的所有主题分类", sectionID)));
            }
            ViewData["section"] = ForumSections.GetSection(sectionID, true);
            return PartialView("Controls/Forums/ListForumCategories.ascx", categories);
        }
        /// <summary>
        /// 编辑主题分类
        /// </summary>
        public ActionResult Control_EditForumThreadCategory(int sectionID, int? categoryID)
        {
            ForumThreadCategory category = null;
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                category = ForumThreadCategories.GetForumThreadCategory(categoryID.Value, true);
                if (category == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format("找不到要更新CategoryID为{0}的主题分类", categoryID.Value)));
                }
            }
            else if (TempData["category"] != null)
            {
                category = TempData.Get<ForumThreadCategory>("category");
            }
            if (category == null)
            {
                category = new ForumThreadCategory();
                category.SectionID = sectionID;
            }
            return PartialView("Controls/Forums/EditForumThreadCategory.ascx", category);
        }
        /// <summary>
        /// 创建,更新主题分类
        /// </summary>
        public ActionResult CreateUpdateForumThreadCategory(int sectionID, int? categoryID)
        {
            ForumThreadCategory category = null;
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                category = ForumThreadCategories.GetForumThreadCategory(categoryID.Value, true);
                if (category == null)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "抱歉，找不到你要更新的主题分类");
                    return RedirectToAction("Control_EditForumThreadCategory", new RouteValueDictionary { { "sectionID", sectionID }, { "categoryID", categoryID } });
                }
            }
            else
            {
                category = new ForumThreadCategory();
            }
            category.CategoryName = Request.Form.GetString("categoryName", string.Empty);
            category.CategoryDescription = Request.Form.GetString("categoryDescription", string.Empty);
            category.SectionID = sectionID;
            category.DisplayOrder = Request.Form.GetInt("displayOrder", 0);
            if (string.IsNullOrEmpty(category.CategoryName))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "主题分类不能为空");
                TempData["category"] = category;
                return RedirectToAction("Control_EditForumThreadCategory", new RouteValueDictionary { { "sectionID", sectionID }, { "categoryID", categoryID } });
            }
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                ForumThreadCategories.UpdateForumThreadCategory(category);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "更新论坛主题分类操作已成功");
            }
            else
            {
                ForumThreadCategories.CreateForumThreadCategory(category);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "创建论坛主题分类操作已成功");
            }
            return RedirectToAction("Control_EditForumThreadCategory", new RouteValueDictionary { { "sectionID", sectionID }, { "categoryID", categoryID } });
        }
        /// <summary>
        /// 删除主题分类
        /// </summary>
        public ActionResult DeleteForumThreadCategory(int categoryID)
        {
            ForumThreadCategory category = ForumThreadCategories.GetForumThreadCategory(categoryID, true);
            if (category == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format("找不到CategoryID为{0}的主题分类", categoryID)));
            }
            int sectionID = category.SectionID;
            ForumThreadCategories.DeleteForumThreadCategory(category);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "删除论坛主题分类操作已成功");
            return RedirectToAction("Control_ListForumCategories", new RouteValueDictionary { { "sectionID", sectionID } });
        }

        /// <summary>
        /// 显示版块
        /// </summary>
        /// <param name="groupID"></param>
        /// <param name="mode"></param>
        /// <returns></returns>
        public ActionResult Control_ListForumSection(int groupID, BrowseMode? mode)
        {

            List<ForumSection> forumSections = ForumSections.GetRootSectionsOfGroup(groupID, HasManagePermission());

            ViewData["ListID"] = groupID;
            UserCookie usercookie = new UserCookie(CurrentUser, HttpContext);

            if (mode.HasValue)
            {
                ViewData["Mode"] = (int)mode;
                usercookie.SaveForumSectionsListMode((int)mode, groupID);
            }

            return PartialView("Controls/Forums/ListForumSections.ascx", forumSections);
        }

        public ActionResult Control_ListForumSectionsOfSection(int sectionID, BrowseMode mode)
        {
            ForumSection section = ForumSections.GetSection(sectionID, HasManagePermission());
            ViewData["ListID"] = sectionID;
            ViewData["Mode"] = (int)mode;
            return PartialView("Controls/Forums/ListForumSections.ascx", section.ActivedChilds);

        }



        #endregion

        #region Action

        /// <summary>
        /// 批量删除主题帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteForumThreads(int sectionID)
        {
            ForumSection section = ForumSections.GetSection(sectionID, true);
            if (!HasManagePermission())
            {
                if (section == null)
                    return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ThreadNotExist", forumApplicationID), sectionID)));

                if (!CurrentUser.IsContentAdministrator && !section.IsModerator(CurrentUser.UserID))
                    throw new SPBException(ExceptionTypes.Instance().GeneralAccessDenied());
            }

            string[] deleteThreadIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("threadIDCheckBox", string.Empty)))
            {
                deleteThreadIDs = Request.Form.GetString("threadIDCheckBox", string.Empty).Split(',');
                if (deleteThreadIDs != null && deleteThreadIDs.Length > 0)
                {
                    foreach (string threadID in deleteThreadIDs)
                    {
                        int id = 0;
                        int.TryParse(threadID, out id);
                        ForumThread thread = ForumThreads.GetThread(id, true);
                        if (thread != null)
                        {
                            ForumThreads.DeleteThread(thread.ThreadID);
                        }
                    }
                }
            }
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_BatchDeleteSucceed", forumApplicationID));
            return RedirectToAction("ManageForumThreads", CurrentRouteValueDictionary);

        }


        /// <summary>
        /// 批量设置主题帖的置顶状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult BatchSetSticky()
        {
            if (CurrentUser == null)
                return new EmptyResult();

            bool isSticky = Request.Form.GetBool("isSticky", false);

            string[] setStickyThreadIDs = null;
            DateTime stickyDate = DateTime.Now.AddDays(1);
            DateTime.TryParse(Request.Form.GetString("StickyDate", string.Empty), out stickyDate);

            if (!string.IsNullOrEmpty(Request.Form.GetString("threadIDs", string.Empty)))
            {
                setStickyThreadIDs = Request.Form.GetString("threadIDs", string.Empty).Split(',');
                if (setStickyThreadIDs != null && setStickyThreadIDs.Length > 0)
                {
                    foreach (string threadID in setStickyThreadIDs)
                    {
                        int id = 0;
                        int.TryParse(threadID, out id);
                        ForumThread thread = ForumThreads.GetThread(id, true);

                        if (thread != null)
                        {


                            if (!ForumPermission.AccessCheck(thread.ForumSection, CurrentUser, PermissionItemKeys.Instance().StickyForumThread(), HttpContext, string.Empty))
                            {
                                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                                TempData["StatusMessageContent"] = "你没有该项操作的管理权限！";
                                return PartialView("Controls/Forums/SetSticky.ascx");
                            }

                            ForumThreads.SetSticky(thread.ThreadID, isSticky, stickyDate);
                        }
                    }
                }
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "设置置顶成功！";

            return PartialView("Controls/Forums/SetSticky.ascx");
        }

        /// <summary>
        /// 批量设置主题帖的锁定状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult BatchSetLock(bool isLocked)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            string[] setlockThreadIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("threadIDCheckBox", string.Empty)))
            {
                setlockThreadIDs = Request.Form.GetString("threadIDCheckBox", string.Empty).Split(',');
                if (setlockThreadIDs != null && setlockThreadIDs.Length > 0)
                {
                    foreach (string threadID in setlockThreadIDs)
                    {
                        int id = 0;
                        int.TryParse(threadID, out id);
                        ForumThread thread = ForumThreads.GetThread(id, true);
                        if (thread != null)
                        {
                            if (!ForumPermission.AccessCheck(thread.ForumSection, CurrentUser, PermissionItemKeys.Instance().EditForumPost(), HttpContext, string.Empty))
                                return Redirect(SiteUrls.Instance().Error("你没有该项操作的管理权限！"));

                            ForumThreads.SetLock(thread.ThreadID, isLocked);
                        }
                    }
                }
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = isLocked ? "设置主题锁定成功！" : "取消主题锁定成功！";
            return RedirectToAction("ManageForumThreads", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 批量转移主题帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult BatchMoveThreads()
        {
            if (CurrentUser == null)
                return new EmptyResult();

            string[] strThreadIDs = null;
            int toSectionID = Request.Form.GetInt("toSectionID", 0);
            if (toSectionID <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_SelectMoveModuel", forumApplicationID);
                return PartialView("Controls/Forums/MoveForumTopices.ascx");
            }
            if (string.IsNullOrEmpty(Request.Form.GetString("threadIDs", string.Empty)))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_SelectOneThread", forumApplicationID);
                return PartialView("Controls/Forums/MoveForumTopices.ascx");
            }

            strThreadIDs = Request.Form.GetString("threadIDs", string.Empty).Split(',');

            if (strThreadIDs != null)
                foreach (string strThreadID in strThreadIDs)
                {
                    int threadID = -1;
                    int.TryParse(strThreadID, out threadID);
                    ForumThread thread = ForumThreads.GetThread(threadID, true);

                    if (thread != null)
                    {
                        if (!ForumPermission.AccessCheck(thread.ForumSection, CurrentUser, PermissionItemKeys.Instance().MoveForumThread(), HttpContext, string.Empty))
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = "你没有该项操作的管理权限！";
                            return PartialView("Controls/Forums/MoveForumThread.ascx");
                        }

                        ForumThreads.MoveThread(threadID, toSectionID);
                    }
                }

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "移动主题成功！";

            return PartialView("Controls/Forums/MoveForumThread.ascx");
        }
        /// <summary>
        /// 批量设置主题帖类别
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult BatchSetThreadCategory(int categoryID)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            string[] strThreadIDs = Request.Form.GetValues("threadIDCheckBox");
            if (categoryID <= 0)
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_BatchSetType", forumApplicationID));

            if (strThreadIDs == null || strThreadIDs.Length <= 0)
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_SelectOneThread", forumApplicationID));

            foreach (string strThreadID in strThreadIDs)
            {
                int threadID = -1;
                int.TryParse(strThreadID, out threadID);
                ForumThread thread = ForumThreads.GetThread(threadID, true);
                if (thread != null)
                {
                    //TODO:
                    if (!ForumPermission.AccessCheck(thread.ForumSection, CurrentUser, PermissionItemKeys.Instance().EditForumPost(), HttpContext, string.Empty))
                        return Redirect(SiteUrls.Instance().Error("你没有该项操作的管理权限！"));

                    thread.CategoryID = categoryID;
                    ForumThreads.UpdateThread(thread);
                }
            }
            return RedirectToAction("ManageForumThreads", CurrentRouteValueDictionary);
        }


        /// <summary>
        /// 创建论坛回复贴
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [ChannelGuideInstallApplication(ApplicationID = 211)]
        public ActionResult CreateForumPost(int? parentID)
        {
            ForumPost parentPost = ForumPosts.GetPost(parentID ?? 0, false);
            if (parentPost == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_ReplyTopicPageNotExist", forumApplicationID))));
            }
            ForumPost post = new ForumPost();
            post.Subject = Request.Form.GetString("subject", string.Empty).Trim();
            post.Body = Request.Form.GetString("body", string.Empty).Trim();
            post.IsUseSignature = Request.Form.GetBool("isUseSignature", false);

            post.UserID = CurrentUser != null ? CurrentUser.UserID : 0;
            post.Author = SPBHtmlHelper.Instance().UserDisplayName(CurrentUser);

            post.ParentID = parentID ?? 0;
            post.ThreadID = parentPost.ThreadID;
            post.SectionID = parentPost.SectionID;
            ForumSection section = ForumSections.GetSection(parentPost.SectionID, false);
            if (!ForumPermission.AccessCheck(section, CurrentUser, PermissionItemKeys.Instance().ReplyForumThread(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();
            if (string.IsNullOrEmpty(post.Subject))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_TitleNotNull", forumApplicationID));
                TempData["post"] = post;
                return RedirectToAction("NewForumPost", new RouteValueDictionary { { "parentID", parentID } });
            }
            if (string.IsNullOrEmpty(post.Body))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_ContentNotNull", forumApplicationID));
                TempData["post"] = post;
                return RedirectToAction("NewForumPost", new RouteValueDictionary { { "parentID", parentID } });
            }

            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                //检查验证码是否录入正确
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                if (verifyCode != null)
                    verifyCode = verifyCode.Trim();

                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_IdentifyingCodeError"));
                    TempData["post"] = post;
                    return RedirectToAction("NewForumPost", new RouteValueDictionary { { "parentID", parentID } });
                }
            }
            ForumPosts.CreatePost(post);
            if (post.PostID > 0)
            {
                if (post.AuditingStatus == AuditingStatuses.Success)
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_IssueReplySucceed", forumApplicationID));
                else
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_SucceedButAdmin", forumApplicationID));
                return Redirect(SiteUrls.Instance().ShowForumThreadGotoLastPost(post.ThreadID, post.PostID));
            }
            else
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_ReplyLostContactAdmin", forumApplicationID));
                TempData["post"] = post;
                return RedirectToAction("NewForumPost", new RouteValueDictionary { { "parentID", parentID } });
            }
        }

        /// <summary>
        /// 快速回复
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public JsonResult CreateQuickForumPost(int? parentID)
        {
            ForumPost parentPost = ForumPosts.GetPost(parentID ?? 0, false);
            if (parentPost == null)
            {
                return Json(new
                {
                    ok = false,
                    message = SPBHtmlHelper.Instance().StatusMessage(StatusMessageType.Error, string.Format(GetResourceString("Exception_ReplyTopicPageNotExist", forumApplicationID)))
                });
            }

            ForumPost post = new ForumPost();
            post.Subject = Request.Form.GetString("subject", string.Empty).Trim();
            post.Body = Request.Form.GetString("body", string.Empty);
            post.UserID = CurrentUser != null ? CurrentUser.UserID : 0;
            post.Author = SPBHtmlHelper.Instance().UserDisplayName(CurrentUser);

            post.ParentID = parentID ?? 0;
            post.ThreadID = parentPost.ThreadID;
            post.SectionID = parentPost.SectionID;
            ForumSection section = ForumSections.GetSection(parentPost.SectionID, false);
            if (string.IsNullOrEmpty(post.Body))
            {
                return Json(new
                {
                    ok = false,
                    message = SPBHtmlHelper.Instance().StatusMessage(StatusMessageType.Error, string.Format(GetResourceString("Message_ContentNotNull", forumApplicationID)))
                });
            }
            string exceptionType = ExceptionTypes.Instance().PostAccessDenied();
            if (!ForumPermission.Validate(section, CurrentUser, PermissionItemKeys.Instance().ReplyForumThread(), ref exceptionType))
            {
                ExceptionMessage exceptionMessage = ResourceManager.GetExceptionMessage(exceptionType, forumApplicationID);
                string message = GetResourceString("Message_SendTopicLost", forumApplicationID);
                if (exceptionMessage != null)
                    message = exceptionMessage.Body;
                return Json(new
                {
                    ok = false,
                    message = SPBHtmlHelper.Instance().StatusMessage(StatusMessageType.Error, message)
                });
            }
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                //检查验证码是否录入正确
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                if (verifyCode != null)
                    verifyCode = verifyCode.Trim();

                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    return Json(new
                    {
                        ok = false,
                        message = SPBHtmlHelper.Instance().StatusMessage(StatusMessageType.Error, GetResourceString("Message_IdentifyingCodeError"))
                    });
                }
            }

            ForumPosts.CreatePost(post);
            if (post.PostID > 0)
            {
                string message = GetResourceString("Message_IssueReplySucceed", forumApplicationID);
                if (post.AuditingStatus != AuditingStatuses.Success)
                    message = GetResourceString("Message_SucceedButAdmin", forumApplicationID);

                return Json(new
                {
                    ok = true,
                    message = SPBHtmlHelper.Instance().StatusMessage(StatusMessageType.Success, message),
                    postID = post.PostID
                });
            }
            else
            {
                return Json(new
                {
                    ok = false,
                    message = SPBHtmlHelper.Instance().StatusMessage(StatusMessageType.Error, GetResourceString("Message_ReplyLostContactAdmin", forumApplicationID))
                });
            }
        }

        /// <summary>
        /// 更新论坛回复贴
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateForumPost(int postID)
        {
            ForumPost post = ForumPosts.GetPost(postID, true);
            if (post == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_ReplyTopicNotExist", forumApplicationID))));
            }

            post.Subject = Request.Form.GetString("subject", string.Empty).Trim();
            post.Body = Request.Form.GetString("body", string.Empty).Trim();
            post.IsUseSignature = Request.Form.GetBool("isUseSignature", false);
            if (string.IsNullOrEmpty(post.Subject))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_TitleNotNull", forumApplicationID));
                return RedirectToAction("EditForumPost", new RouteValueDictionary { { "postID", postID } });
            }
            if (string.IsNullOrEmpty(post.Body))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_ContentNotNull", forumApplicationID));
                return RedirectToAction("EditForumPost", new RouteValueDictionary { { "postID", postID } });
            }
            ForumPosts.UpdatePost(post);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdateReplySucceed", forumApplicationID));
            return Redirect(SiteUrls.Instance().ShowForumThreadGotoLastPost(post.ThreadID, post.PostID));
        }

        /// <summary>
        /// 删除回复帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteForumPost(int postID, string returnUrl)
        {

            ForumPost post = ForumPosts.GetPost(postID, true);
            if (post == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_ReplyTopicNotExist", forumApplicationID))));
            }

            ForumSection section = ForumSections.GetSection(post.SectionID, true);
            string exceptionType = ExceptionTypes.Instance().UnknownError();
            if (CurrentUser.UserID != post.UserID && !ForumPermission.Validate(section, CurrentUser, PermissionItemKeys.Instance().DeleteForumPost(), ref exceptionType))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Message_Pattern_NotDeleteLimit", forumApplicationID), StringUtils.Trim(post.Subject, 14)));
                return Redirect(returnUrl);
            }

            int threadID = post.ThreadID;
            ForumPosts.DeletePost(postID);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteTopicSucceed", forumApplicationID));
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 批量删除主题帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteForumPosts(int? pageIndex, AuditingStatuses auditingStatusesForDisplay)
        {
            if (!CurrentUser.IsContentAdministrator)
                throw new SPBException(ExceptionTypes.Instance().GeneralAccessDenied());
            string[] deletePostsIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("postIDCheckBox", string.Empty)))
            {
                deletePostsIDs = Request.Form.GetString("postIDCheckBox", string.Empty).Split(',');
                if (deletePostsIDs != null && deletePostsIDs.Length > 0)
                {
                    foreach (string postID in deletePostsIDs)
                    {
                        int id = 0;
                        int.TryParse(postID, out id);
                        ForumPost post = ForumPosts.GetPost(id, true);
                        if (post != null)
                        {
                            ForumPosts.DeletePost(post.PostID);
                        }
                    }
                }
            }
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_BatchDeleteSucceed", forumApplicationID));
            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 购买主题
        /// </summary>
        /// <param name="threadID">主题ID</param>
        public ActionResult ExchangeThreadAction(int threadID)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));

            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
                throw new SPBException(ExceptionTypes.Instance().PostNotFound());

            if (CurrentUser.TradePoints < thread.Price)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Message_Pattern_NeedDealIntegral", forumApplicationID), thread.Price.ToString(), CurrentUser.TradePoints.ToString()));
            }
            else
            {
                ForumThreads.ExchangeThread(CurrentUser, threadID);
                UserPointRecord record = new UserPointRecord();
                record.Description = string.Format(GetResourceString("Description_Pattern_ShopForumThread", forumApplicationID), thread.Subject);
                record.PointCategory = UserPointCategory.Trade;
                record.PointItemName = GetResourceString("Description_ShopForumThread", forumApplicationID);
                record.Points = -thread.Price;
                record.UserID = CurrentUser.UserID;
                SpaceBuilder.Common.Points.CreateUserPointRecord(record);

                //系统扣除税点
                //税率
                int tax = SiteSettingsManager.GetSiteSettings().TransactionTax;
                //税点
                int taxPoints = thread.Price - thread.Price * (100 - tax) / 100;
                Points.ChanageSystemTradePoints(taxPoints, "征收论坛主题交易税", string.Format("{0}购买了论坛主题：{1}", SPBHtmlHelper.Instance().UserDisplayNameLink(CurrentUser, true), thread.Subject));


                UserPointRecord recordl = new UserPointRecord();
                recordl.Description = string.Format(GetResourceString("Description_Pattern_ShopYourThread", forumApplicationID), CurrentUser.DisplayName, thread.Subject);
                recordl.PointCategory = UserPointCategory.Trade;
                recordl.PointItemName = GetResourceString("Description_ByShopForumThread", forumApplicationID);
                recordl.Points = thread.Price - taxPoints;
                recordl.UserID = thread.UserID;
                SpaceBuilder.Common.Points.CreateUserPointRecord(recordl);
            }
            return RedirectToAction("ShowForumThread", new { threadID = threadID });
        }

        /// <summary>
        /// 购买附件
        /// </summary>
        public ActionResult ExchangeAttachmentAction(int attachmentID)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));

            Attachment attach = ForumAttachmentManager.Instance().GetAttachment(attachmentID, false);
            if (attach == null)
                throw new ArgumentException(GetResourceString("Exception_SlaveNotExist", forumApplicationID));

            if (CurrentUser.TradePoints < attach.Price)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Message_Pattern_NeedDealIntegral", forumApplicationID), attach.Price.ToString(), CurrentUser.TradePoints.ToString()));
            }
            else
            {
                ForumAttachmentManager.Instance().ExchangeAttachment(CurrentUser, attachmentID);
            }
            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 论坛全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Search(int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_ForumThreadSearch", forumApplicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ForumThreadSearch", forumApplicationID), string.Empty);
            ForumPostFullTextQuery query = new ForumPostFullTextQuery();
            query.ConvertFromQueryString(Request);
            query.PageSize = 10;

            if (pageIndex.HasValue)
                query.PageIndex = pageIndex.Value;

            SearchResultDataSet<ForumThread> results = null;
            if (query.IsValid())
            {
                results = ForumSearchManager.Instance().Search(query);
            }

            ViewData["Query"] = query;
            return View("Pages/Forums/Search.aspx", "Masters/Forum.master", results);
        }

        /// <summary>
        /// 论坛高级搜索
        /// </summary>
        public ActionResult Search()
        {
            RouteValueDictionary parameters = new RouteValueDictionary();
            if (!string.IsNullOrEmpty(Request.Form.GetString("Keyword", string.Empty)))
                parameters["Keyword"] = Request.Form.GetString("Keyword", string.Empty).Trim();
            if (Request.Form.GetInt("SectionID", -1) > 0)
                parameters["SectionID"] = Request.Form.GetInt("SectionID", -1);
            if (!string.IsNullOrEmpty(Request.Form.GetString("Author", string.Empty)))
                parameters["Author"] = Request.Form.GetString("Author", string.Empty).Trim();
            if (!string.IsNullOrEmpty(Request.Form.GetString("TagName", string.Empty)))
                parameters["TagName"] = Request.Form.GetString("TagName", string.Empty).Trim();
            if (!string.IsNullOrEmpty(Request.Form.GetString("DateScopes", string.Empty)))
                parameters["DateScopes"] = Request.Form.GetInt("DateScopes", (int)ForumDateScopes.ThreeMonth);

            int userID = Request.Form.GetInt("UserID", -1);
            if (userID > 0)
                parameters["UserID"] = userID.ToString();

            int searchScopeClubID = Request.Form.GetInt("SearchScopeClubID", -1);
            if (searchScopeClubID > 0)
                parameters["SearchScopeClubID"] = searchScopeClubID.ToString();

            return RedirectToAction("Search", parameters);


        }

        public ActionResult UpdateForumAuditingStatus(int threadID)
        {
            List<int> threadIDs = new List<int>() { threadID };
            ForumThreads.UpdateThreadsAuditingStatus(threadIDs, AuditingStatuses.Success);
            return RedirectToAction("ShowForumThread", new RouteValueDictionary { { "threadID", threadID } });
        }


        #endregion

        #region CommonControls

        /// <summary>
        /// 批量批准论坛帖子
        /// </summary>    
        public ActionResult SetForumAuditingStatus(AuditingStatuses auditingStatuses, int? pageIndex, AuditingStatuses auditingStatusesForDisplay, string KeyWords, int? autoID)
        {
            if (!CurrentUser.IsContentAdministrator && !ForumSections.IsModeratorOfAnySection(CurrentUser.UserName))
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }

            if (!string.IsNullOrEmpty(Request.Form.GetString("postIDCheckBox", string.Empty)))
            {
                string[] forumPostIDs = Request.Params.GetString("postIDCheckBox", string.Empty).Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);

                if (forumPostIDs != null && forumPostIDs.Length > 0)
                {
                    List<int> postIDs = ValueHelper.ParseInt(forumPostIDs);
                    ForumPosts.UpdatePostsAuditingStatus(postIDs, auditingStatuses);
                }
            }
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_TopicPass", forumApplicationID));
            return RedirectToAction("ManageForumPosts", new RouteValueDictionary { { "auditingStatusesForDisplay", auditingStatusesForDisplay }, { "pageIndex", pageIndex }, { "KeyWords", KeyWords }, { "autoID", autoID } });
        }

        /// <summary>
        /// 批准论坛帖子
        /// </summary>
        public ActionResult SetForumPostAuditingStatus(int postID, string returnUrl)
        {
            if (!CurrentUser.IsContentAdministrator && !ForumSections.IsModeratorOfAnySection(CurrentUser.UserName))
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }

            List<int> postIDs = new List<int>() { postID };
            ForumPosts.UpdatePostsAuditingStatus(postIDs, AuditingStatuses.Success);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_TopicPass", forumApplicationID));
            return Redirect(returnUrl);
        }

        #endregion

        #region ForumAttachment
        /// <summary>
        /// 上传论坛附件
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_UploadForumAttachment(int? postID, int? sectionID, bool? isImage, string queueCompleteCallback)
        {
            UploadAttachmentSettings settings = new UploadAttachmentSettings();
            ForumAttachmentSettings attachmentSettings = (ForumAttachmentSettings)ForumAttachmentManager.Instance().Settings;
            string uploadPhotoTypes = ForumAttachmentManager.Instance().Settings.Extensions;
            IList<string> imageExtensions = MimeTypeConfiguration.GetImageExtensions();
            StringBuilder newUploadPhotoTypes = new StringBuilder();
            string[] types = uploadPhotoTypes.Split(',');
            foreach (string type in types)
            {
                if (!string.IsNullOrEmpty(type))
                {
                    if (isImage.HasValue && isImage.Value == imageExtensions.Contains(type.ToLower()))
                        newUploadPhotoTypes.AppendFormat("*.{0};", type);
                }
            }
            settings.AssociateID = postID ?? 0;
            settings.OwnerID = sectionID ?? 0;
            settings.Attributes = "";
            settings.FileTypes = newUploadPhotoTypes.ToString();
            settings.UploadUrl = Globals.GetFullUrlInFileHandlerSite(ForumAttachmentManager.Instance().StorageProviderName, WebUtils.ResolveUrl("~/Services/Forum/UploadAttachments.ashx"));
            settings.FlashUrl = Globals.GetFullUrlInFileHandlerSite(ForumAttachmentManager.Instance().StorageProviderName, WebUtils.ResolveUrl("~/Utility/UploadFiles/swfupload.swf"));
            settings.FileQueryLimit = attachmentSettings.MaxFileCountATime;
            settings.FileSizeLimit = attachmentSettings.MaxAttachmentSize.ToString();
            settings.QueueCompleteCallback = queueCompleteCallback;
            settings.IsImage = isImage ?? true;

            ViewData["UploadAttachmentSettings"] = settings;
            return View("Controls/Forums/UploadAttachmentInEditer.ascx", settings);
        }

        /// <summary>
        /// 管理论坛主题附件
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ManagePostAttachments(int? postID, bool? isImage)
        {
            if (CurrentUser == null)
                return new EmptyResult();
            ManageThreadAttachmentsModel model = new ManageThreadAttachmentsModel();
            model.IsImage = isImage ?? true;

            if (postID.HasValue && postID.Value > 0)
            {
                ForumPost forumPost = ForumPosts.GetPost(postID.Value, true);
                if (forumPost == null)
                    return new EmptyResult();

                IList<Attachment> attachments = ForumAttachmentManager.Instance().GetAttachmentsByAssociateID(postID.Value, true);
                model.Attachments = attachments;
                model.UserFeaturedImageAttachmentID = 0;
            }
            else
            {
                IList<Attachment> attachments = ForumAttachmentManager.Instance().GetTemporaryAttachmentsOfUser(CurrentUser.UserID);
                model.Attachments = attachments;
            }

            model.GetAttachmentUrl = SiteUrls.Instance().DownloadForumAttachment;
            model.DeleteAttachmentUrl = SiteUrls.Instance().DeleteForumAttachmentUrl;
            model.ShowAttachmentThumbnail = SPBHtmlHelper.Instance().ShowForumAttachmentThumbnail;
            return PartialView("Controls/Forums/ManagePostAttachmentsInEditer.ascx", model);
        }

        public ActionResult UserContentSelector_UpdateForumAttachments(int? postID, int? sectionID, bool isImage)
        {
            return PartialView("Controls/Forums/UserContentSelector_UpdateForumAttachments.ascx");
        }

        #region Manage

        /// <summary>
        /// 删除论坛附件
        /// </summary>
        public ActionResult DeleteForumAttachment(int attachmentID, bool? isImage)
        {
            Attachment attachment = ForumAttachmentManager.Instance().GetAttachment(attachmentID, true);
            if (attachment == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "找不到附件");
                return RedirectToAction("Control_ManagePostAttachments", new RouteValueDictionary { { "postID", attachment.AssociateID }, { "isImage", isImage } });
            }

            if (CurrentUser == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "请先登录");
                return RedirectToAction("Control_ManagePostAttachments", new RouteValueDictionary { { "postID", attachment.AssociateID }, { "isImage", isImage } });
            }

            if (attachment.AssociateID > 0)
            {
                ForumPost forumPost = ForumPosts.GetPost(attachment.AssociateID, true);
                if (forumPost == null)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "找不到附件所属的帖子");
                    return RedirectToAction("Control_ManagePostAttachments", new RouteValueDictionary { { "postID", attachment.AssociateID }, { "isImage", isImage } });
                }
            }

            User attachmentOwner = Users.GetUser(attachment.UserID, false);
            if (attachmentOwner == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "找不到附件的上传者");
                return RedirectToAction("Control_ManagePostAttachments", new RouteValueDictionary { { "postID", attachment.AssociateID }, { "isImage", isImage } });
            }

            if (CurrentUser.UserID != attachment.UserID && !CurrentUser.IsContentAdministrator)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "没有权限，只有上传者和网站管理员才能删除附件");
                return RedirectToAction("Control_ManagePostAttachments", new RouteValueDictionary { { "postID", attachment.AssociateID }, { "isImage", isImage } });
            }

            ForumAttachmentManager.Instance().DeleteAttachment(attachmentID);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "删除成功");
            return RedirectToAction("Control_ManagePostAttachments", new RouteValueDictionary { { "postID", attachment.AssociateID }, { "isImage", isImage } });
        }

        #endregion

        #endregion


        #region Help methods

        /// <summary>
        /// 设置论坛版块面包屑
        /// </summary>
        private void SetForumSectionBreadCrumb(int sectionID)
        {
            Stack<ForumSection> stack = new Stack<ForumSection>();
            ForumSection forumSection = ForumSections.GetSection(sectionID, false);
            if (forumSection != null)
            {
                stack.Push(forumSection);
                if (forumSection.ParentID > 0)
                {
                    int parentID = forumSection.ParentID;

                    while (true)
                    {
                        ForumSection parentSection = ForumSections.GetSection(parentID, false);
                        if (parentSection != null)
                        {
                            stack.Push(parentSection);
                            parentID = parentSection.ParentID;
                        }
                        else
                            break;
                    }
                }
                BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Forum", forumApplicationID), SiteUrls.Instance().ChannelForumHome());
                ForumSectionGroup group = ForumSectionGroups.GetGroup(forumSection.GroupID, false);
                if (group != null)
                    BreadCrumb.AddBreadCrumbItem(group.GroupName, SiteUrls.Instance().ShowForumGroup(group.GroupID));

                while (stack.Count > 0)
                {
                    ForumSection popForumSection = stack.Pop();
                    BreadCrumb.AddBreadCrumbItem(popForumSection.SectionName, SiteUrls.Instance().ShowForumSection(popForumSection.SectionID));
                }
            }
        }

        /// <summary>
        /// 设置论坛主题详细显示页面的meta标签
        /// </summary>
        private void SetPageMeta(ForumThread thread)
        {
            string metaKeyWords = string.Empty;
            string metaDescription = string.Empty;

            //标签
            if (thread.Tags != null && thread.Tags.Count > 0)
            {
                foreach (string tagName in thread.Tags)
                {
                    if (tagName != null)
                    {
                        metaKeyWords = string.Format("{1},{0}", metaKeyWords, tagName);
                    }
                }
            }

            //标题和用户名
            metaKeyWords = metaKeyWords + thread.Subject;
            metaKeyWords = string.Format("{0},{1}", metaKeyWords, thread.Author);

            metaDescription = string.Format("{0},{1}", thread.Subject, HtmlUtils.TrimHtml(thread.Body, 150));

            Header.AddMetaDescription(metaDescription);
            Header.AddMetaKeywords(metaKeyWords);
        }

        /// <summary>
        /// 是否具有管理权限
        /// </summary>
        protected bool HasManagePermission()
        {
            return (CurrentUser != null && (CurrentUser.IsContentAdministrator || CurrentUser.IsAdministrator));
        }

        #endregion
    }

    /// <summary>
    /// 频道论坛二级导航
    /// </summary>
    public enum ChannelForumSubMenu
    {
        //论坛首页
        ForumHome,

        //论坛管理
        ManageForum,

        //论坛导航
        ForumNavigation,

        //话题排行
        ForumTopicsRank
    }

    /// <summary>
    /// 浏览方式
    /// </summary>
    public enum BrowseMode
    {
        //详细
        detail,

        //简要
        medium
    }
}
