﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights  reserved.
// </copyright> 
//------------------------------------------------------------------------------ 

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Routing;
using System.Web.Mvc;
using SpaceBuilder.Common.Controllers;
using SpaceBuilder.Club;
using SpaceBuilder.Common;

using SpaceBuilder.Utils;

using SpaceBuilder.Web;
using System.Text.RegularExpressions;
using System.Web.UI;
using SpaceBuilder.LuceneSearch;

namespace SpaceBuilder.Forum.Controllers
{
    [HttpCompress]
    public class ClubForumController : ClubDomainControllerBase
    {
        private int applicationID = ApplicationIDs.Instance().Forum();

        /// <summary>
        /// 圈子论坛首页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Home(string clubDomainName, ForumThreadSortBy? sortBy, bool? isEssential, int? pageIndex)
        {
            Club.Club club = Clubs.GetClub(clubDomainName, false);

            if (club == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));

            try
            {
                ClubPermission.AccessCheck(club, CurrentUser, PermissionItemKeys.Instance().ViewClub(), HttpContext, SiteUrls.Instance().Login(true));
            }
            catch
            {
                return Redirect(SiteUrls.Instance().Error("没有权限浏览当前群组！"));
            }


            #region PageTitle & BreadCrumb
            string applicationNameInclub = GetResourceString("Title_Topic", applicationID);
            SetPageTitle(applicationNameInclub + GetResourceString("Title_Home", applicationID), clubDomainName);
            BreadCrumb.AddBreadCrumbItem(applicationNameInclub, "");
            #endregion

            bool hasPermission = CurrentUser == null || CurrentUser.UserID == club.ClubOwner.UserID || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator;

            int groupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(club.ClubID);

            PagingDataSet<ForumThread> pds = ForumThreads.GetPagingThreadsSortBy(sortBy ?? ForumThreadSortBy.LastRepliedDate, groupID, PresentAreaIDs.Instance().ClubDomain(), isEssential, 20, pageIndex ?? 1);

            ViewData["SortBy"] = sortBy;
            return this.View("Pages/Forums/Home.aspx", "Masters/Forum.master", pds);
        }

        #region ForumSection

        /// <summary>
        /// 论坛版块
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShowForumSection(string clubDomainName, int sectionID, int? categoryID, bool? isEssential, ForumDateScopes? scopes, ForumThreadSortBy? sortBy, int? pageIndex)
        {
            Club.Club club = Clubs.GetClub(clubDomainName, false);
            if (club == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));

            try
            {
                ClubPermission.AccessCheck(club, CurrentUser, PermissionItemKeys.Instance().ViewClub(), HttpContext, SiteUrls.Instance().Login(true));
            }
            catch
            {
                return Redirect(SiteUrls.Instance().Error("没有权限浏览当前群组！"));
            }

            ForumSection section = ForumSections.GetSection(sectionID, true);
            if (section == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ForumModelNotFound", applicationID), section.SectionName)));

            ForumSectionGroup group = ForumSectionGroups.GetGroup(section.GroupID, false);
            if (group == null || group.PresentAreaID != PresentAreaIDs.Instance().ClubDomain() || group.OwnerID != club.ClubID)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ForumModelNotFound", applicationID), section.SectionName)));

            #region PageTitle & BreadCrumb
            SetPageTitle(section.SectionName, section.SectionName, clubDomainName);
            BreadCrumb.AddBreadCrumbItem(section.SectionName, "");
            #endregion

            bool hasPermission = false;
            if (CurrentUser != null && (ClubMembers.HasClubManagerPermission(club.ClubID, CurrentUser.UserID) || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
                hasPermission = true;

            isEssential = isEssential.HasValue && !isEssential.Value ? null : isEssential;
            PagingDataSet<ForumThread> pds = ForumThreads.GetForumThreadsOfSection(sectionID, categoryID, scopes, sortBy ?? ForumThreadSortBy.LastRepliedDate, isEssential, pageIndex ?? 1, hasPermission);

            if (pageIndex == null || pageIndex.Value <= 1)
                ViewData["StickyThreads"] = ForumThreads.GetStickyThreads(sectionID, categoryID, hasPermission);

            ViewData["Section"] = section;
            ViewData["scopes"] = (int?)scopes;
            ViewData["sortBy"] = sortBy.HasValue ? (int)sortBy : (int)ForumThreadSortBy.LastRepliedDate;

            return this.View("Pages/Forums/ShowForumSection.aspx", "Masters/Forum.master", pds);
        }

        /// <summary>
        /// 显示版块
        /// </summary>
        /// <param name="groupID"></param>
        /// <param name="mode"></param>
        /// <returns></returns>
        public ActionResult Control_ListForumSection(string clubDomainName, int sectionID, BrowseMode mode)
        {

            ForumSection section = ForumSections.GetSection(sectionID, false);
            ViewData["ListID"] = sectionID;
            ViewData["Mode"] = (int)mode;
            return PartialView("Controls/Forums/ListForumSections.ascx", section.ActivedChilds);

        }

        /// <summary>
        /// 显示版块列表模块
        /// </summary>
        public ActionResult Control_ListSectionsBlock(string clubDomainName)
        {
            Club.Club club = GetCurrentClub(clubDomainName);
            int groupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(club.ClubID);
            List<ForumSection> forumSections = ForumSections.GetRootSectionsOfGroup(groupID, false);
            return PartialView("Controls/Forums/ListSectionsBlock.ascx", forumSections);
        }

        /// <summary>
        /// 编辑版块
        /// </summary>
        public ActionResult EditForumSection(string clubDomainName, int? sectionID)
        {
            Club.Club currentClub = GetCurrentClub(clubDomainName);
            SetPageTitle(GetResourceString("Title_EditModuel", applicationID), clubDomainName);
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();


            string applicationNameInclub = GetResourceString("Title_Topic", applicationID);
            BreadCrumb.AddBreadCrumbItem(currentClub.ClubName, SPBUrlHelper.Action("Home", "ClubDomain", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Manage", applicationID), SPBUrlHelper.Action("Home", "ClubManage", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_EditModuel", applicationID), "");

            ForumSection section = new ForumSection();
            if (sectionID.HasValue && sectionID.Value > 0)
            {
                section = ForumSections.GetSection(sectionID.Value, true);
            }
            else
            {
                if (TempData["section"] != null)
                {
                    section = TempData["section"] as ForumSection;
                }
            }
            ViewData["isPublic"] = currentClub.IsPublic;
            return View("Pages/Manage/EditForumSection.aspx", "Masters/Manage.master", section);
        }

        /// <summary>
        /// 合并版块
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_MergeSection(string clubDomainName, int filterID)
        {
            if (!ClubPermission.Validate(GetCurrentClub(clubDomainName), CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins()))
                return new EmptyResult();

            ForumSection fromSection = ForumSections.GetSection(filterID, true);

            ViewData["ClubID"] = GetCurrentClub(clubDomainName).ClubID;
            return PartialView("Controls/Manage/MergeSection.ascx", fromSection);
        }

        #endregion

        #region ForumThread

        /// <summary>
        /// 显示新建主题帖页面
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult NewForumThread(string clubDomainName, int? sectionID)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);

            #region 权限验证
            if (!ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().CreateClubForumThread()))
                return Redirect(SPBUrlHelper.Action("ApplyClubMember", "ClubDomain", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));

            #endregion

            ForumThread thread = null;
            if (TempData["thread"] != null)
                thread = TempData.Get<ForumThread>("thread");
            else
            {
                thread = new ForumThread();
                thread.GroupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(currentClub.ClubID);
            }
            string applicationNameInclub = GetResourceString("Title_Topic", applicationID);

            BreadCrumb.AddBreadCrumbItem("发布话题", "");

            if (sectionID.HasValue)
            {
                thread.SectionID = sectionID.Value;
                ForumSection section = ForumSections.GetSection(sectionID.Value, false);

                #region PageTitle & BreadCrumb

                SetPageTitle(applicationNameInclub, section.SectionName, clubDomainName);
                SetClubForumSectionBreadCrumb(clubDomainName, section.SectionID);

                #endregion
            }
            else
                SetPageTitle(applicationNameInclub, clubDomainName, GetResourceString("Title_PublishThread", applicationID));

            ViewData["currentClub"] = currentClub;
            return this.View("Pages/Forums/EditForumThread.aspx", "Masters/Forum.master", thread);
        }

        /// <summary>
        /// 创建主题帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateForumThread(string clubDomainName, int? sectionID)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);

            ForumThread thread = new ForumThread();
            if (sectionID.HasValue && sectionID.Value > 0)
                thread.SectionID = sectionID.Value;
            else
                thread.SectionID = Request.Form.GetInt("sectionID", 0);
            ForumSection section = ForumSections.GetSection(thread.SectionID, false);
            if (section == null)
            {
                Response.Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ForumModelNotFound", applicationID), thread.SectionID)));
            }
            thread.GroupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(currentClub.ClubID);
            thread.UserID = CurrentUser.UserID;
            thread.Author = CurrentUser.DisplayName;
            thread.Subject = Request.Form.GetString("subject", string.Empty).Trim();
            thread.Body = Request.Form.GetString("body", string.Empty).Trim();
            thread.IsLocked = Request.Form.GetBool("isLocked", false);
            thread.IsUseSignature = Request.Form.GetBool("isUseSignature", false);
            thread.IsEssential = Request.Form.GetBool("isEssential", false);

            thread.CategoryID = Request.Form.GetInt("categoryID", 0);
            thread.IsSticky = Request.Form.GetBool("isSticky", false);
            thread.IsHidden = Request.Form.GetBool("IsHidden", false);
            thread.Price = Request.Form.GetInt("Price", thread.Price);
            int stickyDays = Request.Form.GetInt("stickyDays", 1);
            if (thread.IsSticky)
                thread.StickyDate = DateTime.Now.AddDays(stickyDays);

            if (string.IsNullOrEmpty(thread.Subject))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_TitleNotNull", applicationID));
                TempData["thread"] = thread;
                return RedirectToAction("NewForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "sectionID", sectionID } });
            }
            if (string.IsNullOrEmpty(thread.Body))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_ContentNotNull", applicationID));
                TempData["thread"] = thread;
                return RedirectToAction("NewForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "sectionID", sectionID } });
            }

            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().CreateClubForumThread(), HttpContext, SiteUrls.Instance().Login(true)))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Message_Pattern_SendTopicLost", applicationID), section.SectionName));
                TempData["thread"] = thread;
                return RedirectToAction("NewForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "sectionID", sectionID } });
            }

            List<string> tagNameList = new List<string>();
            string[] formattedTagNames = Request.Form.GetString("forumTags", string.Empty).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
            if (formattedTagNames != null && formattedTagNames.Length > 0)
            {
                ApplicationTagManager applicationTagManager = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum());
                foreach (string formattedTagName in formattedTagNames)
                {
                    if (applicationTagManager.GetApplicationTag(formattedTagName, true) == null)
                    {
                        ApplicationTag tag = new ApplicationTag();
                        tag.TagName = formattedTagName;
                        tag.UserID = CurrentUser.UserID;
                        tag.Author = CurrentUser.DisplayName;
                        tag.TagName = Formatter.CleanTagName(tag.TagName);
                        tag.TagName = Formatter.FormatSingleLinePlainTextForStorage(tag.TagName);
                        bool isCreated = applicationTagManager.CreateApplicationTag(tag);
                        if (isCreated)
                            tagNameList.Add(tag.TagName);
                    }
                    else
                    {
                        tagNameList.Add(applicationTagManager.GetApplicationTag(formattedTagName, true).TagName);
                    }
                }
            }

            ForumThreads.CreateThread(thread);
            if (thread.ThreadID > 0)
            {
                ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).UpdateApplicationTagsOfItem(thread.ThreadID, tagNameList);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_IssueThreadTopicSucceed", applicationID));
                return Redirect(SiteUrls.Instance().ShowForumThreadInClub(clubDomainName, thread.ThreadID));
            }
            else
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_SendTopicLost", applicationID));
                TempData["thread"] = thread;
            }
            return RedirectToAction("NewForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "sectionID", sectionID } });

        }

        /// <summary>
        /// 删除单个主题帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteForumThread(string clubDomainName, int threadID)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {
                Response.Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));
            }
            //throw new SPBException(ExceptionType.ClubDomainNotFond);

            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
            {
                Response.Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ForumThreadNotExist", applicationID), threadID)));
            }
            //throw new SPBException(ExceptionType.PostNotFound);
            ForumSection section = ForumSections.GetSection(thread.SectionID, true);
            string exceptionType = ExceptionTypes.Instance().UnknownError();
            if (CurrentUser.UserID != thread.UserID && !ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().DeleteClubForumPost(), ref exceptionType))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Message_Pattern_NotDeleteLimit", applicationID), StringUtils.Trim(thread.Subject, 14)));
                return RedirectToAction("ShowForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "threadID", threadID } });
            }
            int sectionID = thread.SectionID;
            ForumThreads.DeleteThread(threadID);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteThreadTopicSucceed", applicationID));
            return RedirectToAction("ShowForumSection", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "sectionID", sectionID } });
        }

        /// <summary>
        /// 显示编辑主题帖页面
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult EditForumThread(string clubDomainName, int threadID)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            SetPageTitle("编辑话题");
            if (currentClub == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));
            }
            //throw new SPBException(ExceptionType.ClubDomainNotFond);

            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ThreadNotExist", applicationID), threadID)));
            }
            // throw new SPBException(ExceptionType.PostNotFound);

            string exceptionType = ExceptionTypes.Instance().UnknownError();
            if (CurrentUser == null || CurrentUser.UserID != thread.UserID && !ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().EditClubForumPost(), ref exceptionType))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Message_Pattern_NotEditLimit", applicationID), StringUtils.Trim(thread.Subject, 14)));
                return RedirectToAction("ShowForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "threadID", threadID } });
            }

            #region PageTitle & BreadCrumb
            string applicationNameInclub = GetResourceString("Title_Topic", applicationID);
            SetPageTitle(applicationNameInclub, thread.ForumSection.SectionName, clubDomainName);
            BreadCrumb.AddBreadCrumbItem(currentClub.ClubName, SiteUrls.Instance().ClubDomainHome(clubDomainName));
            BreadCrumb.AddBreadCrumbItem(applicationNameInclub, SPBUrlHelper.Action("Home", "ClubForum", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));
            SetClubForumSectionBreadCrumb(clubDomainName, thread.SectionID);
            BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(thread.Subject, 14), SiteUrls.Instance().ShowForumThreadInClub(clubDomainName, thread.ThreadID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_EditThread", applicationID), string.Empty);
            #endregion


            string tagNames = string.Empty;
            foreach (var tag in thread.Tags)
            {
                tagNames += tag + ",";
            }
            ViewData["tagNames"] = tagNames;
            ViewData["currentClub"] = currentClub;
            return this.View("Pages/Forums/EditForumThread.aspx", "Masters/Forum.master", thread);
        }

        /// <summary>
        /// 更新主题帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateForumThread(string clubDomainName, int threadID)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {
                Response.Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));
            }
            //throw new SPBException(ExceptionType.ClubDomainNotFond);

            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
            {
                Response.Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ForumThreadNotExist", applicationID), threadID)));
            }
            //throw new SPBException(ExceptionType.PostNotFound);
            ForumSection section = ForumSections.GetSection(thread.SectionID, false);
            string exceptionType = ExceptionTypes.Instance().UnknownError();
            if (CurrentUser.UserID != thread.UserID && !ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().EditClubForumPost(), ref exceptionType))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Message_Pattern_UpdateThreadLost", applicationID), section.SectionName));
                return RedirectToAction("ShowForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "threadID", threadID } });
            }
            thread.GroupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(currentClub.ClubID);
            thread.Subject = Request.Form.GetString("subject", string.Empty).Trim();
            thread.Body = Request.Form.GetString("body", string.Empty).Trim();
            thread.IsLocked = Request.Form.GetBool("isLocked", false);
            thread.CategoryID = Request.Form.GetInt("categoryID", 0);
            thread.IsUseSignature = Request.Form.GetBool("isUseSignature", false);
            thread.IsSticky = Request.Form.GetBool("isSticky", false);
            thread.IsEssential = Request.Form.GetBool("isEssential", false);
            int stickyDays = Request.Form.GetInt("stickyDays", 1);
            thread.IsHidden = Request.Form.GetBool("IsHidden", false);
            thread.Price = Request.Form.GetInt("Price", thread.Price);

            if (thread.IsSticky)
                thread.StickyDate = DateTime.Now.AddDays(stickyDays);

            if (string.IsNullOrEmpty(thread.Subject))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_FillThreadTitle", applicationID));
                return RedirectToAction("ShowForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "threadID", threadID } });
            }
            if (string.IsNullOrEmpty(thread.Body))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_FillThreadContent", applicationID));
                return RedirectToAction("ShowForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "threadID", threadID } });
            }

            List<string> tagNameList = new List<string>();
            string[] formattedTagNames = Request.Form.GetString("forumTags", string.Empty).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
            if (formattedTagNames != null && formattedTagNames.Length > 0)
            {

                foreach (string formattedTagName in formattedTagNames)
                {
                    if (ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).GetApplicationTag(formattedTagName, true) == null)
                    {
                        ApplicationTag tag = new ApplicationTag();
                        tag.TagName = formattedTagName;
                        tag.UserID = CurrentUser.UserID;
                        tag.Author = CurrentUser.DisplayName;
                        tag.TagName = Formatter.CleanTagName(tag.TagName);
                        tag.TagName = Formatter.FormatSingleLinePlainTextForStorage(tag.TagName);

                        bool isCreated = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).CreateApplicationTag(tag);
                        if (isCreated)
                            tagNameList.Add(tag.TagName);
                    }
                    else
                    {
                        tagNameList.Add(ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).GetApplicationTag(formattedTagName, true).TagName);
                    }
                }
            }

            ForumThreads.UpdateThread(thread);
            ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).UpdateApplicationTagsOfItem(thread.ThreadID, tagNameList);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdateThreadTopicSucceed", applicationID));
            return Redirect(SiteUrls.Instance().ShowForumThreadInClub(clubDomainName, thread.ThreadID));
        }

        /// <summary>
        /// 设置主题分类
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SetThreadCategory(string clubDomainName)
        {

            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {
                Response.Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));
            }

            int threadID = Request.Form.GetInt("threadID", 0);
            ForumThread thread = ForumThreads.GetThread(threadID, true);

            if (thread == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("ID为{0}的主题未找到！", threadID);
                return PartialView("Controls/Forums/SetThreadCategory.ascx");
            }

            int categoryID = Request.Form.GetInt("categoryID", 0);

            if (categoryID <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("ID为{0}的类别不存在！", categoryID);
                return PartialView("Controls/Forums/SetThreadCategory.ascx");
            }

            thread.CategoryID = categoryID;
            ForumThreads.UpdateThread(thread);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "设置主题类别成功！";

            return PartialView("Controls/Forums/SetThreadCategory.ascx");

        }

        /// <summary>
        /// 显示主题
        /// </summary>
        /// <param name="replyUserID">回帖人UserID(例如：只看楼主)</param>        
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShowForumThread(string clubDomainName, int threadID, int? replyUserID, bool? isDesc, int? pageIndex)
        {
            Club.Club club = Clubs.GetClub(clubDomainName, false);
            if (club == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));

            ForumThread forumThread = ForumThreads.GetThread(threadID, false);
            if (forumThread == null)
                return Redirect(SiteUrls.Instance().Error404("论坛主题未找到", string.Format("ID为：{0}的主题未找到！", threadID)));

            ForumSectionGroup group = ForumSectionGroups.GetGroup(forumThread.GroupID, false);
            if (group == null || group.PresentAreaID != PresentAreaIDs.Instance().ClubDomain() || group.OwnerID != club.ClubID)
                return Redirect(SiteUrls.Instance().Error404("论坛主题未找到", string.Format("ID为：{0}的主题未找到！", threadID)));

            PagingDataSet<ForumPost> pds = ForumPosts.GetPostsOfThread(threadID, replyUserID, isDesc ?? false, pageIndex ?? 1, true);

            #region PageTitle & BreadCrumb & meta
            string applicationNameInclub = GetResourceString("Title_Topic", applicationID);
            SetPageTitle(applicationNameInclub, StringUtils.Trim(forumThread.Subject, 14), clubDomainName);
            BreadCrumb.AddBreadCrumbItem(club.ClubName, SiteUrls.Instance().ClubDomainHome(clubDomainName));
            BreadCrumb.AddBreadCrumbItem(applicationNameInclub, SPBUrlHelper.Action("Home", "ClubForum", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));

            try
            {
                SetClubForumSectionBreadCrumb(clubDomainName, forumThread.SectionID);
            }
            catch
            {
                return Redirect(SiteUrls.Instance().Error("没有权限浏览当前话题！"));
            }

            BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(forumThread.Subject, 14), SiteUrls.Instance().ShowForumThreadInClub(clubDomainName, forumThread.ThreadID));

            SetPageMeta(forumThread);
            #endregion

            #region 处理审核

            int auditingStatusValue = (int)Applications.GetApplication(ApplicationIDs.Instance().Forum()).AuditingStatusForPublicDisplay;
            if ((int)forumThread.AuditingStatus < auditingStatusValue)
            {
                if (forumThread.AuditingStatus != AuditingStatuses.Success)
                    ViewData["errorMessage"] = GetResourceString("Message_TopicNotPass", applicationID);

                ViewData["errorAuditingStatus"] = true;
            }

            #endregion
            //浏览计数            
            CounterFactory.IncreaseCount<ForumThreadCounter>(IncreasableCountTypes.WebView, threadID);

            ViewData["ForumThread"] = forumThread;

            return this.View("Pages/Forums/ShowForumThread.aspx", "Masters/Forum.master", pds);
        }

        /// <summary>
        /// 显示某一楼层帖子
        /// </summary>
        public ActionResult Control_ForumPostFloor(string clubDomainName, int? replyUserID, bool? isDesc, int? postID)
        {
            ForumPost forumPost = ForumPosts.GetPost(postID ?? 0, true);
            if (forumPost == null)
                return new EmptyResult();
            ForumThread forumThread = ForumThreads.GetThread(forumPost.ThreadID, false);
            if (forumThread == null)
                return new EmptyResult();
            ForumSection forumSection = ForumSections.GetSection(forumPost.SectionID, false);
            if (forumSection == null)
                return new EmptyResult();
            ViewData["forumPost"] = forumPost;
            ViewData["forumSection"] = forumSection;
            ViewData["number"] = forumThread.ReplyCount + 1;

            return View("Controls/Forums/ForumPostFloor.ascx");
        }

        /// <summary>
        /// 更新单个主题的精华状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult SetEssential(string clubDomainName, int threadID, bool? isEssential)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);

            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
                return new JsonResult();
            ForumSection section = ForumSections.GetSection(thread.SectionID, true);
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().SetClubForumThreadEssential(), HttpContext, SiteUrls.Instance().Login(true)))
                return new JsonResult();

            ForumThreads.SetEssential(threadID, isEssential.Value);
            thread = ForumThreads.GetThread(threadID, true);
            return RedirectToAction("ShowForumThread", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 设置单个主题的置顶状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public JsonResult SetSticky(string clubDomainName, int threadID, bool? isSticky)
        {
            if (isSticky == null)
                return new JsonResult();
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);

            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
                return new JsonResult();
            ForumSection section = ForumSections.GetSection(thread.SectionID, true);
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().StickyClubForumThread(), HttpContext, SiteUrls.Instance().Login(true)))
                return new JsonResult();

            ForumThreads.SetSticky(threadID, isSticky ?? false);//在版块中置顶这个主题

            thread = ForumThreads.GetThread(threadID, true);
            return Json(thread.IsSticky, JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 显示移动主题页面
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShowMoveForumThread(string clubDomainName, int threadID)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));
            }
            //throw new SPBException(ExceptionType.ClubDomainNotFond);

            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ThreadNotExist", applicationID), threadID)));
            }
            // throw new SPBException(ExceptionType.PostNotFound);
            ForumSection section = ForumSections.GetSection(thread.SectionID, true);
            if (CurrentUser == null || !ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().MoveClubForumThread(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            if (section != null)
                ViewData["OldSectionName"] = section.SectionName;
            return this.View("Pages/Forums/MoveForumThread.aspx", "Masters/Forum.master", thread);
        }

        /// <summary>
        /// 显示主题列表
        /// </summary>
        /// <param name="sortBy"></param>
        /// <param name="isEssential"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public ActionResult Control_ListForumThreads(string clubDomainName, ForumThreadSortBy? sortBy, bool? isEssential, int? pageIndex)
        {
            Club.Club currentClub = Clubs.GetClub(clubDomainName, false);

            if (currentClub == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format("未找到的圈子：{0}", clubDomainName)));

            PagingDataSet<ForumThread> pds = ForumThreads.GetPagingThreadsSortBy(sortBy ?? ForumThreadSortBy.LastRepliedDate, OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(currentClub.ClubID), PresentAreaIDs.Instance().ClubDomain(), isEssential, 20, pageIndex ?? 1);

            ViewData["sortBy"] = sortBy.HasValue ? (int)sortBy : (int)ForumThreadSortBy.PostDate;
            ViewData["isEssential"] = isEssential;
            return PartialView("Controls/Forums/ListForumThreads.ascx", pds);
        }

        public ActionResult ListMyForums(string clubDomainName)
        {
            SetPageTitle(GetResourceString("Title_MyJoinTopic", applicationID), clubDomainName);
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_MyJoinTopic", applicationID), null);
            return View("Pages/Forums/ListMyForums.aspx", "Masters/Forum.master");
        }

        ///<summary>
        ///显示我的论坛详细列表
        ///</summary>
        public ActionResult Control_ListMyForums(int? pageIndex, string clubDomainName)
        {
            Club.Club currentClub = GetCurrentClub(clubDomainName);
            int groupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(currentClub.ClubID);
            PagingDataSet<ForumThread> pds = ForumThreads.GetMyGroupThreads(CurrentUser != null ? CurrentUser.UserID : 0, groupID, pageIndex ?? 1);
            return PartialView("Controls/Forums/ListMyForums.ascx", pds);
        }

        /// <summary>
        /// 回复
        /// </summary>       
        public ActionResult Control_ListPostForums(int? pageIndex, string clubDomainName)
        {
            Club.Club currentClub = GetCurrentClub(clubDomainName);
            int groupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(currentClub.ClubID);
            PagingDataSet<ForumPost> pds = ForumPosts.GetPostsOfGroup(groupID, pageIndex ?? 1, false);
            Dictionary<int, ForumThread> forumThreads = new Dictionary<int, ForumThread>();
            foreach (ForumPost post in pds.Records)
            {
                forumThreads[post.ThreadID] = ForumThreads.GetThread(post.ThreadID, false);
            }
            ViewData["forumThreads"] = forumThreads;
            return PartialView("Controls/Forums/ListPostForums.ascx", pds);
        }

        /// <summary>
        /// 论坛主题列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ListForumThreadsBlock(string clubDomainName, int? displayCount, ForumThreadSortBy? sortBy)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));
            }

            int groupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(currentClub.ClubID);
            IList<ForumThread> pds = ForumThreads.GetTopThreadsSortBy(sortBy ?? ForumThreadSortBy.PostDate, groupID, PresentAreaIDs.Instance().ClubDomain(), null, displayCount ?? 10);
            return this.PartialView("Controls/Forums/ListForumThreadsBlock.ascx", pds);
        }

        /// <summary>
        /// 移动单个论坛主题
        /// </summary>
        public ActionResult Control_MoveForumThread(string clubDomainName, int threadID)
        {
            if (threadID <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("ID为{0}的主题未找到！", threadID);
            }

            ViewData["threadIDs"] = threadID;

            return PartialView("Controls/Manage/MoveForumThread.ascx");
        }

        /// <summary>
        /// 设置置顶
        /// </summary>
        public ActionResult Control_SetSticky(string clubDomainName, int threadID)
        {
            if (threadID <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("ID为{0}的主题未找到！", threadID);
            }

            ViewData["threadIDs"] = threadID;

            return PartialView("Controls/Manage/SetSticky.ascx");
        }

        /// <summary>
        /// 更新单个主题的锁定状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult SetLock(string clubDomainName, int threadID, bool? isLocked)
        {

            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format("ID为{0}的主题未找到！", threadID)));

            ForumSection section = ForumSections.GetSection(thread.SectionID, false);
            if (!ForumPermission.AccessCheck(section, CurrentUser, PermissionItemKeys.Instance().SetForumThreadEssential(), HttpContext, SiteUrls.Instance().Login(true)))
                return Redirect(SiteUrls.Instance().Error404("版块不存在！"));

            ForumThreads.SetLock(threadID, isLocked ?? false);

            return RedirectToAction("ShowForumThread", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 设置主题类别
        /// </summary>
        public ActionResult Control_SetThreadCategory(string clubDomainName, int threadID, int sectionID)
        {
            ForumThread thread = ForumThreads.GetThread(threadID, false);
            if (thread == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("ID为{0}的主题未找到！", threadID);
            }

            ForumSection section = ForumSections.GetSection(sectionID, false);

            if (section == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("ID为{0}的版块未找到！", sectionID);
            }

            if (section.ThreadCategories == null || section.ThreadCategories.Count <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "没有可用的版块类别！";
            }

            ViewData["thread"] = thread;
            ViewData["section"] = section;

            return PartialView("Controls/Forums/SetThreadCategory.ascx");
        }

        /// <summary>
        /// 设置高亮
        /// </summary>
        public ActionResult Control_SetHighlight(string clubDomainName, int threadID, int sectionID)
        {

            if (threadID <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_NotFindForumTopicID", applicationID), threadID);
            }

            ViewData["threadIDs"] = threadID;

            return PartialView("Controls/Manage/SetHighlight.ascx");
        }

        /// <summary>
        /// 购买主题
        /// </summary>
        /// <param name="threadID">主题ID</param>
        public ActionResult ExchangeThreadAction(string clubDomainName, int threadID)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));

            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
                throw new SPBException(ExceptionTypes.Instance().PostNotFound());

            if (CurrentUser.TradePoints < thread.Price)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Message_Pattern_NeedDealIntegral", applicationID), thread.Price.ToString(), CurrentUser.TradePoints.ToString()));
            }
            else
            {
                ForumThreads.ExchangeThread(CurrentUser, threadID);
                UserPointRecord record = new UserPointRecord();
                record.Description = string.Format(GetResourceString("Description_Pattern_ShopForumThread", applicationID), thread.Subject);
                record.PointCategory = UserPointCategory.Trade;
                record.PointItemName = GetResourceString("Description_ShopForumThread", applicationID);
                record.Points = -thread.Price;
                record.UserID = CurrentUser.UserID;
                SpaceBuilder.Common.Points.CreateUserPointRecord(record);

                //系统扣除税点
                //税率
                int tax = SiteSettingsManager.GetSiteSettings().TransactionTax;
                //税点
                int taxPoints = thread.Price - thread.Price * (100 - tax) / 100;
                Points.ChanageSystemTradePoints(taxPoints, "征收论坛主题交易税", string.Format("{0}购买了论坛主题：{1}", SPBHtmlHelper.Instance().UserDisplayNameLink(CurrentUser,true), thread.Subject));


                UserPointRecord recordl = new UserPointRecord();
                recordl.Description = string.Format(GetResourceString("Description_Pattern_ShopYourThread", applicationID), CurrentUser.DisplayName, thread.Subject);
                recordl.PointCategory = UserPointCategory.Trade;
                recordl.PointItemName = GetResourceString("Description_ByShopForumThread", applicationID);
                recordl.Points = thread.Price - taxPoints;
                recordl.UserID = thread.UserID;
                SpaceBuilder.Common.Points.CreateUserPointRecord(recordl);
            }

            return RedirectToAction("ShowForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "threadID", threadID } });
        }

        public ActionResult Search(string clubDomainName, int? pageIndex)
        {
            Club.Club currentClub = Clubs.GetClub(clubDomainName, false);
            if (currentClub == null)
                Redirect(SiteUrls.Instance().Error404("群组不存在或已被删除！"));



            SetPageTitle(GetResourceString("Title_Topic", applicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ForumThreadSearch", applicationID), string.Empty);

            ForumPostFullTextQuery query = new ForumPostFullTextQuery();
            query.ConvertFromQueryString(Request);
            query.SearchScopeGroupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(currentClub.ClubID);
            query.SectionID = 0;
            query.PageSize = 10;
            if (pageIndex.HasValue)
                query.PageIndex = pageIndex.Value;

            SearchResultDataSet<ForumThread> results = null;
            if (query.IsValid())
                results = ForumSearchManager.Instance().Search(query);

            ViewData["Query"] = query;
            return View("Pages/Forums/ListSearchForums.aspx", "Masters/Forum.master", results);
        }

        #endregion

        #region ForumPost

        /// <summary>
        /// 显示新建回复帖页面
        /// </summary>
        /// <param name="parentID">被回复的帖子PostID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult NewForumPost(string clubDomainName, int? parentID, bool? quote)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));
            }

            ForumPost parentPost = ForumPosts.GetPost(parentID ?? 0, false);
            if (parentPost == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ByReplyTopicNotFound", applicationID), parentID)));
            }
            ForumThread thread = ForumThreads.GetThread(parentPost.ThreadID, false);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ByReplyTopicThreadNotFound", applicationID), parentPost.ThreadID)));
            }
            if (thread.IsLocked)
                throw new SPBException(ExceptionTypes.Instance().PostLocked());

            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ReplyClubForumThread(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            #region PageTitle & BreadCrumb
            string applicationNameInclub = GetResourceString("Title_Topic", applicationID);
            SetPageTitle(applicationNameInclub, StringUtils.Trim(thread.Subject, 14), clubDomainName);
            BreadCrumb.AddBreadCrumbItem(currentClub.ClubName, SiteUrls.Instance().ClubDomainHome(clubDomainName));
            BreadCrumb.AddBreadCrumbItem(applicationNameInclub, SPBUrlHelper.Action("Home", "ClubForum", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));
            SetClubForumSectionBreadCrumb(clubDomainName, thread.SectionID);
            BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(thread.Subject, 14), SiteUrls.Instance().ShowForumThreadInClub(clubDomainName, thread.ThreadID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ReplyThread", applicationID), string.Empty);
            #endregion

            ForumPost post = null;
            if (TempData["post"] != null)
                post = TempData.Get<ForumPost>("post");
            post = new ForumPost();

            int FloorNumber = Request.QueryString.GetInt("FloorNumber", -1);
            string strFloor = string.Empty;
            if (FloorNumber == 0)
                strFloor = GetResourceString("Label_FloorHost", applicationID);
            else if (FloorNumber > 0)
                strFloor = FloorNumber.ToString() + GetResourceString("Label_Floor", applicationID);
            post.Subject = string.Format("{0} {1} {2}", GetResourceString("Common_Reply"), strFloor, parentPost.Author);

            post.ParentID = parentID ?? 0;
            if (quote.HasValue && quote.Value)
                post.Body = string.Format("[quote user=\"{0}\"]{1}[/quote]", parentPost.Author, parentPost.TrimNoQuoteHtml(200));

            ViewData["currentClub"] = currentClub;
            return this.View("Pages/Forums/EditForumPost.aspx", "Masters/Forum.master", post);
        }



        /// <summary>
        /// 创建论坛回复贴
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateForumPost(string clubDomainName, int parentID)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));
            }
            //throw new SPBException(ExceptionType.ClubDomainNotFond);

            ForumPost post = new ForumPost();
            post.Subject = Request.Form.GetString("subject", string.Empty).Trim();
            post.Body = Request.Form.GetString("body", string.Empty).Trim();
            post.IsUseSignature = Request.Form.GetBool("isUseSignature", false);
            if (string.IsNullOrEmpty(post.Body))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_ContentNotNull", applicationID));
                TempData["post"] = post;
                return RedirectToAction("NewForumPost", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "parentID", parentID } });
            }
            post.UserID = CurrentUser != null ? CurrentUser.UserID : 0;
            post.Author = SPBHtmlHelper.Instance().UserDisplayName(CurrentUser);
            ForumPost parentPost = ForumPosts.GetPost(parentID, false);
            if (parentPost == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ByReplyTopicNotFound", applicationID), parentID)));
            }
            post.ParentID = parentID;
            //post.Subject = string.Format("Re:{0}", parentPost.Subject);
            post.ThreadID = parentPost.ThreadID;
            post.SectionID = parentPost.SectionID;
            if (CurrentUser == null || !ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ReplyClubForumThread(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                //检查验证码是否录入正确
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                if (verifyCode != null)
                    verifyCode = verifyCode.Trim();

                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_IdentifyingCodeError"));
                    TempData["post"] = post;
                    return RedirectToAction("NewForumPost", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "parentID", parentID } });
                }
            }

            ForumPosts.CreatePost(post);
            if (post.PostID > 0)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_IssueReplySucceed", applicationID));
                return Redirect(SiteUrls.Instance().ShowForumThreadGotoLastPostInClub(clubDomainName, post.ThreadID, post.PostID));
            }
            else
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_ReplyLost", applicationID));
                TempData["post"] = post;
                return RedirectToAction("NewForumPost", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "parentID", parentID } });
            }
        }

        /// <summary>
        /// 快速回复
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public JsonResult CreateQuickForumPost(string clubDomainName, int parentID)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {

                return Json(new
                {
                    ok = false,
                    message = SPBHtmlHelper.Instance().StatusMessage(StatusMessageType.Error, string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName))
                });
            }

            ForumPost post = new ForumPost();
            post.Body = Request.Form.GetString("body", string.Empty).Trim();

            post.UserID = CurrentUser != null ? CurrentUser.UserID : 0;
            post.Author = SPBHtmlHelper.Instance().UserDisplayName(CurrentUser);
            ForumPost parentPost = ForumPosts.GetPost(parentID, false);

            if (parentPost == null)
            {
                return Json(new
                {
                    ok = false,
                    message = SPBHtmlHelper.Instance().StatusMessage(StatusMessageType.Error, string.Format(GetResourceString("Exception_ReplyTopicPageNotExist", applicationID)))
                });
            }

            post.ParentID = parentID;

            post.Subject = Request.Form.GetString("subject", string.Empty).Trim();

            post.ThreadID = parentPost.ThreadID;
            post.SectionID = parentPost.SectionID;
            ForumSection section = ForumSections.GetSection(parentPost.SectionID, false);

            string exceptionType = ExceptionTypes.Instance().PostAccessDenied();
            if (CurrentUser == null || !ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ReplyClubForumThread(), HttpContext, SiteUrls.Instance().Login(true)))
            {
                ExceptionMessage exceptionMessage = ResourceManager.GetExceptionMessage(exceptionType, applicationID);
                string message = GetResourceString("Message_SendTopicLost", applicationID);
                if (exceptionMessage != null)
                    message = exceptionMessage.Body;
                return Json(new
                {
                    ok = false,
                    message = SPBHtmlHelper.Instance().StatusMessage(StatusMessageType.Error, message)
                });
            }

            if (string.IsNullOrEmpty(post.Body))
            {
                return Json(new
                {
                    ok = false,
                    message = SPBHtmlHelper.Instance().StatusMessage(StatusMessageType.Error, string.Format(GetResourceString("Message_ContentNotNull", applicationID)))
                });
            }

            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                //检查验证码是否录入正确
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                if (verifyCode != null)
                    verifyCode = verifyCode.Trim();

                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    return Json(new
                    {
                        ok = false,
                        message = SPBHtmlHelper.Instance().StatusMessage(StatusMessageType.Error, GetResourceString("Message_IdentifyingCodeError"))
                    });
                }
            }

            if (SPBConfig.Instance().EnableEmoticon)
                post.Body = EmotionManager.Instance().EmoticonTransforms(post.Body);

            ForumPosts.CreatePost(post);
            if (post.PostID > 0)
            {
                string message = GetResourceString("Message_IssueReplySucceed", applicationID);
                if (post.AuditingStatus != AuditingStatuses.Success)
                    message = GetResourceString("Message_SucceedButAdmin", applicationID);

                return Json(new
                {
                    ok = true,
                    message = SPBHtmlHelper.Instance().StatusMessage(StatusMessageType.Success, message),
                    postID = post.PostID
                });
            }
            else
            {
                return Json(new
                {
                    ok = false,
                    message = SPBHtmlHelper.Instance().StatusMessage(StatusMessageType.Error, GetResourceString("Message_ReplyLostContactAdmin", applicationID))
                });
            }
        }

        /// <summary>
        /// 删除回复帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteForumPost(string clubDomainName, int postID, int pageIndex)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));
            }

            ForumPost post = ForumPosts.GetPost(postID, true);
            if (post == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_TopicNotExist", applicationID), postID)));
            }
            ForumSection section = ForumSections.GetSection(post.SectionID, true);
            string exceptionType = ExceptionTypes.Instance().UnknownError();
            if (CurrentUser.UserID != post.UserID && !ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().DeleteClubForumPost(), ref exceptionType))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Message_Pattern_NotDeleteLimit", applicationID), StringUtils.Trim(post.Subject, 14)));
                return RedirectToAction("ShowForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "threadID", post.ThreadID } });
            }

            int threadID = post.ThreadID;
            ForumPosts.DeletePost(postID);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteTopicSucceed", applicationID));
            return RedirectToAction("ShowForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "threadID", threadID }, { "pageIndex", pageIndex } });

        }

        /// <summary>
        /// 显示编辑回复帖页面
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult EditForumPost(string clubDomainName, int postID)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));
            }

            ForumPost post = ForumPosts.GetPost(postID, true);
            if (post == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_TopicNotExist", applicationID), postID)));
            }

            if (post.UserID != CurrentUser.UserID &&
            !ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().EditClubForumPost(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();
            if (post == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_TopicNotExist", applicationID), postID)));
            }

            #region PageTitle & BreadCrumb
            string applicationNameInclub = GetResourceString("Title_Topic", applicationID);
            SetPageTitle(applicationNameInclub, StringUtils.Trim(post.Subject, 14), clubDomainName);
            //BreadCrumb.AddBreadCrumbItem(SiteSettingsManager.GetSiteSettings().SiteName, SiteUrls.Instance().Home());
            BreadCrumb.AddBreadCrumbItem(currentClub.ClubName, SiteUrls.Instance().ClubDomainHome(clubDomainName));
            BreadCrumb.AddBreadCrumbItem(applicationNameInclub, SPBUrlHelper.Action("Home", "ClubForum", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));
            SetClubForumSectionBreadCrumb(clubDomainName, post.SectionID);
            ForumThread thread = ForumThreads.GetThread(post.ThreadID, false);
            if (thread != null)
                BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(thread.Subject, 14), SiteUrls.Instance().ShowForumThreadInClub(clubDomainName, post.ThreadID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_EditTopic", applicationID), string.Empty);
            #endregion

            ViewData["currentClub"] = currentClub;
            return this.View("Pages/Forums/EditForumPost.aspx", "Masters/Forum.master", post);
        }

        /// <summary>
        /// 更新论坛回复贴
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateForumPost(string clubDomainName, int postID)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));
            }
            //throw new SPBException(ExceptionType.ClubDomainNotFond);

            ForumPost post = ForumPosts.GetPost(postID, true);
            if (post == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_TopicNotExist", applicationID), postID)));
            }
            //throw new SPBException(ExceptionType.PostNotFound);

            ForumSection section = ForumSections.GetSection(post.SectionID, false);
            if (post.UserID != CurrentUser.UserID && !ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().EditClubForumPost(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();


            post.Subject = Request.Form.GetString("subject", post.Subject).Trim();
            post.Body = Request.Form.GetString("body", string.Empty).Trim();
            post.IsUseSignature = Request.Form.GetBool("isUseSignature", false);
            //if (string.IsNullOrEmpty(post.Subject))
            //{
            //    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "请填写回复贴的标题！");
            //    return RedirectToAction("EditForumPost", new RouteValueDictionary { { "clubDomainName", clubDomainName }, {"postID", postID } });
            //}
            if (string.IsNullOrEmpty(post.Body))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_FillReplyContent", applicationID));
                return RedirectToAction("EditForumPost", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "postID", postID } });
            }

            ForumPosts.UpdatePost(post);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdateReplySucceed", applicationID));
            return Redirect(SiteUrls.Instance().ShowForumThreadGotoLastPostInClub(clubDomainName, post.ThreadID, post.PostID));
        }

        #endregion

        #region ForumCategory

        /// <summary>
        /// 编辑主题分类
        /// </summary>
        public ActionResult Control_EditForumThreadCategory(int sectionID, int? categoryID)
        {
            ForumThreadCategory category = null;
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                category = ForumThreadCategories.GetForumThreadCategory(categoryID.Value, true);
                if (category == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_NotFindUpdateForumCategories", applicationID), categoryID.Value)));
                }
            }
            else if (TempData["category"] != null)
            {
                category = TempData.Get<ForumThreadCategory>("category");
            }
            if (category == null)
            {
                category = new ForumThreadCategory();
                category.SectionID = sectionID;
            }
            return PartialView("Controls/Manage/EditForumThreadCategory.ascx", category);
        }

        /// <summary>
        /// 创建,更新主题分类
        /// </summary>
        public ActionResult CreateUpdateForumThreadCategory(int sectionID, int? categoryID)
        {
            ForumThreadCategory category = null;
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                category = ForumThreadCategories.GetForumThreadCategory(categoryID.Value, true);
                if (category == null)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_NotFindForumCategoriesSorry", applicationID));
                    return RedirectToAction("Control_EditForumThreadCategory", new RouteValueDictionary { { "sectionID", sectionID }, { "categoryID", categoryID } });
                }
            }
            else
            {
                category = new ForumThreadCategory();
            }
            category.CategoryName = Request.Form.GetString("categoryName", string.Empty);
            category.CategoryDescription = Request.Form.GetString("categoryDescription", string.Empty);
            category.SectionID = sectionID;
            category.DisplayOrder = Request.Form.GetInt("displayOrder", 0);
            if (string.IsNullOrEmpty(category.CategoryName))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_ForumCategoriesNotNull", applicationID));
                TempData["category"] = category;
                return RedirectToAction("Control_EditForumThreadCategory", new RouteValueDictionary { { "sectionID", sectionID }, { "categoryID", categoryID } });
            }
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                ForumThreadCategories.UpdateForumThreadCategory(category);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdateForumCategoriesSucced", applicationID));
            }
            else
            {
                ForumThreadCategories.CreateForumThreadCategory(category);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_CreateForumCategoriesSucced", applicationID));
            }
            return RedirectToAction("Control_EditForumThreadCategory", new RouteValueDictionary { { "sectionID", sectionID }, { "categoryID", categoryID } });
        }

        /// <summary>
        /// 删除主题分类
        /// </summary>
        public ActionResult DeleteForumThreadCategory(int categoryID)
        {
            ForumThreadCategory category = ForumThreadCategories.GetForumThreadCategory(categoryID, true);
            if (category == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_NotFindForumCategoriesID", applicationID), categoryID)));
            }
            int sectionID = category.SectionID;
            ForumThreadCategories.DeleteForumThreadCategory(category);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteForumCategoriesSucced", applicationID));
            return RedirectToAction("Control_ListForumCategories", new RouteValueDictionary { { "sectionID", sectionID } });

        }

        /// <summary>
        /// 显示版块的主题分类页
        /// </summary>    
        public ActionResult ManageForumCategories(string clubDomainName)
        {
            #region PageTitle & BreadCrumb
            SetPageTitle("主题分类管理", clubDomainName);

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Forum", applicationID), SiteUrls.Instance().ChannelForumHome());
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_ManageForumCategories", applicationID), string.Empty);
            #endregion
            Club.Club currentClub = GetCurrentClub(clubDomainName);
            ViewData["currentClub"] = currentClub;
            return View("Pages/Manage/ManageForumCategories.aspx", "Masters/Manage.master");

        }

        /// <summary>
        /// 主题分类列表
        /// </summary>     
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListForumCategories(int sectionID)
        {
            List<ForumThreadCategory> categories = ForumThreadCategories.GetForumThreadCategories(sectionID, true);
            if (categories == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_NotFindForumCategories", applicationID), sectionID)));
            }
            ViewData["section"] = ForumSections.GetSection(sectionID, true);
            return PartialView("Controls/Manage/ListForumCategories.ascx", categories);
        }

        #endregion

        #region Manage

        #region FroumThread

        /// <summary>
        /// 批量删除主题帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteForumThreads(string clubDomainName, int sectionID)
        {
            ForumSection section = ForumSections.GetSection(sectionID, true);
            if (section == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ForumModelNotFound", applicationID), sectionID)));
            }
            //throw new SPBException(ExceptionType.SectionNotFound);
            if (!CurrentUser.IsContentAdministrator && !section.IsModerator(CurrentUser.UserID))
                throw new SPBException(ExceptionTypes.Instance().GeneralAccessDenied());

            string[] deleteThreadIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("threadIDCheckBox", string.Empty)))
            {
                deleteThreadIDs = Request.Form.GetString("threadIDCheckBox", string.Empty).Split(',');
                if (deleteThreadIDs != null && deleteThreadIDs.Length > 0)
                {
                    foreach (string threadID in deleteThreadIDs)
                    {
                        int id = 0;
                        int.TryParse(threadID, out id);
                        ForumThread thread = ForumThreads.GetThread(id, true);
                        if (thread != null && thread.SectionID == section.SectionID)
                        {
                            ForumThreads.DeleteThread(thread.ThreadID);
                        }
                    }
                }
            }
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_BatchDeleteSucceed", applicationID));
            return RedirectToAction("ManageForumThreads", CurrentRouteValueDictionary);

        }

        /// <summary>
        /// 主题管理
        /// </summary>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult ManageForumThreads(string clubDomainName, int? sectionID, string keywords, int? pageIndex)
        {
            Club.Club currentClub = Clubs.GetClub(clubDomainName, false);
            if (currentClub == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));

            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            int groupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(currentClub.ClubID);

            int userID = Request.QueryString.GetInt("userID", 0);

            if (!sectionID.HasValue)
            {
                List<ForumSection> sections = ForumSections.GetRootSectionsOfGroup(groupID, true);
                if (sections != null && sections.Count > 0)
                {
                    ViewData["section"] = sections[0];
                    sectionID = sections[0].SectionID;
                }
            }
            else
                ViewData["section"] = ForumSections.GetSection(sectionID ?? 0, true);

            #region PageTitle & BreadCrumb
            SetPageTitle("主题管理", clubDomainName);
            BreadCrumb.AddBreadCrumbItem("主题管理", string.Empty);
            #endregion

            PagingDataSet<ForumThread> pds = ForumThreads.GetThreadsForManage(keywords, groupID, sectionID, userID, PresentAreaIDs.Instance().ClubDomain(), null, 40, pageIndex ?? 1);

            ViewData["section"] = ForumSections.GetSection(sectionID ?? 0, true);
            ViewData["sideSubMenu"] = (int)ForumSubMenus.ManageThread;
            ViewData["currentClub"] = currentClub;

            return this.View("Pages/Manage/ManageForumThreads.aspx", "Masters/Manage.master", pds);
        }

        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Control_SetStickys(string clubDomainName)
        {
            string strThreadIDs = Request.Form.GetString("threadIDCheckBox", string.Empty);

            if (string.IsNullOrEmpty(strThreadIDs))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选择要置顶的帖子！";
            }

            ViewData["threadIDs"] = strThreadIDs;
            return PartialView("Controls/Manage/SetSticky.ascx");
        }


        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Control_MoveForumThreads(string clubDomainName)
        {
            string strThreadIDs = Request.Form.GetString("threadIDCheckBox", string.Empty);

            if (string.IsNullOrEmpty(strThreadIDs))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选择要移动的主题！";
            }

            ViewData["threadIDs"] = strThreadIDs;

            return PartialView("Controls/Manage/MoveForumThread.ascx");
        }

        public ActionResult UpdateForumAuditingStatus(int postID, int threadID, string clubDomainName)
        {
            List<int> postIDs = new List<int>() { postID };
            ForumPosts.UpdatePostsAuditingStatus(postIDs, AuditingStatuses.Success);
            return RedirectToAction("ShowForumThread", new RouteValueDictionary { { "threadID", threadID }, { "clubDomainName", clubDomainName } });
        }

        /// <summary>
        /// 移动主题
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult MoveForumThread(string clubDomainName, int threadID)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {
                Response.Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));
            }
            //throw new SPBException(ExceptionType.ClubDomainNotFond);

            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
            {
                Response.Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ForumThreadNotExist", applicationID), threadID)));
            }
            //throw new SPBException(ExceptionType.PostNotFound);
            ForumSection section = ForumSections.GetSection(thread.SectionID, true);
            if (CurrentUser == null || !ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().MoveClubForumThread(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            int sectionID = Request.Form.GetInt("sectionID", 0);
            if (sectionID <= 0)
                return RedirectToAction("ShowMoveForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "threadID", threadID } });
            ForumThreads.MoveThread(threadID, sectionID);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_MoveThreadSucceed", applicationID));
            return RedirectToAction("ShowForumThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "threadID", threadID } });
        }

        /// <summary>
        /// 设置主题图章
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SetThreadStamp(string clubDomainName)
        {
            int threadID = Request.Form.GetInt("threadID", 0);
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "群组已被删除或不存在!";
                return this.PartialView("Controls/Forums/SetForumStamp.ascx");
            }
            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "主题已被删除或不存在!";
                return this.PartialView("Controls/Forums/SetForumStamp.ascx");
            }

            ForumSection section = ForumSections.GetSection(thread.SectionID, true);
            if (CurrentUser == null || !ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().SetClubForumThreadEssential(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            int stampID = Request.Form.GetInt("stampID", 0);

            ForumThreads.SetStamp(thread.ThreadID, stampID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = string.Format("设置成功！");

            return PartialView("Controls/Forums/SetForumStamp.ascx");

        }


        /// <summary>
        /// 主题盖章
        /// </summary>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public ActionResult Control_SetForumStamp(string clubDomainName, int threadID)
        {

            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (currentClub == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "群组已被删除或不存在!";
                return this.PartialView("Controls/Forums/SetForumStamp.ascx");
            }
            ForumThread thread = ForumThreads.GetThread(threadID, true);
            if (thread == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "主题已被删除或不存在!";
                return this.PartialView("Controls/Forums/SetForumStamp.ascx");
            }

            ForumSection section = ForumSections.GetSection(thread.SectionID, true);
            if (CurrentUser == null || !ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().SetClubForumThreadEssential(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            ViewData["forumThread"] = thread;
            ViewData["Stamps"] = StampManager.Instance().GetStamps();
            return this.PartialView("Controls/Forums/SetForumStamp.ascx");
        }

        /// <summary>
        /// 批量设置主题帖的精华状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult BatchSetEssential(string clubDomainName, bool isEssential)
        {
            Club.Club currentClub = Clubs.GetClub(clubDomainName, false);
            if (currentClub == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format("圈子：{0}不存在！", clubDomainName)));

            if (CurrentUser == null || !ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins()))
                return Redirect(SiteUrls.Instance().Error404("你没有此项操作的权限！"));

            string[] setEssentialThreadIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("threadIDCheckBox", string.Empty)))
            {
                setEssentialThreadIDs = Request.Form.GetString("threadIDCheckBox", string.Empty).Split(',');
                if (setEssentialThreadIDs != null && setEssentialThreadIDs.Length > 0)
                {
                    foreach (string threadID in setEssentialThreadIDs)
                    {
                        int id = 0;
                        int.TryParse(threadID, out id);
                        ForumThread thread = ForumThreads.GetThread(id, true);
                        if (thread != null)
                        {
                            ForumThreads.SetEssential(thread.ThreadID, isEssential);
                        }
                    }
                }
            }

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_BatchSetSucceed", applicationID);

            return RedirectToAction("ManageForumThreads", CurrentRouteValueDictionary);
        }
        /// <summary>
        /// 批量设置主题帖的置顶状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult BatchSetSticky(string clubDomainName)
        {

            Club.Club currentClub = Clubs.GetClub(clubDomainName, false);
            if (currentClub == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format("圈子：{0}不存在！", clubDomainName)));

            if (CurrentUser == null || !ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins()))
                return Redirect(SiteUrls.Instance().Error404("你没有此项操作的权限！"));

            bool isSticky = Request.Form.GetBool("isSticky", false);

            string[] setStickyThreadIDs = null;
            DateTime stickyDate = Request.Form.Get<DateTime>("StickyDate", DateTime.Now.AddDays(1));
            if (!string.IsNullOrEmpty(Request.Form.GetString("threadIDs", string.Empty)))
            {
                setStickyThreadIDs = Request.Form.GetString("threadIDs", string.Empty).Split(',');
                if (setStickyThreadIDs != null && setStickyThreadIDs.Length > 0)
                {
                    foreach (string threadID in setStickyThreadIDs)
                    {
                        int id = 0;
                        int.TryParse(threadID, out id);
                        ForumThread thread = ForumThreads.GetThread(id, true);
                        if (thread != null)
                            ForumThreads.SetSticky(thread.ThreadID, isSticky, stickyDate);
                    }
                }
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "设置置顶成功！";

            return PartialView("Controls/Manage/SetSticky.ascx");
        }

        /// <summary>
        /// 批量设置主题帖的锁定状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult BatchSetLock(string clubDomainName, bool isLocked)
        {
            Club.Club currentClub = Clubs.GetClub(clubDomainName, false);
            if (currentClub == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format("圈子：{0}不存在！", clubDomainName)));

            if (CurrentUser == null || !ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins()))
                return Redirect(SiteUrls.Instance().Error404("你没有此项操作的权限！"));

            string[] setlockThreadIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("threadIDCheckBox", string.Empty)))
            {
                setlockThreadIDs = Request.Form.GetString("threadIDCheckBox", string.Empty).Split(',');
                if (setlockThreadIDs != null && setlockThreadIDs.Length > 0)
                {
                    foreach (string threadID in setlockThreadIDs)
                    {
                        int id = 0;
                        int.TryParse(threadID, out id);
                        ForumThread thread = ForumThreads.GetThread(id, true);
                        if (thread != null)
                            ForumThreads.SetLock(thread.ThreadID, isLocked);
                    }
                }
            }
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_BatchSetTopSucceed", applicationID));
            return RedirectToAction("ManageForumThreads", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 批量转移主题帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult BatchMoveThreads(string clubDomainName)
        {

            Club.Club currentClub = Clubs.GetClub(clubDomainName, false);
            if (currentClub == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format("圈子：{0}不存在！", clubDomainName)));

            if (CurrentUser == null && !ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins()))
                return Redirect(SiteUrls.Instance().Error404("你没有此项操作的权限！"));

            string[] strThreadIDs = null;
            int toSectionID = Request.Form.GetInt("toSectionID", 0);
            if (toSectionID <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_SelectMoveModuel", applicationID);
                return PartialView("Controls/Forums/MoveForumTopices.ascx");
            }
            if (string.IsNullOrEmpty(Request.Form.GetString("threadIDs", string.Empty)))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_SelectOneThread", applicationID);
                return PartialView("Controls/Manage/MoveForumTopices.ascx");
            }

            strThreadIDs = Request.Form.GetString("threadIDs", string.Empty).Split(',');

            if (strThreadIDs != null)
                foreach (string strThreadID in strThreadIDs)
                {
                    int threadID = -1;
                    int.TryParse(strThreadID, out threadID);
                    if (threadID > 0)
                        ForumThreads.MoveThread(threadID, toSectionID);
                }

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "移动主题操作成功";
            return PartialView("Controls/Manage/MoveForumThread.ascx");
        }
        /// <summary>
        /// 批量设置主题帖类别
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult BatchSetThreadCategory(string clubDomainName, int categoryID)
        {

            Club.Club currentClub = Clubs.GetClub(clubDomainName, false);
            if (currentClub == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format("圈子：{0}不存在！", clubDomainName)));

            if (CurrentUser == null || !ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins()))
                return Redirect(SiteUrls.Instance().Error404("你没有此项操作的权限！"));

            string[] strThreadIDs = Request.Form.GetValues("threadIDCheckBox");
            if (categoryID <= 0)
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_BatchSetType", applicationID));

            if (strThreadIDs == null || strThreadIDs.Length <= 0)
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_SelectOneThread", applicationID));

            foreach (string strThreadID in strThreadIDs)
            {
                int threadID = -1;
                int.TryParse(strThreadID, out threadID);
                if (threadID > 0)
                {
                    ForumThread thread = ForumThreads.GetThread(threadID, true);
                    thread.CategoryID = categoryID;
                    ForumThreads.UpdateThread(thread);
                }
            }
            return RedirectToAction("ManageForumThreads", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 设置主题高亮
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Control_BatchSetHighlight(string clubDomainName)
        {
            string strThreadIDs = Request.Form.GetString("threadIDCheckBox", string.Empty);

            if (!string.IsNullOrEmpty(strThreadIDs))
                ViewData["threadIDs"] = strThreadIDs;
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选择要设置高亮的主题！";
            }

            return PartialView("Controls/Manage/SetHighlight.ascx");
        }

        /// <summary>
        /// 批量设置高亮
        /// </summary>
        /// <param name="sectionID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult BatchSetHighlight(string clubDomainName)
        {

            Club.Club currentClub = Clubs.GetClub(clubDomainName, false);
            if (CurrentUser == null || !ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins()))
                return Redirect(SiteUrls.Instance().Error404("你没有此项操作的权限！"));

            string[] setEssentialThreadIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("threadIDs", string.Empty)))
            {
                setEssentialThreadIDs = Request.Form.GetString("threadIDs", string.Empty).Split(',');
                if (setEssentialThreadIDs != null && setEssentialThreadIDs.Length > 0)
                {
                    StringBuilder highlightstyle = new StringBuilder();
                    if (Request.Form.GetBool("IsHighlight", false))
                    {
                        if (!string.IsNullOrEmpty(Request.Form.GetString("Fonts", string.Empty)))
                            highlightstyle.Append(Request.Form.GetString("Fonts", string.Empty));

                        if (!string.IsNullOrEmpty(Request.Form.GetString("FontSize", string.Empty)))
                            highlightstyle.Append(Request.Form.GetString("FontSize", string.Empty));

                        if (!string.IsNullOrEmpty(Request.Form.GetString("FontStyle", string.Empty)))
                        {
                            string[] fontStyles = Request.Form.GetString("FontStyle", string.Empty).Split(',');
                            foreach (string fontStyle in fontStyles)
                            {
                                highlightstyle.Append(fontStyle);
                            }
                        }

                        if (!string.IsNullOrEmpty(Request.Form.GetString("FontColor", string.Empty)))
                            highlightstyle.Append("color:" + Request.Form.GetString("FontColor", string.Empty) + ";");

                        if (!string.IsNullOrEmpty(Request.Form.GetString("BGColor", string.Empty)))
                            highlightstyle.Append("background-color:" + Request.Form.GetString("BGColor", string.Empty) + ";");
                    }

                    foreach (string threadID in setEssentialThreadIDs)
                    {
                        int id = 0;
                        int.TryParse(threadID, out id);

                        ForumThread thread = ForumThreads.GetThread(id, true);

                        if (Request.Form.GetBool("IsHighlight", false) && !string.IsNullOrEmpty(highlightstyle.ToString()))
                        {
                            thread.HighlightStyle = "style=\"" + highlightstyle.ToString() + "\"";
                            thread.HighlightDate = Request.Form.Get<DateTime>("HighlightDate", DateTime.Now);
                        }
                        else
                            thread.HighlightStyle = string.Empty;

                        if (thread != null)
                            ForumThreads.UpdateThread(thread);

                    }
                }
            }

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "设置主题高亮成功！";

            return PartialView("Controls/Manage/SetHighlight.ascx");
        }

        /// <summary>
        /// 用于楼层跳转
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="pageIndex"></param>
        /// <param name="floorNum"></param>
        /// <returns></returns>
        public ActionResult ForumElevator(string clubDomainName, int threadID, string floor)
        {

            ForumThread thread = ForumThreads.GetThread(threadID, false);
            if (thread == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ThreadIDNotExist", applicationID), threadID)));

            int? pageIndex;
            int fNum = 0;
            int.TryParse(floor, out fNum);

            RouteValueDictionary dictionary = new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "threadID", threadID } };
            string url = SPBUrlHelper.Action("ShowForumThread", "ClubForum", dictionary);

            if (fNum <= thread.ReplyCount && fNum > 0)
            {
                pageIndex = fNum / ForumConfiguration.Instance().PostsPerPage;

                if (fNum % ForumConfiguration.Instance().PostsPerPage > 0)
                    pageIndex++;

                if (pageIndex > 0)
                    dictionary.Add("pageIndex", pageIndex);
                url = SPBUrlHelper.Action("ShowForumThread", "ClubForum", dictionary) + "#f_" + fNum;
            }

            return Redirect(url);
        }

        #endregion

        #region ForumPost

        /// <summary>
        /// 批量删除圈子帖子
        /// </summary>
        public ActionResult DeleteForumPosts(string clubDomainName, string returnUrl)
        {
            if (!ClubPermission.AccessCheck(GetCurrentClub(clubDomainName), CurrentUser, PermissionItemKeys.Instance().ApproveClubForumPost(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            string strPostIDs = Request.Form.GetString("postIDCheckBox", string.Empty);
            if (!string.IsNullOrEmpty(strPostIDs))
            {
                List<int> intPostIDs = ValueHelper.ParseInt(strPostIDs.Split(','));
                foreach (var item in intPostIDs)
                {
                    ForumPosts.DeletePost(item);
                }
            }
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 帖子审核管理
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult ManageForumPosts(AuditingStatusesForDisplay? auditingStatusesForDisplay, int? pageIndex, string clubDomainName)
        {


            if (!ClubPermission.AccessCheck(GetCurrentClub(clubDomainName), CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            #region PageTitle & BreadCrumb
            SetPageTitle("帖子管理", clubDomainName);

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Forum", applicationID), SiteUrls.Instance().ChannelForumHome());
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_TopicAuditingManage", applicationID), string.Empty);
            #endregion

            if (auditingStatusesForDisplay.HasValue)
                ViewData["auditingStatusesForDisplay"] = auditingStatusesForDisplay;
            int status = Request.QueryString.GetInt("SelectStatuses", -1);
            if (status > 0)
                ViewData["auditingStatusesForDisplay"] = (AuditingStatusesForDisplay)status;
            ViewData["pageIndex"] = pageIndex;
            ViewData["currentClub"] = GetCurrentClub(clubDomainName);
            return this.View("Pages/Manage/ManageForumPosts.aspx", "Masters/Manage.master");
        }

        /// <summary>
        /// 帖子管理列表
        /// </summary>   
        public ActionResult Control_ListForumPosts(AuditingStatusesForDisplay? auditingStatusesForDisplay, int? pageIndex, string clubDomainName)
        {
            Club.Club currentClub = GetCurrentClub(clubDomainName);
            int groupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(currentClub.ClubID);

            string KeyWords = Request.QueryString.GetString("KeyWords", string.Empty);
            int autoID = Request.QueryString.GetInt("autoID", 0);
            int forumSectionSselect = Request.QueryString.GetInt("forumSectionSselect", -1);
            int selectStatuses = Request.QueryString.GetInt("SelectStatuses", -1);
            List<int> sectionIDs = new List<int>();
            if (selectStatuses <= 0)
            {
                sectionIDs = ForumSections.GetIndentedSectionsOfGroup(OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(currentClub.ClubID), null).Select(n => n.SectionID).ToList();
            }
            else
                sectionIDs = new List<int> { forumSectionSselect };

            PagingDataSet<ForumPost> pds = ForumPosts.GetPostsForManage(KeyWords, groupID, autoID, sectionIDs, auditingStatusesForDisplay, 15, pageIndex ?? 1);
            ViewData["auditingStatusesForDisplay"] = auditingStatusesForDisplay;
            ViewData["pageIndex"] = pageIndex;

            return PartialView("Controls/Manage/ListForumPosts.ascx", pds);
        }
        /// <summary>
        /// 帖子审核管理时用于查看帖子详细内容
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShowForumPost(int postID)
        {
            ForumPost forumPost = ForumPosts.GetPost(postID, true);
            return PartialView("Controls/Manage/ShowForumPost.ascx", forumPost);
        }
        /// <summary>
        /// 批量批准论坛帖子
        /// </summary>    
        public ActionResult SetForumAuditingStatus(AuditingStatuses auditingStatuses, int? pageIndex, AuditingStatuses auditingStatusesForDisplay, string KeyWords, int? autoID)
        {
            if (!CurrentUser.IsContentAdministrator && !ForumSections.IsModeratorOfAnySection(CurrentUser.UserName))
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }

            if (!string.IsNullOrEmpty(Request.Form.GetString("postIDCheckBox", string.Empty)))
            {
                string[] forumPostIDs = Request.Params.GetString("postIDCheckBox", string.Empty).Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);

                if (forumPostIDs != null && forumPostIDs.Length > 0)
                {
                    List<int> postIDs = ValueHelper.ParseInt(forumPostIDs);
                    ForumPosts.UpdatePostsAuditingStatus(postIDs, auditingStatuses);
                }
            }
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_TopicPass", applicationID));
            return RedirectToAction("ManageForumPosts", new RouteValueDictionary { { "auditingStatusesForDisplay", auditingStatusesForDisplay }, { "pageIndex", pageIndex }, { "KeyWords", KeyWords }, { "autoID", autoID } });
        }
        /// <summary>
        /// 批量删除主题帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteForumPosts(int? pageIndex, AuditingStatuses auditingStatusesForDisplay)
        {
            if (!CurrentUser.IsContentAdministrator)
                throw new SPBException(ExceptionTypes.Instance().GeneralAccessDenied());
            string[] deletePostsIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("postIDCheckBox", string.Empty)))
            {
                deletePostsIDs = Request.Form.GetString("postIDCheckBox", string.Empty).Split(',');
                if (deletePostsIDs != null && deletePostsIDs.Length > 0)
                {
                    foreach (string postID in deletePostsIDs)
                    {
                        int id = 0;
                        int.TryParse(postID, out id);
                        ForumPost post = ForumPosts.GetPost(id, true);
                        if (post != null)
                        {
                            ForumPosts.DeletePost(post.PostID);
                        }
                    }
                }
            }
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_BatchDeleteSucceed", applicationID));
            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 显示圈子帖子审核
        /// </summary>
        public ActionResult ApproveClubForumPosts(string clubDomainName, ForumThreadSortBy? sortBy, int? pageIndex)
        {
            Club.Club club = Clubs.GetClub(clubDomainName, false);
            if (club == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));
            }
            if (!ClubPermission.AccessCheck(club, CurrentUser, PermissionItemKeys.Instance().ApproveClubForumPost(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            #region PageTitle & BreadCrumb
            string applicationNameInclub = GetResourceString("Title_Topic", applicationID);
            SetPageTitle(applicationNameInclub + GetResourceString("Title_Home", applicationID), clubDomainName);
            BreadCrumb.AddBreadCrumbItem(club.ClubName, SiteUrls.Instance().ClubDomainHome(clubDomainName));
            BreadCrumb.AddBreadCrumbItem(applicationNameInclub, SPBUrlHelper.Action("Home", "ClubForum", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));
            #endregion

            int groupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(club.ClubID);

            PagingDataSet<ForumPost> pds = ForumPosts.GetPostsForManage(string.Empty, groupID, null, null, AuditingStatusesForDisplay.Fail, 15, pageIndex ?? 1);
            ViewData["ApproveClubForumPosts"] = pds;
            ViewData["SortBy"] = sortBy;
            return this.View("Pages/Forums/Home.aspx", "Masters/Forum.master", null);
        }

        /// <summary>
        /// 批量批准通过审核的帖子
        /// </summary>
        public ActionResult SetForumPostsAuditingStatus(string clubDomainName, string returnUrl)
        {
            if (!ClubPermission.AccessCheck(GetCurrentClub(clubDomainName), CurrentUser, PermissionItemKeys.Instance().ApproveClubForumPost(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            string strPostIDs = Request.Form.GetString("postIDCheckBox", string.Empty);
            if (!string.IsNullOrEmpty(strPostIDs))
            {
                List<int> intPostIDs = ValueHelper.ParseInt(strPostIDs.Split(','));
                ForumPosts.UpdatePostsAuditingStatus(intPostIDs, AuditingStatuses.Success);
            }
            return Redirect(returnUrl);
        }

        #endregion

        #region ForumSection

        /// <summary>
        /// 创建更新版块
        /// </summary>
        /// <param name="sectionID"></param>
        /// <param name="parentID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateUpdateForumSection(string clubDomainName, int? sectionID)
        {
            if (!ClubPermission.AccessCheck(GetCurrentClub(clubDomainName), CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            int? statusMessageType = null;
            string resourceString = null;

            ForumSection section = new ForumSection();
            if (sectionID.HasValue && sectionID.Value > 0)
                section = ForumSections.GetSection(sectionID.Value, true);
            else
                section.GroupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(GetCurrentClub(clubDomainName).ClubID);

            section.SectionName = Request.Form.GetString("forumSectionName", string.Empty).Trim();
            section.Description = Request.Form.GetString("description", string.Empty).Trim();

            section.Rules = Request.Form.GetString("rules", string.Empty).Trim();
            section.DisplayOrder = Request.Form.GetInt("displayOrder", 0);
            section.EnableAnonymousPosting = Request.Form.GetBool("enableAnonymousPosting", true);
            section.EnableThreadStatus = Request.Form.GetBool("enableThreadStatus", true);
            section.ThreadCategoryStatus = (ThreadCategoryStatuses)Request.Form.GetInt("threadCategoryStatus", (int)section.ThreadCategoryStatus);
            section.IsActive = Request.Form.GetBool("isActive", true);
            System.Web.HttpPostedFileBase picFile = null;
            if (Request.Files != null && Request.Files.Count > 0)
                picFile = Request.Files[0];

            if (sectionID.HasValue && sectionID.Value > 0)
            {
                ForumSections.Update(section, picFile);
                statusMessageType = (int)StatusMessageType.Success;
                resourceString = "更新板块成功！";
            }
            else
            {
                ForumSections.Create(section, picFile);
                if (section.SectionID > 0)
                {
                    statusMessageType = (int)StatusMessageType.Success;
                    resourceString = GetResourceString("Message_CreateModuelSucceed", applicationID);
                }
                else
                {
                    TempData["section"] = section;
                    statusMessageType = (int)StatusMessageType.Error;
                    resourceString = GetResourceString("Message_CreateModuelLost", applicationID);
                }
            }

            if (statusMessageType != null && statusMessageType.HasValue)
            {
                TempData["StatusMessageType"] = statusMessageType;
                TempData["StatusMessageContent"] = resourceString;
                if (statusMessageType == (int)StatusMessageType.Success)
                    return RedirectToAction("ManageForumSections", new RouteValueDictionary { { "clubDomainName", clubDomainName } });
            }

            return RedirectToAction("EditForumSection", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "sectionID", section.SectionID } });


        }

        /// <summary>
        /// 删除版块
        /// </summary>
        /// <param name="sectionID"></param>
        /// <returns></returns>
        public ActionResult DeleteForumSection(string clubDomainName, int sectionID)
        {
            if (!ClubPermission.AccessCheck(GetCurrentClub(clubDomainName), CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            ForumSection section = ForumSections.GetSection(sectionID, true);
            ForumSections.Delete(section);
            return RedirectToAction("Control_ManageForumSections");
        }

        /// <summary>
        /// 管理版块
        /// </summary>
        /// <returns></returns>
        public ActionResult ManageForumSections(string clubDomainName)
        {
            Club.Club currentClub = GetCurrentClub(clubDomainName);
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            SetPageTitle("管理板块", clubDomainName);

            string applicationNameInclub = GetResourceString("Title_Topic", applicationID);
            BreadCrumb.AddBreadCrumbItem(currentClub.ClubName, SPBUrlHelper.Action("Home", "ClubDomain", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Manage", applicationID), SPBUrlHelper.Action("Home", "ClubManage", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ManageForumModuel", applicationID), "");

            return View("Pages/Manage/ManageForumSections.aspx", "Masters/Manage.master");
        }

        /// <summary>
        /// 管理版块列表
        /// </summary>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ManageForumSections(string clubDomainName)
        {
            Club.Club currentClub = GetCurrentClub(clubDomainName);
            int groupID = OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(currentClub.ClubID);
            List<ForumSection> forumSections = ForumSections.GetIndentedSectionsOfGroupForAdmin(groupID, null);
            return PartialView("Controls/Manage/ManageForumSections.ascx", forumSections);
        }

        /// <summary>
        /// 合并或移动版块
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult MergeSection(string clubDomainName, int fromSectionID)
        {
            if (!ClubPermission.AccessCheck(GetCurrentClub(clubDomainName), CurrentUser, PermissionItemKeys.Instance().ManageClubForumSectioins(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            int toSectionID = Request.Form.GetInt("sectionDDL", 0);
            ForumSections.MergeSection(fromSectionID, toSectionID);
            fromSectionID = toSectionID;

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;

            return RedirectToAction("Control_MergeSection", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "FilterID", fromSectionID } });
        }

        #endregion

        #endregion

        #region Help methods

        /// <summary>
        /// 设置论坛版块面包屑
        /// </summary>
        private void SetClubForumSectionBreadCrumb(string clubDomainName, int sectionID)
        {
            ClubPermission.AccessCheck(GetCurrentClub(clubDomainName), CurrentUser, PermissionItemKeys.Instance().ViewClubForumThread(), HttpContext, SiteUrls.Instance().Login(true));

            Stack<ForumSection> stack = new Stack<ForumSection>();
            ForumSection currentSection = ForumSections.GetSection(sectionID, false);
            while (currentSection != null)
            {
                stack.Push(currentSection);
                currentSection = ForumSections.GetSection(currentSection.ParentID, false);
            }
            while (stack.Count > 0)
            {
                currentSection = stack.Pop();
                BreadCrumb.AddBreadCrumbItem(currentSection.SectionName, SiteUrls.Instance().ShowForumSectionInClub(clubDomainName, currentSection.SectionID));
            }
        }

        /// <summary>
        /// 设置圈子论坛主题详细显示页面的meta标签
        /// </summary>
        private void SetPageMeta(ForumThread thread)
        {
            string metaKeyWords = string.Empty;
            string metaDescription = string.Empty;

            //标签
            if (thread.Tags != null && thread.Tags.Count > 0)
            {
                foreach (string tagName in thread.Tags)
                {
                    if (tagName != null)
                    {
                        metaKeyWords = string.Format("{1},{0}", metaKeyWords, tagName);
                    }
                }
            }

            //标题和用户名
            metaKeyWords = metaKeyWords + thread.Subject;
            metaKeyWords = string.Format("{0},{1}", metaKeyWords, thread.Author);

            metaDescription = string.Format("{0},{1}", thread.Subject, HtmlUtils.TrimHtml(thread.Body, 150));

            Header.AddMetaDescription(metaDescription);
            Header.AddMetaKeywords(metaKeyWords);
        }


        /// <summary>
        /// 是否具有管理圈子权限
        /// </summary>
        protected bool HasManagePermission(string clubDomainName)
        {
            if (clubDomainName == null)
                throw new SPBException(ExceptionTypes.Instance().ClubDomainNotFond());
            Club.Club currentClub = Clubs.GetClub(clubDomainName, false);
            return (CurrentUser != null && (currentClub.ClubOwner.UserID == CurrentUser.UserID || CurrentUser.IsContentAdministrator || CurrentUser.IsAdministrator));
        }

        #endregion

    }
}
