﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using SpaceBuilder.Forum;
using SpaceBuilder.Common;
using SpaceBuilder.Web;
using SpaceBuilder.Common.Controllers;

using SpaceBuilder.Utils;
using SpaceBuilder.Club;
using System.Web.Routing;
using System.Web.UI;

namespace SpaceBuilder.Forum.Controllers
{
    [ManageAuthorize]
    [HttpCompress]
    public class ManageForumController : ManageControllerBase
    {
        #region ForumSectionGroup

        #region Show

        /// <summary>
        /// 编辑版块组
        /// </summary>
        /// <param name="groupID"></param>
        /// <returns></returns>
        int forumApplication = ApplicationIDs.Instance().Forum();
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_EditForumGroup(int? groupID)
        {
            ForumSectionGroup forumGroup = new ForumSectionGroup();
            if (groupID.HasValue)
            {
                forumGroup = ForumSectionGroups.GetGroup(groupID.Value, true);
            }
            else
            {
                if (TempData["forumGroup"] != null)
                {
                    forumGroup = TempData["forumGroup"] as ForumSectionGroup;
                }
            }

            return this.PartialView("Controls/Forums/EditForumGroup.ascx", forumGroup);
        }

        #endregion

        #region Manage

        /// <summary>
        /// 创建更新版块组
        /// </summary>
        /// <param name="groupID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateUpdateForumGroup(int? groupID)
        {
            ForumSectionGroup forumGroup = new ForumSectionGroup();
            if (groupID.HasValue && groupID.Value > 0)
            {
                forumGroup = ForumSectionGroups.GetGroup(groupID.Value, true);
            }
            else
            {
                forumGroup.PresentAreaID = PresentAreaIDs.Instance().Channel();

            }
            forumGroup.GroupName = Request.Form.GetString("forumGroupName", string.Empty).Trim();
            forumGroup.DisplayOrder = Request.Form.GetInt("forumGroupDisplayOrder", 0);

            if (groupID.HasValue && groupID.Value > 0)
            {
                ForumSectionGroups.UpdateGroup(forumGroup);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_UpdateModuelSucceed", forumApplication);
            }
            else
            {
                ForumSectionGroups.CreateGroup(forumGroup);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_CreateModuelTeamSucceed", forumApplication);
            }
            return RedirectToAction("Control_EditForumGroup", new RouteValueDictionary { { "groupID", forumGroup.GroupID } });
        }

        /// <summary>
        /// 删除版块组
        /// </summary>
        /// <param name="clubID"></param>
        /// <returns></returns>
        public ActionResult DeleteForumGroup(int groupID)
        {
            ForumSectionGroup forumGroup = ForumSectionGroups.GetGroup(groupID, true);
            ForumSectionGroups.DeleteGroup(forumGroup);
            return RedirectToAction("ManageForumSections");
        }

        /// <summary>
        ///将板块组转换为群组 
        /// </summary>
        /// <param name="groupID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ConvertForumGroupToClub(int groupID)
        {
            ForumSectionGroup forumGroup = ForumSectionGroups.GetGroup(groupID, true);
            if (forumGroup == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format("未找到ID为{0}的板块组！", groupID)));
            ViewData["ConvertForum"] = forumGroup;
            Club.Club convertCLub = null;
            if (forumGroup.OwnerID > 0)
                convertCLub = Clubs.GetClub(forumGroup.OwnerID, true);
            else
            {
                convertCLub = new SpaceBuilder.Club.Club();
            }
            ViewData["Club"] = convertCLub;
            ClubDomainConfiguration clubDomainConfiguration = ClubDomainConfiguration.Instance();
            ViewData["SiteCategoryLevel"] = clubDomainConfiguration.SiteCategoryLevel;
            ViewData["AreaLevel"] = clubDomainConfiguration.AreaLevel;
            ViewData["RootAreaCode"] = clubDomainConfiguration.RootAreaCode;

            return this.PartialView("Controls/Forums/ConvertForumGroupToClub.ascx");
        }

        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult ConvertForumGroupToClub(int clubID, int groupID)
        {
            Club.Club club = null;
            if (clubID > 0)
                club = Clubs.GetClub(clubID, true);
            else
                club = new SpaceBuilder.Club.Club();

            string clubOwnerName = Request.Form.GetString("clubOwnerName", string.Empty).Trim();
            club.IsEnabled = true;
            club.ClubName = Request.Form.GetString("clubName", string.Empty);
            if (string.IsNullOrEmpty(club.DomainName))
            {
                club.DomainName = Request.Form.GetString("clubDomain", club.DomainName).Trim();
                if (Clubs.DomainNameIsExist(club.DomainName))
                {
                    TempData["newclub"] = club;
                    TempData["StatusMessage"] = GetResourceString("Message_WebExist", forumApplication);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    return RedirectToAction("Control_ConvertForumGroupToClub", new RouteValueDictionary { { "groupID", groupID } });
                }
            }
            club.Brief = Request.Form.GetString("brief", club.Brief);
            if (!string.IsNullOrEmpty(club.Brief))
                club.Brief = club.Brief.Trim();
            club.CategoryID = Request.Form.GetInt("categoryID", 0);
            club.AreaCode = Request.Form.GetString("areaCode", string.Empty);
            club.IsPublic = Request.Form.GetInt("clubSecrecyType", 0) > 0 ? true : false;
            club.JoinNotice = Request.Form.GetString("joinNotice", string.Empty);
            if (club.IsPublic)
                club.IsMemberNeedAuthorize = Request.Form.GetBool("needApproveCheckbox", false);

            User u = Users.GetUser(clubOwnerName, true);
            if (u == null)
            {
                TempData["newclub"] = club;
                TempData["StatusMessage"] = GetResourceString("Message_ClubHostUserNameNotExist", forumApplication);
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                return RedirectToAction("Control_ConvertForumGroupToClub", new RouteValueDictionary { { "groupID", groupID } });
            }
            club.ClubOwnerUserID = u.UserID;

            if (clubID > 0)
                Clubs.UpdateClub(club);
            else
                Clubs.CreateClub(club.ClubOwnerUserID, club);

            ForumSectionGroups.ConvertSectionGroupToClub(groupID, club.ClubID);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            return RedirectToAction("Control_ConvertForumGroupToClub", new RouteValueDictionary { { "groupID", groupID } });
        }
        #endregion

        #endregion

        #region ForumSection

        #region Show

        /// <summary>
        /// 管理论坛版块
        /// </summary>
        /// <param name="groupID"></param>
        /// <returns></returns>
        public ActionResult ManageForumSections()
        {
            Dictionary<ForumSectionGroup, List<ForumSection>> dict = new Dictionary<ForumSectionGroup, List<ForumSection>>();

            IList<ForumSectionGroup> groups = ForumSectionGroups.GetSiteGroups(true);
            foreach (ForumSectionGroup group in groups)
            {
                List<ForumSection> forumSections = ForumSections.GetIndentedSectionsOfGroupForAdmin(group.GroupID, null);
                dict.Add(group, forumSections);
            }
            return View("Pages/Forums/ManageForumSections.aspx", "Masters/Forum.master", dict);
        }

        /// <summary>
        /// 编辑论坛版块
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_EditForumSection(int? sectionID, int? parentID, bool? isChildSection)
        {
            ForumSection section = null;
            if (sectionID.HasValue && sectionID.Value > 0)
            {
                section = ForumSections.GetSection(sectionID.Value, true);
            }
            else if (parentID.HasValue && parentID.Value > 0)
            {
                ViewData["parentID"] = parentID.Value;

                if (TempData["section"] != null)
                {
                    section = TempData["section"] as ForumSection;
                }
            }
            if (section == null)
                section = new ForumSection();

            ViewData["isChildSection"] = isChildSection;

            return this.PartialView("Controls/Forums/EditForumSection.ascx", section);
        }

        /// <summary>
        /// 合并或移动版块
        /// </summary>
        /// <param name="isMerge"></param>
        /// <param name="FilterID">需要排除当前栏目及子栏目</param>
        /// <returns></returns>
        public ActionResult Control_MergeOrMoveSection(bool isMerge, int filterID)
        {
            ForumSection fromSection = ForumSections.GetSection(filterID, true);

            ViewData["isMerge"] = isMerge;
            ViewData["fromSection"] = fromSection;

            return PartialView("Controls/Forums/MergeOrMoveSection.ascx");
        }

        #endregion

        #region Manage

        /// <summary>
        /// 创建更新版块
        /// </summary>
        /// <param name="sectionID"></param>
        /// <param name="parentID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateUpdateForumSection(int? sectionID, int? parentID, bool isChildSection)
        {
            ForumSection section = new ForumSection();
            if (sectionID.HasValue && sectionID.Value > 0)
            {
                section = ForumSections.GetSection(sectionID.Value, true);
            }
            else if (parentID.HasValue && parentID.Value > 0)
            {
                if (isChildSection)
                    section.ParentID = parentID.Value;
                else
                    section.GroupID = parentID.Value;
            }

            section.SectionName = Request.Form.GetString("forumSectionName", string.Empty).Trim();
            section.Description = Request.Form.GetString("description", string.Empty).Trim();

            List<string> list = new List<string>();
            string[] formattedTagNames = Request.Form.GetString("ImmediateModeratorUserNames", string.Empty).Trim().Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);

            if (formattedTagNames != null && formattedTagNames.Length > 0)
            {
                foreach (string formattedTagName in formattedTagNames)
                {
                    if (!string.IsNullOrEmpty(formattedTagName))
                    {
                        list.Add(formattedTagName);
                    }
                }
            }

            section.ImmediateModeratorUserNames = list;
            section.InheriteModerator = Request.Form.GetBool("inheriteModerator", true);
            section.Rules = Request.Form.GetString("rules", string.Empty).Trim();
            section.DisplayOrder = Request.Form.GetInt("displayOrder", 0);
            section.EnableAnonymousPosting = Request.Form.GetBool("enableAnonymousPosting", true);
            section.EnableThreadStatus = Request.Form.GetBool("enableThreadStatus", true);
            section.IsActive = Request.Form.GetBool("isActive", true);
            section.ThreadCategoryStatus = (ThreadCategoryStatuses)Request.Form.GetInt("threadCategoryStatus", (int)section.ThreadCategoryStatus);
            System.Web.HttpPostedFileBase picFile = null;
            if (Request.Files != null && Request.Files.Count > 0)
                picFile = Request.Files[0];

            if (sectionID.HasValue && sectionID.Value > 0)
            {
                try
                {
                    ForumSections.Update(section, picFile);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "更新版块操作已成功";
                }
                catch (Exception)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "更新版块失败，请重试";
                    TempData["section"] = section;
                }
            }
            else
            {
                try
                {
                    ForumSections.Create(section, picFile);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "创建版块操作已成功";
                }
                catch (Exception)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "创建版块失败，请重试";
                    TempData["section"] = section;
                }
            }
            return RedirectToAction("Control_EditForumSection", new RouteValueDictionary { { "sectionID", section.SectionID }, { "isChildSection", isChildSection } });

        }

        /// <summary>
        /// 删除版块
        /// </summary>
        /// <param name="sectionID"></param>
        /// <returns></returns>
        public ActionResult DeleteForumSection(int sectionID)
        {
            ForumSection section = ForumSections.GetSection(sectionID, true);
            ForumSections.Delete(section);
            return RedirectToAction("ManageForumSections");
        }

        /// <summary>
        /// 合并或移动版块
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult MergeOrMoveSection(bool isMerge, int fromSectionID)
        {
            string sectionDDL = Request.Form.GetString("sectionDDL", string.Empty);
            int toSectionID = 0;
            int toClubID = 0;

            if (sectionDDL.StartsWith("g-"))
            {
                toClubID = Convert.ToInt32(sectionDDL.Substring(2));
            }
            else
            {
                toSectionID = Request.Form.GetInt("sectionDDL", 0);
            }

            if (isMerge)
            {
                ForumSections.MergeSection(fromSectionID, toSectionID);
                fromSectionID = toSectionID;
            }
            else
            {
                ForumSections.MoveSection(fromSectionID, toSectionID, toClubID);
            }

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;

            return RedirectToAction("Control_MergeOrMoveSection", new RouteValueDictionary { { "isMerge", true }, { "FilterID", fromSectionID } });
        }

        #endregion

        #endregion

        #region ForumThreadCategory
        /// <summary>
        /// 编辑主题分类
        /// </summary>
        public ActionResult Control_EditForumThreadCategory(int sectionID, int? categoryID)
        {
            ForumThreadCategory category = null;
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                category = ForumThreadCategories.GetForumThreadCategory(categoryID.Value, true);
                if (category == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format("找不到要更新CategoryID为{0}的主题分类", categoryID.Value)));
                }
            }
            else if (TempData["category"] != null)
            {
                category = TempData.Get<ForumThreadCategory>("category");
            }
            if (category == null)
            {
                category = new ForumThreadCategory();
                category.SectionID = sectionID;
            }
            return PartialView("Controls/Forums/EditForumThreadCategory.ascx", category);
        }
        /// <summary>
        /// 创建,更新主题分类
        /// </summary>
        public ActionResult CreateUpdateForumThreadCategory(int sectionID, int? categoryID)
        {
            ForumThreadCategory category = null;
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                category = ForumThreadCategories.GetForumThreadCategory(categoryID.Value, true);
                if (category == null)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "抱歉，找不到你要更新的主题分类");
                    return RedirectToAction("Control_EditForumThreadCategory", new RouteValueDictionary { { "sectionID", sectionID }, { "categoryID", categoryID } });
                }
            }
            else
            {
                category = new ForumThreadCategory();
            }
            category.CategoryName = Request.Form.GetString("categoryName", string.Empty);
            category.CategoryDescription = Request.Form.GetString("categoryDescription", string.Empty);
            category.SectionID = sectionID;
            category.DisplayOrder = Request.Form.GetInt("displayOrder", 0);
            if (string.IsNullOrEmpty(category.CategoryName))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "主题分类不能为空");
                TempData["category"] = category;
                return RedirectToAction("Control_EditForumThreadCategory", new RouteValueDictionary { { "sectionID", sectionID }, { "categoryID", categoryID } });
            }
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                ForumThreadCategories.UpdateForumThreadCategory(category);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "更新论坛主题分类操作已成功");
            }
            else
            {
                ForumThreadCategories.CreateForumThreadCategory(category);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "创建论坛主题分类操作已成功");
            }
            return RedirectToAction("Control_EditForumThreadCategory", new RouteValueDictionary { { "sectionID", sectionID }, { "categoryID", categoryID } });
        }
        /// <summary>
        /// 删除主题分类
        /// </summary>
        public ActionResult DeleteForumThreadCategory(int categoryID)
        {
            ForumThreadCategory category = ForumThreadCategories.GetForumThreadCategory(categoryID, true);
            if (category == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format("找不到CategoryID为{0}的主题分类", categoryID)));
            }
            int sectionID = category.SectionID;
            ForumThreadCategories.DeleteForumThreadCategory(category);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "删除论坛主题分类操作已成功");
            return RedirectToAction("ManageForumThreadCategories", new RouteValueDictionary { { "sectionID", sectionID } });
        }
        /// <summary>
        /// 显示版块的主题分类页
        /// </summary>
        public ActionResult ManageForumThreadCategories(int sectionID)
        {
            List<ForumThreadCategory> categories = ForumThreadCategories.GetForumThreadCategories(sectionID, true);
            if (categories == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format("找不到SectionID为{0}的版块下的所有主题分类", sectionID)));
            }
            ViewData["section"] = ForumSections.GetSection(sectionID, true);
            return View("Pages/Forums/ManageForumThreadCategories.aspx", "Masters/Forum.master", categories);
        }

        #endregion

        #region ForumThreads

        /// <summary>
        /// 管理论坛主题
        /// </summary>
        public ActionResult ManageForumThreads(int? presentAreaID, AuditingStatusesForDisplay? auditingStatusesForDisplay, int? pageIndex)
        {
            if (!presentAreaID.HasValue)
                presentAreaID = PresentAreaIDs.Instance().Channel();

            int sponsorUserID = Request.QueryString.GetInt("userID", -1);
            string subjectKeyword = Request.QueryString.GetString("subjectKeyword", string.Empty);
            int groupID = Request.QueryString.GetInt("groupID", -1);
            int sectionID = Request.QueryString.GetInt("sectonID", -1);
            ViewData["AuditingStatusesForDisplay"] = auditingStatusesForDisplay ?? AuditingStatusesForDisplay.Pending;
            PagingDataSet<ForumThread> pds = ForumThreads.GetThreadsForManage(subjectKeyword, groupID, sectionID, sponsorUserID, presentAreaID, auditingStatusesForDisplay ?? AuditingStatusesForDisplay.Pending, 15, pageIndex ?? 1);
            return View("Pages/Forums/ManageForumThreads.aspx", "Masters/Forum.master", pds);
        }

        /// <summary>
        /// 更新审核状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateThreadApproveStatus(AuditingStatuses auditingStatus)
        {
            string[] strThreadIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("newsCheckBoxGroup", string.Empty)))
            {
                strThreadIDs = Request.Form.GetString("newsCheckBoxGroup", string.Empty).Split(',');
                if (strThreadIDs != null && strThreadIDs.Length > 0)
                {
                    List<int> threadIDs = new List<int>();
                    foreach (string threadID in strThreadIDs)
                    {
                        int int_threadID = 0;
                        int.TryParse(threadID, out int_threadID);

                        if (int_threadID > 0)
                            threadIDs.Add(int_threadID);
                    }
                    ForumThreads.UpdateThreadsAuditingStatus(threadIDs, auditingStatus);
                }
            }
            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 搜索主题
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SearchThreadForum(AuditingStatusesForDisplay? auditingStatusesForDisplay, int? sectionID, int? groupID, int presentAreaID)
        {
            string userID = Request.Form.GetString("userID", string.Empty);
            string subjectKeyword = Request.Form.GetString("subjectKeyword", string.Empty);
            return RedirectToAction("ManageForumThreads", new RouteValueDictionary { { "subjectKeyword", subjectKeyword }, { "auditingStatusesForDisplay", auditingStatusesForDisplay }, { "sectionID", sectionID }, { "groupID", groupID }, { "userID", userID }, { "presentAreaID", presentAreaID } });
        }

        #endregion

        #region ForumPosts

        #region Show

        /// <summary>
        /// 评论管理
        /// </summary>
        /// <param name="status"></param>
        /// <param name="pageIndex"></param>
        /// <param name="sectionID"></param>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public ActionResult ManageForumPosts(AuditingStatusesForDisplay? auditingStatusesForDisplay, int? pageIndex)
        {
            int userID = Request.QueryString.GetInt("userID", -1);
            int groupID = Request.QueryString.GetInt("groupID", -1);
            PagingDataSet<ForumPost> pds = ForumPosts.GetPostsForManage(string.Empty, groupID, userID, null, auditingStatusesForDisplay ?? AuditingStatusesForDisplay.Pending, 15, pageIndex ?? 1);
            ViewData["AuditingStatusesForDisplay"] = auditingStatusesForDisplay ?? AuditingStatusesForDisplay.Pending;
            return View("Pages/Forums/ManageForumPosts.aspx", "Masters/Forum.master", pds);
        }

        /// <summary>
        /// 搜索评论
        /// </summary>
        /// <param name="status"></param>
        /// <param name="sectionID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SearchPostForum(int? sectionID, int? groupID)
        {
            string searchUserIDKeywords = Request.Form.GetString("searchUserID", string.Empty);

            return RedirectToAction("ManageForumPosts", new RouteValueDictionary { { "sectionID", sectionID }, { "groupID", groupID }, { "userID", searchUserIDKeywords } });
        }

        #endregion

        #region Manage

        /// <summary>
        /// 批量批准论坛帖子
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdatePostApproveStatus(AuditingStatuses? auditingStatus)
        {
            if (!string.IsNullOrEmpty(Request.Form.GetString("postIDCheckBox", string.Empty)))
            {
                string[] forumPostIDs = Request.Params.GetString("postIDCheckBox", string.Empty).Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                if (forumPostIDs != null && forumPostIDs.Length > 0)
                {
                    List<int> postIDs = new List<int>();
                    foreach (string postID in forumPostIDs)
                    {
                        int int_postID = 0;
                        int.TryParse(postID, out int_postID);

                        if (int_postID > 0)
                            postIDs.Add(int_postID);
                    }
                    ForumPosts.UpdatePostsAuditingStatus(postIDs, auditingStatus ?? AuditingStatuses.Success);
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "帖子批准完成");
                }
            }
            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 批准论坛帖子
        /// </summary>
        public ActionResult SetForumAuditingStatus(int postID)
        {
            try
            {
                List<int> postIDs = new List<int>() { postID };
                ForumPosts.UpdatePostsAuditingStatus(postIDs, AuditingStatuses.Success);

                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "帖子批准完成");
            }
            catch (Exception)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "帖子批准操作失败");
            }

            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 删除回复帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteForumPost(int postID)
        {
            ForumPost post = ForumPosts.GetPost(postID, true);
            if (post == null)
                throw new SPBException(ExceptionTypes.Instance().PostNotFound());

            try
            {
                ForumPosts.DeletePost(postID);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "删除回帖操作已成功！");

            }
            catch (Exception)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "删除回帖操作失败！");
            }

            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 批量删除主题帖
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteForumPosts()
        {
            try
            {
                string[] deletePostsIDs = null;
                if (!string.IsNullOrEmpty(Request.Form.GetString("postIDCheckBox", string.Empty)))
                {
                    deletePostsIDs = Request.Form.GetString("postIDCheckBox", string.Empty).Split(',');
                    if (deletePostsIDs != null && deletePostsIDs.Length > 0)
                    {
                        foreach (string postID in deletePostsIDs)
                        {
                            int id = 0;
                            int.TryParse(postID, out id);
                            ForumPost post = ForumPosts.GetPost(id, true);
                            if (post != null)
                            {
                                ForumPosts.DeletePost(post.PostID);
                            }
                        }
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "批量删除主题帖操作已成功！");
                    }
                }
            }
            catch (Exception)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "批量删除主题帖操作失败！");
            }

            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        #endregion

        #endregion

        #region ForumTags

        #region Show

        /// <summary>
        /// 标签管理
        /// </summary>
        /// <returns></returns>
        public ActionResult ManageForumTags()
        {
            SetPageTitle("标签管理");
            return View("Pages/Forums/ManageForumTags.aspx", "Masters/Forum.master");
        }

        /// <summary>
        /// 编辑标签
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_EditTag(string tagName)
        {
            ApplicationTag tag = new ApplicationTag();
            if (!string.IsNullOrEmpty(tagName))
            {
                tag = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).GetApplicationTag(tagName, true);
            }
            return PartialView("Controls/Forums/EditTag.ascx", tag);
        }

        /// <summary>
        /// 列表显示标签
        /// </summary>
        /// <returns></returns>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListTags(int? pageIndex)
        {
            string keyWords = Request.Form.GetString("keyWords", string.Empty);
            ViewData["keyWords"] = keyWords;
            PagingDataSet<ApplicationTag> pds = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).GetAppliationTags(pageIndex ?? 1, null, keyWords);
            return PartialView("Controls/Forums/ListTags.ascx", pds);
        }

        #endregion

        #region Manage

        /// <summary>
        /// 创建更新标签
        /// </summary>
        /// <param name="tagName"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateUpdateTag(string tagName)
        {
            ApplicationTag tag = new ApplicationTag();
            //if (!string.IsNullOrEmpty(tagName))
            //{
            //    tag = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).GetApplicationTag(tagName, true);

            //    tag.UserID = CurrentUser.UserID;
            //    tag.Author = CurrentUser.DisplayName;
            //    tag.TagName = StringUtils.Trim(Request.Form["tagName"].ToString(), 30, string.Empty);
            //    try
            //    {
            //        ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).UpdateApplicationTag(tag);
            //        TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            //    }
            //    catch (Exception)
            //    {
            //        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
            //        TempData["StatusMessageContent"] = "更新标签失败，请重试";
            //    }
            //}
            //else
            //{
            string[] formattedTagNames = Request.Form["tagName"].Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);

            if (formattedTagNames != null && formattedTagNames.Length > 0)
            {
                foreach (string formattedTagName in formattedTagNames)
                {
                    if (!string.IsNullOrEmpty(formattedTagName))
                    {
                        tag.UserID = CurrentUser.UserID;
                        tag.Author = CurrentUser.DisplayName;
                        tag.TagName = StringUtils.Trim(formattedTagName, 8, string.Empty);
                        tag.TagName = Formatter.CleanTagName(tag.TagName);
                        tag.TagName = Formatter.FormatSingleLinePlainTextForStorage(tag.TagName);

                        try
                        {
                            ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).CreateApplicationTag(tag);
                            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                        }
                        catch (Exception)
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = "创建标签失败，请重试";
                        }
                    }
                }
            }
            //}

            return RedirectToAction("Control_EditTag", new RouteValueDictionary { { "tagName", tag.TagName } });
        }

        /// <summary>
        /// 删除标签
        /// </summary>
        /// <param name="tagName"></param>
        /// <returns></returns>
        public ActionResult DeleteTag(string tagName, int? pageIndex)
        {
            tagName = WebUtils.UrlDecode(tagName);
            ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).DeleteApplicationTag(tagName);
            return RedirectToAction("Control_ListTags", new RouteValueDictionary { { "pageIndex", pageIndex } });
        }



        public ActionResult DeleteTags(int? pageIndex)
        {
            string selectedTagsString = Request.Form["LinkGroup"];

            if (!string.IsNullOrEmpty(selectedTagsString))
            {
                string[] selectedTagsStringArray = selectedTagsString.Split(',');
                foreach (string tagName in selectedTagsStringArray)
                {
                    ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().Forum()).DeleteApplicationTag(tagName);
                }

                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "批量删除站点标签操作已成功,需要清除缓存，前台页面才会不显示";
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选择要删除的站点标签";
            }

            return RedirectToAction("Control_ListTags", new RouteValueDictionary { { "pageIndex", pageIndex } });
        }


        #endregion

        #endregion

        #region Help methods

        ///// <summary>
        ///// 从QueryString获取评论查询条件
        ///// </summary>
        //private void ConvertCommentFromQueryString(ForumPostQuery query)
        //{
        //    if (!string.IsNullOrEmpty(Request.QueryString["userID"]))
        //    {
        //        query.UserID = Request.QueryString.GetInt("userID", -1);
        //    }
        //}

        ///// <summary>
        ///// 从QueryString获取网摘查询条件
        ///// </summary>
        //private void ConvertForumFromQueryString(ForumThreadQuery query)
        //{
        //    if (!string.IsNullOrEmpty(Request.QueryString["userID"]))
        //    {
        //        query.SponsorUserID = Request.QueryString.GetInt("userID", -1);
        //    }
        //}

        #endregion

        /// <summary>
        /// 获取论坛统计数据
        /// </summary>       
        public JsonResult GetManageableCounts()
        {
            Dictionary<ForumManageableCounts, int> dictonary = ForumThreads.GetManageableCounts();
            Application forumApplication = Applications.GetApplication(ApplicationIDs.Instance().Forum());
            List<ManageableCount> list = new List<ManageableCount>();
            list.Add(new ManageableCount { Name = "新增主题", Value = forumApplication.CountInPast24Hours });
            if (dictonary.ContainsKey(ForumManageableCounts.PendingPostCount))
                list.Add(new ManageableCount { Name = "待审核回帖", Value = dictonary[ForumManageableCounts.PendingPostCount], Url = SPBUrlHelper.Action("ManageForumPosts", "ManageForum", new RouteValueDictionary { { "auditingStatusesForDisplay", AuditingStatusesForDisplay.Pending } }) });
            if (dictonary.ContainsKey(ForumManageableCounts.AgainPostCount))
                list.Add(new ManageableCount { Name = "需再审核回帖", Value = dictonary[ForumManageableCounts.AgainPostCount], Url = SPBUrlHelper.Action("ManageForumPosts", "ManageForum", new RouteValueDictionary { { "auditingStatusesForDisplay", AuditingStatusesForDisplay.Again } }) });
            if (dictonary.ContainsKey(ForumManageableCounts.PendingClubThreadCount))
                list.Add(new ManageableCount { Name = "待审核群组论坛主题", Value = dictonary[ForumManageableCounts.PendingClubThreadCount], Url = SPBUrlHelper.Action("ManageForumThreads", "ManageForum", new RouteValueDictionary { { "auditingStatusesForDisplay", AuditingStatusesForDisplay.Pending }, { "presentAreaID", PresentAreaIDs.Instance().ClubDomain() } }) });
            if (dictonary.ContainsKey(ForumManageableCounts.PendingThreadCount))
                list.Add(new ManageableCount { Name = "待审核独立论坛主题", Value = dictonary[ForumManageableCounts.PendingThreadCount], Url = SPBUrlHelper.Action("ManageForumThreads", "ManageForum", new RouteValueDictionary { { "auditingStatusesForDisplay", AuditingStatusesForDisplay.Pending }, { "presentAreaID", PresentAreaIDs.Instance().Channel() } }) });
            if (dictonary.ContainsKey(ForumManageableCounts.AgainClubThreadCount))
                list.Add(new ManageableCount { Name = "需再审核群组论坛主题", Value = dictonary[ForumManageableCounts.AgainClubThreadCount], Url = SPBUrlHelper.Action("ManageForumThreads", "ManageForum", new RouteValueDictionary { { "auditingStatusesForDisplay", AuditingStatusesForDisplay.Again }, { "presentAreaID", PresentAreaIDs.Instance().ClubDomain() } }) });
            if (dictonary.ContainsKey(ForumManageableCounts.AgainThreadCount))
                list.Add(new ManageableCount { Name = "需再审核独立论坛主题", Value = dictonary[ForumManageableCounts.AgainThreadCount], Url = SPBUrlHelper.Action("ManageForumThreads", "ManageForum", new RouteValueDictionary { { "auditingStatusesForDisplay", AuditingStatusesForDisplay.Again }, { "presentAreaID", PresentAreaIDs.Instance().Channel() } }) });
            list.Add(new ManageableCount { Name = "总主题", Value = forumApplication.TotalCount });
            return Json(list, JsonRequestBehavior.AllowGet);

        }


    }
}
