﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------


using System;
using System.Collections.Generic;
using System.Text;
using System.Web;
using SpaceBuilder.Forum;
using SpaceBuilder.Utils;
using SpaceBuilder.Web;

using System.IO;
using SpaceBuilder.Common;
using SpaceBuilder.Club;
using SpaceBuilder.Common.Handlers;


namespace SpaceBuilder.Forum.Handlers
{
    /// <summary>
    /// 处理论坛附件下载
    /// </summary>
    public class AttachmentHandler : DownloadFileHandlerBase
    {
        /// <summary>
        /// 向客户端输出附件
        /// </summary>        
        public override void ProcessRequest(HttpContext context)
        {
            //防盗链检测
            if (SPBConfig.Instance().EnableAntiLeech && !Globals.IsAllowableReferrer(context.Request))
            {
                context.Response.Redirect(Globals.GetFullUrlInResourceSite("~/Themes/Shared/Styles/Images/anti-Leech.gif"), true);
                return;
            }

            int attachmentID = context.Request.QueryString.Get<int>("AttachmentID", -1);
            if (attachmentID <= 0)
            {
                WebUtils.Return404(context);
                return;
            }

            Attachment attachment = ForumAttachmentManager.Instance().GetAttachment(attachmentID, true);
            if (attachment == null)
            {
                WebUtils.Return404(context);
                return;
            }

            if (!attachment.IsImage)
            {
                ForumSection section = ForumSections.GetSection(attachment.OwnerID, false);
                if (section == null)
                {
                    WebUtils.Return404(context);
                    return;
                }
                User currentUser = Globals.GetCurrentUser();

                ForumSectionGroup group = ForumSectionGroups.GetGroup(section.GroupID, true);

                if (group == null)
                    throw new SPBException(string.Format("ID为:{0}板块组不存在！", section.GroupID));

                if (group.PresentAreaID == PresentAreaIDs.Instance().ClubDomain())
                {
                    if (!ClubPermission.Validate(ClubID2ClubNameDictionary.GetClubDomainName(group.OwnerID), currentUser, PermissionItemKeys.Instance().DownClubForumAttachment()))
                    {
                        WebUtils.Return404(context);
                        return;
                    }
                }
                else
                {
                    if (!ForumPermission.Validate(section, currentUser, PermissionItemKeys.Instance().DownloadForumAttachment()))
                    {
                        WebUtils.Return404(context);
                        return;
                    }
                    if (attachment.Price > 0 && !ForumAttachmentManager.Instance().IsPayOut(attachment.AttachmentID, currentUser.UserID)
                            && attachment.UserID != currentUser.UserID && !ForumPermission.Validate(section, currentUser, PermissionItemKeys.Instance().EditForumPost()))
                    {
                        WebUtils.Return404(context);
                        return;
                    }
                }
                if (currentUser != null || !ForumThreads.IsMultiThreadedDownload(attachmentID))
                {
                    //下载计数 attachmentID=threadID              
                    CounterFactory.IncreaseCount<ForumAttachmentCounter>(IncreasableCountTypes.Download, attachmentID);
                    //记住下载记录，以便下次判断
                    ForumThreads.CreateForumDownLoadRecord(attachmentID);
                }
            }

            DateTime lastModified = attachment.DateCreated.ToUniversalTime();
            if (IsCacheOK(context, lastModified))
            {
                WebUtils.Return304(context);
                return;
            }
            else
            {
                SetResponsesDetails(context, attachment.ContentType.ToLower(), attachment.FriendlyFileName, lastModified);
                FileSystemFile fileSystemFile = attachment.File as FileSystemFile;
                if (fileSystemFile != null && (!fileSystemFile.FullLocalPath.StartsWith(@"\")))
                {
                    // Send files stored on UNC paths explicitly to avoid a bug with TransmitFile.
                    context.Response.TransmitFile(fileSystemFile.FullLocalPath);
                }
                else
                {
                    context.Response.AddHeader("Content-Length", attachment.ContentSize.ToString("0"));
                    try
                    {
                        using (Stream stream = attachment.OpenReadStream())
                        {
                            if (stream == null)
                            {
                                WebUtils.Return404(context);
                                return;
                            }

                            int bufferLength = attachment.ContentSize <= DownloadFileHandlerBase.BufferLength ? attachment.ContentSize : DownloadFileHandlerBase.BufferLength;
                            byte[] buffer = new byte[bufferLength];
                            int readedSize;
                            while ((readedSize = stream.Read(buffer, 0, bufferLength)) > 0)
                            {
                                if (!context.Response.IsClientConnected)
                                    break;

                                context.Response.OutputStream.Write(buffer, 0, readedSize);
                            }
                            context.Response.OutputStream.Flush();
                            context.Response.Flush();
                        }
                    }
                    catch (Exception ex)
                    {
                        context.Response.Redirect(SiteUrls.Instance().Error(ex.ToString()), true);
                        return;
                    }
                }

                context.Response.Cache.VaryByParams["AttachmentID"] = true;
                //context.Response.Close();
                context.Response.End();
            }

        }
    }
}
