﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;

using System.Web;
using SpaceBuilder.Common.Handlers;
using SpaceBuilder.Utils;
using SpaceBuilder.Rss;
using SpaceBuilder.Common;
using SpaceBuilder.Web;

namespace SpaceBuilder.Forum.Handlers
{
    /// <summary>
    /// 处理论坛rss
    /// </summary>
    public class ForumRssHandler : BaseRssHandler
    {
        private int sectionID = -1;
        public int SectionID
        {
            get { return sectionID; }
            set { sectionID = value; }
        }

        private ForumThreadSortBy sortBy = ForumThreadSortBy.LastRepliedDate;
        public ForumThreadSortBy SortBy
        {
            get { return sortBy; }
            set { sortBy = value; }
        }

        /// <summary>
        /// Appends http://Host:Port to all blog urls
        /// </summary>
        protected override string BaseUrl
        {
            get { return WebUtils.HostPath(Context.Request.Url); }
        }

        protected override int CacheTime { get { return ForumConfiguration.Instance().RssCacheWindowInSeconds; } }

        protected override string CacheKey
        {
            get { return string.Format("Rss_ForumThreads::SectionID:{0}-SortBy:{1}", SectionID, (int)sortBy); }
        }

        protected override void ProcessFeed()
        {
            SectionID = this.Context.Request.QueryString.GetInt("SectionID", -1);
            SortBy = (ForumThreadSortBy)this.Context.Request.QueryString.GetInt("SortBy", (int)ForumThreadSortBy.LastRepliedDate);

            base.ProcessFeed();
        }

        protected override CachedFeed BuildFeed()
        {
            ForumSection section = this.SectionID <= 0 ? null : ForumSections.GetSection(sectionID, false);
            if (section == null)
                return new CachedFeed(DateTime.Now, null, string.Empty);

            bool isEssential = this.Context.Request.QueryString.GetBool("IsEssential", false);

            IList<ForumThread> threads = ForumThreads.GetTopForumThreadsOfSection(section.SectionID, null, null, ForumThreadSortBy.PostDate, isEssential, ForumConfiguration.Instance().RssDefaultThreadsPerFeed);

            RssDocument rssDoc = new RssDocument();
            rssDoc.UseExternalNamespaces = true;

            if (!string.IsNullOrEmpty(StyleSheet))
                rssDoc.StyleSheet = FormatUrl(WebUtils.ResolveUrl(StyleSheet));

            ForumSectionGroup group = ForumSectionGroups.GetGroup(section.GroupID, false);

            RssChannel rssChannel = new RssChannel();
            if (sectionID > 0)
            {
                if (section == null)
                    throw new SPBException(ExceptionTypes.Instance().SectionNotFound(), "The forum could not be found");

                rssChannel.Title = WebUtils.HtmlDecode(section.SectionName);

                if (group.PresentAreaID == PresentAreaIDs.Instance().ClubDomain())
                    rssChannel.Link = FormatUrl(SiteUrls.Instance().ShowForumSectionInClub(Club.ClubID2ClubNameDictionary.GetClubDomainName(group.OwnerID), section.SectionID));
                else
                    rssChannel.Link = FormatUrl(SiteUrls.Instance().ShowForumSection(section.SectionID));

                rssChannel.Description = section.Description;
            }
            else if (group != null && group.PresentAreaID == PresentAreaIDs.Instance().ClubDomain())
            {
                Club.Club currentClub = SpaceBuilder.Club.Clubs.GetClub(group.OwnerID, false);
                if (currentClub == null)
                    throw new SPBException(ExceptionTypes.Instance().ClubDomainNotFond(), "The club could not be found");

                rssChannel.Title = WebUtils.HtmlDecode(currentClub.ClubName);
                rssChannel.Link = FormatUrl(SiteUrls.Instance().ClubDomainHome(currentClub.DomainName));
                rssChannel.Description = currentClub.Brief;
            }

            rssChannel.Generator = SPBConfig.Instance().SpaceBuilderVersionInfo;

            List<RssItem> rssItems = new List<RssItem>();

            foreach (ForumThread thread in threads)
            {
                RssItem item = new RssItem();
                item.Title = WebUtils.HtmlDecode(thread.Subject);

                group = ForumSectionGroups.GetGroup(thread.GroupID, false);
                if (group.PresentAreaID == PresentAreaIDs.Instance().ClubDomain())
                {
                    item.Link = FormatUrl(SiteUrls.Instance().ShowForumThreadInClub(Club.ClubID2ClubNameDictionary.GetClubDomainName(group.OwnerID), thread.ThreadID));
                    item.Comments = item.Link;
                }
                else
                {
                    item.Link = FormatUrl(SiteUrls.Instance().ShowForumThread(thread.ThreadID));
                    item.Comments = item.Link;
                }

                item.Author = thread.Author;
                item.PubDate = thread.PostDate;

                RssGuid guid = new RssGuid();
                guid.IsPermaLink = false;
                guid.Text = SiteSettingsManager.GetSiteSettings().SiteKey + ":Forum:" + thread.ThreadID;
                item.Guid = guid;

                item.Description = thread.Body;

                //启用rss阅读计数
                item.Description = string.Format("{0}<img src=\"{1}\" width=\"1\" height=\"1\">", item.Description, FormatUrl(SiteUrls.Instance().AggView(typeof(ForumThreadCounter), thread.ThreadID)));

                item.ReplyCount = thread.ReplyCount;

                rssItems.Add(item);
            }

            rssDoc.Channel = rssChannel;
            rssChannel.Items = rssItems;

            DateTime dt = threads.Count > 0 ? threads[0].PostDate : DateTime.Now;

            return new CachedFeed(dt, null, rssDoc.ToXml());
        }


    }
}
