﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Web;
using SpaceBuilder.Common;


using System.Web.Mvc;
using System.Web.Routing;
using SpaceBuilder.Utils;
using SpaceBuilder.Club;

namespace SpaceBuilder.Forum
{
    public static class SPBHtmlHelperExtensions
    {
        /// <summary>
        /// 显示版块组及版块的的DropDownList
        /// </summary>
        /// <param name="clubID">ForumSection限制的ClubID</param>
        /// <param name="exceptSectionID">需要排除的SectionID</param>
        /// <param name="enableSelectForumGroup">是否允许选择版块组</param>
        /// <param name="htmlAttributes">html属性，例如：new {Class="editor",width="90%"}</param>
        public static string ForumSectionsDropDownList(this SPBHtmlHelper spbHtmlHelper, string name, int? clubID, int? exceptSectionID, string selectedValue, bool enableSelectForumGroup, string firstOptionText, RouteValueDictionary htmlAttributes)
        {

            return ForumSectionsDropDownList(spbHtmlHelper, name, clubID, exceptSectionID, false, selectedValue, enableSelectForumGroup, firstOptionText, htmlAttributes);
        }

        /// <summary>
        /// 仅显示当前用户可以管理的版块组及版块的的DropDownList
        /// </summary>
        /// <param name="clubID">ForumSection限制的ClubID</param>
        /// <param name="exceptSectionID">需要排除的SectionID</param>
        /// <param name="enableSelectForumGroup">是否允许选择版块组</param>
        /// <param name="userName">当前用户的用户名</param>
        /// <param name="htmlAttributes">html属性，例如：new {Class="editor",width="90%"}</param>
        public static string ModerateForumSectionsDropDownList(this SPBHtmlHelper spbHtmlHelper, string name, int? clubID, int? exceptSectionID, string selectedValue, bool enableSelectForumGroup, string firstOptionText, RouteValueDictionary htmlAttributes)
        {
            User currentUser = Globals.GetCurrentUser();
            if (currentUser == null)
                return string.Empty;
            return ForumSectionsDropDownList(spbHtmlHelper, name, clubID, exceptSectionID, true, selectedValue, enableSelectForumGroup, firstOptionText, htmlAttributes);
        }

        /// <summary>
        /// 显示版块组及版块的的DropDownList
        /// </summary>
        /// <param name="clubID">ForumSection限制的ClubID</param>
        /// <param name="exceptSectionID">需要排除的SectionID</param>
        /// <param name="enableSelectForumGroup">是否允许选择版块组</param>
        /// <param name="htmlAttributes">html属性，例如：new {Class="editor",width="90%"}</param>
        /// todo:需要改
        private static string ForumSectionsDropDownList(this SPBHtmlHelper spbHtmlHelper, string name, int? clubID, int? exceptSectionID, bool onlyCanModerate, string selectedValue, bool enableSelectForumGroup, string firstOptionText, RouteValueDictionary htmlAttributes)
        {
            TagBuilder selectBuilder = new TagBuilder("select");
            selectBuilder.MergeAttribute("name", name);
            selectBuilder.MergeAttribute("id", name);
            selectBuilder.MergeAttributes(htmlAttributes);

            if (!string.IsNullOrEmpty(firstOptionText))
            {
                TagBuilder selectOptionBuilder = new TagBuilder("option");
                selectOptionBuilder.MergeAttribute("value", string.Empty);
                selectOptionBuilder.SetInnerText(firstOptionText);
                selectBuilder.InnerHtml += selectOptionBuilder.ToString();
            }

            User currentUser = Globals.GetCurrentUser();

            if (clubID.HasValue)
            {
                Club.Club club = SpaceBuilder.Club.Clubs.GetClub(clubID.Value, false);
                if (club != null)
                {
                    List<ForumSection> forumSectionsOfClub = ForumSections.GetIndentedSectionsOfGroup(OwnerID2GroupIDDictionary.Instance(PresentAreaIDs.Instance().ClubDomain()).GetGroupID(clubID.Value), true);
                    foreach (var forumSection in forumSectionsOfClub)
                    {
                        if (forumSection.SectionID == exceptSectionID)
                            continue;
                        if (onlyCanModerate && !ClubPermission.Validate(club, currentUser, PermissionItemKeys.Instance().ApproveClubForumPost()))
                            continue;

                        TagBuilder selectOptionBuilder = new TagBuilder("option");
                        selectOptionBuilder.MergeAttribute("value", forumSection.SectionID.ToString());
                        string optionInnerHtml = string.Empty;

                        for (int i = 0; i <= forumSection.Depth; i++)
                        {
                            optionInnerHtml += "&nbsp;&nbsp;&nbsp;&nbsp;";
                        }

                        optionInnerHtml += forumSection.SectionName;
                        selectOptionBuilder.InnerHtml = optionInnerHtml;

                        if (!string.IsNullOrEmpty(selectedValue) && selectedValue == forumSection.SectionID.ToString())
                            selectOptionBuilder.MergeAttribute("selected", "selected");

                        selectBuilder.InnerHtml += selectOptionBuilder.ToString();
                    }
                }
            }
            else
            {
                IList<ForumSectionGroup> forumGroups = ForumSectionGroups.GetSiteGroups(false);
                foreach (var group in forumGroups)
                {
                    if (group == null)
                        continue;

                    TagBuilder groupSelectOptionBuilder = new TagBuilder("option");
                    groupSelectOptionBuilder.MergeAttribute("value", "g-" + group.GroupID);

                    if (!string.IsNullOrEmpty(selectedValue) && (selectedValue == "g-" + group.GroupID))
                        groupSelectOptionBuilder.MergeAttribute("selected", "selected");


                    groupSelectOptionBuilder.SetInnerText(group.GroupName);
                    if (!enableSelectForumGroup)
                    {
                        groupSelectOptionBuilder.MergeAttribute("disabled", "disabled");
                    }

                    //selectBuilder.InnerHtml += groupSelectOptionBuilder.ToString();

                    List<ForumSection> forumSectionsOfGroup = ForumSections.GetRootSectionsOfGroup(group.GroupID, true);
                    StringBuilder sectionOptionsOfGroup = new StringBuilder();
                    foreach (var forumSection in forumSectionsOfGroup)
                    {
                        if (forumSection.SectionID == exceptSectionID)
                            continue;

                        if (onlyCanModerate && !ForumPermission.Validate(forumSection, currentUser, PermissionItemKeys.Instance().ApproveForumPost()))
                            continue;

                        TagBuilder selectOptionBuilder = new TagBuilder("option");
                        selectOptionBuilder.MergeAttribute("value", forumSection.SectionID.ToString());
                        string optionInnerHtml = string.Empty;

                        for (int i = 0; i <= forumSection.Depth; i++)
                        {
                            optionInnerHtml += "&nbsp;&nbsp;&nbsp;&nbsp;";
                        }

                        optionInnerHtml += forumSection.SectionName;
                        selectOptionBuilder.InnerHtml = optionInnerHtml;

                        if (!string.IsNullOrEmpty(selectedValue) && selectedValue == forumSection.SectionID.ToString())
                            selectOptionBuilder.MergeAttribute("selected", "selected");

                        //selectBuilder.InnerHtml += selectOptionBuilder.ToString();
                        sectionOptionsOfGroup.Append(selectOptionBuilder.ToString());
                    }
                    if (sectionOptionsOfGroup.Length > 0)
                    {
                        selectBuilder.InnerHtml += groupSelectOptionBuilder.ToString();
                        selectBuilder.InnerHtml += sectionOptionsOfGroup.ToString();
                    }
                }
            }
            return selectBuilder.ToString();
        }

        /// <summary>
        /// 生成论坛版块快速跳转DropDownList
        /// </summary>
        /// <param name="clubID">ForumSection限制的ClubID</param>
        /// <param name="htmlAttributes">html属性，例如：new {Class="editor",width="90%"}</param>
        ///todo:需要改
        public static string ForumSectionJumperDropDownList(this SPBHtmlHelper spbHtmlHelper, int? groupID, string firstOptionText, RouteValueDictionary htmlAttributes)
        {
            TagBuilder selectBuilder = new TagBuilder("select");
            if (htmlAttributes != null)
                selectBuilder.MergeAttributes(htmlAttributes);


            if (!string.IsNullOrEmpty(firstOptionText))
            {
                TagBuilder selectOptionBuilder = new TagBuilder("option");
                selectOptionBuilder.MergeAttribute("value", string.Empty);
                selectOptionBuilder.SetInnerText(firstOptionText);
                selectBuilder.InnerHtml += selectOptionBuilder.ToString();
            }

            if (groupID.HasValue)
            {
                Club.Club club = SpaceBuilder.Club.Clubs.GetClub(groupID.Value, false);
                if (club != null)
                {
                    List<ForumSection> forumSectionsOfGroup = ForumSections.GetRootSectionsOfGroup(groupID.Value, true);
                    foreach (var forumSection in forumSectionsOfGroup)
                    {
                        TagBuilder selectOptionBuilder = new TagBuilder("option");
                        selectOptionBuilder.MergeAttribute("value", SiteUrls.Instance().ShowForumSectionInClub(club.DomainName, forumSection.SectionID));
                        string optionInnerText = string.Empty;
                        if (forumSection.Depth > 0)
                        {
                            for (int i = 0; i < forumSection.Depth * 2 + 1; i++)
                            {
                                optionInnerText += "　";
                            }
                            optionInnerText += "├";
                        }
                        optionInnerText += forumSection.SectionName;

                        selectOptionBuilder.SetInnerText(optionInnerText);
                        selectBuilder.InnerHtml += selectOptionBuilder.ToString();
                    }
                }
            }
            else
            {
                IList<ForumSectionGroup> forumGroups = ForumSectionGroups.GetSiteGroups(false);
                foreach (var group in forumGroups)
                {
                    if (group == null)
                        continue;

                    TagBuilder selectOptionBuilder = new TagBuilder("option");
                    selectOptionBuilder.MergeAttribute("value", SiteUrls.Instance().ShowForumGroup(group.GroupID));
                    selectOptionBuilder.SetInnerText(group.GroupName);
                    selectBuilder.InnerHtml += selectOptionBuilder.ToString();

                    List<ForumSection> forumSectionsOfClub = ForumSections.GetRootSectionsOfGroup(group.GroupID, true);
                    foreach (var forumSection in forumSectionsOfClub)
                    {
                        TagBuilder forumSectionOption = new TagBuilder("option");
                        forumSectionOption.MergeAttribute("value", SiteUrls.Instance().ShowForumSection(forumSection.SectionID));
                        string optionInnerText = string.Empty;

                        for (int i = 0; i < forumSection.Depth * 2 + 1; i++)
                        {
                            optionInnerText += "　";
                        }
                        optionInnerText += "├";
                        optionInnerText += forumSection.SectionName;

                        forumSectionOption.SetInnerText(optionInnerText);
                        selectBuilder.InnerHtml += forumSectionOption.ToString();
                    }
                }
            }
            return selectBuilder.ToString();
        }


        /// <summary>
        /// 在主题列表中显示主题分页
        /// </summary>
        /// <param name="htmlAttributes">html属性，例如：new {Class="editor",width="90%"}</param>
        public static string ShowForumThreadPager(this SPBHtmlHelper spbHtmlHelper, ForumThread forumThread, RouteValueDictionary htmlAttributes)
        {
            int pageSize = ForumConfiguration.Instance().PostsPerPage;

            if (forumThread.ReplyCount <= pageSize)
                return string.Empty;
            else
            {
                int totalPages = forumThread.ReplyCount / pageSize;
                if ((forumThread.ReplyCount % pageSize) > 0)
                    totalPages++;

                TagBuilder pagerContainer = new TagBuilder("span");
                if (htmlAttributes != null)
                    pagerContainer.MergeAttributes(htmlAttributes);

                if (totalPages <= 8)
                {
                    pagerContainer.InnerHtml += "&nbsp;...&nbsp;";
                    for (int i = 2; i <= totalPages; i++)
                    {
                        TagBuilder anchor = new TagBuilder("a");
                        anchor.MergeAttribute("href", SiteUrls.Instance().ShowForumThreadPagedInPresentArea(forumThread, i));
                        anchor.SetInnerText(i.ToString());
                        pagerContainer.InnerHtml += anchor.ToString();
                    }
                }
                else
                {
                    for (int i = 2; i <= 4; i++)
                    {
                        TagBuilder anchor = new TagBuilder("a");
                        anchor.MergeAttribute("href", SiteUrls.Instance().ShowForumThreadPaged(forumThread.ThreadID, i));
                        anchor.SetInnerText(i.ToString());
                        pagerContainer.InnerHtml += anchor.ToString();

                    }
                    pagerContainer.InnerHtml += "&nbsp;...&nbsp;";
                    for (int i = totalPages - 3; i <= totalPages; i++)
                    {
                        TagBuilder anchor = new TagBuilder("a");
                        anchor.MergeAttribute("href", SiteUrls.Instance().ShowForumThreadPaged(forumThread.ThreadID, i));
                        anchor.SetInnerText(i.ToString());
                        pagerContainer.InnerHtml += anchor.ToString();

                    }
                }
                return pagerContainer.ToString();
            }

        }

        /// <summary>
        /// 根据当前用户的权限控制是否显示IP地址，以及显示的格式
        /// </summary>
        /// <param name="userHostAddress">IP地址</param>
        /// <returns>格式化后的IP地址</returns>
        public static string ShowPostIPAddress(this SPBHtmlHelper spbHtmlHelper, string userHostAddress)
        {
            User currentUser = Globals.GetCurrentUser();

            if (userHostAddress == "000.000.000.000")
                return ResourceManager.GetString("Description_NotLogged", Globals.GetCurrentUserLanguage());

            if (SiteSettingsManager.GetSiteSettings().DisplayPostIP)
                return userHostAddress;
            else if ((SiteSettingsManager.GetSiteSettings().DisplayPostIPAdminsModeratorsOnly) && currentUser != null && currentUser.IsContentAdministrator)
                return userHostAddress;
            else
                return ResourceManager.GetString("Description_Logged", Globals.GetCurrentUserLanguage());
        }

        /// <summary>
        /// 获取论坛附件的缩略图
        /// </summary>
        public static string ShowForumAttachmentThumbnail(this SPBHtmlHelper spbHtmlHelper, Attachment attachment, ContentTypeThumbnailSize thumbnailSize)
        {
            return spbHtmlHelper.ShowAttachmentThumbnail(attachment, thumbnailSize, SiteUrls.Instance().DownloadForumAttachment);
        }

    }
}
