﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using SpaceBuilder.Forum;
using SpaceBuilder.Common;
using Lucene.Net.Documents;
using Lucene.Net.Search;
using Lucene.Net.Index;
using Lucene.Net.QueryParsers;
using Lucene.Net.Analysis;
using SpaceBuilder.Utils;
using Lucene.Net.Analysis.Standard;
using SpaceBuilder.Club;

namespace SpaceBuilder.LuceneSearch
{
    /// <summary>
    /// 论坛全文检索
    /// </summary>
    public class ForumSearchManager : SearchManagerBase<ForumThread>
    {
        //博客索引文件夹
        private static readonly string ForumIndexFileDirectory = "Forum";
        private static volatile ForumSearchManager _self = null;
        private static readonly object lockObject = new object();

        private ForumSearchManager(string indexFileDirectory) : base(indexFileDirectory) { }

        public static ForumSearchManager Instance()
        {
            if (_self == null)
            {
                lock (lockObject)
                {
                    if (_self == null)
                    {
                        _self = new ForumSearchManager(ForumIndexFileDirectory);
                    }
                }
            }
            return _self;
        }

        public SearchResultDataSet<ForumThread> Search(ForumPostFullTextQuery query)
        {
            //索引文件不存在时，返回null
            if (!IsIndexFilesExists)
                return new SearchResultDataSet<ForumThread>();

            BooleanQuery currentQuery = new BooleanQuery();
            BooleanQuery queryForFilter = new BooleanQuery();

            if (query.SearchScopeGroupID > 0)
            {
                Term clubIDTerm = new Term(ForumIndexFields.GroupID, query.SearchScopeGroupID.ToString());
                Query clubIDQuery = new TermQuery(clubIDTerm);
                queryForFilter.Add(clubIDQuery, BooleanClause.Occur.MUST);
            }
            else
            {
                QueryParser groupIDQueryParser = new QueryParser(CurrentLuceneVersion, ForumIndexFields.GroupID, new WhitespaceAnalyzer());
                IList<ForumSectionGroup> groups = ForumSectionGroups.GetSiteGroups(false);
                StringBuilder groupIDs = new StringBuilder();
                if (groups != null)
                {
                    foreach (ForumSectionGroup group in groups)
                    {
                        if (group.PresentAreaID == PresentAreaIDs.Instance().Channel())
                            groupIDs.Append(group.GroupID + " ");
                    }
                }
                Query groupIDQuery = groupIDQueryParser.Parse(groupIDs.ToString().TrimEnd(' '));
                queryForFilter.Add(groupIDQuery, BooleanClause.Occur.MUST);
            }

            if (query.UserID > 0)
            {
                Term userIDTerm = new Term(ForumIndexFields.UserID, query.UserID.ToString());
                Query userIDQuery = new TermQuery(userIDTerm);
                queryForFilter.Add(userIDQuery, BooleanClause.Occur.MUST);
            }

            if (query.SectionID > 0)
            {
                if (query.IncludeSectionDescendant)
                {
                    QueryParser sectionIDQueryParser = new QueryParser(ForumIndexFields.SectionID, new WhitespaceAnalyzer());

                    IList<ForumSection> childSections = ForumSections.GetAllChilds(query.SectionID, false);
                    StringBuilder sectionIDs = new StringBuilder();
                    sectionIDs.Append(query.SectionID.ToString());
                    if (childSections != null)
                    {
                        foreach (ForumSection forumSection in childSections)
                        {
                            sectionIDs.Append(" " + forumSection.SectionID);
                        }
                    }

                    Query siteCategoryIDQuery = sectionIDQueryParser.Parse(sectionIDs.ToString());
                    currentQuery.Add(siteCategoryIDQuery, BooleanClause.Occur.MUST);
                }
                else
                {
                    Term siteCategoryIDTerm = new Term(ForumIndexFields.SectionID, query.SectionID.ToString());
                    Query siteCategoryIDQuery = new TermQuery(siteCategoryIDTerm);
                    currentQuery.Add(siteCategoryIDQuery, BooleanClause.Occur.MUST);
                }
            }

            if (!string.IsNullOrEmpty(query.Keyword))
            {
                Query postKeywordQuery = null;
                query.Keyword = StringUtilsForLucene.LuceneKeywordsScrubber(query.Keyword.ToLower());
                string keywordSegments = SegmentForQueryParser(query.Keyword);

                if (!string.IsNullOrEmpty(query.Keyword))
                {
                    string[] searchFieldsForKeyword = new string[4];
                    searchFieldsForKeyword[0] = ForumIndexFields.Subject;
                    searchFieldsForKeyword[1] = ForumIndexFields.Body;
                    searchFieldsForKeyword[2] = ForumIndexFields.Tags;
                    searchFieldsForKeyword[3] = ForumIndexFields.Author;

                    MultiFieldQueryParser keywordParser = new MultiFieldQueryParser(CurrentLuceneVersion, searchFieldsForKeyword, GetChineseAnalyzerOfUnTokenized());
                    keywordParser.SetLowercaseExpandedTerms(true);
                    keywordParser.SetDefaultOperator(QueryParser.OR_OPERATOR);
                    postKeywordQuery = keywordParser.Parse(keywordSegments);
                    currentQuery.Add(postKeywordQuery, BooleanClause.Occur.MUST);
                }
            }

            if (query.DateScopes > 0)
            {
                Term lowerUpdateDateTerm = new Term(ForumIndexFields.LastUpdatedDate, DateTools.DateToString(DateTime.Now.AddDays(0 - query.DateScopes), DateTools.Resolution.DAY));
                Term upperUpdateDateTerm = new Term(ForumIndexFields.LastUpdatedDate, DateTools.DateToString(DateTime.Now, DateTools.Resolution.DAY));
                Query updatedDateQuery = new RangeQuery(lowerUpdateDateTerm, upperUpdateDateTerm, true);
                queryForFilter.Add(updatedDateQuery, BooleanClause.Occur.MUST);
            }

            if (!string.IsNullOrEmpty(query.TagName))
            {
                string[] tagSegments = SegmentForPhraseQuery(query.TagName);
                if (tagSegments != null && tagSegments.Length > 0)
                {
                    PhraseQuery tagQuery = new PhraseQuery();
                    foreach (var tagSegment in tagSegments)
                        tagQuery.Add(new Term(ForumIndexFields.Tags, tagSegment));

                    tagQuery.SetSlop(PhraseQuerySlop);
                    tagQuery.SetBoost((float)Math.Pow(3, 5));
                    currentQuery.Add(tagQuery, BooleanClause.Occur.MUST);
                }
            }

            SortField[] sortFields;
            switch (query.SortBy)
            {
                case FullTextQueryForumPostsSortBy.LastUpdatedDate:
                    sortFields = new SortField[] { new SortField(ForumIndexFields.LastUpdatedDate, SortField.STRING, true) };
                    break;
                default:
                    sortFields = new SortField[] { SortField.FIELD_SCORE };
                    break;
            }

            Filter filter = null;
            if (queryForFilter.Clauses().Count > 0)
                filter = new QueryWrapperFilter(queryForFilter);

            SearchResultDataSet<ForumThread> pds = Search(currentQuery, filter, sortFields, query.PageIndex, query.PageSize);
            foreach (var item in pds.Records)
            {
                item.Subject = HighlighterForKeyWord(item.Subject, query.Keyword);
                item.Body = HighlighterForKeyWord(item.Body, query.Keyword);
            }
            return pds;
        }

        /// <summary>
        /// 初始化索引
        /// </summary>
        public override void InitializeIndex(string indexPath)
        {
            if (!System.IO.Directory.Exists(indexPath))
            {
                try
                {
                    System.IO.Directory.CreateDirectory(indexPath);
                }
                catch
                {
                    throw new ApplicationException(string.Format("create Directory '{0}' failed", PhysicalIndexDirectory));
                }
            }

            int indexPageSize = 2000;
            bool createIndexFile = true;
            PagingDataSet<ForumThread> pds = ForumThreads.GetThreadsForManage(string.Empty, null, null, null, null, AuditingStatusesForDisplay.Success, indexPageSize, 1);
            double tIndex = Convert.ToDouble(pds.TotalRecords) / Convert.ToDouble(indexPageSize);
            int indexPageIndex = (int)Math.Ceiling(tIndex);

            if (pds.TotalRecords > 0)
            {
                //分多次进行索引
                for (int i = 1; i <= indexPageIndex; i++)
                {
                    if (i != 1)
                        pds = ForumThreads.GetThreadsForManage(string.Empty, null, null, null, null, AuditingStatusesForDisplay.Success, indexPageSize, i);
                    Insert(pds.Records, indexPath, createIndexFile);
                    if (createIndexFile)
                        createIndexFile = false;
                }
            }

        }

        /// <summary>
        /// Document转化成PostSearchItem
        /// </summary>
        protected override ForumThread ConvertDocumentToObj(Document doc)
        {
            ForumThread post = new ForumThread();

            int userID;
            int.TryParse(doc.Get(ForumIndexFields.UserID), out userID);
            post.UserID = userID;

            int clubID;
            int.TryParse(doc.Get(ForumIndexFields.GroupID), out clubID);
            post.GroupID = clubID;

            int sectionID;
            int.TryParse(doc.Get(ForumIndexFields.SectionID), out sectionID);
            post.SectionID = sectionID;

            int threadID;
            int.TryParse(doc.Get(ForumIndexFields.ThreadID), out threadID);
            post.ThreadID = threadID;

            int postID;
            int.TryParse(doc.Get(ForumIndexFields.PostID), out postID);
            post.PostID = postID;

            post.Author = doc.Get(ForumIndexFields.Author);
            post.Subject = doc.Get(ForumIndexFields.Subject);
            post.Body = doc.Get(ForumIndexFields.Body);

            try
            {
                post.PostDate = DateTools.StringToDate(doc.Get(ForumIndexFields.PostDate));
            }
            catch { }

            try
            {
                post.LastRepliedDate = DateTools.StringToDate(doc.Get(ForumIndexFields.LastUpdatedDate));
            }
            catch { }

            return post;
        }

        /// <summary>
        /// PostSearchItem 转化成Document进行索引的存储
        /// </summary>
        protected override Document ConvertObjToDocument(ForumThread thread)
        {
            if (thread == null)
                return null;

            //待审核及未通过审核不允许加入索引
            if (thread.AuditingStatus == AuditingStatuses.Pending || thread.AuditingStatus == AuditingStatuses.Fail)
                return null;

            Document doc = new Document();
            Field field;

            field = new Field(ForumIndexFields.UserID, thread.UserID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(ForumIndexFields.GroupID, thread.GroupID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(ForumIndexFields.SectionID, thread.SectionID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(ForumIndexFields.ThreadID, thread.ThreadID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(ForumIndexFields.PostID, thread.PostID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(ForumIndexFields.Author, thread.Author.ToLower(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(ForumIndexFields.Subject, thread.Subject.ToLower(), Field.Store.YES, Field.Index.ANALYZED, Field.TermVector.YES);
            field.SetBoost(2.0F);
            doc.Add(field);

            field = new Field(ForumIndexFields.Body, HtmlUtils.StripAllTags(thread.Body).ToLower(), Field.Store.YES, Field.Index.ANALYZED, Field.TermVector.YES);
            doc.Add(field);

            if (thread.Tags != null)
            {
                foreach (var tag in thread.Tags)
                {
                    field = new Field(ForumIndexFields.Tags, tag.ToLower(), Field.Store.YES, Field.Index.ANALYZED);
                    field.SetBoost(2.0F);
                    doc.Add(field);
                }
            }

            field = new Field(ForumIndexFields.PostDate, DateTools.DateToString(thread.PostDate, DateTools.Resolution.DAY), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(ForumIndexFields.LastUpdatedDate, DateTools.DateToString(thread.LastRepliedDate, DateTools.Resolution.DAY), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            return doc;
        }

    }
}
