﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using SpaceBuilder.Utils;
using SpaceBuilder.Common;
using SpaceBuilder.Web;
using System.Web.Routing;
using System.Text.RegularExpressions;
using SpaceBuilder.LuceneSearch;
using Lucene.Net.Search;
using System.Web.Security;
using System.Web.UI.WebControls;
using System.Threading;
using System.Web.UI;

namespace SpaceBuilder.MicroBlog.Controllers
{
    public partial class MicroBlogController : MicroBlogDomainControllerBase
    {
        private int applicationID = ApplicationIDs.Instance().MicroBlog();
        private delegate void SyncMicroBlogDelegate(string body, int userID, int threadID, bool? hasPhoto);


        #region MicroBlog

        /// <summary>
        /// 登陆
        /// </summary>
        /// <returns></returns>
        public ActionResult MicroBlogHome()
        {
            ViewData["Topices"] = MicroBlogTopics.GetTopTopicSortBy(TopicSortBy.ItemCount, 8);

            if (CurrentUser == null)
                return View("Pages/Login.aspx");
            else
                return RedirectToAction("MyMicroBlogHome", new RouteValueDictionary { { "userDomainName", CurrentUser.UserName } });
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_TopMicroBlogThreads()
        {
            IList<MicroBlogThread> threads = MicroBlogThreads.GetTopThreadsSortBy(MicroBlogThreadSortBy.DateCreated, 25, 3);
            if (threads != null)
                threads = threads.Where(n => n.OriginalThreadID <= 0).ToList();
            return PartialView("Controls/TopMicroBlogThreads.ascx", threads);
        }

        /// <summary>
        /// 微博发布框
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public ActionResult Control_MicroBlogInput(string userDomainName, string body, bool? isSimple)
        {
            ViewData["Body"] = body;
            ViewData["isSimple"] = isSimple ?? false;
            if (AccountBindings.GetAccountBinding(CurrentDomainUser.UserID, AccountTypes.Instance().SinaWeibo()) != null)
                ViewData["isbind"] = true;
            return PartialView("Controls/MicroBlogInput.ascx");
        }

        /// <summary>
        /// 分享视频音乐
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="contentID">要将将内容插入容器的ID</param>
        /// <param name="type">媒体类型</param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_ShareMedia(string userDomainName, string contentID, int type)
        {
            return PartialView("Controls/ShareMedia.ascx");
        }

        /// <summary>
        /// 发布微博
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreateMicroBlogThread(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string body = Request.Form["microBlogBody"];
            try
            {
                Globals.CheckPostDate();
            }
            catch (Exception e)
            {
                return Json(new { error = true, msg = "你发微博的速度太快哦！" });
            }

            if (string.IsNullOrEmpty(body))
                return Json(new { error = true, msg = "微博内容不能为空！" });

            MicroBlogThread thread = new MicroBlogThread();
            thread.Body = body;

            int attachmetID = Request.Form.GetInt("attachmentID", -1);
            if (attachmetID > 0)
            {
                Attachment mAttachment = MicroBlogAttachmentManager.Instance().GetAttachment(attachmetID, true);
                if (mAttachment != null)
                    thread.HasPhoto = true;
            }
            thread.OwnerUserID = currentDomainUser.UserID;
            thread.Author = CurrentUser.DisplayName;
            thread.PostMode = PostModes.Web;
            thread.PostSource = WebUtils.IPAddress;
            thread.UserID = CurrentUser.UserID;
            try
            {
                MicroBlogThreads.CreateThread(thread);
            }
            catch { }

            if (Request.Form.GetBool("syncsina", false) && thread.ThreadID > 0)
            {
                try
                {
                    SyncMicroBlogDelegate sd = new SyncMicroBlogDelegate(SyncSina);
                    sd.BeginInvoke(thread.Body, currentDomainUser.UserID, thread.ThreadID, thread.HasPhoto, null, null);
                }
                catch { }
            }

            return PartialView("Controls/MicroBlogItems.ascx", new List<MicroBlogThread> { thread });

        }

        /// <summary>
        /// 删除微博主题
        /// </summary>
        public ActionResult DeleteMicroBlogThread(int threadID)
        {
            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, true);
            if (thread == null)
                return Redirect(SiteUrls.Instance().Error404("当前微博已被删除！"));
            if (CurrentUser != null && (CurrentUser.UserID == thread.UserID || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
            {
                if (thread != null)
                {
                    if (CurrentUser.IsContentAdministrator || CurrentUser.UserID == thread.OwnerUserID)
                    {
                        MicroBlogThreads.DeleteThread(thread);
                    }
                }
                ContentResult result = new ContentResult();
                result.Content = "ok";
                return result;
            }
            return new EmptyResult();
        }

        /// <summary>
        /// 更新单个微博审核状态
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult updateMicroBlogAuditingStatuses(int threadID, string userDomainName)
        {
            List<int> threadIds = new List<int>();
            threadIds.Add(threadID);
            MicroBlogThreads.UpdateThreadAuditingStatuses(threadIds, AuditingStatuses.Success);
            return RedirectToAction("ShowMicroBlogThread", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", threadID } });
        }

        /// <summary>
        /// 转发微博
        /// </summary>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public ActionResult Control_ForwardMicroBlog(int threadID)
        {

            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, false);
            if (thread == null)
                throw new SPBException(ExceptionTypes.Instance().UnknownError());

            return PartialView("Controls/ForwardMicroBlog.ascx", thread);
        }

        /// <summary>
        /// 转发微博
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult ForwardMicroBlog()
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Error404("你没有该项操作权限！"));
            MicroBlogThread thread = new MicroBlogThread();

            int threadID = Request.Form.GetInt("threadID", 0);
            MicroBlogThread forwardThread = MicroBlogThreads.GetThread(threadID, false);
            if (forwardThread == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "微博不存在！";
                return RedirectToAction("Control_ForwardMicroBlog", "MicroBlog", new RouteValueDictionary { { "threadID", threadID } });
            }

            if (MicroBlogThreads.IsForwarded(CurrentUser.UserID, forwardThread.ThreadID))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "同一微博只能转发一次！";
                return RedirectToAction("Control_ForwardMicroBlog", "MicroBlog", new RouteValueDictionary { { "threadID", threadID } });
            }

            thread.Body = Request.Form.GetString("microBlogBody", string.Empty);

            if (string.IsNullOrEmpty(thread.Body))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "内容不能为空";
                return RedirectToAction("Control_ForwardMicroBlog", "MicroBlog", new RouteValueDictionary { { "threadID", threadID } });
            }
            if (forwardThread.OriginalThreadID <= 0)
            {
                thread.OriginalThreadID = forwardThread.ThreadID;
            }
            else
            {
                thread.OriginalThreadID = forwardThread.OriginalThreadID;
            }

            thread.ForwardedThreadID = forwardThread.ThreadID;
            thread.OwnerUserID = CurrentUser.UserID;
            thread.Author = CurrentUser.DisplayName;

            MicroBlogThreads.CreateThread(thread);
            if (thread.ThreadID > 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "转发成功！";

            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "转发失败！";
            }

            MicroBlogComment comment = new MicroBlogComment();

            if (Request.Form.GetBool("Comment", false))
            {
                comment.Author = CurrentUser.DisplayName;
                comment.UserID = CurrentUser.UserID;
                comment.OwnerUserID = forwardThread.UserID;
                comment.ThreadID = forwardThread.ThreadID;
                comment.Body = thread.Body;
                MicroBlogComments.CreateComment(comment);
            }

            if (Request.Form.GetBool("OriginalComment", false) && forwardThread.OriginalThreadID > 0)
            {
                comment.Author = CurrentUser.DisplayName;
                comment.UserID = CurrentUser.UserID;
                comment.OwnerUserID = forwardThread.GetOriginalThread.UserID;
                comment.ThreadID = forwardThread.OriginalThreadID;
                comment.Body = thread.Body;
                MicroBlogComments.CreateComment(comment);
            }

            return RedirectToAction("Control_ForwardMicroBlog", "MicroBlog", new RouteValueDictionary { { "threadID", threadID } });
        }

        /// <summary>
        /// 微博详细显示页
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public ActionResult ShowMicroBlogThread(string userDomainName, int threadID)
        {
            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, HasManagePermission(userDomainName));
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));

            if (thread == null)
                return Redirect(SiteUrls.Instance().Error("微博不存在或已被删除！"));

            #region 处理审核

            int auditingStatusValue = (int)Applications.GetApplication(ApplicationIDs.Instance().MicroBlog()).AuditingStatusForPublicDisplay;
            if ((int)thread.AuditingStatus < auditingStatusValue)
            {
                if (thread.AuditingStatus == AuditingStatuses.Fail)
                    ViewData["errorMessage"] = GetResourceString("Message_MicrobloggingNotAuditing", applicationID);
                else
                    ViewData["errorMessage"] = GetResourceString("Message_MicrobloggingAuditing", applicationID);
                ViewData["errorAuditingStatus"] = true;
            }

            #endregion

            return View("Pages/ShowMicroBlogThread.aspx", "Masters/MicroBlog.master", thread);
        }

        /// <summary>
        /// 我的微博首页
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public ActionResult MyMicroBlogHome(string userDomainName, int? gid)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            if (CurrentUser == null || CurrentUser.UserID != domainUser.UserID)
            {

                SetPageTitle(domainUser.DisplayName + "的微博");
                ViewData["MicroBlogDomainSubMenu"] = (int)MicroBlogDomainSubMenu.MyMicroBlogs;
                return View("Pages/ListMyMicroBlogThreads.aspx", "Masters/MicroBlog.master");
            }
            SetPageTitle(GetResourceString("Navigation_MyMicroBlogHome", applicationID));
            ViewData["userGroupID"] = gid;
            ViewData["MicroBlogDomainSubMenu"] = (int)MicroBlogDomainSubMenu.MyMicroBlogHome;
            return View("Pages/MyMicroBlogHome.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 我的微博
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public ActionResult ListMyMicroBlogThreads(string userDomainName)
        {
            SetPageTitle(GetResourceString("Navigation_MyMicroBlog", applicationID));
            ViewData["MicroBlogDomainSubMenu"] = (int)MicroBlogDomainSubMenu.MyMicroBlogs;
            return View("Pages/ListMyMicroBlogThreads.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 提到我的微博列表
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult ListReferMeMicroBlogs(string userDomainName)
        {
            SetPageTitle(string.Format(GetResourceString("Navigation_Atme", applicationID), CurrentDomainUser.DisplayName));
            ViewData["MicroBlogDomainSubMenu"] = (int)MicroBlogDomainSubMenu.Atme;
            ViewData["Scopes"] = MicroBlogScopes.ReferMe;
            return View("Pages/ListMicroBlogThreads.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 我的收藏
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult MyFavoriteMicroBlogs(string userDomainName)
        {
            SetPageTitle(userDomainName, "我的收藏");
            ViewData["MicroBlogDomainSubMenu"] = (int)MicroBlogDomainSubMenu.MyFavorite;
            ViewData["Scopes"] = MicroBlogScopes.MyFavorite;
            return View("Pages/ListMicroBlogThreads.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 用户统计信息块
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_MicroBlogInfo(string userDomainName, int? subMenu)
        {
            User domainUser = Users.GetUser(userDomainName, false);
            if (domainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserNotFound());
            MicroBlogUserData userData = new MicroBlogUserData();
            userData = MicroBlogUserDatas.GetUserData(domainUser.UserID, false);
            ViewData["MicroBlogDomainSubMenu"] = subMenu ?? -1;
            return PartialView("Controls/MicroBlogInfoBlock.ascx", userData);
        }

        /// <summary>
        /// 其他用户访问我的微博时显示的用户信息
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_MicroBlogUserCard(string userDomainName)
        {
            return PartialView("Controls/MicroBlogUserCard.ascx");
        }

        /// <summary>
        /// 微博空间列表控件
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="scopes"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_ListMicroBlogThreads(string userDomainName, MicroBlogScopes scopes, MicroBlogTypes? type, int? gid, int? pageIndex)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<MicroBlogThread> pds = null;

            switch (scopes)
            {
                case MicroBlogScopes.MyCreate:
                    if (HasManagePermission(userDomainName))
                        pds = MicroBlogThreads.GetMyThreads(domainUser.UserID, type, pageIndex ?? 1);
                    else
                        pds = MicroBlogThreads.GetThreads(domainUser.UserID, type, pageIndex ?? 1, false);
                    break;
                case MicroBlogScopes.MyFollow:
                    {
                        pds = MicroBlogThreads.GetThreadsOfTimeLine(domainUser.UserID, type, gid.HasValue && gid.Value == 0 ? null : gid, pageIndex ?? 1, false);
                        List<UserCategory> hideGroups = null;
                        List<UserCategory> defaultGroups = null;
                        List<UserCategory> groups = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).GetUserCategories(domainUser.UserID, true);
                        if (groups != null)
                        {
                            groups = groups.Where(n => n.CategoryID > 0).ToList();

                            if (groups.Count >= 4)
                            {
                                hideGroups = groups.GetRange(4, groups.Count - 4);
                                defaultGroups = groups.GetRange(0, 4);
                                if (gid.HasValue)
                                {
                                    if (hideGroups != null && hideGroups.Select(n => n.CategoryID).Contains(gid.Value))
                                    {
                                        int index = hideGroups.FindIndex(n => n.CategoryID == gid.Value);
                                        UserCategory group = defaultGroups.Last();
                                        defaultGroups[defaultGroups.Count - 1] = hideGroups[index];
                                        hideGroups[index] = group;

                                    }
                                }
                            }
                            else
                                defaultGroups = groups;
                        }

                        ViewData["hideGroups"] = hideGroups;
                        ViewData["defaultGroups"] = defaultGroups;
                        ViewData["groups"] = groups;
                        ViewData["gid"] = gid;
                        break;
                    }
                case MicroBlogScopes.ReferMe:
                    pds = MicroBlogThreads.GetReferMeThreads(domainUser.UserID, pageIndex ?? 1);
                    break;
                case MicroBlogScopes.MyFavorite:
                    pds = MicroBlogThreads.GetFavoriteThreads(domainUser.UserID, pageIndex ?? 1);
                    break;
                default:
                    break;
            }

            ViewData["Scopes"] = (int)scopes;
            return PartialView("Controls/ListMicroBlogThreads.ascx", pds);
        }

        /// <summary>
        /// 微博排行
        /// </summary>
        /// <returns></returns>
        public ActionResult MicroBlogRanks(MicroBlogThreadSortBy? sortBy)
        {
            SetPageTitle("微博排行");
            ViewData["SortBy"] = sortBy.HasValue ? (int)sortBy : (int)MicroBlogThreadSortBy.DateCreated;
            return View("Pages/MicroBlogRanks.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 最新微博
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_ListNewMicroBlogs(MicroBlogThreadSortBy? sortBy, int? pageIndex)
        {

            PagingDataSet<MicroBlogThread> pds = MicroBlogThreads.GetPagingThreadsSortBy(sortBy ?? MicroBlogThreadSortBy.DateCreated, 40, pageIndex ?? 1);
            ViewData["sortBy"] = sortBy ?? MicroBlogThreadSortBy.DateCreated;
            return PartialView("Controls/ListSortMicroBlogThreads.ascx", pds);
        }

        /// <summary>
        /// 话题下的微博
        /// </summary>
        /// <param name="keyWord"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public ActionResult SearchMicroBlogs(int? pageIndex)
        {

            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));

            SetPageTitle("搜索话题");
            string keyword = Request.QueryString.GetString("Keyword", string.Empty);
            SearchResultDataSet<MicroBlogThread> results = null;
            if (!string.IsNullOrEmpty(keyword))
                results = MicroBlogSearchManager.Instance().Search(WebUtils.UrlDecode(keyword), true, 20, pageIndex ?? 1);
            TempData["Keyword"] = WebUtils.UrlDecode(keyword);

            return View("Pages/ListMicroBlogsOfTopic.aspx", "Masters/MicroBlog.Master", results);
        }

        /// <summary>
        /// 显示微博图片
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_MicroBlogPhoto(int threadID, int? othreadID, MicroBlogPhotoTypes type)
        {
            return PartialView("Controls/MicroBlogPhoto.ascx");
        }

        /// <summary>
        /// 显示视频
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_MicroBlogVideo(int? threadID, string thumbnailUrl, string playUrl, bool play)
        {
            return PartialView("Controls/MicroBlogVideo.ascx");
        }

        /// <summary>
        /// 显示音乐
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_MicroBlogMusic(int? threadID, string playUrl, bool play)
        {
            return PartialView("Controls/MicroBlogMusic.ascx");
        }

        /// <summary>
        /// 关注微博
        /// </summary>
        public ActionResult FavoriteMicroBlog(int threadID)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Error404("没有操作权限或用户不存在！"));
            if (threadID > 0)
            {
                if (!MicroBlogThreads.IsFavorite(CurrentUser.UserID, threadID))
                {
                    MicroBlogThreads.CreateFavorite(CurrentUser.UserID, threadID);
                }
                else
                {
                    MicroBlogThreads.DeleteFavorite(CurrentUser.UserID, threadID);
                }
            }
            return new EmptyResult();
        }

        /// <summary>
        /// 获取最新的微博
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_Newthreads(bool? inDomain, int cthreadID, string keyword)
        {
            if (CurrentUser == null)
                return new EmptyResult();
            if (cthreadID == 0)
                return new EmptyResult();

            IList<MicroBlogThread> threads = null;
            if (string.IsNullOrEmpty(keyword))
                threads = MicroBlogThreads.GetNewThreads(inDomain ?? true ? CurrentUser.UserID : 0, cthreadID);
            else
            {
                threads = TopicMicroBlogs(WebUtils.UrlDecode(keyword), 50, 1);
                if (threads != null)
                {
                    int index = threads.Select(n => n.ThreadID).ToList().IndexOf(cthreadID);
                    threads = index > 0 ? threads.Take(index).ToList() : null;
                }
                ViewData["isTopic"] = true;
            }

            return PartialView("Controls/MicroBlogItems.ascx", threads);
        }

        /// <summary>
        /// 获取新微博消息通知
        /// </summary>
        public ActionResult NewThreadsMessage(bool? inDomain, int cthreadID, string keyword)
        {
            if (CurrentUser == null)
                return Json(new { }, JsonRequestBehavior.AllowGet);
            if (cthreadID == 0)
                return Json(new { }, JsonRequestBehavior.AllowGet);

            int count = 0;
            if (string.IsNullOrEmpty(keyword))
                count = MicroBlogThreads.GetNewThreadCount(inDomain ?? true ? CurrentUser.UserID : 0, cthreadID);
            else
            {
                IList<MicroBlogThread> threads = TopicMicroBlogs(WebUtils.UrlDecode(keyword), 50, 1);
                if (threads != null)
                {
                    int index = threads.Select(n => n.ThreadID).ToList().IndexOf(cthreadID);
                    count = index > 0 ? index : 0;
                }
            }
            return Json(new { count = count }, JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 同步新浪微博
        /// </summary>
        private void SyncSina(string body, int userID, int threadID, bool? hasPhoto)
        {
            AccountBinding ab = AccountBindings.GetAccountBinding(userID, AccountTypes.Instance().SinaWeibo());
            if (ab == null)
                return;

            string imgurl = string.Empty;
            if (hasPhoto.HasValue && hasPhoto.Value)
            {
                Attachment attachment = MicroBlogAttachmentManager.Instance().GetLatestAttachmentOfAssociateID(threadID);
                if (attachment != null)
                {
                    IFile file = FileStorageProvider.Instance(MicroBlogAttachmentManager.Instance().StorageProviderName).GetFile(attachment.GetRelativePath(), attachment.FileName);

                    if (file != null)
                    {
                        FileSystemFile fileSystemFile = file as FileSystemFile;
                        imgurl = fileSystemFile.FullLocalPath;
                    }
                }
            }

            oAuthSina oauth = new oAuthSina();
            oauth.token = ab.OauthToken;
            oauth.tokenSecret = ab.OauthTokenSecret;

            if (string.IsNullOrEmpty(imgurl))
            {
                string url = "http://api.t.sina.com.cn/statuses/update.xml?";
                oauth.oAuthWebRequest(oAuthSina.Method.POST, url, "status=" + body);
            }
            else
            {
                string url = "http://api.t.sina.com.cn/statuses/upload.xml?";
                oauth.oAuthWebRequestWithPic(oAuthSina.Method.POST, url, "status=" + body, imgurl);
            }
        }

        #endregion

        #region Comment

        /// <summary>
        /// 我的微博首页
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult MyMicroBlogComments(string userDomainName)
        {
            SetPageTitle(GetResourceString("Navigation_MyMicroBlogComments", applicationID));
            if (CurrentDomainUser.UserID != CurrentUser.UserID)
                return RedirectToAction("ListMyMicroBlogThreads", new RouteValueDictionary { { "userDomainName", userDomainName } });

            ViewData["MicroBlogDomainSubMenu"] = (int)MicroBlogDomainSubMenu.MyComment;
            return View("Pages/MyMicroBlogComments.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 我的评论列表控件
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="scopes"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult Control_ListMicroBlogComments(string userDomainName, int? pageIndex)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            if (domainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserNotFound());

            PagingDataSet<MicroBlogComment> pds = null;

            pds = MicroBlogComments.GetMyComments(domainUser.UserID, pageIndex ?? 1);
            return PartialView("Controls/ListMicroBlogComments.ascx", pds);
        }

        /// <summary>
        /// 回复微博
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateMyMicroBlogComment(string userDomainName, int threadID, int parentID, bool isSend)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Error404("你没有该项操作权限！"));
            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, false);

            if (thread != null)
            {
                string body = Request.Form["microBlogCommentBody"];
                if (string.IsNullOrEmpty(body))
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_CommentLimit", applicationID));
                    TempData["microBlogCommentBody"] = body;

                    if (isSend == true)
                        return RedirectToAction("MySendMicroBlogComments", new RouteValueDictionary { { "userDomainName", userDomainName } });
                    else
                        return RedirectToAction("MyMicroBlogComments", new RouteValueDictionary { { "userDomainName", userDomainName } });
                }

                MicroBlogComment comment = new MicroBlogComment();
                comment.ThreadID = threadID;
                comment.Body = body;
                comment.ParentID = parentID;
                comment.OwnerUserID = thread.OwnerUserID;
                comment.PostMode = PostModes.Web;
                comment.PostSource = WebUtils.IPAddress;
                comment.UserID = CurrentUser.UserID;
                comment.Author = CurrentUser.DisplayName;
                try
                {
                    #region 检查发帖时间间隔,防灌水
                    Globals.CheckPostDate();
                    #endregion

                    MicroBlogComments.CreateComment(comment);
                }
                catch (Exception e)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "不允许频繁发帖，请稍等片刻再试");
                    if (isSend == true)
                        return RedirectToAction("MySendMicroBlogComments", new RouteValueDictionary { { "userDomainName", userDomainName } });
                    else
                        return RedirectToAction("MyMicroBlogComments", new RouteValueDictionary { { "userDomainName", userDomainName } });
                }
                //评论同时转发微博
                int isForward = Request.Form.GetInt("microBlogForward", 0);
                if (isForward == 1)
                {
                    MicroBlogThread forwardThread = new MicroBlogThread();
                    forwardThread.Body = body;
                    forwardThread.OwnerUserID = CurrentUser.UserID;
                    forwardThread.Author = CurrentUser.DisplayName;
                    forwardThread.PostMode = PostModes.Web;
                    forwardThread.PostSource = WebUtils.IPAddress;
                    forwardThread.OriginalThreadID = threadID;
                    MicroBlogThreads.CreateThread(forwardThread);
                }

                if ((int)comment.AuditingStatus < (int)SiteSettingsManager.GetSiteSettings().AuditingStatusesForPublicDisplay)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Message_CommentAudit", applicationID));
                }
            }
            if (isSend == true)
                return RedirectToAction("MySendMicroBlogComments", new RouteValueDictionary { { "userDomainName", userDomainName } });
            else
                return RedirectToAction("MyMicroBlogComments", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 删除微博回复
        /// </summary>
        public ActionResult DeleteMyMicroBlogComment(int? threadID, int commentID, string userDomainName, int? pageIndex)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Error404("你没有该项操作权限！"));

            //TODO:验证操作者权限
            MicroBlogComment comment = MicroBlogComments.GetComment(commentID, true);
            if (comment != null)
            {
                if (CurrentUser.IsContentAdministrator || CurrentUser.UserID == comment.UserID)
                {
                    MicroBlogComments.DeleteComment(commentID);
                }
            }

            return Control_ListMicroBlogComments(userDomainName, pageIndex ?? 1);

        }

        /// <summary>
        /// 我的微博首页
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult MySendMicroBlogComments(string userDomainName)
        {
            SetPageTitle(GetResourceString("Navigation_MyMicroBlogComments", applicationID));
            if (CurrentDomainUser.UserID != CurrentUser.UserID)
                return RedirectToAction("ListMyMicroBlogThreads", new RouteValueDictionary { { "userDomainName", userDomainName } });

            ViewData["MicroBlogDomainSubMenu"] = (int)MicroBlogDomainSubMenu.MyComment;
            return View("Pages/MySendMicroBlogComments.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 我的评论列表控件
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="scopes"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult Control_ListSendMicroBlogComments(string userDomainName, int? pageIndex)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            if (domainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserNotFound());

            PagingDataSet<MicroBlogComment> pds = null;

            pds = MicroBlogComments.GetMySendComments(domainUser.UserID, pageIndex ?? 1);
            return PartialView("Controls/ListSendMicroBlogComments.ascx", pds);
        }

        /// <summary>
        /// 删除微博回复
        /// </summary>
        public ActionResult DeleteMySendMicroBlogComment(int? threadID, int commentID, string userDomainName, int? pageIndex)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Error404("你没有该项操作权限！"));

            //TODO:验证操作者权限
            MicroBlogComment comment = MicroBlogComments.GetComment(commentID, true);
            if (comment != null)
            {
                if (CurrentUser.IsContentAdministrator || CurrentUser.UserID == comment.UserID)
                {
                    MicroBlogComments.DeleteComment(commentID);
                }
            }

            return Control_ListSendMicroBlogComments(userDomainName, pageIndex ?? 1);

        }

        /// <summary>
        /// 微博评论
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_MicroBlogThreadComments(int threadID, int? pageIndex)
        {
            MicroBlogThread microBlogThread = MicroBlogThreads.GetThread(threadID, false);
            bool isManager = false;
            if (microBlogThread == null)
            {
                throw new SPBException(ExceptionTypes.Instance().UnknownError());
                // return Redirect(SiteUrls.Instance().Error404(String.Format("找不到微博！")));
            }

            if ((microBlogThread.OwnerUserID == CurrentUser.UserID) || CurrentUser.IsContentAdministrator)
                isManager = true;

            PagingDataSet<MicroBlogComment> Comments = MicroBlogComments.GetCommentsOfThread(microBlogThread.ThreadID, pageIndex ?? 1, isManager);
            ViewData["threadID"] = threadID;
            ViewData["isManager"] = isManager;
            ViewData["microBlogThread"] = microBlogThread;
            return PartialView("Controls/ListMicroBlogThreadComments.ascx", Comments);
        }

        /// <summary>
        /// 微博列表中的评论
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public ActionResult Control_ListCommentsOfThread(int threadID)
        {
            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, false);
            IList<MicroBlogComment> comments = null;

            if (thread != null)
            {
                if (thread.UserID == CurrentUser.UserID)
                    comments = MicroBlogComments.GetTopCommentsOfThread(threadID, 10, true);
                else
                    comments = MicroBlogComments.GetTopCommentsOfThread(threadID, 10, false);
            }

            ViewData["thread"] = thread;
            return PartialView("Controls/ListMicroBlogCommentsOfThread.ascx", comments);

        }

        /// <summary>
        /// 回复微博
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateMicroBlogComment(string userDomainName, int threadID, CommentType? type, int? pageIndex)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Error404("你没有该项操作权限！"));
            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, false);

            type = type.HasValue ? type : CommentType.ListComment;
            if (thread != null)
            {
                string body = Request.Form["microBlogCommentBody"];
                if (string.IsNullOrEmpty(body))
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_CommentLimit", applicationID));
                    TempData["microBlogCommentBody"] = body;
                    if (type == CommentType.ListComment)
                        return Control_MicroBlogThreadComments(threadID, pageIndex ?? 1);
                    else
                        return Control_ListCommentsOfThread(threadID);
                }

                MicroBlogComment comment = new MicroBlogComment();
                comment.ThreadID = threadID;
                comment.Body = body;
                int commentID = Request.Form.GetInt("commentID", 0);
                if (commentID > 0)
                    comment.ParentID = commentID;
                else
                    comment.ParentID = 0;

                comment.OwnerUserID = thread.OwnerUserID;
                comment.PostMode = PostModes.Web;
                comment.PostSource = WebUtils.IPAddress;
                comment.UserID = CurrentUser.UserID;
                comment.Author = CurrentUser.DisplayName;
                try
                {
                    Globals.CheckPostDate();
                    MicroBlogComments.CreateComment(comment);
                }
                catch (Exception e)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "不允许频繁发帖，请稍等片刻再试");
                    if (type == CommentType.ListComment)
                        return Control_MicroBlogThreadComments(threadID, pageIndex ?? 1);
                    else
                        return Control_ListCommentsOfThread(threadID);
                }
                //评论同时转发微博
                int isForward = Request.Form.GetInt("microBlogForward", 0);
                if (isForward == 1)
                {
                    MicroBlogThread forwardThread = new MicroBlogThread();
                    forwardThread.Body = body;
                    forwardThread.OwnerUserID = CurrentUser.UserID;
                    forwardThread.Author = CurrentUser.DisplayName;
                    forwardThread.PostMode = PostModes.Web;
                    forwardThread.PostSource = WebUtils.IPAddress;
                    forwardThread.OriginalThreadID = threadID;
                    MicroBlogThreads.CreateThread(forwardThread);
                }

                if ((int)comment.AuditingStatus < (int)SiteSettingsManager.GetSiteSettings().AuditingStatusesForPublicDisplay)
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Message_CommentAudit", applicationID));
            }

            if (type == CommentType.ListComment)
                return Control_MicroBlogThreadComments(threadID, pageIndex ?? 1);
            else
                return Control_ListCommentsOfThread(threadID);
        }

        /// <summary>
        /// 删除微博回复
        /// </summary>
        public ActionResult DeleteMicroBlogComment(int? threadID, int commentID, CommentType? type, int? pageIndex)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Error404("你没有该项操作权限！"));

            //TODO:验证操作者权限
            MicroBlogComment comment = MicroBlogComments.GetComment(commentID, true);
            if (comment != null)
            {
                if (CurrentUser.IsContentAdministrator || CurrentUser.UserID == comment.UserID)
                {
                    MicroBlogComments.DeleteComment(commentID);
                }
            }

            type = type.HasValue ? type : CommentType.ListComment;
            if (type == CommentType.ListComment)
                return Control_MicroBlogThreadComments(threadID ?? 0, pageIndex ?? 1);
            else
                return Control_ListCommentsOfThread(threadID ?? 0);

        }

        #endregion

        #region Topic

        /// <summary>
        /// 特色话题
        /// </summary>
        /// <param name="topicID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public ActionResult ShowMicroBlogTopic(int topicID, int? pageIndex)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));

            MicroBlogTopic topic = MicroBlogTopics.GetMicroBlogTopic(topicID, false);

            SetPageTitle(topic.Title);

            //搜索相关微薄
            SearchResultDataSet<MicroBlogThread> results = null;
            if (!string.IsNullOrEmpty(topic.Body))
                results = MicroBlogSearchManager.Instance().Search(WebUtils.UrlDecode(topic.Body), true, 20, pageIndex ?? 1);

            ViewData["topic"] = topic;
            return View("Pages/ShowMicroBlogTopic.aspx", "Masters/MicroBlog.Master", results);
        }

        /// <summary>
        /// 分享话题控件
        /// </summary>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public ActionResult Control_ShareMicroBlogTopic(string returnUrl)
        {
            ViewData["returnUrl"] = returnUrl;
            return PartialView("Controls/ShareMicroBlogTopic.ascx");
        }

        /// <summary>
        /// 微博排行
        /// </summary>
        /// <returns></returns>
        public ActionResult MicroBlogsTopicRanks()
        {
            SetPageTitle("话题排行");
            return View("Pages/MicroBlogTopicRanks.aspx", "Masters/MicroBlog.Master");
        }

        /// <summary>
        /// 热门话题排行
        /// </summary>
        /// <param name="commendTypeID"></param>
        /// <param name="disPlayCount"></param>
        /// <returns></returns>
        public ActionResult Control_MicroBlogTopicRanks(int? disPlayCount, TopicSortBy? sortBy)
        {
            IList<MicroBlogTopic> microBlogTopics = MicroBlogTopics.GetTopTopicSortBy(sortBy ?? TopicSortBy.DateCreated, disPlayCount ?? 10);

            ViewData["sortBy"] = sortBy;
            return PartialView("Controls/MicroBlogTopicRanks.ascx", microBlogTopics);
        }

        /// <summary>
        /// 关注话题内容块
        /// </summary>
        public ActionResult TopicesFavitesBlock(string UserDomainName)
        {
            if (string.IsNullOrEmpty(UserDomainName))
                throw new ArgumentNullException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = Users.GetUser(UserDomainName, true);
            IList<string> Topic = MicroBlogTopics.GetTopicesFavorites(currentDomainUser.UserID);

            return View("Controls/TopicesFavoritesBlock.ascx", Topic);

        }

        /// <summary>
        /// 加关注
        /// </summary>       
        public JsonResult FaviteTopic(string keyword)
        {
            if (CurrentUser == null || string.IsNullOrEmpty(keyword))
                return Json(new { }, JsonRequestBehavior.AllowGet);
            string isFavorite = "create";
            keyword = WebUtils.UrlDecode(keyword);
            if (!MicroBlogTopics.IsFavorite(CurrentUser.UserID, WebUtils.UrlDecode(keyword)))
            {
                MicroBlogTopics.CreateFavoriteTopic(CurrentUser.UserID, keyword);
            }
            else
            {
                MicroBlogTopics.DeleteFavoriteTopic(CurrentUser.UserID, keyword);
                isFavorite = "delete";
            }
            return Json(new { ok = true, isFavorite = isFavorite }, JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 删除话题
        /// </summary>      
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteFavoriteTopic(string topic, string UserDomainName)
        {
            if (string.IsNullOrEmpty(UserDomainName))
                throw new ArgumentNullException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(UserDomainName);
            MicroBlogTopics.DeleteFavoriteTopic(currentDomainUser.UserID, topic);
            return RedirectToAction("TopicesFavitesBlock", new RouteValueDictionary { { "UserDomainName", UserDomainName } });
        }

        /// <summary>
        /// 热门话题
        /// </summary>
        /// <param name="commendTypeID"></param>
        /// <param name="disPlayCount"></param>
        /// <returns></returns>
        public ActionResult Control_TopMicroBlogTopic(int? disPlayCount)
        {
            IList<MicroBlogTopic> microBlogTopics = MicroBlogTopics.GetTopTopicSortBy(TopicSortBy.ItemCount, disPlayCount ?? 10);

            return PartialView("Controls/TopMicroBlogTopics.ascx", microBlogTopics);
        }

        #endregion

        #region Screen

        /// <summary>
        /// 我的大屏幕
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult MyScreens(string userDomainName, int? pageIndex)
        {
            SetPageTitle(userDomainName, "我的大屏幕");
            PagingDataSet<MicroBlogScreen> pds = MicroBlogScreens.GetScreens(CurrentDomainUser.UserID, pageIndex ?? 1);
            return View("Pages/MyScreens.aspx", "Masters/MicroBlog.Master", pds);
        }

        /// <summary>
        /// 创建编辑大屏幕
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_EditScreen(string userDomainName)
        {
            if (CurrentDomainUser.Rank < MicroBlogConfiguration.Instance().CreateScreenRank)
                return Redirect(SiteUrls.Instance().Error("用户等级达到" + MicroBlogConfiguration.Instance().CreateScreenRank + "级才能创建大屏幕！"));

            MicroBlogScreen screen = new MicroBlogScreen();
            return PartialView("Controls/EditScreen.ascx", screen);
        }

        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UpdateScreen(string userDomainName)
        {
            if (CurrentDomainUser.Rank < MicroBlogConfiguration.Instance().CreateScreenRank)
                return Redirect(SiteUrls.Instance().Error("用户等级达到" + MicroBlogConfiguration.Instance().CreateScreenRank + "级才能创建大屏幕！"));

            MicroBlogScreen screen = new MicroBlogScreen();
            screen.UserID = CurrentDomainUser.UserID;
            screen.ScreenName = Request.Form.GetString("ScreenName", string.Empty);
            screen.Keyword = Request.Form.GetString("Keyword", string.Empty);
            screen.IsDeadline = Request.Form.GetBool("IsDeadline", true);
            if (screen.IsDeadline)
            {
                screen.StartDate = DateTime.Parse(Request.Form.GetString("startDate", DateTime.Now.ToString()));
                screen.EndDate = DateTime.Parse(Request.Form.GetString("endDate", DateTime.Now.AddDays(1).ToString()));
            }
            screen.IsFilter = Request.Form.GetBool("IsFilter", true);
            if (string.IsNullOrEmpty(screen.ScreenName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "名称不能为空！";
                return PartialView("Controls/EditScreen.ascx", screen);
            }

            if (screen.StartDate > screen.EndDate)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "开始时间不能大于截止时间！";
                return PartialView("Controls/EditScreen.ascx", screen);
            }

            try
            {
                MicroBlogScreens.CreateScreen(screen);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            }
            catch
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "创建失败！";
            }

            return PartialView("Controls/EditScreen.ascx", screen);
        }

        /// <summary>
        /// 删除大屏幕
        /// </summary>
        public ActionResult DeleteScreen(string userDomainName, int? screenID, int? pageIndex)
        {
            if (screenID.HasValue && screenID.Value > 0)
            {
                MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID.Value, true);
                if (screen != null)
                {
                    try
                    {
                        MicroBlogScreens.DeleteScreen(screenID.Value);
                    }
                    catch { }
                }
            }

            return RedirectToAction("MyScreens", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 大屏幕导播室
        /// </summary>
        /// <returns></returns>
        public ActionResult ScreenLeadStudio(int screenID, bool? isPlay, int? pageIndex)
        {

            MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID, true);

            if (screen == null)
                return Redirect(SiteUrls.Instance().Error404("大屏幕不存在", "大屏幕不存在或已被删除！"));

            if (!screen.IsFilter)
                return Redirect(SiteUrls.Instance().Error404("页面不存在"));

            if (CurrentUser == null || (CurrentUser.UserID != screen.UserID && !CurrentUser.IsAdministrator && !CurrentUser.IsContentAdministrator))
                return Redirect(SiteUrls.Instance().Error("没有权限访问当前页面！"));

            if (screen.IsDeadline)
            {
                if (screen.StartDate.CompareTo(DateTime.Now) > 0)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "大屏幕还未开始！");
                    return View("Pages/ScreenLeadStudio.aspx", "Masters/Screen.master", screen);
                }

                if (screen.EndDate.CompareTo(DateTime.Now) < 0)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "大屏幕已经超过截止时间！");
                    return View("Pages/ScreenLeadStudio.aspx", "Masters/Screen.master", screen);
                }
            }


            if (screen.AuditingStatus < AuditingStatuses.Success)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "大屏幕未通过审核！");
                return View("Pages/ScreenLeadStudio.aspx", "Masters/Screen.master");
            }

            ViewData["isPlay"] = isPlay;
            ViewData["screen"] = screen;
            ViewData["pageIndex"] = pageIndex;

            return View("Pages/ScreenLeadStudio.aspx", "Masters/Screen.master", screen);
        }

        /// <summary>
        /// 显示大屏幕中有审核状态的微博
        /// </summary
        public ActionResult Control_ListMicroBlogsOfScreen(int screenID)
        {
            Dictionary<int, MicroBlogThread> dict = MicroBlogThreads.GetPlayList(null, screenID);
            return PartialView("Controls/ListPlayList.ascx", dict);
        }

        /// <summary>
        /// 大屏幕全部微博
        /// </summary>
        public ActionResult Control_ListTopicMicroBlogsOfScreen(string keyword, int? pageIndex)
        {
            SearchResultDataSet<MicroBlogThread> pds = MicroBlogSearchManager.Instance().Search(keyword, true, 20, pageIndex ?? 1);
            ViewData["keyword"] = keyword;
            return PartialView("Controls/ListTopicMicroBlogsOfScreen.ascx", pds);
        }

        /// <summary>
        /// 添加微博到大屏幕
        /// </summary>
        public JsonResult AddItem(int threadID, int screenID, bool? isInsert)
        {

            if (CurrentUser == null)
                return Json(new { }, JsonRequestBehavior.AllowGet);

            MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID, false);
            if (screen == null)
                return Json(new { }, JsonRequestBehavior.AllowGet);

            if (screen.UserID != CurrentUser.UserID && !CurrentUser.IsAdministrator && !CurrentUser.IsContentAdministrator)
                return Json(new { }, JsonRequestBehavior.AllowGet);

            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, false);
            if (thread == null)
                return Json(new { }, JsonRequestBehavior.AllowGet);

            try
            {
                if (isInsert.HasValue && isInsert.Value)
                    MicroBlogThreads.InsertMicroBlogOfScreen(screenID, threadID);
                else
                    MicroBlogThreads.CreateMicroBlogsOfScreen(threadID, screenID);
            }
            catch
            {
                return Json(new { error = true }, JsonRequestBehavior.AllowGet);
            }

            return Json(new { ok = true }, JsonRequestBehavior.AllowGet);
        }
        /// <summary>
        ///批量添加微博到大屏幕
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult AddItems(int screenID, string returnUrl)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Error("你没有操作权限！"));

            MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID, false);
            if (screen == null)
                return Redirect(SiteUrls.Instance().Error404("你访问的页面不存在", "大屏幕不存在或已被删除！"));

            if (screen.UserID != CurrentUser.UserID && !CurrentUser.IsAdministrator && !CurrentUser.IsContentAdministrator)
                return Redirect(SiteUrls.Instance().Error("你没有操作权限！"));
            string strIDs = Request.Form["MicroBlog"];
            if (string.IsNullOrEmpty(strIDs))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "至少选择一条要操作的微博！";
                return Redirect(returnUrl);
            }

            List<int> threadIDs = new List<int>();
            string[] arrIDs = strIDs.Split(',');
            foreach (string strid in arrIDs)
            {
                int id = 0;
                int.TryParse(strid, out id);
                if (id > 0)
                    threadIDs.Add(id);
            }

            if (threadIDs != null)
                MicroBlogThreads.CreateMicroBlogsOfScreen(threadIDs, screenID);

            return Redirect(returnUrl);
        }

        /// <summary>
        /// 创建收藏
        /// </summary>
        public JsonResult CreateFavoriteOfScreen(int screenID, int serialNumber, int threadID)
        {

            if (CurrentUser == null || screenID <= 0 || serialNumber <= 0 || threadID <= 0)
                return Json(new { }, JsonRequestBehavior.AllowGet);

            MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID, false);
            if (screen == null)
                return Json(new { }, JsonRequestBehavior.AllowGet);

            if (screen.UserID != CurrentUser.UserID && !CurrentUser.IsAdministrator && !CurrentUser.IsContentAdministrator)
                return Json(new { }, JsonRequestBehavior.AllowGet);

            try
            {
                MicroBlogThreads.CreateFavoriteOfScreen(screenID, serialNumber, threadID);
            }
            catch
            {
                return Json(new { error = true }, JsonRequestBehavior.AllowGet);
            }
            return Json(new { ok = true }, JsonRequestBehavior.AllowGet);
        }

        public ActionResult ScreenFavorites(int screenID, int serialNumber, int? pageIndex)
        {
            MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID, true);

            if (screen == null)
                return Redirect(SiteUrls.Instance().Error404("大屏幕不存在", "大屏幕不存在或已被删除！"));

            if (!screen.IsFilter)
                return Redirect(SiteUrls.Instance().Error404("页面不存在"));

            if (CurrentUser == null || (CurrentUser.UserID != screen.UserID && !CurrentUser.IsAdministrator && !CurrentUser.IsContentAdministrator))
                return Redirect(SiteUrls.Instance().Error("没有权限访问当前页面！"));

            if (screen.IsDeadline)
            {
                if (screen.StartDate.CompareTo(DateTime.Now) > 0)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "大屏幕还未开始！");
                    return View("Pages/ScreenLeadStudio.aspx", "Masters/Screen.master", screen);
                }

                if (screen.EndDate.CompareTo(DateTime.Now) < 0)
                    return View("Pages/ScreenLeadStudio.aspx", "Masters/Screen.master", screen);
            }

            if (screen.AuditingStatus < AuditingStatuses.Success)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "大屏幕未通过审核！");
                return View("Pages/ScreenFavorites.aspx", "Masters/Screen.master", screen);
            }

            ViewData["screen"] = screen;
            ViewData["pageIndex"] = pageIndex;

            return View("Pages/ScreenFavorites.aspx", "Masters/Screen.master", screen);
        }


        public ActionResult Control_ListMicroBlogsOfScreenFavorite(int screenID, int serialNumber, int? pageIndex)
        {

            PagingDataSet<MicroBlogThread> pds = null;
            MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID, false);
            if (screen != null)
                pds = MicroBlogThreads.GetThreadsOfScreenFavorite(screenID, serialNumber, pageIndex ?? 0);
            ViewData["isFavorite"] = true;
            ViewData["status"] = AuditingStatusesForDisplay.Pending;
            ViewData["isPause"] = screen.IsPause;
            return PartialView("Controls/ListMicroBlogsOfScreen.ascx", pds);
        }

        /// <summary>
        /// 删除收藏夹中的微博
        /// </summary>
        public ActionResult DeleteMciroBlogOfScreenFavorite(int screenID, int serialNumber, string returnUrl)
        {
            if (CurrentUser == null || screenID <= 0 || serialNumber <= 0)
                return Redirect(SiteUrls.Instance().Error404("没有相关操作权限！"));

            MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID, false);
            if (screen == null)
                return Redirect(SiteUrls.Instance().Error404("大屏幕不存在！", "大屏幕不存在或已被删除!"));
            if (screen.UserID != CurrentUser.UserID)
                return Redirect(SiteUrls.Instance().Error404("没有相关操作权限！"));

            try
            {
                string strIDs = Request.Form["MicroBlog"];

                if (string.IsNullOrEmpty(strIDs))
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "至少选择一条要操作的微博！";
                    return Redirect(returnUrl);
                }
                string[] arrIDs = strIDs.Split(',');
                List<int> threadIDs = new List<int>();
                foreach (string strID in arrIDs)
                {
                    int id = 0;
                    int.TryParse(strID, out id);
                    if (id > 0)
                        threadIDs.Add(id);
                }

                if (threadIDs != null)
                    MicroBlogThreads.DeleteFavoriteOfScreen(screenID, serialNumber, threadIDs);

            }
            catch
            {
                TempData["StatusMessageType"] = StatusMessageType.Error;
                TempData["StatusMessageContent"] = "移除失败请联系管理员！";
                return Redirect(returnUrl);
            }

            return Redirect(returnUrl);
        }

        public ActionResult Screen(int screenID)
        {
            if (screenID <= 0)
                return Redirect(SiteUrls.Instance().Error404());
            MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID, false);

            if (screen == null || screen.AuditingStatus < AuditingStatuses.Success || (screen.IsDeadline && (screen.StartDate > DateTime.Now || screen.EndDate < DateTime.Now)))
                return View("Pages/Screen.aspx");

            if (screen == null)
                return Redirect(SiteUrls.Instance().Error404("大屏幕不存在！", "大屏幕不存在或已被删除！"));

            return View("Pages/Screen.aspx", screen);
        }

        public ActionResult ScreenItems(int screenID, int? itemID, int? threadID)
        {
            if (screenID <= 0)
                return new EmptyResult();

            MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID, false);
            if (screen == null)
                return Redirect(SiteUrls.Instance().Error404());

            if (screen.AuditingStatus < AuditingStatuses.Success || screen.IsPause || (screen.IsDeadline && (screen.StartDate > DateTime.Now || screen.EndDate < DateTime.Now)))
                return new EmptyResult();

            if (screen.IsFilter)
            {
                ViewData["DictThreads"] = MicroBlogThreads.GetScreenItems(screenID, itemID ?? 0);
            }
            else
            {
                IList<MicroBlogThread> threads = TopicMicroBlogs(screen.Keyword, 100, 1);
                if (threads != null && threads.Count > 0)
                {
                    if (threadID.HasValue)
                    {
                        threads = threads.Where(n => n.ThreadID > threadID.Value).OrderBy(n => n.ThreadID).ToList();
                        if (threads != null && threads.Count > 0)
                            threads = threads.ToList().GetRange(0, 1);
                    }
                    else
                        threads = threads.OrderBy(n => n.ThreadID).ToList().GetRange(0, 1);
                }
                ViewData["ilthreads"] = threads;
            }

            return View("Controls/ScreenItems.ascx");

        }

        /// <summary>
        /// 更新导播室播放列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public JsonResult GetPlayedItem(int? screenID, int? itemID)
        {
            if (CurrentUser == null || (screenID.HasValue && screenID.Value <= 0))
                return Json(new { });
            MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID ?? 0, false);
            if (screen == null || screen.IsPause)
                return Json(new { });

            int playItemID = MicroBlogThreads.GetPlayItem(screenID ?? 0, itemID ?? 0);
            if (itemID != playItemID)
                return Json(new { ok = true, pid = playItemID });

            return Json(new { });
        }


        /// <summary>
        /// 获取最新的大屏幕素材
        /// </summary>
        public ActionResult GetTopicItem(int cthreadID, string keyword)
        {
            if (CurrentUser == null || cthreadID == 0 || string.IsNullOrEmpty(keyword))
                return new EmptyResult();

            IList<MicroBlogThread> threads = TopicMicroBlogs(WebUtils.UrlDecode(keyword), 50, 1);
            if (threads != null)
            {
                int index = threads.Select(n => n.ThreadID).ToList().IndexOf(cthreadID);
                threads = index > 0 ? threads.Take(index).ToList() : null;
            }

            return PartialView("Controls/TopicItems.ascx", threads);
        }

        /// <summary>
        /// 从播放列表中删除微博
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public JsonResult DeleteItem(int screenID, int itemID)
        {
            if (CurrentUser == null || screenID <= 0 || itemID < 0)
                return Json(new { error = true, msg = "没有相关操作权限！" });

            MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID, false);
            if (screen == null || (screen.UserID != CurrentUser.UserID && !CurrentUser.IsAdministrator && !CurrentUser.IsContentAdministrator))
                return Json(new { error = true, msg = "操作失败！" });

            try
            {
                MicroBlogThreads.DeleteMicroBlogsOfScreen(itemID, screenID);
            }
            catch
            {
                return Json(new { error = true, msg = "移除失败请联系管理员！" });
            }
            return Json(new { ok = true });

        }

        /// <summary>
        /// 从播放列表中删除微博
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public JsonResult DeleteItems(int screenID)
        {
            if (CurrentUser == null || screenID <= 0)
                return Json(new { error = true, msg = "没有相关操作权限！" });

            MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID, false);
            if (screen == null || (screen.UserID != CurrentUser.UserID && !CurrentUser.IsAdministrator && !CurrentUser.IsContentAdministrator))
                return Json(new { error = true, msg = "操作失败！" });

            try
            {
                string strIDs = Request.Form["MicroBlog"];

                if (string.IsNullOrEmpty(strIDs))
                    return Json(new { error = true, msg = "至少选择一条要操作的微博！" });

                string[] arrIDs = strIDs.Split(',');
                List<int> itemIDs = new List<int>();
                foreach (string strID in arrIDs)
                {
                    int id = 0;
                    int.TryParse(strID, out id);
                    if (id > 0)
                        itemIDs.Add(id);
                }

                if (itemIDs != null)
                    MicroBlogThreads.DeleteMicroBlogsOfScreen(itemIDs, screenID);

            }
            catch
            {
                return Json(new { error = true, msg = "移除失败请联系管理员！" });
            }
            return Json(new { ok = true });

        }

        /// <summary>
        /// 暂停
        /// </summary>
        public JsonResult Pause(int? screenID)
        {
            MicroBlogScreen screen = MicroBlogScreens.GetScreen(screenID ?? 0, false);

            if (screen == null)
                return Json(new { }, JsonRequestBehavior.AllowGet);

            screen.IsPause = screen.IsPause ? false : true;
            try
            {
                MicroBlogScreens.UpdateScreen(screen);
            }
            catch
            {
                return Json(new { }, JsonRequestBehavior.AllowGet);
            }
            if (screen.IsPause)
                return Json(new { pause = true }, JsonRequestBehavior.AllowGet);
            else
                return Json(new { }, JsonRequestBehavior.AllowGet);
        }

        #endregion

        #region Theme

        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult ModifyHome(string userDomainName)
        {
            SetPageTitle("个性化装扮");
            ViewData["CurrentDomaiNameUser"] = CurrentDomainUser;
            ViewData["MicroBlogDomainSubMenu"] = (int)MicroBlogDomainSubMenu.CustomDress;
            return View("Pages/MyMicroBlogHome.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 删除WidgetInstance
        /// </summary>
        /// <param name="instanceID"></param>
        /// <returns></returns>
        public ActionResult Control_ThemeSettings(int presentAreaID, int ownerID, string pageCode, string exitUrl)
        {
            PresentAreaThemeManager themeManager = MicroBlogPresentAreaThemeManager.Instance();

            IList<ThemeAppearance> themeAppearances = Themes.GetThemeAppearancesOfEnabled(presentAreaID, true);
            KeyValuePair<string, string> selectedThemeAppearance = new KeyValuePair<string, string>(themeManager.GetSelectedThemeAppearanceName(ControllerContext), themeManager.GetSelectedThemeName(ControllerContext));

            IList<Widget> initialWidgets = Widgets.GetInitialWidgets(presentAreaID, true);

            IList<string> layoutModes = themeManager.LayoutModes;
            string currentLayoutMode = Themes.GetOwnerLayout(ownerID, selectedThemeAppearance.Value, pageCode, true);

            ViewData.Add("presentAreaID", presentAreaID);
            ViewData.Add("ownerID", ownerID);
            ViewData.Add("pageCode", pageCode);

            ViewData.Add("themeAppearances", themeAppearances);
            ViewData.Add("selectedThemeAppearance", selectedThemeAppearance);

            Dictionary<string, int> usedWidgetIDsOfPage = WidgetInstances.GetUsedWidgetIDs(ownerID, pageCode, selectedThemeAppearance.Value);
            ViewData.Add("initialWidgets", initialWidgets);
            ViewData.Add("usedWidgetIDsOfPage", usedWidgetIDsOfPage);

            ViewData.Add("layoutModes", layoutModes);
            ViewData.Add("currentLayoutMode", currentLayoutMode);

            ViewData.Add("themeLocation", themeManager.PresentArea.ThemeLocation);
            ViewData.Add("exitUrl", exitUrl);

            PresentArea presentArea = PresentAreas.GetPresentArea(presentAreaID, true);
            if (presentArea == null)
                throw new SPBException(ExceptionTypes.Instance().UnknownError());
            ViewData["presentArea"] = presentArea;

            return PartialView("Controls/ThemeSettings.ascx");
        }

        #endregion

        #region ShortUrl

        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public JsonResult CreateShortUrl(string userDomainName)
        {
            string url = Request.Form.GetString("urlForShare", string.Empty);
            if (string.IsNullOrEmpty(url))
                return Json(new { });

            UrlInfo info = UrlManager.Instance().CreateUrlInfo(url);
            if (info == null)
                return Json(new { });

            return Json(new { ok = true, shortUrl = SPBConfig.Instance().ShortUrlDomain.TrimEnd('/') + SiteUrls.Instance().RedirectUrl(info.Alias) });

        }

        public ActionResult RedirectUrl(string shortUrl)
        {
            if (string.IsNullOrEmpty(shortUrl))
                return new EmptyResult();
            UrlInfo info = UrlManager.Instance().GetUrlInfo(shortUrl);
            if (info == null)
                return new EmptyResult();
            return Redirect(info.URL);
        }

        #endregion

        #region Photo

        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UploadPhoto(string userDomainName)
        {
            System.Web.HttpPostedFileBase hpf = null;
            string url = string.Empty;
            if (Request.Files.Count > 0)
            {
                hpf = Request.Files[0] as System.Web.HttpPostedFileBase;
                if (hpf.ContentLength == 0)
                    return Content("<textarea>{\"message\":\"文件大小不能为0KB.\"}</textarea>", "text/html");
                else
                {
                    if (hpf.ContentLength > MicroBlogConfiguration.Instance().MaxAttachmentSize * 1024)
                        return Content("<textarea>{\"message\":\"" + string.Format("上传文件大小不能超过{0}K。", MicroBlogConfiguration.Instance().MaxAttachmentSize) + "\"}</textarea>", "text/html");

                    User currentUser = Users.GetUser(userDomainName, false);

                    if (!currentUser.IsContentAdministrator && hpf.ContentLength / 1024 > (currentUser.DatabaseQuota - currentUser.DatabaseQuotaUsed))
                        return Content("<textarea>{\"message\":\"空间容量不足！\"}</textarea>", "text/html");

                    if (!string.IsNullOrEmpty(hpf.FileName))
                    {
                        string fileTypeString = hpf.FileName.Substring(hpf.FileName.LastIndexOf(".") + 1);
                        string uploadPhotoTypes = MicroBlogConfiguration.Instance().MicroBlogAttachmentSettings.Extensions;
                        StringBuilder newUploadPhotoTypes = new StringBuilder();
                        string[] types = uploadPhotoTypes.Split(',');
                        bool isPermitted = false;
                        foreach (string type in types)
                        {
                            if (!string.IsNullOrEmpty(type))
                            {
                                if (string.Compare(type, fileTypeString, true) == 0)
                                {
                                    isPermitted = true;
                                }
                            }
                        }

                        if (!isPermitted)
                            return Content("<textarea>{\"message\":\"不支持所上传的文件类型！\"}</textarea>", "text/html");
                    }
                    string newfileName = hpf.FileName.Substring(hpf.FileName.LastIndexOf("\\") + 1);

                    Attachment attachment = new Attachment(hpf);
                    attachment.ContentSize = hpf.ContentLength;
                    attachment.ContentType = hpf.ContentType;
                    attachment.DateCreated = DateTime.Now;
                    attachment.UserID = CurrentUser.UserID;
                    attachment.FriendlyFileName = newfileName;

                    IList<Attachment> temAttachments = MicroBlogAttachmentManager.Instance().GetTemporaryAttachmentsOfUser(currentUser.UserID);

                    foreach (var temAtt in temAttachments)
                    {
                        if (temAtt != null) MicroBlogAttachmentManager.Instance().DeleteAttachment(temAtt);
                    }

                    MicroBlogAttachmentManager.Instance().CreateTemporaryAttachment(attachment, hpf.InputStream);
                    IList<Attachment> temporaryAttachments = MicroBlogAttachmentManager.Instance().GetTemporaryAttachmentsOfUser(currentUser.UserID);
                    Attachment temAttachment = MicroBlogAttachmentManager.Instance().GetLatestAttachmentOfUser(currentUser.UserID);
                    url = SiteUrls.Instance().MicroBlogPhotoUrl(temAttachment, MicroBlogPhotoTypes.Preview);
                    return Content("<textarea>{\"ok\":\"true\",\"url\":\"" + url + "\",\"PhotoName\":\"" + StringUtils.Trim(temAttachment.FriendlyFileName.Substring(0, temAttachment.FriendlyFileName.IndexOf('.') - 1), 5, "") + "\",\"attachmentID\":\"" + temAttachment.AttachmentID + "\"}</textarea>", "text/html");
                }
            }

            return Content("<textarea>{\"message\":\"上传失败！\"}</textarea>", "text/html");
        }

        [AcceptVerbs(HttpVerbs.Post)]
        public JsonResult DeletePhoto(int attachmentID)
        {
            Attachment attachment = MicroBlogAttachmentManager.Instance().GetAttachment(attachmentID, true);
            if (attachment != null)
                MicroBlogAttachmentManager.Instance().DeleteAttachment(attachment);
            return Json(new { ok = 1 });
        }

        #endregion

        #region Smiley

        /// <summary>
        /// 表情图片选择器
        /// </summary>
        public ActionResult Control_SmileySelector(string controlName, string textAreaID)
        {
            List<EmotionCategory> categories = EmotionManager.Instance().Categories;
            if (categories == null || categories.Count == 0)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_NoSmileyCategory"));
                return new EmptyResult();
            }
            ViewData["Categories"] = categories;

            return PartialView("Controls/SmileySelector.ascx");
        }

        #endregion

        #region Follow

        /// <summary>
        /// 关注用户
        /// </summary>
        public ActionResult FollowUser(int? userID)
        {

            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Error("你没有操作权限！"));
            else if (userID == null || userID.Value <= 0)
                return new EmptyResult();

            User user = Users.GetUser(userID.Value, false);
            if (user == null)
                return new EmptyResult();
            return FollowUserInDomain(user.UserName);
        }

        /// <summary>
        /// 关注用户
        /// </summary>
        public ActionResult FollowUserInDomain(string userDomainName)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            User currentUser = Globals.GetCurrentUser();
            if (currentUser == null || domainUser == null)
                return Redirect(SiteUrls.Instance().Error404("没有操作权限或用户不存在！"));
            MicroBlogFollow follow = new MicroBlogFollow();
            follow.ItemID = domainUser.UserID;
            follow.UserID = currentUser.UserID;

            if (!MicroBlogFollows.IsFollow(currentUser.UserID, domainUser.UserID))
            {
                MicroBlogFollows.CreateFollow(follow);
            }
            else
            {
                follow = MicroBlogFollows.GetFollow(currentUser.UserID, domainUser.UserID);
                if (follow != null)
                    MicroBlogFollows.DeleteFollow(follow.FollowID);
            }
            return new EmptyResult();
        }

        /// <summary>
        /// 显示我的关注的用户列表
        /// </summary>
        public ActionResult ListFollow(string userDomainName, MicroBlogFollowSortBy sortBy, int? gid, int? pageIndex)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));
            User domainUser = GetCurrentDomainUser(userDomainName);
            if (domainUser == null)
                return Redirect(SiteUrls.Instance().Error404("用户不存在！"));

            if (sortBy == MicroBlogFollowSortBy.FollowUser)
                SetPageTitle("我的关注");
            else
                SetPageTitle("我的粉丝");

            ViewData["sortBy"] = (int)sortBy;

            return View("Pages/ListFollow.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 显示用户关注列表
        /// </summary>
        public ActionResult Control_ListFollow(string userDomainName, MicroBlogFollowSortBy sortBy, int? gid, int? pageIndex)
        {
            if (CurrentUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserNotFound());

            User domainUser = GetCurrentDomainUser(userDomainName);
            if (domainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());

            PagingDataSet<MicroBlogFollow> pds = null;
            if (sortBy == MicroBlogFollowSortBy.Follower)
            {
                pds = MicroBlogFollows.GetFollowers(domainUser.UserID, false, pageIndex ?? 1);
            }
            else
            {
                List<UserCategory> hideGroups = null;
                List<UserCategory> defaultGroups = null;
                List<UserCategory> groups = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).GetUserCategories(domainUser.UserID, true);

                if (groups != null)
                {
                    groups = groups.Where(n => n.CategoryID > 0).ToList();

                    if (groups.Count >= 4)
                    {
                        hideGroups = groups.GetRange(4, groups.Count - 4);
                        defaultGroups = groups.GetRange(0, 4);
                        if (gid.HasValue)
                        {
                            if (hideGroups != null && hideGroups.Select(n => n.CategoryID).Contains(gid.Value))
                            {
                                int index = hideGroups.FindIndex(n => n.CategoryID == gid.Value);
                                UserCategory group = defaultGroups.Last();
                                defaultGroups[defaultGroups.Count - 1] = hideGroups[index];
                                hideGroups[index] = group;

                            }
                        }
                    }
                    else
                        defaultGroups = groups;
                }

                ViewData["hideGroups"] = hideGroups;
                ViewData["defaultGroups"] = defaultGroups;
                ViewData["groups"] = groups;
                ViewData["gid"] = gid;

                pds = MicroBlogFollows.GetFollows(domainUser.UserID, gid.HasValue && gid.Value == 0 ? null : gid, false, pageIndex ?? 1);
            }
            ViewData["sortBy"] = (int)sortBy;

            return PartialView("Controls/ListFollow.ascx", pds);
        }

        /// <summary>
        /// 编辑用户分组
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_EditUserGroup(string userDomainName, int? gid)
        {
            UserCategory group = new UserCategory();
            if (gid.HasValue && gid.Value > 0)
                group = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).GetUserCategory(gid.Value, true);

            if (TempData["userGroup"] != null)
                group = TempData["userGroup"] as UserCategory;

            return PartialView("Controls/EditUserGroup.ascx", group);
        }

        /// <summary>
        /// 更新分组成功
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UpdateUserGroup(string userDomainName, int? gid)
        {
            UserCategory group = new UserCategory();
            if (gid.HasValue && gid.Value > 0)
            {
                group = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).GetUserCategory(gid.Value, true);
                if (group == null)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "分组不存在或已被删除！";
                    return RedirectToAction("Control_EditUserGroup", "MicroBlog", new RouteValueDictionary { { "userDomainName", userDomainName } });
                }
            }

            group.CategoryName = Request.Form.GetString("GroupName", string.Empty);
            TempData["userGroup"] = group;

            if (string.IsNullOrEmpty(group.CategoryName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "分组名不能为空！";
                return RedirectToAction("Control_EditUserGroup", "MicroBlog", new RouteValueDictionary { { "userDomainName", userDomainName } });
            }

            group.UserID = CurrentDomainUser.UserID;
            try
            {
                if (gid.HasValue && gid.Value > 0)
                    MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).UpdateUserCategory(group);
                else
                    MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).CreateUserCategory(group);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            }
            catch
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "创建分组失败！";
            }

            return RedirectToAction("Control_EditUserGroup", "MicroBlog", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 删除用户分组
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteUserGroup(string userDomainName, int? gid)
        {

            if (gid.HasValue && gid.Value > 0)
            {
                UserCategory group = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).GetUserCategory(gid.Value, true);

                if (group != null)
                {
                    try
                    {
                        MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).DeleteUserCategory(gid.Value, CurrentDomainUser.UserID);
                    }
                    catch { }
                }
            }
            return RedirectToAction("ListFollow", "MicroBlog", new RouteValueDictionary { { "userDomainName", userDomainName }, { "sortBy", MicroBlogFollowSortBy.FollowUser } });
        }

        /// <summary>
        /// 显示关注用户选择器
        /// </summary>
        /// <param name="controlName">控件Name</param>
        /// <param name="selectedUserIDs">初始选中值</param>
        /// <param name="selectionLimit">最大可选关注用户数量</param>
        public ActionResult Control_FollowSelector(string controlName, string selectedUserIDs, int? selectionLimit)
        {
            IList<UserIdentifier> identifiers = new List<UserIdentifier>();
            if (!string.IsNullOrEmpty(selectedUserIDs))
            {
                string[] strUserIDs = selectedUserIDs.Split(UserTagManager.SplitCharacters);
                foreach (string strUserID in strUserIDs)
                {
                    int userID = 0;
                    int.TryParse(strUserID, out userID);
                    UserIdentifier userIdentifier = Users.GetUserIdentifier(userID);
                    if (userIdentifier != null)
                        identifiers.Add(userIdentifier);
                }
            }
            return PartialView("Controls/FollowSelector.ascx", identifiers);
        }

        /// <summary>
        /// 更新用户的分组
        /// </summary>
        /// <param name="followID">关注ID</param>
        /// <param name="gid">分组ID</param>
        /// <param name="sgid">被选中分组的ID</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ContentResult SetFollowToGroup(string userDomainName, int followID, int gid)
        {
            List<UserCategory> selectedCategory = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).GetUserCategoriesOfItem(followID, true);
            List<int> tempGroupIDs = new List<int>(selectedCategory.Select(n => n.CategoryID));

            if (tempGroupIDs.Contains(gid))
            {
                tempGroupIDs.Remove(gid);
                selectedCategory.RemoveAll(n => n.CategoryID == gid);
            }
            else
            {
                tempGroupIDs.Add(gid);
                UserCategory userCategory = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).GetUserCategory(gid, true);
                selectedCategory.Add(userCategory);
            }

            MicroBlogFollows.UpdateFollowToGroup(followID, tempGroupIDs);
            MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).UpdateUserCategoriesOfItem(followID, tempGroupIDs);
            var result = selectedCategory.Select(n => n.CategoryName).Take(2);
            if (result.Count() == 0)
                return Content("未分组");
            else
                return Content(string.Join(",", result.ToArray()));
        }

        /// <summary>
        /// 批量设置关注用户分组
        /// </summary>
        public ActionResult BatchSetFollowsToGroup(string userDomainName, int? gid)
        {

            if (gid.HasValue && gid.Value > 0)
            {
                string followIDs = Request.Form["followIDs"];
                if (!string.IsNullOrEmpty(followIDs))
                {

                    string[] arrFollowID = followIDs.Split(',');
                    foreach (string strfollowID in arrFollowID)
                    {
                        int followID = 0;
                        int.TryParse(strfollowID, out followID);
                        if (followID > 0)
                        {
                            List<UserCategory> selectedCategory = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).GetUserCategoriesOfItem(followID, true);
                            List<int> tempGroupIDs = new List<int>(selectedCategory.Select(n => n.CategoryID));

                            if (tempGroupIDs.Contains(gid.Value))
                                continue;
                            else
                                tempGroupIDs.Add(gid.Value);

                            MicroBlogFollows.UpdateFollowToGroup(followID, tempGroupIDs);
                            MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).UpdateUserCategoriesOfItem(followID, tempGroupIDs);
                        }
                    }
                }
            }
            return RedirectToAction("ListFollow", new RouteValueDictionary { { "userDomainName", userDomainName }, { "sortBy", MicroBlogFollowSortBy.FollowUser }, { "gid", gid } });
        }

        /// <summary>
        /// 我的粉丝
        /// </summary>
        public ActionResult Control_TopFollowersBlock(string userDomainName, int? disPlayCount)
        {

            User domainUser = GetCurrentDomainUser(userDomainName);

            PagingDataSet<MicroBlogFollow> pds = MicroBlogFollows.GetFollowers(domainUser.UserID, false, 1);

            IList<MicroBlogFollow> follows = null;
            if (pds != null)
                follows = pds.Records;

            ViewData["disPlayCount"] = disPlayCount ?? 9;

            return PartialView("Controls/TopFollowers.ascx", follows);
        }

        /// <summary>
        /// 设置用户分组
        /// </summary>
        /// <param name="userID">当前用户ID</param>
        /// <param name="itemID">被关注用户ID</param>
        /// <returns></returns>
        public ActionResult Control_SetUserGroup(int itemID)
        {
            if (CurrentUser == null)
            {
                return new EmptyResult();
            }

            List<UserCategory> userCategories = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).GetUserCategories(CurrentUser.UserID, false);
            ViewData["selectCategories"] = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).GetUserCategoriesOfItem(itemID, false);
            return PartialView("Controls/SetUserGroup.ascx", userCategories);
        }

        /// <summary>
        ///设置用户分组 
        /// </summary>
        /// <param name="itemID">被关注用户ID</param>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SetUserToGroups(int itemID)
        {
            if (CurrentUser == null)
            {
                return new EmptyResult();
            }

            string gIDs = Request.Form["gid"];
            if (!string.IsNullOrEmpty(gIDs))
            {
                MicroBlogFollow follow = MicroBlogFollows.GetFollow(CurrentUser.UserID, itemID);
                if (follow == null)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "用户还未被关注！";
                    return RedirectToAction("Control_SetUserGroup", "MicroBlog", new RouteValueDictionary { { "itemID", itemID } });
                }

                List<UserCategory> selectedCategory = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).GetUserCategoriesOfItem(follow.FollowID, true);
                List<int> tempGroupIDs = new List<int>(selectedCategory.Select(n => n.CategoryID));

                string[] arrUserCategoryID = gIDs.Split(',');

                foreach (var strGID in arrUserCategoryID)
                {
                    int gid = 0;
                    int.TryParse(strGID, out gid);

                    if (tempGroupIDs.Contains(gid))
                        tempGroupIDs.Remove(gid);
                    else
                        tempGroupIDs.Add(gid);

                }

                MicroBlogFollows.UpdateFollowToGroup(follow.FollowID, tempGroupIDs);
                MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).UpdateUserCategoriesOfItem(follow.FollowID, tempGroupIDs);
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            return RedirectToAction("Control_SetUserGroup", "MicroBlog", new RouteValueDictionary { { "itemID", itemID } });
        }

        /// <summary>
        /// 创建用户分组
        /// </summary>
        /// <returns></returns>
        public JsonResult CreateGroup()
        {
            if (CurrentUser == null)
                return Json(new { });

            UserCategory group = new UserCategory();
            group.CategoryName = Request.Form.GetString("GroupName", string.Empty);

            if (string.IsNullOrEmpty(group.CategoryName))
            {
                return Json(new { error = true, message = "分组名为空或超出长度限制！" });
            }

            group.UserID = CurrentUser.UserID;
            try
            {
                MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).CreateUserCategory(group);
            }
            catch
            {
                return Json(new { error = true, message = "创建群组失败！" });
            }
            return Json(new { ok = true, groupName = group.CategoryName, gid = group.CategoryID });
        }

        /// <summary>
        /// 获取我关注的用户
        /// </summary>
        /// <param name="topNumber"></param>
        /// <returns></returns>
        public JsonResult GetMyFollowUsers(int? topNumber)
        {
            string term = Request.QueryString.GetString("term", string.Empty);
            if (string.IsNullOrEmpty(term) || CurrentUser == null)
                return Json(new { }, JsonRequestBehavior.AllowGet);
            term = WebUtils.UrlDecode(term);
            IList<MicroBlogFollow> follows = MicroBlogFollows.GetAllFollows(CurrentUser.UserID);

            MicroBlogFollow follow = new MicroBlogFollow();
            follow.ItemID = CurrentUser.UserID;
            follows.Add(follow);

            return Json(follows.Select<MicroBlogFollow, UserIdentifier>(n => Users.GetUserIdentifier(n.ItemID))
            .Where(
            n => (
            n != null
            && (n.CommonName.Contains(term)
            || n.NickName.Contains(term)
            || n.UserName.Contains(term))))
            .Select(n => new
            {
                userName = n.UserName,
                displayName = n.DisplayName,
                commonName = n.CommonName,
                nickName = n.NickName
            }), JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 会员全文检索
        /// </summary>
        public ActionResult SearchUsers(int? pageIndex)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));

            SetPageTitle("搜索用户");
            int type = Request.QueryString.GetInt("type", (int)MicroBlogSearchType.User);
            string keyword = Request.QueryString.GetString("Keyword", string.Empty);
            PersonUserFullTextQuery query = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(keyword))
            {
                if (type == (int)MicroBlogSearchType.User)
                    query.NameKeyword = WebUtils.UrlDecode(keyword);
                else if (type == (int)MicroBlogSearchType.UserTag)
                    query.TagName = keyword;
            }

            query.PageSize = 10;

            if (pageIndex.HasValue)
                query.PageIndex = pageIndex.Value;

            SearchResultDataSet<PersonUser> results = null;
            if (query.IsValid())
            {
                results = PersonUserSearchManager.Instance().Search(CurrentUser, query);

                for (int i = 0; i < results.Records.Count; i++)
                {
                    results.Records[i] = Users.GetUser(results.Records[i].UserID, false) as PersonUser;
                }
            }

            TempData["Keyword"] = WebUtils.UrlDecode(keyword);
            ViewData["type"] = type;
            return View("Pages/SearchUsers.aspx", "Masters/MicroBlog.master", results);
        }

        /// <summary>
        /// 获取当前用户的关注用户列表
        /// </summary>
        public JsonResult GetMyFollows(int? categoryID)
        {

            List<MicroBlogSimpleFollow> simpleFollows = MicroBlogFollows.GetAllSimpleFollows(CurrentUser != null ? CurrentUser.UserID : 0, categoryID, true);
            return Json(
           simpleFollows
           .Select(n => new
           {
               followID = n.FollowID,
               userID = n.ItemID,
               displayName = n.DisplayName,
               commonName = n.CommonName,
               nickName = n.NickName,
               userAvatarUrl = SiteUrls.Instance().UserAvatarUrl(n.UserID, AvatarSizeTypes.Small)
           }), JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 搜索用户
        /// </summary>
        public JsonResult SearchFollows()
        {
            string term = Request.QueryString.GetString("q", string.Empty);
            if (string.IsNullOrEmpty(term))
                return Json(new { }, JsonRequestBehavior.AllowGet);
            term = WebUtils.UrlDecode(term);


            List<MicroBlogSimpleFollow> simpleFollows = MicroBlogFollows.GetAllSimpleFollows(CurrentUser != null ? CurrentUser.UserID : 0, null, true);
            return Json(simpleFollows
            .Where(
            n => (
            n != null
            && (n.CommonName.Contains(term)
            || n.NickName.Contains(term)
            || n.UserName.Contains(term))))
            .Select(n => new
            {
                followID = n.FollowID,
                userID = n.ItemID,
                displayName = n.DisplayName,
                commonName = n.CommonName,
                nickName = n.NickName,
                userAvatarUrl = SiteUrls.Instance().UserAvatarUrl(n.ItemID, AvatarSizeTypes.Small)
            }), JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 获取我关注用户的类别
        /// </summary>
        public JsonResult GetMyFollowCategories()
        {
            List<UserCategory> categories = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().MicroBlogFollow()).GetUserCategories(CurrentUser.UserID, false);
            return Json(
            categories.Select(n => new
            {
                value = n.CategoryID,
                name = n.CategoryName
            }), JsonRequestBehavior.AllowGet);
        }

        #endregion

        #region User

        /// <summary>
        /// 微博达人
        /// </summary>
        public ActionResult Control_RecommendUsersBlock(int commendTypeID, int? disPlayCount)
        {
            IList<RecommendedItem> recommends = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().User(), commendTypeID, disPlayCount ?? 3);
            IList<User> users = new List<User>();
            ViewData["commendTypeID"] = commendTypeID;
            ViewData["RecommendationType"] = RecommendationTypes.GetRecommendationType(commendTypeID, false);

            foreach (RecommendedItem recommend in recommends)
            {
                users.Add(Users.GetUser(recommend.ItemID, false));
            }

            return PartialView("Controls/RecommendUsersBlock.ascx", users);
        }

        /// <summary>
        /// 可能认识的人
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_MayknowUsers(string userDomainName, int? displayCount)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            int userCount;
            List<PersonUser> users = MayKnowUsers(domainUser, displayCount ?? 9, 50, out userCount);
            ViewData["displayCount"] = displayCount;
            ViewData["userCount"] = userCount;
            return PartialView("Controls/MayknowUsers.ascx", users);
        }

        /// <summary>
        /// 用户排行页面
        /// </summary>
        /// <returns></returns>
        public ActionResult TopUsersList()
        {
            SetPageTitle("用户排行");
            return View("Pages/TopUsersList.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 用户排行控件
        /// </summary>       
        public ActionResult Control_TopUsersRank(UserDataSortBy? sortBy)
        {
            IList<MicroBlogUserData> userDatas = MicroBlogUserDatas.GetTopUserdatasSortBy(20, sortBy ?? UserDataSortBy.DateCreated);
            ViewData["sortBy"] = sortBy;
            return View("Controls/TopUsersRankBlock.ascx", userDatas);

        }

        /// <summary>
        /// 获取用户卡片
        /// </summary>
        /// <param name="userID"></param>
        /// <returns></returns>
        public ActionResult Control_MemberCard(int? userID)
        {
            if (userID == null)
                return new EmptyResult();
            MicroBlogUserData userData = MicroBlogUserDatas.GetUserData(userID ?? 0, false);
            if (userData == null)
                return new EmptyResult();
            PersonUser personUser = Users.GetUser(userID ?? 0, false) as PersonUser;
            if (personUser == null)
                return new EmptyResult();
            Random random = new Random();
            string controlID = Request.QueryString.GetString("controlID", random.Next(9999).ToString());
            ViewData["controlID"] = controlID;
            ViewData["PersonUser"] = personUser;
            return PartialView("Controls/MemberCard.ascx", userData);
        }

        #endregion

        #region UserTag

        /// <summary>
        /// 显示个人标签页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UserTags(string userDomainName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            SetPageTitle(userDomainName, GetResourceString("Title_UserTags", applicationID));

            //取用户标签
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTags(currentDomainUser.UserID, true);
            ViewData["userTags"] = userTags;

            return View("Pages/ManageUserTags.aspx", "Masters/MicroBlog.master");
        }

        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_UserSiteTags(string userDomainName, int? pageIndex)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            //取站点标签，换页
            PagingDataSet<SiteTag> siteTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetTopSiteTagsSortBy(SiteTagsSortBy.UserCount, 100, 11, pageIndex ?? 1);
            //取用户标签
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTags(currentDomainUser.UserID, true);
            siteTags.Records = siteTags.Records.Where(n => !userTags.Select(m => m.TagName).ToList().Contains(n.TagName)).ToList();
            return PartialView("Controls/UserSiteTags.ascx", siteTags);
        }

        [OutputCache(Location = OutputCacheLocation.None)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_ListMyUserSiteTags(string userDomainName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            //取用户标签
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTags(currentDomainUser.UserID, true);
            return PartialView("Controls/ListMyUserSiteTags.ascx", userTags);
        }

        /// <summary>
        /// 微博用户标签
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_ListUserSiteTags(string userDomainName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            //取用户标签
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTags(currentDomainUser.UserID, true);
            return PartialView("Controls/ListUserSiteTags.ascx", userTags);
        }

        /// <summary>

        /// 创建个人标签
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreateUserTag(string userDomainName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            //判断是否超过10个
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTags(currentDomainUser.UserID, true);
            if (userTags.Count >= 10)
                TempData["statusMessageData"] = new StatusMessageData(StatusMessageType.Hint, "你最多可以有10个标签");
            else
            {
                #region 处理标签
                string selectedTagNames = Request.Form.GetString("userTags", string.Empty).Trim();
                List<UserTag> selectedTags = null;
                if (!string.IsNullOrEmpty(selectedTagNames))
                {
                    string[] tagNames = selectedTagNames.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                    if (tagNames != null && tagNames.Length > 0)
                    {
                        selectedTags = new List<UserTag>();
                        foreach (string tagName in tagNames)
                        {
                            if (!string.IsNullOrEmpty(tagName.Trim()))
                            {
                                UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTag(tagName.Trim(), currentDomainUser.UserID, true);
                                if (tag != null)
                                {
                                    selectedTags.Add(tag);
                                }
                                else
                                {
                                    if (userTags.Count < 10)
                                    {
                                        UserTag userTag = new UserTag();
                                        userTag.UserID = currentDomainUser.UserID;
                                        userTag.TagName = HtmlUtils.TrimHtml(tagName, 8).Trim();
                                        userTag.TagName = Formatter.CleanTagName(userTag.TagName);
                                        userTag.TagName = Formatter.FormatSingleLinePlainTextForStorage(userTag.TagName);

                                        bool isSuccess = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).CreateUserTag(userTag);
                                        if (isSuccess)
                                            selectedTags.Add(userTag);
                                    }
                                    else
                                    {
                                        TempData["statusMessageData"] = new StatusMessageData(StatusMessageType.Hint, "你最多可以有10个标签");
                                        break;
                                    }
                                }
                            }
                        }
                    }
                }
                #endregion

                if (selectedTags != null && selectedTags.Count > 0)
                {
                    UserTagManager.Instance(UserTagItemKeys.Instance().Person()).UpdateUserTagsOfItem(currentDomainUser.UserID, currentDomainUser.UserID, selectedTags.Select(n => n.TagName).ToList());
                }
            }

            return RedirectToAction("Control_ListMyUserSiteTags", "MicroBlog");
        }

        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreateUserTag(string userDomainName, string tageName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            tageName = tageName.Trim();

            //判断是否超过10个
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTags(currentDomainUser.UserID, true);
            if (userTags.Count == 10)
                TempData["statusMessageData"] = new StatusMessageData(StatusMessageType.Hint, "你最多可以有10个标签");
            else
            {
                #region 处理标签
                List<UserTag> selectedTags = new List<UserTag>();
                UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTag(tageName, currentDomainUser.UserID, true);
                if (tag != null)
                {
                    selectedTags.Add(tag);
                }
                else
                {
                    UserTag userTag = new UserTag();
                    userTag.UserID = currentDomainUser.UserID;
                    //userTag.TagName = StringUtils.Trim(tageName.Trim(), 8, string.Empty);
                    userTag.TagName = HtmlUtils.TrimHtml(tageName, 8).Trim();
                    userTag.TagName = Formatter.CleanTagName(userTag.TagName);
                    userTag.TagName = Formatter.FormatSingleLinePlainTextForStorage(userTag.TagName);
                    bool isSuccess = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).CreateUserTag(userTag);
                    if (isSuccess)
                        selectedTags.Add(userTag);
                }
                #endregion

                if (selectedTags != null && selectedTags.Count > 0)
                {
                    UserTagManager.Instance(UserTagItemKeys.Instance().Person()).UpdateUserTagsOfItem(currentDomainUser.UserID, currentDomainUser.UserID, selectedTags.Select(n => n.TagName).ToList());
                }
            }
            return RedirectToAction("Control_ListMyUserSiteTags", "MicroBlog", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteUserTag(string userDomainName, string tagName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            UserTagManager.Instance(UserTagItemKeys.Instance().Person()).DeleteUserTag(tagName, currentDomainUser.UserID);

            return RedirectToAction("Control_ListMyUserSiteTags", "MicroBlog", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_CreateUserSiteTagInModal(string userDomainName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));

            return PartialView("Controls/CreateUserSiteTagInModal.ascx");
        }

        /// <summary>
        /// 创建个人标签
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreateUserTagInModal(string userDomainName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            #region 处理标签
            string selectedTagNames = Request.Form.GetString("tagName", string.Empty).Trim();
            List<UserTag> selectedTags = null;
            if (!string.IsNullOrEmpty(selectedTagNames))
            {
                string[] tagNames = selectedTagNames.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                if (tagNames != null && tagNames.Length > 0)
                {
                    selectedTags = new List<UserTag>();
                    foreach (string tagName in tagNames)
                    {
                        if (!string.IsNullOrEmpty(tagName.Trim()))
                        {
                            UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTag(tagName.Trim(), currentDomainUser.UserID, true);
                            if (tag != null)
                            {
                                selectedTags.Add(tag);
                            }
                            else
                            {
                                UserTag userTag = new UserTag();
                                userTag.UserID = currentDomainUser.UserID;
                                userTag.TagName = StringUtils.Trim(tagName, 8, string.Empty);
                                bool isSuccess = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).CreateUserTag(userTag);
                                if (isSuccess)
                                {
                                    selectedTags.Add(userTag);
                                    TempData["statusMessageData"] = new StatusMessageData(StatusMessageType.Success, "添加新标签成功");
                                }
                            }
                        }
                    }
                }
            }
            #endregion

            if (selectedTags != null && selectedTags.Count > 0)
            {
                UserTagManager.Instance(UserTagItemKeys.Instance().Person()).UpdateUserTagsOfItem(currentDomainUser.UserID, currentDomainUser.UserID, selectedTags.Select(n => n.TagName).ToList());
            }

            return RedirectToAction("Control_CreateUserSiteTagInModal", "MicroBlog", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        #endregion

        #region UserSettings

        //帐号基本信息设置
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult BasicInfoSetting(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_EditPersonUserProfile", ApplicationIDs.Instance().User()));
            User domainUser = Users.GetUser(userDomainName, true);
            PersonUser userToEdit = domainUser as PersonUser;
            if (userToEdit == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }

            return View("Pages/BasicInfo.aspx", "Masters/MicroBlog.master", userToEdit);
        }

        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UpdateSettingBasicInfo(string userDomainName)
        {
            User domainUser = Users.GetUser(userDomainName, true);
            PersonUser userToEdit = domainUser as PersonUser;
            if (userToEdit == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }
            userToEdit.CommonName = Request.Form.GetString("commonName", userToEdit.CommonName);
            if (SiteSettingsManager.GetSiteSettings().EnableNickName)
                userToEdit.NickName = Request.Form.GetString("nickName", userToEdit.NickName);
            userToEdit.Gender = (Genders)Request.Form.GetInt("gender", (int)userToEdit.Gender);
            try
            {
                userToEdit.Birthday = DateTime.Parse(Request.Form.GetString("birthday", userToEdit.Birthday.ToString()));
            }
            catch
            {
            }
            userToEdit.NowAreaCode = Request.Form.GetString("nowAreaCode", userToEdit.NowAreaCode);

            userToEdit.School = Request.Form.GetString("school", userToEdit.School);
            userToEdit.SchoolID = Request.Form.GetInt("schoolID", Convert.ToInt32(userToEdit.SchoolID));
            userToEdit.SelfAppraisal = Request.Form.GetString("selfAppraisal", userToEdit.SelfAppraisal);
            userToEdit.MsnIM = Request.Form.GetString("msnIM", userToEdit.MsnIM);
            userToEdit.QQIM = Request.Form.GetString("qqIM", userToEdit.QQIM);
            Users.UpdateUser(userToEdit);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_UpdateProfileSucceed", ApplicationIDs.Instance().User());
            return View("Pages/BasicInfo.aspx", "Masters/MicroBlog.Master", userToEdit);
        }

        /// <summary>
        /// 帐号职位信息设置
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult BussinessSetting(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Action_InputProfessionalInfo", ApplicationIDs.Instance().User()));
            User domainUser = Users.GetUser(userDomainName, true);
            PersonUser userToEdit = domainUser as PersonUser;
            if (userToEdit == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }

            return View("Pages/Bussiness.aspx", "Masters/MicroBlog.master", userToEdit);
        }

        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UpdateSettingBussiness(string userDomainName)
        {
            User domainUser = Users.GetUser(userDomainName, true);
            PersonUser userToEdit = domainUser as PersonUser;
            if (userToEdit == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }
            userToEdit.Title = Request.Form.GetString("title", userToEdit.Title);
            userToEdit.JobSort = Request.Form.GetString("jobSort", userToEdit.JobSort);
            userToEdit.CompanyName = Request.Form.GetString("companyName", userToEdit.CompanyName);
            userToEdit.CompanyTrade = Request.Form.GetInt("companyTrade", userToEdit.CompanyTrade);
            Users.UpdateUser(userToEdit);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_UpdateProfileSucceed", ApplicationIDs.Instance().User());
            return View("Pages/Bussiness.aspx", "Masters/MicroBlog.Master", userToEdit);
        }

        /// <summary>
        /// 上传头像
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UploadAvatar(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Button_UploadPersonUserAvatar", ApplicationIDs.Instance().User()));
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }
            if (currentDomainUser.HasAvatar)
                ViewData["canCropAvatar"] = true;

            ViewData["UserType"] = currentDomainUser.UserType;
            return View("Pages/MicroBlogUploadAvatar.aspx", "Masters/MicroBlog.Master", currentDomainUser);
        }
        /// <summary>
        /// 裁剪头像
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CropAvatar(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Button_UploadPersonUserAvatar", ApplicationIDs.Instance().User()));
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }
            StatusMessageData statusMessageData = null;
            if (!currentDomainUser.HasAvatar)
            {
                statusMessageData = new StatusMessageData(StatusMessageType.Error, "您还没有上传头像，无法进行裁剪!");
                TempData["StatusMessageData"] = statusMessageData;
                return RedirectToAction("UploadAvatar", new RouteValueDictionary { { "userDomainName", userDomainName } });
            }
            ViewData["UserType"] = currentDomainUser.UserType;
            return View("Pages/CutoutAvatar.aspx", "Masters/MicroBlog.Master");
        }

        /// <summary>
        /// 裁剪头像处理
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ContentResult CropAvatarAction(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Button_UploadPersonUserAvatar", ApplicationIDs.Instance().User()));
            string jsoncall = Request.QueryString.GetString("jsoncallback", string.Empty);
            System.Web.Script.Serialization.JavaScriptSerializer serializer = new System.Web.Script.Serialization.JavaScriptSerializer();

            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Content(string.Format("{0}({1})", jsoncall, serializer.Serialize(new { Status = "Error", Message = string.Format(GetResourceString("Exception_Pattern_UserNotExist", applicationID), userDomainName) })));

            }

            //StatusMessageData statusMessageData = null;

            bool autoCrop = Request.QueryString.GetBool("autoCrop", true);
            if (autoCrop)
            {
                Avatars.CropAvatar(currentDomainUser.UserID);
            }
            else
            {
                SiteSettings siteSettings = SiteSettingsManager.GetSiteSettings();
                float cropImageWidth = Request.QueryString.GetFloat("cropImageWidth", siteSettings.AvatarWidth);
                float cropImageHeight = cropImageWidth / siteSettings.AvatarWidth * siteSettings.AvatarHeight;
                float cropImageX = Request.QueryString.GetFloat("cropImageX", 0F);
                float cropImageY = Request.QueryString.GetFloat("cropImageY", 0F);

                Avatars.CropAvatar(currentDomainUser.UserID, cropImageWidth, cropImageHeight, cropImageX, cropImageY);

                //statusMessageData = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_AvatarFileCutOffSucceed", applicationID));
                //TempData["StatusMessageData"] = statusMessageData;
                //return RedirectToAction("CropAvatar");
            }
            return Content(string.Format("{0}({1})", jsoncall, serializer.Serialize(new { Status = "Success", Message = GetResourceString("Message_AvatarFileCutOffSucceed", ApplicationIDs.Instance().User()) })));
        }

        #endregion

        #region Notice

        /// <summary>
        /// 通知列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListNotices(string userDomainName, int? pageIndex)
        {
            SetPageTitle(userDomainName, GetResourceString("Action_Notice"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<InstationNoticeRecord> pds = InstationNotices.GetInstationNoticeRecords(pageIndex ?? 1, currentDomainUser.UserID);
            InstationNotices.BatchSetIsRead(currentDomainUser.UserID, pageIndex ?? 1, pds.Records);
            return View("Pages/ListNotice.aspx", "Masters/MicroBlog.master", pds);

        }

        #endregion

        #region Message

        /// <summary>
        /// 收件箱
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListMessageInBox(string userDomainName, int? pageIndex)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_MessageInBox", ApplicationIDs.Instance().User()));
            ViewData["PageIndex"] = pageIndex;
            return View("Pages/ListPrivateMessageIn.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 收件箱控件
        /// </summary>
        public ActionResult Control_ListMessageInBox(string userDomainName, int? pageIndex)
        {

            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            ViewData["NoReadCount"] = PrivateMessages.GetNoReadCount(currentDomainUser.UserID);
            PagingDataSet<PrivateMessageInboxItem> pds = PrivateMessages.GetPrivateMessagesFromInbox(currentDomainUser.UserID, null, null, pageIndex ?? 1);
            return PartialView("Controls/PrivateListMessageIn.ascx", pds);
        }

        /// <summary>
        /// 从收件箱中删除私信
        /// </summary>      
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteInBoxMessage(string userDomainName, int threadID, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PrivateMessages.DeletePrivateMessageFromInbox(currentDomainUser.UserID, threadID);
            return RedirectToAction("ListMessageInBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 发件箱
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListMessageOutBox(string userDomainName, int? pageIndex)
        {

            SetPageTitle(userDomainName, GetResourceString("Title_MessageOutBox", ApplicationIDs.Instance().User()));
            ViewData["PageIndex"] = pageIndex;
            return View("Pages/ListPrivateMessageOut.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 发件箱控件
        /// </summary>
        public ActionResult Control_ListMessageOutBox(string userDomainName, int? pageIndex)
        {

            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<PrivateMessageOutboxItem> pds = PrivateMessages.GetPrivateMessagesFromOutbox(currentDomainUser.UserID, null, pageIndex ?? 1);
            return PartialView("Controls/PrivateListMessageOut.ascx", pds);
        }

        /// <summary>
        /// 从发件箱中删除一条私信
        /// </summary>         
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteOutBoxMessage(string userDomainName, int threadID, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PrivateMessages.DeletePrivateMessageFromOutbox(CurrentDomainUser.UserID, threadID);
            return RedirectToAction("ListMessageOutBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });

        }

        /// <summary>
        /// 显示私信详细页
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ShowMessage(string userDomainName, int threadID, string returnUrl)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            SetPageTitle(userDomainName, GetResourceString("Title_ShowMessage", ApplicationIDs.Instance().User()));
            return View("Pages/ShowMessage.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 显示私信控件
        /// </summary> 
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_ShowMessage(string userDomainName, int threadID)
        {
            User currentDoaminUser = GetCurrentDomainUser(userDomainName);
            SetPageTitle(userDomainName, GetResourceString("Title_ShowMessage", ApplicationIDs.Instance().User()));
            PrivateMessage message = PrivateMessages.GetPrivateMessage(threadID);
            if (CurrentUser != null && !PrivateMessages.HasOperationPermission(message, CurrentUser))
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied(), GetResourceString("Exception_ViewMessageAccessDenied", ApplicationIDs.Instance().User()));
            }
            //设置私信为已读
            PrivateMessages.SetReadFlag(CurrentUser != null ? CurrentUser.UserID : 0, message.ThreadID, true);

            return View("Controls/ShowPrivateMessage.ascx", message);
        }

        /// <summary>
        /// 回复私信
        /// </summary>        
        public ActionResult ReplyMessage(string userDomainName, int threadID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (threadID > 0)
            {
                PrivateMessage pm = PrivateMessages.GetPrivateMessage(threadID);
                string body = Request.Form["body"];
                if (!string.IsNullOrEmpty(body))
                {
                    PrivateMessagePost post = new PrivateMessagePost();
                    post.ThreadID = threadID;
                    post.ParentPostID = pm.PostID;
                    post.UserID = CurrentUser.UserID;
                    post.Author = CurrentUser.DisplayName;
                    post.Subject = string.Format("{0}:{1}", GetResourceString("Common_Reply"), pm.Sender);
                    post.Body = Formatter.FormatPlainTextComment(body);
                    if (SPBConfig.Instance().EnableEmoticon)
                        post.Body = EmotionManager.Instance().EmoticonTransforms(post.Body);
                    PrivateMessages.ReplyMessage(post, pm);
                }
            }
            return RedirectToAction("Control_ShowMessage", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", threadID } });
        }

        /// <summary>
        /// 显示私信页面
        /// </summary>       
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult NewMessage(string userDomainName, int? toUserID)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_SendMessage", ApplicationIDs.Instance().User()));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            if (toUserID.HasValue && toUserID.Value == (int)PrivateMessageEspecialUserID.Serve)
            {
                ViewData["sendToServe"] = true;
                return View("Pages/NewMessage.aspx", "Masters/MicroBlog.master");
            }
            //给某会员发普通消息
            if (toUserID.HasValue && toUserID.Value > 0)
            {
                User user = Users.GetUser(toUserID.Value, false);

                if (user != null)
                {
                    //判断该会员是否允许当前用户向其发私信
                    if (Privacys.Validate(user.UserID, CurrentUser, PrivacyItemKeys.Instance().Action_ReceivePrivateMessage()))
                    {
                        ViewData["IsAllowed"] = true;
                        ViewData["ToUser"] = user;
                    }
                    else
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        TempData["StatusMessageContent"] = GetResourceString("Message_StopSendMessageForPrivacy", ApplicationIDs.Instance().User());
                    }
                }

                ViewData["IsMultiple"] = false;
            }
            else
            {
                if (currentDomainUser.UserType == UserTypes.CompanyUser)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_StopForCompanySendToMorePerson", ApplicationIDs.Instance().User());
                }

                ViewData["IsMultiple"] = true;
            }

            return View("Pages/NewMessage.aspx", "Masters/MicroBlog.master");

        }

        /// <summary>
        /// 发送私信
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreateMessage(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string[] receivers;
            string receiverIDs = Request.Form["receiverIDs"];
            string subject = Request.Form.GetString("subject", string.Empty);
            string body = Request.Form.GetString("body", string.Empty);
            if (string.IsNullOrEmpty(receiverIDs) || string.IsNullOrEmpty(subject) || string.IsNullOrEmpty(body))
            {
                TempData["subject"] = subject;
                TempData["body"] = body;
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                if (string.IsNullOrEmpty(receiverIDs))
                    TempData["StatusMessageContent"] = GetResourceString("Message_SendErrorForReceiverEmpty", ApplicationIDs.Instance().User());
                else if (string.IsNullOrEmpty(subject))
                    TempData["StatusMessageContent"] = GetResourceString("Message_SendErrorForSubjectEmpty", ApplicationIDs.Instance().User());
                else
                    TempData["StatusMessageContent"] = GetResourceString("Message_SendErrorForBodyEmpty", ApplicationIDs.Instance().User());
                return RedirectToAction("NewMessage", new RouteValueDictionary { { "userDomainName", userDomainName } });
            }

            //收件人集合
            receivers = receiverIDs.Split(',');
            //每个消息最多允许100个接收人
            if (receivers.Length > 100)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_SendErrorForBeyondMaxPersonCount", ApplicationIDs.Instance().User());
                return RedirectToAction("NewMessage");
            }
            //发送私信成功的接收人集合
            List<string> successReceivers = new List<string>();
            int toUserID = -1;

            if (int.TryParse(receiverIDs, out toUserID) && toUserID == (int)PrivateMessageEspecialUserID.Serve)
            {
                PrivateMessage pm = new PrivateMessage();
                pm.Subject = subject;
                pm.Body = body;
                pm.SenderUserID = CurrentUser.UserID;
                pm.Sender = CurrentUser.DisplayName;
                pm.MessageType = PrivateMessageType.Common;

                pm.ReceiverUserID = toUserID;
                pm.Receiver = "客服";

                #region 格式化处理
                pm.Subject = Formatter.FormatSingleLinePlainTextForStorage(pm.Subject);
                pm.Body = Formatter.FormatMultiLinePlainTextForStorage(pm.Body, true);

                if (SPBConfig.Instance().EnableEmoticon)
                    pm.Body = EmotionManager.Instance().EmoticonTransforms(pm.Body);
                #endregion

                PrivateMessages.SendMessage(pm);
                successReceivers.Add(pm.Receiver);
            }



            //为每个收件人建立私信
            for (int i = 0; i < receivers.Length; i++)
            {
                //查找收件人
                User user = null;

                if (int.TryParse(receivers[i], out toUserID))
                {
                    user = Users.GetUser(toUserID, false);
                }

                if (user != null)
                {
                    //判断当前空间者是否接受访问者的私信
                    if (Privacys.Validate(user.UserID, CurrentUser, PrivacyItemKeys.Instance().Action_ReceivePrivateMessage()))
                    {
                        PrivateMessage pm = new PrivateMessage();
                        pm.Subject = subject;
                        pm.Body = body;
                        pm.SenderUserID = CurrentUser.UserID;
                        pm.Sender = CurrentUser.DisplayName;
                        pm.MessageType = PrivateMessageType.Common;

                        pm.ReceiverUserID = user.UserID;
                        pm.Receiver = user.DisplayName;

                        #region 格式化处理
                        pm.Subject = Formatter.FormatSingleLinePlainTextForStorage(pm.Subject);
                        pm.Body = Formatter.FormatMultiLinePlainTextForStorage(pm.Body, true);

                        if (SPBConfig.Instance().EnableEmoticon)
                            pm.Body = EmotionManager.Instance().EmoticonTransforms(pm.Body);
                        #endregion

                        PrivateMessages.SendMessage(pm);
                        successReceivers.Add(user.DisplayName);
                        toUserID = user.UserID;
                    }
                }
            }
            //如果有成功发送的消息则显示提示信息
            if (successReceivers.Count > 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                string strReceivers = string.Empty;
                foreach (string displayName in successReceivers)
                {
                    strReceivers += "," + displayName;
                }
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_SendSuccess", ApplicationIDs.Instance().User()), strReceivers.Substring(1));
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_SendErrorForOther", ApplicationIDs.Instance().User());
            }
            return RedirectToAction("NewMessage", new RouteValueDictionary { { "userDomainName", userDomainName }, { "toUserID", toUserID } });

        }

        #endregion

        #region Login

        /// <summary>
        /// 简洁首页登陆
        /// </summary>
        /// <returns></returns>
        public ActionResult LoginInSampleAction()
        {
            string userName = Request.Form.GetString("LoginUserName", string.Empty).Trim();
            string password = Request.Form.GetString("LoginPassword", string.Empty).Trim();
            bool autoLogin = Request.Form.GetBool("AutoLogin", false);

            User userToLogin = new User();
            userToLogin.UserName = userName;
            userToLogin.Password = password;

            LoginUserStatuses loginStatus = Users.ValidUser(userToLogin.UserName, userToLogin.Password);
            //如果作为用户名登录失败，则作为Email重试
            if (loginStatus == LoginUserStatuses.InvalidCredentials)
            {
                User userByEmail = Users.FindUserByEmail(userName);
                if (userByEmail != null)
                {
                    userToLogin.UserName = userByEmail.UserName;
                    loginStatus = Users.ValidUser(userToLogin.UserName, userToLogin.Password);
                }
            }

            //是否允许未批准用户登录
            bool enablePendingUsersToLogin = SiteSettingsManager.GetSiteSettings().EnablePendingUsersToLogin;

            if (loginStatus == LoginUserStatuses.Success || (enablePendingUsersToLogin && loginStatus == LoginUserStatuses.AccountPending))
            {
                if (autoLogin)
                {
                    FormsAuthentication.SetAuthCookie(userToLogin.UserName, true);
                }
                else
                {
                    FormsAuthentication.SetAuthCookie(userToLogin.UserName, false);
                }
                return RedirectToAction("MicroBlogHome");
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                if (loginStatus == LoginUserStatuses.InvalidCredentials)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UserLoginFailedForUserNameOrPassword", ApplicationIDs.Instance().User());
                }
                else if (!enablePendingUsersToLogin && loginStatus == LoginUserStatuses.AccountPending)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UserLoginFailedForAccountPending", ApplicationIDs.Instance().User());
                }
                else if (loginStatus == LoginUserStatuses.AccountDisapproved)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UserLoginFailedForAccountDisapproved", ApplicationIDs.Instance().User());
                }
                else if (loginStatus == LoginUserStatuses.UnknownError)
                {
                    throw new SPBException(ExceptionTypes.Instance().UserUnknownLoginError());
                }
                else if (loginStatus == LoginUserStatuses.AccountBanned)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UserLoginFailedForAccountBanned", ApplicationIDs.Instance().User());
                }

                return RedirectToAction("login", "ChannelUser", new RouteValueDictionary { { "returnUrl", SPBUrlHelper.Action("MicroBlogHome", "MicroBlog") } });
            }
        }

        /// <summary>
        /// 注销
        /// </summary>
        //TODO: 暂时方法
        public ActionResult Logout()
        {
            FormsAuthentication.SignOut();
            ///移除在线用户
            if (CurrentUser != null)
                UserOnlines.DeleteOnlineUser(CurrentUser.UserID);

            return RedirectToAction("MicroBlogHome");
        }





        #endregion

        #region help Method

        /// <summary>
        /// 是否具有管理空间权限
        /// </summary>
        protected bool HasManagePermission(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            return (CurrentUser != null && (currentDomainUser.UserID == CurrentUser.UserID || CurrentUser.IsContentAdministrator || CurrentUser.IsAdministrator));
        }

        /// <summary>
        ///  取主题回复的数目
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        protected ActionResult microBlogThreadCommentCount(int threadID)
        {
            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, false);
            int count = 0;
            if (thread != null)
                count = thread.ReplyCount;
            return Content(count.ToString());
        }

        /// <summary>
        /// 获取我可能认识的人
        /// </summary>
        private List<PersonUser> MayKnowUsers(User domainUser, int displayCount, int pageSize, out int userCount)
        {
            PersonUser personUser = domainUser as PersonUser;
            List<PersonUser> users = new List<PersonUser>();

            #region 同事

            SearchResultDataSet<PersonUser> resultsCompany = null;
            PersonUserFullTextQuery queryCompany = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(personUser.CompanyName))
            {
                queryCompany.CompanyName = personUser.CompanyName;
                queryCompany.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                queryCompany.SortOrder = SortOrder.Descending;
                resultsCompany = PersonUserSearchManager.Instance().Search(CurrentUser, queryCompany);
                queryCompany.PageSize = pageSize;
            }
            if (resultsCompany != null)
                users.AddRange(resultsCompany.Records.Where(n => !users.Select(m => m.UserID).ToList().Contains(n.UserID)));

            #endregion

            #region 二层朋友

            SearchResultDataSet<PersonUser> resultsSecondFriends = null;
            PersonUserFullTextQuery querySecondFriends = new PersonUserFullTextQuery();
            querySecondFriends.SearchRange = 2;
            querySecondFriends.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
            querySecondFriends.SortOrder = SortOrder.Descending;
            querySecondFriends.PageSize = pageSize;

            resultsSecondFriends = PersonUserSearchManager.Instance().Search(CurrentUser, querySecondFriends);

            if (resultsSecondFriends != null)
                users.AddRange(resultsSecondFriends.Records.Where(n => !users.Select(m => m.UserID).ToList().Contains(n.UserID)));

            #endregion

            #region 同行
            SearchResultDataSet<PersonUser> resultsCompanyTrade = null;
            PersonUserFullTextQuery queryCompanyTrade = new PersonUserFullTextQuery();
            if (personUser.CompanyTrade > 0)
            {
                queryCompanyTrade.CompanyTrade = personUser.CompanyTrade;
                queryCompanyTrade.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                queryCompanyTrade.SortOrder = SortOrder.Descending;
                queryCompany.PageSize = pageSize;
                resultsCompanyTrade = PersonUserSearchManager.Instance().Search(CurrentUser, queryCompanyTrade);

                if (resultsCompanyTrade != null)
                    users.AddRange(resultsCompanyTrade.Records.Where(n => !users.Select(m => m.UserID).ToList().Contains(n.UserID)));
            }
            #endregion

            #region 校友
            SearchResultDataSet<PersonUser> resultsSchool = null;
            PersonUserFullTextQuery querySchool = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(personUser.School))
            {
                querySchool.School = personUser.School;
                querySchool.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                querySchool.SortOrder = SortOrder.Descending;
                querySchool.PageSize = pageSize;
                resultsSchool = PersonUserSearchManager.Instance().Search(CurrentUser, querySchool);
                if (resultsSchool != null)
                    users.AddRange(resultsSchool.Records.Where(n => !users.Select(m => m.UserID).ToList().Contains(n.UserID)));
            }
            #endregion

            #region 同城
            SearchResultDataSet<PersonUser> resultsCity = null;
            PersonUserFullTextQuery queryCity = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(personUser.NowAreaCode))
            {
                queryCity.NowAreaCode = personUser.NowAreaCode;
                queryCity.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                queryCity.SortOrder = SortOrder.Descending;
                queryCity.PageSize = pageSize;
                resultsCity = PersonUserSearchManager.Instance().Search(CurrentUser, queryCity);

                if (resultsCity != null)
                    users.AddRange(resultsCity.Records.Where(n => !users.Select(m => m.UserID).ToList().Contains(n.UserID)));
            }
            #endregion

            #region 同乡
            SearchResultDataSet<PersonUser> resultsHome = null;
            PersonUserFullTextQuery queryHome = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(personUser.HomeAreaCode))
            {
                queryHome.HomeAreaCode = personUser.HomeAreaCode;
                queryHome.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                queryHome.SortOrder = SortOrder.Descending;
                queryHome.PageSize = pageSize;
                resultsHome = PersonUserSearchManager.Instance().Search(CurrentUser, queryHome);

                if (resultsHome != null)
                    users.AddRange(resultsHome.Records.Where(n => !users.Select(m => m.UserID).ToList().Contains(n.UserID)));

            }
            #endregion

            users = users.Where(n => n.UserID != domainUser.UserID && !MicroBlogFollows.IsFollow(domainUser.UserID, n.UserID)).ToList();

            userCount = users != null ? users.Count : 0;
            if (users != null && users.Count > displayCount)
            {
                Random random = new Random();
                List<PersonUser> displayUsers = new List<PersonUser>();

                int count = users.Count;
                while (displayUsers.Count < displayCount)
                {
                    PersonUser user = users[random.Next(count)];
                    if (!displayUsers.Select(n => n.UserID).Contains(user.UserID))
                    {
                        displayUsers.Add(user);
                    }
                }
                return displayUsers;
            }

            return users;

        }

        private IList<MicroBlogThread> TopicMicroBlogs(string keyword, int pageSize, int pageIndex)
        {
            SearchResultDataSet<MicroBlogThread> threads = MicroBlogSearchManager.Instance().Search(keyword, true, pageSize, pageIndex);
            return threads.Records;
        }

        #endregion

    }

    /// <summary>
    /// 微博导航
    /// </summary>
    public enum MicroBlogDomainSubMenu
    {
        /// <summary>
        /// 我的首页
        /// </summary>
        MyMicroBlogHome,

        /// <summary>
        /// 我的微博
        /// </summary>
        MyMicroBlogs,

        /// <summary>
        ///随便看看 
        /// </summary>
        News,

        /// <summary>
        ///提到我 
        /// </summary>
        Atme,

        /// <summary>
        /// 我朋友的微博
        /// </summary>
        MyFriends,

        /// <summary>
        /// 话题
        /// </summary>
        Topic,

        /// <summary>
        /// 我的评论
        /// </summary>
        MyComment,

        /// <summary>
        /// 模板
        /// </summary>
        CustomDress,

        /// <summary>
        /// 我的收藏
        /// </summary>
        MyFavorite,

        /// <summary>
        /// 我的大屏幕
        /// </summary>
        MyScreen

    }

    /// <summary>
    /// 关注列表分类
    /// </summary>
    public enum MicroBlogFollowSortBy
    {
        /// <summary>
        /// 我的关注
        /// </summary>
        FollowUser = 211,

        /// <summary>
        /// 我的粉丝
        /// </summary>
        Follower = 212

    }

    /// <summary>
    /// 
    /// </summary>
    public enum CommentType
    {
        /// <summary>
        /// 微博列表中显示的评论
        /// </summary>
        TopCommnet,

        /// <summary>
        /// 详细显示页显示的评论
        /// </summary>
        ListComment
    }

    public enum MicroBlogBrowseSubMenu
    {
        /// <summary>
        /// 随便看看
        /// </summary>
        News,

        /// <summary>
        /// 话题排行
        /// </summary>
        TopicRanks,

        /// <summary>
        /// 用户排行
        /// </summary>
        UserRanks
    }

    public enum MicroBlogSearchType
    {
        /// <summary>
        ///搜索微博 
        /// </summary>
        MicroBlog,

        /// <summary>
        /// 搜索用户
        /// </summary>
        User,

        /// <summary>
        /// 搜索用户标签
        /// </summary>
        UserTag
    }
}


