﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using SpaceBuilder.Utils;
using SpaceBuilder.Common;
using SpaceBuilder.Web;
using System.Web.Routing;
using System.Text.RegularExpressions;
using SpaceBuilder.LuceneSearch;
using Lucene.Net.Search;
using System.Web.Security;

namespace SpaceBuilder.MicroBlog.Controllers
{
    [HttpCompress]
    public partial class MobileMicroBlogController : MicroBlogDomainControllerBase
    {
        #region View

        /// <summary>
        /// 登陆
        /// </summary>
        /// <returns></returns>
        public ActionResult Login()
        {

            if (CurrentUser == null)
                return View("Pages/Login.aspx", "Masters/MicroBlog.Master");
            else
                return RedirectToAction("MyMicroBlogHome", new RouteValueDictionary { { "userDomainName", CurrentUser.UserName } });
        }

        /// <summary>
        /// 忘记密码
        /// </summary>
        /// <returns></returns>
        public ActionResult ForgetPassword()
        {
            return View("Pages/ForgetPassword.aspx", "Masters/MicroBlog.Master");
        }

        /// <summary>
        /// 我的微博首页
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public ActionResult MyMicroBlogHome(string userDomainName, int? pageIndex)
        {
            if (CurrentUser == null)
                return RedirectToAction("Login");

            if (CurrentDomainUser.UserID != CurrentUser.UserID)
                return RedirectToAction("ListMyMicroBlogThreads", new RouteValueDictionary { { "userDomainName", userDomainName } });
            PagingDataSet<MicroBlogThread> pds = MicroBlogThreads.GetThreadsOfTimeLine(CurrentDomainUser.UserID, null, null, pageIndex ?? 1, true);

            return View("Pages/MyMicroBlogHome.aspx", "Masters/MicroBlog.master", pds);
        }

        /// <summary>
        /// 上传照片
        /// </summary>
        /// <returns></returns>
        public ActionResult UploadPhoto()
        {
            return View("Pages/UploadPhoto.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 微博列表
        /// </summary>
        /// <returns></returns>
        public ActionResult ListMicroBlogThreads(MicroBlogThreadSortBy? sortBy, int? pageIndex)
        {
            PagingDataSet<MicroBlogThread> pds = MicroBlogThreads.GetPagingThreadsSortBy(sortBy ?? MicroBlogThreadSortBy.DateCreated, 15, pageIndex ?? 1);
            if (!sortBy.HasValue)
                sortBy = MicroBlogThreadSortBy.DateCreated;

            ViewData["sortBy"] = (int)sortBy;

            return View("Pages/ListMicroBlogThreads.aspx", "Masters/MicroBlog.master", pds);
        }

        /// <summary>
        /// 话题下微博列表
        /// </summary>
        /// <returns></returns>
        public ActionResult ListTopicMicroBlogThreads(string keyWord, int? pageIndex)
        {
            SearchResultDataSet<MicroBlogThread> results = null;

            if (!string.IsNullOrEmpty(keyWord))
                results = MicroBlogSearchManager.Instance().Search(WebUtils.UrlDecode(keyWord), true, 15, pageIndex ?? 1);
            ViewData["keyWord"] = WebUtils.UrlDecode(keyWord);

            return View("Pages/ListTopicMicroBlogThreads.aspx", "Masters/MicroBlog.master", results);
        }

        /// <summary>
        /// 微博详细页
        /// </summary>
        /// <returns></returns>
        public ActionResult ShowMicroBlogThread(string userDomainName, int threadID, int? pageIndex)
        {
            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, HasManagePermission(userDomainName));
            if (thread == null)
            {
                ViewData["StatusMessageContent"] = "微博不存在！";
                ViewData["returnUrl"] = Request.UrlReferrer.AbsoluteUri;
                return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
            }
            #region 处理审核

            int auditingStatusValue = (int)Applications.GetApplication(ApplicationIDs.Instance().MicroBlog()).AuditingStatusForPublicDisplay;
            if ((int)thread.AuditingStatus < auditingStatusValue)
            {
                if (thread.AuditingStatus == AuditingStatuses.Fail)
                    ViewData["StatusMessageContent"] = "该文章尚未审核通过！";
                else
                    ViewData["StatusMessageContent"] = "该文章正在审核当中";
                ViewData["returnUrl"] = Request.UrlReferrer.AbsoluteUri;
                return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
            }

            #endregion

            ViewData["MicroBlogThread"] = thread;
            PagingDataSet<MicroBlogComment> pds = MicroBlogComments.GetCommentsOfThread(threadID, pageIndex ?? 1, true);
            return View("Pages/ShowMicroBlogThread.aspx", "Masters/MicroBlog.master", pds);
        }

        /// <summary>
        /// 转发微博
        /// </summary>
        /// <returns></returns>
        public ActionResult ForwardMicroBlogThread(int threadID, string returnUrl)
        {
            if (string.IsNullOrEmpty(returnUrl))
                ViewData["returnUrl"] = Request.UrlReferrer.AbsoluteUri;
            else
                ViewData["returnUrl"] = returnUrl;

            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, false);
            return View("Pages/ForwardMicroBlogThread.aspx", "Masters/MicroBlog.Master", thread);
        }

        /// <summary>
        /// 我的粉丝
        /// </summary>
        /// <returns></returns>
        public ActionResult MyFuns()
        {
            return View("Pages/MyFuns.aspx", "Masters/MicroBlog.Master");
        }

        /// <summary>
        /// 我的关注/我的粉丝
        /// </summary>
        /// <returns></returns>
        public ActionResult ListFollows(string userDomainName, MicroBlogFollowSortBy? sortBy, int? pageIndex)
        {

            ViewData["sortBy"] = (int)(sortBy ?? MicroBlogFollowSortBy.FollowUser);

            PagingDataSet<MicroBlogFollow> pds = null;
            if (sortBy == MicroBlogFollowSortBy.FollowUser)
                pds = MicroBlogFollows.GetFollows(CurrentDomainUser.UserID, null, false, pageIndex ?? 1);
            else
                pds = MicroBlogFollows.GetFollowers(CurrentDomainUser.UserID, false, pageIndex ?? 1);

            return View("Pages/ListFollows.aspx", "Masters/MicroBlog.Master", pds);
        }

        /// <summary>
        /// 我的微博
        /// </summary>
        /// <returns></returns>

        public ActionResult MyMicroBlog(string userDomainName, int? pageIndex)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);

            PagingDataSet<MicroBlogThread> pds = null;
            if (HasManagePermission(userDomainName))
                pds = MicroBlogThreads.GetMyThreads(domainUser.UserID, null, pageIndex ?? 1);
            else
                pds = MicroBlogThreads.GetThreads(domainUser.UserID, null, pageIndex ?? 1, false);

            return View("Pages/MyMicroBlog.aspx", "Masters/MicroBlog.Master", pds);
        }

        /// <summary>
        /// 注册用户
        /// </summary>
        public ActionResult Register(string token)
        {
            SetPageTitle(GetResourceString("Action_Register"));

            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.RegistrationMode == RegistrationModes.DisableAll)
            {
                return Redirect(SiteUrls.Instance().ShowExceptionMessage(ExceptionTypes.Instance().RegisterDenied()));
            }
            if (settings.RegistrationMode == RegistrationModes.OnlyByInvitation)
            {
                if (string.IsNullOrEmpty(token))
                    token = TempData.GetString("Token", string.Empty);

                int inviteUserID = 0;
                int.TryParse(EncryptManager.DecryptTokenForInviteFriend(token), out inviteUserID);
                User user = Users.GetUser(inviteUserID, false);
                if (user == null)
                    return Redirect(SiteUrls.Instance().ShowExceptionMessage(ExceptionTypes.Instance().RegisterDenied()));
            }

            return View("Pages/Register.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 话题列表
        /// </summary>
        /// <returns></returns>
        public ActionResult ListTopics(int? disPlayCount)
        {
            IList<MicroBlogTopic> microBlogTopics = MicroBlogTopics.GetTopTopicSortBy(TopicSortBy.ItemCount, disPlayCount ?? 10);

            return View("Pages/ListTopics.aspx", "Masters/MicroBlog.Master", microBlogTopics);
        }

        /// <summary>
        /// 提到我的微博
        /// </summary>
        /// <returns></returns>
        public ActionResult AtMe(string userDomainName, int? pageIndex)
        {
            PagingDataSet<MicroBlogThread> pds = MicroBlogThreads.GetReferMeThreads(CurrentDomainUser.UserID, pageIndex ?? 1);
            return View("Pages/AtMe.aspx", "Masters/MicroBlog.master", pds);
        }

        /// <summary>
        /// 推荐用户
        /// </summary>
        /// <returns></returns>
        public ActionResult RecommendUser(int commendTypeID, int? disPlayCount)
        {
            IList<RecommendedItem> recommends = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().User(), commendTypeID, disPlayCount ?? 9);
            IList<User> users = new List<User>();
            ViewData["commendTypeID"] = commendTypeID;

            foreach (RecommendedItem recommend in recommends)
            {
                users.Add(Users.GetUser(recommend.ItemID, false));
            }

            return View("Pages/RecommendUser.aspx", "Masters/MicroBlog.Master", users);
        }


        public ActionResult DeleteItem(int? threaID, int? commentID)
        {
            if (threaID.HasValue && threaID.Value > 0)
            {
                ViewData["ItemID"] = threaID;
                ViewData["Message"] = "确定删除当前微博么？";
                ViewData["deleteUrl"] = SPBUrlHelper.Action("DeleteMicroBlogThread", "MobileMicroBlog", new RouteValueDictionary { { "threadID", threaID } });
            }
            if (commentID.HasValue && commentID.Value > 0)
            {
                ViewData["ItemID"] = commentID;
                ViewData["Message"] = "确定删除当前评论么？";
                ViewData["deleteUrl"] = SPBUrlHelper.Action("DeleteMicroBlogComment", "MobileMicroBlog", new RouteValueDictionary { { "commentID", commentID } });
            }
            ViewData["returnUrl"] = Request.UrlReferrer.AbsoluteUri;
            return View("Pages/DeleteItem.aspx", "Masters/MicroBlog.master");
        }

        /// <summary>
        /// 上传图片
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public ActionResult UploadPhoto(string userDomainName, string returnUrl)
        {
            return View("Pages/UploadPhoto.aspx", "Masters/MicroBlog.master");
        }

        #endregion

        #region Action

        /// <summary>
        /// 删除微博主题
        /// </summary>
        public ActionResult DeleteMicroBlogThread(int threadID)
        {
            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, true);

            if (CurrentUser != null && (CurrentUser.UserID == thread.UserID || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
            {
                if (thread != null)
                {
                    if (CurrentUser.IsContentAdministrator || CurrentUser.UserID == thread.OwnerUserID)
                    {
                        MicroBlogThreads.DeleteThread(thread);
                    }
                }
            }

            string returnUrl = Request.QueryString.GetString("returnUrl", string.Empty);
            return Redirect(WebUtils.UrlDecode(returnUrl));
        }


        /// <summary>
        /// 登陆
        /// </summary>
        /// <returns></returns>
        public ActionResult LoginInSampleAction()
        {
            string userName = Request.Form.GetString("LoginUserName", string.Empty).Trim();
            string password = Request.Form.GetString("LoginPassword", string.Empty).Trim();

            bool autoLogin = Request.Form.GetBool("AutoLogin", false);

            if (string.IsNullOrEmpty(userName))
            {
                ViewData["StatusMessageContent"] = "用户名不能为空！";
                return View("Pages/Message.aspx", "Masters/MicroBlog.master");
            }

            if (string.IsNullOrEmpty("password"))
            {
                ViewData["StatusMessageContent"] = "密码不能为空！";
                return View("Pages/Message.aspx", "Masters/MicroBlog.master");
            }

            User userToLogin = new User();
            userToLogin.UserName = userName;
            userToLogin.Password = password;

            LoginUserStatuses loginStatus = Users.ValidUser(userToLogin.UserName, userToLogin.Password);
            //如果作为用户名登录失败，则作为Email重试
            if (loginStatus == LoginUserStatuses.InvalidCredentials)
            {
                User userByEmail = Users.FindUserByEmail(userName);
                if (userByEmail != null)
                {
                    userToLogin.UserName = userByEmail.UserName;
                    loginStatus = Users.ValidUser(userToLogin.UserName, userToLogin.Password);
                }
            }

            //是否允许未批准用户登录
            bool enablePendingUsersToLogin = SiteSettingsManager.GetSiteSettings().EnablePendingUsersToLogin;
            StatusMessageType statusMessagetype = StatusMessageType.Success;

            if (loginStatus == LoginUserStatuses.Success || (enablePendingUsersToLogin && loginStatus == LoginUserStatuses.AccountPending))
            {
                if (autoLogin)
                    FormsAuthentication.SetAuthCookie(userToLogin.UserName, true);
                else
                    FormsAuthentication.SetAuthCookie(userToLogin.UserName, false);
            }
            else
            {
                if (loginStatus == LoginUserStatuses.InvalidCredentials)
                {
                    ViewData["StatusMessageContent"] = "用户名或密码出错";
                    statusMessagetype = StatusMessageType.Error;
                }
                else if (!enablePendingUsersToLogin && loginStatus == LoginUserStatuses.AccountPending)
                {
                    ViewData["StatusMessageContent"] = "超级管理员正在审核您的帐号，审核通过后您将收到邮件通知";
                    statusMessagetype = StatusMessageType.Error;

                }
                else if (loginStatus == LoginUserStatuses.AccountDisapproved)
                {
                    ViewData["StatusMessageContent"] = "您的帐号未通过审核，不能进行登录";
                    statusMessagetype = StatusMessageType.Error;

                }
                else if (loginStatus == LoginUserStatuses.UnknownError)
                {
                    ViewData["StatusMessageContent"] = "未知错误"; statusMessagetype = StatusMessageType.Error;

                }
                else if (loginStatus == LoginUserStatuses.AccountBanned)
                {
                    ViewData["StatusMessageContent"] = "您的帐号处于封禁状态，不能登录系统";
                    statusMessagetype = StatusMessageType.Error;

                }
            }

            if (statusMessagetype == StatusMessageType.Success)
            {
                ViewData["returnUrl"] = SPBUrlHelper.Action("MyMicroBlogHome", "MobileMicroBlog", new RouteValueDictionary { { "userDomainName", userToLogin.UserName } });
                ViewData["StatusMessageType"] = statusMessagetype;
                ViewData["StatusMessageContent"] = "登录成功";
            }
            return View("Pages/Message.aspx", "Masters/MicroBlog.master");

        }

        /// <summary>
        /// 处理获取密码
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult RecoverPassword()
        {
            string email = Request.Form.GetString("email", string.Empty).Trim();

            if (string.IsNullOrEmpty(email))
            {
                ViewData["StatusMessageContent"] = "请输入电子邮箱";
                return View("Pages/Message.aspx", "Masters/MicroBlog.master");
            }

            User user = Users.FindUserByEmail(email.Trim());
            if (user == null)
            {
                ViewData["StatusMessageContent"] = "您输入的Email不存在，请检查是否有拼写错误";
                return View("Pages/Message.aspx", "Masters/MicroBlog.master");
            }

            if (user.SendRecoverPasswordSecureLink())
            {
                ViewData["StatusMessageContent"] = "重新获取密码的信息已经发生到您的电子信箱";
            }
            return View("Pages/Message.aspx", "Masters/MicroBlog.master");
        }

        public ActionResult ForwardMicroBlog()
        {
            if (CurrentUser == null)
            {
                ViewData["StatusMessageContent"] = "微博不存在！";
                ViewData["StatusMessageType"] = StatusMessageType.Error;
                return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
            }
            MicroBlogThread thread = new MicroBlogThread();

            ViewData["returnUrl"] = Request.Form.GetString("returnUrl", string.Empty);
            int threadID = Request.Form.GetInt("threadID", 0);
            MicroBlogThread forwardThread = MicroBlogThreads.GetThread(threadID, false);
            string returnUrl = Request.Form.GetString("returnUrl", string.Empty);
            if (forwardThread == null)
            {
                ViewData["StatusMessageContent"] = "微博不存在！";
                ViewData["StatusMessageType"] = StatusMessageType.Error;
                return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
            }

            if (forwardThread.UserID == CurrentUser.UserID)
            {
                ViewData["StatusMessageContent"] = "不能转发自己的微博！";
                ViewData["StatusMessageType"] = StatusMessageType.Error;
                return View("Pages/Message.aspx", "Masters/MicroBlog.Master");

            }

            if (MicroBlogThreads.IsForwarded(CurrentUser.UserID, forwardThread.ThreadID))
            {
                ViewData["StatusMessageContent"] = "同一微博只能转发一次！";
                ViewData["StatusMessageType"] = StatusMessageType.Error;
                return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
            }

            thread.Body = Request.Form.GetString("microBlogBody", string.Empty);

            if (string.IsNullOrEmpty(thread.Body))
            {
                ViewData["StatusMessageContent"] = "内容不能为空";
                ViewData["StatusMessageType"] = StatusMessageType.Error;
                return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
            }
            if (forwardThread.OriginalThreadID <= 0)
            {
                thread.OriginalThreadID = forwardThread.ThreadID;
            }
            else
            {
                thread.OriginalThreadID = forwardThread.OriginalThreadID;
            }

            thread.ForwardedThreadID = forwardThread.ThreadID;
            thread.OwnerUserID = CurrentUser.UserID;
            thread.Author = CurrentUser.DisplayName;
            thread.PostMode = PostModes.Mobile;

            MicroBlogThreads.CreateThread(thread);
            if (thread.ThreadID > 0)
            {
                TempData["StatusMessageContent"] = "转发成功！";
                ViewData["StatusMessageType"] = StatusMessageType.Success;

                MicroBlogComment comment = new MicroBlogComment();

                if (Request.Form.GetBool("Comment", false))
                {
                    comment.Author = CurrentUser.DisplayName;
                    comment.UserID = CurrentUser.UserID;
                    comment.OwnerUserID = forwardThread.UserID;
                    comment.ThreadID = forwardThread.ThreadID;
                    comment.Body = thread.Body;
                    MicroBlogComments.CreateComment(comment);
                }

                if (Request.Form.GetBool("OriginalComment", false) && forwardThread.OriginalThreadID > 0)
                {
                    comment.Author = CurrentUser.DisplayName;
                    comment.UserID = CurrentUser.UserID;
                    comment.OwnerUserID = forwardThread.GetOriginalThread.UserID;
                    comment.ThreadID = forwardThread.OriginalThreadID;
                    comment.Body = thread.Body;
                    MicroBlogComments.CreateComment(comment);
                }
                return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
            }
            else
            {
                TempData["StatusMessageType"] = StatusMessageType.Error;
                TempData["StatusMessageContent"] = "转发失败！";
            }
            return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
        }

        /// <summary>
        /// 发布微博
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateMicroBlogThread(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string body = Request.Form["microBlogBody"];

            if (!string.IsNullOrEmpty(Request.Form.GetString("uploadImage", string.Empty)))
            {
                ViewData["returnUrl"] = Request.UrlReferrer.AbsoluteUri;
                return View("Pages/UploadPhoto.aspx", "Masters/MicroBlog.master");
            }
            string returnUrl = Request.Form.GetString("returnUrl", string.Empty);
            if (body.Length > 140)
            {
                ViewData["StatusMessageContent"] = "内容超过140字！";
                ViewData["StatusMessageType"] = StatusMessageType.Error;
                TempData["Body"] = body;
                ViewData["returnUrl"] = returnUrl;
                return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
            }

            MicroBlogThread thread = new MicroBlogThread();
            thread.Body = body;
            Regex rg = new Regex("(?<=(\\@))[a-zA-Z\u4e00-\u9fa5]*", RegexOptions.Multiline | RegexOptions.Singleline);
            string userName = rg.Match(body).Value;
            if (!string.IsNullOrEmpty(userName))
            {
                if (thread.Body.Length - 1 == userName.Length)
                {
                    User user = Users.GetUser(userName, false);
                    if (user != null && user.UserID > 0)
                        return RedirectToAction("MyMicroBlog", new RouteValueDictionary { { "userDomainName", user.UserName } });
                }
            }

            thread.OwnerUserID = currentDomainUser.UserID;
            thread.Author = CurrentUser.DisplayName;
            thread.PostMode = PostModes.Mobile;
            thread.PostSource = WebUtils.IPAddress;
            thread.UserID = CurrentUser.UserID;
            thread.HasPhoto = true;
            MicroBlogThreads.CreateThread(thread);

            System.Web.HttpPostedFileBase hpf = null;

            if (Request.Files.Count > 0 && thread.ThreadID > 0)
            {
                hpf = Request.Files[0] as System.Web.HttpPostedFileBase;

                if (hpf.ContentLength == 0)
                {
                    ViewData["StatusMessageContent"] = "请选择要上传的附件！";
                    ViewData["StatusMessageType"] = StatusMessageType.Error;
                    ViewData["returnUrl"] = returnUrl;
                    return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
                }
                else
                {
                    if (hpf.ContentLength > MicroBlogConfiguration.Instance().MaxAttachmentSize * 1024)
                    {
                        ViewData["StatusMessageContent"] = string.Format("上传文件大小不能超过{0}KB", MicroBlogConfiguration.Instance().MaxAttachmentSize); ;
                        return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
                    }
                    if (!string.IsNullOrEmpty(hpf.FileName))
                    {
                        string fileTypeString = hpf.FileName.Substring(hpf.FileName.LastIndexOf(".") + 1);
                        string uploadPhotoTypes = MicroBlogConfiguration.Instance().MicroBlogAttachmentSettings.Extensions;
                        StringBuilder newUploadPhotoTypes = new StringBuilder();
                        string[] types = uploadPhotoTypes.Split(',');
                        bool isPermitted = false;
                        foreach (string type in types)
                        {
                            if (!string.IsNullOrEmpty(type))
                            {
                                if (string.Compare(type, fileTypeString, true) == 0)
                                {
                                    isPermitted = true;
                                }
                            }
                        }
                        if (!isPermitted)
                        {
                            ViewData["StatusMessageContent"] = "只支持gif,jpg,jpeg,png,bmp类型图片";
                            return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
                        }
                    }
                    string newfileName = hpf.FileName.Substring(hpf.FileName.LastIndexOf("\\") + 1);

                    Attachment attachment = new Attachment(hpf);
                    attachment.ContentSize = hpf.ContentLength;
                    attachment.ContentType = hpf.ContentType;
                    attachment.DateCreated = DateTime.Now;
                    attachment.UserID = CurrentUser.UserID;
                    attachment.FriendlyFileName = newfileName;
                    attachment.AssociateID = thread.ThreadID;

                    MicroBlogAttachmentManager.Instance().CreateAttachment(attachment, hpf.InputStream);
                }
            }
            if (string.IsNullOrEmpty(returnUrl))
                return Redirect(Request.UrlReferrer.AbsoluteUri);
            else
                return Redirect(returnUrl);
        }

        /// <summary>
        /// 回复微博
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateMicroBlogComment(string userDomainName, int threadID)
        {
            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, false);

            if (thread != null)
            {
                string body = Request.Form["microBlogCommentBody"];
                if (string.IsNullOrEmpty(body) || body.Length > 140)
                {
                    ViewData["StatusMessageContent"] = "没有输入内容或内容超过140字";
                    TempData["microBlogCommentBody"] = body;
                    return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
                }
                MicroBlogComment comment = new MicroBlogComment();
                comment.ThreadID = threadID;
                comment.Body = body;
                comment.OwnerUserID = thread.OwnerUserID;
                comment.PostMode = PostModes.Mobile;
                comment.PostSource = WebUtils.IPAddress;
                comment.UserID = CurrentUser.UserID;
                comment.Author = CurrentUser.DisplayName;
                MicroBlogComments.CreateComment(comment);

                if ((int)comment.AuditingStatus < (int)SiteSettingsManager.GetSiteSettings().AuditingStatusesForPublicDisplay)
                {
                    ViewData["StatusMessageContent"] = "发布评论成功，需要通过审核后才能显示";
                    ViewData["StatusMessageType"] = StatusMessageType.Error;
                    return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
                }
            }
            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }


        /// <summary>
        /// 删除微博回复
        /// </summary>
        public ActionResult DeleteMicroBlogComment(int commentID)
        {
            //TODO:验证操作者权限
            MicroBlogComment comment = MicroBlogComments.GetComment(commentID, true);
            if (comment != null)
            {
                if (CurrentUser != null && (CurrentUser.IsContentAdministrator || CurrentUser.UserID == comment.UserID))
                {
                    MicroBlogComments.DeleteComment(commentID);
                }
            }
            string returnUrl = Request.QueryString.GetString("returnUrl", string.Empty);
            return Redirect(WebUtils.UrlDecode(returnUrl));
        }

        /// <summary>
        /// 关注用户
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="returnUrl"></param>
        /// <returns></returns>
        public ActionResult FollowUser(int userID)
        {
            User user = Users.GetUser(userID, false);
            if (user == null)
                return new EmptyResult();
            return FollowUserInDomain(user.UserName);
        }

        /// <summary>
        /// 关注用户
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="returnUrl"></param>
        /// <returns></returns>
        public ActionResult FollowUserInDomain(string userDomainName)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            User currentUser = Globals.GetCurrentUser();
            if (currentUser == null || currentUser == null)
                return Redirect(SiteUrls.Instance().Error404("用户不存在！"));
            MicroBlogFollow follow = new MicroBlogFollow();
            follow.ItemID = domainUser.UserID;
            follow.UserID = currentUser.UserID;

            if (!MicroBlogFollows.IsFollow(currentUser.UserID, domainUser.UserID))
            {
                MicroBlogFollows.CreateFollow(follow);
            }
            else
            {
                follow = MicroBlogFollows.GetFollow(currentUser.UserID, domainUser.UserID);
                MicroBlogFollows.DeleteFollow(follow.FollowID);
            }

            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 注册用户
        /// </summary>
        /// <param name="token">邀请码</param>

        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult RegisterAction(string token)
        {
            PersonUser personUser = new PersonUser();
            personUser.Language = Globals.GetCurrentUserLanguage();
            personUser.UserName = Request.Form.GetString("userName", string.Empty).Trim();
            string apassword = Request.Form.GetString("apassword", string.Empty);
            personUser.Password = Request.Form.GetString("password", string.Empty).Trim();
            personUser.PrivateEmail = Request.Form.GetString("userEmail", string.Empty).Trim();
            personUser.CommonName = Request.Form.GetString("commonName", string.Empty).Trim();
            personUser.NickName = Request.Form.GetString("nickName", personUser.CommonName).Trim();
            personUser.Gender = (Genders)Request.Form.GetInt("gender", 1);

            //检查验证码是否录入正确
            //string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
            //string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
            //if (verifyCode != null)
            //    verifyCode = verifyCode.Trim();

            //if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
            //{
            //    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
            //    TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
            //    TempData["personUser"] = personUser;
            //    return RedirectToAction("Register", new RouteValueDictionary { { "token", token } });
            //}
            string errorMessage = "";
            string registerUrl = SPBUrlHelper.Action("Register", "MobileMicroBlog", new RouteValueDictionary { { "token", token } });
            if (!Globals.ValidateUserName(personUser.UserName, out errorMessage))
            {
                TempData["personUser"] = personUser;

                return RedirectToAction("Message", new RouteValueDictionary { { "message", errorMessage }, { "messageType", (int)StatusMessageType.Error }, { "url", registerUrl } });
            }
            if (!Globals.ValidatePassword(personUser.Password, out errorMessage))
            {
                TempData["personUser"] = personUser;

                return RedirectToAction("Message", new RouteValueDictionary { { "message", errorMessage }, { "messageType", (int)StatusMessageType.Error }, { "url", registerUrl } });
            }

            CreateUpdateUserStatuses status = CreateUpdateUserStatuses.UnknownFailure;
            AccountActivations activation = SiteSettingsManager.GetSiteSettings().AccountActivation;
            // Does the user require approval?
            if (activation == AccountActivations.Automatic)
                personUser.AccountStatus = UserAccountStatuses.Approved;
            else
                personUser.AccountStatus = UserAccountStatuses.ApprovalPending;
            User inviteUser = null;
            int inviteUserID = 0;
            if (!string.IsNullOrEmpty(token))
            {
                try
                {
                    inviteUserID = int.Parse(EncryptManager.DecryptTokenForInviteFriend(token));
                }
                catch { }
            }

            if (inviteUserID > 0)
            {
                inviteUser = Users.GetUser(inviteUserID, false);
                if (inviteUser != null)
                    personUser.InviteUserID = inviteUser.UserID;
            }

            if (personUser.UserName.Equals("Anonymous", StringComparison.CurrentCultureIgnoreCase))
            {
                status = CreateUpdateUserStatuses.DuplicateUsername;
            }
            else
            {
                try
                {
                    personUser = ConvertUserForStorage(personUser);
                    status = Users.Create(personUser, true);
                }
                catch
                {
                    //status = exception.;
                }
            }

            // Determine if the account was created successfully
            string message = string.Empty;
            switch (status)
            {
                // Username already exists!
                case CreateUpdateUserStatuses.DuplicateUsername:
                    ViewData["StatusMessageType"] = (int)StatusMessageType.Error;
                    message = "对不起，您输入的账号已被注册，请输入其他名称";
                    break;
                case CreateUpdateUserStatuses.BeyondLicensedCount:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    message = "对不起，网站的注册人数超过了授权数量";
                    break;

                // Email already exists!
                case CreateUpdateUserStatuses.DuplicateEmailAddress:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    message = string.Format("对不起，您输入的电子邮箱地址已经被使用。如果您曾经注册，则可以通过&lt;a href=&quot;{0}&quot;&gt;[获取密码]&lt;/a&gt;重新得到您的账号、密码", SiteUrls.Instance().ForgotPassword());
                    break;
                // Unknown failure has occurred!
                case CreateUpdateUserStatuses.UnknownFailure:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    message = "对不起，注册新用户的时候产生了一个未知错误";
                    break;

                // Username is disallowed
                case CreateUpdateUserStatuses.DisallowedUsername:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    message = "对不起，您输入的账号禁止使用，请输入其他名称";
                    break;

                case CreateUpdateUserStatuses.InvalidQuestionAnswer:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    message = "不正确的密码问题和答案";
                    break;

                case CreateUpdateUserStatuses.InvalidPassword:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    message = "您的密码长度少于本站要求的最小密码长度，请重新输入";
                    break;

                // Everything went off fine, good
                case CreateUpdateUserStatuses.Created:
                    //建立朋友连接
                    if (inviteUser != null && inviteUser.UserType == UserTypes.PersonUser)
                    {
                        Friends.CreateFriend(inviteUser.UserID, personUser.UserID);

                        #region 触发 成功邀请站外朋友事件
                        Friend newFriend = new Friend();
                        newFriend.UserID = inviteUser.UserID;
                        newFriend.FriendUserID = personUser.UserID;
                        FriendEvents.SuccessInviteFriend(newFriend);
                        #endregion

                        #region 更新邀请者 邀请的朋友数
                        inviteUser.InviteFriendCount += 1;
                        Users.UpdateUserStatistics(inviteUser);
                        #endregion
                    }
                    switch (activation)
                    {
                        case AccountActivations.AdminApproval:
                            LeaveSecureConnection(SiteUrls.Instance().ShowExceptionMessage(ExceptionTypes.Instance().UserAccountPending()));
                            break;

                        case AccountActivations.Email:
                            LeaveSecureConnection(SPBUrlHelper.Action("UserAccountCreatedValidateEmail", "ChannelUser", new RouteValueDictionary { { "userName", personUser.UserName } }));
                            break;

                        case AccountActivations.Automatic:
                            if (!Request.IsAuthenticated)
                                FormsAuthentication.SetAuthCookie(personUser.UserName, false);
                            LeaveSecureConnection(SPBUrlHelper.Action("MyMicroBlogHome", "MobileMicroBlog", new RouteValueDictionary { { "userDomainName", personUser.UserName } }));
                            break;
                    }
                    return new EmptyResult();
            }
            TempData["personUser"] = personUser;
            return RedirectToAction("Message", new RouteValueDictionary { { "message", message }, { "messageType", (int)StatusMessageType.Error }, { "url", registerUrl } });
        }

        /// <summary>
        /// 提示信息
        /// </summary>
        /// <param name="message"></param>
        /// <param name="messageType"></param>
        /// <param name="url"></param>
        /// <returns></returns>
        public ActionResult Message(string message, int messageType, string url)
        {
            ViewData["StatusMessageContent"] = message;
            ViewData["StatusMessageType"] = (StatusMessageType)messageType;
            ViewData["returnUrl"] = url;
            return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
        }

        private void LeaveSecureConnection(string url)
        {
            if (Request.IsSecureConnection)
                url = string.Format("https://{0}{1}", Request.Url.Host, url);
            Response.Redirect(url, true);
        }

        /// <summary>
        /// 用户档案存储前进行转化
        /// </summary>
        private static PersonUser ConvertUserForStorage(PersonUser pu)
        {
            pu.CommonName = Formatter.FormatSingleLinePlainTextForStorage(pu.CommonName);
            pu.NickName = Formatter.FormatSingleLinePlainTextForStorage(pu.NickName);
            pu.PublicEmail = Formatter.FormatSingleLinePlainTextForStorage(pu.PublicEmail);
            pu.MsnIM = Formatter.FormatSingleLinePlainTextForStorage(pu.MsnIM);
            pu.QQIM = Formatter.FormatSingleLinePlainTextForStorage(pu.QQIM);
            pu.YahooIM = Formatter.FormatSingleLinePlainTextForStorage(pu.YahooIM);
            pu.SkypeIM = Formatter.FormatSingleLinePlainTextForStorage(pu.SkypeIM);
            pu.PersonalHomepage = Formatter.FormatSingleLinePlainTextForStorage(pu.PersonalHomepage);
            return pu;
        }

        /// <summary>
        /// 注销
        /// </summary>
        //TODO: 暂时方法
        public ActionResult Logout()
        {
            FormsAuthentication.SignOut();
            ///移除在线用户
            if (CurrentUser != null)
                UserOnlines.DeleteOnlineUser(CurrentUser.UserID);

            return RedirectToAction("Login");
        }

        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UploadPicture(string userDomainName)
        {

            if (CurrentUser == null || CurrentUser.UserID != CurrentDomainUser.UserID)
                return View();

            System.Web.HttpPostedFileBase hpf = null;

            if (Request.Files.Count > 0)
            {
                hpf = Request.Files[0] as System.Web.HttpPostedFileBase;

                if (hpf.ContentLength == 0)
                {
                    ViewData["StatusMessageContent"] = "请选择要上传的附件！";
                    ViewData["StatusMessageType"] = StatusMessageType.Error;
                    return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
                }
                else
                {
                    if (hpf.ContentLength > MicroBlogConfiguration.Instance().MaxAttachmentSize * 1024)
                    {
                        ViewData["StatusMessageContent"] = string.Format("<script type=\"text/javacript\">alert('上传文件大小不能超过{0}K。');</script>", MicroBlogConfiguration.Instance().MaxAttachmentSize); ;
                        return View("Pages/Message.aspx", "Masters/MicroBlog.Master");
                    }
                    if (!string.IsNullOrEmpty(hpf.FileName))
                    {
                        string fileTypeString = hpf.FileName.Substring(hpf.FileName.LastIndexOf(".") + 1);
                        string uploadPhotoTypes = MicroBlogConfiguration.Instance().MicroBlogAttachmentSettings.Extensions;
                        StringBuilder newUploadPhotoTypes = new StringBuilder();
                        string[] types = uploadPhotoTypes.Split(',');
                        bool isPermitted = false;
                        foreach (string type in types)
                        {
                            if (!string.IsNullOrEmpty(type))
                            {
                                if (string.Compare(type, fileTypeString, true) == 0)
                                {
                                    isPermitted = true;
                                }
                            }
                        }
                        if (!isPermitted)
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            return Redirect(Request.UrlReferrer.AbsoluteUri);
                        }
                    }
                    string newfileName = hpf.FileName.Substring(hpf.FileName.LastIndexOf("\\") + 1);

                    Attachment attachment = new Attachment(hpf);
                    attachment.ContentSize = hpf.ContentLength;
                    attachment.ContentType = hpf.ContentType;
                    attachment.DateCreated = DateTime.Now;
                    attachment.UserID = CurrentUser.UserID;
                    attachment.FriendlyFileName = newfileName;

                    MicroBlogAttachmentManager.Instance().CreateTemporaryAttachment(attachment, hpf.InputStream);

                    IList<Attachment> temporarys = MicroBlogAttachmentManager.Instance().GetTemporaryAttachmentsOfUser(CurrentUser.UserID);
                }
            }
            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        #endregion

    }

    public enum MobileMicroBlogSubMenu
    {
        /// <summary>
        /// 我的首页
        /// </summary>
        MyMicroBlogHome,

        /// <summary>
        /// 随便看看
        /// </summary>
        NewMicroBlog,

        /// <summary>
        /// 话题
        /// </summary>
        MicroBlogTopic
    }
}


