﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------ 

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using SpaceBuilder.Utils;
using SpaceBuilder.Common;
using SpaceBuilder.Web;
using System.Web.Routing;
using System.Text.RegularExpressions;
using SpaceBuilder.LuceneSearch;
using Lucene.Net.Search;
using System.Threading;
using System.Web.UI;
namespace SpaceBuilder.MicroBlog.Controllers
{
    [HttpCompress]
    public class UserDomainMicroBlogController : UserDomainControllerBase
    {
        private int applicationID = ApplicationIDs.Instance().MicroBlog();
        private delegate void SyncMicroBlogDelegate(string body, int userID, int threadID, bool? hasPhoto);

        #region Control

        /// 微博发布框
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult Control_MicroBlogInput(string userDomainName)
        {

            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            List<EmotionCategory> listEmotionCategory = EmotionManager.Instance("MicroBlogEmotion.xml").Categories;
            if (listEmotionCategory != null && listEmotionCategory.Count > 0)
            {
                List<Emotion> listEmotion = listEmotionCategory[0].Emotions;
                ViewData["listEmotion"] = listEmotion;

                if (AccountBindings.GetAccountBinding(currentDomainUser.UserID, AccountTypes.Instance().SinaWeibo()) != null)
                    ViewData["isbind"] = true;
            }
            return PartialView("Controls/MicroBlogs/MicroBlogInput.ascx");
        }

        /// <summary>
        ///  在动态中显示主题的回复
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListMicroBlogCommentsInActive(string userDomainName, int threadID, bool? isWide)
        {
            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, false);
            IList<MicroBlogComment> comments = null;
            if (thread != null)
                comments = MicroBlogComments.GetTopCommentsOfThread(threadID, 10, false);
            ViewData["MicroBlogThread"] = thread;
            return PartialView("Controls/MicroBlogs/ListMicroBlogCommentsInActive.ascx", comments);
        }

        #endregion

        #region Action

        /// <summary>
        /// 发布微博
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreateMicroBlogThread(string userDomainName, string returnUrl)
        {
            User currentDomainUser = GetCurrentDomainUser(WebUtils.UrlDecode(userDomainName));

            MicroBlogThread thread = new MicroBlogThread();
            thread.Body = Request.Form["microBlogBody"];
            try
            {
                Globals.CheckPostDate();
            }
            catch (Exception e)
            {
                return Json(new { error = true, msg = "你发微博的速度太快哦！" });
            }

            if (string.IsNullOrEmpty(thread.Body))
                return Json(new { error = true, msg = "微博内容不能为空！" });

            int attachmetID = Request.Form.GetInt("attachmentID", -1);
            if (attachmetID > 0)
            {
                Attachment mAttachment = MicroBlogAttachmentManager.Instance().GetAttachment(attachmetID, true);
                if (mAttachment != null)
                    thread.HasPhoto = true;
            }
            thread.OwnerUserID = currentDomainUser.UserID;
            thread.Author = CurrentUser.DisplayName;
            thread.PostMode = PostModes.Web;
            thread.PostSource = WebUtils.IPAddress;
            thread.UserID = CurrentUser.UserID;

            try
            {
                MicroBlogThreads.CreateThread(thread);
            }
            catch { }

            if (Request.Form.GetBool("syncsina", false) && thread.ThreadID > 0)
            {
                try
                {
                    SyncMicroBlogDelegate sd = new SyncMicroBlogDelegate(SyncSina);
                    sd.BeginInvoke(thread.Body, currentDomainUser.UserID, thread.ThreadID, thread.HasPhoto, null, null);
                }
                catch { }
            }

            if (thread.AuditingStatus != AuditingStatuses.Success)
                return Json(new { ok = true, msg = "发布成功，需要通过审核才能显示！" });

            return Json(new { ok = true });
        }

        /// <summary>
        /// 同步新浪微博
        /// </summary>
        private void SyncSina(string body, int userID, int threadID, bool? hasPhoto)
        {
            AccountBinding ab = AccountBindings.GetAccountBinding(userID, AccountTypes.Instance().SinaWeibo());
            if (ab == null)
                return;

            string imgurl = string.Empty;
            if (hasPhoto.HasValue && hasPhoto.Value)
            {
                Attachment attachment = MicroBlogAttachmentManager.Instance().GetLatestAttachmentOfAssociateID(threadID);
                if (attachment != null)
                {
                    IFile file = FileStorageProvider.Instance(MicroBlogAttachmentManager.Instance().StorageProviderName).GetFile(attachment.GetRelativePath(), attachment.FileName);

                    if (file != null)
                    {
                        FileSystemFile fileSystemFile = file as FileSystemFile;
                        imgurl = fileSystemFile.FullLocalPath;
                    }
                }
            }
            oAuthSina oauth = new oAuthSina();
            oauth.token = ab.OauthToken;
            oauth.tokenSecret = ab.OauthTokenSecret;

            if (string.IsNullOrEmpty(imgurl))
            {
                string url = "http://api.t.sina.com.cn/statuses/update.xml?";
                oauth.oAuthWebRequest(oAuthSina.Method.POST, url, "status=" + body);
            }
            else
            {
                string url = "http://api.t.sina.com.cn/statuses/upload.xml?";
                oauth.oAuthWebRequestWithPic(oAuthSina.Method.POST, url, "status=" + body, imgurl);
            }
        }

        /// <summary>
        /// 在动态里回复微博
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateMicroBlogCommentInActive(string userDomainName, int threadID, bool? isWide)
        {
            MicroBlogThread thread = MicroBlogThreads.GetThread(threadID, false);
            if (CurrentUser != null && thread != null)
            {
                string body = Request.Form["microBlogCommentBody"];
                MicroBlogComment comment = new MicroBlogComment();
                comment.ThreadID = threadID;
                comment.Body = body;
                comment.OwnerUserID = thread.OwnerUserID;
                comment.PostMode = PostModes.Web;
                comment.PostSource = WebUtils.IPAddress;
                comment.UserID = CurrentUser.UserID;
                comment.Author = CurrentUser.DisplayName;
                try
                {
                    MicroBlogComments.CreateComment(comment);
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "发表评论成功！");
                }
                catch (SPBException e)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, e.Message);
                }
            }

            return RedirectToAction("Control_ListMicroBlogCommentsInActive", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 在动态中删除微博回复
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteMicroBlogCommentInActive(string userDomainName, int commentID)
        {
            //TODO:验证操作者权限
            MicroBlogComment comment = MicroBlogComments.GetComment(commentID, true);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "评论不存在 ！");

            if (comment != null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "没有该项操作权限！");

                if (CurrentUser != null && (CurrentUser.IsContentAdministrator || CurrentUser.UserID == comment.OwnerUserID))
                {
                    try
                    {
                        MicroBlogComments.DeleteComment(commentID);
                        TempData["StatusMEssageData"] = new StatusMessageData(StatusMessageType.Success, "删除成功！");
                    }
                    catch (SPBException e)
                    {
                        TempData["StatusMEssageData"] = new StatusMessageData(StatusMessageType.Error, "删除失败如有问题请联系管理员！");
                    }
                }
            }

            return RedirectToAction("Control_ListMicroBlogCommentsInActive", CurrentRouteValueDictionary);
        }

        #region 微博图片

        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UploadPhoto(string userDomainName)
        {
            System.Web.HttpPostedFileBase hpf = null;
            string url = string.Empty;
            if (Request.Files.Count > 0)
            {
                hpf = Request.Files[0] as System.Web.HttpPostedFileBase;

                if (hpf.ContentLength == 0)
                    return Content("<textarea>{\"message\":\"文件大小不能为0KB.\"}</textarea>", "text/html");
                else
                {
                    if (hpf.ContentLength > MicroBlogConfiguration.Instance().MaxAttachmentSize * 1024)
                        return Content("<textarea>{\"message\":\"" + string.Format("上传文件大小不能超过{0}K。", MicroBlogConfiguration.Instance().MaxAttachmentSize) + "\"}</textarea>", "text/html");

                    User currentUser = Users.GetUser(userDomainName, false);

                    if (!currentUser.IsContentAdministrator && hpf.ContentLength / 1024 > (currentUser.DatabaseQuota - currentUser.DatabaseQuotaUsed))
                        return Content("<textarea>{\"message\":\"空间容量不足！\"}</textarea>", "text/html");

                    if (!string.IsNullOrEmpty(hpf.FileName))
                    {
                        string fileTypeString = hpf.FileName.Substring(hpf.FileName.LastIndexOf(".") + 1);
                        string uploadPhotoTypes = MicroBlogConfiguration.Instance().MicroBlogAttachmentSettings.Extensions;
                        StringBuilder newUploadPhotoTypes = new StringBuilder();
                        string[] types = uploadPhotoTypes.Split(',');
                        bool isPermitted = false;
                        foreach (string type in types)
                        {
                            if (!string.IsNullOrEmpty(type))
                            {
                                if (string.Compare(type, fileTypeString, true) == 0)
                                {
                                    isPermitted = true;
                                }
                            }
                        }

                        if (!isPermitted)
                            return Content("<textarea>{\"message\":\"不支持所上传的文件类型！\"}</textarea>", "text/html");
                    }
                    string newfileName = hpf.FileName.Substring(hpf.FileName.LastIndexOf("\\") + 1);

                    Attachment attachment = new Attachment(hpf);
                    attachment.ContentSize = hpf.ContentLength;
                    attachment.ContentType = hpf.ContentType;
                    attachment.DateCreated = DateTime.Now;
                    attachment.UserID = CurrentUser.UserID;
                    attachment.FriendlyFileName = newfileName;

                    IList<Attachment> temAttachments = MicroBlogAttachmentManager.Instance().GetTemporaryAttachmentsOfUser(currentUser.UserID);

                    foreach (var temAtt in temAttachments)
                    {
                        if (temAtt != null) MicroBlogAttachmentManager.Instance().DeleteAttachment(temAtt);
                    }

                    MicroBlogAttachmentManager.Instance().CreateTemporaryAttachment(attachment, hpf.InputStream);
                    IList<Attachment> temporaryAttachments = MicroBlogAttachmentManager.Instance().GetTemporaryAttachmentsOfUser(currentUser.UserID);
                    Attachment temAttachment = MicroBlogAttachmentManager.Instance().GetLatestAttachmentOfUser(currentUser.UserID);
                    url = SiteUrls.Instance().MicroBlogPhotoUrl(temAttachment, MicroBlogPhotoTypes.Preview);
                    return Content("<textarea>{\"ok\":\"true\",\"url\":\"" + url + "\",\"PhotoName\":\"" + StringUtils.Trim(temAttachment.FriendlyFileName.Substring(0, temAttachment.FriendlyFileName.IndexOf('.') - 1), 5, "") + "\",\"attachmentID\":\"" + temAttachment.AttachmentID + "\"}</textarea>", "text/html");
                }
            }

            return Content("<textarea>{\"message\":\"上传失败！\"}</textarea>", "text/html");
        }

        [AcceptVerbs(HttpVerbs.Post)]
        public JsonResult DeletePhoto(int attachmentID)
        {
            Attachment attachment = MicroBlogAttachmentManager.Instance().GetAttachment(attachmentID, true);
            if (attachment != null)
                MicroBlogAttachmentManager.Instance().DeleteAttachment(attachment);
            return Json(new { ok = 1 });
        }

        #endregion

        #endregion
    }
}
