﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.MicroBlog;
using Lucene.Net.Documents;
using SpaceBuilder.Utils;
using SpaceBuilder.Common;
using Lucene.Net.QueryParsers;
using Lucene.Net.Search;
using Lucene.Net.Index;

namespace SpaceBuilder.LuceneSearch
{
    public class MicroBlogSearchManager : SearchManagerBase<MicroBlogThread>
    {
        //索引文件夹
        private static readonly string microBlogIndexFileDirectory = "MicroBlog";
        private static volatile MicroBlogSearchManager _self = null;
        private static readonly object lockObject = new object();

        private MicroBlogSearchManager(string indexFileDirectory) : base(indexFileDirectory) { }

        public static MicroBlogSearchManager Instance()
        {
            if (_self == null)
            {
                lock (lockObject)
                {
                    if (_self == null)
                    {
                        _self = new MicroBlogSearchManager(microBlogIndexFileDirectory);
                    }
                }
            }
            return _self;
        }

        /// <summary>
        /// 搜索
        /// </summary>
        public SearchResultDataSet<MicroBlogThread> Search(string keyWord, bool isTopic, int pageSize, int pageIndex)
        {
            keyWord = StringUtilsForLucene.LuceneKeywordsScrubber(keyWord.ToLower());

            if (string.IsNullOrEmpty(keyWord) || !IsIndexFilesExists)
                return new SearchResultDataSet<MicroBlogThread>();
            
            SortField[] sortFields = new SortField[] { new SortField(MicroBlogIndexFields.ThreadID, SortField.INT, true) };
            if (isTopic)
            {
                string[] listKeyWords = SegmentForPhraseQuery(keyWord);
                PhraseQuery query = new PhraseQuery();
                foreach (var item in listKeyWords)
                    query.Add(new Term(MicroBlogIndexFields.Body, item));

                query.SetSlop(PhraseQuerySlop);
                SearchResultDataSet<MicroBlogThread> pds = Search(query, null, sortFields, pageIndex, pageSize);
                return pds;
            }
            else
            {
                string keywordSegments = SegmentForQueryParser(keyWord);
                QueryParser parser = new QueryParser(CurrentLuceneVersion, MicroBlogIndexFields.Body, GetChineseAnalyzerOfUnTokenized());
                Query query = parser.Parse(keywordSegments);
                SearchResultDataSet<MicroBlogThread> pds = Search(query, null, sortFields, pageIndex, pageSize);
                return pds;
            }
        }

        public override void InitializeIndex(string indexPath)
        {
            if (!System.IO.Directory.Exists(indexPath))
            {
                try
                {
                    System.IO.Directory.CreateDirectory(indexPath);
                }
                catch
                {
                    throw new ApplicationException(string.Format("create Directory '{0}' failed", PhysicalIndexDirectory));
                }
            }

            #region 索引

            int indexPageSize = 2000;
            bool createIndexFile = true;
            PagingDataSet<MicroBlogThread> pds = MicroBlogThreads.GetThreadsForAdmin(indexPageSize, 1, Applications.GetApplication(ApplicationIDs.Instance().MicroBlog()).AuditingStatusForPublicDisplay, string.Empty, string.Empty, -1);
            double tIndex = Convert.ToDouble(pds.TotalRecords) / Convert.ToDouble(indexPageSize);
            int indexPageIndex = (int)Math.Ceiling(tIndex);

            if (pds.TotalRecords > 0)
            {
                //分多次进行索引
                for (int i = 1; i <= indexPageIndex; i++)
                {
                    if (i != 1)
                        pds = MicroBlogThreads.GetThreadsForAdmin(indexPageSize, i, Applications.GetApplication(ApplicationIDs.Instance().MicroBlog()).AuditingStatusForPublicDisplay, string.Empty, string.Empty, -1);
                    Insert(pds.Records, indexPath, createIndexFile);
                    if (createIndexFile)
                        createIndexFile = false;
                }
            }

            #endregion
        }

        //protected override Lucene.Net.Analysis.Analyzer GetChineseAnalyzer()
        //{
        //    return new Lucene.Net.Analysis.WhitespaceAnalyzer();
        //}

        protected override Lucene.Net.Documents.Document ConvertObjToDocument(MicroBlogThread thread)
        {
            if (thread == null)
                return null;

            //待审核及未通过审核不允许加入索引
            if (thread.AuditingStatus == AuditingStatuses.Pending || thread.AuditingStatus == AuditingStatuses.Fail)
                return null; 

            Document doc = new Document();
            Field field;

            field = new Field(MicroBlogIndexFields.UserID, thread.UserID.ToString(), Field.Store.YES, Field.Index.NO);
            doc.Add(field);

            field = new Field(MicroBlogIndexFields.ThreadID, thread.ThreadID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(MicroBlogIndexFields.Author, thread.Author.ToLower(), Field.Store.YES, Field.Index.NO);
            doc.Add(field);

            field = new Field(MicroBlogIndexFields.HasMusic, thread.HasMusic.ToString(), Field.Store.YES, Field.Index.NO);
            doc.Add(field);

            field = new Field(MicroBlogIndexFields.HasPhoto, thread.HasPhoto.ToString(), Field.Store.YES, Field.Index.NO);
            doc.Add(field);

            field = new Field(MicroBlogIndexFields.HasVideo, thread.HasVideo.ToString(), Field.Store.YES, Field.Index.NO);
            doc.Add(field);

            field = new Field(MicroBlogIndexFields.PlayUrl, thread.PlayUrl == null ? string.Empty : thread.PlayUrl, Field.Store.YES, Field.Index.NO);
            doc.Add(field);

            field = new Field(MicroBlogIndexFields.ThumbnailUrl, thread.ThumbnailUrl == null ? string.Empty : thread.ThumbnailUrl, Field.Store.YES, Field.Index.NO);
            doc.Add(field);

            field = new Field(MicroBlogIndexFields.Body, thread.LinkBody, Field.Store.NO, Field.Index.ANALYZED);
            doc.Add(field);

            field = new Field(MicroBlogIndexFields.LinkBody, thread.LinkBody, Field.Store.YES, Field.Index.NO);
            doc.Add(field);

            field = new Field(MicroBlogIndexFields.DateCreated, DateTools.DateToString(thread.DateCreated, DateTools.Resolution.MINUTE), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(MicroBlogIndexFields.OriginalThreadID, thread.OriginalThreadID.ToString(), Field.Store.YES, Field.Index.NO);
            doc.Add(field);

            return doc;
        }

        protected override MicroBlogThread ConvertDocumentToObj(Lucene.Net.Documents.Document doc)
        {
            MicroBlogThread blogThread = new MicroBlogThread();
            int userID;
            int.TryParse(doc.Get(MicroBlogIndexFields.UserID), out userID);
            blogThread.OwnerUserID = userID;

            blogThread.Author = doc.Get(MicroBlogIndexFields.Author);

            int threadID;
            int.TryParse(doc.Get(MicroBlogIndexFields.ThreadID), out threadID);
            blogThread.ThreadID = threadID;

            blogThread.LinkBody = doc.Get(MicroBlogIndexFields.LinkBody);

            bool hasVideo;
            bool.TryParse(doc.Get(MicroBlogIndexFields.HasVideo), out hasVideo);
            blogThread.HasVideo = hasVideo;

            bool hasPhoto;
            bool.TryParse(doc.Get(MicroBlogIndexFields.HasPhoto), out hasPhoto);
            blogThread.HasPhoto = hasPhoto;

            bool hasMusic;
            bool.TryParse(doc.Get(MicroBlogIndexFields.HasMusic), out hasMusic);
            blogThread.HasMusic = hasMusic;

            blogThread.PlayUrl = doc.Get(MicroBlogIndexFields.PlayUrl);
            blogThread.ThumbnailUrl = doc.Get(MicroBlogIndexFields.ThumbnailUrl);

            int OriginalThreadID;
            int.TryParse(doc.Get(MicroBlogIndexFields.OriginalThreadID), out OriginalThreadID);
            blogThread.OriginalThreadID = OriginalThreadID;

            try
            {
                blogThread.DateCreated = DateTools.StringToDate(doc.Get(MicroBlogIndexFields.DateCreated));
            }
            catch { }

            return blogThread;
        }
    }
}
