﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using SpaceBuilder.Utils;
using SpaceBuilder.Common;
using SpaceBuilder.Web;
using System.Web.Routing;
using System.Web;

namespace SpaceBuilder.MicroBlog
{
    /// <summary>
    /// Blog相关Url
    /// </summary>
    public static class SiteUrlsExtensions
    {

        /// <summary>
        /// 根据AttachmentID获取标题图片Url
        /// </summary>
        public static string MicroBlogPhotoUrl(this SiteUrls siteUrls, Attachment attachment, MicroBlogPhotoTypes photoType)
        {
            //如果启用URL直连 则直接使用直连URL
            if (FileStorageProvider.Instance(MicroBlogAttachmentManager.Instance().StorageProviderName).EnableDirectlyUrl)
            {
                if (attachment == null)
                    return string.Empty;

                //默认仅预生成标题图
                if (photoType == MicroBlogPhotoTypes.View || photoType == MicroBlogPhotoTypes.Preview)
                    return MicroBlogAttachmentManager.Instance().GetDirectlyUrl(attachment, photoType);
            }
            return Globals.GetFullUrlInFileHandlerSite(MicroBlogAttachmentManager.Instance().StorageProviderName, string.Format("~/Services/MicroBlog/MicroBlogImage.ashx?AttachmentID={0}&ImageType={1}", attachment.AttachmentID, photoType));
        }

        /// <summary>
        /// 微博话题Logo图片地址
        /// </summary>
        public static string MicroBlogTopicLogoUrl(this SiteUrls siteUrls, MicroBlogTopic topic)
        {
            if (FileStorageProvider.Instance(MicroBlogTopics.StorageProviderName).EnableDirectlyUrl)
            {
                return MicroBlogTopics.GetTopicPhotoDirectlyUrl(topic);
            }

            return Globals.GetFullUrlInFileHandlerSite(ConvertibleGifts.StorageProviderName, string.Format("~/Services/MicroBlog/MicroBlogTopicLogo.ashx?topicID={0}", topic.TopicID));
        }

        #region 重复配置的URL

        /// <summary>
        /// 我的微博首页
        /// </summary>
        /// <param name="siteUrls"></param>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public static string UserMicroBlogHome(this SiteUrls siteUrls, int userID)
        {
            string userName = UserID2UserNameDictionary.GetUserName(userID);
            return UserMicroBlogHome(siteUrls, userName);
        }


        /// <summary>
        /// 我的微博首页
        /// </summary>
        /// <param name="siteUrls"></param>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public static string UserMicroBlogHome(this SiteUrls siteUrls, int userID, User currentUser)
        {
            string userName = UserID2UserNameDictionary.GetUserName(userID);
            if (currentUser != null && currentUser.UserName.Equals(userName, StringComparison.CurrentCultureIgnoreCase))
                return MyMicroBlogs(siteUrls, userID);
            else
                return UserMicroBlogHome(siteUrls, userName);

        }

        /// <summary>
        /// 我的微博首页
        /// </summary>
        /// <param name="siteUrls"></param>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public static string UserMicroBlogHome(this SiteUrls siteUrls, string userDomainName)
        {
            return SPBUrlHelper.RouteUrl("MicroBlogDomain_MyHome", new RouteValueDictionary { { "action", "MyMicroBlogHome" }, { "UserDomainName", WebUtils.UrlDecode(userDomainName) } });
        }

        /// <summary>
        /// 微博详细页
        /// </summary>
        /// <param name="siteRuls"></param>
        /// <param name="userID"></param>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public static string ShowMicroBlog(this SiteUrls siteRuls, int userID, int threadID)
        {
            string userName = UserID2UserNameDictionary.GetUserName(userID);
            return ShowMicroBlog(siteRuls, userName, threadID);
        }

        /// <summary>
        /// 微博详细页
        /// </summary>
        /// <param name="siteRuls"></param>
        /// <param name="userID"></param>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public static string ShowMicroBlog(this SiteUrls siteRuls, string userName, int threadID)
        {
            return SPBUrlHelper.Action("ShowMicroBlogThread", "MicroBlog", new RouteValueDictionary { { "userDomainName", userName }, { "threadID", threadID } });
        }


        public static string RedirectUrl(this SiteUrls siteRuls, string shortUrl)
        {
            return SPBUrlHelper.RouteUrl("MicroBlogDomain_ShortUrl", new RouteValueDictionary { { "action", "RedirectUrl" }, { "controller", "MicroBlog" }, { "shortUrl", shortUrl } });
        }


        /// <summary>
        /// 我的微博
        /// </summary>
        /// <param name="siteRuls"></param>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public static string MyMicroBlogs(this SiteUrls siteRuls, int userID)
        {
            string userName = UserID2UserNameDictionary.GetUserName(userID);
            return MyMicroBlogs(siteRuls, userName);
        }

        /// <summary>
        /// 我的微博
        /// </summary>
        /// <param name="siteRuls"></param>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public static string MyMicroBlogs(this SiteUrls siteRuls, string userDomainName)
        {
            return SPBUrlHelper.RouteUrl("MicroBlogDomain_UserMicroBlogs", new RouteValueDictionary { { "action", "ListMyMicroBlogThreads" }, { "UserDomainName", userDomainName } });
        }

        /// <summary>
        /// 提到我的微博
        /// </summary>
        /// <param name="siteRuls"></param>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public static string AtMe(this SiteUrls siteRuls, string userDomainName)
        {
            return SPBUrlHelper.RouteUrl("MicroBlogDomain_ReferUserMicroBlogs", new RouteValueDictionary { { "action", "ListReferMeMicroBlogs" }, { "UserDomainName", userDomainName } });
        }

        /// <summary>
        /// 登录
        /// </summary>
        /// <param name="includeReturnUrl">是否包含returnUrl</param>
        public static string MicroBlogHome(this SiteUrls siteRuls, bool includeReturnUrl)
        {
            if (includeReturnUrl)
            {
                HttpContext httpContext = HttpContext.Current;

                string currentPath = httpContext.Request.Url.PathAndQuery;
                string returnUrl = SiteUrls.ExtractQueryParams(currentPath)["ReturnUrl"];

                if (string.IsNullOrEmpty(returnUrl))
                    return MicroBlogHome(siteRuls, WebUtils.UrlEncode(HttpContext.Current.Request.RawUrl));
                else
                    return MicroBlogHome(siteRuls, returnUrl);
            }
            else
                return MicroBlogHome(siteRuls);
        }

        /// <summary>
        /// 登录
        /// </summary>
        public static string MicroBlogHome(this SiteUrls siteRuls)
        {
            return MicroBlogHome(siteRuls, string.Empty);
        }

        /// <summary>
        /// 登录
        /// </summary>
        private static string MicroBlogHome(this SiteUrls siteRuls, string ReturnUrl)
        {
            return SPBUrlHelper.RouteUrl("MicroBlogDomain_Home", new RouteValueDictionary { { "action", "MicroBlogHome" }, { "ReturnUrl", ReturnUrl } });
        }

        /// <summary>
        /// 关注用户/粉丝列表
        /// </summary>
        /// <param name="siteRuls"></param>
        /// <param name="userDomainName"></param>
        /// <param name="sortBy"></param>
        /// <returns></returns>
        public static string ListFollowUser(this SiteUrls siteRuls, string userDomainName, int sortBy)
        {
            return SPBUrlHelper.Action("ListFollow", "MicroBlog", new RouteValueDictionary { { "userDomainName", userDomainName }, { "SortBy", sortBy } });
        }

        /// <summary>
        /// 显示话题
        /// </summary>
        /// <param name="topic">话题关键字</param>
        /// <param name="isFull">是否返回完整Url</param>
        public static string Topic(this SiteUrls siteRuls, string topic, bool? isFull)
        {
            int topicID = MicroBlogTopics.GetTopicIDByTopic(topic);
            return Topic(siteRuls, topicID, isFull);
        }

        /// <summary>
        /// 显示话题
        /// </summary>
        /// <param name="topic">话题关键字</param>
        /// <param name="isFull">是否返回完整Url</param>
        public static string Topic(this SiteUrls siteRuls, int topicID, bool? isFull)
        {
            if (isFull.HasValue && isFull.Value)
                return WebUtils.FullPath(SPBUrlHelper.Action("ShowMicroBlogTopic", "MicroBlog", new RouteValueDictionary { { "topicID", topicID } }));
            else
                return SPBUrlHelper.Action("ShowMicroBlogTopic", "MicroBlog", new RouteValueDictionary { { "topicID", topicID } });
        }

        public static string SearchTopic(this SiteUrls siteRuls, string topic, PostModes? mode)
        {
            mode = mode ?? PostModes.Web;
            if (mode == PostModes.Mobile)
                return SPBUrlHelper.Action("ListTopicMicroBlogThreads", "MobileMicroBlog", new RouteValueDictionary { { "keyWord", WebUtils.UrlEncode(topic) } });
            else
                return SPBUrlHelper.Action("SearchMicroBlogs", "MicroBlog", new RouteValueDictionary { { "keyWord", WebUtils.UrlEncode(topic) } });

        }

        /// <summary>
        /// 话题访问连接
        /// </summary>
        public static string TopicUrl(this SiteUrls siteUrls, string topic, int mode)
        {
            return WebUtils.ResolveUrl(String.Format("~/Services/MicroBlog/TopicUrl.ashx?topic={0}&mode={1}", WebUtils.UrlEncode(topic), mode));
        }

        #endregion
    }
}
