﻿//------------------------------------------------------------------------------
//<copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights reserved.
//</copyright> 
//------------------------------------------------------------------------------ 
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using SpaceBuilder.Common.Controllers;
using SpaceBuilder.Common;
using SpaceBuilder.Web;
using SpaceBuilder.LuceneSearch;
using System.Web.Routing;
using SpaceBuilder.Utils;
using System.Threading;
using System.Web.UI;

namespace SpaceBuilder.News.Controllers
{
    /// <summary>
    /// 频道资讯
    /// </summary>
    [HttpCompress]
    public class ChannelNewsController : ChannelControllerBase
    {
        int applicationID = ApplicationIDs.Instance().News();
        private static readonly string pageSeparator = "<!-- pagebreak -->";

        #region News

        /// <summary>
        /// 资讯首页
        /// </summary>
        /// <returns></returns>
        public ActionResult Home()
        {
            SetPageTitle(GetResourceString("Title_News", applicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_News", applicationID), string.Empty);

            string viewName = "SimpleHome";
            viewName = NewsConfiguration.Instance().HomePageName;

            return View("Pages/News/" + viewName + ".aspx", "Masters/News.master");
        }


        public ActionResult Control_ListNewsInSection()
        {
            //1级栏目资讯
            List<NewsSection> rootList = NewsSections.GetSections(null, true, NewsSectionsOrganizeOptions.None, NewsSectionsSortBy.DisplayOrder, SortOrder.Ascending);
            Dictionary<NewsSection, IList<NewsThread>> dict = new Dictionary<NewsSection, IList<NewsThread>>();

            foreach (NewsSection section in rootList)
            {
                IList<NewsThread> pdsThread = NewsThreads.GetTopThreadsSortBy(NewThreadsSortBy.PostDate, section.SectionID, 5);
                dict.Add(section, pdsThread);
            }
            ViewData["sectionThread"] = dict;
            return PartialView("Controls/News/ListNewsInSection.ascx");
        }

        /// <summary>
        /// 显示列表、概要、图片资讯
        /// </summary>
        /// <param name="sectionID">栏目ID</param>
        /// <param name="sortBy">排序方式</param>
        /// <param name="viewName">控件名称（不加后缀）</param>
        /// <param name="titleName">控件标题（不填写时不显示）</param>
        /// <param name="pageSize">每页显示条瘦身</param>
        /// <param name="pageIndex">显示第几页</param>
        /// <returns></returns>
        public ActionResult Control_ListOrSummaryNewsBlock(int? sectionID, NewThreadsSortBy? sortBy, string viewName, string titleName, int pageSize, int? pageIndex)
        {
            if (!sectionID.HasValue || sectionID.Value <= 0)
                sectionID = -1;
            PagingDataSet<NewsThread> pds = NewsThreads.GetNewsThreadsSortBy(sortBy ?? NewThreadsSortBy.StickyAndLastRepliedDate, sectionID.Value, pageSize, pageIndex ?? 1);
            ViewData["sortBy"] = sortBy ?? NewThreadsSortBy.StickyAndLastRepliedDate;
            ViewData["titleName"] = titleName;
            ViewData["viewName"] = viewName;
            return PartialView("Controls/News/" + viewName + ".ascx", pds);
        }

        /// <summary>
        /// 首页资讯模块
        /// </summary>
        /// <returns></returns>        
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListNewsSildeBlock(int? commendTypeID, int displayCount)
        {
            //置顶资讯
            IList<NewsThread> stickyNewsThread = NewsThreads.GetTopThreadsSortBy(NewThreadsSortBy.StickyAndPostDate, -1, displayCount);

            return PartialView("Controls/News/ListNewsSildeBlock.ascx", stickyNewsThread);
        }



        /// <summary>
        /// 资讯首页资讯模块
        /// </summary>
        /// <returns></returns>        
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListNewsSilde(int? commendTypeID, int displayCount)
        {
            //幻灯片
            IList<RecommendedItem> pdsSlide = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().News(), commendTypeID, 5);
            List<NewsThread> listSlideNews = new List<NewsThread>();
            foreach (var item in pdsSlide)
            {
                NewsThread thread = NewsThreads.GetThread(item.ItemID, false);
                if (thread == null)
                    continue;

                listSlideNews.Add(thread);
            }

            ViewData["listSlideNews"] = listSlideNews;

            //置顶资讯
            IList<NewsThread> stickyNewsThread = NewsThreads.GetTopThreadsSortBy(NewThreadsSortBy.StickyAndPostDate, -1, displayCount);

            return PartialView("Controls/News/ListNewsSilde.ascx", stickyNewsThread);
        }

        /// <summary>
        /// 设置置顶状态
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="isSticky"></param>
        /// <returns></returns>
        public ActionResult SetSticky(int threadID, bool isSticky)
        {
            NewsThread thread = NewsThreads.GetThread(threadID, true);
            if (thread == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ArticleNoFound", ApplicationIDs.Instance().News()), threadID)));

            if (CurrentUser != null && CurrentUser.IsContentAdministrator)
            {
                if (isSticky)
                    thread.SpecialOrder = 1;
                else
                    thread.SpecialOrder = 0;

                NewsThreads.UpdateStickyStatuses(new List<int>() { thread.ThreadID }, isSticky);
            }

            return new EmptyResult();
        }

        /// <summary>
        /// 按照类别显示列表页（包含子版块）
        /// </summary>
        /// <param name="sectionID"></param>
        /// <returns></returns>
        public ActionResult ListNewsForChild(int sectionID)
        {
            //当前栏目
            NewsSection section = NewsSections.GetSection(sectionID, false);
            if (section == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ListNotFound", applicationID), sectionID)));

            //栏目或子栏目下资讯
            if (section.ChildCount > 0)
            {
                List<NewsSection> childList = NewsSections.GetSections(sectionID, false, NewsSectionsOrganizeOptions.None, NewsSectionsSortBy.DisplayOrder, SortOrder.Ascending);
                if (childList.Count == 0)
                    return RedirectToAction("ListNews", new RouteValueDictionary { { "sectionID", sectionID } });

                Dictionary<NewsSection, IList<NewsThread>> dict = new Dictionary<NewsSection, IList<NewsThread>>();

                foreach (NewsSection childSection in childList)
                {
                    IList<NewsThread> childThreads = NewsThreads.GetTopThreadsSortBy(NewThreadsSortBy.StickyAndPostDate, childSection.SectionID, 5);
                    dict.Add(childSection, childThreads);
                }

                ViewData["sectionThread"] = dict;
            }
            else
            {
                return RedirectToAction("ListNews", new RouteValueDictionary { { "sectionID", sectionID } });
            }

            ViewData["section"] = section;
            ViewData["SecondaryID"] = sectionID;

            #region 生成面包屑
            BreadCrumb.AddBreadCrumbItem(section.GetMultilingualSectionName(CurrentUserLanguage), SPBUrlHelper.Action("ListNewsForChild", "ChannelNews", new RouteValueDictionary { { "sectionID", section.SectionID } }));

            //ShowCategoryBreadCrumb(sectionID);
            #endregion

            SetPageTitle(GetResourceString("Title_NewsList", applicationID));
            //置顶资讯
            IList<NewsThread> stickyNewsThread = NewsThreads.GetTopThreadsSortBy(NewThreadsSortBy.StickyAndPostDate, sectionID, 11);

            string pageName = section.PageShowSection;
            if (string.IsNullOrEmpty(pageName))
                pageName = "ListNewsForChild";

            return View("Pages/News/" + pageName + ".aspx", "Masters/News.master", stickyNewsThread);
        }

        /// <summary>
        /// 生成活动类别面包屑
        /// </summary>
        private void ShowCategoryBreadCrumb(int sectionID)
        {
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_News", applicationID), SiteUrls.Instance().ChannelNewsHome());
            Stack<NewsSection> nsection = new Stack<NewsSection>();
            NewsSection section = NewsSections.GetSection(sectionID, false);
            while (section != null)
            {
                nsection.Push(section);
                section = NewsSections.GetSection(section.ParentID, false);
            }
            while (nsection.Count > 0)
            {
                section = nsection.Pop();
                BreadCrumb.AddBreadCrumbItem(section.GetMultilingualSectionName(CurrentUserLanguage), SPBUrlHelper.Action("ListNewsForChild", "ChannelNews", new RouteValueDictionary { { "sectionID", section.SectionID } }));
            }
        }

        /// <summary>
        /// 按照类别显示列表页
        /// </summary>
        /// <param name="sectionID"></param>
        public ActionResult ListNews(int sectionID, int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_NewsList", applicationID));

            //置顶资讯
            IList<NewsThread> stickyThreads = NewsThreads.GetTopThreadsSortBy(NewThreadsSortBy.StickyAndPostDate, sectionID, 1);
            NewsThread topThread = null;
            if (stickyThreads.Count > 0)
            {
                topThread = stickyThreads[0];
            }
            ViewData["topThread"] = topThread;

            //当前栏目
            NewsSection section = NewsSections.GetSection(sectionID, false);
            #region 生成面包屑
            BreadCrumb.AddBreadCrumbItem(section.GetMultilingualSectionName(CurrentUserLanguage), SPBUrlHelper.Action("ListNewsForChild", "ChannelNews", new RouteValueDictionary { { "sectionID", section.SectionID } }));

            //ShowCategoryBreadCrumb(sectionID);
            #endregion
            if (section == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ListNotFound", applicationID), sectionID)));
                // throw new SPBException(ExceptionType.ResourceNotFound);
            }
            ViewData["section"] = section;

            //栏目下资讯

            bool includeSectionDescendant = false;
            if (section.ChildCount > 0)
                includeSectionDescendant = true;

            if (!pageIndex.HasValue || pageIndex.Value <= 0)
                pageIndex = 1;

            PagingDataSet<NewsThread> allpds = NewsThreads.GetThreadsInSection(sectionID, includeSectionDescendant, pageIndex.Value);
            ViewData["SecondaryID"] = sectionID;

            string pageName = section.PageList;
            if (string.IsNullOrEmpty(pageName))
                pageName = "ListNews";

            return View("Pages/News/" + pageName + ".aspx", "Masters/News.master", allpds);
        }

        /// <summary>
        /// 最新推荐模块
        /// </summary>
        /// <returns></returns>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.RelativelyStable)]
        public ActionResult Control_ListCommendsBlock()
        {
            IList<RecommendedItem> pds = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().News(), 201003, 6);
            ViewData["RecommendationType"] = RecommendationTypes.GetRecommendationType(201003, false);

            return PartialView("Controls/News/ListCommendsBlock.ascx", pds);
        }

        /// <summary>
        /// 图片资讯模块
        /// </summary>
        /// <param name="commendTypeID">推荐类别ID</param>
        /// <param name="viewName">控件名称（不加后缀）</param>
        /// <param name="titleName">控件标题</param>
        /// <param name="topNumber">显示几条</param>
        /// <returns></returns>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListPhotoNewsBlock(int commendTypeID, string viewName, string titleName, int topNumber)
        {
            IList<RecommendedItem> pds = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().News(), commendTypeID, topNumber);
            List<NewsThread> listPhotoNews = new List<NewsThread>();

            foreach (var item in pds)
            {
                NewsThread thread = NewsThreads.GetThread(item.ItemID, false);
                if (thread == null)
                    continue;

                listPhotoNews.Add(thread);
            }

            ViewData["RecommendationType"] = RecommendationTypes.GetRecommendationType(commendTypeID, false);
            ViewData["TitleName"] = titleName;

            return PartialView("Controls/News/" + viewName + ".ascx", listPhotoNews);


        }

        /// <summary>
        /// 图片资讯页
        /// </summary>
        /// <returns></returns>
        public ActionResult ListPhotoNews(int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_PhotoNews", applicationID));

            PagingDataSet<RecommendedItem> pds = RecommendedItems.GetRecommendedItems(36, pageIndex ?? 1, ApplicationIDs.Instance().News(), 201002);
            List<NewsThread> listPhotoNews = new List<NewsThread>();

            foreach (var item in pds.Records)
            {
                NewsThread thread = NewsThreads.GetThread(item.ItemID, false);
                if (thread == null)
                    continue;

                listPhotoNews.Add(thread);
            }
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_PhotoNews", ApplicationIDs.Instance().News()), string.Empty);

            PagingDataSet<NewsThread> pdsPhoto = new PagingDataSet<NewsThread>();
            pdsPhoto.PageIndex = pds.PageIndex;
            pdsPhoto.PageSize = pds.PageSize;
            pdsPhoto.Records = listPhotoNews;
            pdsPhoto.TotalRecords = pds.TotalRecords;

            return View("Pages/News/ListPhotoNews.aspx", "Masters/News.master", pdsPhoto);
        }

        /// <summary>
        /// 资讯内容块
        /// </summary>
        /// <param name="sectionID">栏目ID</param>
        /// <param name="sortBy">排序方式</param>
        /// <param name="viewName">控件名称（不加后缀）</param>
        /// <param name="titleName">控件标题（不填写时不显示）</param>
        /// <param name="topNumber">显示条数</param>
        /// <returns></returns>
        public ActionResult Control_ListNewsBlock(int? sectionID, NewThreadsSortBy? sortBy, string viewName, string titleName, int topNumber)
        {
            if (!sectionID.HasValue || sectionID.Value <= 0)
                sectionID = -1;
            IList<NewsThread> threads = NewsThreads.GetTopThreadsSortBy(sortBy ?? NewThreadsSortBy.HitTimes, sectionID.Value, topNumber);
            ViewData["TitleName"] = titleName;
            ViewData["sortBy"] = sortBy ?? NewThreadsSortBy.HitTimes;
            return PartialView("Controls/News/" + viewName + ".ascx", threads);
        }

        /// <summary>
        /// 资讯详细页
        /// </summary>
        /// <returns></returns>
        public ActionResult ShowNews(int newsID, int? pageIndex, int? cPageIndex)
        {
            NewsThread thread = null;
            if (CurrentUser != null && (CurrentUser.IsContentAdministrator || CurrentUser.IsContentAdministrator))
                thread = NewsThreads.GetThread(newsID, true);
            else
                thread = NewsThreads.GetThread(newsID, false);

            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NewsThemeNotFound", applicationID), newsID)));
            }
            SetPageTitle(thread.Title, GetResourceString("Title_News", applicationID));

            #region 生成面包屑
            ShowCategoryBreadCrumb(thread.SectionID);
            BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(thread.Title, 20), SiteUrls.Instance().ShowNews(thread.ThreadID));
            #endregion

            List<Attachment> attachmentList = NewsAttachmentManager.Instance().GetAttachmentsByAssociateID(newsID, false);
            ViewData["attachmentList"] = attachmentList;

            #region 处理审核
            int auditingStatusValue = (int)Applications.GetApplication(ApplicationIDs.Instance().News()).AuditingStatusForPublicDisplay;
            if ((int)thread.AuditingStatus < auditingStatusValue)
            {
                if (thread.AuditingStatus == AuditingStatuses.Fail)
                    ViewData["errorMessage"] = GetResourceString("Message_NewsWithoutAuditing", applicationID);
                else
                    ViewData["errorMessage"] = GetResourceString("Message_NewsAuditing", applicationID);
                ViewData["errorAuditingStatus"] = true;
            }
            #endregion

            string blogThreadBody = string.Empty;
            if (CurrentUser != null && CurrentUser.IsAdministrator || CurrentUser != null && CurrentUser.IsContentAdministrator || CurrentUser != null && CurrentUser.UserID == thread.UserID)
            {
                ViewData["hasManagePermission"] = true;
                blogThreadBody = thread.GetBody(true);
            }
            else
                blogThreadBody = thread.GetBody(false);

            string[] contents = blogThreadBody.Split(new string[] { pageSeparator }, StringSplitOptions.RemoveEmptyEntries);
            if (contents != null && contents.Length > 1)
            {
                if (!cPageIndex.HasValue || cPageIndex.Value > contents.Length + 1)
                    cPageIndex = 1;

                blogThreadBody = contents[cPageIndex.Value - 1];
                ViewData["totalPage"] = contents.Length;
            }

            ViewData["blogThreadBody"] = blogThreadBody;


            if (CurrentUser == null || (thread.UserID != CurrentUser.UserID && !CurrentUser.IsContentAdministrator))
            {
                if (CurrentUser == null && thread.AllowededUserRank > 0)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                    TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_PleaseRead", applicationID) + "<a href=\"{0}\">" + GetResourceString("Action_Login") + "</a>", SiteUrls.Instance().Login(true));
                }
                else if (CurrentUser != null && CurrentUser.Rank < thread.AllowededUserRank)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                    TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_ReadNeedLevel", applicationID), thread.AllowededUserRank, CurrentUser.Rank);
                }
                else
                {
                    //判断阅读积分
                    if (thread.ReadingNeedPoint)
                    {
                        if (CurrentUser == null)
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                            TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_PleaseRead", applicationID) + "<a href=\"{0}\">" + GetResourceString("Action_Login") + "</a>", SiteUrls.Instance().Login(true));
                        }
                        else
                        {
                            if (!IsCookieExistNews(newsID))
                            {
                                if (!IsDBExistNews(newsID))
                                {
                                    if (CurrentUser.TradePoints < thread.PointConsumption)
                                    {
                                        TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                                        TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_ReadSpareIntegral", applicationID), thread.PointConsumption, CurrentUser.TradePoints);
                                    }
                                    else
                                    {
                                        TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                                        TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_ReadWillCutPoint", applicationID), CurrentUser.TradePoints, thread.PointConsumption);
                                    }
                                }
                                else
                                {
                                    UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
                                    userCookie.SaveReadedNewsThreadIDs(newsID);
                                }
                            }
                        }
                    }
                }
            }

            #region 浏览计数

            if (TempData["StatusMessageType"] == null)
            {
                CounterFactory.IncreaseCount<NewsThreadCounter>(IncreasableCountTypes.WebView, newsID);
            }

            #endregion

            #region 设置Meta标签

            string metaDescription = string.Empty;
            metaDescription = string.Format("{0}：{1}", thread.Title, StringUtils.Trim(thread.ForcedExcerpt, 150));
            Header.AddMetaKeywords(thread.Keywords);
            Header.AddMetaDescription(metaDescription);

            #endregion


            #region 相关文章

            //List<NewsThread> correlativeNewsThreads = new List<NewsThread>();
            //List<int> correlativeNewsThreadIDs = new List<int>();
            //SearchResultDataSet<NewsThread> srs;

            //NewsThreadFullTextQuery fullTextQuery = new NewsThreadFullTextQuery();
            //fullTextQuery.PageSize = 11;
            //if (thread.Tags != null && thread.Tags.Count > 0)
            //{
            //    string[] tagNames = new string[thread.Tags.Count];
            //    for (int i = 0; i < thread.Tags.Count; i++)
            //    {
            //        tagNames[i] = thread.Tags[i].TagName;
            //    }
            //    fullTextQuery.TagNames = tagNames;

            //    try
            //    {
            //        srs = NewsSearchManager.Instance().Search(fullTextQuery);
            //        if (srs != null)
            //        {
            //            correlativeNewsThreads.AddRange(srs.Records);
            //            foreach (var correlativeNewsThread in correlativeNewsThreads)
            //            {
            //                correlativeNewsThreadIDs.Add(correlativeNewsThread.ThreadID);
            //            }
            //        }
            //    }
            //    catch { }
            //}

            ////至少保证6条相关资讯
            //if (correlativeNewsThreads.Count < 6)
            //{
            //    fullTextQuery.TagNames = null;
            //    fullTextQuery.Keyword = thread.Title;
            //    fullTextQuery.SectionID = thread.SectionID;

            //    try
            //    {
            //        srs = NewsSearchManager.Instance().Search(fullTextQuery);
            //        if (srs != null)
            //        {
            //            foreach (var _thread in srs.Records)
            //            {
            //                if (!correlativeNewsThreadIDs.Contains(_thread.ThreadID))
            //                    correlativeNewsThreads.Add(_thread);

            //                if (correlativeNewsThreads.Count > 11)
            //                    break;
            //            }
            //        }
            //    }
            //    catch { }
            //}

            #endregion
            //ViewData["RelatedNewsThread"] = correlativeNewsThreads;

            IList<Rating> ratings = NewsThreads.GetRatings(newsID, 30);
            ViewData["ratings"] = ratings;

            string pageName = string.Empty;
            if (thread.Section != null)
                pageName = thread.Section.PageDetail;
            if (string.IsNullOrEmpty(pageName))
                pageName = "ShowNews";

            return View("Pages/News/" + pageName + ".aspx", "Masters/News.master", thread);
        }

        /// <summary>
        /// 相关资讯
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_CorrelativeNewsThreads(int threadID, string title)
        {
            NewsThread thread = NewsThreads.GetThread(threadID, false);
            ICollection<NewsThread> correlativeNewsThreads;
            try
            {
                correlativeNewsThreads = NewsSearchManager.Instance().GetCorrelativeThreads(thread, 10);
            }
            catch
            {
                correlativeNewsThreads = new List<NewsThread>();
            }

            return PartialView("Controls/News/CorrelativeNewsThreads.ascx", correlativeNewsThreads);
        }

        /// <summary>
        /// 评论列表
        /// </summary>
        /// <returns></returns>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListComments(int newsID, int? pageIndex)
        {
            NewsThread thread = NewsThreads.GetThread(newsID, false);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ReferNotFound", applicationID), newsID)));
            }

            if (!pageIndex.HasValue || pageIndex.Value <= 0)
                pageIndex = 1;
            PagingDataSet<NewsComment> pds = null;
            if (CurrentUser != null && (thread.UserID == CurrentUser.UserID || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
            {
                pds = NewsComments.GetCommentsOfThread(newsID, pageIndex.Value, true);
            }
            else
            {
                pds = NewsComments.GetCommentsOfThread(newsID, pageIndex.Value, false);
            }
            ViewData["thread"] = thread;

            return PartialView("Controls/News/ListComments.ascx", pds);
        }

        /// <summary>
        /// 创建评论
        /// </summary>
        /// <param name="newsID"></param>
        /// <returns></returns>
        public ActionResult CreateComment(int newsID, int? pageIndex)
        {
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                if (verifyCode != null)
                    verifyCode = verifyCode.Trim();

                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
                    TempData["commentBody"] = Request.Form.GetString("commentBody", string.Empty).Trim();
                    return RedirectToAction("Control_ListComments", new RouteValueDictionary { { "newsID", newsID }, { "pageIndex", pageIndex } });
                }
            }
            NewsThread thread = NewsThreads.GetThread(newsID, false);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NewsThemeNotFound", applicationID), newsID)));
                //throw new SPBException(ExceptionType.AccessDenied);
            }

            if (!NewsPermission.AccessCheck(thread, CurrentUser, PermissionItemKeys.Instance().CreateNewsComment(), HttpContext))
                return new EmptyResult();

            NewsComment comment = new NewsComment();
            comment.Body = Request.Form.GetString("commentBody", string.Empty).Trim();
            comment.ThreadID = newsID;
            comment.SectionID = thread.SectionID;
            comment.Author = CurrentUser.DisplayName;
            comment.UserID = CurrentUser.UserID;
            comment.ParentID = Request.Form.GetInt("parentID", 0);
            if (comment.ParentID > 0)
            {

                NewsComment parentComment = NewsComments.GetComment(comment.ParentID, false);
                if (parentComment != null)
                {
                    comment.ParentAuthor = parentComment.Author;
                    comment.ParentBody = parentComment.Body;
                    comment.ParentUserID = parentComment.UserID;
                    comment.ParentPostDate = parentComment.PostDate;
                    if (comment.Body.StartsWith(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author), StringComparison.CurrentCultureIgnoreCase))
                    {
                        if (comment.Body.Length > string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author).Length)
                        {
                            comment.Body = comment.Body.Substring(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author).Length);
                        }
                        else
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = GetResourceString("Message_ImportReplyContent", applicationID);
                            TempData["commentBody"] = comment.Body;
                            TempData["parentID"] = comment.ParentID;
                            return RedirectToAction("Control_ListComments", new RouteValueDictionary { { "newsID", newsID }, { "pageIndex", pageIndex } });
                        }
                    }
                }
            }

            try
            {
                #region 检查发帖时间间隔,防灌水
                Globals.CheckPostDate();
                #endregion
                NewsComments.CreateComment(comment);
            }
            catch (SPBException e)
            {
                ExceptionMessage exceptionMessage = ResourceManager.GetExceptionMessage(e.ExceptionType, ApplicationIDs.Instance().News());
                if (exceptionMessage != null)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = exceptionMessage.Title + "，" + exceptionMessage.Body;
                }
                return RedirectToAction("Control_ListComments", new RouteValueDictionary { { "newsID", newsID }, { "pageIndex", pageIndex } });
            }

            if ((int)comment.AuditingStatus >= (int)SiteSettingsManager.GetSiteSettings().AuditingStatusesForPublicDisplay)
            {
                TempData["StatusMessageContent"] = GetResourceString("Message_AddCommentSucceed", applicationID);
            }
            else
            {
                TempData["StatusMessageContent"] = GetResourceString("Message_ShowAfterAuditing", applicationID);
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;

            return RedirectToAction("Control_ListComments", new RouteValueDictionary { { "newsID", newsID }, { "pageIndex", pageIndex } });
            //return Control_ListComments(newsID, pageIndex);
        }

        /// <summary>
        /// 删除评论
        /// </summary>
        /// <param name="newsID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public ActionResult DeleteComment(int commentID, int newsID, int? pageIndex)
        {
            NewsComment comment = NewsComments.GetComment(commentID, true);
            if (comment == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_CommentNotFound", applicationID), commentID)));
                //throw new SPBException(ExceptionType.AccessDenied);
            }
            else if (CurrentUser != null && comment.UserID == CurrentUser.UserID || CurrentUser.IsContentAdministrator)
            {
                NewsComments.DeleteComment(commentID);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSucecssful", applicationID);
            }
            else
            {
                // throw new SPBException(ExceptionType.AdministrationAccessDenied);
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_DeleteFail", applicationID);
            }

            return RedirectToAction("Control_ListComments", new RouteValueDictionary { { "newsID", newsID }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 删除资讯
        /// </summary>
        /// <param name="newsID"></param>
        /// <returns></returns>
        public ActionResult DeleteNews(int newsID)
        {

            NewsThread thread = NewsThreads.GetThread(newsID, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NewsThemeNotFound", applicationID), newsID)));
            }

            if (!NewsPermission.AccessCheck(thread, CurrentUser, PermissionItemKeys.Instance().DeleteNewsThread(), HttpContext))
                return new EmptyResult();

            NewsThreads.DeleteThread(thread);
            return RedirectToAction("Home");
        }

        /// <summary>
        /// 更新评论审核状态
        /// </summary>
        /// <param name="commentID"></param>
        /// <param name="newsID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public ActionResult UpdateCommentAuditingStatus(int commentID, int newsID, int? pageIndex)
        {
            NewsComment comment = NewsComments.GetComment(commentID, true);
            if (comment == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_CommentNotFound", applicationID), commentID)));
            }
            else if (CurrentUser.IsContentAdministrator)
            {
                List<int> list = new List<int>();
                list.Add(commentID);
                NewsComments.UpdateAuditingStatus(list, AuditingStatuses.Success);
            }
            else
            {
                throw new SPBException(ExceptionTypes.Instance().AdministrationAccessDenied());
            }

            return RedirectToAction("Control_ListComments", new RouteValueDictionary { { "newsID", newsID }, { "pageIndex", pageIndex } });
        }

        #region Tag

        /// <summary>
        /// 标签地图Module
        /// </summary>
        /// <returns></returns>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_TagMapBlock(int? topNumber)
        {
            Dictionary<string, KeyValuePair<int, ApplicationTag>> applicationTags = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).GetTopApplicationTags(topNumber ?? 15, ApplicationTagsSortBy.ItemCount, null);

            return PartialView("Controls/News/TagMapBlock.ascx", applicationTags);
        }

        /// <summary>
        /// 标签地图页
        /// </summary>
        /// <returns></returns>
        public ActionResult TagMap()
        {
            SetPageTitle(GetResourceString("Title_TagMap", applicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_TagMap", applicationID), string.Empty);

            Dictionary<string, KeyValuePair<int, ApplicationTag>> applicationTags = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).GetTopApplicationTags(300, ApplicationTagsSortBy.ItemCount, null);

            return View("Pages/News/TagMap.aspx", "Masters/News.master", applicationTags);
        }

        /// <summary>
        /// 标签下资讯
        /// </summary>
        /// <returns></returns>
        public ActionResult ListNewsForTag(string tagName, int? pageIndex)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = WebUtils.UrlDecode(tagName);
            SetPageTitle(GetResourceString("Title_NewsTag", applicationID));
            BreadCrumb.AddBreadCrumbItem(tagName, string.Empty);

            if (!pageIndex.HasValue || pageIndex.Value <= 0)
                pageIndex = 1;

            PagingDataSet<NewsThread> pds = NewsThreads.GetThreadsInSiteTag(tagName, true, pageIndex.Value);

            return View("Pages/News/ListNewsForTag.aspx", "Masters/News.master", pds);
        }

        #endregion

        #region Rating

        /// <summary>
        /// 对资讯投稿进行星级评价
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Rate()
        {
            //资讯投稿ThreadID
            int itemID = Request.QueryString.GetInt("itemID", -1);

            //评价的星级
            int value = Request.QueryString.GetInt("value", -1);


            ContentResult result = new ContentResult();
            result.ContentType = "text/html";
            if (!this.HttpContext.User.Identity.IsAuthenticated)
            {
                result.Content = GetResourceString("Message_PleaceLogin");
            }
            else
            {
                if (CurrentUser != null && CurrentUser != null && itemID > 0 && value > 0)
                {
                    Rating rating = new Rating();
                    rating.UserID = CurrentUser.UserID;
                    rating.Author = CurrentUser.DisplayName;
                    rating.ItemID = itemID;
                    rating.Rate = value;
                    NewsThreads.Rate(rating);

                    result.Content = GetResourceString("Message_ThanksForRating");
                }
                else
                {
                    result.Content = GetResourceString("Message_ParameterIncorrect");
                }
            }
            return result;
        }

        /// <summary>
        /// 显示评价人列表
        /// </summary>
        public ActionResult Control_ListRatingUsers(int itemID, int? topNumber)
        {
            IList<Rating> ratings = NewsThreads.GetRatings(itemID, topNumber ?? 30);
            return View("Controls/Common/ListRatingUsers.ascx", ratings);
        }


        #endregion

        #region FullTextSearch

        /// <summary>
        /// 资讯全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Search(int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_SearchNews", applicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_SearchNews", applicationID), string.Empty);

            NewsThreadFullTextQuery query = new NewsThreadFullTextQuery();
            query.ConvertFromQueryString(Request);
            query.PageSize = 10;

            if (pageIndex.HasValue)
                query.PageIndex = pageIndex.Value;

            SearchResultDataSet<NewsThread> results = null;
            if (query.IsValid())
            {
                results = NewsSearchManager.Instance().Search(query);
            }
            else
                results = new SearchResultDataSet<NewsThread>();
            ViewData["Query"] = query;
            return this.View("Pages/News/Search.aspx", "Masters/News.master", results);
        }

        /// 资讯全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Search()
        {
            RouteValueDictionary parameters = new RouteValueDictionary();
            parameters["Keyword"] = Request.Form.GetString("Keyword", string.Empty).Trim();

            int searchScopeUserID = Request.Form.GetInt("SearchScopeUserID", -1);
            if (searchScopeUserID > 0)
                parameters["SearchScopeUserID"] = searchScopeUserID.ToString();

            return RedirectToAction("Search", parameters);
        }

        #endregion

        #region 更新资讯审核状态

        public ActionResult UpdateNewsShareThread(int threadid)
        {
            List<int> threadIds = new List<int>();
            threadIds.Add(threadid);
            NewsThreads.UpdateAuditingStatuses(threadIds, AuditingStatuses.Success);
            return RedirectToAction("ShowNews", new RouteValueDictionary { { "newsID", threadid } });
        }

        #endregion

        #region SiteCategory

        /// <summary>
        /// 获取所有的根栏目
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_SubMenus()
        {
            List<NewsSection> newsSections = NewsSections.GetSections(null, true, NewsSectionsOrganizeOptions.None, NewsSectionsSortBy.DisplayOrder, SortOrder.Ascending);
            StringBuilder sb = new StringBuilder();
            foreach (NewsSection section in newsSections)
            {
                sb.AppendFormat("<li class=\"spb-nav-item\"><a {0} href=\"{1}\"><span>{2}</span></a>", section.ChildCount > 0 ? "class=\"sf-with-ul\"" : "", section.ChildCount > 0 ? Globals.GetFullUrlInMainSite(SPBUrlHelper.Action("ListNewsForChild", "ChannelNews", new RouteValueDictionary { { "sectionID", section.SectionID } })) : Globals.GetFullUrlInMainSite(SPBUrlHelper.Action("ListNews", "ChannelNews", new RouteValueDictionary { { "sectionID", section.SectionID } })), section.GetMultilingualSectionName(CurrentUserLanguage));
                if (section.ChildCount > 0)
                    sb.Append(GetChildCategoriesTag(section.SectionID));
                sb.Append("</li>");
            }
            return PartialView("Controls/News/SubMenus.ascx", sb.ToString());
        }

        /// <summary>
        /// 递归返回子栏目的标签
        /// </summary>
        /// <param name="categoryID"></param>
        /// <returns></returns>
        private string GetChildCategoriesTag(int sectionID)
        {
            List<NewsSection> sections = NewsSections.GetChilds(sectionID);
            if (sections != null && sections.Count > 0)
            {
                StringBuilder sb = new StringBuilder();

                sb.Append("<ul class=\"spb-nav-submenu\" style=\"disply:none;visibility:hidden\">");
                foreach (NewsSection childSection in sections)
                {
                    sb.AppendFormat("<li class=\"spb-nav-item\"><a {0} href=\"{1}\"><span>{2}</span></a>", childSection.ChildCount > 0 ? "class=\"sf-with-ul\"" : "", Globals.GetFullUrlInMainSite(SPBUrlHelper.Action("ListNewsForChild", "ChannelNews", new RouteValueDictionary { { "sectionID", childSection.SectionID } })), childSection.GetMultilingualSectionName(CurrentUserLanguage));
                    if (childSection.ChildCount > 0)
                        sb.Append(GetChildCategoriesTag(childSection.SectionID));
                    sb.Append("</li>");
                }
                sb.Append("</ul>");

                return sb.ToString();
            }

            return string.Empty;
        }

        #endregion

        #endregion

        #region NewsAttachment
        /// <summary>
        /// 管理资讯主题附件
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_UploadNewsAttachment(int? threadID, bool? isImage, string queueCompleteCallback)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            UploadAttachmentSettings settings = new UploadAttachmentSettings();
            NewsAttachmentSettings attachmentSettings = (NewsAttachmentSettings)NewsAttachmentManager.Instance().Settings;
            string uploadPhotoTypes = NewsAttachmentManager.Instance().Settings.Extensions;
            IList<string> imageExtensions = MimeTypeConfiguration.GetImageExtensions();
            StringBuilder newUploadPhotoTypes = new StringBuilder();
            string[] types = uploadPhotoTypes.Split(',');
            foreach (string type in types)
            {
                if (!string.IsNullOrEmpty(type))
                {
                    if (isImage.HasValue && isImage.Value == imageExtensions.Contains(type.ToLower()))
                        newUploadPhotoTypes.AppendFormat("*.{0};", type);
                }
            }
            settings.AssociateID = threadID ?? 0;
            settings.FileTypes = newUploadPhotoTypes.ToString();
            settings.UploadUrl = Globals.GetFullUrlInFileHandlerSite(NewsAttachmentManager.Instance().StorageProviderName, WebUtils.ResolveUrl("~/Services/News/UploadAttachments.ashx"));
            settings.FlashUrl = Globals.GetFullUrlInFileHandlerSite(NewsAttachmentManager.Instance().StorageProviderName, WebUtils.ResolveUrl("~/Utility/UploadFiles/swfupload.swf"));
            settings.FileQueryLimit = attachmentSettings.MaxFileCountATime;
            settings.FileSizeLimit = attachmentSettings.MaxAttachmentSize.ToString();
            settings.QueueCompleteCallback = queueCompleteCallback;
            settings.IsImage = isImage ?? true;

            ViewData["UploadAttachmentSettings"] = settings;
            return View("Controls/UploadAttachmentInEditer.ascx", settings);
        }

        /// <summary>
        /// 管理资讯主题附件
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ManageThreadAttachments(int? threadID, bool? isImage)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            ManageThreadAttachmentsModel model = new ManageThreadAttachmentsModel();
            model.IsImage = isImage ?? true;

            if (threadID.HasValue && threadID.Value > 0)
            {
                NewsThread newsThread = NewsThreads.GetThread(threadID.Value, false);
                if (newsThread == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_Pattern_VisitNewsArticleNoFound", applicationID)));
                }

                IList<Attachment> attachments = NewsAttachmentManager.Instance().GetAttachmentsByAssociateID(threadID.Value, true);
                model.Attachments = attachments;
                model.UserFeaturedImageAttachmentID = newsThread.FeaturedImageAttachmentID;
            }
            else
            {
                IList<Attachment> attachments = NewsAttachmentManager.Instance().GetTemporaryAttachmentsOfUser(CurrentUser.UserID);
                model.Attachments = attachments;
            }
            model.GetAttachmentUrl = SiteUrls.Instance().DownloadNewsAttachment;
            model.DeleteAttachmentUrl = SiteUrls.Instance().DeleteNewsAttachmentUrl;
            model.ShowAttachmentThumbnail = SPBHtmlHelper.Instance().ShowNewsAttachmentThumbnail;
            return PartialView("Controls/ManageThreadAttachmentsInEditer.ascx", model);
        }

        public ActionResult UserContentSelector_UpdateNewsAttachments(int? threadID)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            return PartialView("Controls/News/UserContentSelector_UpdateNewsAttachments.ascx");
        }

        #region Manage

        /// <summary>
        /// 删除资讯附件
        /// </summary>
        public ActionResult DeleteNewsAttachment(int attachmentID, bool? isImage)
        {

            Attachment attachment = NewsAttachmentManager.Instance().GetAttachment(attachmentID, true);
            if (attachment == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "找不到附件");
                return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID }, { "isImage", isImage } });
            }

            if (CurrentUser == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "请先登录");
                return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID }, { "isImage", isImage } });
            }

            if (attachment.AssociateID > 0)
            {
                NewsThread newsThread = NewsThreads.GetThread(attachment.AssociateID, false);
                if (newsThread == null)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "找不到附件所属的资讯");
                    return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID }, { "isImage", isImage } });
                }
            }

            User attachmentOwner = Users.GetUser(attachment.UserID, false);
            if (attachmentOwner == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "找不到附件的上传者");
                return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID }, { "isImage", isImage } });
            }

            if (CurrentUser.UserID != attachment.UserID && !CurrentUser.IsContentAdministrator)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "没有权限，只有上传者和网站管理员才能删除附件");
                return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID }, { "isImage", isImage } });
            }

            NewsAttachmentManager.Instance().DeleteAttachment(attachmentID);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "删除成功");

            return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID }, { "isImage", isImage } });
        }

        #endregion

        #endregion


        #region Help

        /// <summary>
        /// Cookie中是否存在
        /// </summary>
        /// <param name="newsID"></param>
        /// <returns></returns>
        private bool IsCookieExistNews(int newsID)
        {
            UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
            string readedNewsThreadIds = userCookie.GetReadedNewsThreadIDs();
            if (!string.IsNullOrEmpty(readedNewsThreadIds))
                return readedNewsThreadIds.IndexOf(newsID + ",") != -1;
            else
                return false;
        }

        /// <summary>
        /// 数据库中是否存在
        /// </summary>
        /// <param name="newsID"></param>
        /// <returns></returns>
        private bool IsDBExistNews(int newsID)
        {
            if (CurrentUser != null && NewsThreads.IsReader(newsID, CurrentUser.UserID))
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// 扣积分
        /// </summary>
        /// <param name="newsID"></param>
        /// <returns></returns>
        public ActionResult CutPoint(int newsID)
        {
            NewsThread thread = NewsThreads.GetThread(newsID, false);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NewsThemeNotFound", applicationID), newsID)));
                //throw new SPBException(ExceptionType.AccessDenied);
            }
            if (CurrentUser.UserID != newsID)
            {
                if (CurrentUser.Rank < thread.AllowededUserRank || CurrentUser.TradePoints < thread.PointConsumption)
                {
                    throw new SPBException(ExceptionTypes.Instance().AccessDenied());
                }
                else
                {
                    if (NewsThreads.CreateNewsReader(newsID, CurrentUser))
                    {
                        UserPointRecord userPointRecord = new UserPointRecord();
                        userPointRecord.UserID = CurrentUser.UserID;
                        userPointRecord.PointCategory = UserPointCategory.Trade;
                        userPointRecord.PointItemName = GetResourceString("Description_ReadNews", applicationID);
                        userPointRecord.Points = 0 - thread.PointConsumption;
                        userPointRecord.Description = string.Format(GetResourceString("Description_Pattern_ReadNews", applicationID), thread.Title);

                        Points.CreateUserPointRecord(userPointRecord);

                        UserPointRecord contributorPointRecord = new UserPointRecord();
                        contributorPointRecord.UserID = thread.UserID;
                        contributorPointRecord.PointCategory = UserPointCategory.Trade;
                        contributorPointRecord.PointItemName = GetResourceString("Description_NewsReaded", applicationID);
                        contributorPointRecord.Points = thread.PointConsumption;
                        contributorPointRecord.Description = string.Format(GetResourceString("Description_Pattern_NewsReaded", applicationID), thread.Title);

                        Points.CreateUserPointRecord(contributorPointRecord);

                        UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
                        userCookie.SaveReadedNewsThreadIDs(newsID);
                    }

                }
            }
            return RedirectToAction("ShowNews", new RouteValueDictionary { { "newsID", newsID } });
        }

        #endregion
    }
}
