﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using SpaceBuilder.Web;
using SpaceBuilder.Common;
using SpaceBuilder.Utils;
using System.Web.Routing;
using System.Text.RegularExpressions;
using System.Net;
using System.IO;
using System.Web.UI;

namespace SpaceBuilder.News.Controllers
{
    [ManageAuthorize]
    [HttpCompress]
    public class ManageNewsController : ManageControllerBase
    {
        int applicationID = ApplicationIDs.Instance().News();
        private static readonly string pageSeparator = "<!-- pagebreak -->";
        #region Section

        #region Show

        /// <summary>
        /// 管理栏目
        /// </summary>
        /// <returns></returns>
        public ActionResult ManageSections()
        {
            SetPageTitle(GetResourceString("Title_ManageNewsSection", applicationID));
            return View("Pages/News/ManageSections.aspx", "Masters/News.master");
        }

        /// <summary>
        /// 编辑栏目
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_EditSection(int? sectionID, int? parentID)
        {
            NewsSection section = new NewsSection();
            if (parentID.HasValue && (int)parentID > 0)
            {
                section.ParentID = parentID.Value;
            }
            if (sectionID.HasValue && (int)sectionID > 0)
            {
                section = NewsSections.GetSection(sectionID.Value, true);
                if (section == null)
                {
                    throw new SPBException(ExceptionTypes.Instance().ResourceNotFound());
                }
            }

            return PartialView("Controls/News/EditSection.ascx", section);
        }

        /// <summary>
        /// 列表显示栏目
        /// </summary>
        /// <returns></returns>+
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListSections()
        {
            List<NewsSection> sortList = NewsSections.GetSectionsForAdmin(null, false, NewsSectionsOrganizeOptions.Indented, ActiveStatuses.IsActive);

            return PartialView("Controls/News/ListSections.ascx", sortList);
        }

        /// <summary>
        /// 合并或移动栏目
        /// </summary>
        /// <param name="isMerge"></param>
        /// <param name="FilterID">需要排除当前栏目及子栏目</param>
        /// <returns></returns>
        public ActionResult Control_MergeOrMoveSection(bool isMerge, int filterID)
        {
            NewsSection fromSection = NewsSections.GetSection(filterID, true);
            if (fromSection == null)
            {
                throw new SPBException(ExceptionTypes.Instance().ResourceNotFound());
            }

            List<NewsSection> allSections = NewsSections.GetSectionsForAdmin(null, false, NewsSectionsOrganizeOptions.Indented, ActiveStatuses.IsActive);

            List<NewsSection> removedSections = NewsSections.GetAllChilds(filterID, true);
            removedSections.Add(fromSection);

            List<NewsSection> filteredSections = new List<NewsSection>();

            foreach (var section in allSections)
            {
                bool isMeet = false;
                foreach (var removedSection in removedSections)
                {
                    if (section.SectionID == removedSection.SectionID)
                    {
                        isMeet = true;
                        break;
                    }
                }

                if (isMeet)
                {
                    continue;
                }
                filteredSections.Add(section);
            }

            ViewData["isMerge"] = isMerge;
            ViewData["fromSection"] = fromSection;

            return PartialView("Controls/News/MergeOrMoveSection.ascx", filteredSections);
        }

        #endregion

        #region Manage

        /// <summary>
        /// 创建更新栏目
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateUpdateSection(int? sectionID, int parentID)
        {
            NewsSection section = new NewsSection();
            if (sectionID.HasValue && (int)sectionID > 0)
            {
                section = NewsSections.GetSection(sectionID.Value, true);
                if (section == null)
                {
                    throw new SPBException(ExceptionTypes.Instance().AccessDenied());
                }
            }
            else
            {
                section.ParentID = parentID;
            }
            section.SectionName = Request.Form.GetString("title", string.Empty).Trim();
            section.Description = Request.Form.GetString("body", string.Empty).Trim();
            section.DisplayOrder = Request.Form.GetInt("sortNumber", 0);
            section.PageDetail = Request.Form.GetString("PageDetail", string.Empty).Trim();
            section.PageList = Request.Form.GetString("PageList", string.Empty).Trim();
            section.PageShowSection = Request.Form.GetString("PageShowSection", string.Empty).Trim();

            if (sectionID.HasValue && (int)sectionID > 0)
            {
                NewsSections.Update(section);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            }
            else
            {
                if (SiteSettingsManager.GetSiteSettings().EnableCensorship)
                {
                    section.SectionName = Censors.CensorPost(section.SectionName);
                    section.Description = Censors.CensorPost(section.Description);
                }

                NewsSections.Create(section);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            }
            //更新地区名称语言包
            Dictionary<string, string> sectionNameRes = new Dictionary<string, string>();
            foreach (Language language in Languages.GetLanguages(true).Values)
            {
                string sectionName = Request.Form.GetString("sectionName_" + language.LanguageCode, string.Empty);
                if (!string.IsNullOrEmpty(sectionName))
                    sectionNameRes[language.LanguageCode] = sectionName;
            }
            if (sectionNameRes.Count > 0)
                NewsSections.UpdateSectionNameRes(section.SectionID, sectionNameRes);
            return RedirectToAction("Control_EditSection", new RouteValueDictionary { { "sectionID", section.SectionID } });
        }

        /// <summary>
        /// 删除栏目
        /// </summary>
        /// <param name="sectionID"></param>
        /// <returns></returns>
        public ActionResult DeleteSection(int sectionID)
        {
            GetChildListForDelete(sectionID);

            return RedirectToAction("Control_ListSections");
        }

        /// <summary>
        /// 递归删除List<NewsSection>
        /// </summary>
        /// <param name="sortlist"></param>
        /// <param name="parentID"></param>
        private void GetChildListForDelete(int sectionID)
        {
            NewsSections.Delete(sectionID);
            List<NewsSection> list = NewsSections.GetSectionsForAdmin(sectionID, false, NewsSectionsOrganizeOptions.None, ActiveStatuses.IsActive);
            foreach (var section in list)
            {
                GetChildListForDelete(section.SectionID);
            }
        }

        /// <summary>
        /// 合并或移动栏目及资讯
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult MergeOrMoveSection(bool isMerge, int fromSectionID)
        {
            int toSectionID = Request.Form.GetInt("sectionDDL", 0);
            if (isMerge)
            {
                NewsSections.MergeSection(fromSectionID, toSectionID);
                fromSectionID = toSectionID;
            }
            else
            {
                NewsSections.MoveSection(fromSectionID, toSectionID);
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;

            return RedirectToAction("Control_MergeOrMoveSection", new RouteValueDictionary { { "isMerge", true }, { "FilterID", fromSectionID } });
        }

        #endregion

        #endregion

        #region News

        #region Show

        /// <summary>
        /// 管理资讯
        /// </summary>
        /// <returns></returns>
        public ActionResult ManageNews(AuditingStatusesForDisplay? status, int? sectionID, int? pageIndex, string key)
        {
            SetPageTitle(GetResourceString("Title_ManageNews", applicationID));
            if (status == null)
                status = AuditingStatusesForDisplay.Pending;

            List<NewsSection> allSections = NewsSections.GetSectionsForAdmin(null, false, NewsSectionsOrganizeOptions.Indented, ActiveStatuses.IsActive);
            ViewData["allSections"] = allSections;

            if (!sectionID.HasValue || sectionID.Value <= 0)
                sectionID = -1;

            if (!pageIndex.HasValue || pageIndex.Value < 1)
                pageIndex = 1;

            PagingDataSet<NewsThread> pds = NewsThreads.GetThreadsForAdmin(15, pageIndex.Value, status, sectionID.Value, -1, key);

            ViewData["status"] = status;
            ViewData["sectionID"] = sectionID;

            return View("Pages/News/ManageNews.aspx", "Masters/News.master", pds);
        }

        /// <summary>
        /// 发布资讯
        /// </summary>
        /// <returns></returns>
        public ActionResult EditNews(int? newsID)
        {
            SetPageTitle(GetResourceString("Title_ReleaseNews", applicationID));
            NewsThread thread = new NewsThread();

            if (NewsSections.GetSections(null, true, NewsSectionsOrganizeOptions.None, NewsSectionsSortBy.DisplayOrder, SortOrder.Ascending).Count > 0)
            {
                ViewData["isSection"] = true;

                List<NewsSection> allSections = NewsSections.GetSectionsForAdmin(null, false, NewsSectionsOrganizeOptions.Indented, ActiveStatuses.IsActive);
                ViewData["allSections"] = allSections;

                Dictionary<int, string> dict = NewsThreads.GetTitlePrefixes(true);
                ViewData["dict"] = dict;

                if (newsID.HasValue && (int)newsID > 0)
                {
                    thread = NewsThreads.GetThread(newsID.Value, true);
                    if (thread == null)
                    {
                        throw new SPBException(ExceptionTypes.Instance().ResourceNotFound());
                    }
                }
                else
                {
                    if (TempData["thread"] != null)
                    {
                        thread = TempData["thread"] as NewsThread;
                    }
                }

                string tagList = string.Empty;
                foreach (var tag in thread.Tags)
                {
                    tagList = tagList + tag + ",";
                }

                ViewData["tagList"] = tagList;

            }

            return View("Pages/News/EditNews.aspx", "Masters/News.master", thread);
        }

        /// <summary>
        /// 设置资讯栏目
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_EditNewsSection()
        {
            string selectedThreadIDsString = Request.Form.GetString("newsCheckBoxGroup", string.Empty);
            if (!string.IsNullOrEmpty(selectedThreadIDsString))
            {
                ViewData["ListThreadID"] = selectedThreadIDsString;
            }
            else
            {
                ViewData["StatusMessageType"] = (int)StatusMessageType.Error;
                ViewData["StatusMessageContent"] = GetResourceString("Message_NoSelectedNews", applicationID);
                return PartialView("Controls/News/EditNewsSection.ascx");
            }

            List<NewsSection> sortList = NewsSections.GetSections(null, false, NewsSectionsOrganizeOptions.Indented, NewsSectionsSortBy.DisplayOrder, SortOrder.Ascending);

            return PartialView("Controls/News/EditNewsSection.ascx", sortList);
        }

        /// <summary>
        /// 搜索资讯
        /// </summary>
        /// <param name="status"></param>
        /// <param name="sectionID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SearchNews(AuditingStatusesForDisplay? status, int? sectionID)
        {
            string searchBodyKeywords = Request.Form.GetString("searchBodyKeywords", string.Empty).Trim();

            return RedirectToAction("ManageNews", new RouteValueDictionary { { "status", status }, { "sectionID", sectionID }, { "key", searchBodyKeywords } });
        }

        #endregion

        #region Manage

        /// <summary>
        /// 创建更新资讯
        /// </summary>
        /// <param name="newsID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateUpdateNews(int? newsID)
        {
            NewsThread thread = new NewsThread();
            if (newsID.HasValue && (int)newsID > 0)
            {
                thread = NewsThreads.GetThread(newsID.Value, true);
                if (thread == null)
                {
                    throw new SPBException(ExceptionTypes.Instance().AccessDenied());
                }
            }
            else
            {
                thread.UserID = CurrentUser.UserID;
                thread.Contributor = CurrentUser.DisplayName;
            }

            thread.SectionID = Request.Form.GetInt("sectionDDL", 0);
            thread.IsContributed = false;
            thread.PrefixID = Request.Form.GetInt("prefixDDL", 0);
            thread.Title = Request.Form.GetString("title", string.Empty).Trim();
            thread.Subtitle = Request.Form.GetString("subtitle", string.Empty).Trim();
            thread.Keywords = Request.Form.GetString("keywords", string.Empty).Trim();

            string metaKeyWords = string.Empty;
            List<string> tagNameList = new List<string>();
            string[] formattedTagNames = Request.Form.GetString("newsTags", string.Empty).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
            if (formattedTagNames != null && formattedTagNames.Length > 0)
            {
                foreach (string formattedTagName in formattedTagNames)
                {
                    if (ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).GetApplicationTag(formattedTagName.Trim(), true) == null)
                    {
                        ApplicationTag tag = new ApplicationTag();
                        tag.TagName = formattedTagName.Trim();
                        tag.UserID = CurrentUser.UserID;
                        tag.Author = CurrentUser.DisplayName;
                        tag.TagName = Formatter.CleanTagName(tag.TagName);
                        tag.TagName = Formatter.FormatSingleLinePlainTextForStorage(tag.TagName);

                        bool isCreated = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).CreateApplicationTag(tag);
                        if (isCreated)
                            tagNameList.Add(tag.TagName);
                    }
                    else
                    {
                        tagNameList.Add(ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).GetApplicationTag(formattedTagName, true).TagName);
                    }
                    if (string.IsNullOrEmpty(thread.Keywords))
                        metaKeyWords = string.Format("{1},{0}", metaKeyWords, formattedTagName);
                }
            }

            if (string.IsNullOrEmpty(thread.Keywords))
            {
                metaKeyWords = metaKeyWords + thread.Title;
                metaKeyWords = string.Format("{0},{1}", metaKeyWords, thread.Contributor);
                if (thread.Section != null)
                    metaKeyWords = string.Format("{0},{1}", metaKeyWords, thread.Section.SectionName);
                thread.Keywords = metaKeyWords;
            }

            bool isAutoPage = Request.Form.GetBool("isAutoPage", false);
            int pageSize = Request.Form.GetInt("pageSize", 0);
            string body = Request.Form.GetString("body", string.Empty);
            if (isAutoPage && pageSize > 0)
                body = string.Join(pageSeparator, ContentPages.GetPageContentForStorage(body, pageSize * 1000, true).ToArray());


            thread.OriginalAuthor = Request.Form.GetString("author", string.Empty).Trim();
            thread.CopyFrom = Request.Form.GetString("copyFrom", string.Empty).Trim();
            thread.Excerpt = Request.Form.GetString("excerpt", string.Empty).Trim();
            thread.BodyForDataTransmission = body;
            thread.AllowededUserRank = Request.Form.GetInt("allowededUserRank", 0);
            thread.EnableComments = Request.Form.GetBool("enableComments", true);
            thread.PointConsumption = Request.Form.GetInt("pointConsumption", 0);
            thread.IsDownloadRemoteImage = Request.Form.GetBool("downloadRemoteImage", false);

            bool isSticky = Request.Form.GetBool("isSticky", false);
            if (isSticky)
            {
                thread.SpecialOrder = 1;//在版块中置顶
                thread.StickyDate = DateTime.Now.AddDays(Request.Form.GetInt("stickyDays", 1));
            }
            else
            {
                thread.SpecialOrder = 0;//取消置顶
            }

            //设置标题图片
            int featurePicAttachmentID = Request.Form.GetInt("featurePicAttachmentID", -1);
            if (featurePicAttachmentID > 0)
                thread.FeaturedImageAttachmentID = featurePicAttachmentID;

            if (newsID.HasValue && (int)newsID > 0)
            {
                NewsThreads.UpdateThread(thread);
                ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).UpdateApplicationTagsOfItem(thread.ThreadID, tagNameList);
                return RedirectToAction("ManageNews");
            }
            else
            {
                NewsThreads.CreateThread(thread);
                ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).UpdateApplicationTagsOfItem(thread.ThreadID, tagNameList);
                if (thread.ThreadID > 0)
                {
                    return RedirectToAction("ManageNews", new RouteValueDictionary { { "status", AuditingStatusesForDisplay.Success } });
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_ReleaseNewsAborted", applicationID);
                    TempData["thread"] = thread;
                    return RedirectToAction("EditNews", new RouteValueDictionary { { "newsID", newsID } });
                }
            }
        }

        /// <summary>
        /// 删除资讯
        /// </summary>
        /// <param name="newsID"></param>
        /// <param name="status"></param>
        /// <param name="sectionID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public ActionResult DeleteNews(int newsID)
        {
            NewsThread thread = NewsThreads.GetThread(newsID, true);
            if (thread == null)
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }
            NewsThreads.DeleteThread(thread);

            if (Request.UrlReferrer != null)
            {
                return Redirect(Request.UrlReferrer.AbsoluteUri);
            }
            return RedirectToAction("ManageNews");
        }

        /// <summary>
        /// 批量删除资讯
        /// </summary>
        /// <param name="status"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteNewss()
        {
            string[] deletePostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("newsCheckBoxGroup", string.Empty)))
            {
                deletePostIDs = Request.Form.GetString("newsCheckBoxGroup", string.Empty).Split(',');
                if (deletePostIDs != null && deletePostIDs.Length > 0)
                {
                    int newsID = 0;
                    foreach (string postID in deletePostIDs)
                    {
                        int.TryParse(postID, out newsID);
                        NewsThread thread = NewsThreads.GetThread(newsID, true);
                        if (thread == null)
                            throw new SPBException(ExceptionTypes.Instance().AccessDenied());

                        NewsThreads.DeleteThread(thread);
                    }
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = GetResourceString("Message_BatchUpdateSuccess", applicationID);
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_NoSelectedNews", applicationID);
            }

            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 更新置顶状态
        /// </summary>
        /// <param name="status"></param>
        /// <param name="sectionID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateStickyStatuses(bool isSticky)
        {
            string[] deletePostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("newsCheckBoxGroup", string.Empty)))
            {
                deletePostIDs = Request.Form.GetString("newsCheckBoxGroup", string.Empty).Split(',');
                if (deletePostIDs != null && deletePostIDs.Length > 0)
                {
                    List<int> list = new List<int>();
                    foreach (string postID in deletePostIDs)
                    {
                        if (Convert.ToInt32(postID) > 0)
                            list.Add(Convert.ToInt32(postID));
                    }
                    NewsThreads.UpdateStickyStatuses(list, isSticky);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = GetResourceString("Message_BatchUpdateSuccess", applicationID);
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_NoSelectedNews", applicationID);
            }

            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 更新审核状态
        /// </summary>
        /// <param name="isPass"></param>
        /// <param name="status"></param>
        /// <param name="sectionID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateNewsAuditingStatuses(bool isPass)
        {
            string[] deletePostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("newsCheckBoxGroup", string.Empty)))
            {
                deletePostIDs = Request.Form.GetString("newsCheckBoxGroup", string.Empty).Split(',');
                if (deletePostIDs != null && deletePostIDs.Length > 0)
                {
                    List<int> list = new List<int>();
                    foreach (string postID in deletePostIDs)
                    {
                        int id = Convert.ToInt32(postID);
                        if (id > 0)
                        {
                            list.Add(id);

                            //发私信
                            NewsThread thread = NewsThreads.GetThread(id, true);
                            if (thread != null)
                            {
                                PersonUser user = Users.GetUser(thread.UserID, false) as PersonUser;
                                if (user != null && !user.IsContentAdministrator)
                                {
                                    PrivateMessage pm = new PrivateMessage();
                                    if (isPass)
                                    {
                                        pm.Subject = string.Format(GetResourceString("Title_Pattern_Point_NewsThreadAuditingSuccess", applicationID), thread.Title);
                                        pm.Body = string.Format(GetResourceString("Title_Pattern_Point_NewsThreadAuditingSuccess", applicationID), thread.Title);
                                    }
                                    else
                                    {
                                        pm.Subject = string.Format(GetResourceString("Title_Pattern_Point_NewsThreadAuditingFail", applicationID), thread.Title);
                                        pm.Body = string.Format(GetResourceString("Title_Pattern_Point_NewsThreadAuditingFail", applicationID), thread.Title);
                                    }

                                    pm.SenderUserID = (int)PrivateMessageEspecialUserID.Serve;

                                    pm.ReceiverUserID = user.UserID;
                                    pm.Receiver = user.DisplayName;
                                    pm.MessageType = PrivateMessageType.Serve;

                                    PrivateMessages.SendMessage(pm);
                                }
                            }
                        }
                    }
                    NewsThreads.UpdateAuditingStatuses(list, isPass ? AuditingStatuses.Success : AuditingStatuses.Fail);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = GetResourceString("Message_BatchUpdateSuccess", applicationID);
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_NoSelectedNews", applicationID);
            }

            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 更新站点类别
        /// </summary>
        /// <param name="selectedThreadIDsString"></param>
        /// <returns></returns>
        public ActionResult Control_UpdateNewsSection(string selectedThreadIDsString)
        {
            string[] selectedThreadIDsStringArray = selectedThreadIDsString.Split(',');
            if (selectedThreadIDsStringArray != null && selectedThreadIDsStringArray.Length > 0)
            {
                List<int> ListThreadID = new List<int>();
                foreach (string Id in selectedThreadIDsStringArray)
                {
                    if (Convert.ToInt32(Id) > 0)
                        ListThreadID.Add(Convert.ToInt32(Id));
                }
                int sectionID = Request.Form.GetInt("toCategoryID", -1);
                if (sectionID > 0)
                {
                    NewsThreads.UpdateNewsSection(ListThreadID, sectionID);
                    ViewData["StatusMessageType"] = (int)StatusMessageType.Success;
                    ViewData["StatusMessageContent"] = GetResourceString("Message_SetSiteCategoryBatchWorkSuccess", applicationID);
                }
                else
                {
                    ViewData["StatusMessageType"] = (int)StatusMessageType.Hint;
                    ViewData["StatusMessageContent"] = GetResourceString("Message_SetSiteCategoryWorkFailed", applicationID);
                }

            }
            ViewData["ListThreadID"] = selectedThreadIDsString;

            List<NewsSection> sortList = NewsSections.GetSectionsForAdmin(null, false, NewsSectionsOrganizeOptions.Indented, ActiveStatuses.IsActive);

            return PartialView("Controls/News/EditNewsSection.ascx", sortList);
        }

        #endregion

        #endregion

        #region Comment

        #region Show

        /// <summary>
        /// 管理评论
        /// </summary>
        /// <returns></returns>
        public ActionResult ManageComments(AuditingStatusesForDisplay? status, int? pageIndex, int? userID)
        {
            SetPageTitle(GetResourceString("Title_ManageNewsComment", applicationID));
            if (status == null)
                status = AuditingStatusesForDisplay.Pending;

            if (!userID.HasValue || userID.Value <= 0)
                userID = -1;

            if (!pageIndex.HasValue || pageIndex.Value <= 0)
                pageIndex = 1;

            PagingDataSet<NewsComment> pds = NewsComments.GetCommentsForAdmin(userID.Value, null, status, pageIndex.Value);

            ViewData["status"] = status;

            return View("Pages/News/ManageComments.aspx", "Masters/News.master", pds);
        }

        /// <summary>
        /// 搜索评论
        /// </summary>
        /// <param name="status"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SearchNewsComments(AuditingStatusesForDisplay? status)
        {
            int searchUserID = Request.Form.GetInt("searchUserID", 0);

            return RedirectToAction("ManageComments", new RouteValueDictionary { { "status", status }, { "userID", searchUserID } });
        }

        #endregion

        #region Manage

        /// <summary>
        /// 删除评论
        /// </summary>
        /// <param name="commentID"></param>
        /// <param name="bookmarkID"></param>
        /// <param name="status"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public ActionResult DeleteComment(int commentID)
        {
            NewsComments.DeleteComment(commentID);

            if (Request.UrlReferrer != null)
                return Redirect(Request.UrlReferrer.AbsoluteUri);
            else
                return RedirectToAction("ManageComments");
        }

        /// <summary>
        /// 批量删除评论
        /// </summary>
        /// <param name="commentID"></param>
        /// <param name="bookmarkID"></param>
        /// <param name="status"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteComments()
        {
            string[] deletePostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("commentCheckBoxGroup", string.Empty)))
            {
                deletePostIDs = Request.Form.GetString("commentCheckBoxGroup", string.Empty).Split(',');
                if (deletePostIDs != null && deletePostIDs.Length > 0)
                {
                    List<int> list = new List<int>();
                    foreach (string postID in deletePostIDs)
                    {
                        if (Convert.ToInt32(postID) > 0)
                            list.Add(Convert.ToInt32(postID));
                    }
                    NewsComments.DeleteComments(list.ToArray());
                }
            }

            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 更新审核状态
        /// </summary>
        /// <param name="isPass"></param>
        /// <param name="status"></param>
        /// <param name="sectionID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateCommentAuditingStatuses(bool isPass)
        {
            string[] deletePostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("commentCheckBoxGroup", string.Empty)))
            {
                deletePostIDs = Request.Form.GetString("commentCheckBoxGroup", string.Empty).Split(',');
                if (deletePostIDs != null && deletePostIDs.Length > 0)
                {
                    List<int> list = new List<int>();
                    foreach (string postID in deletePostIDs)
                    {
                        if (Convert.ToInt32(postID) > 0)
                            list.Add(Convert.ToInt32(postID));
                    }
                    NewsComments.UpdateAuditingStatus(list, isPass ? AuditingStatuses.Success : AuditingStatuses.Fail);
                }
            }

            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        #endregion

        #endregion

        #region Tags

        #region Show

        /// <summary>
        /// 管理标签
        /// </summary>
        /// <returns></returns>
        public ActionResult ManageTags()
        {
            SetPageTitle(GetResourceString("Title_ManageNewsTag", applicationID));
            return View("Pages/News/ManageTags.aspx", "Masters/News.master");
        }

        /// <summary>
        /// 编辑标签
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_EditTag(string tagName)
        {
            ApplicationTag tag = new ApplicationTag();
            if (!string.IsNullOrEmpty(tagName))
            {
                tag = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).GetApplicationTag(tagName, true);
                if (tag == null)
                {
                    throw new SPBException(ExceptionTypes.Instance().ResourceNotFound());
                }
            }
            return PartialView("Controls/News/EditTag.ascx", tag);
        }

        /// <summary>
        /// 列表显示标签
        /// </summary>
        /// <returns></returns>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListTags(int? pageIndex)
        {
            string keyWords = Request.Form.GetString("keyWords", string.Empty);
            ViewData["keyWords"] = keyWords;
            PagingDataSet<ApplicationTag> pds = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).GetAppliationTags(pageIndex ?? 1, null, keyWords);
            return PartialView("Controls/News/ListTags.ascx", pds);
        }

        #endregion

        #region Manage

        /// <summary>
        /// 创建更新标签
        /// </summary>
        /// <param name="tagName"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateUpdateTag(string tagName)
        {
            ApplicationTag tag = new ApplicationTag();
            //if (!string.IsNullOrEmpty(tagName))
            //{
            //    tag = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).GetApplicationTag(tagName, true);
            //    if (tag == null)
            //    {
            //        throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            //    }

            //    tag.TagName = StringUtils.Trim(Request.Form.GetString("tagName", string.Empty).Trim(), 30, string.Empty);

            //    ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).UpdateApplicationTag(tag);
            //    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            //}
            //else
            //{
            string[] formattedTagNames = Request.Form.GetString("tagName", string.Empty).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);

            if (formattedTagNames != null && formattedTagNames.Length > 0)
            {
                foreach (string formattedTagName in formattedTagNames)
                {
                    if (!string.IsNullOrEmpty(formattedTagName))
                    {
                        tag.UserID = CurrentUser.UserID;
                        tag.Author = CurrentUser.DisplayName;
                        tag.TagName = StringUtils.Trim(formattedTagName.Trim(), 8, string.Empty);
                        tag.TagName = Formatter.CleanTagName(tag.TagName);
                        tag.TagName = Formatter.FormatSingleLinePlainTextForStorage(tag.TagName);

                        ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).CreateApplicationTag(tag);
                        TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    }
                }
            }
            //}

            return RedirectToAction("Control_EditTag", new RouteValueDictionary { { "tagName", tag.TagName } });
        }

        /// <summary>
        /// 删除标签
        /// </summary>
        /// <param name="tagName"></param>
        /// <returns></returns>
        public ActionResult DeleteTag(string tagName, int? pageIndex)
        {
            tagName = WebUtils.UrlDecode(tagName);
            ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).DeleteApplicationTag(tagName);
            return RedirectToAction("Control_ListTags", new RouteValueDictionary { { "pageIndex", pageIndex } });
        }

        public ActionResult DeleteTags(int? pageIndex)
        {
            string selectedTagsString = Request.Form["LinkGroup"];

            if (!string.IsNullOrEmpty(selectedTagsString))
            {
                string[] selectedTagsStringArray = selectedTagsString.Split(',');
                foreach (string tagName in selectedTagsStringArray)
                {
                    ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).DeleteApplicationTag(tagName);
                }

                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "批量删除站点标签操作已成功,需要清除缓存，前台页面才会不显示";
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选择要删除的站点标签";
            }

            return RedirectToAction("Control_ListTags", new RouteValueDictionary { { "pageIndex", pageIndex } });
        }

        #endregion

        #endregion

        #region TitlePrefixes

        #region Show

        /// <summary>
        /// 管理标题前缀
        /// </summary>
        /// <returns></returns>
        public ActionResult ManageTitlePrefixes()
        {
            SetPageTitle(GetResourceString("Title_ManageTitlePrefixes", applicationID));
            return View("Pages/News/ManageTitlePrefixes.aspx", "Masters/News.master");
        }

        /// <summary>
        /// 列表显示标题前缀
        /// </summary>
        /// <returns></returns>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListTitlePrefixes()
        {
            Dictionary<int, string> dict = NewsThreads.GetTitlePrefixes(true);
            return PartialView("Controls/News/ListTitlePrefixes.ascx", dict);
        }

        /// <summary>
        /// 编辑标题前缀
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_EditTitlePrefix(int? prefixID)
        {
            if (prefixID.HasValue && (int)prefixID > 0)
            {
                Dictionary<int, string> dict = NewsThreads.GetTitlePrefixes(true);
                foreach (var prefix in dict)
                {
                    if (prefixID.Value == prefix.Key)
                    {
                        ViewData["id"] = prefix.Key;
                        ViewData["name"] = prefix.Value;
                        break;
                    }
                }
            }

            return PartialView("Controls/News/EditTitlePrefix.ascx");
        }

        #endregion

        #region Manage

        /// <summary>
        /// 创建更新前缀
        /// </summary>
        /// <param name="prefixID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateUpdatePrefix(int? prefixID)
        {
            if (prefixID.HasValue && (int)prefixID > 0)
            {
                string name = StringUtils.Trim(Request.Form.GetString("name", string.Empty).Trim(), 30, string.Empty);

                NewsThreads.UpdateTitlePrefix(prefixID.Value, name);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            }
            else
            {
                string[] formattedTagNames = Request.Form.GetString("name", string.Empty).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);

                if (formattedTagNames != null && formattedTagNames.Length > 0)
                {
                    foreach (string formattedTagName in formattedTagNames)
                    {
                        if (!string.IsNullOrEmpty(formattedTagName))
                        {
                            NewsThreads.CreateTitlePrefix(StringUtils.Trim(formattedTagName.Trim(), 8, string.Empty));
                            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                        }
                    }
                }
            }

            return RedirectToAction("Control_EditTitlePrefix", new RouteValueDictionary { { "prefixID", prefixID } });
        }

        /// <summary>
        /// 删除标题前缀
        /// </summary>
        /// <param name="prefixID"></param>
        /// <returns></returns>
        public ActionResult DeletePrefix(int prefixID)
        {
            NewsThreads.DeleteTitlePrefix(prefixID);
            return RedirectToAction("Control_ListTitlePrefixes");
        }

        #endregion

        #endregion

        #region Attachment

        #region Show

        /// <summary>
        /// 管理附件
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_ManageThreadAttachments(int? threadID)
        {
            if (threadID.HasValue && (int)threadID > 0)
            {
                NewsThread newsThread = NewsThreads.GetThread(threadID.Value, true);
                if (newsThread == null)
                {
                    throw new SPBException(ExceptionTypes.Instance().ResourceNotFound());
                }
                IList<Attachment> attachments = NewsAttachmentManager.Instance().GetAttachmentsByAssociateID(threadID.Value, true);
                ViewData["Attachments"] = attachments;
                ViewData["ThreadID"] = threadID;
                ViewData["UserFeaturedImageAttachmentID"] = newsThread.FeaturedImageAttachmentID;
            }
            else
            {
                IList<Attachment> attachments = NewsAttachmentManager.Instance().GetTemporaryAttachmentsOfUser(CurrentUser.UserID);
                ViewData["Attachments"] = attachments;
            }
            return PartialView("Controls/News/ManageThreadAttachments.ascx");
        }

        /// <summary>
        /// 新建附件
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_EditNewsAttachment(int? threadID)
        {
            if (threadID.HasValue)
            {
                ViewData["ThreadID"] = threadID;
            }
            return PartialView("Controls/News/EditNewsAttachment.ascx");
        }

        #endregion

        #region Manage

        #region 不再使用，已改为多文件上传形式

        /// <summary>
        /// 创建附件
        /// </summary>
        /// <param name="threadID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateNewsAttachment(int? threadID)
        {
            System.Web.HttpPostedFileBase postedFile = null;

            if (Request.Files != null && Request.Files.Count > 0)
                postedFile = Request.Files[0];

            ContentResult result = new ContentResult();
            result.ContentType = "text/html";

            if (postedFile == null || postedFile.ContentLength == 0)
            {
                result.Content = GetResourceString("Message_SelectOnloadFile", applicationID);
                return result;
            }

            #region 检查上传文件格式
            AttachmentSettings fs = NewsConfiguration.Instance().AttachmentSettings;
            if (fs.ValidateExtensions && !CurrentUser.IsContentAdministrator)
            {
                if (!fs.ValidationRegex.IsMatch(postedFile.FileName))
                {
                    //statusMessageData = new StatusMessageData(StatusMessageType.Error, string.Format("只允许上传以下文件类型: {0}.", fs.Extensions.Replace(";", ",")));
                    //TempData["StatusMessageData"] = statusMessageData;
                    //return RedirectToAction("Control_NewBlogAttachment");

                    result.Content = string.Format(GetResourceString("Message_Pattern_OnlyOnloadType", applicationID), fs.Extensions.Replace(";", ","));
                    return result;
                }
            }
            #endregion

            #region 检查上传文件大小
            if ((postedFile.ContentLength / 1024) > NewsConfiguration.Instance().AttachmentSettings.MaxAttachmentSize)
            {
                //statusMessageData = new StatusMessageData(StatusMessageType.Error, string.Format("上传附件不能超过{0}K", WeblogConfiguration.Instance().AttachmentSettings.MaxAttachmentSize));
                //TempData["StatusMessageData"] = statusMessageData;
                //return RedirectToAction("Control_NewBlogAttachment");

                result.Content = string.Format(GetResourceString("Message_Pattern_OnloadAttachmentMaxKB", applicationID), NewsConfiguration.Instance().AttachmentSettings.MaxAttachmentSize);
                return result;
            }
            #endregion

            Attachment attachment = new Attachment(postedFile);
            attachment.UserID = CurrentUser.UserID;

            if (!string.IsNullOrEmpty(Request.Form.GetString("friendlyFileName", string.Empty)))
                attachment.FriendlyFileName = Request.Form.GetString("friendlyFileName", string.Empty).Trim();
            if (threadID.HasValue)
            {
                attachment.AssociateID = threadID.Value;
                NewsAttachmentManager.Instance().CreateAttachment(attachment, postedFile.InputStream);
            }
            else
            {
                NewsAttachmentManager.Instance().CreateTemporaryAttachment(attachment, postedFile.InputStream);
            }

            result.Content = "ok";
            return result;

        }

        #endregion

        /// <summary>
        /// 删除附件
        /// </summary>
        public ActionResult DeleteNewsAttachment(int threadID, int attachmentID)
        {
            NewsAttachmentManager.Instance().DeleteAttachment(attachmentID);

            return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", threadID } });
        }

        public ActionResult UserContentSelector_UpdateNewsAttachments(int? threadID)
        {
            return PartialView("Controls/News/UserContentSelector_UpdateNewsAttachments.ascx");
        }

        #endregion

        #endregion

        /// <summary>
        /// 获取资讯数据统计
        /// </summary>
        /// <returns></returns>
        public JsonResult GetManageableCounts()
        {
            Dictionary<NewsManageableCounts, int> dictionary = NewsThreads.GetManageableCounts();
            Application newsApplication = Applications.GetApplication(ApplicationIDs.Instance().News());
            List<ManageableCount> list = new List<ManageableCount>();
            list.Add(new ManageableCount { Name = "新增资讯", Value = newsApplication.CountInPast24Hours });
            if (dictionary.ContainsKey(NewsManageableCounts.PendingThreadCount))
                list.Add(new ManageableCount { Name = "待审核资讯", Value = dictionary[NewsManageableCounts.PendingThreadCount], Url = SPBUrlHelper.Action("ManageNews", "ManageNews", new RouteValueDictionary { { "status", AuditingStatusesForDisplay.Pending } }) });
            if (dictionary.ContainsKey(NewsManageableCounts.PendingThreadCount))
                list.Add(new ManageableCount { Name = "需再审核资讯", Value = dictionary[NewsManageableCounts.AgainThreadCount], Url = SPBUrlHelper.Action("ManageNews", "ManageNews", new RouteValueDictionary { { "status", AuditingStatusesForDisplay.Again } }) });
            if (dictionary.ContainsKey(NewsManageableCounts.PendingThreadCount))
                list.Add(new ManageableCount { Name = "待审核评论", Value = dictionary[NewsManageableCounts.PendingCommentCount], Url = SPBUrlHelper.Action("ManageComments", "ManageNews", new RouteValueDictionary { { "status", AuditingStatusesForDisplay.Pending } }) });
            list.Add(new ManageableCount { Name = "总资讯数", Value = newsApplication.TotalCount });
            return Json(list,JsonRequestBehavior.AllowGet);
        }
    }

    public enum NewsSubMenus
    {
        /// <summary>
        /// 发布资讯
        /// </summary>
        CreateNews = 1201,

        /// <summary>
        /// 资讯管理
        /// </summary>
        ManageNews = 1202,

        /// <summary>
        /// 标题前缀管理
        /// </summary>
        ManageTitlePrefixes = 1203,

        /// <summary>
        /// 资讯评论管理
        /// </summary>
        ManageNewsComments = 1204,

        /// <summary>
        /// 资讯标签管理
        /// </summary>
        ManageNewsTags = 1205,

        /// <summary>
        /// 资讯栏目管理
        /// </summary>
        ManageNewsSections = 1206,
    }
}