﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using SpaceBuilder.Common;
using SpaceBuilder.Web;
using System.Web.Routing;
using SpaceBuilder.Utils;

namespace SpaceBuilder.News.Controllers
{
    [UserDomainGuideInstallApplication(ApplicationID = 201)]
    [HttpCompress]
    public class NewsController : UserDomainControllerBase
    {
        private int applicationID = ApplicationIDs.Instance().News();
        private static readonly string pageSeparator = "<!-- pagebreak -->";

        #region NewsThread

        /// <summary>
        /// 发布博客同时投稿到资讯
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public ActionResult BlogToNews(string userDomainName)
        {
            Post post = TempData.Get<Post>("post", new Post());

            NewsThread newsThread = new NewsThread();

            newsThread.Title = post.Subject;
            newsThread.BodyForDataTransmission = post.BodyForDataTransmission;
            newsThread.Excerpt = post.Excerpt;
            newsThread.OriginalLinks = post.FromUrl;
            newsThread.OriginalAuthor = post.Author;
            //将文件的附件拷贝到资讯的临时附件中
            List<Attachment> attachments = post.Attachments;
            if (attachments != null)
            {
                foreach (var attach in attachments)
                {
                    if (attach.File == null)
                        continue;
                    Attachment attachment = new Attachment(NewsAttachmentManager.Instance().StorageProviderName, null);
                    attachment.UserID = attach.UserID;
                    attachment.ContentSize = attach.ContentSize;
                    attachment.ContentType = attach.ContentType;
                    attachment.DateCreated = attach.DateCreated;
                    attachment.FriendlyFileName = attach.FriendlyFileName;

                    NewsAttachmentManager.Instance().CreateTemporaryAttachment(attachment, attach.File.OpenReadStream());
                }
            }
            List<NewsSection> allSections = NewsSections.GetSections(null, false, NewsSectionsOrganizeOptions.Indented, NewsSectionsSortBy.DisplayOrder, SortOrder.Ascending);
            ViewData["allSections"] = allSections;

            ViewData["isSection"] = true;

            return View("Pages/News/EditNews.aspx", "Masters/News.master", newsThread);
        }

        /// <summary>
        /// 发表资讯
        /// </summary>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserTypeAuthorize(UserType = UserTypes.PersonUser)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult EditNews(string userDomainName, int? newsID)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_ReleaseNews", applicationID));
            #endregion

            if (!NewsConfiguration.Instance().EnableUserContribute)
            {
                throw new SPBException(ExceptionTypes.Instance().ResourceNotFound());
            }

            if (!NewsPermission.AccessCheck(null, CurrentUser, PermissionItemKeys.Instance().ContributeNewsThread(), HttpContext))
                return new EmptyResult();

            NewsThread thread = new NewsThread();

            if (NewsSections.GetSections(null, true, NewsSectionsOrganizeOptions.None, NewsSectionsSortBy.DisplayOrder, SortOrder.Ascending).Count > 0)
            {
                if (newsID.HasValue && (int)newsID > 0)
                {
                    thread = NewsThreads.GetThread(newsID.Value, true);
                    if (thread == null)
                    {
                        return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NewsThemeNotFound", applicationID), newsID.Value)));
                        //throw new SPBException(ExceptionType.ResourceNotFound);
                    }
                    if (!NewsPermission.AccessCheck(thread, CurrentUser, PermissionItemKeys.Instance().EditNewsThread(), HttpContext))
                        return new EmptyResult();

                    if (thread.AuditingStatus == AuditingStatuses.Success && !NewsPermission.AccessCheck(thread, CurrentUser, PermissionItemKeys.Instance().EidtApprovedContributedNewsThread(), HttpContext))
                        return new EmptyResult();
                    else if (thread.AuditingStatus != AuditingStatuses.Success && !NewsPermission.AccessCheck(thread, CurrentUser, PermissionItemKeys.Instance().EditNewsThread(), HttpContext))
                        return new EmptyResult();
                    if (thread.GetBodyForEdit().IndexOf("<!-- pagebreak -->") > 0)
                        ViewData["isShowPage"] = false;
                }
                else
                {
                    if (TempData["thread"] != null)
                    {
                        thread = TempData["thread"] as NewsThread;
                    }
                }

                ViewData["isSection"] = true;


                List<NewsSection> allSections = NewsSections.GetSections(null, false, NewsSectionsOrganizeOptions.Indented, NewsSectionsSortBy.DisplayOrder, SortOrder.Ascending);
                ViewData["allSections"] = allSections;

                Dictionary<int, string> dict = NewsThreads.GetTitlePrefixes(true);
                ViewData["dict"] = dict;
            }


            //string tagList = string.Empty;
            //foreach (var tag in thread.Tags)
            //{
            //    tagList = tagList + tag + ",";
            //}

            //ViewData["tagList"] = tagList;

            return View("Pages/News/EditNews.aspx", "Masters/News.master", thread);
        }

        /// <summary>
        /// 显示资讯列表
        /// </summary>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserTypeAuthorize(UserType = UserTypes.PersonUser)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListNews(string userDomainName, AuditingStatusesForDisplay? status, int? pageIndex)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_NewsList", applicationID));
            #endregion

            if (!NewsConfiguration.Instance().EnableUserContribute)
            {
                throw new SPBException(ExceptionTypes.Instance().ResourceNotFound());
            }

            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            if (status.HasValue)
                ViewData["status"] = status;

            if (!pageIndex.HasValue || pageIndex.Value <= 0)
                pageIndex = 1;

            PagingDataSet<NewsThread> pds = NewsThreads.GetMyThreads(currentDomainUser.UserID, status, pageIndex.Value);

            return View("Pages/News/ListNews.aspx", "Masters/News.master", pds);
        }

        /// <summary>
        /// 创建更新资讯
        /// </summary>
        /// <param name="newsID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserTypeAuthorize(UserType = UserTypes.PersonUser)]
        public ActionResult CreateUpdateNews(string userDomainName, int? newsID)
        {
            if (!NewsConfiguration.Instance().EnableUserContribute)
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            NewsThread thread = new NewsThread();
            if (newsID.HasValue && (int)newsID > 0)
            {
                thread = NewsThreads.GetThread(newsID.Value, false);
                if (thread == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NewsThemeNotFound", applicationID), newsID.Value)));
                }
                if (!NewsPermission.AccessCheck(thread, CurrentUser, PermissionItemKeys.Instance().EditNewsThread(), HttpContext))
                    return new EmptyResult();

                if (thread.AuditingStatus == AuditingStatuses.Success && !NewsPermission.AccessCheck(thread, CurrentUser, PermissionItemKeys.Instance().EidtApprovedContributedNewsThread(), HttpContext))
                    return new EmptyResult();
                else if (thread.AuditingStatus != AuditingStatuses.Success && !NewsPermission.AccessCheck(thread, CurrentUser, PermissionItemKeys.Instance().EditNewsThread(), HttpContext))
                    return new EmptyResult();
            }
            else
            {
                if (!NewsPermission.AccessCheck(thread, CurrentUser, PermissionItemKeys.Instance().ContributeNewsThread(), HttpContext))
                    return new EmptyResult();

                thread.UserID = CurrentUser.UserID;
                thread.Contributor = CurrentUser.DisplayName;
            }

            bool isAutoPage = Request.Form.GetBool("isAutoPage", false);
            int pageSize = Request.Form.GetInt("pageSize", 0);
            string body = Request.Form.GetString("body", string.Empty);
            if (isAutoPage && pageSize > 0)
                body = string.Join(pageSeparator, ContentPages.GetPageContentForStorage(body, pageSize * 1000, true).ToArray());


            thread.SectionID = Request.Form.GetInt("sectionDDL", 0);
            if (CurrentUser.IsContentAdministrator)
                thread.IsContributed = false;
            else
                thread.IsContributed = true;

            thread.PrefixID = Request.Form.GetInt("prefixDDL", 0);
            thread.Title = Request.Form.GetString("title", string.Empty).Trim();
            thread.Subtitle = Request.Form.GetString("subtitle", string.Empty).Trim();
            thread.Keywords = Request.Form.GetString("keywords", string.Empty).Trim();
            thread.OriginalLinks = Request.Form.GetString("OriginalLinks", string.Empty).Trim();


            string metaKeyWords = string.Empty;
            List<string> tagNameList = new List<string>();
            string[] formattedTagNames = Request.Form.GetString("newsTags", string.Empty).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
            if (formattedTagNames != null && formattedTagNames.Length > 0)
            {
                ApplicationTagManager applicationTagManager = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News());
                foreach (string formattedTagName in formattedTagNames)
                {
                    if (applicationTagManager.GetApplicationTag(formattedTagName.Trim(), true) == null)
                    {
                        ApplicationTag tag = new ApplicationTag();
                        tag.TagName = formattedTagName.Trim();
                        tag.UserID = CurrentUser.UserID;
                        tag.Author = currentDomainUser.DisplayName;
                        tag.TagName = Formatter.CleanTagName(tag.TagName);
                        tag.TagName = Formatter.FormatSingleLinePlainTextForStorage(tag.TagName);

                        bool isCreated = applicationTagManager.CreateApplicationTag(tag);
                        if (isCreated)
                            tagNameList.Add(tag.TagName);
                    }
                    else
                    {
                        tagNameList.Add(applicationTagManager.GetApplicationTag(formattedTagName, true).TagName);
                    }
                    if (string.IsNullOrEmpty(thread.Keywords))
                        metaKeyWords = string.Format("{1},{0}", metaKeyWords, formattedTagName);
                }
            }

            if (string.IsNullOrEmpty(thread.Keywords))
            {
                metaKeyWords = metaKeyWords + thread.Title;
                metaKeyWords = string.Format("{0},{1}", metaKeyWords, thread.Contributor);
                if (thread.Section != null)
                    metaKeyWords = string.Format("{0},{1}", metaKeyWords, thread.Section.GetMultilingualSectionName(CurrentUserLanguage));
                thread.Keywords = metaKeyWords;
            }

            thread.OriginalAuthor = Request.Form.GetString("author", string.Empty).Trim();
            thread.CopyFrom = Request.Form.GetString("copyFrom", string.Empty).Trim();
            thread.Excerpt = Request.Form.GetString("excerpt", string.Empty).Trim();
            thread.BodyForDataTransmission = body;

            //thread.AllowededUserRank = Request.Form.GetInt("allowededUserRank", 0);
            thread.EnableComments = Request.Form.GetBool("enableComments", true);
            thread.SpecialOrder = Request.Form.GetBool("isSticky", false) ? 1 : 0;
            thread.PointConsumption = Request.Form.GetInt("pointConsumption", 0);
            thread.IsDownloadRemoteImage = Request.Form.GetBool("downloadRemoteImage", false);

            //设置标题图片
            int featurePicAttachmentID = Request.Form.GetInt("featurePicAttachmentID", -1);
            if (featurePicAttachmentID > 0)
                thread.FeaturedImageAttachmentID = featurePicAttachmentID;

            if (newsID.HasValue && (int)newsID > 0)
            {
                NewsThreads.UpdateThread(thread);

                ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).UpdateApplicationTagsOfItem(thread.ThreadID, tagNameList);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                if (thread.AuditingStatus == AuditingStatuses.Success)
                    TempData["StatusMessageContent"] = GetResourceString("Message_AmendNewsSucceed", applicationID);
                else
                    TempData["StatusMessageContent"] = GetResourceString("Message_NewsShowAfterAuditing", applicationID);
                return Redirect(SiteUrls.Instance().ShowNews(thread.ThreadID));
            }
            else
            {
                NewsThreads.CreateThread(thread);
                ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().News()).UpdateApplicationTagsOfItem(thread.ThreadID, tagNameList);
                if (thread.ThreadID > 0)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    if (thread.AuditingStatus == AuditingStatuses.Success)
                        TempData["StatusMessageContent"] = GetResourceString("Message_ReleaseNewsSucceed", applicationID);
                    else
                        TempData["StatusMessageContent"] = GetResourceString("Message_ShowAfterRelease", applicationID);
                    return Redirect(SiteUrls.Instance().ShowNews(thread.ThreadID));
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_ReleaseNewsAborted", applicationID);
                    TempData["thread"] = thread;
                    return RedirectToAction("EditNews", new RouteValueDictionary { { "userDomainName", userDomainName }, { "newsID", newsID } });

                }
            }

        }

        /// <summary>
        /// 删除资讯
        /// </summary>
        /// <param name="newsID"></param>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteNews(string userDomainName, int newsID, AuditingStatusesForDisplay? status)
        {
            if (!NewsConfiguration.Instance().EnableUserContribute)
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }

            NewsThread thread = NewsThreads.GetThread(newsID, false);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NewsThemeNotFound", applicationID), newsID)));
                //throw new SPBException(ExceptionType.AccessDenied);
            }
            ValidateItemOwner(thread.UserID);

            if (!NewsPermission.AccessCheck(thread, CurrentUser, PermissionItemKeys.Instance().DeleteNewsThread(), HttpContext))
            {
                return new EmptyResult();
            }

            NewsThreads.DeleteThread(thread);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSucecssful", applicationID);

            if (Request.UrlReferrer != null)
            {
                return Redirect(Request.UrlReferrer.AbsoluteUri);
            }

            return RedirectToAction("ListNews", new RouteValueDictionary { { "userDomainName", userDomainName }, { "status", status } });
        }

        #endregion
    }

    #region Enums

    /// <summary>
    /// 资讯类型
    /// </summary>
    public enum NewsMenuTypes
    {
        /// <summary>
        /// 发表资讯
        /// </summary>
        Create,

        /// <summary>
        /// 所有资讯
        /// </summary>
        ListAll,
    }

    #endregion
}