﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common;

using System.Web;

namespace SpaceBuilder.News
{
    public class NewsPermission
    {
        /// <summary>
        /// 验证用户是否具有某种权限
        /// </summary>
        /// <param name="newsThread">资讯主题</param>
        /// <param name="user">操作用户</param>
        /// <param name="permissionItemKey">permissionItemKey</param>
        /// <returns>是否通过验证</returns>
        public static bool Validate(NewsThread newsThread, User user, string permissionItemKey)
        {
            string exceptionType = ExceptionTypes.Instance().UnknownError();
            return Validate(newsThread, user, permissionItemKey, ref exceptionType);
        }


        /// <summary>
        /// 验证用户是否具有某种权限
        /// </summary>
        /// <param name="newsThread">资讯主题</param>
        /// <param name="user">操作用户</param>
        /// <param name="permissionItemKey">permissionItemKey</param>
        /// <param name="exceptionType">抛出异常类型</param>
        /// <returns>是否通过验证</returns>
        public static bool Validate(NewsThread newsThread, User user, string permissionItemKey, ref string exceptionType)
        {
            if (user != null && user.IsContentAdministrator)
                return true;

            if ((permissionItemKey == PermissionItemKeys.Instance().CreateNewsComment()) && (newsThread != null) && (newsThread.IsLocked))
            {
                exceptionType = ExceptionTypes.Instance().GeneralAccessDenied();
                return false;
            }

            PermissionCollection userPermissionCollection = Permissions.ResolveUserPermission(user, ApplicationIDs.Instance().News());
            PermissionCollection permissionCollectionForNews = userPermissionCollection.Clone();

            //资讯投稿
            if (newsThread == null)
            {
                if (permissionCollectionForNews.Validate(permissionItemKey))
                {
                    return true;
                }
                else
                {
                    exceptionType = ExceptionTypes.Instance().GeneralAccessDenied();
                    return false;
                }
            }

            //处理Owner
            if (newsThread != null && newsThread.UserID == user.UserID)
            {
                Role moderatorRole = Roles.GetRole(UserRoleNames.Instance().Owner(), false);
                if (moderatorRole != null)
                {
                    PermissionCollection ownerPermissionCollection = Permissions.GetPermissionItemRolesInUserRole(moderatorRole.RoleID, ApplicationIDs.Instance().News());
                    permissionCollectionForNews.Merge(ownerPermissionCollection);
                }
            }

            if (permissionCollectionForNews.Validate(permissionItemKey))
            {
                return true;
            }
            else
            {
                exceptionType = ExceptionTypes.Instance().GeneralAccessDenied();
                return false;
            }
        }


        #region AccessCheck

        /// <summary>
        /// 检查用户是否有权限进行某项操作，如果没有则转到登录页面或者提示错误
        /// </summary>
        public static bool AccessCheck(NewsThread newsThread, User user, string permissionItemKey, HttpContextBase httpContext)
        {
            string exceptionType = ExceptionTypes.Instance().UnknownError();
            if (!Validate(newsThread, user, permissionItemKey, ref exceptionType))
            {
                if (httpContext.Request.IsAuthenticated)
                    throw new SPBException(ExceptionTypes.Instance().AccessDenied());
                else
                    httpContext.Response.Redirect(SiteUrls.Instance().Login(true), true);

                return false;
            }
            return true;
        }

        #endregion

    }
}
