﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using SpaceBuilder.Photo;
using SpaceBuilder.Common;
using SpaceBuilder.Common.Controllers;
using SpaceBuilder.Utils;

using SpaceBuilder.Web;

using System.Web.Routing;
using SpaceBuilder.LuceneSearch;

namespace SpaceBuilder.Photo.Controllers
{
    [HttpCompress]
    public class ChannelPhotoController : ChannelControllerBase
    {
        int applicationID = ApplicationIDs.Instance().Photo();

        /// <summary>
        /// 图片首页
        /// </summary>
        public ActionResult Home()
        {
            SetPageTitle(GetResourceString("Title_Callery", applicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Callery", applicationID), string.Empty);
            return View("Pages/Photos/Home.aspx", "Masters/Photo.master");
        }

        #region PhotoThread

        /// <summary>
        /// 标签地图
        /// </summary>
        public ActionResult PhotoTagMap()
        {
            SetPageTitle(GetResourceString("Title_Callery", applicationID), Globals.SiteName, GetResourceString("Title_TagMap", applicationID));

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_TagMap", applicationID), string.Empty);

            Dictionary<string, KeyValuePair<int, SiteTag>> siteTags = UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).GetTopSiteTagsSortBy(SiteTagsSortBy.ItemCount, 300);
            return View("Pages/Photos/PhotoTagMap.aspx", "Masters/Photo.master", siteTags);
        }

        /// <summary>
        /// 根据标签显示图片列表
        /// </summary>
        /// <param name="tagName">标签名</param>
        public ActionResult ListPhotosByTag(string tagName)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            SetPageTitle(GetResourceString("Title_Callery", applicationID), Globals.SiteName, string.Format(GetResourceString("Title_Pattern_TagInImage", applicationID), tagName));
            //BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Callery", applicationID), SiteUrls.Instance().ChannelPhotoHome());
            //BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_TagMap", applicationID), SPBUrlHelper.Action("PhotoTagMap", "ChannelPhoto"));
            BreadCrumb.AddBreadCrumbItem(string.Format(GetResourceString("Title_Pattern_TagInImage", applicationID), tagName), string.Empty);
            ViewData["tagName"] = tagName;
            return View("Pages/Photos/ListPhotosByTag.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 根据类别显示照片列表
        /// </summary>
        public ActionResult ListPhotosByCategory(int? categoryID)
        {
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                int siteCategoryID = (int)categoryID.Value;

                SiteCategory currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Photo()).Get(siteCategoryID, false);
                BreadCrumb.AddBreadCrumbItem(currentCategory.GetMultilingualCategoryName(CurrentUserLanguage), string.Empty);

                SiteCategory category = SiteCategories.Instance(ApplicationIDs.Instance().Photo()).Get(categoryID.Value, false);
                SetPageTitle(GetResourceString("Title_Callery", applicationID), Globals.SiteName, string.Format(GetResourceString("Title_Pattern_categoryInImage", applicationID), category.GetMultilingualCategoryName(CurrentUserLanguage)));
            }
            else
                BreadCrumb.AddBreadCrumbItem(string.Format(GetResourceString("Title_PhotoRank", applicationID)), string.Empty);

            return View("Pages/Photos/ListPhotosByCategory.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 生成活动类别面包屑
        /// </summary>
        private void ShowCategoryBreadCrumb(int siteCategoryID)
        {
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Callery", applicationID), SiteUrls.Instance().ChannelPhotoHome());
            Stack<SiteCategory> stack = new Stack<SiteCategory>();
            SiteCategory currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Photo()).Get(siteCategoryID, false);
            while (currentCategory != null)
            {
                stack.Push(currentCategory);
                currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Photo()).Get(currentCategory.ParentID, false);
            }
            while (stack.Count > 0)
            {
                currentCategory = stack.Pop();
                BreadCrumb.AddBreadCrumbItem(currentCategory.GetMultilingualCategoryName(CurrentUserLanguage), SPBUrlHelper.Action("ListPhotosByCategory", "ChannelPhoto", new RouteValueDictionary { { "categoryID", currentCategory.CategoryID } }));
            }
        }

        /// <summary>
        /// 相册排行
        /// </summary>
        /// <param name="sortBy">排序条件</param>
        /// <param name="pageIndex">页码</param>
        public ActionResult GalleryRanks(UserCategorySortBy? sortBy)
        {
            SetPageTitle(GetResourceString("Title_Callery", applicationID), GetResourceString("Title_CalleryRank", applicationID));
            //BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Callery", applicationID), SiteUrls.Instance().ChannelPhotoHome());
            BreadCrumb.AddBreadCrumbItem(string.Format(GetResourceString("Title_CalleryRank", applicationID)), string.Empty);
            ViewData["sortBy"] = sortBy ?? UserCategorySortBy.MostRecentUpdateDate;
            return View("Pages/Photos/GalleryRanks.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 站点热门图片
        /// </summary>
        /// <param name="pageSize"></param>
        /// <returns></returns>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListStickyPhotosBlock(int displayCount)
        {
            bool isShow = false;
            if (Applications.IsEnabled(ApplicationIDs.Instance().Photo()))
                isShow = true;
            ViewData["isShow"] = isShow;
            IList<PhotoThread> photos = PhotoThreads.GetTopThreadsSortBy(PhotoSortBy.StickyAndMostRecent, null, displayCount);
            return PartialView("Controls/Photos/ListStickyPhotosBlock.ascx", photos);
        }

        #region Control Actions

        /// <summary>
        /// 相册列表模块
        /// </summary>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListGalleriesBlock(UserCategorySortBy? sortBy, string viewName, string titleName, int topNumber)
        {
            IList<UserCategory> list = null;
            if (sortBy.HasValue && sortBy == UserCategorySortBy.MostRecentUpdateDate)
                list = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetTopCategoriesSortBy(sortBy ?? UserCategorySortBy.MostRecentUpdateDate, 100);
            else
                list = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetTopCategoriesSortBy(sortBy ?? UserCategorySortBy.MostRecentUpdateDate, topNumber);
            ViewData["sortBy"] = sortBy ?? UserCategorySortBy.MostRecentUpdateDate;
            ViewData["titleName"] = titleName;
            if (list != null)
                list = list.Where(n => n.ItemCount > 0).Take(topNumber).ToList();
            return PartialView("Controls/Photos/" + viewName + ".ascx", list);
        }
        /// <summary>
        /// 图片简单列表模块
        /// 非分页列表
        /// </summary>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListPhotosBlock(PhotoSortBy? sortBy, string viewName, string titleName, int topNumber)
        {
            IList<PhotoThread> list = PhotoThreads.GetTopThreadsSortBy(sortBy ?? PhotoSortBy.StickyAndMostRecent, null, topNumber);
            ViewData["sortBy"] = sortBy ?? PhotoSortBy.StickyAndMostRecent;
            ViewData["titleName"] = titleName;
            return PartialView("Controls/Photos/" + viewName + ".ascx", list);
        }

        /// <summary>
        /// 图片标签列表模块
        /// </summary>
        /// <param name="pageSize">显示标签数量</param>
        /// <param name="tagType">标签类型</param>
        /// <param name="sortBy">按文章数或用户数排序</param>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListSiteTagsBlock(int pageSize, SiteTagsSortBy? sortBy, string controlTitle)
        {
            Dictionary<string, KeyValuePair<int, SiteTag>> siteTags = UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).GetTopSiteTagsSortBy(sortBy ?? SiteTagsSortBy.ItemCount, pageSize);
            ViewData["sortBy"] = sortBy ?? SiteTagsSortBy.ItemCount;
            ViewData["controlTitle"] = controlTitle;
            return PartialView("Controls/Photos/ListSiteTagsBlock.ascx", siteTags);
        }
        /// <summary>
        /// 根据标签显示照片列表
        /// </summary>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListPhotosByTag(string tagName, int? pageIndex)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");

            PagingDataSet<PhotoThread> pds = PhotoThreads.GetPagingThreadsInSiteTag(PhotoSortBy.MostRecent, tagName, 20, pageIndex ?? 1);
            ViewData["tagName"] = tagName;
            return PartialView("Controls/Photos/ListPhotosByTag.ascx", pds);
        }

        /// <summary>
        /// 根据类别显示照片列表
        /// </summary>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListPhotosByCategory(int? categoryID, PhotoSortBy? sortBy, int pageSize, int? pageIndex)
        {
            PagingDataSet<PhotoThread> pds = PhotoThreads.GetPagingThreadsSortBy(sortBy ?? PhotoSortBy.MostRecent, categoryID, pageSize, pageIndex ?? 1);

            ViewData["categoryID"] = categoryID;
            if (categoryID.HasValue && categoryID.Value > 0)
                ViewData["categoryName"] = SiteCategories.Instance(ApplicationIDs.Instance().Photo()).Get(categoryID.Value, false);
            ViewData["sortBy"] = sortBy;
            return PartialView("Controls/Photos/ListPhotosByCategory.ascx", pds);
        }

        /// <summary>
        /// 相册排行
        /// </summary>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_GalleryRanks(UserCategorySortBy? sortBy, int pageSize, int? pageIndex)
        {
            //PagingDataSet<PhotoSection> pds = PhotoSections.GetPagingsSortBy(sortBy ?? PhotoSectionsSortBy.PictureCount, pageSize, pageIndex ?? 1);
            //ViewData["sortBy"] = sortBy;
            PagingDataSet<UserCategory> pds = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategories(sortBy ?? UserCategorySortBy.MostRecentUpdateDate, pageSize, pageIndex ?? 1);
            ViewData["sortBy"] = sortBy;
            return PartialView("Controls/Photos/GalleryRanks.ascx", pds);
        }

        #endregion

        #region Rating

        /// <summary>
        /// 对图片进行星级评价
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Rate()
        {
            //图片ThreadID
            int itemID = Request.QueryString.GetInt("itemID", -1);

            //评价的星级
            int value = Request.QueryString.GetInt("value", -1);


            ContentResult result = new ContentResult();
            result.ContentType = "text/html";
            if (!this.HttpContext.User.Identity.IsAuthenticated)
            {
                result.Content = GetResourceString("Message_PleaceLogin");
            }
            else
            {
                if (CurrentUser != null && itemID > 0 && value > 0)
                {
                    Rating rating = new Rating();
                    rating.UserID = CurrentUser.UserID;
                    rating.Author = CurrentUser.DisplayName;
                    rating.ItemID = itemID;
                    rating.Rate = value;
                    PhotoThreads.Rate(rating);
                    result.Content = GetResourceString("Message_ThanksForRating");
                }
                else
                {
                    result.Content = GetResourceString("Message_ParameterIncorrect");
                }
            }
            return result;
        }

        /// <summary>
        /// 显示评价人列表
        /// </summary>
        public ActionResult Control_ListRatingUsers(int itemID, int? topNumber)
        {
            IList<Rating> ratings = PhotoThreads.GetRatings(itemID, topNumber ?? 30);
            return View("Controls/Common/ListRatingUsers.ascx", ratings);
        }

        #endregion

        #region 用户内容选择器

        /// <summary>
        /// 在用户内容选择器中显示用户的照片
        /// </summary>
        /// <param name="categoryID">相簿ID</param>
        public ActionResult UserContentSelector_ListPhotos(int? categoryID)
        {
            List<PhotoThread> photos = null;

            User currentUser = Globals.GetCurrentUser();
            if (currentUser != null && currentUser != null)
            {
                PagingDataSet<PhotoThread> photoSet = PhotoThreads.GetThumbnailThreads(currentUser.UserID, string.Empty, categoryID, 1);
                photos = new List<PhotoThread>(photoSet.Records);
            }
            else
            {
                photos = new List<PhotoThread>();
            }

            List<UserCategory> userCategories = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategories(currentUser.UserID, false);

            ViewData["userCategories"] = userCategories;
            return PartialView("Controls/UserContentSelector_ListPhotos.ascx", photos);
        }

        public JsonResult GetReaderHtmls(string selectedPhotoIDs, PhotoImageType photoImageType, PhotoLinkType photoLinkType)
        {
            if (string.IsNullOrEmpty(selectedPhotoIDs))
                return Json(new { }, JsonRequestBehavior.AllowGet);
            string[] selectedPhotoIDsArray = selectedPhotoIDs.Split(',');
            List<string> list = new List<string>();
            foreach (string strPhotoID in selectedPhotoIDsArray)
            {
                int photoID = 0;
                int.TryParse(strPhotoID, out photoID);
                if (photoID > 0)
                    list.Add(string.Format("<img src='{0}' border='0' />", SiteUrls.Instance().PictureUrl(photoID, photoImageType)));
            }
            return Json(list, JsonRequestBehavior.AllowGet);
        }

        #endregion

        #region FullTextSearch

        /// <summary>
        /// 图片全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Search(int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_SearchImage", applicationID));

            PhotoFullTextQuery query = new PhotoFullTextQuery();
            query.ConvertFromQueryString(Request);
            query.PageSize = 10;

            if (pageIndex.HasValue)
                query.PageIndex = pageIndex.Value;

            SearchResultDataSet<PhotoThread> results = null;
            if (query.IsValid())
            {
                results = PhotoSearchManager.Instance().Search(query);
            }

            ViewData["Query"] = query;
            return this.View("Pages/Photos/Search.aspx", "Masters/Photo.master", results);
        }

        /// 图片全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Search()
        {
            RouteValueDictionary parameters = new RouteValueDictionary();
            parameters["Keyword"] = Request.Form.GetString("Keyword", string.Empty);

            int searchScopeUserID = Request.Form.GetInt("SearchScopeUserID", -1);
            if (searchScopeUserID > 0)
                parameters["SearchScopeUserID"] = searchScopeUserID.ToString();

            return RedirectToAction("Search", parameters);
        }

        #endregion

        #region SiteCategory

        /// <summary>
        /// 获取所有的根类别
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        //[ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_SubMenus()
        {
            List<SiteCategory> siteCategories = SiteCategories.Instance(ApplicationIDs.Instance().Photo()).GetRoots(false);
            StringBuilder sb = new StringBuilder();
            foreach (SiteCategory rootCategory in siteCategories)
            {
                sb.AppendFormat("<li class=\"spb-nav-item\"><a {0} href=\"{1}\"><span>{2}</span></a>", rootCategory.ChildCount > 0 ? "class=\"sf-with-ul\"" : "", Globals.GetFullUrlInMainSite(SPBUrlHelper.Action("ListPhotosByCategory", "ChannelPhoto", new RouteValueDictionary { { "categoryID", rootCategory.CategoryID } })), rootCategory.GetMultilingualCategoryName(CurrentUserLanguage));
                if (rootCategory.ChildCount > 0)
                    sb.Append(GetChildCategoriesTag(rootCategory.CategoryID));
                sb.Append("</li>");
            }
            return PartialView("Controls/Photos/SubMenus.ascx", sb.ToString());
        }

        /// <summary>
        /// 递归返回子类的标签
        /// </summary>
        /// <param name="categoryID"></param>
        /// <returns></returns>
        private string GetChildCategoriesTag(int categoryID)
        {
            SiteCategory category = SiteCategories.Instance(ApplicationIDs.Instance().Photo()).Get(categoryID, false);
            StringBuilder sb = new StringBuilder();
            sb.Append("<ul class=\"spb-nav-submenu\" style=\"disply:none;visibility:hidden\">");
            if (category.ChildCount > 0)
            {
                foreach (SiteCategory childCategory in category.Childs)
                {
                    sb.AppendFormat("<li class=\"spb-nav-item\"><a {0} href=\"{1}\"><span>{2}</span></a>", childCategory.ChildCount > 0 ? "class=\"sf-with-ul\"" : "", Globals.GetFullUrlInMainSite(SPBUrlHelper.Action("ListPhotosByCategory", "ChannelPhoto", new RouteValueDictionary { { "categoryID", childCategory.CategoryID } })), childCategory.GetMultilingualCategoryName(CurrentUserLanguage));
                    if (childCategory.ChildCount > 0)
                        sb.Append(GetChildCategoriesTag(childCategory.CategoryID));
                    sb.Append("</li>");
                }
            }
            sb.Append("</ul>");

            return sb.ToString();
        }

        #endregion

        #region 推荐图片

        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_RecommendPhotoSlideBlock(int? commendTypeID, int? displayCount)
        {
            IList<RecommendedItem> recommendItems = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().Photo(), commendTypeID, displayCount ?? 11);
            IList<PhotoThread> photos = new List<PhotoThread>();
            foreach (RecommendedItem item in recommendItems)
            {
                PhotoThread photo = PhotoThreads.GetThread(item.ItemID, PrivacyStatuses.Public, false);
                if (photo != null)
                {
                    photos.Add(photo);
                }

            }
            ViewData["recommendPhotos"] = photos;
            ViewData["RecommendationType"] = RecommendationTypes.GetRecommendationType(commendTypeID ?? 0, false);

            return PartialView("Controls/Photos/RecommendPhotoSlideBlock.ascx");
        }

        /// <summary>
        /// 推荐相册
        /// </summary>
        /// <param name="commendTypeID"></param>
        /// <param name="displayCount"></param>
        /// <returns></returns>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_RecommendPhotoAlbumBlock(int? commendTypeID, int? displayCount)
        {
            IList<RecommendedItem> recommendItems = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().Photo(), commendTypeID, displayCount ?? 11);
            //IList<UserCategory> albums = new List<UserCategory>();
            //foreach (RecommendedItem item in recommendItems)
            //{
            //    UserCategory album = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(item.ItemID, false);
            //    if (album != null)
            //    {
            //        albums.Add(album);
            //    }
            //}
            return PartialView("Controls/Photos/RecommendPhotoAlbumBlock.ascx", recommendItems);
        }

        #endregion

        #endregion
    }
}