﻿//------------------------------------------------------------------------------
//<copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights reserved.
//</copyright> 
//------------------------------------------------------------------------------ 

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using System.Collections;
using System.Web;
using SpaceBuilder.Common;
using SpaceBuilder.Photo;
using SpaceBuilder.Common.Controllers;
using SpaceBuilder.Utils;
using SpaceBuilder.Web;
using System.Web.Routing;
using System.Xml;
using SpaceBuilder.LuceneSearch;
using System.Web.UI;

namespace SpaceBuilder.Photo.Controllers
{
    /// <summary>
    /// 个人空间中图片
    /// </summary>
    [UserDomainGuideInstallApplication(ApplicationID = 112)]
    [HttpCompress]
    public class PhotoController : UserDomainControllerBase
    {
        private int applicationID = ApplicationIDs.Instance().Photo();

        /// <summary>
        /// 空间相册主页
        /// </summary>
        /// <param name="userDomainName">空间所有者用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult Home(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());

            #region 标题
            SetPageTitle(userDomainName, string.Empty);
            Header.AddRSDDiscovery(SiteUrls.Instance().GalleryRss(userDomainName));
            #endregion

            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                ViewData["isManger"] = true;

            List<UserCategory> categories = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategories(CurrentUser != null ? CurrentUser.UserID : 0, true);
            if (categories != null && categories.Count == 1)
                ViewData["isAllowDel"] = false;

            List<UserCategory> userCategoryList = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategories(currentDomainUser.UserID, true);

            return View("Pages/Photos/Home.aspx", "Masters/Photo.master", userCategoryList);
        }

        #region PhotoThread

        /// 图片全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Search(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());

            RouteValueDictionary parameters = new RouteValueDictionary();
            parameters["Keyword"] = Request.Form.GetString("Keyword", string.Empty);
            parameters["SearchScopeUserID"] = currentDomainUser.UserID.ToString();

            return RedirectToAction("Search", "ChannelPhoto", parameters);
        }

        /// <summary>
        /// 显示我被圈的照片
        /// </summary>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListTaggedPhotos(string userDomainName)
        {
            #region 标题
            SetPageTitle(userDomainName, string.Empty);
            #endregion
            return View("Pages/Photos/ListTaggedPhotos.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 显示被圈照片用户控件
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        //[ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListTaggedPhotos(string userDomainName, int? pageIndex)
        {
            if (!pageIndex.HasValue)
                pageIndex = 1;
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<PhotoThread> pds = PhotoThreads.GetMyTaggedThreads(currentDomainUser.UserID, pageIndex.Value);
            return PartialView("Controls/Photos/ListTaggedPhotos.ascx", pds);
        }

        /// <summary>
        /// 显示编辑用户分类模式框
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_EditUserCategory(string userDomainName, int? categoryID)
        {
            UserCategory UserCategory = new UserCategory();
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                UserCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(categoryID.Value, true);
                ValidateItemOwner(UserCategory.UserID);
                TempData["selectedCategoryID"] = categoryID.Value;
            }
            ViewData["categoryID"] = categoryID;
            return PartialView("Controls/Photos/EditUserCategory.ascx", UserCategory);
        }

        /// <summary>
        /// 创建更新用户分类
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreateUpdateUserCategory(string userDomainName, int? categoryID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string userCategoryItemKey = UserCategoryItemKeys.Instance().Photo();

            UserCategory userCategory = new UserCategory();
            if (categoryID != null && (int)categoryID > 0)
            {
                userCategory = UserCategoryManager.Instance(userCategoryItemKey).GetUserCategory(categoryID.Value, true);
            }
            else
            {
                userCategory.UserID = currentDomainUser.UserID;
            }
            userCategory.CategoryName = Request.Form.GetString("categoryName", string.Empty).Trim();
            userCategory.CategoryName = HtmlUtils.TrimHtml(userCategory.CategoryName, 20);
            userCategory.CategoryName = Formatter.FormatSingleLinePlainTextForStorage(userCategory.CategoryName);
            if (string.IsNullOrEmpty(userCategory.CategoryName))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_UserCategoryNameRequired", applicationID));
            }
            else
            {
                userCategory.Description = Request.Form.GetString("description", string.Empty);
                userCategory.Description = HtmlUtils.TrimHtml(userCategory.Description, 100);
                userCategory.Description = Formatter.FormatMultiLinePlainTextForStorage(userCategory.Description, true);
                userCategory.DisplayOrder = Request.Form.GetInt("displayOrder", 0);
                userCategory.PrivacyStatus = (PrivacyStatuses)Request.Form.GetInt("setPhotoPirvacyStatus", (int)PrivacyStatuses.Public);
                userCategory.Password = Request.Form.GetString("setPhotoPirvacyStatus_Password", string.Empty);

                if (categoryID.HasValue && (int)categoryID > 0)
                {
                    ValidateItemOwner(userCategory.UserID);
                    UserCategoryManager.Instance(userCategoryItemKey).UpdateUserCategory(userCategory);
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdateUserCategorySucceed", applicationID));
                }
                else
                {
                    UserCategoryManager.Instance(userCategoryItemKey).CreateUserCategory(userCategory);
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_CreateUserCategorySucceed", applicationID));
                }
            }
            return RedirectToAction("Control_EditUserCategory", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", userCategory.CategoryID } });
        }

        /// <summary>
        /// 删除用户分类
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteUserCategory(string userDomainName, int categoryID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string userCategoryItemKey = UserCategoryItemKeys.Instance().Photo();
            UserCategory userCategory = UserCategoryManager.Instance(userCategoryItemKey).GetUserCategory(categoryID, true);
            if (userCategory == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_UserCategoryNotExist", applicationID), categoryID)));
            }
            ValidateItemOwner(userCategory.UserID);

            List<UserCategory> categories = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategories(userCategory.UserID, true);
            if (categories != null && categories.Count == 1)
                return Redirect(SiteUrls.Instance().Error(GetResourceString("Message_DenyDelete", applicationID)));

            //删除相册下照片
            PagingDataSet<PhotoThread> pds = PhotoThreads.GetThreads(userCategory.UserID, null, userCategory.CategoryID, ValueHelper.GetSqlMaxInt() / PhotoConfiguration.Instance().CacheablePageCount, 1);
            PhotoThreads.DeleteThreads(pds.Records.Select(n => n.ThreadID).ToList());

            UserCategoryManager.Instance(userCategoryItemKey).DeleteUserCategory(categoryID, currentDomainUser.UserID);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteUserCategorySucceed", applicationID));
            return RedirectToAction("Home", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 显示缩略图
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListIconPhotos(string userDomainName, int? pageIndex)
        {
            #region 标题
            SetPageTitle(userDomainName, string.Empty);
            #endregion

            return View("Pages/Photos/ListIconPhotos.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 最新相册
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ListPhotos(string userDomainName, int? pageIndex)
        {
            return View("Pages/Photos/ListPhotos.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 摘要显示某个标签下的图片
        /// </summary>
        /// <param name="userDomainName">当前空间用户名</param>
        /// <param name="tagName">标签名称</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListIconPhotosByTag(string userDomainName, string tagName, int? pageIndex)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            #region 标题
            SetPageTitle(userDomainName, tagName);
            #endregion

            return View("Pages/Photos/ListIconPhotosByTag.aspx", "Masters/Photo.master");
        }


        /// <summary>
        /// 提交照片密码密码
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="categoryID"></param>
        /// <param name="photoBrowseType"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult ListPhotosForPassword(string userDomainName, int categoryID, PhotoBrowseType photoBrowseType)
        {
            RouteValueDictionary parameters = new RouteValueDictionary();
            parameters["password"] = Request.Form.GetString("password", string.Empty);
            parameters["userDomainName"] = userDomainName;
            parameters["categoryID"] = categoryID;

            UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);

            string actionName = "ListThumbnailPhotosByCategory";
            switch (photoBrowseType)
            {
                case PhotoBrowseType.ListWall:
                    actionName = "ListWallPhotos";
                    break;
                case PhotoBrowseType.SlideShow:
                    actionName = "SlideShowPhotosByCategory";
                    break;
                default:
                    break;
            }

            return RedirectToAction(actionName, parameters);
        }

        /// <summary>
        /// 图标显示（小视图）某个相簿下的图片
        /// </summary>
        /// <param name="userDomainName">当前空间用户名</param>
        /// <param name="categoryID">相簿ID</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = false)]
        public ActionResult ListThumbnailPhotosByCategory(string userDomainName, int categoryID, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());

            ViewData["currentDomainUser"] = currentDomainUser;

            UserCategory photoUserCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(categoryID, false);
            if (photoUserCategory != null)
            {
                SetPageTitle(userDomainName, photoUserCategory.CategoryName);

                bool isManger = false;
                if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                    isManger = true;

                ViewData["isManger"] = isManger;
                ViewData["currentCategory"] = photoUserCategory;

                switch (photoUserCategory.PrivacyStatus)
                {
                    case PrivacyStatuses.NeedPassword:
                        UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
                        ViewData["passwordRight"] = userCookie.AuthorizePhotoCatetoryID(photoUserCategory.CategoryID);
                        break;
                    case PrivacyStatuses.OnlyFriend:
                        if (CurrentUser != null && Friends.IsFriend(CurrentUser.UserID, currentDomainUser.UserID))
                            ViewData["isMyFriend"] = true;
                        break;
                    default:
                        break;
                }
            }

            return View("Pages/Photos/ListThumbnailPhotosByCategory.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 幻灯显示 相簿下的图片
        /// </summary>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult SlideShowPhotosByCategory(string userDomainName, int categoryID, int? pageSize)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());

            ViewData["currentDomainUser"] = currentDomainUser;

            UserCategory photoUserCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(categoryID, false);
            if (photoUserCategory == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_CalleryNotExist", applicationID), categoryID)));
            }
            else
            {
                SetPageTitle(userDomainName, photoUserCategory.CategoryName);

                bool isManager = false;
                if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                    isManager = true;
                ViewData["currentCategory"] = photoUserCategory;
                ViewData["isManger"] = isManager;

                switch (photoUserCategory.PrivacyStatus)
                {
                    case PrivacyStatuses.NeedPassword:
                        UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
                        ViewData["passwordRight"] = userCookie.AuthorizePhotoCatetoryID(photoUserCategory.CategoryID);
                        break;
                    case PrivacyStatuses.OnlyFriend:
                        if (CurrentUser != null && Friends.IsFriend(CurrentUser.UserID, currentDomainUser.UserID))
                            ViewData["isMyFriend"] = true;
                        break;
                    default:
                        break;
                }
            }
            return View("Pages/Photos/SlideShowPhotosByCategory.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 根据类别显示图片列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListWallPhotos(string userDomainName, int categoryID, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());

            ViewData["currentDomainUser"] = currentDomainUser;

            UserCategory photoUserCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(categoryID, false);
            if (photoUserCategory != null)
            {
                SetPageTitle(userDomainName, photoUserCategory.CategoryName);

                bool isManger = false;
                if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                    isManger = true;
                ViewData["isManger"] = isManger;
                ViewData["currentCategory"] = photoUserCategory;

                switch (photoUserCategory.PrivacyStatus)
                {
                    case PrivacyStatuses.NeedPassword:
                        UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
                        ViewData["passwordRight"] = userCookie.AuthorizePhotoCatetoryID(photoUserCategory.CategoryID);
                        break;
                    case PrivacyStatuses.OnlyFriend:
                        if (CurrentUser != null && Friends.IsFriend(CurrentUser.UserID, currentDomainUser.UserID))
                            ViewData["isMyFriend"] = true;
                        break;
                    default:
                        break;
                }
            }
            return View("Pages/Photos/ListWallPhotos.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 根据标签显示图片列表
        /// </summary>
        /// [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListPhotosByTag(string userDomainName, string tagName, int? pageIndex)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            #region 标题
            SetPageTitle(userDomainName, tagName);
            #endregion

            return View("Pages/Photos/ListPhotosByTag.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 显示某个相册中的所有图片
        /// </summary>
        /// <param name="userDomainName">空间用户名</param>
        /// <param name="categoryID">相簿ID</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ShowPhotoCategory(string userDomainName, int categoryID, int? pageIndex)
        {
            #region 标题
            UserCategory photoUserCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(categoryID, false);
            if (photoUserCategory != null)
                SetPageTitle(userDomainName, photoUserCategory.CategoryName);
            #endregion

            return View("Pages/Photos/ShowPhotoCategory.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 提交照片密码密码
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="categoryID"></param>
        /// <param name="photoBrowseType"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult ShowPhotoForPassword(string userDomainName, int threadID)
        {
            RouteValueDictionary parameters = new RouteValueDictionary();
            parameters["password"] = Request.Form.GetString("password", string.Empty);
            parameters["userDomainName"] = userDomainName;
            parameters["threadID"] = threadID;

            return RedirectToAction("ShowPhoto", parameters);
        }

        /// <summary>
        /// 图片详细显示
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        /// <param name="postID">显示图片的PostID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ShowPhoto(string userDomainName, int threadID, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PhotoThread currentPhoto = PhotoThreads.GetThread(threadID, null, HasManagePermission(userDomainName));
            if (currentPhoto == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), threadID)));
            }

            #region 处理审核
            int auditingStatusValue = (int)Applications.GetApplication(ApplicationIDs.Instance().Photo()).AuditingStatusForPublicDisplay;
            if ((int)currentPhoto.AuditingStatus < auditingStatusValue)
            {
                if (currentPhoto.AuditingStatus == AuditingStatuses.Fail)
                    ViewData["errorMessage"] = GetResourceString("Message_ImageNotAuditingPass", applicationID);
                else
                    ViewData["errorMessage"] = GetResourceString("Message_ThisImageIsAuditinging", applicationID);
                ViewData["errorAuditingStatus"] = true;
            }
            #endregion

            ViewData["currentPhoto"] = currentPhoto;
            ViewData["currentDomainUser"] = currentDomainUser;
            ViewData["currentUser"] = CurrentUser;
            if (CurrentUser == null || (!IsUserDomainOwner(userDomainName) && !CurrentUser.IsContentAdministrator))
            {
                if (currentPhoto.UserCategory != null)
                {
                    switch (currentPhoto.UserCategory.PrivacyStatus)
                    {
                        case PrivacyStatuses.NeedPassword:
                            UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
                            ViewData["passwordRight"] = userCookie.AuthorizePhotoCatetoryID(currentPhoto.UserCategoryID);
                            break;
                        case PrivacyStatuses.OnlyFriend:
                            if (CurrentUser != null && Friends.IsFriend(CurrentUser.UserID, currentDomainUser.UserID))
                                ViewData["isMyFriend"] = true;
                            break;
                        default:
                            break;
                    }
                }
            }

            #region 标题 计数
            AddWebCount<PhotoSectionCounter>(currentDomainUser.UserID);
            AddWebCount<PhotoCounter>(threadID);
            base.SetPageTitle(userDomainName, currentPhoto.Subject);
            #endregion

            #region 设置meta标签
            SetPageMeta(currentPhoto);
            #endregion

            bool isOwner = false;
            if (CurrentUser != null && (CurrentUser.IsContentAdministrator || IsUserDomainOwner(userDomainName)))
            {
                isOwner = true;
            }

            ViewData["sildePhotosPageIndex"] = PhotoThreads.GetSequenceNumber(currentDomainUser.UserID, currentPhoto.UserCategoryID, threadID, isOwner);

            IList<Rating> ratings = PhotoThreads.GetRatings(threadID, 30);
            ViewData["ratings"] = ratings;

            ViewData["topSpecialOrder"] = this.setSpecialOrder;

            return View("Pages/Photos/ShowPhoto.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 验证隐私密码
        /// </summary>
        [HttpGet]
        public ActionResult Control_PrivacyPassword(string userDomainName, int categoryID)
        {

            return PartialView("Controls/Photos/PrivacyPassword.ascx");
        }
        /// <summary>
        /// 验证隐私密码
        /// </summary>
        [HttpPost]
        public ActionResult AuthorizePrivacyPassword(string userDomainName, int categoryID)
        {
            UserCategory userCagetory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(categoryID, true);
            if (userCagetory == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format("找不到CategoryID为{0}的相册", categoryID));
                return RedirectToAction("Control_PrivacyPassword", CurrentRouteValueDictionary);
            }

            string password = Request.Form.GetString("password", string.Empty);
            bool isSuccess = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).ValidPrivacyPassword(password, userCagetory.Password);
            if (isSuccess)
            {
                UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);

                userCookie.SaveAuthorizedPhotoCatetoryIDs(categoryID, password);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "通过验证");
            }
            else
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "密码错误");
            }
            return RedirectToAction("Control_PrivacyPassword", CurrentRouteValueDictionary);
        }


        /// <summary>
        /// 查看不同尺寸大小的图片
        /// </summary>
        /// <param name="userDomainName">空间用户的用户名</param>
        /// <param name="postID">图片的PostID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ShowSizesPhoto(string userDomainName, int threadID, PhotoImageType? photoImageType)
        {
            PhotoThread currentPhoto = PhotoThreads.GetThread(threadID, null, false);
            if (currentPhoto == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), threadID)));
            }
            PhotoImageType imageType = PhotoImageType.Icon;

            if (photoImageType != null)
                imageType = (PhotoImageType)photoImageType;


            #region 计数 标题
            SetPageTitle(userDomainName, GetResourceString("Title_DifferSize", applicationID));
            #endregion

            ViewData["PhotoImageType"] = imageType;
            return View("Pages/Photos/ShowSizesPhoto.aspx", "Masters/Photo.master", currentPhoto);
        }

        /// <summary>
        /// 图片摘要列表
        /// </summary>
        /// <param name="userDomainName">当前空间用户名</param>
        /// <param name="categoroyID">所属相簿ID</param>
        /// <param name="tagName">所属标签</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListPhotos(string userDomainName, int? categoryID, string tagName, int? pageIndex, PhotoSortBy? sortBy, SortOrder? sortOrder)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            #region 计数
            AddWebCount<PhotoSectionCounter>(currentDomainUser.UserID);
            #endregion

            if (!pageIndex.HasValue || pageIndex.Value < 0)
                pageIndex = 1;

            PagingDataSet<PhotoThread> photos = new PagingDataSet<PhotoThread>();

            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
            {
                ViewData["IsManager"] = true;
                photos = PhotoThreads.GetMyThumbnailThreads(currentDomainUser.UserID, tagName, categoryID, pageIndex.Value);
            }
            else
            {
                ViewData["IsManager"] = false;
                photos = PhotoThreads.GetThumbnailThreads(currentDomainUser.UserID, tagName, categoryID, pageIndex.Value);
            }
            return PartialView("Controls/Photos/ListPhotos.ascx", photos);
        }

        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListWallPhotos(string userDomainName, int? categoryID, string tagName, int? pageIndex, PhotoSortBy? sortBy, SortOrder? sortOrder)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            #region 计数
            AddWebCount<PhotoSectionCounter>(currentDomainUser.UserID);
            #endregion

            if (!pageIndex.HasValue || pageIndex.Value < 0)
                pageIndex = 1;

            PagingDataSet<PhotoThread> photos = new PagingDataSet<PhotoThread>();

            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
            {
                ViewData["IsManager"] = true;
                photos = PhotoThreads.GetMyWallThreads(currentDomainUser.UserID, tagName, categoryID, pageIndex.Value);
            }
            else
            {
                ViewData["IsManager"] = false;
                photos = PhotoThreads.GetWallThreads(currentDomainUser.UserID, tagName, categoryID, pageIndex.Value);
            }
            return PartialView("Controls/Photos/ListWallPhotos.ascx", photos);
        }

        /// <summary>
        /// 朋友的图片
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        //[UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        //[ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListFriendsPhotos(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            if (!pageIndex.HasValue)
                pageIndex = 0;
            PagingDataSet<PhotoSection> friendsPhotoSections = PhotoSections.GetFriendRecentPhotos(currentDomainUser.UserID, pageIndex.Value, 10);

            return PartialView("Controls/Photos/ListFriendsPhotos.ascx", friendsPhotoSections);
        }

        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_ShowUserCategory(string userDomainName, int categoryID)
        {
            UserCategory userCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(categoryID, true);
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                ViewData["isManager"] = true;
            else
                ViewData["isManager"] = false;

            return PartialView("Controls/Photos/ShowUserCategory.ascx", userCategory);
        }

        /// <summary>
        /// 图标显示图片列表
        /// </summary>
        /// <param name="userName">当前空间用户名</param>
        /// <param name="categoryID">所属类别ID</param>
        /// <param name="tagName">标签名称</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        public ActionResult Control_ListThumbnailPhotos(string userDomainName, int? categoryID, string tagName, int? pageIndex)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            #region 计数
            AddWebCount<PhotoSectionCounter>(currentDomainUser.UserID);
            #endregion

            PagingDataSet<PhotoThread> photos = new PagingDataSet<PhotoThread>();
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                photos = PhotoThreads.GetMyThumbnailThreads(currentDomainUser.UserID, tagName, categoryID, pageIndex.Value);
            else
                photos = PhotoThreads.GetThumbnailThreads(currentDomainUser.UserID, tagName, categoryID, pageIndex.Value);

            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                ViewData["IsManager"] = true;
            else
                ViewData["IsManager"] = false;
            return PartialView("Controls/Photos/ListThumbnailPhotos.ascx", photos);
        }

        /// <summary>
        /// 显示不同尺寸大小图片控件
        /// </summary>
        /// <param name="userDomainName">当前空间用户</param>
        /// <param name="postID">图片ID</param>
        /// <param name="photoImageType">显示图片类型</param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ShowSizesView(string userDomainName, int threadID, PhotoImageType photoImageType)
        {

            return PartialView("Controls/Photos/ShowSizesView.ascx");
        }

        public ActionResult Control_Search(string userDomainName)
        {

            return PartialView("Controls/Photos/Search.ascx");
        }

        public ActionResult Control_PhotoRss(string userDomainName)
        {

            return PartialView("Controls/Photos/PhotoRss.ascx");
        }

        /// <summary>
        /// 最新照片块
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_LatestPhotosBlock(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<PhotoThread> photos = null;
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
            {
                photos = PhotoThreads.GetMyThumbnailThreads(currentDomainUser.UserID, null, null, 1);
            }
            else
            {
                photos = PhotoThreads.GetThumbnailThreads(currentDomainUser.UserID, null, null, 1);
            }

            return PartialView("Controls/Photos/LatestPhotosBlock.ascx", photos.Records);
        }

        /// <summary>
        /// 显示图片的Exif信息
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ShowPhotoExif(string userDomainName, int? threadID)
        {
            if (!threadID.HasValue)
                threadID = 0;

            #region 计数 标题
            SetPageTitle(userDomainName, GetResourceString("Action_Info", applicationID));
            #endregion

            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PhotoThread currentPhoto = PhotoThreads.GetThread(threadID.Value, null, false);
            if (currentPhoto == null)
                throw new ArgumentException(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), threadID));

            List<TunyNet.ImageMetadata.Tag> mateTags = PhotoThreads.GetExifTags(threadID.Value);
            ViewData["currentPhoto"] = currentPhoto;
            return View("Pages/Photos/ShowPhotoExif.aspx", "Masters/Photo.master", mateTags);
        }

        /// <summary>
        /// 页面右侧滑动显示图片
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        //[ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_SlideListPhotos(string userDomainName, int threadID, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PhotoThread currentPhoto = PhotoThreads.GetThread(threadID, null, false);
            if (currentPhoto == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), threadID)));
            }

            if (!pageIndex.HasValue || pageIndex.Value < 0)
                pageIndex = 1;

            PagingDataSet<PhotoThread> photos = new PagingDataSet<PhotoThread>();
            if (CurrentUser != null && (CurrentUser.IsContentAdministrator || IsUserDomainOwner(userDomainName)))
            {
                photos = PhotoThreads.GetMySildeThreads(currentDomainUser.UserID, currentPhoto.UserCategoryID, pageIndex.Value);
            }
            else
                photos = PhotoThreads.GetSildeThreads(currentDomainUser.UserID, currentPhoto.UserCategoryID, pageIndex.Value);

            return PartialView("Controls/Photos/SlideListPhotos.ascx", photos);
        }

        /// <summary>
        /// 幻灯片显示图片
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_SlideShowPhotos(string userDomainName, int? categoryID, string tagName)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<PhotoThread> photos = new PagingDataSet<PhotoThread>();
            if (HasManagePermission(userDomainName))
                photos = PhotoThreads.GetMySildeThreads(currentDomainUser.UserID, tagName, categoryID, 1);
            else
                photos = PhotoThreads.GetSlideThreads(currentDomainUser.UserID, tagName, categoryID, 1);

            StringBuilder elements = new StringBuilder();
            int i = 0;
            elements.AppendFormat("<script type=\"text/javascript\">  var FullGalleryShowArr =[");
            foreach (PhotoThread photo in photos.Records)
            {
                if (photo != null)
                {
                    if (i > 0)
                    {
                        elements.AppendFormat(",new SlideshowElement('{0}',{1},'{2}','{3}','{4}','{5}','{6}')", WebUtils.FullPath(SiteUrls.Instance().PictureUrl(photo.ThreadID, PhotoImageType.Details)), i, HtmlUtils.StripAllTags(photo.Subject, false), HtmlUtils.StripAllTags(photo.Subject, false), SPBUrlHelper.Action("ShowPhoto", "Photo", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", photo.ThreadID } }), HtmlUtils.StripAllTags(photo.Subject, false), WebUtils.FullPath(SiteUrls.Instance().PictureUrl(photo.ThreadID, PhotoImageType.Icon)));
                    }
                    else
                    {
                        elements.AppendFormat("new SlideshowElement('{0}',{1},'{2}','{3}','{4}','{5}','{6}')", WebUtils.FullPath(SiteUrls.Instance().PictureUrl(photo.ThreadID, PhotoImageType.Details)), i, HtmlUtils.StripAllTags(photo.Subject, false), HtmlUtils.StripAllTags(photo.Subject, false), SPBUrlHelper.Action("ShowPhoto", "Photo", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", photo.ThreadID } }), HtmlUtils.StripAllTags(photo.Subject, false), WebUtils.FullPath(SiteUrls.Instance().PictureUrl(photo.ThreadID, PhotoImageType.Icon)));
                    }
                    i++;
                }
            }
            elements.AppendFormat(" ];  </script>");
            ViewData["elements"] = elements.ToString();

            return PartialView("Controls/Photos/SlideShowPhotos.ascx");
        }

        #endregion

        #region Manage

        /// <summary>
        /// 批量上传图片
        /// </summary>
        /// <param name="userDomainName">当前空间用户名</param>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult UploadPhotos(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_SetManyImage", applicationID));
            #endregion

            PhotoAttachmentSettings photoAttachmentSettings = (PhotoAttachmentSettings)PhotoAttachmentManager.Instance().Settings;
            string uploadPhotoTypes = PhotoAttachmentManager.Instance().Settings.Extensions;
            StringBuilder newUploadPhotoTypes = new StringBuilder();
            string[] types = uploadPhotoTypes.Split(',');
            foreach (string type in types)
            {
                if (!string.IsNullOrEmpty(type))
                    newUploadPhotoTypes.AppendFormat("*.{0};", type);
            }
            ViewData["uploadPhotoTypes"] = newUploadPhotoTypes.ToString();
            ViewData["photoAttachmentSettings"] = photoAttachmentSettings;
            return View("pages/Photos/UploadPhotos.aspx", "Masters/Photo.master", currentDomainUser);
        }

        /// <summary>
        /// 编辑图片
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        /// <param name="postID">需要编辑的PictureID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult EditPhoto(string userDomainName, int? threadID)
        {
            if (!threadID.HasValue && SPBConfig.Instance().EnableDistributedStorage)
                return RedirectToAction("UploadPhotos", new RouteValueDictionary { { "userDomainName", userDomainName } });

            User CurrentDomainUser = GetCurrentDomainUser(userDomainName);

            PhotoSection currentPhotoSection = PhotoSections.Get(CurrentDomainUser.UserID, false);
            PhotoThread currentPhoto = new PhotoThread();
            if (threadID != null && (int)threadID > 0)
            {
                currentPhoto = PhotoThreads.GetThread((int)threadID, null, true);
                if (currentPhoto == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), threadID)));
                }
                ValidateItemOwner(currentPhoto.OwnerUserID);
                UserCategory category = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(currentPhoto.UserCategoryID, true);
                ViewData["isFeaturedPhoto"] = category.FeaturedItemID == currentPhoto.ThreadID;
            }
            else
            {
                currentPhoto.ThreadID = -1;
            }

            #region 标题
            if (currentPhoto.ThreadID > 0)
                SetPageTitle(userDomainName, string.Format(GetResourceString("Title_Pattern_Edit", applicationID), currentPhoto.Subject));
            else
                SetPageTitle(userDomainName, GetResourceString("Title_CreateImage", applicationID));
            #endregion

            ViewData["currentDomainUser"] = CurrentDomainUser;
            ViewData["restrictIllustration"] = string.Format(GetResourceString("Message_Pattern_AllowUpLoadTypeSizeMax", applicationID), PhotoAttachmentManager.Instance().Settings.Extensions, PhotoAttachmentManager.Instance().Settings.MaxAttachmentSize);
            if (threadID.HasValue && threadID.Value > 0)
                ViewData["SelectedUserTags"] = UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).GetUserTagsOfItem(currentPhoto.ThreadID, HasManagePermission(CurrentDomainUser));
            return View("Pages/Photos/EditPhoto.aspx", "Masters/Photo.master", currentPhoto);
        }

        /// <summary>
        ///  评论编辑图片
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult EditPhotos(string userDomainName, int? photoCount)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (!photoCount.HasValue || photoCount.Value < 0)
                photoCount = 1;

            PagingDataSet<PhotoThread> editPhotos = PhotoThreads.GetThreadsForAdmin(photoCount.Value, 1, null, currentDomainUser.UserID, string.Empty);
            StringBuilder editPhotoIDs = new StringBuilder();
            foreach (PhotoThread photo in editPhotos.Records)
            {
                if (photo != null)
                {
                    editPhotoIDs.AppendFormat("{0},", photo.ThreadID);
                    ValidateItemOwner(photo.OwnerUserID);
                }
            }

            ViewData["editPhotoIDs"] = editPhotoIDs.ToString();

            return View("Pages/Photos/EditPhotos.aspx", "Masters/Photo.master", editPhotos);
        }

        /// <summary>
        /// 批量更新图片
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UpdatePhotos(string userDomainName, string photoIDs)
        {
            int categoryID = 0;

            if (!string.IsNullOrEmpty(photoIDs))
            {
                string[] editeIDsString = photoIDs.Split(',');
                List<int> editIDs = new List<int>();
                foreach (string editIDstring in editeIDsString)
                {
                    if (!string.IsNullOrEmpty(editIDstring))
                        editIDs.Add(Convert.ToInt32(editIDstring));
                }

                int featureItemID = Request.Form.GetInt("categoryFeatureItemID", 0);

                foreach (int photoID in editIDs)
                {
                    PhotoThread photo = PhotoThreads.GetThread(photoID, null, true);
                    if (photo != null)
                    {
                        if (categoryID == 0)
                            categoryID = photo.UserCategoryID;

                        ValidateItemOwner(photo.OwnerUserID);
                        if (photoID == featureItemID && photo.UserCategory != null)
                        {
                            photo.UserCategory.FeaturedItemID = featureItemID;
                            UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).UpdateUserCategory(photo.UserCategory);
                        }

                        photo.Subject = Request.Form.GetString(string.Format("subject_{0}", photoID), photo.Subject).Trim();
                        photo.BodyForDataTransmission = Request.Form.GetString(string.Format("body_{0}", photoID), string.Empty);
                        photo.BodyForDataTransmission = Formatter.FormatMultiLinePlainTextForStorage(photo.BodyForDataTransmission, true, false);
                        if (PhotoConfiguration.Instance().EableUserEntrySiteCategory)
                            photo.SiteCategoryID = Request.Form.GetInt(string.Format("photoSiteCategoryDDL_{0}", photoID), 0);
                        string userTags = Request.Form.GetString(string.Format("userTags_{0}", photoID), string.Empty);
                        List<string> selectedTagNames = null;
                        List<UserTag> selectedTags = null;

                        if (!string.IsNullOrEmpty(userTags))
                        {
                            string[] tagNames = userTags.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                            if (tagNames != null && tagNames.Length > 0)
                            {
                                selectedTagNames = new List<string>();
                                selectedTags = new List<UserTag>();

                                foreach (string tagName in tagNames)
                                {
                                    if (!string.IsNullOrEmpty(tagName.Trim()))
                                    {
                                        UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).GetUserTag(tagName.Trim(), photo.OwnerUserID, true);
                                        if (tag == null)
                                        {
                                            UserTag newTag = new UserTag();
                                            newTag.UserID = photo.OwnerUserID;
                                            newTag.TagName = StringUtils.Trim(tagName, 8, string.Empty);
                                            newTag.TagName = Formatter.CleanTagName(newTag.TagName);
                                            newTag.TagName = Formatter.FormatSingleLinePlainTextForStorage(newTag.TagName);

                                            if (UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).CreateUserTag(newTag))
                                            {
                                                selectedTagNames.Add(newTag.TagName);
                                                selectedTags.Add(newTag);
                                            }
                                        }
                                        else
                                        {
                                            selectedTagNames.Add(tag.TagName);
                                            selectedTags.Add(tag);
                                        }
                                    }
                                }
                            }
                        }

                        PhotoThreads.UpdateThread(photo, photo.UserCategoryID, selectedTags, null);
                        if (selectedTagNames != null && selectedTagNames.Count > 0)
                            UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).UpdateUserTagsOfItem(photo.ThreadID, photo.OwnerUserID, selectedTagNames);
                    }
                }
            }

            return RedirectToAction("ListThumbnailPhotosByCategory", "Photo", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", categoryID } });
        }

        /// <summary>
        /// 更新图片
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        /// <param name="postID">图片的PostID</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UpdatePhoto(string userDomainName, int? threadID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PhotoThread currentPhoto = new PhotoThread();
            PhotoSection currentPhotoSectioin = PhotoSections.Get(currentDomainUser.UserID, false);
            if (threadID != null && (int)threadID > 0)
            {
                currentPhoto = PhotoThreads.GetThread((int)threadID, null, true);
                if (currentPhoto == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), (int)threadID)));
                }
                //throw new ArgumentException(string.Format("{0}图片不存在", (int)threadID));
                ValidateItemOwner(currentPhoto.OwnerUserID);
            }
            else
                currentPhoto.ThreadID = -1;

            currentPhoto.Subject = Request.Form.Get<string>("PhotoName", string.Empty).Trim();
            if (string.IsNullOrEmpty(currentPhoto.Subject))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_PleaseWriteImageName", applicationID);
            }
            else
            {
                int oldUserCategoryID = currentPhoto.UserCategoryID;
                currentPhoto.BodyForDataTransmission = Request.Form.Get<string>("PhotoDescription", string.Empty);
                int oldCategoryID = currentPhoto.UserCategoryID;
                currentPhoto.UserCategoryID = Request.Form.GetInt("photoUserCategoryDDL", 0);
                currentPhoto.SiteCategoryID = Request.Form.GetInt("photoSiteCategoryDDL", 0);

                if (!(currentPhoto.ThreadID > 0))
                {
                    currentPhoto.OwnerUserID = currentDomainUser.UserID;
                    currentPhoto.Author = currentDomainUser.DisplayName;
                }

                #region 已经注释掉
                //特征图片被移到其他相簿时，如果原相簿中还有图片那么选择最早上传的图片为特征图片，如果原相簿没有其他图片了那么就是默认特征图片；如果写在图片更新里不太合适也因为只有变更所述相簿的时候才会出现这种情况
                //实现功能：如上所述，并且如果转移到的相簿没有图片，那么设置特征图片；
                //这些写在这里不合适....yangmj 200807081422
                //begin
                //UserCategory oldCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(currentPhoto.UserCategoryID);
                //if (oldCategory != null && currentPhoto.ThreadID == oldCategory.FeaturedItemID && currentPhoto.UserCategoryID != Convert.ToInt32(Request.Form["PhotoCategoriesDropDownList"]))
                //{
                //    if (currentPhoto.UserCategory.ItemCount > 1)
                //    {
                //        if (oldCategory != null)
                //        {
                //            PhotoQuery query = new PhotoQuery();
                //            query.UserCategoryID = currentPhoto.UserCategoryID;
                //            query.OwnerUserID = currentPhoto.OwnerUserID;
                //            query.SortBy = PhotoSortBy.MostRecent;
                //            query.SortOrder = SortOrder.Ascending;
                //            PagingDataSet<PhotoThread> pics = PhotoThreads.GetPhotos(query);
                //            if (pics.Records.Count > 1)
                //            {
                //                foreach (PhotoThread pic in pics.Records)
                //                {
                //                    if (pic.ThreadID != currentPhoto.ThreadID)
                //                    {
                //                        oldCategory.FeaturedItemID = pic.ThreadID;
                //                        break;
                //                    }
                //                }
                //            }
                //            else
                //            {
                //                oldCategory.FeaturedItemID = 0;
                //            }
                //        }
                //    }
                //    else
                //        oldCategory.FeaturedItemID = 0;
                //    UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).UpdateUserCategory(oldCategory);
                //}
                //UserCategory newCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(Convert.ToInt32(Request.Form["PhotoCategoriesDropDownList"]));
                //if (newCategory != null && newCategory.ItemCount == 0)
                //{
                //    newCategory.FeaturedItemID = currentPhoto.ThreadID;
                //    UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).UpdateUserCategory(newCategory);
                //}
                //end

                //currentPhoto.UserCategoryID = Convert.ToInt32(Request.Form["PhotoCategoriesDropDownList"]);

                ////处理图片类别
                //// IList<GalleryPostCategory> currentCategories = Pictures.GetPictureCategories(currentPhoto);
                ////if (currentCategories == null || currentCategories.Count == 0)
                ////{
                ////    if (!ValueHelper.IsNullOrEmpty(categories.SelectedValue))
                ////    {
                ////        Pictures.UpdatePictureCategories(currentPhoto, new string[] { categories.SelectedItem.Text });
                ////    }
                ////}
                ////else
                ////{
                ////    if (ValueHelper.IsNullOrEmpty(categories.SelectedValue))
                ////    {
                ////        Pictures.UpdatePictureCategories(currentPhoto, null);
                ////    }
                ////    else if (!currentCategories[0].CategoryName.Equals(categories.SelectedItem.Text, StringComparison.CurrentCultureIgnoreCase))
                ////    {
                ////        Pictures.UpdatePictureCategories(currentPhoto, new string[] { categories.SelectedItem.Text });
                ////    }
                ////}

                //if ((pictureData != null) && (pictureData.PostedFile != null) && (pictureData.PostedFile.ContentLength > 0))
                //{
                //    PhotoAttachmentSettings gas = PhotoConfiguration.Instance().AttachmentSettings;
                //    if (gas.ValidationRegex.IsMatch(pictureData.PostedFile.FileName))
                //    {
                //        Pictures.CreatePicture(currentGallery.SectionID, currentPhoto, new PostAttachment(pictureData.PostedFile));
                //    }
                //    else
                //    {
                //        Pictures.UpdatePicture(currentPhoto);
                //    }
                //}
                //else
                //{
                //    PhotoThreads.UpdateThread(currentPhoto);
                //}
                #endregion

                bool fileChanged = false;
                System.Web.HttpPostedFileBase hpf = null;
                if (Request.Files.Count > 0)
                {
                    hpf = Request.Files[0] as System.Web.HttpPostedFileBase;
                    if (hpf.ContentLength == 0 && currentPhoto.ThreadID > 0)
                    {
                        fileChanged = false;
                    }
                    else
                    {
                        if (hpf.ContentLength == 0)
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_UpLoadImageSizeCannotMax", applicationID), 0);
                            return RedirectToAction("EditPhoto", "Photo", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", currentPhoto.ThreadID } });
                        }
                        if (hpf.ContentLength > PhotoAttachmentManager.Instance().Settings.MaxAttachmentSize * 1024)
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_UpLoadImageSizeCannotMax", applicationID), PhotoAttachmentManager.Instance().Settings.MaxAttachmentSize);
                            return RedirectToAction("EditPhoto", "Photo", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", currentPhoto.ThreadID } });
                        }
                        if (!string.IsNullOrEmpty(hpf.FileName))
                        {
                            string fileTypeString = hpf.FileName.Substring(hpf.FileName.LastIndexOf(".") + 1);
                            string uploadPhotoTypes = PhotoAttachmentManager.Instance().Settings.Extensions;
                            StringBuilder newUploadPhotoTypes = new StringBuilder();
                            string[] types = uploadPhotoTypes.Split(',');
                            //List<string> typesString = new List<string>();
                            bool isPermitted = false;
                            foreach (string type in types)
                            {
                                if (!isPermitted && !string.IsNullOrEmpty(type))
                                {
                                    if (string.Compare(type, fileTypeString, true) == 0)
                                    {
                                        isPermitted = true;
                                    }
                                }
                            }
                            if (!isPermitted)
                            {
                                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_AllowUpLoadType", applicationID), PhotoAttachmentManager.Instance().Settings.Extensions);
                                return RedirectToAction("EditPhoto", "Photo", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", currentPhoto.ThreadID } });
                            }
                        }
                        string newfileName = hpf.FileName.Substring(hpf.FileName.LastIndexOf("\\") + 1);
                        if (!(currentPhoto.ThreadID > 0) || (currentPhoto != null && !(newfileName == currentPhoto.Attachment.FriendlyFileName && hpf.ContentLength == (int)currentPhoto.Attachment.ContentSize)))
                        {
                            Attachment attachment = new Attachment(PhotoAttachmentManager.Instance().StorageProviderName, newfileName);
                            attachment.ContentSize = hpf.ContentLength;
                            attachment.ContentType = hpf.ContentType;
                            attachment.DateCreated = DateTime.Now;
                            attachment.FriendlyFileName = newfileName;
                            currentPhoto.Attachment = attachment;
                            fileChanged = true;
                        }
                    }
                }


                List<UserTag> selectedTags = new List<UserTag>();
                string tags = Request.Form.GetString("userTags", string.Empty);
                if (!string.IsNullOrEmpty(tags))
                {
                    string[] tagNames = tags.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                    if (tagNames != null && tagNames.Length > 0)
                    {
                        foreach (string tagName in tagNames)
                        {
                            if (!string.IsNullOrEmpty(tagName.Trim()))
                            {
                                UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).GetUserTag(tagName.Trim(), currentPhoto.OwnerUserID, true);
                                if (tag == null)
                                {
                                    UserTag newTag = new UserTag();
                                    newTag.UserID = currentPhoto.OwnerUserID;
                                    newTag.TagName = StringUtils.Trim(tagName, 8, string.Empty);
                                    newTag.TagName = Formatter.CleanTagName(newTag.TagName);
                                    newTag.TagName = Formatter.FormatSingleLinePlainTextForStorage(newTag.TagName);

                                    if (UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).CreateUserTag(newTag))
                                        selectedTags.Add(newTag);
                                }
                                else
                                    selectedTags.Add(tag);
                            }
                        }
                    }
                }

                if (currentPhoto.ThreadID < 0 || currentPhoto.ThreadID == 0)
                {
                    PhotoThreads.CreateThread(currentPhoto, selectedTags, hpf.InputStream);
                    TempData["StatusMessageContent"] = GetResourceString("Message_ImageCreateSucecss", applicationID);
                }
                else if (fileChanged && hpf != null)
                {
                    PhotoThreads.UpdateThread(currentPhoto, oldUserCategoryID, selectedTags, hpf.InputStream);
                    TempData["StatusMessageContent"] = GetResourceString("Message_ImageUpdateSucecss", applicationID);
                }
                else
                {
                    PhotoThreads.UpdateThread(currentPhoto, oldUserCategoryID, selectedTags, null);
                    TempData["StatusMessageContent"] = GetResourceString("Message_ImageUpdateSucecss", applicationID);
                }
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;

                if (selectedTags.Count > 0)
                    UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).UpdateUserTagsOfItem(currentPhoto.ThreadID, currentPhoto.OwnerUserID, selectedTags.Select(n => n.TagName).ToList());
                else
                    UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).DeleteUserTagsOfItem(currentPhoto.ThreadID);

                bool setFeature = false;
                setFeature = Request.Form.Get<bool>("IsFeature", false);
                if (oldCategoryID != currentPhoto.UserCategoryID)
                {
                    UserCategory oldCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(oldCategoryID, true);
                    if (oldCategory.FeaturedItemID == currentPhoto.ThreadID)
                    {
                        oldCategory.FeaturedItemID = -1;
                        UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).UpdateUserCategory(oldCategory);
                    }
                }
                if (currentPhoto.UserCategory != null)
                {
                    if (currentPhoto.UserCategory.FeaturedItemID > 0)
                    {
                        if (currentPhoto.ThreadID == currentPhoto.UserCategory.FeaturedItemID)
                        {
                            if (!setFeature)
                            {
                                currentPhoto.UserCategory.FeaturedItemID = -1;
                                UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).UpdateUserCategory(currentPhoto.UserCategory);
                            }
                        }
                        else
                        {
                            if (setFeature)
                            {
                                currentPhoto.UserCategory.FeaturedItemID = currentPhoto.ThreadID;
                                UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).UpdateUserCategory(currentPhoto.UserCategory);
                            }
                        }
                    }
                    else
                    {
                        if (setFeature)
                        {
                            currentPhoto.UserCategory.FeaturedItemID = currentPhoto.ThreadID;
                            UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).UpdateUserCategory(currentPhoto.UserCategory);
                        }
                    }
                }

                #region 推荐到群组
                ////总站是否启用了群组模块
                //if (Applications.IsEnabled(ApplicationIDs.Instance().Photo()))
                //{
                //    string commendClubIDs = Request.Form.Get<string>("commendToMyClubs");
                //    //删除所有以前推荐的
                //    PhotoClubCommendManager.Instance().DeleteCommendItemsByItemID(currentPhoto.ThreadID, currentDomainUser.UserID);
                //    //获取推荐到的群组ID集合
                //    string[] selectedClubIDs = null;
                //    if (!string.IsNullOrEmpty(commendClubIDs))
                //    {
                //        selectedClubIDs = commendClubIDs.Split(',');

                //        if (selectedClubIDs != null && selectedClubIDs.Length > 0)
                //        {
                //            foreach (string clubIDstring in selectedClubIDs)
                //            {
                //                if (clubIDstring != null)
                //                {
                //                    ClubApplicationCommend commendItem = new ClubApplicationCommend();
                //                    commendItem.ThreadID = currentPhoto.ThreadID;
                //                    commendItem.AuthorUserID = currentDomainUser.UserID;
                //                    commendItem.ClubID = Convert.ToInt32(clubIDstring);
                //                    commendItem.Author = currentDomainUser.DisplayName;
                //                    commendItem.Commender = currentDomainUser.DisplayName;
                //                    commendItem.CommenderUserID = currentDomainUser.UserID;
                //                    commendItem.Subject = currentPhoto.Subject;

                //                    PhotoClubCommendManager.Instance().OperateThreadToClub(commendItem, DataProviderAction.Create);
                //                }
                //            }
                //        }
                //    }
                //}

                #endregion
            }
            return RedirectToAction("ShowPhoto", "Photo", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", currentPhoto.ThreadID } });
        }

        /// <summary>
        /// 删除图片
        /// </summary>
        /// <param name="postID">将要删除图片的threadID</param>
        /// <param name="userDomainName">图片所属用户用户名</param>
        /// <param name="pageIndex">当前页码（可为空）</param>       
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeletePhotoThread(int threadID, string userDomainName, int? pageIndex, bool isManage, int? categoryID)
        {
            PhotoThread photo = PhotoThreads.GetThread(threadID, null, true);
            if (photo == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), threadID)));
            }
            //throw new ArgumentException(string.Format("{0}图片不存在", threadID));
            ValidateItemOwner(photo.OwnerUserID);

            PhotoThreads.DeleteThread(photo);

            if (isManage)
                return RedirectToAction("Control_ManagePhotos", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", categoryID } });
            else
                return RedirectToAction("ListThumbnailPhotosByCategory", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", photo.UserCategoryID } });
        }

        /// <summary>
        /// 删除所有选中图片
        /// </summary>
        /// <param name="userDomainName">用户名</param>       
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeletePhotos(string userDomainName, int? categoryID)
        {
            PhotoSection currentPhotoSectioin = GetPhotoSection(userDomainName);
            string deletedIDsString = Request.Form["PhotoCheckBoxGroup"];
            if (!string.IsNullOrEmpty(deletedIDsString))
            {
                List<int> photoIDs = new List<int>();
                int currentID = 0;
                string[] deleteIDsStrArray = deletedIDsString.Split(',');
                foreach (string IDString in deleteIDsStrArray)
                {
                    currentID = int.Parse(IDString);
                    if (currentID > 0)
                    {
                        photoIDs.Add(currentID);
                    }
                }
                PhotoThreads.DeleteThreads(photoIDs);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_ImageSelectSetSuccess", applicationID));
            }
            return RedirectToAction("Control_ManagePhotos", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", categoryID } });
        }

        /// <summary>
        /// 批量设置标签
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetPhotosTags(string userDomainName, int? categoryID)
        {
            User photoOwner = GetCurrentDomainUser(userDomainName);
            string[] setPostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("PhotoCheckBoxGroup", string.Empty)))
            {
                //设置标签
                string selectedTagNames = Request.Form.Get<string>("setUserTags", string.Empty);
                List<string> tagNames = new List<string>();
                if (!string.IsNullOrEmpty(selectedTagNames))
                {

                    string[] selectdTagNamesArry = selectedTagNames.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                    if (selectdTagNamesArry.Length > 0)
                    {
                        foreach (string tagname in selectdTagNamesArry)
                        {
                            if (!string.IsNullOrEmpty(tagname.Trim()))
                            {
                                UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).GetUserTag(tagname, photoOwner.UserID, true);
                                if (tag != null) //设置本标签
                                {
                                    tagNames.Add(tag.TagName);
                                }
                                else //创建并设置标签
                                {
                                    UserTag userTag = new UserTag();
                                    userTag.UserID = photoOwner.UserID;
                                    userTag.TagName = StringUtils.Trim(tagname, 8, string.Empty);
                                    userTag.TagName = Formatter.CleanTagName(userTag.TagName);
                                    userTag.TagName = Formatter.FormatSingleLinePlainTextForStorage(userTag.TagName);
                                    bool isCreated = UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).CreateUserTag(userTag);
                                    if (isCreated)
                                        tagNames.Add(userTag.TagName);
                                }
                            }
                        }
                    }
                }
                if (tagNames.Count > 0)
                {
                    setPostIDs = Request.Form.GetString("PhotoCheckBoxGroup", string.Empty).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                    if (setPostIDs != null && setPostIDs.Length > 0)
                    {
                        List<int> setPhotoIDs = new List<int>();
                        foreach (string postID in setPostIDs)
                        {
                            int id = int.Parse(postID);
                            PhotoThread photo = PhotoThreads.GetThread(id, null, true);
                            if (photo != null)
                            {
                                UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).UpdateUserTagsOfItem(photo.ThreadID, photo.OwnerUserID, tagNames);
                            }
                        }
                    }
                }
            }
            return RedirectToAction("Control_ManagePhotos", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", categoryID } });
        }

        /// <summary>
        /// 批量设置用户分类
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetPhotosUserCategory(string userDomainName, int userCategoryID, int? showUserCategoryID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string[] setPostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("PhotoCheckBoxGroup", string.Empty)))
            {
                setPostIDs = Request.Form.GetString("PhotoCheckBoxGroup", string.Empty).Split(',');
                if (setPostIDs != null && setPostIDs.Length > 0)
                {
                    List<int> setPhotoIDs = new List<int>();
                    foreach (string postID in setPostIDs)
                    {
                        int id = int.Parse(postID);
                        PhotoThread photo = PhotoThreads.GetThread(id, null, true);
                        if (photo != null)
                        {
                            //ValidateItemOwner(blogThread.OwnerUserID);
                            setPhotoIDs.Add(id);
                        }
                    }
                    PhotoThreads.UpdateUserCategory(setPhotoIDs, userCategoryID, currentDomainUser.UserID);
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_ImageSelectSetSuccess", applicationID));
                }
            }
            return RedirectToAction("Control_ManagePhotos", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", showUserCategoryID } });
        }

        /// <summary>
        /// 批量设置站点分类
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetPhotosSiteCategory(string userDomainName, int siteCategoryID, int? userCategoryID)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            string[] setPostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("PhotoCheckBoxGroup", string.Empty)))
            {
                setPostIDs = Request.Form.GetString("PhotoCheckBoxGroup", string.Empty).Split(',');
                if (setPostIDs != null && setPostIDs.Length > 0)
                {
                    List<int> setPhotoIDs = new List<int>();
                    foreach (string postID in setPostIDs)
                    {
                        int id = int.Parse(postID);
                        PhotoThread photo = PhotoThreads.GetThread(id, null, true);
                        if (photo != null)
                        {
                            setPhotoIDs.Add(id);
                        }
                    }
                    PhotoThreads.UpdateSiteCategory(setPhotoIDs, siteCategoryID);

                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_ImageSelectSetSuccess", applicationID));
                }
            }
            return RedirectToAction("Control_ManagePhotos", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", userCategoryID } });
        }

        /// <summary>
        /// 管理用户图片
        /// </summary>
        /// <param name="userDomainName">用户名称</param>
        /// <param name="pageIndex">当前页码</param>
        /// <param name="categoryID">所属类别ID（可设置为空）</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult ManagePhotos(string userDomainName, int? pageIndex, int? categoryID)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManagerImage", applicationID));
            #endregion

            ViewData["pageIndex"] = pageIndex;
            ViewData["categoryID"] = categoryID;

            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            List<UserCategory> userCategories = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategories(currentDomainUser.UserID, true);
            ViewData["userCategoriesList"] = userCategories;
            List<SiteCategory> allSiteCategoryList = SiteCategories.Instance(applicationID).GetAll(false);
            ViewData["allSiteCategoryList"] = allSiteCategoryList;
            return View("Pages/Photos/ManagePhotos.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 管理图片控件
        /// </summary>
        /// <param name="userDomainName">当前用户名</param>
        /// <param name="categoryID">相簿ID</param>
        /// <param name="pageIndex">当前显示页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ManagePhotos(string userDomainName, int? pageIndex, int? categoryID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            PagingDataSet<PhotoThread> threads = PhotoThreads.GetMyThumbnailThreads(currentDomainUser.UserID, string.Empty, categoryID, pageIndex ?? 1);

            ViewData["allSiteCategoryList"] = SiteCategories.Instance(ApplicationIDs.Instance().Photo()).GetAll(false);
            ViewData["userCategoriesList"] = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategories(currentDomainUser.UserID, true);
            return PartialView("Controls/Photos/ManagePhotos.ascx", threads);
        }

        /// <summary>
        /// 设置为精华
        /// </summary>
        public ActionResult SetEssential(string userDomainName, int threadID)
        {
            PhotoThread photo = PhotoThreads.GetThread(threadID, PrivacyStatuses.Public, true);
            if (photo == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), threadID)));
            }
            //throw new ArgumentException(string.Format("图片：{0}不存在", threadID));
            if (!CurrentUser.IsContentAdministrator)
            {
                throw new ArgumentException(GetResourceString("Exception_AdminSetHighLimit"));
            }

            photo.IsEssential = true;
            PhotoThreads.SetEssential(photo);

            return new EmptyResult();
        }

        /// <summary>
        /// 设置为所属相簿标示图片
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetFeature(string userDomainName, int threadID)
        {
            PhotoThread photo = PhotoThreads.GetThread(threadID, PrivacyStatuses.Public, true);
            if (photo == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_ImageReviewNotExist", applicationID), threadID)));
            }
            //throw new ArgumentException(string.Format("图片：{0}不存在", threadID));
            if (photo.UserCategory == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Excepton_ImageNotHaveCallery", applicationID), threadID)));
                //throw new ArgumentException(string.Format("图片：{0}不属于任何相簿", threadID));
            }
            photo.UserCategory.FeaturedItemID = photo.ThreadID;
            UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).UpdateUserCategory(photo.UserCategory);

            return new EmptyResult();
        }

        /// <summary>
        /// 取消精华设置
        /// </summary>
        public ActionResult CancelEssential(string userDomainName, int threadID)
        {
            PhotoThread photo = PhotoThreads.GetThread(threadID, null, true);
            if (photo == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), threadID)));
            }
            //throw new ArgumentException(string.Format("图片：{0}不存在", threadID));
            if (!CurrentUser.IsContentAdministrator)
            {
                throw new ArgumentException(GetResourceString("Exception_AdminSetHighLimit"));
            }

            photo.IsEssential = false;
            PhotoThreads.SetEssential(photo);
            //PhotoThreads.UpdateThread(photo, photo.UserCategoryID, null, null);

            return new EmptyResult();
        }

        /// <summary>
        /// 设置为置顶
        /// </summary>
        public ActionResult SetTop(string userDomainName, int threadID)
        {
            PhotoThread photo = PhotoThreads.GetThread(threadID, PrivacyStatuses.Public, true);
            if (photo == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), threadID)));
            }
            //throw new ArgumentException(string.Format("图片：{0}不存在", threadID));
            if (!CurrentUser.IsContentAdministrator)
            {
                throw new ArgumentException(GetResourceString("Exception_AdminSetHighLimit"));
            }

            photo.SpecialOrder = this.setSpecialOrder;
            PhotoThreads.SetSticky(photo);

            return new EmptyResult();
        }

        /// <summary>
        /// 取消置顶
        /// </summary>
        public ActionResult CancelTop(string userDomainName, int threadID)
        {
            PhotoThread photo = PhotoThreads.GetThread(threadID, null, true);
            if (photo == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), threadID)));
            }
            //throw new ArgumentException(string.Format("图片：{0}不存在", threadID));
            if (!CurrentUser.IsContentAdministrator)
            {
                throw new ArgumentException(GetResourceString("Exception_AdminSetHighLimit"));
            }

            photo.SpecialOrder = 0;
            PhotoThreads.SetSticky(photo);

            //PhotoThreads.UpdateThread(photo, photo.UserCategoryID, null, null);

            return new EmptyResult();
        }

        #endregion

        #region PhotoComment

        #region Show

        /// <summary>
        /// 显示照片评论列表和评论框
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        /// <param name="postID">图片ID</param>
        /// <param name="pageIndex">当前评论页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_PhotoComments(string userDomainName, int threadID, bool isMiniStyle, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PhotoSection currentPhotoSectioin = GetPhotoSection(userDomainName);
            PhotoThread currentPhoto = PhotoThreads.GetThread(threadID, null, false);
            if (currentPhoto == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), threadID)));
            }

            ViewData["currentDomainUser"] = currentDomainUser;
            ViewData["currentUser"] = CurrentUser;
            ViewData["threadID"] = threadID;
            ViewData["currentPhotoSection"] = currentPhotoSectioin;
            UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
            ViewData["rememberMyName"] = userCookie.GetRememberMeInfor_name();
            ViewData["rememberMyUrl"] = userCookie.GetRememberMeInfor_url();

            if (isMiniStyle)
                pageIndex = 1;

            PagingDataSet<PhotoComment> pds = PhotoComments.GetCommentsOfThread(currentPhoto.ThreadID, pageIndex.HasValue ? pageIndex.Value : 1, HasManagePermission(userDomainName));

            bool isManager = false;
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                isManager = true;
            ViewData["isManager"] = isManager;

            if (isMiniStyle)
                return PartialView("Controls/Photos/PhotoMiniComments.ascx", pds);
            else
                return PartialView("Controls/Photos/PhotoComments.ascx", pds);
        }

        #endregion

        #region Manage

        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult CreatePhotoComment(string userDomainName, int threadID, bool isMiniStyle, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            PhotoSection currentPhotoSection = PhotoSections.Get(currentDomainUser.UserID, false);
            PhotoThread commentPhoto = PhotoThreads.GetThread(threadID, null, false);
            if (commentPhoto == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), threadID)));
            }
            //throw new ArgumentException(string.Format("{0}图片不存在", threadID));
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = string.Empty;
                if (isMiniStyle)
                    verifyCode = Request.Form.GetString("verifyCode_" + threadID, string.Empty);
                else
                    verifyCode = Request.Form.GetString("verifyCode", string.Empty);

                if (verifyCode != null)
                    verifyCode = verifyCode.Trim();
                int parentID = 0;
                if (isMiniStyle)
                    parentID = Request.Form.GetInt("parentID_" + threadID, 0);
                else
                    parentID = Request.Form.GetInt("parentID", 0);

                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_ValidateError", applicationID);
                    if (CurrentUser == null)
                    {
                        if (isMiniStyle)
                            TempData["url"] = Request.Form.GetString("url_" + threadID, string.Empty);
                        else
                            TempData["url"] = Request.Form.GetString("url", string.Empty);

                        if (isMiniStyle)
                            TempData["author"] = Request.Form.GetString("author_" + threadID, string.Empty);
                        else
                            TempData["author"] = Request.Form.GetString("author", string.Empty);
                    }
                    TempData["parentID"] = parentID;

                    if (isMiniStyle)
                        TempData["commentBody"] = Request.Form.GetString("commentBody_" + threadID, string.Empty);
                    else
                        TempData["commentBody"] = Request.Form.GetString("commentBody", string.Empty);

                    return Control_PhotoComments(userDomainName, threadID, isMiniStyle, pageIndex);
                }
            }

            PhotoComment comment = new PhotoComment();
            if (CurrentUser != null)
            {
                comment.UserID = CurrentUser.UserID;
            }
            comment.OwnerUserID = currentDomainUser.UserID;
            comment.ThreadID = commentPhoto.ThreadID;
            if (!currentPhotoSection.ModerateComments)
                comment.AuditingStatus = AuditingStatuses.Success;

            if (isMiniStyle)
                comment.Body = Request.Form.GetString("commentBody_" + threadID, string.Empty);
            else
                comment.Body = Request.Form.GetString("commentBody", string.Empty);

            if (isMiniStyle)
                comment.ParentID = Request.Form.GetInt("parentID_" + threadID, 0);
            else
                comment.ParentID = Request.Form.GetInt("parentID", 0);

            if (comment.ParentID > 0)
            {
                PhotoComment parentComment = PhotoComments.Get(comment.ParentID, null, true);
                if (parentComment != null)
                {
                    comment.ParentAuthor = parentComment.Author;
                    comment.ParentBody = parentComment.Body;
                    comment.ParentUserID = parentComment.UserID;
                    comment.ParentPostDate = parentComment.PostDate;
                    if (comment.Body.StartsWith(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author), StringComparison.CurrentCultureIgnoreCase))
                    {
                        if (comment.Body.Length > string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author).Length)
                        {
                            comment.Body = comment.Body.Substring(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author).Length);
                        }
                        else
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = GetResourceString("Message_PleaseRestoreWord", applicationID);
                            TempData["commentBody"] = comment.Body;
                            TempData["parentID"] = comment.ParentID;
                            return Control_PhotoComments(userDomainName, threadID, isMiniStyle, pageIndex);
                        }
                    }
                }
            }

            if (string.IsNullOrEmpty(comment.Body))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_RestoreWordNotNull", applicationID);
            }
            else
            {
                if (currentPhotoSection.ModerateComments)
                    comment.IsApproved = false;
                else
                    comment.IsApproved = true;

                if (CurrentUser == null)
                {
                    comment.Author = Request.Form.Get<string>("author", GetResourceString("Common_AnonymousDisplayName"));
                    comment.TitleUrl = Formatter.FormatSingleLinePlainTextForStorage(Request.Form.Get<string>("commentWebsite", string.Empty));
                    if (!string.IsNullOrEmpty(comment.TitleUrl) && !comment.TitleUrl.StartsWith("http://"))
                        comment.TitleUrl = string.Format("http://{0}", comment.TitleUrl);

                    UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
                    if (Request.Form.Get<int>("rememberMe", -1) > 0)
                    {
                        userCookie.SaveRememberMeInfor_name(WebUtils.HtmlEncode(comment.Author));
                        userCookie.SaveRememberMeInfor_url(WebUtils.HtmlEncode(comment.TitleUrl));
                    }
                    else
                    {
                        userCookie.SaveRememberMeInfor_name(string.Empty);
                        userCookie.SaveRememberMeInfor_url(string.Empty);
                    }
                }
                else
                {
                    comment.Author = CurrentUser.DisplayName;
                    comment.TitleUrl = SiteUrls.Instance().UserChannelHome(comment.UserID, UserDomainSpecialAppIDs.Instance().Home());
                }
                try
                {
                    #region 检查发帖时间间隔,防灌水
                    Globals.CheckPostDate();
                    #endregion

                    comment = PhotoComments.Create(comment);
                }
                catch (SPBException e)
                {
                    ExceptionMessage exceptionMessage = ResourceManager.GetExceptionMessage(e.ExceptionType, ApplicationIDs.Instance().Photo());
                    if (exceptionMessage != null)
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        TempData["StatusMessageContent"] = exceptionMessage.Title + "，" + exceptionMessage.Body;
                    }
                    return Control_PhotoComments(userDomainName, threadID, isMiniStyle, pageIndex);
                }

                if (comment.PostID > 0)
                {
                    if (comment.AuditingStatus == AuditingStatuses.Success)
                    {
                        if (comment.IsApproved)
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                            TempData["StatusMessageContent"] = GetResourceString("Message_ValidateSucecess", applicationID);
                        }
                        else
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                            TempData["StatusMessageContent"] = GetResourceString("Message_ValidateHostAuditingDoShow", applicationID);
                        }
                    }
                    else
                    {
                        switch (comment.AuditingStatus)
                        {
                            case AuditingStatuses.Again:
                            case AuditingStatuses.Pending:
                                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                                TempData["StatusMessageContent"] = GetResourceString("Message_ValidateManagerAuditingDoshow", applicationID);
                                break;
                            case AuditingStatuses.Fail:
                                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                                TempData["StatusMessageContent"] = GetResourceString("Message_ValidateManagerAuditingDoshow", applicationID);
                                break;
                            default:
                                break;
                        }
                    }
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_ValidateFailContactManager", applicationID);
                }
            }
            return Control_PhotoComments(userDomainName, threadID, isMiniStyle, pageIndex);
        }

        /// <summary>
        /// 删除选中的图片评论
        /// </summary>
        /// <param name="userDomainName">当前空间用户名</param>    
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeletePhotoComments(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PhotoSection currentPhotoSection = GetPhotoSection(userDomainName);
            int getApprovalStatus = int.Parse(Request.Form["approveDropDownList"]);
            ApprovalStatuses? approvalStatus = null;
            if (getApprovalStatus == 0)
                approvalStatus = ApprovalStatuses.NotApprove;
            else if (getApprovalStatus == 1)
                approvalStatus = ApprovalStatuses.IsApproved;

            string deleteCommentsString = Request.Form["PhotoCommentsCheckBoxGroup"];
            if (!string.IsNullOrEmpty(deleteCommentsString))
            {
                string[] deleteIDStrings = deleteCommentsString.Split(',');
                if (deleteIDStrings != null && deleteIDStrings.Length > 0)
                {
                    List<int> selectedIDArray = new List<int>();
                    foreach (string IDString in deleteIDStrings)
                    {
                        Int32 deleteID = int.Parse(IDString);
                        if (deleteID > 0)
                            selectedIDArray.Add(deleteID);
                    }
                    PhotoComments.Deletes(selectedIDArray);
                }
            }
            return Control_ManagePhotoComments(userDomainName, 0, approvalStatus);
        }

        /// <summary>
        /// 删除单条图片评论
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="photoPostID">评论所在的图片的PostID</param>
        /// <param name="commentPostID">要删除的评论的PostID</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult DeletePhotoComment(string userDomainName, int threadID, int postID, bool isMiniStyle, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PhotoSection currentPhotoSection = GetPhotoSection(userDomainName);
            PhotoComments.Delete(postID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSuccess");
            return Control_PhotoComments(userDomainName, threadID, isMiniStyle, pageIndex);
        }

        /// <summary>
        /// 设置图片评论是否通过验证
        /// </summary>
        /// <param name="userDomainName">当前空间用户名</param>
        /// <param name="isApproved">选中的评论设置为通过验证（true），选中的评论设置为未通过验证（false）</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetPhotoCommentsApproveStatus(string userDomainName, bool isApproved)
        {
            PhotoSection currentPhotoSection = GetPhotoSection(userDomainName);
            int getApprovalStatus = int.Parse(Request.Form["approveDropDownList"]);
            ApprovalStatuses? approvalStatus = null;
            if (getApprovalStatus == 0)
                approvalStatus = ApprovalStatuses.NotApprove;
            else if (getApprovalStatus == 1)
                approvalStatus = ApprovalStatuses.IsApproved;

            string approveCommentsString = Request.Form["PhotoCommentsCheckBoxGroup"];
            if (!string.IsNullOrEmpty(approveCommentsString))
            {
                string[] approveIDStrings = approveCommentsString.Split(',');
                if (approveIDStrings != null && approveIDStrings.Length > 0)
                {
                    List<int> selectedIDArray = new List<int>();
                    foreach (string IDString in approveIDStrings)
                    {
                        Int32 approveID = int.Parse(IDString);
                        if (approveID > 0)
                            selectedIDArray.Add(approveID);
                    }

                    PhotoComments.Approve(currentPhotoSection.OwnerUserID, selectedIDArray, isApproved);

                }
            }
            return Control_ManagePhotoComments(userDomainName, 0, approvalStatus);
        }

        /// <summary>
        /// 设置图片评论状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetPhotoCommentApproveStatus(string userDomainName, int threadID, int postID, bool isApproved, bool isMiniStyle, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PhotoComment photoComment = PhotoComments.Get(postID, false, true);
            if (photoComment == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), postID)));
            }
            //throw new ArgumentException(string.Format("图片评论：{0}不存在", postID));
            List<int> postIDs = new List<int>();
            postIDs.Add(postID);
            PhotoComments.Approve(currentDomainUser.UserID, postIDs, isApproved);
            return Control_PhotoComments(userDomainName, threadID, isMiniStyle, pageIndex);
        }

        /// <summary>
        /// 设置单条评论通过验证
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="photoPostID">评论所属图片的PostID</param>
        /// <param name="commentPostID">要删除的评论的PostID</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult ApproveThePhotoComment(string userDomainName, int threadID, int postID, bool isMiniStyle, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PhotoComment photoComment = PhotoComments.Get(postID, false, true);
            if (photoComment == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotHaveImage", applicationID), postID)));
            }
            //throw new ArgumentException(string.Format("图片评论：{0}不存在", postID));
            ValidateItemOwner(photoComment.OwnerUserID);

            PhotoSection currentPhotoSection = GetPhotoSection(userDomainName);
            List<int> postIDs = new List<int>();
            postIDs.Add(postID);
            PhotoComments.Approve(currentDomainUser.UserID, postIDs, true);
            return Control_PhotoComments(userDomainName, threadID, isMiniStyle, pageIndex);
        }

        /// <summary>
        /// 管理相册评论
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        /// <param name="pageIndex">当前页码</param>
        /// <param name="approvalStatus">是否批准</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult ManagePhotoComments(string userDomainName, int? pageIndex, ApprovalStatuses? approvalStatus)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManagerImageAudting", applicationID));
            #endregion

            ViewData["pageIndex"] = pageIndex;
            ViewData["approvalStatus"] = approvalStatus;
            return View("pages/Photos/ManagePhotoComments.aspx", "Masters/Photo.master");
        }

        /// <summary>
        /// 根据评论状态显示评论列表
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="approvalStatus">选择状态</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult ManagePhotoCommentsByApprovalStatus(string userDomainName, int? pageIndex)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManagerImageAudting", applicationID));
            #endregion

            int approvalStatus = int.Parse(Request.Form["approveDropDownList"]);
            ApprovalStatuses getApprovalStatus = ApprovalStatuses.IsApproved;
            if (approvalStatus == 0)
                getApprovalStatus = ApprovalStatuses.NotApprove;
            else if (approvalStatus == 1)
                getApprovalStatus = ApprovalStatuses.IsApproved;
            else
                return Control_ManagePhotoComments(userDomainName, pageIndex, null);

            return Control_ManagePhotoComments(userDomainName, pageIndex, getApprovalStatus);
        }

        /// <summary>
        /// 管理照片评论
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        /// <param name="pageIndex">当前显示页码</param>
        /// <param name="approvalStatus">显示状态评论</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_ManagePhotoComments(string userDomainName, int? pageIndex, ApprovalStatuses? approvalStatus)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            if (!pageIndex.HasValue)
                pageIndex = 1;

            PagingDataSet<PhotoComment> pds = PhotoComments.GetMyComments(currentDomainUser.UserID, approvalStatus, pageIndex.Value);

            ViewData["ApprovalStatus"] = approvalStatus;
            return PartialView("Controls/Photos/ManagePhotoComments.ascx", pds);
        }

        #endregion

        #endregion

        #region PhotoSection

        /// <summary>
        /// 相册设置管理
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ManagePhotoSectionSettings(string userDomainName)
        {
            PhotoSection currentPhotoSection = GetPhotoSection(userDomainName);

            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_MangerCallerySet", applicationID));
            #endregion

            return View("Pages/Photos/ManagePhotoSectionSettings.aspx", "Masters/Photo.master", currentPhotoSection);
        }

        /// <summary>
        /// 更新相册设置
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UpdateGallerySettings(string userDomainName)
        {
            PhotoSection currentGallery = GetPhotoSection(userDomainName);
            //currentGallery.SectionName = Request.Form.Get<string>("GalleryName", GetResourceString("Action_Callery", applicationID)).Trim();
            if (string.IsNullOrEmpty(currentGallery.SectionName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
            }
            else
            {
                //currentGallery.Description = Formatter.FormatMultiLinePlainTextForStorage(Request.Form["Description"], true);
                if (Request.Form["EnableComment"] == "true" || Request.Form["EnableComment"] == "True")
                    currentGallery.EnableComments = true;
                else
                    currentGallery.EnableComments = false;

                if (Request.Form["EnableAnonymousComment"] == "true" || Request.Form["EnableAnonymousComment"] == "True")
                    currentGallery.EnableAnonymousPosting = true;
                else
                    currentGallery.EnableAnonymousPosting = false;

                if (Request.Form["ModerateComments"] == "true" || Request.Form["ModerateComments"] == "True")
                    currentGallery.ModerateComments = true;
                else
                    currentGallery.ModerateComments = false;

                PhotoSections.Update(currentGallery);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;

            }

            return RedirectToAction("ManagePhotoSectionSettings", "Photo", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        #endregion

        #region PhotoUserCategory

        #region Show


        /// <summary>
        /// 用户相簿列表
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        /// <param name="currentCategoryID">当前相簿ID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        public ActionResult Control_ListPhotoCategories(string userDomainName, int? currentCategoryID, int? listPhotoCategoryCount)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            #region 计数
            AddWebCount<PhotoSectionCounter>(currentDomainUser.UserID);
            #endregion

            string title = string.Empty;
            List<UserCategory> categoriesArrayList = null;
            UserCategory currentPostCategory = null;
            bool isShowOtherGallery = currentCategoryID != null && currentCategoryID > 0;

            if (currentDomainUser != null)
            {
                if (IsUserDomainOwner(userDomainName))
                {
                    title = GetResourceString("Label_MyGallery", applicationID);
                    if (isShowOtherGallery)
                        title = GetResourceString("Label_MyOtherGallery", applicationID);
                }
                else if ((currentDomainUser as PersonUser) != null)
                {
                    switch ((currentDomainUser as PersonUser).Gender)
                    {
                        case Genders.Female:
                            title = GetResourceString("Label_HerGallery", applicationID);
                            if (isShowOtherGallery)
                                title = GetResourceString("Label_HerOtherGallery", applicationID);
                            break;
                        case Genders.Male:
                        default:
                            title = GetResourceString("Label_HisGallery", applicationID);
                            if (isShowOtherGallery)
                                title = GetResourceString("Label_HisOtherGallery", applicationID);
                            break;
                    }
                }
                ViewData["title"] = title;

                categoriesArrayList = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategories(currentDomainUser.UserID, HasManagePermission(userDomainName));

                if (categoriesArrayList != null)
                    categoriesArrayList = categoriesArrayList.Where(n => n.PrivacyStatus == PrivacyStatuses.Public).ToList();

                if (currentCategoryID != null && (int)currentCategoryID > 0)
                {
                    currentPostCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory((int)currentCategoryID, false);
                }
            }
            int listCount = listPhotoCategoryCount.HasValue ? (int)listPhotoCategoryCount : 3;
            ViewData["listPhotoCategoryCount"] = listCount;
            ViewData["userDomainName"] = userDomainName;
            ViewData["currentCategory"] = currentPostCategory;
            return PartialView("Controls/Photos/ListPhotoCategories.ascx", categoriesArrayList);
        }



        /// <summary>
        /// 相博下拉列表
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_PhotoUserCategoryDropdownList(string userDomainName, int? currentCategoryID, string controlName, string cssClass)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            List<UserCategory> userCategories = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategories(currentDomainUser.UserID, true);
            if (currentCategoryID != null && currentCategoryID > 0)
            {
                ViewData["currentCategoryID"] = (int)currentCategoryID;
            }
            else
                ViewData["currentCategoryID"] = 0;
            ViewData["controlName"] = controlName;
            ViewData["cssClass"] = cssClass;
            return PartialView("Controls/Photos/PhotoUserCategoryDropdownList.ascx", userCategories);
        }

        #endregion

        #region Manage


        /// <summary>
        /// 创建新相簿
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_NewPhotoUserCategory(string userDomainName)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_CreateCallery", applicationID));
            #endregion

            return View("Controls/Photos/NewPhotoUserCategory.ascx");
        }

        /// <summary>
        /// 编辑用户相簿
        /// </summary>
        /// <param name="userDomainName">空间用户名</param>
        /// <param name="categoryID">需要编辑的相簿ID</param>
        //TODO: 选择特征图片，需要异步刷新
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult EditPhotoCategory(string userDomainName, int categoryID)
        {
            PhotoSection currentPhotoSection = GetPhotoSection(userDomainName);

            UserCategory category = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(categoryID, false);
            if (currentPhotoSection == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_CalleryNotExist", applicationID), userDomainName)));
            }
            // throw new ArgumentException(string.Format("{0}相簿不存在", categoryID));
            ValidateItemOwner(category.UserID);

            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_EditCallery", applicationID));
            #endregion

            //选择特征图片，需要异步刷新
            PagingDataSet<PhotoThread> ts = PhotoThreads.GetMyThumbnailThreads(currentPhotoSection.OwnerUserID, string.Empty, categoryID, 1);

            ViewData["categoryPhotos"] = ts;

            return View("Pages/Photos/EditPhotoCategory.aspx", "Masters/Photo.master", category);
        }

        /// <summary>
        /// 添加新相簿
        /// </summary>
        /// <param name="userDomainName">用户名称</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreatePhotoUserCategory(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            PhotoSection currentPhotoSection = GetPhotoSection(userDomainName);

            UserCategory category = new UserCategory();
            category.UserID = currentDomainUser.UserID;
            category.CategoryName = HtmlUtils.TrimHtml(Request.Form.GetString("userCategoryName", string.Empty), 20).Trim();
            category.CategoryName = Formatter.FormatSingleLinePlainTextForStorage(category.CategoryName);
            if (string.IsNullOrEmpty(category.CategoryName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_PleaseWriteTypeName", applicationID);
            }
            else
            {
                category.DisplayOrder = Request.Form.GetInt("userCategoryOrder", 0);
                category.Description = Request.Form.Get<string>("userCategoryDescription", string.Empty);
                category.Description = HtmlUtils.TrimHtml(category.Description, 2000);
                category.Description = Formatter.FormatMultiLinePlainTextForStorage(category.Description, true);
                category.DateCreated = DateTime.Now;
                try
                {
                    UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).CreateUserCategory(category);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["currentCategoryIDInDropDownlist"] = category.CategoryID;
                }
                catch (SPBException ex)
                {
                    if (ex.ExceptionType == ExceptionTypes.Instance().PostDuplicate())
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        TempData["StatusMessageContent"] = GetResourceString("Exception_PleaseTryOtherName", applicationID);
                    }
                    else
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        TempData["StatusMessageContent"] = GetResourceString("Exception_PleaseAgain", applicationID);
                    }
                }
            }
            return RedirectToAction("Control_NewPhotoUserCategory", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 更新用户相簿
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        /// <param name="categoryID">相簿ID</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UpdatePhotoCategory(string userDomainName, int categoryID)
        {
            PhotoSection currentPhotoSection = GetPhotoSection(userDomainName);

            UserCategory category = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(categoryID, false);
            if (category == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Excetpton_CalleryNotExis", applicationID), categoryID)));
            }
            //throw new ArgumentException(string.Format("{0}相簿不存在", categoryID));
            ValidateItemOwner(category.UserID);

            category.CategoryName = Request.Form.Get<string>("CategoryName", string.Empty).Trim();
            category.CategoryName = Formatter.FormatSingleLinePlainTextForStorage(category.CategoryName);
            category.CategoryName = HtmlUtils.TrimHtml(category.CategoryName, 20);
            category.DisplayOrder = Request.Form.Get<int>("userCategoryOrder", 0);
            if (string.IsNullOrEmpty(category.CategoryName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
            }
            else
            {
                category.Description = Request.Form.Get<string>("Description", string.Empty);
                category.Description = HtmlUtils.TrimHtml(category.Description, 2000);
                category.Description = Formatter.FormatMultiLinePlainTextForStorage(category.Description, true);
                category.FeaturedItemID = Request.Form.Get<int>("photosDropdownList", 0);

                try
                {
                    UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).UpdateUserCategory(category);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                }
                catch (SPBException ex)
                {
                    if (ex.ExceptionType == ExceptionTypes.Instance().PostDuplicate())
                    {
                        throw new ArgumentException(GetResourceString(GetResourceString("Exception_PleaseTryOtherName", applicationID)));
                    }
                    else
                    {
                        throw new ArgumentException(GetResourceString("Exception_PleaseAgain", applicationID));
                    }
                }
            }
            return RedirectToAction("EditPhotoCategory", "Photo", new RouteValueDictionary { { "userDomainName", userDomainName }, { "categoryID", categoryID } });
        }

        /// <summary>
        /// 显示好友的图片
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListFriendsPhotos(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_FriendCallery", applicationID));
            #endregion

            return View("Pages/Photos/ListFriendsPhotos.aspx", "Masters/Photo.master");
        }

        #endregion

        #endregion

        #region PhotoUserTag

        /// <summary>
        /// 文件用户标签
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_ListPhotoUserTags(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().Photo()).GetUserTags(currentDomainUser.UserID, false);
            return PartialView("Controls/Photos/ListPhotoUserTags.ascx", userTags);
        }

        /// <summary>
        /// 管理用户博客标签
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult ManagePhotoUserTags(string userDomainName)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManagerCalleryTag", applicationID));
            #endregion

            ViewData["userDomainName"] = userDomainName;
            return View("Pages/Photos/ManagePhotoUserTags.aspx", "Masters/Photo.master");
        }

        #endregion

        #region TagUserInPhoto
        /// <summary>
        /// 照片圈人
        /// </summary>
        public ActionResult PhotoTags(string userDomainName)
        {
            int ID = Request.QueryString.GetInt("ID", -1);
            int threadID = Request.QueryString.GetInt("threadID", -1);
            if (threadID > 0)
            {
                //删除标记
                if (ID > 0)
                {
                    TagUserInPhotos.Delete(ID, threadID);
                }
                //新建标记
                else if (Request.QueryString.Count > 0)
                {
                    int AreaX = Request.QueryString.GetInt("AreaX", -1);
                    int AreaY = Request.QueryString.GetInt("AreaY", -1);
                    int AreaWidth = Request.QueryString.GetInt("AreaWidth", -1);
                    int AreaHeight = Request.QueryString.GetInt("AreaHeight", -1);
                    int UserID = Request.QueryString.GetInt("UserID", -1);
                    User user = Users.GetUser(UserID, false);
                    string DisplayName = string.Empty;
                    if (user != null)
                        DisplayName = user.DisplayName;
                    int OperatorUserID = CurrentUser.UserID;
                    if (OperatorUserID > 0 && UserID > 0)
                    {
                        TagUserInPhoto t = new TagUserInPhoto();
                        t.ThreadID = threadID;
                        t.AreaX = AreaX;
                        t.AreaY = AreaY;
                        t.AreaWidth = AreaWidth;
                        t.AreaHeight = AreaHeight;
                        t.UserID = UserID;
                        t.DisplayName = DisplayName;
                        t.OperatorUserID = OperatorUserID;
                        TagUserInPhotos.Create(t);
                    }
                }

                List<TagUserInPhoto> list = TagUserInPhotos.Get(threadID, true);
                return Content(ToHtml(list));
            }
            else
            {
                return Content(string.Empty);
            }
        }

        private string ToHtml(List<TagUserInPhoto> tuips)
        {
            if (tuips == null)
                return string.Empty;
            StringBuilder photoTags = new StringBuilder();
            foreach (TagUserInPhoto tuip in tuips)
            {
                string hn_tag = "{" + "\"x1\":" + tuip.AreaX.ToString() + ",\"y1\":" + tuip.AreaY.ToString() + ",\"height\":" + tuip.AreaHeight.ToString() + ",\"width\":" + tuip.AreaWidth.ToString() + ",\"note\":\"" + tuip.DisplayName + "\"}";
                photoTags.Append("<span class=\"usertag\" hn_tag='" + hn_tag + "'>\n");
                photoTags.Append(string.Format("<a href=\"{0}\">{1}</a>\n", SiteUrls.Instance().UserDomainHome(tuip.UserID), tuip.DisplayName));
                photoTags.Append("(");
                photoTags.Append(string.Format("<a href=\"{0}\" title=\"{1}\" >" + GetResourceString("Action_Callery", applicationID) + "</a>\n", SiteUrls.Instance().UserChannelHome(tuip.UserID, ApplicationIDs.Instance().Photo()), string.Format(GetResourceString("Dscription_SeeCaller", applicationID), tuip.DisplayName)));
                if (CurrentUser.UserID == tuip.OperatorUserID || CurrentUser != null && CurrentUser.IsContentAdministrator)
                {
                    photoTags.Append("<span class='pipe'>|</span>\n");
                    photoTags.Append("<a class=\"tag-del\" value=\"" + tuip.ID + "\"href=\"javascript:void(0)\">" + GetResourceString("Action_Delete") + "</a>\n");
                }
                photoTags.Append(")");
                photoTags.Append("</span>");
            }
            return photoTags.ToString();
        }

        #endregion

        #region 辅助方法

        private int setSpecialOrder = 1;

        /// <summary>
        /// 获取相册实体
        /// </summary>
        /// <param name="userDomainName">相册所属用户用户名</param>
        private PhotoSection GetPhotoSection(string userDomainName)
        {
            User currrentDomainUser = GetCurrentDomainUser(userDomainName);

            PhotoSection currentPhotoSection = PhotoSections.Get(currrentDomainUser.UserID, false);
            if (currentPhotoSection == null)
            {
                Response.Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_CalleryNotExist", applicationID), userDomainName)));
            }
            //throw new ArgumentException(string.Format("{0}用户相册不存在", userDomainName));
            return currentPhotoSection;
        }

        /// <summary>
        /// 设置页面标题
        /// </summary>
        protected override void SetPageTitle(string userDomainName, string text)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            base.SetPageTitle(userDomainName, text);
            if (string.IsNullOrEmpty(text))
                SpaceBuilder.Web.Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, GetResourceString("Title_Callery", applicationID), GetCurrentDomainUser(userDomainName).DisplayName, Globals.SiteName));
            else
                SpaceBuilder.Web.Header.AddTitle(string.Format("{1}{0}{2}{0}{3}{0}{4}", Globals.BrowserTitleSeparator, text, GetResourceString("Title_Callery", applicationID), GetCurrentDomainUser(userDomainName).DisplayName, Globals.SiteName));
        }

        /// <summary>
        /// 设置Meta标签：KeyWords, Description
        /// </summary>
        /// <param name="currentPhoto"></param>
        private void SetPageMeta(PhotoThread currentPhoto)
        {
            string metaKeyWords = string.Empty;
            string metaDescription = string.Empty;

            //标签
            if (currentPhoto.UserTags != null && currentPhoto.UserTags.Count > 0)
            {
                foreach (string userTag in currentPhoto.UserTags)
                {
                    if (userTag != null)
                    {
                        metaKeyWords = string.Format("{1},{0}", metaKeyWords, userTag);
                    }
                }
            }

            //标题和用户名
            metaKeyWords = metaKeyWords + currentPhoto.Subject;
            metaKeyWords = string.Format("{0},{1}", metaKeyWords, currentPhoto.Author);

            //站点分类和用户分类
            //if (file.SiteCategory != null)
            //    metaKeyWords = string.Format("{0},{1}",metaKeyWords,file.SiteCategory.CategoryName);
            //if (file.UserCategory != null)
            //    metaKeyWords = string.Format("{0},{1}",metaKeyWords,file.UserCategory.CategoryName);

            metaDescription = string.Format("{0}：{1}", currentPhoto.Subject, HtmlUtils.TrimHtml(currentPhoto.GetBodyForOwner(), 150));

            Header.AddMetaDescription(metaDescription);
            Header.AddMetaKeywords(metaKeyWords);
        }

        ///// <summary>
        ///// 格式化UUrl
        ///// </summary>
        //private string FormatUrl(string url)
        //{
        //    if (url.StartsWith("http", StringComparison.CurrentCultureIgnoreCase))
        //        return url;
        //    else
        //        return string.Concat(WebUtils.HostPath(Context.Request.Url), url);
        //}

        #endregion

        #region 更新相片审核状态

        public ActionResult UpdatePhotoAuditingStatus(int photoThreadID, string userDomainName)
        {
            List<int> threadIDs = new List<int>();
            threadIDs.Add(photoThreadID);
            PhotoThreads.UpdateAuditingStatus(threadIDs, AuditingStatuses.Success);

            return RedirectToAction("ShowPhoto", new RouteValueDictionary { { " userDomainName", userDomainName }, { "threadID", photoThreadID } });
        }

        #endregion
    }

    /// <summary>
    /// 空间相册二级菜单
    /// </summary>
    public enum PhotoUserDomainSubMenus
    {
        /// <summary>
        /// 我的相册
        /// </summary>
        myPhotoCategory = 1,

        /// <summary>
        /// 最新照片
        /// </summary>
        latestPhoto = 5,

        /// <summary>
        /// 朋友的相册
        /// </summary>
        friendPhotoCategory = 10,

        /// <summary>
        /// 站点中的相册
        /// </summary>
        sitePhotoCategory = 20,
        /// <summary>
        /// 我被圈的图片
        /// </summary>
        taggedPhotoCategory = 30
    }
}
