﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using System.Web;
using System.IO;

using SpaceBuilder.Common;
using SpaceBuilder.Photo;
using SpaceBuilder.Utils;

using SpaceBuilder.Web;
using SpaceBuilder.Common.Handlers;

namespace SpaceBuilder.Photo.Handlers
{
    public class ShowPhotoHandler : DownloadFileHandlerBase
    {
        public override void ProcessRequest(HttpContext context)
        {
            //防盗链检测
            if (SPBConfig.Instance().EnableAntiLeech && !Globals.IsAllowableReferrer(context.Request))
            {
                context.Response.Redirect(Globals.GetFullUrlInResourceSite("~/Themes/Shared/Styles/Images/anti-Leech.gif"), true);
                return;
            }

            int threadID = context.Request.QueryString.GetInt("threadID", -1);
            if (threadID < 0)
            {
                WebUtils.Return404(context);
                return;
            }

            PhotoThread photo = PhotoThreads.GetThread(threadID, null, false);
            if (photo == null)
            {
                WebUtils.Return404(context);
                return;
            }

            PhotoImageType imageType = PhotoImageType.Icon;
            imageType = (PhotoImageType)context.Request.QueryString.GetInt("imageType", (int)PhotoImageType.Icon);

            User currentUser = Globals.GetCurrentUser();

            if (!UserCanView(photo, imageType))
            {
                WebUtils.Return404(context);
                return;
            }

            IFile photoFile = PhotoAttachmentManager.Instance().GetResizedPhoto(photo.Attachment, imageType);
            if (photoFile == null)
            {
                WebUtils.Return404(context);
                return;
            }

            DateTime lastModified = photoFile.LastModified.ToUniversalTime();
            if (IsCacheOK(context, lastModified))
            {
                WebUtils.Return304(context);
                return;
            }
            else
            {
                SetResponsesDetails(context, photo.ContentType, photo.FileName, lastModified);
                FileSystemFile fileSystemFile = photoFile as FileSystemFile;
                if (fileSystemFile != null && (!fileSystemFile.FullLocalPath.StartsWith(@"\")))
                {
                    context.Response.TransmitFile(fileSystemFile.FullLocalPath);
                }
                else
                {
                    context.Response.AddHeader("Content-Length", photoFile.ContentLength.ToString("0"));
                    context.Response.Buffer = false;
                    context.Response.BufferOutput = false;
                    Stream stream = null;
                    try
                    {
                        stream = photoFile.OpenReadStream();
                        if (stream == null)
                        {
                            WebUtils.Return404(context);
                            return;
                        }

                        int bufferLength = photoFile.ContentLength <= DownloadFileHandlerBase.BufferLength ? photoFile.ContentLength : DownloadFileHandlerBase.BufferLength;
                        byte[] buffer = new byte[bufferLength];
                        int readedSize;
                        while ((readedSize = stream.Read(buffer, 0, bufferLength)) > 0)
                        {
                            if (!context.Response.IsClientConnected)
                                break;

                            context.Response.OutputStream.Write(buffer, 0, readedSize);
                            context.Response.OutputStream.Flush();
                        }
                        context.Response.OutputStream.Flush();
                        context.Response.Flush();
                    }
                    catch (Exception ex)
                    {
                        context.Response.Redirect(SiteUrls.Instance().Error(ex.ToString()), true);
                        return;
                    }
                    finally
                    {
                        if (stream != null)
                            stream.Close();
                    }
                }
            }

            context.Response.Cache.VaryByParams["threadID"] = true;
            context.Response.Cache.VaryByParams["imageType"] = true;
            context.Response.End();
        }

        /// <summary>
        /// 当前用户是否有浏览图片的权限
        /// </summary>
        /// <param name="photo"></param>
        /// <param name="imageType"></param>
        /// <param name="password"></param>
        /// <returns></returns>
        public static bool UserCanView(PhotoThread photo, PhotoImageType imageType)
        {
            if (photo.PrivacyStatus == PrivacyStatuses.Public)
                return true;

            if (photo.PrivacyStatus > PrivacyStatuses.Privacy && imageType < PhotoImageType.Slideshow)
                return true;

            User currentUser = Globals.GetCurrentUser();

            if (currentUser != null && (currentUser.UserID == photo.OwnerUserID || currentUser.IsContentAdministrator))
                return true;

            switch (photo.PrivacyStatus)
            {
                case PrivacyStatuses.NeedPassword:
                    UserCookie userCookie = new UserCookie(currentUser, HttpContext.Current);
                    return userCookie.AuthorizePhotoCatetoryID(photo.UserCategoryID);
                case PrivacyStatuses.OnlyFriend:
                    return currentUser != null && Friends.IsFriend(currentUser.UserID, photo.OwnerUserID);
                default:
                    return false;
            }
        }
    }
}