﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using SpaceBuilder.Common;

using SpaceBuilder.Web;
using SpaceBuilder.Utils;
using SpaceBuilder.Common.Mail;

namespace SpaceBuilder.Photo.Modules
{
    /// <summary>
    /// 处理与相册的动态、通知
    /// </summary>
    public class DisposeActivityForPhotoSection : IPhotoModule
    {

        #region IPostModule 成员
        public void Init(PhotoEventManager pem, System.Xml.Linq.XElement node)
        {
            pem.AfterPhotoCommentChange += new PhotoPostEventHandler(pem_AfterPhotoCommentChange);
            pem.AfterPhotoThreadChange += new PhotoThreadEventHandler(pem_AfterPhotoThreadChange);
            pem.AfterTagUserInPhotoChange += new TagUserInPhotoEventHandler(pem_AfterTagUserInPhotoChange);
        }

        void pem_AfterTagUserInPhotoChange(TagUserInPhoto tagUserInPhoto, GlobalEventArgs e)
        {
            if (e.State == ObjectState.Create)
            {
                User user = Users.GetUser(tagUserInPhoto.OperatorUserID, false);
                User toUser = Users.GetUser(tagUserInPhoto.UserID, false);
                if (user == null || toUser == null)
                    return;

                string itemKey = ActivityItemKeys.Instance().NewTagUserInPhoto();
                ActivityItem activityItem = Activities.GetActivityItem(itemKey);
                if (activityItem == null)
                    return;
                PhotoThread photo = PhotoThreads.GetThread(tagUserInPhoto.ThreadID, null, false);

                #region 处理个人动态

                UserActivity userActivity = new UserActivity();
                userActivity.ApplicationID = activityItem.ApplicationID;
                userActivity.ActivityItemID = activityItem.ItemID;
                userActivity.LeadingActorUserID = user.UserID;
                userActivity.LeadingActor = user.DisplayName;
                userActivity.Subject = string.Format(ResourceManager.GetString("Title_Pattern_Activity_NewTagUserInPhoto", user.Language, e.ApplicationID), SPBHtmlHelper.Instance().UserDisplayNameLink(user,true), GetPhotoNameWithHyperlink(photo), SPBHtmlHelper.Instance().UserDisplayNameLink(toUser,true));
                userActivity.Body = GetImageHtml(photo);
                userActivity.IsPublic = photo.PrivacyStatus == PrivacyStatuses.Privacy ? false : true;

                userActivity.RelativeItemID = photo.ThreadID;
                userActivity.RelativeItemName = photo.Subject;

                Activities.CreateUserActivity(userActivity);

                #endregion

                #region 处理朋友动态
                if (photo.PrivacyStatus == PrivacyStatuses.Public || photo.PrivacyStatus == PrivacyStatuses.OnlyFriend)
                {
                    UserActivitySettings userActivitySettings = Activities.GetUserActivitySettings(user.UserID, false);
                    //判断用户是否允许发送给 朋友动态项目
                    if (userActivitySettings.IsAllowedInActivityItemsOfSending(itemKey))
                    {
                        FriendActivity friendActivity = new FriendActivity();
                        friendActivity.ApplicationID = activityItem.ApplicationID;
                        friendActivity.ActivityItemID = activityItem.ItemID;
                        friendActivity.LeadingActorUserID = user.UserID;
                        friendActivity.LeadingActor = user.DisplayName;

                        friendActivity.Subject = userActivity.Subject;
                        friendActivity.Body = GetImageHtml(photo);

                        friendActivity.RelativeItemID = photo.ThreadID;
                        friendActivity.RelativeItemName = photo.Subject;

                        List<int> receiverUserIDs = Activities.GetFriendActivityReceiverUserIDsFromFriends(user, itemKey);
                        receiverUserIDs.Remove(toUser.UserID);
                        Activities.CreateFriendActivity(friendActivity, receiverUserIDs);
                    }
                }
                #endregion
            }
        }

        void pem_AfterPhotoThreadChange(PhotoThread photo, GlobalEventArgs e)
        {
            if (e.State == ObjectState.Create)
            {
                User user = Users.GetUser(photo.OwnerUserID, false);
                if (user == null)
                    return;

                string itemKey = ActivityItemKeys.Instance().NewPhoto();
                ActivityItem activityItem = Activities.GetActivityItem(itemKey);

                if (activityItem == null)
                    return;

                UserCategory uc = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().Photo()).GetUserCategory(photo.UserCategoryID, true);

                #region 处理个人动态

                UserActivity userActivity = new UserActivity();
                userActivity.ApplicationID = activityItem.ApplicationID;
                userActivity.ActivityItemID = activityItem.ItemID;
                userActivity.LeadingActorUserID = user.UserID;
                userActivity.LeadingActor = user.DisplayName;
                userActivity.Subject = string.Format(ResourceManager.GetString("Title_Pattern_Activity_NewPhoto", user.Language, e.ApplicationID), SPBHtmlHelper.Instance().UserDisplayNameLink(user,true), GetCategoryNameWithHyperlink(photo), GetPhotoNameWithHyperlink(photo));
                userActivity.Body = GetImageHtml(photo);
                userActivity.IsPublic = photo.PrivacyStatus == PrivacyStatuses.Privacy ? false : true;
                userActivity.RelativeItemID = photo.ThreadID;
                userActivity.RelativeItemName = photo.Subject;
                Activities.CreateUserActivity(userActivity);
                #endregion

                #region 处理朋友动态
                if ((photo.PrivacyStatus == PrivacyStatuses.Public || photo.PrivacyStatus == PrivacyStatuses.OnlyFriend) && (uc.PrivacyStatus == PrivacyStatuses.Public || uc.PrivacyStatus == PrivacyStatuses.OnlyFriend))
                {
                    UserActivitySettings userActivitySettings = Activities.GetUserActivitySettings(user.UserID, false);
                    //判断用户是否允许发送给 朋友动态项目
                    if (userActivitySettings.IsAllowedInActivityItemsOfSending(itemKey))
                    {
                        FriendActivity friendActivity = new FriendActivity();
                        friendActivity.ApplicationID = activityItem.ApplicationID;
                        friendActivity.ActivityItemID = activityItem.ItemID;
                        friendActivity.LeadingActorUserID = user.UserID;
                        friendActivity.LeadingActor = user.DisplayName;

                        friendActivity.Subject = userActivity.Subject;
                        friendActivity.Body = GetImageHtml(photo);

                        friendActivity.RelativeItemID = photo.ThreadID;
                        friendActivity.RelativeItemName = photo.Subject;

                        List<int> receiverUserIDs = Activities.GetFriendActivityReceiverUserIDsFromFriends(user, itemKey);
                        Activities.CreateFriendActivity(friendActivity, receiverUserIDs);
                    }
                }
                #endregion
            }
            else if (e.State == ObjectState.Delete)
            {
                Activities.DeleteUserAndFriendActivities(ApplicationIDs.Instance().Photo(), photo.ThreadID);
            }
        }

        void pem_AfterPhotoCommentChange(PhotoComment photoComment, SpaceBuilder.Common.GlobalEventArgs e)
        {
            if (e.State == ObjectState.Create)
            {
                User user = Users.GetUser(photoComment.UserID, false);
                if (user == null)
                    return;

                bool activityIsPublic = false;
                PhotoThread photo = PhotoThreads.GetThread(photoComment.ThreadID, null, false);

                User owner = Users.GetUser(photo.OwnerUserID, false);
                if (owner == null)
                    return;

                string itemKey = ActivityItemKeys.Instance().NewPhotoComment();
                ActivityItem activityItem = Activities.GetActivityItem(itemKey);

                if (activityItem == null)
                    return;

                if ((photo.PrivacyStatus == PrivacyStatuses.OnlyFriend || photo.PrivacyStatus == PrivacyStatuses.Public) && photoComment.IsApproved)
                    activityIsPublic = true;

                #region 处理个人动态
                UserActivity userActivity = new UserActivity();
                userActivity.ApplicationID = activityItem.ApplicationID;
                userActivity.ActivityItemID = activityItem.ItemID;
                userActivity.LeadingActorUserID = user.UserID;
                userActivity.LeadingActor = user.DisplayName;
                userActivity.Subject = string.Format(ResourceManager.GetString("Title_Pattern_Activity_NewPhotoComment", user.Language, e.ApplicationID), SPBHtmlHelper.Instance().UserDisplayNameLink(user,true), SPBHtmlHelper.Instance().UserDisplayNameLink(owner,true), GetPhotoNameWithHyperlink(photo));
                userActivity.Body = HtmlUtils.TrimHtml(photoComment.Body, 250);
                userActivity.IsPublic = activityIsPublic;
                userActivity.RelativeItemID = photoComment.PostID;
                Activities.CreateUserActivity(userActivity);
                #endregion

                #region 处理朋友动态
                if (activityIsPublic)
                {
                    UserActivitySettings userActivitySettings = Activities.GetUserActivitySettings(user.UserID, false);
                    //判断用户是否允许发送给 朋友动态项目
                    if (userActivitySettings.IsAllowedInActivityItemsOfSending(itemKey))
                    {
                        FriendActivity friendActivity = new FriendActivity();
                        friendActivity.ApplicationID = activityItem.ApplicationID;
                        friendActivity.ActivityItemID = activityItem.ItemID;
                        friendActivity.LeadingActorUserID = user.UserID;
                        friendActivity.LeadingActor = user.DisplayName;

                        friendActivity.Subject = userActivity.Subject;
                        friendActivity.Body = HtmlUtils.TrimHtml(photoComment.Body, 250);

                        friendActivity.RelativeItemID = photoComment.PostID;

                        List<int> receiverUserIDs = Activities.GetFriendActivityReceiverUserIDsFromFriends(user, itemKey);
                        Activities.CreateFriendActivity(friendActivity, receiverUserIDs);
                    }
                }
                #endregion

            }
        }

        #endregion

        /// <summary>
        /// 获取带链接的图片名称的html代码
        /// </summary>
        private static string GetPhotoNameWithHyperlink(PhotoThread photo)
        {
            return string.Format("<a {0} href=\"{1}\" title=\"{2}\">{3}</a>", HyperlinkTargetOfUserActivity, SiteUrls.Instance().ShowPicture(UserID2UserNameDictionary.GetUserName(photo.OwnerUserID), photo.ThreadID), photo.Subject, StringUtils.Trim(photo.Subject, 9, string.Empty));
        }

        /// <summary>
        /// 获取带链接的相簿名称的html代码
        /// </summary>
        private static string GetCategoryNameWithHyperlink(PhotoThread photo)
        {
            return string.Format("<a {0} href=\"{1}\">{2}</a>", HyperlinkTargetOfUserActivity, SiteUrls.Instance().ShowPhotoCategory(UserID2UserNameDictionary.GetUserName(photo.OwnerUserID), photo.UserCategoryID), photo.UserCategory != null ? photo.UserCategory.CategoryName : string.Empty);
        }

        /// <summary>
        /// 图片的html呈现代码
        /// </summary>
        public static string GetImageHtml(PhotoThread photo)
        {
            return string.Format("<a {2} href=\"{3}\"><img src=\"{0}\" border=\"0\" alt=\"{1}\" /></a>", SiteUrls.Instance().PictureUrl(photo.ThreadID, PhotoImageType.Thumbnail), photo.Subject, HyperlinkTargetOfUserActivity, SiteUrls.Instance().ShowPicture(UserID2UserNameDictionary.GetUserName(photo.OwnerUserID), photo.ThreadID));
        }

        private static readonly string HyperlinkTargetOfUserActivity = SiteSettingsManager.GetSiteSettings().OpenNewWindowInUserActivity ? "target=\"_blank\"" : string.Empty;

    }

}
