﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Lucene.Net.Search;
using Lucene.Net.Index;
using SpaceBuilder.Photo;
using SpaceBuilder.Common;

using Lucene.Net.QueryParsers;
using Lucene.Net.Documents;
using Lucene.Net.Analysis;
using SpaceBuilder.Utils;
using Lucene.Net.Analysis.Standard;

namespace SpaceBuilder.LuceneSearch
{
    /// <summary>
    /// 图片全文检索
    /// </summary>
    public class PhotoSearchManager : SearchManagerBase<PhotoThread>
    {

        private static readonly string forumIndexFileDirectory = "Photo";
        private static volatile PhotoSearchManager _self = null;
        private static readonly object lockObject = new object();
        private PhotoSearchManager(string indexFileDirectory) : base(indexFileDirectory) { }

        public static PhotoSearchManager Instance()
        {
            if (_self == null)
            {
                lock (lockObject)
                {
                    if (_self == null)
                    {
                        _self = new PhotoSearchManager(forumIndexFileDirectory);
                    }
                }
            }

            return _self;
        }

        /// <summary>
        /// 搜索
        /// </summary>
        /// <param name="query"></param>
        /// <returns></returns>
        public SearchResultDataSet<PhotoThread> Search(PhotoFullTextQuery query)
        {
            //索引文件不存在时，返回null
            if (!IsIndexFilesExists)
                return new SearchResultDataSet<PhotoThread>();

            BooleanQuery currentQuery = new BooleanQuery();
            BooleanQuery queryForFilter = new BooleanQuery();

            if (query.SearchScopeUserID > 0)
            {
                Term userIDTerm = new Term(PhotoIndexFields.UserID, query.SearchScopeUserID.ToString());
                Query userIDQuery = new TermQuery(userIDTerm);
                queryForFilter.Add(userIDQuery, BooleanClause.Occur.MUST);
            }

            if (query.SiteCategoryID > 0)
            {
                if (query.IncludeSiteCategoryDescendant)
                {
                    QueryParser siteCategoryIDQueryParser = new QueryParser(CurrentLuceneVersion, PhotoIndexFields.SiteCategoryID, new WhitespaceAnalyzer());

                    List<SiteCategory> childSiteCategories = SiteCategories.Instance(ApplicationIDs.Instance().Photo()).GetAllChilds(query.SiteCategoryID, false);
                    StringBuilder siteCategoryIDs = new StringBuilder();
                    siteCategoryIDs.Append(query.SiteCategoryID.ToString());
                    if (childSiteCategories != null)
                    {
                        foreach (SiteCategory siteCategory in childSiteCategories)
                        {
                            siteCategoryIDs.Append(" " + siteCategory.CategoryID);
                        }
                    }

                    Query siteCategoryIDQuery = siteCategoryIDQueryParser.Parse(siteCategoryIDs.ToString());
                    currentQuery.Add(siteCategoryIDQuery, BooleanClause.Occur.MUST);
                }
                else
                {
                    Term siteCategoryIDTerm = new Term(PhotoIndexFields.SiteCategoryID, query.SiteCategoryID.ToString());
                    Query siteCategoryIDQuery = new TermQuery(siteCategoryIDTerm);
                    currentQuery.Add(siteCategoryIDQuery, BooleanClause.Occur.MUST);
                }
            }

            if (query.IsPublicFilter)
            {
                Term isPublicTerm = new Term(PhotoIndexFields.PrivacyStatus, ((int)PrivacyStatuses.Public).ToString());
                Query isPublicQuery = new TermQuery(isPublicTerm);
                queryForFilter.Add(isPublicQuery, BooleanClause.Occur.MUST);
            }

            if (!string.IsNullOrEmpty(query.Keyword))
            {
                Query postKeywordQuery = null;
                query.Keyword = StringUtilsForLucene.LuceneKeywordsScrubber(query.Keyword.ToLower());
                string keywordSegments = SegmentForQueryParser(query.Keyword);
                if (!string.IsNullOrEmpty(query.Keyword))
                {
                    string[] searchFieldsForKeyword = new string[4];
                    searchFieldsForKeyword[0] = PhotoIndexFields.Subject;
                    searchFieldsForKeyword[1] = PhotoIndexFields.Body;
                    searchFieldsForKeyword[2] = PhotoIndexFields.Tags;
                    searchFieldsForKeyword[3] = PhotoIndexFields.Author;
                    MultiFieldQueryParser keywordParser = new MultiFieldQueryParser(CurrentLuceneVersion, searchFieldsForKeyword, GetChineseAnalyzerOfUnTokenized());
                    keywordParser.SetLowercaseExpandedTerms(true);
                    keywordParser.SetDefaultOperator(QueryParser.OR_OPERATOR);
                    postKeywordQuery = keywordParser.Parse(keywordSegments);
                    currentQuery.Add(postKeywordQuery, BooleanClause.Occur.MUST);
                }
            }

            if (!string.IsNullOrEmpty(query.TagName))
            {
                string[] tagSegments = SegmentForPhraseQuery(query.TagName);
                if (tagSegments != null && tagSegments.Length > 0)
                {
                    PhraseQuery tagQuery = new PhraseQuery();
                    foreach (var tagSegment in tagSegments)
                        tagQuery.Add(new Term(PhotoIndexFields.Tags, tagSegment));

                    tagQuery.SetSlop(PhraseQuerySlop);
                    tagQuery.SetBoost((float)Math.Pow(3, 5));
                    currentQuery.Add(tagQuery, BooleanClause.Occur.MUST);
                }
            }

            if (!string.IsNullOrEmpty(query.Author))
            {
                string[] authorSegments = SegmentForPhraseQuery(query.Author);
                if (authorSegments != null && authorSegments.Length > 0)
                {
                    PhraseQuery authorQuery = new PhraseQuery();
                    foreach (var authorSegment in authorSegments)
                        authorQuery.Add(new Term(PhotoIndexFields.Author, authorSegment));

                    authorQuery.SetSlop(PhraseQuerySlop);
                    authorQuery.SetBoost((float)Math.Pow(3, 5));
                    currentQuery.Add(authorQuery, BooleanClause.Occur.MUST);
                }
            }

            SortField[] sortFields;
            switch (query.SortBy)
            {
                case FullTextQueryPhotosSortBy.DateCreated:
                    sortFields = new SortField[] { new SortField(PhotoIndexFields.ThreadID, SortField.INT, true) };
                    break;
                default:
                    sortFields = new SortField[] { SortField.FIELD_SCORE };
                    break;
            }
            Filter filter = null;
            if (queryForFilter.Clauses().Count > 0)
                filter = new QueryWrapperFilter(queryForFilter);

            SearchResultDataSet<PhotoThread> pds = Search(currentQuery, filter, sortFields, query.PageIndex, query.PageSize);            
            return pds;
        }

        /// <summary>
        /// 初始化索引
        /// </summary>
        public override void InitializeIndex(string indexPath)
        {
            if (!System.IO.Directory.Exists(indexPath))
            {
                try
                {
                    System.IO.Directory.CreateDirectory(indexPath);
                }
                catch
                {
                    throw new ApplicationException(string.Format("create Directory '{0}' failed", PhysicalIndexDirectory));
                }
            }

            #region 索引图片

            int indexPageSize = 2000;
            bool createIndexFile = true;
            PagingDataSet<PhotoThread> pds = PhotoThreads.GetThreadsForAdmin(indexPageSize, 1, Applications.GetApplication(ApplicationIDs.Instance().Photo()).AuditingStatusForPublicDisplay, -1, string.Empty);
            double tIndex = Convert.ToDouble(pds.TotalRecords) / Convert.ToDouble(indexPageSize);
            int indexPageIndex = (int)Math.Ceiling(tIndex);

            if (pds.TotalRecords > 0)
            {
                //分多次进行索引
                for (int i = 1; i <= indexPageIndex; i++)
                {
                    if (i != 1)
                        pds = PhotoThreads.GetThreadsForAdmin(indexPageSize, i, Applications.GetApplication(ApplicationIDs.Instance().Photo()).AuditingStatusForPublicDisplay, -1, string.Empty);
                    Insert(pds.Records, indexPath, createIndexFile);
                    if (createIndexFile)
                        createIndexFile = false;
                }
            }

            #endregion
        }

        /// <summary>
        /// Document转化成PhotoThread
        /// </summary>
        protected override PhotoThread ConvertDocumentToObj(Document doc)
        {
            PhotoThread photo = new PhotoThread();
            int userID;
            int.TryParse(doc.Get(PhotoIndexFields.UserID), out userID);
            photo.OwnerUserID = userID;

            photo.Author = doc.Get(PhotoIndexFields.Author);

            int threadID;
            int.TryParse(doc.Get(PhotoIndexFields.ThreadID), out threadID);
            photo.ThreadID = threadID;

            photo.Subject = doc.Get(PhotoIndexFields.Subject);
            //photo.Body = doc.Get(PhotoIndexFields.Body);

            int userCategoryID;
            int.TryParse(doc.Get(PhotoIndexFields.UserCategoryID), out userCategoryID);
            photo.UserCategoryID = userCategoryID;

            int siteCategoryID;
            int.TryParse(doc.Get(PhotoIndexFields.SiteCategoryID), out siteCategoryID);
            photo.SiteCategoryID = siteCategoryID;


            int auditingStatusValue = (int)AuditingStatuses.Success;
            int.TryParse(doc.Get(PhotoIndexFields.AuditingStatus), out auditingStatusValue);
            photo.AuditingStatus = (AuditingStatuses)auditingStatusValue;

            try
            {
                photo.PostDate = DateTools.StringToDate(doc.Get(PhotoIndexFields.PostDate));
            }
            catch { }

            try
            {
                photo.LastRepliedDate = DateTools.StringToDate(doc.Get(PhotoIndexFields.LastUpdatedDate));
            }
            catch { }

            return photo;
        }

        /// <summary>
        /// PhotoThread 转化成Document进行索引的存储
        /// </summary>
        protected override Document ConvertObjToDocument(PhotoThread thread)
        {
            if (thread == null)
                return null;

            //待审核及未通过审核不允许加入索引
            if (thread.AuditingStatus == AuditingStatuses.Pending || thread.AuditingStatus == AuditingStatuses.Fail)
                return null; 

            Document doc = new Document();
            Field field;
            field = new Field(PhotoIndexFields.UserID, thread.OwnerUserID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(PhotoIndexFields.ThreadID, thread.ThreadID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(PhotoIndexFields.Author, thread.Author.ToLower(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(PhotoIndexFields.Subject, thread.Subject.ToLower(), Field.Store.YES, Field.Index.ANALYZED, Field.TermVector.YES);
            field.SetBoost(2.0F);
            doc.Add(field);

            field = new Field(PhotoIndexFields.Body, HtmlUtils.StripAllTags(thread.GetBody()).ToLower(), Field.Store.YES, Field.Index.ANALYZED, Field.TermVector.YES);
            doc.Add(field);

            field = new Field(PhotoIndexFields.UserCategoryID, thread.UserCategoryID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(PhotoIndexFields.SiteCategoryID, thread.SiteCategoryID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            if (thread.UserTags != null)
            {
                foreach (var tag in thread.UserTags)
                {
                    field = new Field(PhotoIndexFields.Tags, tag.ToLower(), Field.Store.YES, Field.Index.ANALYZED);
                    field.SetBoost(2.0F);
                    doc.Add(field);
                }
            }

            field = new Field(PhotoIndexFields.PrivacyStatus, ((int)thread.PrivacyStatus).ToString().ToLower(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(PhotoIndexFields.AuditingStatus, ((int)thread.AuditingStatus).ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(PhotoIndexFields.PostDate, DateTools.DateToString(thread.PostDate, DateTools.Resolution.DAY), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            return doc;
        }
    }
}
