﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using SpaceBuilder.Common.Mail;
using SpaceBuilder.Common;
using SpaceBuilder.Web;
using System.Web.Routing;
using SpaceBuilder.Utils;
using SpaceBuilder.LuceneSearch;

namespace SpaceBuilder.Share.Controllers
{
    [HttpCompress]
    public class ChannelShareController : ChannelControllerBase
    {

        private int applicationID = ApplicationIDs.Instance().Share();

        #region Share

        #region View

        /// <summary>
        /// 分享首页
        /// </summary>
        public ActionResult Home()
        {
            SetPageTitle(GetResourceString("Title_ShareHome", applicationID));

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_Share", ApplicationIDs.Instance().Share()), string.Empty);

            List<SiteCategory> siteCategory = SiteCategories.Instance(ApplicationIDs.Instance().Share()).GetRoots(false);

            ViewData["siteCategories"] = siteCategory;

            return View("Pages/Shares/Home.aspx", "Masters/Share.Master");
        }

        /// <summary>
        /// 分享排行
        /// </summary>
        /// <returns></returns>
        public ActionResult ShareRanks()
        {
            SetPageTitle(GetResourceString("Title_ShareCompositor", applicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_Share", ApplicationIDs.Instance().Share()), string.Empty);
            return View("Pages/Shares/ShareRanks.aspx", "Masters/Share.Master");
        }

        /// <summary>
        /// 显示用户分享页
        /// </summary>
        public ActionResult ShowShareThread(int threadID)
        {
            ShareThread shareThread = ShareThreads.GetThread(threadID, false);
            if (shareThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ShareNotFound", applicationID), threadID)));
            }

            SetPageTitle(shareThread.Subject);

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ShareHome", applicationID), SiteUrls.Instance().ChannelShareHome());
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ShareCompositor", applicationID), SPBUrlHelper.Action("ShareRanks", "ChannelShare"));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ShowShare", applicationID), string.Empty);


            if (CurrentUser != null && (CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
                ViewData["hasManagePermission"] = true;

            #region 处理审核
            int auditingStatusValue = (int)Applications.GetApplication(applicationID).AuditingStatusForPublicDisplay;
            if ((int)shareThread.AuditingStatus < auditingStatusValue)
            {
                if (shareThread.AuditingStatus == AuditingStatuses.Fail)
                    ViewData["errorMessage"] = GetResourceString("Message_ShareWithoutAuditing", applicationID);
                else
                    ViewData["errorMessage"] = GetResourceString("Message_ShareAuditing", applicationID);
                ViewData["errorAuditingStatus"] = true;
            }
            #endregion

            #region 处理隐私
            ViewData["errorPrivacyStatus"] = true;
            if (shareThread.PrivacyStatus != PrivacyStatuses.Public)
            {
                ViewData["errorPrivacyStatus"] = false;
            }
            #endregion

            #region 计数 标题
            CounterFactory.IncreaseCount<ShareThreadCounter>(IncreasableCountTypes.WebView, threadID);
            SetPageTitle(shareThread.Subject);
            #endregion

            #region 设置Meta标签
            string metaDescription = string.Empty;
            metaDescription = string.Format("{0}：{1}", shareThread.Subject, StringUtils.Trim(shareThread.Body, 150));
            Header.AddMetaKeywords(shareThread.Subject);
            Header.AddMetaDescription(metaDescription);
            #endregion

            if (shareThread.MediaType == MediaTypes.Website)
                return View("Pages/Shares/ShowShareLink.aspx", shareThread);
            else
                return View("Pages/Shares/ShowShareThread.aspx", "Masters/Share.master", shareThread);
        }

        /// <summary>
        /// 根据标签显示站点分享
        /// </summary>
        public ActionResult ListShareThreadsBySiteTag(string tagName)
        {
            BreadCrumb.AddBreadCrumbItem(HtmlUtils.StripScriptTags(WebUtils.UrlDecode(tagName)), string.Empty);
            SetPageTitle(GetResourceString("Common_UserTag") + "：" + HtmlUtils.StripScriptTags(WebUtils.UrlDecode(tagName)), GetResourceString("Navigation_ClubDomain_Share", applicationID));
            return View("Pages/Shares/ListShareThreadsBySiteTag.aspx", "Masters/Share.master");
        }

        /// <summary>
        ///类别下的分享
        /// </summary>
        /// <param name="categoryID"></param>
        /// <returns></returns>
        public ActionResult ListShareThreadsByCategories(int? categoryID)
        {
            SetPageTitle(GetResourceString("Title_ShareCompositor", applicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_Share", ApplicationIDs.Instance().Share()), string.Empty);


            Stack<SiteCategory> categoryStack = null;
            List<SiteCategory> siteCategoties = GetChildCategories(categoryID ?? 0, out categoryStack);

            ViewData["categoryStack"] = categoryStack;
            ViewData["siteCategoties"] = siteCategoties;
            ViewData["CurrentCategory"] = SiteCategories.Instance(ApplicationIDs.Instance().Share()).Get(categoryID ?? 0, false);

            return View("Pages/Shares/ListShareByCategories.aspx", "Masters/Share.Master", siteCategoties);
        }

        /// <summary>
        /// 标签地图
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShareTagMap()
        {
            SetPageTitle(GetResourceString("Title_TagMap", applicationID), GetResourceString("Navigation_ClubDomain_Share", applicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_HotTag", applicationID), string.Empty);

            Dictionary<string, KeyValuePair<int, SiteTag>> siteTags = UserTagManager.Instance(UserTagItemKeys.Instance().UserShareThread()).GetTopSiteTagsSortBy(SiteTagsSortBy.ItemCount, 200);
            return View("Pages/Shares/ShareTagMap.aspx", "Masters/Share.master", siteTags);
        }

        /// <summary>
        /// 显示分享对话框
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShareDialog()
        {
            SetPageTitle(GetResourceString("Title_ShareTo", applicationID) + Globals.SiteName);
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));
            ShareThread shareThread = null;
            int threadID = Request.QueryString.GetInt("threadID", 0);
            if (threadID > 0)
            {
                shareThread = ShareThreads.GetThread(threadID, false);
                if (shareThread == null)
                    ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, string.Format(GetResourceString("Message_Pattern_SiteShareNotFound", applicationID), threadID));
                else if (UserShareThreads.IsShared(CurrentUser.UserID, threadID))
                    ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Message_ShareNotRepeat", applicationID));
            }
            else
            {
                string url = Request.QueryString.GetString("url", string.Empty);
                if (string.IsNullOrEmpty(url))
                    ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Validate_LinkNotEmpty", applicationID));
                if (ShareThreads.UrlIsExist(url, out threadID))
                {
                    shareThread = ShareThreads.GetThread(threadID, false);
                    if (shareThread == null)
                        ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, string.Format(GetResourceString("Message_Pattern_SiteShareNotFound", applicationID), threadID));
                    else if (UserShareThreads.IsShared(CurrentUser.UserID, threadID))
                        ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Message_ShareNotRepeat", applicationID));
                }
                else
                {
                    int sharedApplicationID = Request.QueryString.GetInt("applicationID", 0);
                    if (sharedApplicationID <= 0)
                    {
                        UrlInfo info = UrlParserFactory.Instance().ParseUrl(url);
                        if (info == null)
                        {
                            shareThread = new ShareThread();
                            return View("Pages/Shares/ShareDialog.aspx", shareThread);
                        }
                        shareThread = new ShareThread(info);
                    }
                    else
                        shareThread = new ShareThread();

                    shareThread.UserID = CurrentUser.UserID;
                    shareThread.Author = CurrentUser.DisplayName;
                    shareThread.ApplicationID = sharedApplicationID;
                    shareThread.Subject = Request.QueryString.GetString("subject", shareThread.Subject);
                    shareThread.ThumbnailUrl = Request.QueryString.GetString("thumbnailUrl", shareThread.ThumbnailUrl);
                    shareThread.Body = Request.QueryString.GetString("body", shareThread.Body);
                    shareThread.ContentUserID = Request.QueryString.GetInt("contentUserID", 0);
                    shareThread.ContentAuthor = Request.QueryString.GetString("contentAuthor", string.Empty);
                }
            }

            return View("Pages/Shares/ShareDialog.aspx", shareThread);
        }

        /// <summary>
        /// 搜索分享
        /// </summary>
        /// <param name="keyWord"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Search(string keyword)
        {
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ShareSearch", applicationID), string.Empty);
            ShareSearchQuery query = new ShareSearchQuery();
            query.PageSize = 20;
            query.PageIndex = Request.QueryString.GetInt("pageIndex", 1);
            query.KeyWord = keyword;
            SearchResultDataSet<ShareThread> pds = ShareSearchManager.Instance().Search(query);
            ViewData["ShareSearchQuery"] = query;
            return View("Pages/Shares/SearchShareThreads.aspx", "Masters/Share.Master", pds);
        }

        /// <summary>
        /// 搜索
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Search()
        {
            return RedirectToAction("Search", new RouteValueDictionary { { "keyword", Request.Form["keyword"] } });
        }
        #endregion

        #region Control

        /// <summary>
        /// 频道分享列表
        /// </summary>
        public ActionResult Control_ListShareThreads(int? siteCategoryID, MediaTypes? mediaType, int? applicationID, string tagName, ShareThreadSortBy? sortBy, int? pageIndex)
        {
            PagingDataSet<ShareThread> pds = null;
            if (string.IsNullOrEmpty(tagName))
            {
                ViewData["siteCategoryID"] = siteCategoryID;
                ViewData["mediaType"] = mediaType.HasValue ? mediaType : MediaTypes.Video;
                ViewData["shareApplicationID"] = applicationID;
                pds = ShareThreads.GetThreadsSortBy(sortBy ?? ShareThreadSortBy.DateCreated, siteCategoryID, mediaType, applicationID, pageIndex ?? 1);
            }
            else
            {
                ViewData["tagName"] = tagName;
                pds = ShareThreads.GetPagingThreadsInSiteTag(tagName, pageIndex ?? 1);
            }
            ViewData["sortBy"] = sortBy ?? ShareThreadSortBy.DateCreated;
            return PartialView("Controls/Shares/ListShareThreads.ascx", pds);
        }

        /// <summary>
        /// 人脉关系：最近分享此视频的好友和最近分享此视频的会员
        /// </summary>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public ActionResult Control_Relations(int threadID)
        {
            ShareThread thread = ShareThreads.GetThread(threadID, false);
            if (thread == null)
                thread = new ShareThread();
            ViewData["userID"] = CurrentUser == null ? 0 : CurrentUser.UserID;
            ViewData["shareThread"] = thread;
            ViewData["MediaType"] = thread.MediaType;
            PagingDataSet<UserShareThread> shareThreads = UserShareThreads.GetFriendThreads(CurrentUser == null ? 0 : CurrentUser.UserID, thread.MediaType, null, 1);
            IList<UserShareThread> friendsShareThreads = new List<UserShareThread>();

            foreach (UserShareThread friendShareThread in shareThreads.Records)
            {
                if (friendShareThread.ThreadID == threadID)
                    friendsShareThreads.Add(friendShareThread);
            }
            ViewData["friendsShareThreads"] = friendsShareThreads;
            IList<UserShareThread> userShareThreads = UserShareThreads.GetUserThreadsByThreadID(threadID);

            return PartialView("Controls/Shares/Relations.ascx", userShareThreads);
        }

        /// <summary>
        /// 分享首页带头条的分享
        /// </summary>
        /// <param name="categoryID"></param>
        /// <param name="applicationID"></param>
        /// <param name="sortBy"></param>
        /// <param name="displayCount"></param>
        /// <returns></returns>
        public ActionResult Control_ListStickyShareThreadsBlock(MediaTypes? mediaType, int? applicationID, ShareThreadSortBy? sortBy, int? displayCount)
        {
            IList<ShareThread> shareThreads = ShareThreads.GetTopThreadsSortBy(sortBy ?? ShareThreadSortBy.HitTimes, displayCount ?? 10, null, mediaType ?? MediaTypes.Video, applicationID);
            return PartialView("Controls/Shares/ListStickyShareThreadsBlock.ascx", shareThreads);
        }

        /// <summary>
        /// 分享首页视频分享
        /// </summary>
        /// <param name="categoryID"></param>
        /// <param name="applicationID"></param>
        /// <param name="sortBy"></param>
        /// <param name="displayCount"></param>
        /// <returns></returns>
        public ActionResult Control_ListVideoShareBlock(MediaTypes? mediaType, ShareThreadSortBy? sortBy, int? displayCount)
        {
            IList<ShareThread> shareThreads = ShareThreads.GetTopThreadsSortBy(sortBy ?? ShareThreadSortBy.HitTimes, displayCount ?? 9, null, mediaType ?? MediaTypes.Video, null);
            return PartialView("Controls/Shares/ListVideoSharesBlock.ascx", shareThreads);
        }

        /// <summary>
        /// 分享首页视频分享
        /// </summary>
        /// <param name="mediaType"></param>
        /// <param name="sortBy"></param>
        /// <param name="displayCount"></param>
        /// <returns></returns>
        public ActionResult Control_ListVideoShareThreadsBlock(MediaTypes? mediaType, ShareThreadSortBy? sortBy, int? displayCount)
        {
            IList<ShareThread> shareThreads = ShareThreads.GetTopThreadsSortBy(sortBy ?? ShareThreadSortBy.HitTimes, displayCount ?? 9, null, mediaType ?? MediaTypes.Video, null);
            return PartialView("Controls/Shares/ListVideoShareThreadsBlock.ascx", shareThreads);
        }

        /// <summary>
        /// 分享首页图片分享
        /// </summary>
        /// <param name="applicationID"></param>
        /// <param name="sortBy"></param>
        /// <returns></returns>
        public ActionResult Control_ListThumbnailShareThreadsBlock(int? applicationID, MediaTypes? mediaType, ShareThreadSortBy? sortBy, int? displayCount)
        {
            IList<ShareThread> shareThreads = ShareThreads.GetTopThreadsSortBy(sortBy ?? ShareThreadSortBy.HitTimes, displayCount ?? 5, null, mediaType, applicationID);
            return PartialView("Controls/Shares/ListThumbnailShareThreadsBlock.ascx", shareThreads);
        }

        /// <summary>
        /// 分享首页音乐分享
        /// </summary>
        /// <param name="applicationID"></param>
        /// <param name="sortBy"></param>
        /// <param name="displayCount"></param>
        /// <returns></returns>
        public ActionResult Control_ListMusicShareThreadsBlock(MediaTypes? mediaType, ShareThreadSortBy? sortBy, int? displayCount)
        {
            IList<ShareThread> shareThreads = ShareThreads.GetTopThreadsSortBy(sortBy ?? ShareThreadSortBy.HitTimes, displayCount ?? 10, null, mediaType ?? MediaTypes.Music, null);
            ViewData["sortBy"] = (int)sortBy;
            return PartialView("Controls/Shares/ListMusicShareThreadsBlock.ascx", shareThreads);
        }

        public ActionResult Control_AddShare()
        {
            return PartialView("Controls/Shares/AddShare.ascx");
        }

        /// <summary>
        /// 图标分享列表
        /// </summary>
        /// <param name="applicationID"></param>
        /// <param name="sortBy"></param>
        /// <param name="displayCount"></param>
        /// <returns></returns>
        public ActionResult Control_ListIconShareThreadsBlock(MediaTypes? mediaType, int? applicationID, ShareThreadSortBy? sortBy, int? displayCount)
        {
            IList<ShareThread> shareThreads = ShareThreads.GetTopThreadsSortBy(sortBy ?? ShareThreadSortBy.HitTimes, displayCount ?? 5, null, mediaType, applicationID);
            ViewData["sortBy"] = (int)sortBy;
            return PartialView("Controls/Shares/ListIconShareThreadsBlock.ascx", shareThreads);
        }

        /// <summary>
        ///简单分享列表
        /// </summary>
        /// <param name="applicationID"></param>
        /// <param name="sortBy"></param>
        /// <returns></returns>
        public ActionResult Control_ListSimpleShareThreadsBlock(int? siteCategoryID, MediaTypes? mediaType, int? applicationID, ShareThreadSortBy? sortBy, int? displayCount)
        {
            IList<ShareThread> shareThreads = ShareThreads.GetTopThreadsSortBy(sortBy ?? ShareThreadSortBy.HitTimes, displayCount ?? 10, siteCategoryID, mediaType, applicationID);
            return PartialView("Controls/Shares/ListSimpleShareThreadsBlock.ascx", shareThreads);
        }

        public ActionResult Control_ListShareThreadsBlockByCategory(int? applicationID, int siteCategoryID, MediaTypes? mediaType, ShareThreadSortBy? sortBy, int? displayCount)
        {
            IList<ShareThread> shareThreads = ShareThreads.GetTopThreadsSortBy(sortBy ?? ShareThreadSortBy.HitTimes, displayCount ?? 10, siteCategoryID, mediaType, applicationID);
            return PartialView("Controls/Shares/ListShareThreadsByCategoryBlock.ascx");
        }

        /// <summary>
        /// 编辑站点分享
        /// </summary>
        public ActionResult Control_EditShareThread(int threadID)
        {
            ShareThread shareThread = ShareThreads.GetThread(threadID, true);
            if (shareThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ShareNotFound", applicationID), threadID)));
            }
            if (CurrentUser == null || (!CurrentUser.IsAdministrator && !CurrentUser.IsContentAdministrator))
                throw new Exception(GetResourceString("Exception_ShareEditAccessDenied", applicationID));
            return PartialView("Controls/Shares/EditShareThread.ascx", shareThread);
        }

        /// <summary>
        /// 频道标签列表显示
        /// </summary>
        /// <param name="pageSize">显示标签数量</param>
        /// <param name="sortBy">按分享数或用户数排序</param>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_SiteShareTagsBlock(int? displayCount, SiteTagsSortBy? sortBy)
        {
            Dictionary<string, KeyValuePair<int, SiteTag>> siteTags = UserTagManager.Instance(UserTagItemKeys.Instance().UserShareThread()).GetTopSiteTagsSortBy(sortBy ?? SiteTagsSortBy.ItemCount, displayCount ?? 50);
            return PartialView("Controls/Shares/SiteShareTagsBlock.ascx", siteTags);
        }

        /// <summary>
        /// 获取所有的根类别
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_SubMenus(int? categoryID, ShareChannelSubMenu? submenu)
        {
            List<SiteCategory> siteCategories = null;
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                SiteCategory siteCategory = SiteCategories.Instance(ApplicationIDs.Instance().Share()).Get(categoryID.Value, false);
                if (siteCategory != null)
                {
                    siteCategories = siteCategory.Childs;
                }
            }
            else
                siteCategories = SiteCategories.Instance(ApplicationIDs.Instance().Share()).GetRoots(false);
            ViewData["submenu"] = submenu;
            return PartialView("Controls/Shares/SubMenu.ascx", siteCategories);
        }

        /// <summary>
        /// 在模式框中进行分享
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_ShareInModal()
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));

            int threadID = Request.QueryString.GetInt("threadID", 0);
            bool? allowToClub = Request.QueryString.GetBool("allowToClub", false);
            ShareThread shareThread = ShareThreads.GetThread(threadID, false);

            if (CurrentUser != null && threadID > 0)
            {
                shareThread = ShareThreads.GetThread(threadID, false);
                if (shareThread == null)
                    ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, string.Format(GetResourceString("Message_Pattern_SiteShareNotFound", applicationID), threadID));
                else if (UserShareThreads.IsShared(CurrentUser.UserID, threadID))
                    ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Message_ShareNotRepeat", applicationID));
            }
            else
            {
                string url = Request.QueryString.GetString("url", string.Empty);
                if (string.IsNullOrEmpty(url))
                    ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Validate_LinkNotEmpty", applicationID));
                if (ShareThreads.UrlIsExist(url, out threadID))
                {
                    shareThread = ShareThreads.GetThread(threadID, false);
                    if (shareThread == null)
                        ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, string.Format(GetResourceString("Message_Pattern_SiteShareNotFound", applicationID), threadID));
                    else if (UserShareThreads.IsShared(CurrentUser.UserID, threadID))
                        ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Message_ShareNotRepeat", applicationID));
                }
                else
                {
                    int sharedApplicationID = Request.QueryString.GetInt("applicationID", 0);
                    if (sharedApplicationID <= 0)
                        shareThread = new ShareThread(UrlParserFactory.Instance().ParseUrl(url));
                    else
                        shareThread = new ShareThread();

                    shareThread.UserID = CurrentUser.UserID;
                    shareThread.Author = CurrentUser.DisplayName;
                    shareThread.ApplicationID = sharedApplicationID;
                    shareThread.Subject = Request.QueryString.GetString("subject", shareThread.Subject);
                    shareThread.ThumbnailUrl = Request.QueryString.GetString("thumbnailUrl", shareThread.ThumbnailUrl);
                    shareThread.Body = Request.QueryString.GetString("body", shareThread.Body);
                    shareThread.ContentUserID = Request.QueryString.GetInt("contentUserID", 0);
                    shareThread.ContentAuthor = Request.QueryString.GetString("contentAuthor", string.Empty);
                    shareThread.URL = url;
                }
            }

            ViewData["allowToClub"] = allowToClub ?? false;
            return PartialView("Controls/Shares/ShareInModal.ascx", shareThread);

        }


        #endregion

        #region Operation

        /// <summary>
        /// 更新站点分享
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateShareThread(int threadID)
        {
            ShareThread thread = ShareThreads.GetThread(threadID, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ShareNotFound", applicationID), threadID)));
            }
            if (CurrentUser == null || (!CurrentUser.IsAdministrator && !CurrentUser.IsContentAdministrator))
                throw new Exception(GetResourceString("Exception_ShareEditAccessDenied", applicationID));
            thread.Subject = Request.Form.GetString("subject", thread.Subject).Trim();
            if (string.IsNullOrEmpty(thread.Subject))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_TitleExist", applicationID));
                return RedirectToAction("Control_EditShareThread", new RouteValueDictionary { { "threadID", threadID } });
            }
            thread.PlayerUrl = Request.Form.GetString("playerUrl", thread.PlayerUrl);
            thread.Body = Request.Form.GetString("body", thread.Body).Trim();
            thread.ThumbnailUrl = Request.Form.GetString("thumbnailUrl", thread.ThumbnailUrl).Trim();
            thread.PlayerUrl = Request.Form.GetString("playerUrl", thread.PlayerUrl).Trim();
            thread.SiteCategoryID = Request.Form.GetInt("siteCategoryID", thread.SiteCategoryID);
            thread.MediaType = Request.Form.Get<MediaTypes>("mediaType", thread.MediaType);
            ShareThreads.UpdateThread(thread);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_EditShareOk", applicationID));
            return RedirectToAction("Control_EditShareThread", new RouteValueDictionary { { "threadID", threadID } });
        }

        /// <summary>
        /// 删除分享
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteShareThread(int threadID, string returnUrl)
        {
            ShareThread thread = ShareThreads.GetThread(threadID, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ShareNotFound", applicationID), threadID)));
            }
            if (CurrentUser == null || (!CurrentUser.IsAdministrator && !CurrentUser.IsContentAdministrator))
                throw new Exception(GetResourceString("Exception_ShareDeleteAccessDenied", applicationID));

            ShareThreads.DeleteThread(threadID);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteSucceed", applicationID));
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="isback"></param>
        /// <param name="returnUrl"></param>
        /// <returns></returns>
        public ActionResult IsBack(int threadID, bool isback, string returnUrl)
        {
            ShareThread thread = ShareThreads.GetThread(threadID, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ShareNotFound", applicationID), threadID)));
            }

            ShareThreads.UpdateThread(thread);
            if (isback)
            {
                thread.BackerCount = thread.BackerCount + 1;
            }
            else
            {
                thread.DissenterCount = thread.DissenterCount + 1;
            }
            return RedirectToAction(returnUrl);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="threadid"></param>
        /// <returns></returns>
        public ActionResult updateUserShareThread(int threadid)
        {
            List<int> threadId = new List<int>();
            threadId.Add(threadid);
            ShareThreads.UpdateAuditingStatus(threadId, AuditingStatuses.Success);
            return RedirectToAction("ShowShareThread", new RouteValueDictionary { { "threadID", threadid } });
        }

        /// <summary>
        /// 在模式中处理创建更新用户分享
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateUserShareThread(int threadID)
        {
            ShareThread shareThread = null;
            if (threadID > 0)
            {
                shareThread = ShareThreads.GetThread(threadID, true);
                if (shareThread == null)
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, string.Format(GetResourceString("Message_Pattern_SiteShareNotFound", applicationID), threadID));
                if (UserShareThreads.IsShared(CurrentUser.UserID, threadID))
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Message_ShareNotRepeat", applicationID));

                if (TempData["StatusMessageData"] != null)
                    return RedirectToAction("ShareDialog");
            }
            else
            {

                string url = Request.Form.GetString("url", string.Empty);
                if (ShareThreads.UrlIsExist(url, out threadID))
                {
                    shareThread = ShareThreads.GetThread(threadID, true);
                    if (shareThread == null)
                        return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_Pattern_SiteShareNotFound", applicationID), threadID)));
                    if (UserShareThreads.IsShared(CurrentUser.UserID, threadID))
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Message_ShareNotRepeat", applicationID));
                }
                else
                {
                    shareThread = new ShareThread();
                    shareThread.URL = url;
                    shareThread.UserID = CurrentUser.UserID;
                    shareThread.Author = CurrentUser.DisplayName;
                    shareThread.ContentUserID = Request.Form.GetInt("contentUserID", 0);
                    shareThread.ContentAuthor = Request.Form.GetString("contentAuthor", string.Empty);
                    shareThread.Subject = Request.Form.GetString("subject", string.Empty);
                    shareThread.Body = Request.Form.GetString("body", string.Empty);
                    shareThread.ThumbnailUrl = Request.Form.GetString("thumbnailUrl", string.Empty);
                    shareThread.ApplicationID = Request.Form.GetInt("applicationID", 0);
                    shareThread.MediaType = (MediaTypes)Request.Form.Get<int>("mediaType", (int)shareThread.MediaType);
                    shareThread.PlayerUrl = Request.Form.GetString("playerUrl", string.Empty);
                    shareThread.SiteCategoryID = Request.Form.GetInt("siteCategoryID", 0);
                    shareThread.ApplicationID = Request.Form.GetInt("applicationID", 0);
                    shareThread.PrivacyStatus = (PrivacyStatuses)Request.Form.GetInt("privacyStatus", (int)PrivacyStatuses.Public);
                    ShareThreads.CreateThread(shareThread);
                    if (shareThread.ThreadID <= 0)
                        throw new Exception(string.Format(GetResourceString("Exception_CreateShareAborted", applicationID)));
                    threadID = shareThread.ThreadID;
                }
            }
            //处理标签
            List<UserTag> selectedTags = null;
            string selectedTagNames = Request.Form.GetString("userTags", string.Empty);
            if (!string.IsNullOrEmpty(selectedTagNames))
            {
                string[] tagNames = selectedTagNames.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                if (tagNames != null && tagNames.Length > 0)
                {
                    selectedTags = new List<UserTag>();
                    foreach (string tagName in tagNames)
                    {
                        if (!string.IsNullOrEmpty(tagName.Trim()))
                        {
                            UserTag userTag = UserTagManager.Instance(UserTagItemKeys.Instance().UserShareThread()).GetUserTag(tagName, CurrentUser.UserID, false);
                            if (userTag == null)
                            {
                                userTag = new UserTag();
                                userTag.UserID = CurrentUser.UserID;
                                userTag.TagName = StringUtils.Trim(tagName, 8, string.Empty);
                                bool isCreated = UserTagManager.Instance(UserTagItemKeys.Instance().UserShareThread()).CreateUserTag(userTag);
                                if (!isCreated)
                                    continue;
                            }
                            selectedTags.Add(userTag);
                        }
                    }
                }
            }
            UserShareThread userShareThread = new UserShareThread();
            userShareThread.ThreadID = threadID;
            userShareThread.UserID = CurrentUser.UserID;
            userShareThread.Author = CurrentUser.DisplayName;
            userShareThread.Body = Request.Form.GetString("commentBody", string.Empty);
            userShareThread.PrivacyStatus = (PrivacyStatuses)Request.Form.GetInt("privacyStatus", (int)userShareThread.PrivacyStatus);
            if (userShareThread.PrivacyStatus == PrivacyStatuses.NeedPassword)
                userShareThread.Password = Request.Form.GetString("privacyStatus_Password", userShareThread.Password);

            UserShareThreads.CreateThread(userShareThread, selectedTags);
            if (selectedTags != null && selectedTags.Count > 0)
                UserTagManager.Instance(UserTagItemKeys.Instance().UserShareThread()).UpdateUserTagsOfItem(userShareThread.UserThreadID, userShareThread.UserID, selectedTags.Select(n => n.TagName).ToList());
            if (userShareThread.UserThreadID > 0)
            {
                if (userShareThread.SiteShareThread.AuditingStatus != AuditingStatuses.Success)
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "分享成功，需要管理员通过审核才能对外显示！");
                else
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_AddShareSucceed", applicationID));
            }
            else
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_AddShareAborted", applicationID));
            return RedirectToAction("ShareDialog", new RouteValueDictionary { { "threadID", threadID } });
        }

        public ActionResult ShareToFirend()
        {
            return null;
        }

        #endregion

        #endregion

        #region ShareComment

        #region Control

        /// <summary>
        /// 评论列表
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ListShareComments(int threadID, string controlName, int? pageIndex)
        {
            ShareThread shareThread = ShareThreads.GetThread(threadID, false);
            if (shareThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ShareNotFound", applicationID), threadID)));
            }
            PagingDataSet<ShareComment> pds;
            if (CurrentUser != null && shareThread.UserID == CurrentUser.UserID)
                pds = ShareComments.GetCommentsOfThread(shareThread.ThreadID, null, pageIndex ?? 1, true);
            else
                pds = ShareComments.GetCommentsOfThread(shareThread.ThreadID, null, pageIndex ?? 1, false);
            return PartialView(string.Format("Controls/Shares/{0}.ascx", string.IsNullOrEmpty(controlName) ? "ListShareComments" : controlName), pds);
        }

        #endregion

        #region Operation

        /// <summary>
        /// 创建分享评论
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateShareComment(int threadID, string controlName, int? pageIndex)
        {

            if (!SiteSettingsManager.GetSiteSettings().EnableAnonymousPosting && CurrentUser == null)
                return Redirect(SiteUrls.Instance().Error("你没有评论的权限!"));

            string body = Request.Form.GetString("commentBody", string.Empty).Trim();
            if (string.IsNullOrEmpty(body))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Validate_ImportCommont", applicationID);
                TempData["commentBody"] = body;
                return RedirectToAction("Control_ListShareComments", CurrentRouteValueDictionary);
            }

            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                TempData["parentID"] = Request.Form.GetInt("parentID", 0);
                if (verifyCode != null)
                {
                    verifyCode = verifyCode.Trim();
                }
                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Validate_ValidateInformationError", applicationID);
                    TempData["commentBody"] = body;
                    return RedirectToAction("Control_ListShareComments", CurrentRouteValueDictionary);
                }
            }
            ShareThread shareThread = ShareThreads.GetThread(threadID, false);
            if (shareThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format("ID：“{0}”主题未找到！", threadID)));
            }
            ShareComment shareComment = new ShareComment();
            shareComment.Body = Request.Form.GetString("commentBody", string.Empty).Trim();
            shareComment.ThreadID = threadID;
            shareComment.Author = CurrentUser == null ? "匿名用户" : CurrentUser.DisplayName;
            shareComment.UserID = CurrentUser == null ? 0 : CurrentUser.UserID;
            shareComment.ParentID = Request.Form.GetInt("parentID", 0);

            if (shareComment.ParentID > 0)
            {
                ShareComment parentCommet = ShareComments.GetComment(shareComment.ParentID, true);
                if (parentCommet != null)
                {
                    shareComment.ParentAuthor = parentCommet.Author;
                    shareComment.ParentBody = parentCommet.Body;
                    shareComment.ParentUserID = parentCommet.UserID;
                    if (shareComment.Body.StartsWith(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentCommet.Author), StringComparison.CurrentCultureIgnoreCase))
                    {
                        if (shareComment.Body.Length > string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentCommet.Author).Length)
                        {
                            shareComment.Body = shareComment.Body.Substring(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentCommet.Author).Length);
                        }
                        else
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = GetResourceString("Message_ImportReversion", applicationID);
                            TempData["commentBody"] = shareComment.Body;
                            TempData["parentID"] = shareComment.ParentID;
                            return RedirectToAction("Control_ListShareComments", CurrentRouteValueDictionary);
                        }
                    }
                }
            }
            try
            {
                ShareComments.CreateComment(shareComment);
            }
            catch (SPBException e)
            {
                ExceptionMessage exceptionMessage = ResourceManager.GetExceptionMessage(e.ExceptionType, ApplicationIDs.Instance().Share());
                if (exceptionMessage != null)
                {
                    TempData["StatusMessageType"] = StatusMessageType.Error;
                    TempData["StatusMessageContent"] = exceptionMessage.Title + "，" + exceptionMessage.Body;
                }
                return RedirectToAction("Control_ListShareComments", CurrentRouteValueDictionary);
            }
            if ((int)shareComment.AuditingStatus >= (int)SiteSettingsManager.GetSiteSettings().AuditingStatusesForPublicDisplay)
            {
                TempData["StatusMessageContent"] = GetResourceString("Message_AddCommontSucceed", applicationID);
            }
            else
            {
                TempData["StatusMessageContent"] = GetResourceString("Message_DescryAfterAuditing", applicationID);
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            ViewData["ShareThread"] = shareThread;
            return RedirectToAction("Control_ListShareComments", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 删除单个评论
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteShareComment(int commentID, int threadID, string controlName, int? pageIndex)
        {
            ShareComment comment = ShareComments.GetComment(commentID, true);
            if (comment == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_CommentNotFound", applicationID), commentID)));
            }
            ShareThread shareThread = ShareThreads.GetThread(comment.ThreadID, false);
            if (shareThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ShareNotFound", applicationID), comment.ThreadID)));
            }
            if (CurrentUser != null && (comment.UserID != CurrentUser.UserID && !CurrentUser.IsContentAdministrator && !CurrentUser.IsAdministrator))
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());

            ShareComments.DeleteComment(commentID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSucceed", applicationID);
            return RedirectToAction("Control_ListShareComments", new RouteValueDictionary { { "threadID", comment.ThreadID }, { "controlName", controlName }, { "pageIndex", pageIndex ?? 1 } });
        }

        #endregion

        #endregion

        #region Help Method

        /// <summary>
        /// 生成分享类别面包屑
        /// </summary>
        private void ShowCategoryBreadCrumb()
        {
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ShareHome", applicationID), SiteUrls.Instance().ChannelShareHome());
            Stack<SiteCategory> siteCategory = new Stack<SiteCategory>();

            Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, "", "", Globals.SiteName));
            BreadCrumb.AddBreadCrumbItem("", SiteUrls.Instance().ChannelShareHome());
            BreadCrumb.AddBreadCrumbItem("", string.Empty);
        }

        /// <summary>
        /// 获取当前类别的子类别，如果是最后一级，则获取同辈类别
        /// </summary>
        /// <param name="siteCategoryID">当前类别</param>
        private List<SiteCategory> GetChildCategories(int? currentCagoryID, out Stack<SiteCategory> categoryStack)
        {
            categoryStack = new Stack<SiteCategory>();
            SiteCategory currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Share()).Get(currentCagoryID ?? 0, false);

            List<SiteCategory> siteCategoties = null;

            if (currentCategory == null)
                siteCategoties = SiteCategories.Instance(ApplicationIDs.Instance().Share()).GetRoots(false);
            else if (currentCategory.ChildCount > 0)
                siteCategoties = currentCategory.Childs;
            else
            {
                if (currentCategory.ParentID <= 0)
                    siteCategoties = SiteCategories.Instance(ApplicationIDs.Instance().Share()).GetRoots(false);
                else
                    siteCategoties = SiteCategories.Instance(ApplicationIDs.Instance().Share()).Get(currentCategory.ParentID, false).Childs;
            }

            while (currentCategory != null)
            {
                if (currentCategory.ChildCount > 0)
                    categoryStack.Push(currentCategory);
                currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Share()).Get(currentCategory.ParentID, false);
            }
            return siteCategoties;
        }

        #endregion
    }

    /// <summary>
    ///频道分享BLock类型 
    /// </summary>
    public enum ShareBlockStyle
    {
        //头条
        First = 0,
        //视频
        video = 1,
        //音乐
        music = 2,
        //图片
        Photo = 3,
        //普通
        Simple = 4,
        //头像
        Avatar = 5
    }

    /// <summary>
    /// 分享模式框内菜单
    /// </summary>
    public enum ShareInModalMenu
    {
        //添加我的分享
        ShareToMyShare,
        //分享给朋友
        ShareToMyFriends,
        //分享到群组
        shareToMyClubs
    }

    /// <summary>
    /// 分享子菜单
    /// </summary>
    public enum ShareChannelSubMenu
    {
        home,
        rank
    }
}