﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Share;
using SpaceBuilder.Common;
using SpaceBuilder.Utils;
using Lucene.Net.Documents;
using Lucene.Net.Search;
using Lucene.Net.Index;
using Lucene.Net.QueryParsers;

namespace SpaceBuilder.LuceneSearch
{
    public class ShareSearchManager : SearchManagerBase<ShareThread>
    {
        /// <summary>
        /// 索引文件目录名称
        /// </summary>
        public static readonly string rewardIndexFileDirectory = "Share";

        private ShareSearchManager(string indexFileDirectory) : base(indexFileDirectory) { }
        private static volatile ShareSearchManager _self = null;
        private static readonly object lockObject = new object();

        public static ShareSearchManager Instance()
        {
            if (_self == null)
            {
                lock (lockObject)
                {
                    if (_self == null)
                    {
                        _self = new ShareSearchManager(rewardIndexFileDirectory);
                    }
                }
            }
            return _self;
        }

        public override void InitializeIndex(string indexPath)
        {
            if (!System.IO.Directory.Exists(indexPath))
            {
                try
                {
                    System.IO.Directory.CreateDirectory(indexPath);
                }
                catch
                {
                    throw new ApplicationException(string.Format("create Directory '{0}' failed", PhysicalIndexDirectory));
                }
            }

            #region 索引

            int indexPageSize = 2000;
            bool createIndexFile = true;
            PagingDataSet<ShareThread> pds = ShareThreads.GetThreadsForAdmin(-1, string.Empty, Applications.GetApplication(ApplicationIDs.Instance().Share()).AuditingStatusForPublicDisplay, indexPageSize, 1);
            double tIndex = Convert.ToDouble(pds.TotalRecords) / Convert.ToDouble(indexPageSize);
            int indexPageIndex = (int)Math.Ceiling(tIndex);

            if (pds.TotalRecords > 0)
            {
                //分多次进行索引
                for (int i = 1; i <= indexPageIndex; i++)
                {
                    if (i != 1)
                        pds = ShareThreads.GetThreadsForAdmin(-1, string.Empty, Applications.GetApplication(ApplicationIDs.Instance().Share()).AuditingStatusForPublicDisplay, indexPageSize, i);
                    Insert(pds.Records, indexPath, createIndexFile);
                    if (createIndexFile)
                        createIndexFile = false;
                }
            }

            #endregion
        }

        protected override Lucene.Net.Documents.Document ConvertObjToDocument(ShareThread post)
        {
            if (post == null)
                return null;
            Document doc = new Document();
            Field f;
            f = new Field(ShareIndexFields.AuditingStatus, ((int)post.AuditingStatus).ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(f);
            //f = new Field(ShareIndexFields.PrivacyStatus, ((int)post.PrivacyStatus).ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            //doc.Add(f);
            f = new Field(ShareIndexFields.Body, post.Body.ToLower(), Field.Store.YES, Field.Index.ANALYZED);
            doc.Add(f);
            f = new Field(ShareIndexFields.DateCreated, DateTools.DateToString(post.DateCreated, DateTools.Resolution.DAY), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(f);
            f = new Field(ShareIndexFields.MediaType, ((int)post.MediaType).ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(f);
            f = new Field(ShareIndexFields.ThreadID, post.ThreadID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(f);
            f = new Field(ShareIndexFields.Subject, post.Subject.ToLower(), Field.Store.YES, Field.Index.ANALYZED);
            doc.Add(f);
            f = new Field(ShareIndexFields.UserID, post.UserID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(f);
            f = new Field(ShareIndexFields.Author, post.Author, Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(f);
            f = new Field(ShareIndexFields.URL, post.URL, Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(f);
            f = new Field(ShareIndexFields.URLSalt, post.URLSalt, Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(f);
            f = new Field(ShareIndexFields.ThumbnailUrl, post.ThumbnailUrl, Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(f);
            f = new Field(ShareIndexFields.SiteCategoryID, post.SiteCategoryID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(f);
            f = new Field(ShareIndexFields.ApplicationID, post.ApplicationID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(f);
            return doc;
        }

        protected override ShareThread ConvertDocumentToObj(Lucene.Net.Documents.Document doc)
        {
            ShareThread newThread = new ShareThread();

            int auditingStatusValue = (int)AuditingStatuses.Success;
            int.TryParse(doc.Get(ShareIndexFields.AuditingStatus), out auditingStatusValue);
            newThread.AuditingStatus = (AuditingStatuses)auditingStatusValue;

            int threadID;
            int.TryParse(doc.Get(ShareIndexFields.ThreadID), out threadID);
            newThread.ThreadID = threadID;

            newThread.Body = doc.Get(ShareIndexFields.Body);
            newThread.URL = doc.Get(ShareIndexFields.URL);
            newThread.URLSalt = doc.Get(ShareIndexFields.URLSalt);

            int userID;
            int.TryParse(doc.Get(ShareIndexFields.UserID), out userID);
            newThread.UserID = userID;

            newThread.Author = doc.Get(ShareIndexFields.Author);
            newThread.Subject = doc.Get(ShareIndexFields.Subject);
            newThread.ThumbnailUrl = doc.Get(ShareIndexFields.ThumbnailUrl);

            try
            {
                newThread.DateCreated = DateTools.StringToDate(doc.Get(ShareIndexFields.DateCreated));
            }
            catch { }

            int SiteCategoryID;
            int.TryParse(doc.Get(ShareIndexFields.SiteCategoryID), out SiteCategoryID);
            newThread.SiteCategoryID = SiteCategoryID;

            int ApplicationID;
            int.TryParse(doc.Get(ShareIndexFields.ApplicationID), out ApplicationID);
            newThread.ApplicationID = ApplicationID;

            if (!string.IsNullOrEmpty(doc.Get(ShareIndexFields.MediaType)))
            {
                int MediaType = 1;
                int.TryParse(doc.Get(ShareIndexFields.MediaType), out MediaType);
                newThread.MediaType = (MediaTypes)MediaType;
            }
            return newThread;
        }
        

        /// <summary>
        /// 根据条件执行查询并返回结果
        /// </summary>
        /// <param name="searchQuery">查询条件</param>
        public SearchResultDataSet<ShareThread> Search(ShareSearchQuery searchQuery)
        {
            //索引文件不存在时，返回null
            if (!IsIndexFilesExists)
                return new SearchResultDataSet<ShareThread>();

            DateTime startTime = DateTime.Now;
            BooleanQuery currentQuery = new BooleanQuery();
            BooleanQuery queryForFilter = new BooleanQuery();
            if (searchQuery.MediaType.HasValue)
            {
                currentQuery.Add(new TermQuery(new Term(ShareIndexFields.MediaType, ((int)searchQuery.MediaType.Value).ToString())), BooleanClause.Occur.MUST);
            }

            if (searchQuery.UserID > 0)
            {
                queryForFilter.Add(new TermQuery(new Term(ShareIndexFields.UserID, searchQuery.UserID.ToString())), BooleanClause.Occur.MUST);
            }

            if (searchQuery.ApplicationID > 0)
            {
                currentQuery.Add(new TermQuery(new Term(ShareIndexFields.ApplicationID, searchQuery.ApplicationID.ToString())), BooleanClause.Occur.MUST);
            }


            if (searchQuery.SiteCategoryID > 0)
            {
                Term siteCategoryIDTerm = new Term(ShareIndexFields.SiteCategoryID, searchQuery.SiteCategoryID.ToString());
                Query siteCategoryIDQuery = new TermQuery(siteCategoryIDTerm);
                currentQuery.Add(siteCategoryIDQuery, BooleanClause.Occur.MUST);
            }


            if (!string.IsNullOrEmpty(searchQuery.KeyWord))
            {
                Query postKeywordQuery = null;
                searchQuery.KeyWord = StringUtilsForLucene.LuceneKeywordsScrubber(searchQuery.KeyWord.ToLower());
                string keywordSegments = SegmentForQueryParser(searchQuery.KeyWord);

                if (!string.IsNullOrEmpty(searchQuery.KeyWord.Trim()))
                {
                    MultiFieldQueryParser keywordParser = new MultiFieldQueryParser(CurrentLuceneVersion, new string[] { ShareIndexFields.Body, ShareIndexFields.Subject }, GetChineseAnalyzerOfUnTokenized());
                    keywordParser.SetDefaultOperator(QueryParser.OR_OPERATOR);
                    keywordParser.SetLowercaseExpandedTerms(true);

                    postKeywordQuery = keywordParser.Parse(keywordSegments);

                    currentQuery.Add(postKeywordQuery, BooleanClause.Occur.MUST);
                }
            }

            SortField[] sortFields = new SortField[] { SortField.FIELD_SCORE };

            Filter filter = null;
            if (queryForFilter.Clauses().Count > 0)
                filter = new QueryWrapperFilter(queryForFilter);

            SearchResultDataSet<ShareThread> pds = Search(currentQuery, filter, sortFields, searchQuery.PageIndex, searchQuery.PageSize);
            foreach (var item in pds.Records)
            {
                item.Subject = HighlighterForKeyWord(item.Subject, searchQuery.KeyWord);
                item.Body = HighlighterForKeyWord(item.Body, searchQuery.KeyWord);
            }
            return pds;
        }
    }
}
