﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Web;
using SpaceBuilder.Vote;
using SpaceBuilder.Common;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using System.IO;
using System.Collections;
using SpaceBuilder.Utils;
using SpaceBuilder.Club;
using System.Web.Routing;
using System.Web.UI;

namespace SpaceBuilder.Vote.Controllers
{
    [HttpCompress]
    public class ChannelVoteController : ChannelControllerBase
    {
        int applicationID = ApplicationIDs.Instance().Vote();

        #region Vote

        #region View

        public ActionResult Home()
        {
            SetPageTitle(GetResourceString("Title_VoteHome", applicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_Vote", applicationID), string.Empty);
            return View("Pages/Votes/Home.aspx", "Masters/Vote.master");
        }
        public ActionResult VoteThreadRanks(VoteThreadSortBy? sortBy)
        {
            SetPageTitle(GetResourceString("Tilte_VoteRank", applicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_Vote", applicationID), string.Empty);
            ViewData["sortBy"] = sortBy.HasValue ? (int)sortBy : (int)VoteThreadSortBy.DateCreated;

            return View("Pages/Votes/VoteThreadRanks.aspx", "Masters/Vote.master");
        }

        /// <summary>
        /// 显示投票详细页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShowVoteThread(int threadID, bool? isResult)
        {
            SetPageTitle(GetResourceString("Title_VoteDetail", applicationID));

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_Vote", applicationID), SiteUrls.Instance().ChannelVoteHome());
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_VoteRanks", applicationID), SPBUrlHelper.Action("VoteThreadRanks", "ChannelVote"));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_ShowVote", applicationID), string.Empty);
            VoteThread voteThread = VoteThreads.GetThread(threadID, true);

            if (voteThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote", applicationID), threadID)));
            }

            if (CurrentUser == null || (voteThread.UserID != CurrentUser.UserID && !CurrentUser.IsContentAdministrator && !CurrentUser.IsAdministrator))
            {

                switch (voteThread.PrivacyStatus)
                {
                    case PrivacyStatuses.OnlyFriend:
                        //仅朋友可见
                        bool ismyFriend = Friends.IsFriend(CurrentUser == null ? 0 : CurrentUser.UserID, voteThread.UserID);
                        if (!ismyFriend)
                        {
                            ViewData["isShow"] = false;
                            return View("Pages/Votes/ShowVoteThread.aspx", "Masters/Vote.master", voteThread);
                        }
                        break;
                    case PrivacyStatuses.Privacy:
                        // 私密
                        ViewData["isShow"] = false;
                        return View("Pages/Votes/ShowVoteThread.aspx", "Masters/Vote.master", voteThread);
                    case PrivacyStatuses.Public:
                        // 公开
                        break;
                    default:
                        break;
                }
            }

            int auditingStatusValue = (int)Applications.GetApplication(ApplicationIDs.Instance().Vote()).AuditingStatusForPublicDisplay;
            if ((int)voteThread.AuditingStatus < auditingStatusValue)
            {
                if (voteThread.AuditingStatus == AuditingStatuses.Fail)
                    ViewData["errorMessage"] = GetResourceString("Message_ThisVoteNotPass", applicationID);
                else
                    ViewData["errorMessage"] = GetResourceString("Message_ThisVoteIsAuditing", applicationID);
                ViewData["errorAuditingStatus"] = true;
            }

            isResult = isResult.HasValue ? isResult.Value : false;
            if (voteThread.VoteStaus == VoteStatuses.Expired || CurrentUser == null)
                isResult = true;
            bool isVote = VoteThreads.IsVoted(threadID, CurrentUser != null ? CurrentUser.UserID : 0);
            if (isVote || voteThread.EndDate < DateTime.Now)
            {
                isResult = true;
                isVote = true;
            }

            ViewData["isVote"] = isVote;
            ViewData["isResult"] = isResult;
            ViewData["TopThreads"] = VoteThreads.GetTopOwnerThreads(PresentAreaIDs.Instance().UserDomain(), voteThread.UserID, 9);

            #region 浏览器标题、计数、meta标签
            SetPageTitle(voteThread.Subject);
            CounterFactory.IncreaseCount<VoteThreadCounter>(IncreasableCountTypes.WebView, threadID);
            SetPageMeta(voteThread);
            #endregion
            return View("Pages/Votes/ShowVoteThread.aspx", "Masters/Vote.master", voteThread);
        }

        #endregion

        #region Controls

        /// <summary>
        /// 显示投票首页各列表块
        /// </summary>
        /// <param name="sortBy"></param>
        /// <param name="status"></param>
        /// <param name="modelTitle"></param>
        /// <returns></returns>
        public ActionResult Control_ListVoteThreadsBlock(VoteThreadSortBy? sortBy, VoteStatuses status, ListBlockStyle style, string Title, int? disPlayCount)
        {
            ViewData["Title"] = Title;
            ViewData["sortBy"] = (int)sortBy;
            ViewData["status"] = (int)status;
            ViewData["Style"] = (int)style;
            IList<VoteThread> threads = VoteThreads.GetTopThreadsSortBy(PresentAreaIDs.Instance().UserDomain(), null, sortBy ?? VoteThreadSortBy.HitTimes, status, disPlayCount ?? 5);
            return PartialView("Controls/Votes/ListVoteThreadsBlock.ascx", threads);
        }

        /// <summary>
        /// 侧边栏排序内容块
        /// </summary>
        /// <param name="sortBy"></param>
        /// <param name="Title"></param>
        /// <param name="disPlayCount"></param>
        /// <returns></returns>
        public ActionResult Control_SideVoteSortBlock(VoteThreadSortBy? sortBy, VoteStatuses? status, string Title, int? disPlayCount)
        {
            ViewData["Title"] = Title;
            ViewData["sortBy"] = (int)sortBy;
            IList<VoteThread> threads = VoteThreads.GetTopThreadsSortBy(PresentAreaIDs.Instance().UserDomain(), null, sortBy ?? VoteThreadSortBy.HitTimes, status, disPlayCount ?? 5);
            return PartialView("Controls/Votes/SideVoteSortBlock.ascx", threads);
        }

        /// <summary>
        /// 侧边推荐投票内容块
        /// </summary>
        /// <param name="commendTypeID"></param>
        /// <param name="disPlayCount"></param>
        /// <returns></returns>
        public ActionResult Control_SideVotesCommendBlok(int commendTypeID, int? disPlayCount)
        {
            IList<RecommendedItem> recommends = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().Vote(), commendTypeID, disPlayCount ?? 3);
            IList<VoteThread> threads = new List<VoteThread>();

            RecommendationType recommendationType = RecommendationTypes.GetRecommendationType(commendTypeID, false);
            ViewData["Title"] = recommendationType != null ? recommendationType.Name : string.Empty;

            foreach (RecommendedItem recommend in recommends)
            {
                threads.Add(VoteThreads.GetThread(recommend.ItemID, false));
            }

            return PartialView("Controls/Votes/SideVoteSortBlock.ascx", threads);
        }

        ///// <summary>
        ///// 邀请好友
        ///// </summary>
        ///// <returns></returns>
        //public ActionResult Control_InviteFriends(int threadID)
        //{
        //    ViewData["threadID"] = threadID;
        //    return PartialView("Controls/Votes/InviteFriends.ascx");
        //}

        /// <summary>
        /// 推荐投票
        /// </summary>
        /// <returns></returns>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_VotesCommendBlock(int commendTypeID, int? disPlayCount, ListBlockStyle style)
        {
            IList<RecommendedItem> recommends = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().Vote(), commendTypeID, disPlayCount ?? 3);
            IList<VoteThread> threads = new List<VoteThread>();

            RecommendationType recommendationType = RecommendationTypes.GetRecommendationType(commendTypeID, false);
            ViewData["Title"] = recommendationType != null ? recommendationType.Name : string.Empty;

            foreach (RecommendedItem recommend in recommends)
            {
                threads.Add(VoteThreads.GetThread(recommend.ItemID, false));
            }

            return PartialView("Controls/Votes/ListVoteThreadsBlock.ascx", threads);
        }

        /// <summary>
        /// 显示投票结果
        /// </summary>
        public ActionResult Control_ListOptionVoters(int optionID, int? pageSize, int? pageIndex)
        {
            PagingDataSet<VoteRecord> pds = VoteThreads.GetVoteRecordsOfOption(optionID, pageSize ?? 8, pageIndex ?? 1);
            ViewData["optionID"] = optionID;
            ViewData["pageSize"] = pageSize;
            return PartialView("Controls/Votes/ListOptionVoters.ascx", pds);
        }

        /// <summary>
        /// 显示投票列表
        /// </summary>
        public ActionResult Control_ListVoteThreads(VoteThreadSortBy? sortBy, VoteStatuses? status, int? pageIndex)
        {
            ViewData["sortBy"] = sortBy.HasValue ? (int)sortBy : (int)VoteThreadSortBy.DateCreated;
            ViewData["status"] = status;
            PagingDataSet<VoteThread> pds = VoteThreads.GetPagingThreadsSortBy(PresentAreaIDs.Instance().UserDomain(), null, sortBy ?? VoteThreadSortBy.HitTimes, status, 12, pageIndex ?? 1);

            return PartialView("Controls/Votes/ListVoteThreads.ascx", pds);
        }

        /// <summary>
        /// 在频道里显示投票
        /// </summary>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public ActionResult Control_ShowVoteThreadBlock(int? threadID)
        {
            bool isVote = VoteThreads.IsVoted(threadID ?? 0, CurrentUser != null ? CurrentUser.UserID : 0);
            ViewData["isVote"] = isVote;
            VoteThread thread = VoteThreads.GetThread(threadID ?? 0, false);
            return PartialView("Controls/Votes/ShowVoteThreadBlock.ascx", thread);
        }

        /// <summary>
        /// 更新截止日期
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_UpdateEndDate(int threadID)
        {
            ViewData["threadID"] = threadID;
            return PartialView("Controls/Votes/UpdateEndDate.ascx");
        }

        /// <summary>
        /// 在频道圈子首页主区域显示圈子投票模块
        /// </summary>
        /// <param name="displayCount"></param>
        /// <param name="sortBy"></param>
        /// <param name="status"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ListClubVoteThreadsBlock(int? disPlayCount)
        {
            ViewData["Title"] = GetResourceString("Title_NewVote", applicationID);
            IList<VoteThread> threads = VoteThreads.GetTopThreadsSortBy(PresentAreaIDs.Instance().ClubDomain(), null, VoteThreadSortBy.DateCreated, null, 10);
            return PartialView("Controls/Votes/ListClubVotesBlock.ascx", threads);
        }

        #endregion

        #region Action

        /// <summary>
        /// 投票
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Vote(int threadID)
        {
            VoteThread voteThread = VoteThreads.GetThread(threadID, false);
            if (voteThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote", applicationID), threadID)));
            }
            if (CurrentUser == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_PleaseLogin", applicationID));
                return RedirectToAction("ShowVoteThread", new RouteValueDictionary { { "threadID", threadID } });
            }
            bool isResult = Request.Form.GetBool("isResult", false);
            bool isAnonymousVote = Request.Form.GetBool("IsAnonymousVote", false);
            string optionIDStr = Request.Form.GetString("optionID", string.Empty);
            string[] strOptionIDs = optionIDStr.Split(',');
            IList<VoteRecord> voteRecords = new List<VoteRecord>();
            foreach (string strOptionID in strOptionIDs)
            {
                int optionID = 0;
                int.TryParse(strOptionID, out optionID);
                if (optionID > 0)
                {
                    VoteRecord record = new VoteRecord();
                    record.OptionID = optionID;
                    record.ThreadID = threadID;
                    record.UserID = CurrentUser.UserID;
                    record.UserDisplayName = CurrentUser.DisplayName;
                    record.IsAnonymousVote = isAnonymousVote;
                    voteRecords.Add(record);
                }
            }
            if (voteRecords.Count > 0 && voteRecords.Count <= voteThread.MaxChooseCount)
            {
                VoteThreads.Vote(CurrentUser.UserID, threadID, voteRecords);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_VoteSucecss", applicationID));
            }
            else
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_VoteFailPleaseSelectOptions", applicationID));
            }
            return RedirectToAction("ShowVoteThread", new RouteValueDictionary { { "threadID", threadID }, { "isResult", isResult } });
        }

        /// <summary>
        /// 更新投票
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateVoteThread(int threadID)
        {
            VoteThread voteThread = VoteThreads.GetThread(threadID, true);
            if (voteThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote", applicationID), threadID)));
            }
            string optionName = Request.Form.GetString("optionName", string.Empty);
            if (CurrentUser != null && (CurrentUser.UserID == voteThread.UserID || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
            {
                if (!string.IsNullOrEmpty(optionName))
                {
                    optionName = Formatter.FormatSingleLinePlainTextForStorage(optionName);
                    VoteOption option = new VoteOption();
                    option.OptionName = optionName;
                    option.ThreadID = voteThread.ThreadID;
                    VoteThreads.CreateVoteOption(option);
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_AddOptionsSucecss", applicationID));
                }
                else
                {

                    int privacyStatus = Request.Form.GetInt("privacyStatus", -1);
                    if (privacyStatus >= 0)
                    {
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdatePrivacySetSucecss", applicationID));
                        voteThread.PrivacyStatus = (PrivacyStatuses)privacyStatus;
                        VoteThreads.UpdateThread(voteThread);
                        return RedirectToAction("ShowVoteThread", new RouteValueDictionary { { "threadID", threadID } });
                    }

                    DateTime endDate = Request.Form.Get<DateTime>("endDate", voteThread.EndDate);
                    if (endDate > DateTime.Now && endDate != voteThread.EndDate)
                    {
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdateOntimeSucecss", applicationID));
                    }

                    voteThread.EndDate = endDate;

                    VoteThreads.UpdateThread(voteThread);
                }
            }
            else
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_FailNotThisLimit", applicationID));
            }
            return RedirectToAction("ShowVoteThread", new RouteValueDictionary { { "threadID", threadID } });
        }

        /// <summary>
        /// 删除投票
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteVoteThread(int threadID)
        {
            VoteThread voteThread = VoteThreads.GetThread(threadID, true);

            if (voteThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote", applicationID), threadID)));
            }
            if (CurrentUser != null && (CurrentUser.UserID == voteThread.UserID || CurrentUser.IsContentAdministrator || CurrentUser.IsAdministrator))
            {
                int userID = voteThread.UserID;
                VoteThreads.DeleteThread(threadID);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteVoteSucecss", applicationID));
                return Redirect(SiteUrls.Instance().UserChannelHome(userID, ApplicationIDs.Instance().Vote()));
            }
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_FailNotDeleteThisLimit", applicationID));
            return RedirectToAction("ShowVoteThread", new RouteValueDictionary { { "threadID", threadID } });

        }

        /// <summary>
        /// 邀请好友
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult InviteFriends()
        {
            NoticeItem noticeItem = NoticeSettingsManager.GetNoticeItem(NoticeItemKeys.Instance().NewVoteComment());
            if (noticeItem == null)
                throw new SPBException("邀请好友时出现异常！");

            int threadID = Request.Form.GetInt("threadID", 0);
            VoteThread voteThread = VoteThreads.GetThread(threadID, false);

            string strUserIDs = Request.Form.GetString("receiverIDs", string.Empty);
            if (string.IsNullOrEmpty(strUserIDs))
                throw new SPBException("邀请好友不存在！");
            string[] strUserIDsArray = strUserIDs.Split(',');
            if (strUserIDsArray != null && strUserIDsArray.Count() > 0)
            {

                foreach (string strUserID in strUserIDsArray)
                {
                    int toUserID = Convert.ToInt32(strUserID.Trim());
                    if (toUserID > 0)
                    {
                        User toUser = Users.GetUser(toUserID, false);
                        if (toUser != null)
                            continue;
                        NoticeRecord record = new NoticeRecord();
                        record.UserID = toUser.UserID;
                        record.Receiver = toUser.DisplayName;
                        record.ApplicationID = ApplicationIDs.Instance().Vote();
                        record.NoticeItemID = noticeItem.ItemID;
                        record.NoticeItemKey = noticeItem.ItemKey;
                        record.LeadingActorUserID = CurrentUser.UserID;
                        record.LeadingActor = CurrentUser.DisplayName;
                        record.LeadingActorUrl = SiteUrls.Instance().UserChannelHome(record.LeadingActorUserID, ApplicationIDs.Instance().Vote());
                        record.RelativeItemID = voteThread.ThreadID;
                        record.RelativeItemName = voteThread.Subject;
                        record.Subject = "你的好友邀请您参加投票！";
                        record.RelativeItemUrl = SiteUrls.Instance().ShowVoteThread(voteThread.ThreadID);
                        NoticeProcessorFactory.Instance().ProcessNotice(record);
                    }
                }
            }
            return RedirectToAction("Control_InviteFriends");
        }

        /// <summary>
        /// 更新投票审核状态
        /// </summary>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public ActionResult UpdateVoteAuditingStatuses(int threadID)
        {
            List<int> threadIds = new List<int>();
            threadIds.Add(threadID);
            VoteThreads.UpdateThreadAuditingStatus(threadIds, AuditingStatuses.Success);
            return RedirectToAction("ShowVoteThread", new RouteValueDictionary { { "threadID", threadID } });
        }

        #endregion

        #endregion

        #region VoteComments

        #region Control

        /// <summary>
        /// 投票评论
        /// </summary>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListVoteThreadComments(int threadID, int? pageIndex)
        {
            VoteThread thread = VoteThreads.GetThread(threadID, false);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(String.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote", applicationID), threadID)));
            }

            PagingDataSet<VoteComment> pds = new PagingDataSet<VoteComment>();
            if (CurrentUser != null && (thread.UserID == thread.UserID || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
            {
                pds = VoteComments.GetCommentsOfThread(threadID, pageIndex ?? 1, true);

            }
            else
            {
                pds = VoteComments.GetCommentsOfThread(threadID, pageIndex ?? 1, false);
            }

            ViewData["VoteThread"] = thread;
            return PartialView("Controls/Votes/ListVoteThreadComments.ascx", pds);
        }

        #endregion

        #region Operation

        /// <summary>
        /// 创建评论
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateComment(int threadID, int? pageIndex)
        {
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                if (verifyCode != null)
                    verifyCode = verifyCode.Trim();

                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    TempData["parentID"] = Request.Form.GetInt("parentID", 0);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
                    TempData["commentBody"] = Request.Form.GetString("commentBody", string.Empty).Trim();
                    return RedirectToAction("Control_ListVoteThreadComments", new RouteValueDictionary { { "threadID", threadID }, { "pageIndex", pageIndex } });
                }
            }
            VoteThread thread = VoteThreads.GetThread(threadID, false);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_VoteThemesCannotFind"), threadID)));
                //throw new SPBException(ExceptionType.AccessDenied);
            }

            VoteComment comment = new VoteComment();
            comment.Body = Request.Form.GetString("commentBody", string.Empty).Trim();
            comment.ThreadID = threadID;
            comment.Author = SPBHtmlHelper.Instance().UserDisplayName(CurrentUser);
            comment.UserID = CurrentUser != null ? CurrentUser.UserID : 0;
            comment.ParentID = Request.Form.GetInt("parentID", 0);
            comment.ThreadOwnerID = thread.UserID;
            comment.PresentAreaID = PresentAreaIDs.Instance().UserDomain();
            if (comment.ParentID > 0)
            {

                VoteComment parentComment = VoteComments.GetComment(comment.ParentID, false);
                if (parentComment != null)
                {
                    comment.ParentAuthor = parentComment.Author;
                    comment.ParentBody = parentComment.Body;
                    comment.ParentUserID = parentComment.UserID;
                    comment.ParentPostDate = parentComment.DateCreated;
                    if (comment.Body.StartsWith(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author), StringComparison.CurrentCultureIgnoreCase))
                    {
                        if (comment.Body.Length > string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author).Length)
                        {
                            comment.Body = comment.Body.Substring(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author).Length);
                        }
                        else
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = GetResourceString("Message_PleaseInputReplyWord", applicationID);
                            TempData["commentBody"] = comment.Body;
                            TempData["parentID"] = comment.ParentID;
                            return RedirectToAction("Control_ListVoteThreadComments", new RouteValueDictionary { { "threadID", threadID }, { "pageIndex", pageIndex } });
                        }
                    }
                }
            }

            try
            {
                #region 检查发帖时间间隔,防灌水
                Globals.CheckPostDate();
                #endregion
                VoteComments.CreateComment(comment);
            }
            catch (SPBException e)
            {
                ExceptionMessage exceptionMessage = ResourceManager.GetExceptionMessage(e.ExceptionType, ApplicationIDs.Instance().Vote());
                if (exceptionMessage != null)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = exceptionMessage.Title + "，" + exceptionMessage.Body;
                }
                return RedirectToAction("Control_ListVoteThreadComments", new RouteValueDictionary { { "threadID", threadID }, { "pageIndex", pageIndex } });
            }

            if ((int)comment.AuditingStatus >= (int)SiteSettingsManager.GetSiteSettings().AuditingStatusesForPublicDisplay)
            {
                TempData["StatusMessageContent"] = GetResourceString("Message_YouAlreadyAddAinding", applicationID);
            }
            else
            {
                TempData["StatusMessageContent"] = GetResourceString("Message_AddReviewSucecss", applicationID);
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;

            return RedirectToAction("Control_ListVoteThreadComments", new RouteValueDictionary { { "threadID", threadID }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 删除评论
        /// </summary>
        /// <param name="commentID"></param>
        /// <returns></returns>
        public ActionResult DeleteComment(int commentID)
        {

            VoteComment comment = VoteComments.GetComment(commentID, true);

            if (comment == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_CannotIDReview", applicationID), commentID)));
            }
            VoteThread thread = VoteThreads.GetThread(comment.ThreadID, false);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote", applicationID), comment.ThreadID)));
            }

            if (CurrentUser != null && (CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator || comment.UserID == CurrentUser.UserID))
            {
                VoteComments.DeleteComment(commentID);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSucecssful", applicationID);
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_DeleteFail", applicationID);
            }
            return RedirectToAction("Control_ListVoteThreadComments", new RouteValueDictionary { { "threadID", comment.ThreadID } });

        }

        #endregion

        #endregion

        #region Help Methods

        /// <summary>
        /// 设置投票详细显示页面的meta标签
        /// </summary>
        private void SetPageMeta(VoteThread thread)
        {
            string metaDescription = string.Empty;
            //标题
            metaDescription = string.Format("{0},{1}", thread.Subject, HtmlUtils.TrimHtml(thread.Body, 150));

            Header.AddMetaDescription(metaDescription);
            Header.AddMetaKeywords(thread.Subject);
        }

        #endregion

    }
    /// <summary>
    /// SubMenu
    /// </summary>
    public enum VoteChannelSubMenu
    {
        //投票首页
        voteHome = 1,
        //投票排行
        voteSortby = 2
    }

    /// <summary>
    /// BlockStyle
    /// </summary>
    public enum ListBlockStyle
    {
        //详细
        Detailed,
        //简洁
        Simple
    }
}