﻿//------------------------------------------------------------------------------
//<copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights reserved.
//</copyright> 
//------------------------------------------------------------------------------ 

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using SpaceBuilder.Common;
using SpaceBuilder.Web;
using SpaceBuilder.Club;
using System.Web.Routing;
using SpaceBuilder.Utils;
using System.Web.UI;


namespace SpaceBuilder.Vote.Controllers
{
    /// <summary>
    /// 群组中有关投票的Controller
    /// </summary>
    [HttpCompress]
    public class ClubVoteController : ClubDomainControllerBase
    {

        int applicationID = ApplicationIDs.Instance().Vote();

        #region Vote

        #region View

        /// <summary>
        /// 显示群组投票首页
        /// </summary>
        public ActionResult ListClubVoteThreads(string clubDomainName)
        {
            SpaceBuilder.Club.Club currentClub = GetCurrentClub(clubDomainName);

            if (currentClub == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));

            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ViewClub(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Vote", applicationID), string.Empty);
            return View("Pages/Votes/ListClubVoteThreads.aspx", "Masters/Vote.master");
        }

        /// <summary>
        /// 发起投票
        /// </summary>
        public ActionResult NewVoteThread(string clubDomainName)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));

            Club.Club currentClub = GetCurrentClub(clubDomainName);


            if (CurrentUser != null && !ClubPermission.Validate(clubDomainName, CurrentUser, PermissionItemKeys.Instance().CreateClubVoteThread()))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_SorryNotVateLimit", applicationID));
                return RedirectToAction("ListClubVoteThreads", new RouteValueDictionary { { "clubDomainName", clubDomainName } });
            }
            SetPageTitle(GetResourceString("Title_DoVote", applicationID), clubDomainName);
            ViewData["MaxVoteOptionCount"] = VoteConfiguration.Instance().MaxVoteOptionCount;

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_DoVote", applicationID), "");
            VoteThread votethread = new VoteThread();
            votethread.PrivacyStatus = PrivacyStatuses.Public;
            votethread.EnableComments = true;
            if (TempData["VoteThread"] != null)
                votethread = TempData["VoteThread"] as VoteThread;

            return View("Pages/Votes/NewVoteThread.aspx", "Masters/Vote.Master", votethread);
        }

        /// <summary>
        /// 显示群组投票详细页
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="isResult"></param>
        /// <returns></returns>
        [ClubMemberVisitAndClubDomainHitTimeAttribute]
        public ActionResult ShowClubVoteThread(string clubDomainName, int threadID, bool? isResult)
        {
            Club.Club club = Clubs.GetClub(clubDomainName, false);
            if (club == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", ApplicationIDs.Instance().Club()), clubDomainName)));

            if (!club.IsPublic && (CurrentUser == null || !ClubMembers.IsClubMember(club.ClubID, CurrentUser.UserID)))
                return Redirect(SiteUrls.Instance().Error404("没有浏览当前群组的权限！"));

            if (!ClubPermission.AccessCheck(club, CurrentUser, PermissionItemKeys.Instance().ViewClub(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            VoteThread thread = VoteThreads.GetThread(threadID, true);
            ViewData["clubDomainName"] = clubDomainName;
            ViewData["club"] = club;

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_ShowVote", applicationID), "");

            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote", applicationID), threadID)));
            }
            if (club == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_CallMinBlog", applicationID))));
            }
            if (CurrentUser == null || (ClubMembers.HasClubManagerPermission(club.ClubID, CurrentUser.UserID) && !CurrentUser.IsAdministrator && !CurrentUser.IsContentAdministrator))
            {
                switch (thread.PrivacyStatus)
                {
                    //仅群组成员可见
                    case PrivacyStatuses.OnlyFriend:
                        bool isMember = ClubMembers.IsClubMember(thread.OwnerID, CurrentUser == null ? 0 : CurrentUser.UserID);
                        if (!isMember)
                        {
                            ViewData["isShow"] = false;
                            return View("Pages/Votes/ShowClubVoteThread.aspx", "Masters/Vote.Master", thread);
                        }
                        break;
                    //仅群组管理员可见
                    case PrivacyStatuses.Privacy:
                        ViewData["isShow"] = false;
                        return View("Pages/Votes/ShowClubVoteThread.aspx", "Masters/Vote.Master", thread);
                    case PrivacyStatuses.Public:
                        break;
                    default:
                        break;
                }
            }

            isResult = isResult.HasValue ? isResult.Value : false;
            if (thread.VoteStaus == VoteStatuses.Expired || CurrentUser == null)
                isResult = true;
            bool isVote = VoteThreads.IsVoted(threadID, CurrentUser == null ? 0 : CurrentUser.UserID);
            ViewData["isVote"] = isVote;
            if (isVote)
                isResult = true;
            ViewData["isResult"] = isResult;
            #region 浏览器标题、计数、meta标签
            SetPageTitle(thread.Subject);
            CounterFactory.IncreaseCount<VoteThreadCounter>(IncreasableCountTypes.WebView, thread.ThreadID);
            SetPageMeta(thread);
            #endregion
            return View("Pages/Votes/ShowClubVoteThread.aspx", "Masters/Vote.Master", thread);
        }

        #endregion

        #region Controls

        /// <summary>
        /// 群组投票列表
        /// </summary>
        /// <param name="clubDomainName"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public ActionResult Control_ListClubVotethreads(string clubDomainName, int? pageIndex)
        {
            Club.Club currentClub = GetCurrentClub(clubDomainName);

            PagingDataSet<VoteThread> pds = VoteThreads.GetOwnerThreads(PresentAreaIDs.Instance().ClubDomain(), currentClub.ClubID, pageIndex ?? 1, true);

            return PartialView("Controls/Votes/ListClubVoteThreads.ascx", pds);
        }

        /// <summary>
        /// 群组投票列表快
        /// </summary>
        /// <param name="sortBy"></param>
        /// <param name="modelTitle"></param>
        /// <returns></returns>
        public ActionResult Control_VotesSideBlock(string clubDomainName, VoteThreadSortBy? sortBy, int? presentAreaID, string modelTitle, int? disPlayCount)
        {
            ViewData["sortBy"] = sortBy;
            ViewData["Title"] = modelTitle;

            Club.Club currentClub = Clubs.GetClub(clubDomainName, false);
            //todo:需要换方法

            int? ownerID = currentClub.ClubID;
            if (presentAreaID.HasValue && presentAreaID.Value > 0)
                ownerID = null;
            IList<VoteThread> therads = VoteThreads.GetTopThreadsSortBy(presentAreaID ?? PresentAreaIDs.Instance().ClubDomain(), ownerID, sortBy ?? VoteThreadSortBy.DateCreated, null, disPlayCount ?? 7);
            return PartialView("Controls/Votes/VotesSideBlock.ascx", therads);
        }

        /// <summary>
        /// 在群组首页显示投票
        /// </summary>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public ActionResult Control_ShowVoteThreadBlock(string clubDomainName, int threadID)
        {

            bool isVote = CurrentUser != null && VoteThreads.IsVoted(threadID, CurrentUser.UserID);
            ViewData["isVote"] = isVote;
            VoteThread thread = VoteThreads.GetThread(threadID, false);
            return PartialView("Controls/Votes/ShowVoteThreadBlock.ascx", thread);
        }

        /// <summary>
        /// 显示投票结果
        /// </summary>
        public ActionResult Control_ListOptionVoters(string clubDomainName, int optionID, int? pageSize, int? pageIndex)
        {
            PagingDataSet<VoteRecord> pds = VoteThreads.GetVoteRecordsOfOption(optionID, pageSize ?? 9, pageIndex ?? 1);
            ViewData["optionID"] = optionID;
            ViewData["pageSize"] = pageSize;
            return PartialView("Controls/Votes/ListOptionVoters.ascx", pds);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="clubDomainName"></param>
        /// <param name="displayCount"></param>
        /// <returns></returns>
        public ActionResult Control_ListVoteThreadsBlock(string clubDomainName, int? displayCount, bool hasBorder, ListBlockStyle? style)
        {
            Club.Club currentClubDomain = Clubs.GetClub(clubDomainName, false);
            if (currentClubDomain == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote", applicationID), clubDomainName)));
            }
            ViewData["Title"] = GetResourceString("Title_NewVote", applicationID);
            ViewData["Style"] = style ?? ListBlockStyle.Simple;
            ViewData["hasBorder"] = hasBorder;
            AuditingStatusesForDisplay? auditingStatusesForDisplay = Applications.GetApplication(ApplicationIDs.Instance().Vote()).AuditingStatusForPublicDisplay;
            IList<VoteThread> threads = VoteThreads.GetTopOwnerThreads(PresentAreaIDs.Instance().ClubDomain(), currentClubDomain.ClubID, displayCount ?? 10);
            return PartialView("Controls/Votes/ListClubVotesBlock.ascx", threads);
        }

        /// <summary>
        /// 更新截止日期
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_UpdateEndDate(string clubDomainName, int threadID)
        {
            ViewData["threadID"] = threadID;
            return PartialView("Controls/Votes/UpdateEndDate.ascx");
        }

        #endregion

        #region Operation

        /// <summary>
        /// 创建群组投票
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateClubVoteThread(string clubDomainName)
        {
            User currentUser = Globals.GetCurrentUser();
            Club.Club currentClub = GetCurrentClub(clubDomainName);
            bool isError = false;

            if (currentUser != null && ClubMembers.IsClubMember(currentClub.ClubID, currentUser.UserID) || currentUser.IsAdministrator || currentUser.IsContentAdministrator)
            {
                VoteThread thread = new VoteThread();
                thread.UserID = currentUser.UserID;
                thread.Subject = Request.Form.GetString("subject", string.Empty);
                if (string.IsNullOrEmpty(thread.Subject))
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("xception_VoteTitleCannotNull", applicationID));
                    isError = true;
                }
                thread.Body = Request.Form.GetString("body", string.Empty);
                if (thread.Body.Length > 160)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Exception_VoteMaxWord", applicationID), 160));

                    isError = true;
                }

                thread.EnableComments = Request.Form.GetBool("enableComments", false);
                thread.EndDate = Request.Form.Get<DateTime>("expireTime", DateTime.Now.AddMonths(1));
                if (thread.EndDate < DateTime.Now)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_EndTimeCannotLessThanCurrentTime", applicationID));
                    isError = true;
                }
                //设置隐私状态
                thread.PrivacyStatus = (PrivacyStatuses)Request.Form.GetInt("privacyStatus", (int)PrivacyStatuses.Public);
                thread.Author = currentUser.DisplayName;
                //todo:presentAreaID
                thread.PresentAreaID = PresentAreaIDs.Instance().ClubDomain();
                thread.Owner = currentClub.ClubName;
                thread.OwnerID = currentClub.ClubID;
                thread.MaxChooseCount = Request.Form.GetInt("maxChooseCount", 1);
                if (thread.MaxChooseCount == 1)
                    thread.IsMultiple = false;
                else
                    thread.IsMultiple = true;

                int totalOptionCount = Request.Form.GetInt("totalOptionCount", 5);

                for (int i = 1; i <= totalOptionCount; i++)
                {
                    string optionName = Request.Form.GetString("voteOption_" + i, string.Empty);
                    int optionID = Request.Form.GetInt("OptionID_" + i, 0);

                    VoteOption voteOption = new VoteOption();
                    voteOption.OptionName = optionName;
                    if (optionID > 0)
                        voteOption.OptionID = optionID;

                    thread.VoteOptions[i] = voteOption;
                }


                if (thread.VoteOptions.Where(n => !String.IsNullOrEmpty(n.Value.OptionName)).Count() < 2)
                {
                    isError = true;
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "投票项不能少于两项！");
                }

                if (isError)
                {
                    TempData["VoteThread"] = thread;
                    return RedirectToAction("NewVoteThread", new RouteValueDictionary { { "clubDomainName", clubDomainName } });
                }

                VoteThreads.CreateThread(thread);
                if (thread.ThreadID < 0)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_CreatFail", applicationID));
                    return RedirectToAction("NewVoteThread", new RouteValueDictionary { { "clubDomainName", clubDomainName } });
                }

                Dictionary<int, VoteOption> voteOptions = new Dictionary<int, VoteOption>(thread.VoteOptions);

                thread.VoteOptions.Clear();
                foreach (KeyValuePair<int, VoteOption> pair in voteOptions)
                {
                    if (!string.IsNullOrEmpty(pair.Value.OptionName))
                    {
                        pair.Value.OptionName = Formatter.FormatSingleLinePlainTextForStorage(pair.Value.OptionName);
                        pair.Value.ThreadID = thread.ThreadID;

                        if (pair.Value.OptionID > 0)
                        {
                            VoteThreads.UpdateVoteOption(pair.Value);
                        }
                        else
                        {
                            VoteThreads.CreateVoteOption(pair.Value);
                        }
                    }
                }

                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_VoteSucecss", applicationID));
                return RedirectToAction("ShowClubVoteThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "threadID", thread.ThreadID } });
            }
            else
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NotMinBlogMe", applicationID), clubDomainName)));
            }
        }

        /// <summary>
        /// 投票
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Vote(int threadID, string clubDomainName)
        {
            VoteThread thread = VoteThreads.GetThread(threadID, false);
            User currnetUser = Globals.GetCurrentUser();
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote"), threadID)));
            }
            if (currnetUser == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_PleaceLogin"));
                return RedirectToAction("ShowClubVoteThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "ThreadID", threadID } });
            }
            bool isAnonymousVote = Request.Form.GetBool("IsAnonymousVote", false);
            string optionIDStr = Request.Form.GetString("optionID", string.Empty);
            string[] strOptionIDs = optionIDStr.Split(',');
            IList<VoteRecord> voteRecords = new List<VoteRecord>();
            foreach (string strOptionID in strOptionIDs)
            {
                int optionID = 0;
                int.TryParse(strOptionID, out optionID);
                if (optionID > 0)
                {
                    VoteRecord record = new VoteRecord();
                    record.OptionID = optionID;
                    record.ThreadID = threadID;
                    record.UserID = currnetUser.UserID;
                    record.UserDisplayName = CurrentUser.DisplayName;
                    record.IsAnonymousVote = isAnonymousVote;
                    voteRecords.Add(record);
                }
            }
            if (voteRecords.Count > 0 && voteRecords.Count <= thread.MaxChooseCount)
            {
                VoteThreads.Vote(currnetUser.UserID, threadID, voteRecords);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_VoteSucecss", applicationID));
            }
            else
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_VoteFail", applicationID));
            }
            return RedirectToAction("ShowClubVoteThread", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "ThreadID", threadID } });
        }

        public ActionResult DeleteVote(int threadID, int userID, int clubID, string returnUrl)
        {

            VoteThread thread = VoteThreads.GetThread(threadID, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote", applicationID), threadID)));
            }
            Club.Club currentClub = Clubs.GetClub(clubID, false);
            User currentUser = Globals.GetCurrentUser();
            if (CurrentUser != null && (currentUser.IsAdministrator || currentUser.UserID == currentClub.ClubOwnerUserID || currentUser.UserID == thread.UserID))
            {
                VoteThreads.DeleteThread(threadID);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteSucecssful", applicationID));
                if (returnUrl != null)
                {
                    return Redirect(returnUrl);
                }
                else
                {
                    return RedirectToAction("ListClubVoteThreads", new RouteValueDictionary { { "clubDomainName", currentClub.DomainName } });
                }
            }
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_DeleteFail", applicationID));

            if (returnUrl != null)
            {
                return Redirect(returnUrl);
            }
            else
            {
                return RedirectToAction("ShowClubVoteThread", new RouteValueDictionary { { "clubDomainName", currentClub.DomainName }, { "threadID", thread.ThreadID } });
            }

        }

        /// <summary>
        /// 更新投票
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateVoteThread(int threadID, string clubDomainName)
        {

            Club.Club currentClub = GetCurrentClub(clubDomainName);
            VoteThread voteThread = VoteThreads.GetThread(threadID, true);
            if (voteThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote", applicationID), threadID)));
            }

            if (CurrentUser != null && (currentClub.ClubOwnerUserID == CurrentUser.UserID || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator || ClubPermission.Validate(clubDomainName, CurrentUser, PermissionItemKeys.Instance().CreateClubVoteThread())))
            {
                string optionName = Request.Form.GetString("optionName", string.Empty);
                if (!string.IsNullOrEmpty(optionName))
                {
                    optionName = Formatter.FormatSingleLinePlainTextForStorage(optionName);
                    VoteOption option = new VoteOption();
                    option.OptionName = optionName;
                    option.ThreadID = voteThread.ThreadID;
                    VoteThreads.CreateVoteOption(option);
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_AddOptionsSucecss", applicationID));
                }
                else
                {

                    PrivacyStatuses privacyStatus = (PrivacyStatuses)Request.Form.GetInt("privacyStatus", (int)voteThread.PrivacyStatus);
                    if (privacyStatus != voteThread.PrivacyStatus)
                    {
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdatePrivacySetSucecss", applicationID));
                        voteThread.PrivacyStatus = privacyStatus;
                        VoteThreads.UpdateThread(voteThread);
                    }
                    DateTime endDate = Request.Form.Get<DateTime>("endDate", voteThread.EndDate);

                    if (endDate != voteThread.EndDate && endDate > voteThread.EndDate)
                    {
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdateOntimeSucecss", applicationID));
                    }
                    voteThread.EndDate = endDate;

                    VoteThreads.UpdateThread(voteThread);
                }
            }
            else
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_SorryYouNotThisLimit", applicationID));
            }
            return RedirectToAction("ShowClubVoteThread", new RouteValueDictionary { { "threadID", threadID }, { "clubDomainName", clubDomainName } });
        }

        #endregion

        #endregion

        #region Comment

        #region View

        /// <summary>
        /// 管理评论
        /// </summary>
        /// <param name="clubDomainName"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public ActionResult ManageComments(string clubDomainName, int? pageIndex)
        {

            Club.Club currnetClub = GetCurrentClub(clubDomainName);
            if (currnetClub == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_CannotFindMinBlog", applicationID), currnetClub.ClubID)));

            SetPageTitle("管理投票评论", clubDomainName);

            PagingDataSet<VoteComment> pds = VoteComments.GetComments(currnetClub.ClubID, pageIndex ?? 1);
            BreadCrumb.AddBreadCrumbItem(currnetClub.ClubName, SiteUrls.Instance().ClubDomainHome(clubDomainName));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Vote", applicationID), SPBUrlHelper.Action("ListClubVoteThreads", "ClubVote", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_MangerReview", applicationID), "");
            return View("Pages/Votes/ManageComments.aspx", "Masters/Manage.master", pds);
        }

        #endregion

        #region Control

        /// <summary>
        /// 群组投票评论列表
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListVoteThreadComments(string clubDomainName, int threadID, int? pageIndex)
        {
            VoteThread thread = VoteThreads.GetThread(threadID, false);
            ViewData["VoteThread"] = thread;
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote", applicationID), threadID)));
            }
            PagingDataSet<VoteComment> pds = new PagingDataSet<VoteComment>();
            if (CurrentUser != null && (CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
            {
                pds = VoteComments.GetCommentsOfThread(threadID, pageIndex ?? 1, true);
            }
            else
            {
                pds = VoteComments.GetCommentsOfThread(threadID, pageIndex ?? 1, true);
            }

            return PartialView("Controls/Votes/ListVoteThreadComments.ascx", pds);
        }



        #endregion

        #region Operation

        /// <summary>
        /// 创建评论
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateComment(string clubDomainName, int threadID, int? pageIndex)
        {
            Club.Club currentClub = GetCurrentClub(clubDomainName);
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                if (verifyCode != null)
                    verifyCode = verifyCode.Trim();

                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
                    TempData["commentBody"] = Request.Form.GetString("commentBody", string.Empty).Trim();
                    return RedirectToAction("Control_ListVoteThreadComments", new RouteValueDictionary { { "threadID", threadID }, { "pageIndex", pageIndex } });
                }
            }
            VoteThread thread = VoteThreads.GetThread(threadID, false);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_VoteThemesCannotFind", applicationID), threadID)));
                //throw new SPBException(ExceptionType.AccessDenied);
            }

            VoteComment comment = new VoteComment();
            comment.Body = Request.Form.GetString("commentBody", string.Empty).Trim();
            comment.ThreadID = threadID;
            comment.Author = CurrentUser == null ? "匿名用户" : CurrentUser.DisplayName;
            comment.UserID = CurrentUser == null ? 0 : CurrentUser.UserID;
            comment.ParentID = Request.Form.GetInt("parentID", 0);
            comment.ThreadOwnerID = currentClub.ClubID;
            comment.PresentAreaID = PresentAreaIDs.Instance().ClubDomain();
            if (comment.ParentID > 0)
            {

                VoteComment parentComment = VoteComments.GetComment(comment.ParentID, false);
                if (parentComment != null)
                {
                    comment.ParentAuthor = parentComment.Author;
                    comment.ParentBody = parentComment.Body;
                    comment.ParentUserID = parentComment.UserID;
                    comment.ParentPostDate = parentComment.DateCreated;
                    if (comment.Body.StartsWith(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author), StringComparison.CurrentCultureIgnoreCase))
                    {
                        if (comment.Body.Length > string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author).Length)
                        {
                            comment.Body = comment.Body.Substring(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author).Length);
                        }
                        else
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = GetResourceString("Message_PleaseInputReplyWord", applicationID);
                            TempData["commentBody"] = comment.Body;
                            TempData["parentID"] = comment.ParentID;
                            return RedirectToAction("Control_ListVoteThreadComments", new RouteValueDictionary { { "threadID", threadID }, { "pageIndex", pageIndex } });
                        }
                    }
                }
            }
            try
            {
                #region 检查发帖时间间隔,防灌水
                Globals.CheckPostDate();
                #endregion
                VoteComments.CreateComment(comment);
            }
            catch (SPBException e)
            {
                ExceptionMessage exceptionMessage = ResourceManager.GetExceptionMessage(e.ExceptionType, ApplicationIDs.Instance().Club());
                if (exceptionMessage != null)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = exceptionMessage.Title + "，" + exceptionMessage.Body;
                }
                return RedirectToAction("Control_ListVoteThreadComments", new RouteValueDictionary { { "threadID", threadID }, { "pageIndex", pageIndex } });
            }

            if ((int)comment.AuditingStatus >= (int)SiteSettingsManager.GetSiteSettings().AuditingStatusesForPublicDisplay)
            {
                TempData["StatusMessageContent"] = GetResourceString("Message_YouAlreadyAddAinding", applicationID);
            }
            else
            {
                TempData["StatusMessageContent"] = GetResourceString("Message_AddReviewSucecss", applicationID);
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;

            return RedirectToAction("Control_ListVoteThreadComments", new RouteValueDictionary { { "threadID", threadID }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 删除评论
        /// </summary>
        /// <param name="commentID"></param>
        /// <returns></returns>
        public ActionResult DeleteComment(int commentID)
        {
            VoteComment comment = VoteComments.GetComment(commentID, true);
            if (comment == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_CannotIDReview", applicationID), commentID)));
            }
            VoteThread thread = VoteThreads.GetThread(comment.ThreadID, false);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote", applicationID), comment.ThreadID)));
            }
            if (CurrentUser != null && (comment.UserID == CurrentUser.UserID || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
            {
                VoteComments.DeleteComment(commentID);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSucecssful", applicationID);
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_DeleteFailYouNotThisOptionsLimits", applicationID);
            }
            return RedirectToAction("Control_ListVoteThreadComments", new RouteValueDictionary { { "threadID", comment.ThreadID } });
        }

        /// <summary>
        /// 管理批量删除评论
        /// </summary>
        /// <param name="returnUrl"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteComments(string clubDomainName, string returnUrl)
        {
            Club.Club currentClub = Clubs.GetClub(clubDomainName, false);
            if (currentClub == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_CallMinBlog", applicationID)));

            }
            if (CurrentUser != null && (currentClub.ClubOwnerUserID == CurrentUser.UserID || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
            {
                string selectIDsString = Request.Form["ManageCommentsCheckBoxGroup"];
                int commentID = 0;
                if (!string.IsNullOrEmpty(selectIDsString))
                {
                    string[] selectIDsStringArray = selectIDsString.Split(',');
                    if (selectIDsStringArray != null && selectIDsStringArray.Length > 0)
                    {
                        foreach (string idString in selectIDsStringArray)
                        {
                            commentID = int.Parse(idString);
                            VoteComments.DeleteComment(commentID);
                        }
                        TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                        TempData["StatusMessageContent"] = GetResourceString("Message_ReviewDeleteSucecss", applicationID);
                    }
                    else
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        TempData["StatusMessageContent"] = GetResourceString("Message_DeleteFailYouNotThisOptionsLimits", applicationID);
                    }
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_PleaseSelectDeleteReview", applicationID);
                }
                return Redirect(returnUrl);
            }

            return Redirect(returnUrl);
        }

        /// <summary>
        /// 管理删除评论
        /// </summary>
        /// <param name="commentID"></param>
        /// <param name="returnUrl"></param>
        /// <returns></returns>
        public ActionResult DeleteCommentInManage(int commentID, string returnUrl)
        {
            VoteComment comment = new VoteComment();
            comment = VoteComments.GetComment(commentID, true);
            if (comment == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_CannotIDReview", applicationID), commentID)));
            }

            if (CurrentUser != null && (comment.UserID == CurrentUser.UserID || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
            {
                VoteComments.DeleteComment(commentID);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_ReviewDeleteSucecss", applicationID);
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_DeleteFailConcoatAdmin", applicationID);
            }
            return Redirect(returnUrl);
        }

        #endregion

        #endregion

        #region Help Method

        /// <summary>
        /// 设置活动详细显示页面的meta标签
        /// </summary>
        private void SetPageMeta(VoteThread thread)
        {
            string metaKeyWords = string.Empty;
            string metaDescription = string.Empty;
            //标题
            metaDescription = string.Format("{0},{1}", thread.Subject, HtmlUtils.TrimHtml(thread.Body, 150));

            Header.AddMetaDescription(metaDescription);
            Header.AddMetaKeywords(thread.Subject);
        }

        #endregion

    }
}





