﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using SpaceBuilder.Common;
using SpaceBuilder.Club;
using SpaceBuilder.Web;
using System.Web.Routing;

namespace SpaceBuilder.Vote.Controllers
{
    /// <summary>
    /// 后台管理投票的Controller
    /// </summary>
    [ManageAuthorize]
    [HttpCompress]
    public class ManageVoteController : ManageControllerBase
    {

        int applicationID = ApplicationIDs.Instance().Vote();

        #region Vote

        #region View

        /// <summary>
        /// 用户投票管理
        /// </summary>
        public ActionResult ManageVoteThreads(int? presentAreaID, AuditingStatusesForDisplay? auditingStatusesForDisplay, int? pageIndex, int? ownerID, string subjectKeyWords)
        {
            SetPageTitle(GetResourceString("Title_MangerVote", applicationID));
            auditingStatusesForDisplay = auditingStatusesForDisplay ?? AuditingStatusesForDisplay.Pending;
            ViewData["AuditingStatusForDisplay"] = auditingStatusesForDisplay;
            ViewData["presentAreaID"] = presentAreaID;
            PagingDataSet<VoteThread> pds = VoteThreads.GetThreadsForAdmin(presentAreaID ?? PresentAreaIDs.Instance().UserDomain(), ownerID ?? 0, null, subjectKeyWords, auditingStatusesForDisplay, 50, pageIndex ?? 1);
            return View("Pages/Votes/ManageVoteThreads.aspx", "Masters/Vote.master", pds);
        }

        #endregion

        #region Operation

        /// <summary>
        /// 搜索投票
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SearchVotes(int? presentAreaID, AuditingStatusesForDisplay auditingStautsesForDisplay)
        {
            string subjectKeyWords = Request.Form.GetString("subjectKeywords", string.Empty).Trim();
            int UserID = Request.Form.GetInt("UserID", 0);
            return RedirectToAction("ManageVoteThreads", new RouteValueDictionary { { "presentAreaID", presentAreaID }, { "auditingStatusesForDisplay", auditingStautsesForDisplay }, { "subjectKeywords", subjectKeyWords }, { "ownerID", UserID } });
        }

        /// <summary>
        /// 删除单个投票
        /// </summary>
        /// <param name="threadID">投票ID</param>
        /// <param name="returnUrl"></param>
        /// <returns></returns>
        public ActionResult DeleteVote(int threadID, string returnUrl)
        {
            VoteThread thread = VoteThreads.GetThread(threadID, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_SorryCannotFindVote", applicationID), threadID)));
            }

            VoteThreads.DeleteThread(threadID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_DeleteVoteSucecss", applicationID);

            return Redirect(returnUrl);
        }

        /// <summary>
        /// 批量删除投票
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteVotes(string returnUrl)
        {
            string selectIDsString = Request.Form["ManageVotesCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectIDsString))
            {
                string[] selectIDsStringArray = selectIDsString.Split(',');
                if (selectIDsStringArray != null && selectIDsStringArray.Length > 0)
                {
                    foreach (string idStrings in selectIDsStringArray)
                    {
                        string[] idString = idStrings.Split(':');
                        int threadID = int.Parse(idString[0]);
                        if (threadID > 0)
                            VoteThreads.DeleteThread(threadID);
                    }
                    TempData["StatusMessageType"] = StatusMessageType.Success;
                    TempData["StatusMessageContent"] = GetResourceString("Message_AllDeleteSucecss", applicationID);
                }
                else
                {
                    TempData["StatusMessageType"] = StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_AllDeleteFailConcoatAdmin", applicationID);
                }
            }
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 批量设置审核状态
        /// </summary>
        /// <param name="isPass"></param>
        /// <param name="ownerTypeName"></param>
        /// <param name="returnUrl"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SetVoteAuditingStatus(bool isPass, string returnUrl)
        {
            string selectIDsString = Request.Form["ManageVotesCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectIDsString))
            {
                string[] selectIDsStringArray = selectIDsString.Split(',');
                if (selectIDsStringArray != null && selectIDsStringArray.Length > 0)
                {
                    List<int> threadIDs = new List<int>();
                    foreach (string idStrings in selectIDsStringArray)
                    {
                        string[] idString = idStrings.Split(':');
                        int threadID = int.Parse(idString[0]);
                        if (threadID > 0)
                        {
                            threadIDs.Add(threadID);
                        }
                    }
                    if (isPass)
                    {
                        VoteThreads.UpdateThreadAuditingStatus(threadIDs, AuditingStatuses.Success);
                        TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    }
                    else
                    {
                        VoteThreads.UpdateThreadAuditingStatus(threadIDs, AuditingStatuses.Fail);
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    }
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = GetResourceString("Message_SetSucecss", applicationID);
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_SetFail", applicationID);

                }
            }
            return Redirect(returnUrl);
        }

        #endregion

        #endregion

        #region VoteComment

        #region View

        /// <summary>
        /// 
        /// </summary>
        /// <param name="auditingStatusesForDisplay"></param>
        /// <param name="pageIndex"></param>
        /// <param name="ownerID"></param>
        /// <param name="ownerTypeName"></param>
        /// <param name="bodyKeyWords"></param>
        /// <returns></returns>
        public ActionResult ManageVoteComments(int? presentAreaID, int? userID, AuditingStatusesForDisplay? auditingStatusesForDisplay, int? pageIndex, int? ownerID, string bodyKeyWords)
        {
            SetPageTitle(GetResourceString("Title_MangerVote", applicationID));
            auditingStatusesForDisplay = auditingStatusesForDisplay ?? AuditingStatusesForDisplay.Pending;
            ViewData["AuditingStatusForDisplay"] = auditingStatusesForDisplay;

            ViewData["presentAreaID"] = presentAreaID;
            PagingDataSet<VoteComment> pds = VoteComments.GetCommentsForAdmin(presentAreaID ?? PresentAreaIDs.Instance().UserDomain(), ownerID ?? 0, userID ?? 0, bodyKeyWords, auditingStatusesForDisplay, 50, pageIndex ?? 1);

            return View("Pages/Votes/ManageVoteComment.aspx", "Masters/Vote.master", pds);
        }

        #endregion

        #region Operation

        /// <summary>
        /// 搜索投票评论
        /// </summary>
        /// <param name="auditingStatusForDisplay"></param>
        /// <param name="ownerTypeName">判断是群组还是用户</param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SearchComments(AuditingStatusesForDisplay auditingStatusForDisplay, int presentAreaID)
        {
            string bodyKeyWords = Request.Form.GetString("bodyKeywords", string.Empty);
            int userID = Request.Form.GetInt("UserID", 0);
            return RedirectToAction("ManageVoteComments", new RouteValueDictionary { { "auditingStatusesForDisplay", auditingStatusForDisplay }, { "presentAreaID", presentAreaID }, { "bodyKeyWords", bodyKeyWords }, { "ownerID", userID } });
        }

        /// <summary>
        /// 删除评论
        /// </summary>
        /// <param name="commentID"></param>
        /// <param name="ownerTypeName">判断是群组还是用户</param>
        /// <param name="returnUrl"></param>
        /// <returns></returns>
        public ActionResult DeleteComment(int commentID, string returnUrl)
        {

            VoteComments.DeleteComment(commentID);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_ReviewDeleteSucecss", applicationID);
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 批量删除评论
        /// </summary>
        /// <param name="ownerTypeName">判断是群组还是用户</param>
        /// <param name="returnUrl"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteComments(string returnUrl)
        {
            string[] deleteCommentIDs = Request.Form.GetString("CheckBoxGroup", string.Empty).Split(',');
            if (deleteCommentIDs != null && deleteCommentIDs.Length > 0)
            {
                int commentID = 0;
                List<int> commentIDs = new List<int>();
                foreach (string strCommentID in deleteCommentIDs)
                {
                    int.TryParse(strCommentID, out commentID);
                    commentIDs.Add(commentID);

                }
                VoteComments.DeleteComments(commentIDs);
                TempData["StatusMessageType"] = StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_AllDeleteSucecss");
            }
            else
            {
                TempData["StatusMessageType"] = StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_AllDeleteFailConcoatAdmin", applicationID);
            }
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 设置投票审核状态
        /// </summary>
        /// <param name="auditingStatuses"></param>
        /// <param name="ownerTypeName"></param>
        /// <param name="returnUrl"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SetCommentAuditingStatus(AuditingStatuses? auditingStatuses, string returnUrl)
        {
            string[] StrCommentIDs = Request.Form.GetString("CheckBoxGroup", string.Empty).Split(',');

            if (StrCommentIDs != null && StrCommentIDs.Length > 0)
            {
                List<int> commentIDs = new List<int>();
                int commentID = 0;
                foreach (string strCommentID in StrCommentIDs)
                {
                    int.TryParse(strCommentID, out commentID);
                    commentIDs.Add(commentID);
                }

                VoteComments.UpdateAuditingStatus(commentIDs, auditingStatuses.Value);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_SetSucecss", applicationID);
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_SetFail", applicationID);
            }
            return Redirect(returnUrl);
        }

        #endregion

        #endregion

        /// <summary>
        /// 获取投票数据统计
        /// </summary>
        /// <returns></returns>
        public JsonResult GetManageableCounts()
        {
            Dictionary<VoteManageableCounts, int> dictionary = VoteThreads.GetManageableCounts();
            Application voteApplication = Applications.GetApplication(ApplicationIDs.Instance().Vote());
            List<ManageableCount> list = new List<ManageableCount>();
            list.Add(new ManageableCount { Name = "新增投票", Value = voteApplication.CountInPast24Hours }); 
            if (dictionary.ContainsKey(VoteManageableCounts.PendingThreadCount))
                list.Add(new ManageableCount { Name = "待审核用户投票", Value = dictionary[VoteManageableCounts.PendingThreadCount], Url = SPBUrlHelper.Action("ManageVoteThreads", "ManageVote", new RouteValueDictionary { { "auditingStatusesForDisplay", AuditingStatusesForDisplay.Pending }, { "presentAreaID", PresentAreaIDs.Instance().UserDomain() } }) });
            if (dictionary.ContainsKey(VoteManageableCounts.AgainThreadCount))
                list.Add(new ManageableCount { Name = "需再审核用户投票", Value = dictionary[VoteManageableCounts.AgainThreadCount], Url = SPBUrlHelper.Action("ManageVoteThreads", "ManageVote", new RouteValueDictionary { { "auditingStatusesForDisplay", AuditingStatusesForDisplay.Again }, { "presentAreaID", PresentAreaIDs.Instance().UserDomain() } }) });
            if (dictionary.ContainsKey(VoteManageableCounts.PendingCommentCount))
                list.Add(new ManageableCount { Name = "待审核用户投票评论", Value = dictionary[VoteManageableCounts.PendingCommentCount], Url = SPBUrlHelper.Action("ManageVoteComments", "ManageVote", new RouteValueDictionary { { "auditingStatusesForDisplay", AuditingStatusesForDisplay.Pending }, { "presentAreaID", PresentAreaIDs.Instance().UserDomain() } }) });

            if (dictionary.ContainsKey(VoteManageableCounts.PendingClubThreadCount))
                list.Add(new ManageableCount { Name = "待审核群组投票", Value = dictionary[VoteManageableCounts.PendingClubThreadCount], Url = SPBUrlHelper.Action("ManageVoteThreads", "ManageVote", new RouteValueDictionary { { "auditingStatusesForDisplay", AuditingStatusesForDisplay.Pending }, { "presentAreaID", PresentAreaIDs.Instance().ClubDomain() } }) });
            if (dictionary.ContainsKey(VoteManageableCounts.AgainClubThreadCount))
                list.Add(new ManageableCount { Name = "需再审核群组投票", Value = dictionary[VoteManageableCounts.AgainClubThreadCount], Url = SPBUrlHelper.Action("ManageVoteThreads", "ManageVote", new RouteValueDictionary { { "auditingStatusesForDisplay", AuditingStatusesForDisplay.Again }, { "presentAreaID", PresentAreaIDs.Instance().ClubDomain() } }) });
            if (dictionary.ContainsKey(VoteManageableCounts.PendingClubCommentCount))
                list.Add(new ManageableCount { Name = "待审核群组投票评论", Value = dictionary[VoteManageableCounts.PendingClubCommentCount], Url = SPBUrlHelper.Action("ManageVoteComments", "ManageVote", new RouteValueDictionary { { "auditingStatusesForDisplay", AuditingStatusesForDisplay.Pending }, { "presentAreaID", PresentAreaIDs.Instance().ClubDomain() } }) });
     
            list.Add(new ManageableCount { Name = "总投票数", Value = voteApplication.TotalCount });
            return Json(list,JsonRequestBehavior.AllowGet);
        }
    }
}

/// <summary>
/// SubMenu
/// </summary>
public enum VoteSubMenu
{
    ManageVote = 24101,
    ManageVoteComment = 24102
}