﻿//------------------------------------------------------------------------------
//<copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights reserved.
//</copyright> 
//------------------------------------------------------------------------------ 

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using SpaceBuilder.Common;
using System.Collections;
using System.Web;
using SpaceBuilder.Utils;
using System.IO;
using System.Web.Routing;
using SpaceBuilder.Web;
using SpaceBuilder.LuceneSearch;
using System.Web.UI;

namespace SpaceBuilder.Vote.Controllers
{
    [UserDomainGuideInstallApplication(ApplicationID = 241)]
    [HttpCompress]
    public class UserDomainVoteController : UserDomainControllerBase
    {

        int applicationID = ApplicationIDs.Instance().Vote();

        #region Vote

        #region View

        /// <summary>
        /// 显示空间内我的投票列表
        /// </summary>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListMyVoteThreads(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_MyVote", applicationID));
            return View("Pages/Votes/ListMyVoteThreads.aspx", "Masters/Vote.master");
        }

        /// <summary>
        /// 显示空间内我朋友的投票列表
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListMyFriendsVoteThreads(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_MyFriendVote", applicationID));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            PagingDataSet<VoteThread> pds = VoteThreads.GetThreadsOfFriends(currentDomainUser.UserID, Request.QueryString.GetInt("pageIndex", 1));

            return View("Pages/Votes/ListMyFriendsVoteThreads.aspx", "Masters/Vote.master", pds);
        }

        /// <summary>
        /// 发起投票
        /// </summary>
        /// <returns></returns>
        /// 
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult NewVoteThread(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            ViewData["isCreate"] = true;
            SetPageTitle(userDomainName, GetResourceString("Title_DoVote", applicationID));
            ViewData["MaxVoteOptionCount"] = VoteConfiguration.Instance().MaxVoteOptionCount;
            VoteThread thread = new VoteThread();
            thread.PrivacyStatus = PrivacyStatuses.Public;
            thread.EnableComments = true;
            if (TempData["VoteThread"] != null)
                thread = TempData["VoteThread"] as VoteThread;
            return this.View("Pages/Votes/NewVoteThread.aspx", "Masters/Vote.master", thread);

        }

        /// <summary>
        /// 编辑投票
        /// </summary>
        /// <param name="threadID"></param>
        /// <param name="userDomainName"></param>
        /// <param name="ownerTypeName"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult EditVoteThread(int? threadID, string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_EditVote", applicationID));

            ViewData["currentUser"] = GetCurrentDomainUser(userDomainName);
            ViewData["isCreate"] = false;

            VoteThread thread = VoteThreads.GetThread(threadID ?? 0, true);

            if (TempData["VoteThread"] != null)
                thread = TempData["VoteThread"] as VoteThread;
            return View("Pages/Votes/NewVoteThread.aspx", "Masters/Vote.master", thread);
        }

        #endregion

        #region Control

        /// <summary>
        /// 空间投票列表
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="votescopes"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Controller_ListVoteThreads(string userDomainName, VoteScopes? votescopes, int? pageIndex)
        {
            User currentdomainName = GetCurrentDomainUser(userDomainName);
            ViewData["userDomainName"] = currentdomainName.UserName;
            ViewData["scopes"] = votescopes;
            PagingDataSet<VoteThread> pds = new PagingDataSet<VoteThread>();
            if (votescopes.HasValue)
            {
                if (votescopes.Value == VoteScopes.myCreated)
                    pds = VoteThreads.GetOwnerThreads(PresentAreaIDs.Instance().UserDomain(), currentdomainName.UserID, pageIndex ?? 1, HasManagePermission(userDomainName));
                else if (votescopes.Value == VoteScopes.myJoined)
                    pds = VoteThreads.GetJoinedUserThreads(currentdomainName.UserID, pageIndex ?? 1);
            }
            return PartialView("Controls/Votes/ListVoteThreads.ascx", pds);
        }

        /// <summary>
        /// 在频道里显示投票
        /// </summary>
        /// <param name="threadID"></param>
        /// <returns></returns>
        public ActionResult Control_ShowVoteThreadBlock(string userDomainName, int threadID)
        {

            bool isVote = CurrentUser != null && VoteThreads.IsVoted(threadID, CurrentUser.UserID);
            ViewData["isVote"] = isVote;
            VoteThread thread = VoteThreads.GetThread(threadID, false);
            return PartialView("Controls/Votes/ShowVoteThreadBlock.ascx", thread);
        }

        #endregion

        #region Action

        /// <summary>
        /// 创建用户投票
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreateUpdateVoteThread(int? threadID, string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            VoteThread voteThread = null;
            Dictionary<int, int> voteCounts = new Dictionary<int, int>();
            if (threadID.HasValue && threadID.Value > 0)
            {
                voteThread = VoteThreads.GetThread(threadID.Value, true);
            }

            if (voteThread == null)
                voteThread = new VoteThread();

            voteThread.Author = currentDomainUser.DisplayName;
            voteThread.Subject = Request.Form.GetString("subject", string.Empty).Trim();
            StatusMessageData messageData = null;
            bool isError = false;

            if (string.IsNullOrEmpty(voteThread.Subject))
            {
                messageData = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_VoteTitleCannotNull", applicationID));
                isError = true;
            }
            if (voteThread.Subject.Length > 80)
            {
                messageData = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Exception_VoteMaxWord", applicationID), 80));
                isError = true;
            }
            voteThread.Body = Request.Form.GetString("body", string.Empty).Trim();
            if (voteThread.Body.Length > 160)
            {
                messageData = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_DescriptionVoteCannotOver160Word", applicationID));
                isError = true;
            }
            voteThread.EnableComments = Request.Form.GetBool("enableComments", false);
            voteThread.EndDate = Request.Form.Get<DateTime>("expireTime", DateTime.Now.AddMonths(1));
            if (voteThread.EndDate < DateTime.Now)
            {
                messageData = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_EndTimeCannotLessThanCurrentTime", applicationID));
                isError = true;
            }

            //设置隐状态
            voteThread.PrivacyStatus = (PrivacyStatuses)Request.Form.GetInt("privacyStatus", (int)PrivacyStatuses.Public);
            voteThread.VoteStaus = VoteStatuses.Publishing;

            voteThread.UserID = CurrentUser.UserID;
            voteThread.OwnerID = CurrentUser.UserID;
            voteThread.Owner = CurrentUser.DisplayName;
            //todo:表现区域ID
            voteThread.PresentAreaID = PresentAreaIDs.Instance().UserDomain();

            voteThread.MaxChooseCount = Request.Form.GetInt("maxChooseCount", 1);
            if (voteThread.MaxChooseCount <= 1)
                voteThread.IsMultiple = false;
            else
                voteThread.IsMultiple = true;


            int totalOptionCount = Request.Form.GetInt("totalOptionCount", 5);

            for (int i = 1; i <= totalOptionCount; i++)
            {
                string optionName = Request.Form.GetString("voteOption_" + i, string.Empty);
                int optionID = Request.Form.GetInt("OptionID_" + i, 0);

                VoteOption voteOption = new VoteOption();
                voteOption.OptionName = optionName;
                if (optionID > 0)
                    voteOption.OptionID = optionID;

                voteThread.VoteOptions[i] = voteOption;
            }

            if (voteThread.VoteOptions.Count < voteThread.MaxChooseCount)
                voteThread.MaxChooseCount = voteThread.VoteOptions.Count;

            if (voteThread.VoteOptions.Where(n => !String.IsNullOrEmpty(n.Value.OptionName)).Count() < 2)
            {
                isError = true;
                messageData = new StatusMessageData(StatusMessageType.Error, "投票项不能少于两项！");
            }

            if (isError)
            {
                TempData["VoteThread"] = voteThread;
                TempData["StatusMessageData"] = messageData;
                if (threadID.HasValue && threadID.Value > 0)
                    return RedirectToAction("EditVoteThread", new RouteValueDictionary { { "threadID", threadID.Value }, { "userDomainName", userDomainName } });
                else
                    return RedirectToAction("NewVoteThread", new RouteValueDictionary { { "userDomainName", userDomainName } });
            }

            if (threadID.HasValue && threadID.Value > 0)
            {
                VoteThreads.UpdateThread(voteThread);
                messageData = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_EditVoteSucecss", applicationID));
            }
            else
            {
                VoteThreads.CreateThread(voteThread);
                if (voteThread.ThreadID <= 0)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_SendVoteFail", applicationID));
                    return RedirectToAction("NewVoteThread", new RouteValueDictionary { { "userDomainName", userDomainName } });
                }
            }

            Dictionary<int, VoteOption> voteOptions = new Dictionary<int, VoteOption>(voteThread.VoteOptions);

            voteThread.VoteOptions.Clear();
            foreach (KeyValuePair<int, VoteOption> pair in voteOptions)
            {
                if (!string.IsNullOrEmpty(pair.Value.OptionName))
                {
                    pair.Value.OptionName = Formatter.FormatSingleLinePlainTextForStorage(pair.Value.OptionName);
                    pair.Value.ThreadID = voteThread.ThreadID;

                    if (pair.Value.OptionID > 0)
                    {
                        VoteThreads.UpdateVoteOption(pair.Value);
                    }
                    else
                    {
                        VoteThreads.CreateVoteOption(pair.Value);
                    }
                }
            }

            //编辑成功
            if (messageData != null)
            {
                TempData["StatusMessageData"] = messageData;
                return Redirect(SiteUrls.Instance().ShowVoteThread(voteThread.ThreadID));
            }

            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Exception_SendVoteSuccess", applicationID));
            return Redirect(SiteUrls.Instance().ShowVoteThread(voteThread.ThreadID));
        }

        /// <summary>
        /// 删除投票
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteVoteThread(string userDomainName, int threadID, VoteScopes votescopes, int? pageIndex)
        {
            VoteThread thread = VoteThreads.GetThread(threadID, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_VoteNoHave", applicationID)));
            }
            User domainUser = GetCurrentDomainUser(userDomainName);
            if (CurrentUser != null && (CurrentUser.UserID == domainUser.UserID || CurrentUser.IsAdministrator))
            {
                VoteThreads.DeleteThread(threadID);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteSucecssful", applicationID));
                return RedirectToAction("Controller_ListVoteThreads", new RouteValueDictionary { { "userDomainName", userDomainName }, { "votescopes", votescopes }, { "pageIndex", pageIndex } });
            }


            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_DeleteFail", applicationID));
            return RedirectToAction("Controller_ListVoteThreads", new RouteValueDictionary { { "userDomainName", userDomainName }, { "votescopes", votescopes }, { "pageIndex", pageIndex } });
        }

        #endregion

        #endregion

        #region Comment

        #region View

        /// <summary>
        /// 显示空间内管理评论
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ManageComments(string userDomainName, int? pageIndex)
        {
            User currentUserDomainName = GetCurrentDomainUser(userDomainName);
            PagingDataSet<VoteComment> pds = VoteComments.GetComments(currentUserDomainName.UserID, pageIndex ?? 1);
            return View("Pages/Votes/ManageComments.aspx", "Masters/Vote.master", pds);
        }

        #endregion

        #region Control

        /// <summary>
        /// 空间投票首页热点投票列表块
        /// </summary>      
        public ActionResult Control_SideVoteSortBlock(VoteThreadSortBy? sortBy, string Title, int? disPlayCount)
        {
            ViewData["Title"] = Title;
            ViewData["sortBy"] = (int)sortBy;
            IList<VoteThread> threads = VoteThreads.GetTopThreadsSortBy(PresentAreaIDs.Instance().UserDomain(), null, sortBy ?? VoteThreadSortBy.HitTimes, VoteStatuses.Publishing, disPlayCount ?? 7);
            return PartialView("Controls/Votes/SideVoteSortBlock.ascx", threads);
        }

        #endregion

        #region Operation

        /// <summary>
        /// 删除单个评论
        /// </summary>
        /// <returns></returns>
        public ActionResult DeleteComment(int commentID, string returnUrl)
        {
            VoteComment comment = VoteComments.GetComment(commentID, true);
            if (comment == null)
            {
                throw new ArgumentException(string.Format(GetResourceString("Exception_CannotExistIDIsReview", applicationID), commentID));
            }
            if (CurrentUser != null && (CurrentUser.UserID == comment.UserID || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
            {
                VoteComments.DeleteComment(commentID);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_ReviewDeleteSucecss", applicationID);
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_DeleteFailConcoatAdmin", applicationID);
            }
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="returnUrl"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteComments(string userDomainName, string returnUrl)
        {
            User ownerUser = GetCurrentDomainUser(userDomainName);
            string selectIDsString = Request.Form["ManageCommentsCheckBoxGroup"];
            int commentID = 0;
            if (!string.IsNullOrEmpty(selectIDsString))
            {
                string[] selectIDsStringArray = selectIDsString.Split(',');
                if (CurrentUser != null && (selectIDsStringArray != null && selectIDsStringArray.Length > 0 || CurrentUser.IsContentAdministrator || CurrentUser.IsAdministrator || CurrentUser.UserID == ownerUser.UserID))
                {
                    foreach (string idString in selectIDsStringArray)
                    {
                        if (string.IsNullOrEmpty(idString))
                            continue;
                        commentID = int.Parse(idString);
                        VoteComments.DeleteComment(commentID);
                    }
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = GetResourceString("Message_ReviewDeleteSucecss", applicationID);
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_FailNotDeleteThisLimit", applicationID);
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_PleaseSelectDeleteReview", applicationID);
            }

            return Redirect(returnUrl);
        }

        #endregion

        #endregion

        #region helper Method

        /// <summary>
        /// 是否具有管理空间权限
        /// </summary>
        protected bool HasManagePermission(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());
            return (CurrentUser != null && (currentDomainUser.UserID == CurrentUser.UserID || CurrentUser.IsContentAdministrator || CurrentUser.IsAdministrator));
        }

        #endregion

    }

    /// <summary>
    /// 空间投票二级菜单
    /// </summary>
    public enum VoteUserDomainSubMenus
    {

        //我发起的投票
        myCreated,

        //我朋友的投票
        myFriend
    }
}