(function() {
		  
var doc = document.documentElement, body = $(document.body);


function encodeHtmlEntity(str) {
	switch (str) {
		case "<" : return "&lt;";
		case ">" : return "&gt;";
		case "\"" : return "&quot;";
		case "&" : return "&amp;";
		default : return $0;
	}
}
function toHtmlSpace(str) {
	return str.replace(/ /g, "&nbsp;");
}
var htmlEntity = /\<|\>|\"|\&/g, multiSpaces = / {2,}/g, newLine = /(?:\r\n)|\n/g, tab = /\t/g, htmlBr = /<br(\s*)\/?>/gi,
	/// html工具类
	html = {
		/// HTML编码
		/// @param {String} 要编码的字符串
		/// @return {String} 编码后的字符串
		encode : function(str) {
			return str.replace(htmlEntity, encodeHtmlEntity)
					.replace(multiSpaces, toHtmlSpace)
					.replace(newLine, '<br />')
					.replace(tab, "　　");
		},
		
		/// HTML解码
		/// @param {String} 要解码的字符串
		/// @return {String} 解码后的字符串
		decode : function(str) {
			return str.replace(htmlBr, "\r\n");
		}
	};
	

/// 替换节点
/// @param {HtmlElement} 新节点
/// @param {HtmlElement} 原节点
/// @return {HtmlElement} 新节点
$.dom.replaceNode = function(newNode, oldNode) {
	if (oldNode.replaceNode) {
		oldNode.replaceNode(newNode);
	} else if (oldNode.parentNode.replaceChild) {
		oldNode.parentNode.replaceChild(newNode, oldNode);
	}
	return newNode;
};


/// 获取离指定节点最近的特定标签名的祖先节点
/// @param {HtmlElement} 节点
/// @param {String} 标签名
/// @return {HtmlElement} 祖先节点
$.dom.getAncestorByTagName = function(elem, tagName) {
	var parent = elem.parentNode;
	tagName = tagName.toUpperCase();
	while (parent) {
		if (tagName === parent.nodeName) {
			return parent;
		}
		parent = parent.parentNode;
	}
};


/// 弹出层控制类
/// @param {HtmlElement,HtmlElementArray,HtmlCollection} 控制弹出的元素
/// @param {HtmlElement,HtmlElementArray,HtmlCollection} 弹出层
/// @param {String} 触发显示弹出层的事件名
/// @param {Number} 隐藏延时，单位毫秒，默认为300毫秒
$.ui.PopupLayer = function(ctrl, content, eventName, delay) {
	if (!arguments.length) {
		return;
	}
	
	var t = this;
	t.constructor = arguments.callee;
	
	t._ctrl = ctrl;
	t._content = content;
	t.delay = isNaN(delay) ? 300 : delay;
	
	var delayHide = function(e) { e.cancelBubble = true; t.delayHide(); },
		clearDelay = function(e) { e.cancelBubble = true; t.clearDelay(); };
	
	$.event.addEvent(ctrl, eventName, function() { t.show(); });
	$.event.addEvent(ctrl, "mouseout", delayHide);
	$.event.addEvent(ctrl, "mouseover", clearDelay);
	$.event.addEvent(content, "mouseover", clearDelay);
	$.event.addEvent(content, "mouseout", delayHide);
}

// 弹出层控制类方法
$.ui.PopupLayer.prototype = {
	
	// 显示弹出层
	show : function() {
		$.style.addCss(this._content, "display:block;");
	},
	
	// 隐藏弹出层
	hide : function() {
		$.style.addCss(this._content, "display:none;");
	},
	
	// 延时隐藏弹出层
	delayHide : function() {
		this._timerId = setTimeout(this.hide.bind(this), this.delay);
	},
	
	// 取消延时隐藏
	clearDelay : function() {
		this._timerId && clearTimeout(this._timerId);
	}
};


/// 编辑器命令类
/// @param {TidyEditor} 编辑器对象
function EditorCommand(editor) {
	this.editor = editor;
}
/// 编辑器命令类原型(方法定义)
EditorCommand.prototype = {
		
	/// 发送编辑命令
	/// @param {String} 命令
	/// @param {String} 命令值
	/// @param {Boolean} 是否使用CSS，默认为false
	/// @param {Boolean} 是否选中范围，默认为true
	exec : function(cmd, value, styleWithCss, needRange) {
		var parent = this.editor;
		(null == needRange || true === needRange) && parent._selectRange();
		styleWithCss = styleWithCss != null ? styleWithCss : false;
		try {
			parent._areaDoc.execCommand("StyleWithCss", false, styleWithCss);
		} catch (e) {
		}
		try {
			parent._areaDoc.execCommand(cmd, false, value);
		} catch (e) {
		}
	},
		
	/// 设为粗体
	toBold : function() { this.exec("Bold"); },
	
	/// 设为斜体
	toItalic : function() { this.exec("Italic"); },
	
	/// 添加下划线
	addUnderline : function() { this.exec("Underline"); },
	
	/// 设置颜色
	/// @param {String} 颜色
	setForecolor : function(color) { this.exec("ForeColor", color, true); },
	
	/// 设置对齐方式
	/// @param {String} 对齐方式
	setAlign : function(align) {
		var node = this.editor.getLocalBlock(), style = "text-align:" + align + ";";
		if (!node) {
			this.formatBlock("p");
			node = this.editor.getLocalBlock();
		}
		if ("center" == align) {
			style += "text-indent:0;";
		}
		node && $.style.addCss(node, style);
	},
	
	/// 添加链接
	/// @param {String} 链接地址
	/// @param {String} 目标
	addLink : function(url, target) {
		this.exec("CreateLink", url);
		var parent = this.editor, a = parent.getLocalNode();
		a = "A" == a.tagName ? a : $.dom.getAncestorByTagName(a, "a");
		if (!a && parent._curRange.getRangeAt) {
			a = parent._curRange.getRangeAt(0).endContainer.parentNode;
			if (a && a.tagName != "A") {
				a = null;
			}
		}
		if (a) {
			a.target = target;
		}
	},
	
	/// 取消链接
	unlink : function() { this.exec("UnLink"); },
	
	/// 设为下标
	toSubscript : function() { this.exec("Subscript"); },
	
	/// 设为上标
	toSuperscript : function() { this.exec("Superscript"); },
	
	/// 格式化当前块
	/// @param {String} 块标签名
	formatBlock : function(tagName) { this.exec("FormatBlock", "<" + tagName + ">"); },
	
	/// 插入HTML
	/// @param {String} HTML代码
	/// @param {Number} 插入位置：0－开头，1－光标所在处，2末尾，默认为1
	insertHtml : function(html, position) {
		var areaBody = this.editor._areaDoc.body;
		position = null == position ? 1 : position;
		switch (position) {
			case 0:
				areaBody.innerHTML = html + areaBody.innerHTML;
			break;
			
			case 1:
				var range = this.editor._curRange;
				if (range && range.pasteHTML) {
					range.pasteHTML(html);
				} else {
					this.exec("InsertHTML", html);
				}
			break;
			
			case 2:
				areaBody.innerHTML = areaBody.innerHTML + html;
			break;
		}
	},
	
	/// 插入表格
	/// @param {Number} 行数
	/// @param {Number} 列数
	/// @param {Number} 宽度
	/// @param {Boolean} 是否有左标题
	/// @param {Boolean} 是否有顶标题
	insertTable : function(rowCount, colCount, width, thLeft, thTop) {
		var html = ['<table'], i, j;
		if (width) {
			html.push(' style="width:' + width + 'px;"');
		}
		html.push('>');
		if (thTop) {
			html.push('<thead><tr>');
			for (j = 0; j < colCount; j++) {
				html.push('<th></th>');
			}
			html.push('</tr></thead>');
			rowCount--;		// 第一行是标题，减去1
		}
		html.push('<tbody>');
		for (i = 0; i < rowCount; i++) {
			html.push('<tr>');
			for (j = 0; j < colCount; j++) {
				html.push(thLeft && 0 === j ? '<th></th>' : '<td></td>');
			}
			html.push('</tr>');
		}
		html.push('</tbody></table>');
		
		this.insertHtml(html.join(""));
	},
	
	/// 插入图片
	/// @param {String} 图片地址
	/// @param {String} 图片说明
	insertImage : function(url, alt) {
		var html = ['<p style="text-align:center;text-indent:0;"><img src="' + url + '"'];
		if (alt) {
			html.push(' alt="' + alt + '"');
		}
		html.push(' /></p>');
		
		this.insertHtml(html.join(""));
	},
	
	/// 插入代码块
	/// @param {String} 代码
	insertCode : function(code) {
		code = code.replace(/\t/g, "　　");
		this.insertHtml('<code>' + code + '</code>');
	}
};

/// TidyEditor编辑器类
/// @param {HtmlElement} 编辑区域
function TidyEditor(editor) {
	if (!arguments.length) { return; }
	
	var t = this;
	t.constructor = arguments.callee;
	
	t._codeArea = $("textarea.code", editor)[0];
	t._area = $("iframe.view", editor)[0];
	
	t._areaWin = t._area.contentWindow;
	t._areaDoc = t._areaWin.document;
	
	t.listenHandler = [];
	
	t.command = new EditorCommand(this);
}

// TidyEditor类原型(方法定义)
TidyEditor.prototype = {
	
	/// 初始化编辑器
	/// @param {String} 预设Css的地址
	/// @param {String} 相对地址基础部分
	/// @param {String} 初始内容内容
	init : function(defaultCss, baseHref, content) {
		var t = this;
		
		t._areaDoc.designMode = "on"; t._areaDoc.contentEditable = true; t._areaDoc.charset = "utf-8";
	
		// 写入预设Html
		t._areaDoc.open(); 
		t._areaDoc.write('<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd"><html xmlns="http://www.w3.org/1999/xhtml"><head><meta http-equiv="Content-Type" content="text/html; charset=utf-8" />');
		if (defaultCss) {
			t._areaDoc.write('<link href="' + defaultCss + '" rel="stylesheet" type="text/css" />');
		}
		if (baseHref) {
			t._areaDoc.write('<base href="' + baseHref + '" />');
		}
		t._areaDoc.write('</head><body>' + (typeof content !== 'string' || "" == content ? '<p></p>' : content)  + '</body></html>');
		t._areaDoc.close();
		
		// 当前模式为视图模式
		t._mode= "view";
		
		// 聚焦时开始监听
		$.event.addEvent(t._areaWin, "focus", function() {
			if (!t._listenTimer) {
				t._listen();
				t._listenTimer = setInterval(t._listen.bind(t), 550);
			}
		});
		
		// 失去焦点时停止监听
		$.event.addEvent(t._areaWin, "blur", function() {
			t._listenTimer && clearInterval(t._listenTimer);
			t._listenTimer = null;
		});
		
		// 处理不同浏览器的按键兼容
		$.event.addEvent(t._areaDoc, "keydown", t._fixKeys.bind(t));
		
		// 把光标设在第一个p内
		if (!$.browser.msie) {
			$.event.addEvent(t._areaDoc, "keydown", t._setCursorInFirstP.bind(t));
		}
	},
	
	// 定时监听状态
	_listen : function() {
		var len = this.listenHandler.length, i;
		if (len) {
			i = -1;
			while (++i < len) {
				this.listenHandler[i].call(this);
			}
		}
		this._setRange();
	},
	
	/// 聚焦到编辑器
	focus : function() { this._areaWin.focus(); },
	
	/// 切换编辑模式
	/// @param {String} view为可视化模式，code为代码模式
	switchMode : function(mode) {
		var t = this;
		mode = mode.toLowerCase();
		if (t._mode !== mode) {
			switch (mode) {
				case "view":
					t._codeArea.style.display = 'none';
					t._area.style.display = 'block';
					t._areaDoc.body.innerHTML = t._codeArea.value;
				break;
				
				case "code":
					t._codeArea.style.display = 'block';
					t._area.style.display = 'none';
					t._codeArea.value = t.getXhtml();
				break;
			}
			t._mode = mode;
		}
	},
	
	/// 获取当前编辑器模式
	/// @return {String} 当前编辑器模式，view为可视化模式，code为代码
	getMode : function() { return this._mode; },
	
	/// 处理按键兼容性
	/// @param {Object} 事件对象
	_fixKeys : function(e) {
		if (!e.shiftKey && 13 === e.keyCode && !$.browser.msie && !$.browser.opera) {
			this.command.insertHtml("<p></p>");
			e.preventDefault();
		}
	},
	
	/// 记录当前范围
	/// @return {Object} 当前范围
	_setRange : function() {
		var t = this;
		//if (document.activeElement === t._area) {
			if (t._areaDoc.selection) {		// For IE
				t._curRange = t._areaDoc.selection.createRange();
			} else if (t._areaWin.getSelection) {	// For FF
				t._curRange = t._areaWin.getSelection();
			}
		//}
		return t._curRange;
	},
	
	/// 选中当前光标所在位置
	/// @return {Object} 选中的范围
	_selectRange : function() {
		var range = this._curRange;
		if (!range) {
			this._areaDoc.body.focus();
			range = this._setRange();
		}
		try {
			if (range && range.select) {
				range.select();
			}
		} catch(e) {
		}
		return range;
	},
	
	/// 如果只有一个段落，把光标设到该处，For FF
	_setCursorInFirstP : function() {
		var areaBody = this._areaDoc.body, child = areaBody.firstChild;
		
		if (!child) {	// 没有子节点，建立一个空段落
			child = this.createElement("p");
			areaBody.appendChild(child);
		}
		
		if (child === areaBody.lastChild) {
			switch (child.nodeName) {
				case "P":
					if ("" === child.innerHTML) {
						var range = this._setRange();
						if (range && range.getRangeAt) {
							range = range.getRangeAt(0);
							range.setStart(child, 0);
							range.setEnd(child, 0);
							range.collapse(false);
							this._curRange = range;
						}
					}
				break;
				
				case "BR":
					areaBody.removeChild(child);
					areaBody.appendChild(this.createElement("p"));
				break;
			}
		}
	},
	
	/// 获取当前光标所在的节点
	/// @return {HtmlElement} 当前光标所在的节点
	getLocalNode : function() {
		var range = this._curRange, node;
		if (range) {
			if (range.getRangeAt) {
				range = range.getRangeAt(0);
				//node = 1 === range.startContainer.nodeType ? range.startContainer.childNodes[range.startOffset] : range.startContainer;
				if (1 === range.startContainer.nodeType) {
					node = range.startContainer.childNodes[range.startOffset];
				}
				if (!node) {
					node = range.startContainer;
				}
			} else if (range.select) {
				node = null != range.text ? range.parentElement() : range(0);
			}
			if (node && node.nodeType !== 1) {
				node = node.parentNode;
			}
		}
		return node;
	},
	
	/// 获取当前光标所在的块级节点
	/// @return {HtmlElement} 当前光标所在的块级节点
	getLocalBlock : function() {
		var node = this.getLocalNode(), display;
		while (node && node.nodeName !== "BODY") {
			display = $.style.getCurrentStyle(node, "display", this._areaWin).toLowerCase();
			if ("block" === display || "table" === display) {
				return node;
			} else {
				node = node.parentNode;
			}
		}
	},
	
	/// 创建编辑区域内的元素
	/// @param {String} 标签名
	/// @return {HtmlElement} 元素
	createElement : function(tagName) { return this._areaDoc.createElement(tagName); },
	
	// 命令对象
	command : {
		
		// 记录父对象
		editor : null,
		
		// 发送编辑命令
		// @param {String} 命令
		// @param {String} 命令值
		// @param {Boolean} 是否使用CSS，默认为false
		// @param {Boolean} 是否选中范围，默认为true
		exec : function(cmd, value, styleWithCss, needRange) {
			var parent = this.editor;
			(null == needRange || true === needRange) && parent._selectRange();
			styleWithCss = styleWithCss != null ? styleWithCss : false;
			try {
				parent._areaDoc.execCommand("StyleWithCss", false, styleWithCss);
			} catch (e) {
			}
			try {
				parent._areaDoc.execCommand(cmd, false, value);
			} catch (e) {
			}
		},
			
		// 设为粗体
		toBold : function() { this.exec("Bold"); },
		
		// 设为斜体
		toItalic : function() { this.exec("Italic"); },
		
		// 添加下划线
		addUnderline : function() { this.exec("Underline"); },
		
		// 设置颜色
		// @param {String} 颜色
		setForecolor : function(color) { this.exec("ForeColor", color, true); },
		
		// 设置对齐方式
		// @param {String} 对齐方式
		setAlign : function(align) {
			var node = this.editor.getLocalBlock(), style = "text-align:" + align + ";";
			if (!node) {
				this.formatBlock("p");
				node = this.editor.getLocalBlock();
			}
			if ("center" == align) {
				style += "text-indent:0;";
			}
			node && $.style.addCss(node, style);
		},
		
		// 添加链接
		// @param {String} 链接地址
		// @param {String} 目标
		addLink : function(url, target) {
			this.exec("CreateLink", url);
			var parent = this.editor, a = parent.getLocalNode();
			a = "A" == a.tagName ? a : $.dom.getAncestorByTagName(a, "a");
			if (!a && parent._curRange.getRangeAt) {
				a = parent._curRange.getRangeAt(0).endContainer.parentNode;
				if (a && a.tagName != "A") {
					a = null;
				}
			}
			if (a) {
				a.target = target;
			}
		},
		
		// 取消链接
		unlink : function() { this.exec("UnLink"); },
		
		// 设为下标
		toSubscript : function() { this.exec("Subscript"); },
		
		// 设为上标
		toSuperscript : function() { this.exec("Superscript"); },
		
		// 格式化当前块
		// @param {String} 块标签名
		formatBlock : function(tagName) { this.exec("FormatBlock", "<" + tagName + ">"); },
		
		// 插入HTML
		// @param {String} HTML代码
		// @param {Number} 插入位置：0－开头，1－光标所在处，2末尾，默认为1
		insertHtml : function(html, position) {
			var areaBody = this.editor._areaDoc.body;
			position = null == position ? 1 : position;
			switch (position) {
				case 0:
					areaBody.innerHTML = html + areaBody.innerHTML;
				break;
				
				case 1:
					var range = this.editor._curRange;
					if (range && range.pasteHTML) {
						range.pasteHTML(html);
					} else {
						this.exec("InsertHTML", html);
					}
				break;
				
				case 2:
					areaBody.innerHTML = areaBody.innerHTML + html;
				break;
			}
		},
		
		// 插入表格
		// @param {Number} 行数
		// @param {Number} 列数
		// @param {Number} 宽度
		// @param {Boolean} 是否有左标题
		// @param {Boolean} 是否有顶标题
		insertTable : function(rowCount, colCount, width, thLeft, thTop) {
			var html = ['<table'], i, j;
			if (width) {
				html.push(' style="width:' + width + 'px;"');
			}
			html.push('>');
			if (thTop) {
				html.push('<thead><tr>');
				for (j = 0; j < colCount; j++) {
					html.push('<th></th>');
				}
				html.push('</tr></thead>');
				rowCount--;		// 第一行是标题，减去1
			}
			html.push('<tbody>');
			for (i = 0; i < rowCount; i++) {
				html.push('<tr>');
				for (j = 0; j < colCount; j++) {
					html.push(thLeft && 0 === j ? '<th></th>' : '<td></td>');
				}
				html.push('</tr>');
			}
			html.push('</tbody></table>');
			
			this.insertHtml(html.join(""));
		},
		
		// 插入图片
		// @param {String} 图片地址
		// @param {String} 图片说明
		insertImage : function(url, alt) {
			var html = ['<p style="text-align:center;text-indent:0;"><img src="' + url + '"'];
			if (alt) {
				html.push(' alt="' + alt + '"');
			}
			html.push(' /></p>');
			
			this.insertHtml(html.join(""));
		},
		
		// 插入代码块
		// @param {String} 代码
		insertCode : function(code) {
			code = code.replace(/\t/g, "　　");
			this.insertHtml('<code>' + code + '</code>');
		}
	},
	
	// 获取原始内容
	// @return {String} 内容
	getHtml : function() {
		this.switchMode("view");
		return this._areaDoc.body.innerHTML;
	},

	// 获取Xhtml格式的内容
	// @return {String} Xhtml格式的内容
	getXhtml : function() {
		// 先切换到可视化模式
		this.switchMode("view");
		
		var body = this._areaDoc.body, html = body.innerHTML;

		html = html.replace(/<i(\s+|>)/ig, "<em$1")		// i -> em
				.replace(/<\/i>/ig, "</em>")
				.replace(/<b(\s+|>)/ig, "<strong$1")	// b -> strong
				.replace(/<\/b>/ig, "</strong>");

		body.innerHTML = html;
		
		// font -> span
		var fonts = $("font", body), span, i = fonts.length;
		while (--i >= 0) {
			span = this.createElement("span");
			if (fonts[i].color) {
				span.style.color = fonts[i].color;
			}
			span.innerHTML = fonts[i].innerHTML;
			$.dom.replaceNode(span, fonts[i]);
		}
		
		html = body.innerHTML;
		
		// 去注释
		html = html.replace(/<\!--[\W\w]*?-->/g, "");
		
		// 去掉多余的标签和属性，标签名和属性小写化，属性值加双引号
		html = html.replace(/<(\/?)([\w1-6]+)([^<>]*?)((?:\s*\/)?)>/g, function($0, $1, $2, $3, $4) {
			// 标签名小写化
			$2 = $2.toLowerCase();
			
			if ($3) {		// 如果有属性
				// 临时存放属性
				var temp = [];
				
				// 属性值加引号，单引号换为双引号
				$3 = $3.replace(/\s+(\w+=)(?=[^\"])(\'?)([^\2>\s]+)\2/g, ' $1"$3"');
				// 把属性转移到数组中
				$3 = $3.replace(/(\w+)\s*=\s*"([^"]*)"/g, function($$0, $$1, $$2) {
					$$1 = $$1.toLowerCase();	// 属性名小写
					if ($$2 !== "" && $$1 !== "uniqueidforjess") {
						temp.push($$1 + '="' + $$2 + '"');
					}
					return "";
				});
				// 给无值属性添加值
				$3 = $3.replace(/\s+(\w+)\s+/g, function($$0, $$1) {
					temp.push($$1 + '="' + $$1 + '"');
					return "";
				});
				
				// 重组属性
				$3 = temp.length ? " " + temp.join(" ") : "";
			}
			
			// 结束前加空格
			if ("/" === $4) {
				$4 = " /";
			}

			return "<" + $1 + $2 + $3 + $4 + ">";
		});
		
		// 关闭单标签
		html = html.replace(/<(br|img|hr|param|embed)([^<>]*?)\/?>/gi, function($0, $1, $2) {
			return "<" + $1 + ($2 ? " " + $2.trim() : "") + " />";
		});
		
		// IE下缺li结束标签，补全之
		html = html.replace(/<li(\s+[^<>]*)?>/gi, "</li><li$1>")
				.replace(/(?:\s*<\/li>\s*)+/gi, "</li>\r\n")
				.replace(/<([ou])l(\s+[^<>]*)?>\s*<\/li>/gi, "<$1l$2>");
		
		// 清除空段落、空链接、空span
		html = html.replace(/<(p|a|span)(?:\s+[^<>]*)?>(?:\s*|(?:&nbsp;)*)<\/\1>/gi, "");
		// 清除段首空格
		// html = html.replace(/(<p(?:\s+[^<>]*)?>)[　]+/gi, "$1");
		
		return html;
	},
	
	test : function() {
		alert(this.getLocalNode());
	}
};

/**
 * 以父窗口的document为上下文选取元素
 * @param {String,HtmlElement,HtmlElementArray,HtmlCollection} CSS选择器
 * @return {HtmlElement,HtmlElementArray} 匹配到的经扩展的HTML元素
 */
function $$(selector) {
	return $(selector, parent.document);
}

// 获取当前Url
var myUrl = document.location.href;

// 匹配出所在iframe的id
/id=(\w+)/.test(myUrl);
var parentFrame = $$("#" + RegExp.$1);

if (parentFrame) {
	// 设置所在iframe的高度
	window.addEvent("load", function() {
		parentFrame.style.height = document.body.scrollHeight + "px";
	});
}

// 建立编辑器实例
var tidyEditor = window.tidyEditor = new TidyEditor($("#editor"));


//　编辑器按钮
var editorBtns = {};

// 按钮样式控制
var btnElems = $("#editor div.buttons > ul > li"), i = btnElems.length;
while (--i >= 0) {
	editorBtns[btnElems[i].className] = $(btnElems[i]);
	btnElems[i].addEvent("mouseover", function() {
		if ("view" === tidyEditor.getMode()) {
			$.style.addCss(this, "active");
		}
	});
	btnElems[i].addEvent("mouseout", function() {
		if (!this.state) {
			$.style.removeCss(this, "active");
		}
	});
}

// 颜色面板弹出层
var colorPopup = new $.ui.PopupLayer(editorBtns["forecolor"], editorBtns["forecolor"].$("div.popup")[0], "click", 200);

// 链接弹窗
var linkWin = new $.ui.Win($("#linkWin"));
linkWin.onShow = function() {
	var node = tidyEditor.getLocalNode(), urlInput = $("#linkUrl");
	if (node && "A" == node.nodeName) {
		urlInput.value = node.href;
		$("#linkTarget").value = node.target;
		linkWin.currentNode = node;
	}
	urlInput.focus();
};
$("#linkWin form").addEvent("submit", function(e) {
	e.preventDefault();
	var url = $("#linkUrl").value.trim(), target = $("#linkTarget").value;
	if ("" == url) {
		alert("请填写Url");
		return;
	}
	if (linkWin.currentNode) {
		linkWin.currentNode.href = url;
		linkWin.currentNode.target = target;
	} else {
		tidyEditor.command.addLink(url, target);
	}
	this.reset();
	linkWin.hide();
});

// 图片弹窗
var imgWin = new $.ui.Win($("#imgWin"));
imgWin.onShow = function() {
	var node = tidyEditor.getLocalNode(), urlInput = $("#imgUrl");
	if (node && "IMG" == node.nodeName) {
		urlInput.value = node.src;
		$("#imgAlt").value = node.alt;
		imgWin.currentNode = node;
	}
	urlInput.focus();
};
$("#imgWin form").addEvent("submit", function(e) {
	e.preventDefault();
	var url = $("#imgUrl").value.trim(), alt = $("#imgAlt").value.trim();
	if ("" == url) {
		alert("请填写Url");
		return;
	}
	if (imgWin.currentNode) {
		imgWin.currentNode.src = url;
		imgWin.currentNode.alt = alt;
		imgWin.currentNode = null;
	} else {
		tidyEditor.command.insertImage(url, alt);
	}
	this.reset();
	imgWin.hide();
});

// 代码弹窗
var codeWin = new $.ui.Win($("#codeWin"));
codeWin.onShow = function() {
	var codeInput = $("#code"), node = tidyEditor.getLocalNode();
	if (node && "CODE" == node.nodeName) {
		codeInput.value = html.decode(node.innerHTML);
		codeWin.currentNode = node;
	}
	codeInput.focus();
};
$("#codeWin form").addEvent("submit", function(e) {
	e.preventDefault();
	var code = $("#code").value.trim();
	if ("" == code) {
		alert("请填写代码");
		return;
	}
	code = html.encode(code);
	if (codeWin.currentNode) {
		codeWin.currentNode.innerHTML = code;
	} else {
		tidyEditor.command.insertCode(code);
	}
	codeWin.currentNode = null;
	this.reset();
	codeWin.hide();
});

// 表格弹窗
var tblWin = new $.ui.Win($("#tblWin"));
tblWin.onShow = function() {
	$("#tblWidth").focus();
};
$("#tblWin form").addEvent("submit", function(e) {
	e.preventDefault();
	var width = parseInt($("#tblWidth").value.trim()), rowCount = parseInt($("#tblRow").value.trim()), colCount = parseInt($("#tblCol").value.trim()),
		isThLeft = $("#tblThLeft").checked, isThTop = $("#tblThTop").checked;
	if (isNaN(width) || width < 0) {
		alert("表格宽度必须为大于0的数字");
		return;
	}
	if (isNaN(rowCount) || rowCount < 0) {
		alert("表格行数必须为大于0的数字");
		return;
	}
	if (isNaN(colCount) || colCount < 0) {
		alert("表格列数必须为大于0的数字");
		return;
	}
	tidyEditor.command.insertTable(rowCount, colCount, width, isThLeft, isThTop);
	this.reset();
	tblWin.hide();
});
	
// 绑定按钮功能
editorBtns["bold"].addEvent("click", function() { tidyEditor.command.toBold(); });
editorBtns["italic"].addEvent("click", function() { tidyEditor.command.toItalic(); });
editorBtns["underline"].addEvent("click", function() { tidyEditor.command.addUnderline(); });
editorBtns["unlink"].addEvent("click", function() { tidyEditor.command.unlink(); });
editorBtns["alignCenter"].addEvent("click", function() { tidyEditor.command.setAlign("center"); });
editorBtns["alignJustify"].addEvent("click", function() { tidyEditor.command.setAlign("justify"); });
editorBtns["alignLeft"].addEvent("click", function() { tidyEditor.command.setAlign("left"); });
editorBtns["alignRight"].addEvent("click", function() { tidyEditor.command.setAlign("right"); });
editorBtns["subscript"].addEvent("click", function() { tidyEditor.command.toSubscript(); });
editorBtns["superscript"].addEvent("click", function() { tidyEditor.command.toSuperscript(); });
editorBtns["blocks"].$("select").addEvent("change", function() {
	tidyEditor.command.formatBlock(this.value);
});
editorBtns["forecolor"].$("span").addEvent("click", function(e) {
	tidyEditor.command.setForecolor($.style.getCurrentStyle(this, "backgroundColor"));
	e.cancelBubble = true;
	colorPopup.hide();
});
editorBtns["table"].addEvent("click", function(e) {
	e.preventDefault();
	tblWin.show();
});
editorBtns["image"].addEvent("click", function(e) {
	e.preventDefault();
	imgWin.show();
});
editorBtns["code"].addEvent("click", function(e) {
	e.preventDefault();
	codeWin.show();
});
editorBtns["link"].addEvent("click", function(e) {
	e.preventDefault();
	linkWin.show();
});
	
$("input.toViewMode").addEvent("click", function() { tidyEditor.switchMode("view"); });
$("input.toCodeMode").addEvent("click", function() { tidyEditor.switchMode("code"); });

var blocks = $("#blocks");
tidyEditor.listenHandler.push(function() {
	var node = this.getLocalBlock();
	node && (blocks.value = node.nodeName);
});

})();