﻿using Microsoft.Practices.Prism.Commands;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace SmokeRegex.Client.ViewModels
{
    public class MainViewModel : Base.ViewModelBase
    {
        public MainViewModel()
        {
            this.RegexOptions = RegexOptions.None;

            this.LoadInitPatterns();
            this.HistoryPatterns = new ObservableCollection<Models.PatternInfo>();

            if (!IsInDesignMode)
            {
                this.SubscribeEvent<Events.DownloadHtmlEvent>(this.OnDownloadHtmlEventReceived);
                this.SubscribeEvent<Events.ReplaceEvent>(this.OnReplaceEventReceived);
            }
        }

        #region 属性
        private string _input;

        /// <summary>
        /// 输入的文本
        /// </summary>
        public string Input
        {
            get
            {
                return _input;
            }
            set
            {
                if (_input != value)
                {
                    _input = value;
                    this.RaisePropertyChanged("Input", "CanMatchCommandExecute");
                }
            }
        }

        private string _pattern;

        /// <summary>
        /// 正则表达式
        /// </summary>
        public string Pattern
        {
            get
            {
                return _pattern;
            }
            set
            {
                if (_pattern != value)
                {
                    _pattern = value;
                    this.RaisePropertyChanged("Pattern", "CanMatchCommandExecute");
                }
            }
        }

        private string _result;

        /// <summary>
        /// 结果
        /// </summary>
        public string Result
        {
            get
            {
                return _result;
            }
            set
            {
                if (_result != value)
                {
                    _result = value;
                    this.RaisePropertyChanged("Result");

                    //Result更改的时候一般就是点击正则按钮的时候
                    //此时添加历史记录
                    this.AddToHistory();
                }
            }
        }

        private RegexOptions _regexOptions;

        /// <summary>
        /// 正则选项
        /// </summary>
        public RegexOptions RegexOptions
        {
            get
            {
                return _regexOptions;
            }
            set
            {
                if (_regexOptions != value)
                {
                    _regexOptions = value;
                    this.RaisePropertyChanged("RegexOptions");
                }
            }
        }

        /// <summary>
        /// 表示操作按钮是否是激活状态
        /// </summary>
        public bool CanMatchCommandExecute
        {
            get
            {
                return !string.IsNullOrEmpty(this.Pattern) && !string.IsNullOrEmpty(this.Input);
            }
        }

        /// <summary>
        /// 预选的表达式列表
        /// </summary>
        public ObservableCollection<Models.PatternInfo> InitPatterns
        {
            get;
            private set;
        }

        /// <summary>
        /// 最近使用的表达式列表
        /// </summary>
        public ObservableCollection<Models.PatternInfo> HistoryPatterns
        {
            get;
            private set;
        }

        private Models.PatternInfo _currentPattern;

        /// <summary>
        /// 当前选中的表达式
        /// </summary>
        public Models.PatternInfo CurrentPattern
        {
            get
            {
                return _currentPattern;
            }
            set
            {
                if (_currentPattern != value)
                {
                    _currentPattern = value;
                    if (value != null)
                    {
                        this.Pattern = value.Pattern;
                    }
                    else
                    {
                        this.Pattern = string.Empty;
                    }
                }
            }
        }

        #endregion

        #region 匹配单个命令
        private DelegateCommand _matchCommand;

        /// <summary>
        /// 匹配单个的命令
        /// </summary>
        public DelegateCommand MatchCommand
        {
            get
            {
                if (_matchCommand == null)
                {
                    _matchCommand = new DelegateCommand(this.Match);
                }
                return _matchCommand;
            }
        }

        private void Match()
        {
            Regex regex = this.CreateRegex();
            var match = regex.Match(this.Input);
            this.Result = match.Value;
        }
        #endregion

        #region 清空命令
        private DelegateCommand _clearCommand;

        /// <summary>
        /// 清空数据命令
        /// </summary>
        public DelegateCommand ClearCommand
        {
            get
            {
                if (_clearCommand == null)
                {
                    _clearCommand = new DelegateCommand(this.Clear);
                }
                return _clearCommand;
            }
        }

        private void Clear()
        {
            this.Input = null;
            this.Pattern = null;
            this.Result = null;
            this.RegexOptions = RegexOptions.None;
        }
        #endregion

        #region 是否匹配命令
        private DelegateCommand _isMatchCommand;

        /// <summary>
        /// 是否匹配命令
        /// </summary>
        public DelegateCommand IsMatchCommand
        {
            get
            {
                if (_isMatchCommand == null)
                {
                    _isMatchCommand = new DelegateCommand(this.IsMatch);
                }
                return _isMatchCommand;
            }
        }

        private void IsMatch()
        {
            Regex regex = this.CreateRegex();
            bool isMatch = regex.IsMatch(this.Input);
            this.Result = isMatch ? "匹配成功" : "匹配失败";
        }

        #endregion

        #region 匹配所有命令
        private DelegateCommand _matchesCommand;

        /// <summary>
        /// 匹配所有命令
        /// </summary>
        public DelegateCommand MatchesCommand
        {
            get
            {
                if (_matchesCommand == null)
                {
                    _matchesCommand = new DelegateCommand(this.Matches);
                }
                return _matchesCommand;
            }
        }

        private void Matches()
        {
            Regex regex = this.CreateRegex();
            var matches = regex.Matches(this.Input);
            StringBuilder sb = new StringBuilder();
            foreach (Match match in matches)
            {
                sb.Append(match.Value);
                sb.AppendLine();
                sb.AppendLine();
            }
            this.Result = sb.ToString();
        }

        #endregion

        #region 文本拆分命令
        private DelegateCommand _splitCommand;

        /// <summary>
        /// 文本拆分命令
        /// </summary>
        public DelegateCommand SplitCommand
        {
            get
            {
                if (_splitCommand == null)
                {
                    _splitCommand = new DelegateCommand(this.Split);
                }
                return _splitCommand;
            }
        }

        private void Split()
        {
            Regex regex = this.CreateRegex();
            string[] result = regex.Split(this.Input);
            StringBuilder sb = new StringBuilder();
            foreach (string item in result)
            {
                sb.Append(item);
                sb.AppendLine();
                sb.AppendLine();
            }
            this.Result = sb.ToString();
        }
        #endregion

        #region 组匹配命令

        private DelegateCommand _groupMatchCommand;

        public DelegateCommand GroupMatchCommand
        {
            get
            {
                if (_groupMatchCommand == null)
                {
                    _groupMatchCommand = new DelegateCommand(this.GroupMatch);
                }
                return _groupMatchCommand;
            }
        }

        private void GroupMatch()
        {
            Regex regex = this.CreateRegex();
            Match match = regex.Match(this.Input);
            StringBuilder sb = new StringBuilder();
            foreach (Group group in match.Groups)
            {
                sb.Append(group.Value);
                sb.AppendLine();
                sb.AppendLine();
            }
            this.Result = sb.ToString();
        }


        #endregion

        #region 方法

        /// <summary>
        /// 根据内部变量,创造一个Regex
        /// </summary>
        /// <returns></returns>
        private Regex CreateRegex()
        {
            Regex regex = new Regex(this.Pattern, this.RegexOptions);
            return regex;
        }

        /// <summary>
        /// 加载预选表达式列表
        /// </summary>
        private void LoadInitPatterns()
        {
            this.InitPatterns = new ObservableCollection<Models.PatternInfo>();

            this.InitPatterns.Add(new Models.PatternInfo { Name = "不选择", Pattern = string.Empty });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "邮箱", Pattern = @"^[\w-]+(\.[\w-]+)*@[\w-]+(\.[\w-]+)+$" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "URL地址", Pattern = @"^[a-zA-z]+://(\w+(-\w+)*)(\.(\w+(-\w+)*))*(\?\S*)?$" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "身份证号码", Pattern = @"\d{18}|\d{15}" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "国内座机号码", Pattern = @"(\d{3}-|\d{4}-)?(\d{8}|\d{7})" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "字母数字及下划线", Pattern = @"^\w+$" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "数字和字母", Pattern = @"^[A-Za-z0-9]+$" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "字母", Pattern = @"^[A-Za-z]+$" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "数字", Pattern = @"^\d+$" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "中文字符", Pattern = @"[\u4e00-\u9fa5]+" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "双字节字符(包括汉字)", Pattern = @"[^\x00-\xff]+" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "手机号码", Pattern = @"^1{1}\d{10}$" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "邮政编码", Pattern = @"^[a-zA-Z0-9 ]{3,12}$" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "匹配空行", Pattern = @"\n[\s| ]*\r" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "匹配首尾空格", Pattern = @"(^\s*)|(\s*$)" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "IP地址", Pattern = @"\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "匹配A标签", Pattern = @"<a[^>]+href=\s*(?:'(?<href>[^']+)'|""(?<href>[^""]+)""|(?<href>[^>\s]+))\s*[^>]*>" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "HTML标签(一)", Pattern = @"<(.*)>.*<\/\1>|<(.*) \/>" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "HTML标签(二)", Pattern = @"<(.[^\<])*>|<\/\w+>" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "日期格式", Pattern = @"\d{4}\W\d{1,2}\W\d{1,2}" });
            this.InitPatterns.Add(new Models.PatternInfo { Name = "日期+时间", Pattern = @"^((((1[6-9]|[2-9]\d)\d{2})-(0?[13578]|1[02])-(0?[1-9]|[12]\d|3[01]))|(((1[6-9]|[2-9]\d)\d{2})-(0?[13456789]|1[012])-(0?[1-9]|[12]\d|30))|(((1[6-9]|[2-9]\d)\d{2})-0?2-(0?[1-9]|1\d|2[0-9]))|(((1[6-9]|[2-9]\d)(0[48]|[2468][048]|[13579][26])|((16|[2468][048]|[3579][26])00))-0?2-29-)) (20|21|22|23|[0-1]?\d):[0-5]?\d:[0-5]?\d$" });
        }

        /// <summary>
        /// 在下载Html成功后触发的事件
        /// </summary>
        /// <param name="e"></param>
        private void OnDownloadHtmlEventReceived(Events.DownloadHtmlEvent e)
        {
            this.Input = e.Html;
        }

        /// <summary>
        /// 在请求替换文本时触发的事件
        /// </summary>
        /// <param name="e"></param>
        private void OnReplaceEventReceived(Events.ReplaceEvent e)
        {
            Regex regex = this.CreateRegex();
            this.Result = regex.Replace(this.Input, e.ReplaceString);
        }

        /// <summary>
        /// 在历史记录中添加当前的表达式
        /// </summary>
        private void AddToHistory()
        {
            if (string.IsNullOrEmpty(this.Pattern))
            {
                return;
            }

            if (this.HistoryPatterns.Count(t => t.Pattern == this.Pattern) == 0)
            {
                Models.PatternInfo item = new Models.PatternInfo
                {
                    Name = this.Pattern,
                    Pattern = this.Pattern
                };
                this.HistoryPatterns.Insert(0, item);
            }

            int allowHistoryCount = 20;

            while (this.HistoryPatterns.Count > allowHistoryCount)
            {
                this.HistoryPatterns.RemoveAt(allowHistoryCount);
            }
        }
        #endregion
    }
}
