// 代码压缩
const UglifyJsPlugin = require('uglifyjs-webpack-plugin');
// gzip压缩
const CompressionWebpackPlugin = require('compression-webpack-plugin');
// 是否为生产环境
const isProduction = process.env.NODE_ENV !== 'development';
// 生产环境是否需要使用cdn
const prodNeedCdn = true;

// cdn链接
const cdn = {
    // cdn：模块名称和模块作用域命名（对应window里面挂载的变量名称）
    externals: {
        vue: 'Vue',
        'vuex': 'Vuex',
        'vue-router': 'VueRouter',
        axios: 'axios',
        'element-ui': 'ELEMENT',
        'qs': 'Qs',
        'nprogress': 'NProgress',
        'screenfull': 'screenfull',
        'e-icon-picker': 'eIconPicker',
        'vue-cropper': '["vue-cropper"]',
    },
    // cdn的css链接
    css: [
        'https://cdn.jsdelivr.net/npm/e-icon-picker@1.0.14/dist/index.css',
        'https://cdn.jsdelivr.net/npm/font-awesome@4.7.0/css/font-awesome.min.css',
        'https://cdn.jsdelivr.net/npm/element-ui@2.14.0/lib/theme-chalk/index.css',
        // 'https://cdn.jsdelivr.net/npm/element-ui@2.13.2/lib/theme-chalk/base.css',
        'https://cdn.jsdelivr.net/npm/nprogress@0.2.0/nprogress.min.css'
    ],
    // cdn的js链接
    js: [
        'https://cdn.jsdelivr.net/npm/vue@2.6.12/dist/vue.min.js',
        'https://cdn.jsdelivr.net/npm/vuex@3.5.1/dist/vuex.min.js',
        'https://cdn.jsdelivr.net/npm/vue-router@3.4.8/dist/vue-router.min.js',
        'https://cdn.jsdelivr.net/npm/axios@0.21.0/dist/axios.min.js',
        'https://cdn.jsdelivr.net/npm/element-ui@2.14.0/lib/index.js',
        'https://cdn.jsdelivr.net/npm/qs@6.9.4/dist/qs.js',
        'https://cdn.jsdelivr.net/npm/nprogress@0.2.0/nprogress.min.js',
        'https://cdn.jsdelivr.net/npm/screenfull@5.0.2/dist/screenfull.min.js',
        'https://cdn.jsdelivr.net/npm/vue-cropper@0.5.5/dist/index.min.js',
        'https://cdn.jsdelivr.net/npm/e-icon-picker@1.0.14/dist/index.min.js',
        'https://cdn.jsdelivr.net/npm/e-icon-picker@1.0.14/dist/symbol.js',
    ]
};

module.exports = {
    //vue 中文配置文档地址
    //https://cli.vuejs.org/zh/config/#css-loaderoptions
    // 如果你不需要生产环境的 source map，可以将其设置为 false 以加速生产环境构建。
    productionSourceMap: false,
    chainWebpack: config => {
        config.optimization.minimizer('terser').tap((args) => {
            args[0].terserOptions.compress.drop_console = true;
            return args
        });

        // 移除 prefetch 插件(针对生产环境首屏请求数进行优化)
        config.plugins.delete('prefetch');
        // 移除 preload 插件(针对生产环境首屏请求数进行优化)   preload 插件的用途：https://cli.vuejs.org/zh/guide/html-and-static-assets.html#preload
        config.plugins.delete('preload');
        // ============注入cdn start============
        config.plugin('html').tap(args => {
            // 生产环境或本地需要cdn时，才注入cdn
            if (isProduction && prodNeedCdn) args[0].cdn = cdn;
            return args
        });
        // ============注入cdn start============
        // ============压缩图片 start============
        config.module
            .rule('images')
            .use('image-webpack-loader')
            .loader('image-webpack-loader')
            .options({bypassOnDebug: true})
            .end();
        // ============压缩图片 end============
    },
    configureWebpack: config => {
        // 生产环境相关配置
        if (isProduction) {
            // 用cdn方式引入，则构建时要忽略相关资源
            if (isProduction && prodNeedCdn) {
                config.externals = cdn.externals;
            }
            // 代码压缩
            config.plugins.push(
                new UglifyJsPlugin({
                    uglifyOptions: {
                        warnings: false,
                        //生产环境自动删除console
                        compress: {
                            drop_debugger: true,
                            drop_console: true,
                        }
                    },
                    sourceMap: false,
                    parallel: true
                })
            );
            // gzip压缩
            const productionGzipExtensions = ['html', 'js', 'css'];
            config.plugins.push(
                new CompressionWebpackPlugin({
                    filename: '[path].gz[query]',
                    algorithm: 'gzip',
                    test: new RegExp(
                        '\\.(' + productionGzipExtensions.join('|') + ')$'
                    ),
                    threshold: 1024, // 只有大小大于该值的资源会被处理 10240
                    minRatio: 0.8, // 只有压缩率小于这个值的资源才会被处理
                    deleteOriginalAssets: false // 删除原文件
                })
            );

            // 公共代码抽离
            config.optimization = {
                splitChunks: {
                    cacheGroups: {
                        vendor: {
                            chunks: 'all',
                            test: /node_modules/,
                            name: 'vendor',
                            minChunks: 1,
                            maxInitialRequests: 5,
                            minSize: 0,
                            priority: 100
                        },
                        common: {
                            chunks: 'all',
                            test: /[\\/]src[\\/]utils[\\/]/,
                            name: 'common',
                            minChunks: 2,
                            maxInitialRequests: 5,
                            minSize: 0,
                            priority: 60
                        },
                        styles: {
                            name: 'styles',
                            test: /\.(sa|sc|c)ss$/,
                            chunks: 'all',
                            enforce: true
                        },
                        runtimeChunk: {
                            name: 'manifest'
                        }
                    }
                }
            }
        }
    },
    // 反向代理
    devServer: {
        open: true,
        host: '0.0.0.0',
        port: 8080,
        https: false,
        hotOnly: false,
        proxy: {
            // 配置跨域
            [process.env.VUE_APP_BASE_API]: {
                target: 'http://localhost:8880',
                ws: true,
                changOrigin: true,
                pathRewrite: {
                    ['^' + process.env.VUE_APP_BASE_API]: ''
                }
            }
        },
        before: app => {
        }
    }
};
