<?php

    // +----------------------------------------------------------------------
    // | 模块初始化行为文件
    // +----------------------------------------------------------------------
    // | Copyright (c) 2015-2019 http://www.yicmf.com, All rights reserved.
    // +----------------------------------------------------------------------
    // | Author: 微尘 <yicmf@qq.com>
    // +----------------------------------------------------------------------

    namespace app\common\behavior;

    use app\ucenter\event\User as UserEvent;
    use app\ucenter\event\AuthGroup as AuthGroupEvent;
    use think\facade\Config;
    use think\exception\HttpResponseException;
    use think\facade\Lang;
    use think\facade\Response;
    use think\facade\Url;
    use think\Container;
    use think\facade\Cookie;
    use think\response\Redirect;
    use think\facade\Env;

    class ModuleInit
    {

        /**
         * 模块的初始化操作
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/3/24 22:06
         */
        public function run()
        {
            $request = Container::get('request');
                // 禁用ip检查
                $this->checkIp($request);
                // 权限检查
                $controllers = explode('.', $request->controller());
                if ( Config::get('auth.auth_on') && in_array($request->module(), Config::get('auth.auth_module'))
                    || (count($controllers) == 2 && strtolower($controllers[0]) == 'admin') ) {
                    // 登陆检查
                    $this->checkLogin($request);
                    // 检查是否是开放权限
                    if ( !in_array($request->module() . '/' . $request->controller() . '/' . $request->action(), Config::get('auth.public_visit')) ) {
                        // 认证模块检查
                        $this->checkAuth($request);
                    }
                }
        }

        /**
         * 检查权限.
         * @param Request $request
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/3/25 8:26
         */
        public function checkLogin($request)
        {
            // 未登陆进入登陆认证
            if ( !UserEvent::isLogin() ) {
                // 缓存要登陆的模块
                Cookie::set('login_module', $request->module());
                // 缓存要访问的地址
                Cookie::set('login_redirect', $request->url(true));
                $response = new Redirect(Url::build('ucenter/Foreign/login', ['module' => $request->module()]));
                $response->code(302);
                throw new HttpResponseException($response);
            }
        }

        /**
         * 安全检查
         * @param $request
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/3/25 8:31
         */
        protected function checkIp($request)
        {
            // 检查IP地址访问
            if ( in_array($request->ip(), explode(',', Config::get('setting.admin_allow_ip'))) ) {
                $response = Response::create(Lang::get('ip not allow'), 'html');
                throw new HttpResponseException($response);
            }
        }

        /**
         * 检查权限
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/3/25 8:26
         */
        public function checkAuth($request)
        {

            // 检测系统权限
            if ( !UserEvent::isAdministrator() && Config::get('auth.auth_on') ) {
                $rule = strtolower($request->module() . '/' . $request->controller() . '/' . $request->action());
                $access = $this->_accessControl($rule);
                if ( false === $access ) {
                    $response = Response::create(Lang::get('no auth access'), 'html');
                    throw new HttpResponseException($response);
                } elseif ( null === $access ) {
                    // 检测访问权限
                    if ( !AuthGroupEvent::checkRule($rule, UserEvent::getLogin()) ) {
                        $response = Response::create(Lang::get('no auth access'), 'html');
                        throw new HttpResponseException($response);
                    } else {
                        // TODO:其他类型的权限
                    }
                }
            }
        }

        /**
         * action访问控制，在 **登陆成功** 后执行的第一项权限检测任务
         * 返回 **false**, 不允许任何人访问
         * 返回 **true**, 允许任何角色访问,无需执行节点权限检测
         * 返回 **null**, 需要继续执行节点权限检测决定是否允许访问.
         * @return bool|null 返回值必须使用 `===` 进行判断
         * @param $rule
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/3/25 17:04
         */
        protected function _accessControl($rule)
        {
            $allow = array_map('strtolower', Config::get('auth.allow_visit'));
            $deny = array_map('strtolower', Config::get('auth.deny_visit'));
            if ( !empty($deny) && in_array($rule, $deny) ) {
                return false; //非超管禁止访问deny中的方法
            }
            if ( !empty($allow) && in_array($rule, $allow) ) {
                return true;
            }
            //next:需要检测节点权限
        }

    }
