<?php

    // +----------------------------------------------------------------------
    // | builder
    // +----------------------------------------------------------------------
    // | Copyright (c) 2015-2019 http://www.yicmf.com, All rights reserved.
    // +----------------------------------------------------------------------
    // | Author: 微尘 <yicmf@qq.com>
    // +----------------------------------------------------------------------

    namespace app\common\builder;

    use think\Db;
    use think\Exception;
    use think\facade\Lang;
    use think\Model;
    use think\facade\Url;

    class Edit extends Builder
    {
        private $_title;

        private $_suggest;

        private $_warning;

        private $_keyList = [];

        // 提示信息
        private $_explaints = [];

        /**
         * @var Model|array
         */
        private $_data;

        private $_buttonList = [];

        private $_reload = 'true';
        private $_mask = 'true';
        private $_savePostUrl;

        private $_group = [];

        private $_triggers = [];

        // 默认获取主键的字段
        protected $_default_pk = 'id';

        /**
         * 设置请求表单头
         * @param string $title
         * @return Edit
         */
        public function form($reload = true, $mask = true)
        {
            $this->_mask = $mask;
            $this->_reload = $reload;
            return $this;
        }

        /**
         * 设置标题.
         * @param string $title
         * @return Edit
         */
        public function title($title)
        {
            $this->_title = $title;

            return $this;
        }

        /**
         * suggest 页面标题边上的提示信息.
         * @param string $suggest
         * @return Edit
         */
        public function suggest($suggest)
        {
            $this->_suggest = $suggest;

            return $this;
        }

        /**
         * 页面下面的提示.
         * @param string|array $explain
         * @return $this
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/4/13 7:53
         */
        public function explain($explain)
        {
            if ( is_array($explain) ) {
                $this->_explaints = array_merge($this->_explaints, $explain);
            } else {
                $this->_explaints[] = $explain;
            }
            return $this;
        }

        /**
         * 设置触发.
         * @param string|array $trigger 需要触发的表单项名，目前支持select（单选类型）、text、radio三种
         * @param string       $values  触发的值
         * @param string       $show    触发后要显示的表单项名，目前不支持普通联动、范围、拖动排序、静态文本
         * @return $this
         */
        public function setTrigger($trigger, $values = '', $show = '')
        {
            if ( is_array($trigger) ) {
                $this->_triggers = array_merge($this->_triggers, $trigger);
            } else {
                $this->_triggers[$trigger][] = ['value' => $values, 'show' => $show];
                //参与的所有字段合集
                if ( isset($this->_triggers[$trigger]['field']) ) {
                    $this->_triggers[$trigger]['field'] .= ',' . $show;
                } else {
                    $this->_triggers[$trigger]['field'] = $show;
                }
            }

            return $this;
        }

        /**
         * warning 页面标题边上的错误信息.
         * @param $warning
         * @return $this
         */
        public function warning($warning)
        {
            $this->_warning = $warning;

            return $this;
        }

        /**
         * 直接显示html.
         * @param string $name
         * @param string $html
         * @param string $title
         * @param string $tips
         * @return Edit
         */
        public function keyHtml($name, $html, $title, $tips = null)
        {
            return $this->key($name, $title, $tips, 'html', $html);
        }

        /**
         * 隐藏表单.
         * @param string $name
         * @param string $title
         * @return Edit
         */
        /**
         * @param      $name
         * @return Edit
         * @throws Exception
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/4/12 11:31
         */
        public function keyHidden($name)
        {
            return $this->key($name, null, null, 'hidden');
        }

        /**
         * 只读文本.
         * @param string      $name
         * @param string      $title
         * @param string|null $tips
         * @return Edit
         * @throws Exception
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/4/12 11:31
         */
        public function keyReadOnly($name, $title, $tips = null)
        {
            return $this->key($name, $title, $tips, 'readonly');
        }

        /**
         * 文本输入框.
         * @param string      $name
         * @param string      $title
         * @param string|null $tips
         * @param int         $size
         * @param array|null  $validate_rule
         * @return Edit
         */
        public function keyText($name, $title, $tips = null, $size = 35, $validate_rule = null)
        {
            return $this->key($name, $title, $tips, 'string', null, $size, $validate_rule);
        }

        /**
         * 数组输入框，内容以’,‘分隔
         * @param string      $name
         * @param string      $title
         * @param string|null $tips
         * @param int         $size
         * @param array|null  $validate_rule
         * @return Edit
         */
        public function keyArray($name, $title, $tips = null, $size = 30, $validate_rule = null)
        {
            return $this->key($name, $title, $tips, 'string', null, $size, $validate_rule);
        }

        /**
         * 文本输入框常用显示title.
         * @param string      $name
         * @param string      $title
         * @param string|null $tips
         * @param int         $size
         * @param array|null  $validate_rule
         * @return Edit
         */
        public function keyTitle($name = 'title', $title = '标题', $tips = null, $size = 30, $validate_rule = null)
        {
            return $this->keyText($name, $title, $tips, $size, $validate_rule);
        }

        /**
         * 地图选择（需安装高德插件）.
         * @param $title
         * @param $tips
         * @return Edit
         */
        public function keyAmap($name, $title, $tips = null, $validate_rule = null)
        {
            return $this->key($name, $title, $tips, 'amap', null, 30, $validate_rule);
        }

        /**
         * 单选按钮.
         * @param string      $name
         * @param string      $title
         * @param array       $options
         * @param string|null $tips
         * @param array|null  $validate_rule
         * @return Edit
         */
        public function keyRadio($name, $title, $options, $tips = null, $validate_rule = null)
        {
            return $this->key($name, $title, $tips, 'radio', $options, 30, $validate_rule);
        }

        /**
         * 是与否的单选.
         * @param string      $name
         * @param string      $title
         * @param string|null $tips
         * @param array|null  $validate_rule
         * @return Edit
         */
        public function keyBool($name, $title, $tips = null, $validate_rule = null)
        {
            $options = [
                1 => '是',
                0 => '否',
            ];

            return $this->keyRadio($name, $title, $options, $tips);
        }

        /**
         * 性别
         * @param string      $name
         * @param string      $title
         * @param string|null $tips
         * @param array|null  $validate_rule
         * @return Edit
         */
        public function keySex($name, $title = '性别', $tips = null, $validate_rule = null)
        {
            $options = [
                2 => '女',
                1 => '男',
                0 => '保密',
            ];

            return $this->keyRadio($name, $title, $options, $tips);
        }

        /**
         * 新窗口选择
         * @param string      $name
         * @param string      $title
         * @param array       $options
         * @param string|null $tips
         * @param array|null  $validate_rule
         * @param int         $size
         * @return Edit
         */
        public function keyFind($name, $url, $title, $tips = null, $field = 'id', $size = 20, $validate_rule = null)
        {
            return $this->key($name, $title, $tips, 'find', ['url' => $url, 'field' => $field, 'num' => 1], $size, $validate_rule);
        }

        /**
         * 新窗口选择多个
         * @param        $name
         * @param        $url
         * @param        $limit
         * @param        $title
         * @param null   $tips
         * @param string $field
         * @param int    $size
         * @param null   $validate_rule
         * @return Edit
         */
        public function keyBelongsToMany($name, $url, $title, $field = 'id', $tips = null, $size = 20, $validate_rule = null)
        {
            return $this->key($name, $title, $tips, 'belongsToMany', ['url' => $url, 'field' => $field, 'limit' => 0], $size, $validate_rule);
        }

        /**
         * 新窗口选择一个
         * @param        $name
         * @param        $url
         * @param        $limit
         * @param        $title
         * @param null   $tips
         * @param string $field
         * @param int    $size
         * @param null   $validate_rule
         * @return Edit
         */
        public function keyBelongTo($name, $url, $title, $field = 'id', $tips = null, $size = 20, $validate_rule = null)
        {
            return $this->key($name, $title, $tips, 'belongTo', ['url' => $url, 'field' => $field, 'limit' => 0], $size, $validate_rule);
        }

        /**
         * 下拉列表.
         * @param string      $name
         * @param string      $title
         * @param array       $options
         * @param string|null $tips
         * @param bool        $multiple 是否开启多项选择
         * @param array|null  $validate_rule
         * @param int         $size
         * @return Edit
         */
        public function keySelect($name, $title, $options, $tips = null, $size = 200, $validate_rule = null)
        {
            return $this->key($name, $title, $tips, 'select', $options, $size, $validate_rule);
        }

        /**
         * 下拉列表多选.
         * @param string      $name
         * @param string      $title
         * @param array       $options
         * @param string|null $tips
         * @param bool        $multiple 是否开启多项选择
         * @param array|null  $validate_rule
         * @param int         $size
         * @return Edit
         */
        public function keySelectMultiple($name, $title, $options, $tips = null, $size = 0, $validate_rule = null)
        {
            return $this->key($name, $title, $tips, 'select_multiple', $options, $size, $validate_rule);
        }

        /**
         * 用于状态的下拉选择.
         * @param string     $name
         * @param string     $title
         * @param string     $tips
         * @param number     $size
         * @param array|null $options
         * @param array|null $validate_rule
         * @return Edit
         */
        public function keyStatus($name = 'status', $title = '状态', $options = null, $tips = null, $validate_rule = null)
        {
            $options = $options ?: [
                -2 => '删除',
                -1 => '禁用',
                1 => '启用',
                0 => '未审核',
                2 => '推荐',
            ];

            return $this->keySelect($name, $title, $options, $tips, $validate_rule);
        }

        /**
         * 复选框.
         * @param string      $name
         * @param string      $title
         * @param array       $options
         * @param string|null $tips
         * @param array|null  $validate_rule
         * @return Edit
         */
        public function keyCheckBox($name, $title, $options, $tips = null, $validate_rule = null)
        {
            return $this->key($name, $title, $tips, 'checkbox', $options, 30, $validate_rule);
        }

        /**
         * 关联单选复选框.
         * @param string      $name
         * @param string      $title
         * @param array       $options
         *                                   string $ajax_url 请求的url
         *                                   string $field 关联的字段
         * @param string|null $tips
         * @param array|null  $validate_rule
         * @return Edit
         */
        public function keyCheckBoxLinkCheck($name, $title, $options, $tips = null, $validate_rule = null)
        {
            return $this->key($name, $title, $tips, 'checkbox_link_check', $options, 30, $validate_rule);
        }

        /**
         * 用户组选择.
         * @param string     $name
         * @param string     $title
         * @param string     $tips
         * @param number     $size
         * @param array|null $validate_rule
         * @return Edit
         */
        public function keyMultiUserGroup($name, $title, $tips = null, $module = 'admin', $validate_rule = null)
        {
            return $this->keyCheckBox($name, $title, $this->readUserGroups($module), $tips, $validate_rule);
        }

        /**
         * TextArea
         * @param      $name
         * @param      $title
         * @param null $tips
         * @param int  $cols 列
         * @param int  $rows 行
         * @param null $validate_rule
         * @return Edit
         * @throws Exception
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/4/23 15:11
         */
        public function keyTextArea($name, $title, $tips = null, $cols = 50, $rows = 2, $validate_rule = null)
        {
            $default_rule = [
                //  'toggle' => 'autoheight',
            ];
            $validate_rule = is_array($validate_rule) ? array_merge($default_rule, $validate_rule) : $default_rule;
            $size = [$cols, $rows];

            return $this->key($name, $title, $tips, 'textarea', null, $size, $validate_rule);
        }

        /**
         * 使用label显示内容
         * @param      $name
         * @param      $title
         * @param null $tips
         * @return Edit
         * @throws Exception
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/4/23 15:12
         */
        public function keyLabel($name, $title, $tips = null)
        {
            return $this->key($name, $title, $tips, 'label');
        }

        /**
         * 闭包函数
         * @param      $name
         * @param      $title
         * @param      $closure
         * @param null $tips
         * @return Edit
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/4/6 13:18
         */
        public function keyClosure($name, $title, $closure, $tips = null)
        {
            return $this->key($name, text($title), $tips, $closure);
        }

        /**
         * 输入密码
         * @param string      $name
         * @param string      $title
         * @param string|null $tips
         * @return Edit
         */
        public function keyPassword($name, $title, $tips = null)
        {
            return $this->key($name, $title, $tips, 'password');
        }

        /**
         * 输入url地址
         * @param string      $name
         * @param string      $title
         * @param string|null $tips
         * @param int         $size
         * @param array|null  $validate_rule
         * @return Edit
         */
        public function keyUrl($name, $title, $tips = null, $size = 50, $validate_rule = null)
        {
            $default_rule = [
                'rule' => 'url',
            ];
            $tips = empty($tips) ? '需要以http或者https开头' : $tips;
            $validate_rule = is_array($validate_rule) ? array_merge($default_rule, $validate_rule) : $default_rule;

            return $this->key($name, $title, $tips, 'url', null, $size, $validate_rule);
        }

        /**
         * 颜色选择器.
         * @param string      $name
         * @param string      $title
         * @param string|null $tips
         * @return Edit
         */
        public function keyColor($name, $title, $tips = null)
        {
            return $this->key($name, $title, $tips, 'color');
        }

        /**
         * 可拖动插件列表.
         * @param string      $name
         * @param string      $title
         * @param string|null $tips
         * @return Edit
         */
        public function keyDragsortLi($name, $title, $tips = null, $options = null)
        {
            return $this->key($name, $title, $tips, 'dragsort_li', $options);
        }

        /**
         * 带字体颜色选择的输入框，一般为文章标题.
         * @param string      $name
         * @param string      $title
         * @param string|null $tips
         * @return Edit
         */
        public function keyHeadline($name, $title, $tips = null, $validate_rule = null)
        {
            return $this->key($name, $title, $tips, 'headline', null, null, $validate_rule);
        }

        /**
         * 自带的标签功能.
         * @param string      $name
         * @param string      $title
         * @param string|null $tips
         * @param array|null  $validate_rule
         * @return Edit
         */
        public function keyTags($name, $title, $tips = null, $validate_rule = null)
        {
            return $this->key($name, $title, $tips, 'tags', null, null, $validate_rule);
        }

        /**
         * 重量配置
         * @param string      $name          字段
         * @param string      $title         标题
         * @param array|null  $validate_rule
         *                                   string $tips 为空的时候提示
         *                                   number $min [可选] 最小值。
         *                                   number $max [可选] 最大值。
         *                                   number $step [可选] 步长，每次调整的值大小。
         *                                   number $decimalPlace [可选] 小数位数。
         * @param string|null $url           如果填写url，则用户的每一步操作都会请求该地址验证
         * @return Edit
         */
        public function keyWeight($name, $title, $tips = null, $validate_rule = null, $url = null)
        {
            $validate_rule_default = [
                'min' => 0,
                'max' => 50,
                'step' => 1,
                'decimal-place' => 3,
            ];
            $validate_rule = is_array($validate_rule) ? array_merge($validate_rule_default, $validate_rule) : $validate_rule_default;

            return $this->key($name, $title, $tips, 'spinner', $url, strlen($validate_rule['max']) + 10, $validate_rule);
        }

        /**
         * 微调器.
         * @param string      $name          字段
         * @param string      $title         标题
         * @param array|null  $validate_rule
         *                                   string $tips 为空的时候提示
         *                                   number $min [可选] 最小值。
         *                                   number $max [可选] 最大值。
         *                                   number $step [可选] 步长，每次调整的值大小。
         *                                   number $decimalPlace [可选] 小数位数。
         * @param string|null $url           如果填写url，则用户的每一步操作都会请求该地址验证
         * @return Edit
         */
        public function keySpinner($name, $title, $tips = null, $validate_rule = null, $url = null)
        {
            $validate_rule_default = [
                'min' => 0,
                'max' => 255,
                'step' => 1,
                'decimal-place' => 0,
            ];
            $validate_rule = is_array($validate_rule) ? array_merge($validate_rule_default, $validate_rule) : $validate_rule_default;

            return $this->key($name, $title, $tips, 'spinner', $url, strlen($validate_rule['max']) + 10, $validate_rule);
        }

        /**
         * 价格
         * @param      $name
         * @param      $title
         * @param null $tips
         * @param null $validate_rule
         * @param int  $size
         * @return Edit
         * @throws Exception
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/4/23 14:57
         */
        public function keyDecimal($name, $title, $tips = null, $validate_rule = null, $size = 15)
        {
            $validate_rule_default = [
                'rule' => 'money',
                'rule-money' => '[/^(?!0+(?:\.0+)?$)(?:[1-9]\d*|0)(?:\.\d{1,2})?$/, \'金额必须大于0并且只能精确到分\']',
                'tip' => '请填写金额',
                'ok' => '',
            ];
            $validate_rule = is_array($validate_rule) ? array_merge($validate_rule_default, $validate_rule) : $validate_rule_default;

            return $this->key($name, $title, $tips, 'number', null, $size, $validate_rule);
        }

        /**
         * 排序
         * @param string $name
         * @param string $title
         * @param string $tips
         * @param null   $validate_rule
         * @param int    $size
         * @return Edit
         * @throws Exception
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/4/23 14:57
         */
        public function keySort($name = 'sort', $title = '排序', $tips = '请填写0-255之间的数字', $validate_rule = null, $size = 15)
        {

            $validate_rule_default = [
                'rule' => 'sort',
                'rule-sort' => '[/^(\d|[1-9]\d?|1\d{2}?|2[0-4][0-9]?|25[0-5])$/, \'请填写0-255之间的数字\']',
                'tip' => '请填写0-255之间的数字',
                'ok' => '',
            ];
            $validate_rule = is_array($validate_rule) ? array_merge($validate_rule_default, $validate_rule) : $validate_rule_default;
            return $this->key($name, $title, $tips, 'number', null, $size, $validate_rule);
        }

        /**
         * 要求验证填写数字.
         * @param string     $name
         * @param string     $title
         * @param string     $tips
         * @param number     $size
         * @param array|null $validate_rule
         * @return Edit
         */
        public function keyNumber($name, $title, $tips = null, $validate_rule = null, $size = 15)
        {
            $validate_rule_default = [
                'rule' => 'number',
                'tip' => Lang::get('please fill in the numbers'),
                'ok' => '',
            ];
            $validate_rule = is_array($validate_rule) ? array_merge($validate_rule_default, $validate_rule) : $validate_rule_default;

            return $this->key($name, $title, $tips, 'number', null, $size, $validate_rule);
        }

        /**
         * 邮箱.
         * @return Edit
         */
        public function keyEmail($name, $title, $tips = null, $validate_rule = null, $size = 25)
        {
            $validate_rule_default = [
                'rule' => (true === $validate_rule) ? 'required;email' : 'email',
                'tip' => '请填写邮箱',
                'ok' => '',
            ];
            $validate_rule = is_array($validate_rule) ? array_merge($validate_rule_default, $validate_rule) : $validate_rule_default;

            return $this->key($name, $title, $tips, 'default', null, $size, $validate_rule);
        }

        /**
         * 手机
         * @return Edit
         */
        public function keyMobile($name, $title, $tips = null, $validate_rule = null, $size = 15)
        {
            $validate_rule_default = [
                'rule' => (true === $validate_rule) ? 'required;mobile' : 'mobile',
                'tip' => '请填写手机号',
                'ok' => '',
            ];
            $validate_rule = is_array($validate_rule) ? array_merge($validate_rule_default, $validate_rule) : $validate_rule_default;

            return $this->key($name, $title, $tips, 'default', null, $size, $validate_rule);
        }

        /**
         * 选择来源、有待改进.
         * @param string     $name
         * @param string     $title
         * @param string     $tips
         * @param number     $size
         * @param array|null $validate_rule
         * @return Edit
         */
        public function keySourse($name, $title, $tips, $options)
        {
            return $this->key($name, $title, $tips, 'sourse', $options);
        }

        /**
         * 富文本编辑器
         * @param string $name
         * @param string $title
         * @param string $tips
         * @param string $config
         * @param array  $style
         * @return Edit
         */
        public function keyEditor($name, $title, $tips = null, $config = [], $style = ['width' => '500px', 'height' => '400px'])
        {
            $default = [
                'items' => 'fontname, fontsize, |,forecolor, hilitecolor, bold, italic, underline, removeformat, |,justifyleft, justifycenter, justifyright, insertorderedlist, insertunorderedlist, |, emoticons, image, link',
                'manager' => Url::build('file/Editor/manager', ['module' => $this->request->module()]),
                'upload' => Url::build('file/Editor/upload', ['module' => $this->request->module()])
            ];
            $config = array_merge($default, $config);
            $key = [
                'name' => $name,
                'title' => $title,
                'tips' => $tips,
                'type' => 'editor',
                'config' => $config,
                'style' => $style,
            ];
            $this->_keyList[] = $key;

            return $this;
        }

        /**
         * 时间选择器.
         * @param string      $name
         * @param string      $title
         * @param string      $tips
         * @param array|null  $validate_rule
         *                                   pattern [可选] 日期格式（y=年， M=月，d=日，H=时，m=分，s=秒）。
         *                                   min-date [可选]日历范围的开始日期。支持动态格式，如data-min-date="{%y-10}-%M-%d"表示将最开始日期定为比当前日期小十年。%y = 本年，%M = 本月，%d = 今天。
         *                                   max-date [可选] 日期格式（y=年， M=月，d=日，H=时，m=分，s=秒）。
         *                                   mm-step [可选] 调整时间：[分] 的步长。日期格式包含 [m] 时有效
         *                                   ss-step [可选] 调整时间：[秒] 的步长。日期格式包含 [s] 时有效
         *                                   nobtn [可选] 文本框右边是否显示日历按钮
         *                                   onlybtn [可选] 只有点击日历按钮时才触发日期选择器
         * @param string|null $url           如果填写url，则用户的每一步操作都会请求该地址验证
         * @return Edit
         */
        public function keyTime($name, $title, $tips = null, $validate_rule = null, $url = '1')
        {
            $validate_rule_default = [
                'pattern' => 'yyyy-MM-dd H:m:s',
                'min-date' => '1990-01-01',
                'max-date' => '2039-12-31',
                'mm-step' => 1,
                'ss-step' => 1,
                'nobtn' => 'false',
                'onlybtn' => 'false',
            ];
            $validate_rule = is_array($validate_rule) ? array_merge($validate_rule_default, $validate_rule) : $validate_rule_default;

            return $this->key($name, $title, $tips, 'datetime', $url, strlen($validate_rule['pattern']) + 5, $validate_rule);
        }

        /**
         * 日期选择器.
         * @param string      $name
         * @param string      $title
         * @param string      $tips
         * @param array|null  $validate_rule
         *                                   pattern [可选] 日期格式（y=年， M=月，d=日，H=时，m=分，s=秒）。
         *                                   min-date [可选]日历范围的开始日期。支持动态格式，如data-min-date="{%y-10}-%M-%d"表示将最开始日期定为比当前日期小十年。%y = 本年，%M = 本月，%d = 今天。
         *                                   max-date [可选] 日期格式（y=年， M=月，d=日，H=时，m=分，s=秒）。
         *                                   mm-step [可选] 调整时间：[分] 的步长。日期格式包含 [m] 时有效
         *                                   ss-step [可选] 调整时间：[秒] 的步长。日期格式包含 [s] 时有效
         *                                   nobtn [可选] 文本框右边是否显示日历按钮
         *                                   onlybtn [可选] 只有点击日历按钮时才触发日期选择器
         * @param string|null $url           如果填写url，则用户的每一步操作都会请求该地址验证
         * @return Edit
         */
        public function keyDate($name, $title, $tips = null, $validate_rule = null, $url = '1')
        {
            $validate_rule_default = [
                'pattern' => 'yyyy-MM-dd',
                'min-date' => '1990-01-01',
                'max-date' => '2039-12-31',
                'mm-step' => 1,
                'ss-step' => 1,
                'nobtn' => 'false',
                'onlybtn' => 'false',
            ];
            $validate_rule = is_array($validate_rule) ? array_merge($validate_rule_default, $validate_rule) : $validate_rule_default;

            return $this->key($name, $title, $tips, 'date', $url, strlen($validate_rule['pattern']) + 5, $validate_rule);
        }

        public function keyTimeRange($name, $title, $tips = null, $validate_rule = null)
        {
            $validate_rule_default = [
                'pattern' => 'yyyy-MM-dd H:m:s',
                'min-date' => '1990-01-01',
                'max-date' => '2019-12-31',
                'mm-step' => 1,
                'ss-step' => 1,
                'nobtn' => true,
                'onlybtn' => 'false',
            ];
            $validate_rule = is_array($validate_rule) ? array_merge($validate_rule_default, $validate_rule) : $validate_rule_default;

            return $this->key($name, $title, $tips, 'timerange', null, 19, $validate_rule);
        }

        /**
         * 仅展示一张图片.
         * @param string $name
         * @param string $title
         * @param string $tips
         * @return Edit
         */
        public function keyShowImg($name, $title, $tips = '点击图片即可下载')
        {
            return $this->key($name, $title, $tips, 'showImg');
        }

        /**
         * 上传单个音频
         * @param 标题 $title
         * @param 描述 $tips
         * @return Edit
         */
        public function keyVoice($name, $title, $remark = null, $size = 50, $validate_rule = null)
        {
            $extensions = 'mp3,wav,wma,amr';
            return $this->key($name, $title, $remark, 'attachment', ['remark' => $remark, 'limit' => 1, 'extensions' => $extensions, 'size' => $size], 30, $validate_rule);
        }

        /**
         * 上传单个视频
         * @param 标题 $title
         * @param 描述 $tips
         * @return Edit
         */
        public function keyVideo($name, $title, $remark = null, $size = 300, $validate_rule = null)
        {
            $extensions = 'mp4,avi,rmvb,rm,wmv';
            return $this->key($name, $title, $remark, 'attachment', ['remark' => $remark, 'limit' => 1, 'extensions' => $extensions, 'size' => $size], 30, $validate_rule);
        }

        /**
         * 上传单个图片.
         * @param string $name 需要保存的字段，为URL地址，且必须是以_url结尾的字符串
         * @param string $title
         * @param null   $remark
         * @param null   $validate_rule
         * @return Edit
         * @throws Exception
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/5/8 11:57
         */
        public function keyImage($name, $title, $remark = null, $validate_rule = null)
        {
            $max_size = 0;
            $exts = '';
            $mimes = '';
            return $this->key($name, $title, null, 'image',
                ['remark' => $remark, 'limit' => 1, 'max_size' => $max_size, 'mimes' => $mimes, 'exts' => $exts]
                , 30, $validate_rule);
        }

        /**
         * 多图片上传
         * @param string $name 需要保存的字段，为URL地址，且必须是以_url结尾的字符串
         * @param string $title
         * @param null   $remark
         * @param int    $limit
         * @param null   $validate_rule
         * @return Edit
         * @throws Exception
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/5/8 11:58
         */

        public function keyImages($name, $title, $remark = null, $limit = 5, $validate_rule = null)
        {
            $max_size = 0;
            $exts = '';
            $mimes = '';
            return $this->key($name, $title, null, 'image',
                ['remark' => $remark, 'limit' => $limit, 'max_size' => $max_size, 'mimes' => $mimes, 'exts' => $exts]
                , 30, $validate_rule);
        }

        /**
         * 上传单个附件
         * @param      $name
         * @param      $title
         * @param null $tips
         * @param int  $size
         * @param null $validate_rule
         * @return Edit
         * @throws Exception
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/5/8 11:58
         */
        public function keyAttachment($name, $title, $tips = null, $size = 50, $validate_rule = null)
        {
            $extensions = '*';
            $remark = '';
            return $this->key($name, $title, $tips, 'attachment', ['remark' => $remark, 'limit' => 1, 'extensions' => $extensions, 'size' => $size], 30, $validate_rule);
        }

        /**
         * 添加城市选择（需安装城市联动插件）
         * @param      $name
         * @param      $title
         * @param null $tips
         * @param null $validate_rule
         * @return Edit
         * @throws Exception
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/5/8 11:58
         */
        public function keyCity($name, $title, $tips = null, $validate_rule = null)
        {
            // 修正在编辑信息时无法正常显示已经保存的地区信息
            return $this->key($name, $title, $tips, 'city', null, 30, $validate_rule);
        }

        /**
         * 批量添加字段信息.
         * @param array $fields
         * @return Edit
         * @author 微尘 <yicmf@qq.com>
         */
        public function setKeys($fields = [])
        {
            $this->_keyList = empty($this->_keyList)?$fields:array_merge($this->_keyList, $fields);

            return $this;
        }

        /**
         * 表单参数组合
         * @param      $name
         * @param      $title
         * @param      $tips
         * @param      $type
         * @param null $options
         * @param int  $size
         * @param null $validate_rule
         * @return $this
         * @throws Exception
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/4/12 10:35
         */
        public function key($name, $title, $tips, $type, $options = null, $size = 30, $validate_rule = null)
        {
            $key = [
                'name' => $name,
                'title' => $title,
                'tips' => (string)$tips,
                'type' => $type,
                'size' => $size,
                'rule' => $this->_getValidaorRule($validate_rule),
                'options' => $options,
            ];
            $this->_keyList[] = $key;

            return $this;
        }

        /**
         * 批量配置key
         * @param $keyList
         * @return $this
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/4/12 10:25
         */
        public function keys($keyList)
        {
            $this->_keyList = array_merge($this->_keyList, $keyList);
            return $this;
        }

        /**
         * 按钮基础调用，一般内部使用.
         * @param string $title
         * @param array  $attr
         * @return Edit
         */
        public function button($title, $attr = [])
        {
            $this->_buttonList[] = [
                'title' => $title,
                'attr' => $attr,
            ];

            return $this;
        }

        /**
         * 提交按钮.
         * @param string $url 提交的url地址，默认当前页
         * @param string $title
         * @return Edit
         */
        public function buttonSubmit($url = null, $title = '保存')
        {
            if ( !is_null($url) ) {
                $this->savePostUrl($url);
            }
            $attr['class'] = 'btn-default';
            $attr['type'] = 'submit';
            $attr['data-icon'] = 'save';

            return $this->button($title, $attr);
        }

        /**
         * 关闭按钮.
         * @param string $title
         * @return Edit
         */
        public function buttonMore($url, $title = '更多操作')
        {
            $attr = [];
            $attr['href'] = Url::build($url);
            $attr['type'] = 'button';
            $attr['data-icon'] = 'save';
            $attr['data-toggle'] = 'doajax';
            $attr['class'] = 'btn-default';

            return $this->button($title, $attr);
        }

        /**
         * 关闭按钮.
         * @param string $title
         * @return Edit
         */
        public function buttonBack($title = '关闭')
        {
            $attr = [];
            $attr['type'] = 'button';
            $attr['data-icon'] = 'close';
            $attr['class'] = 'btn-close';

            return $this->button($title, $attr);
        }

        /**
         * 当前表单值
         * @param array $data
         * @return Edit
         */
        public function data($data = [])
        {
            $this->_data = $data;

            return $this;
        }

        /**
         * 当前表单提交的地址
         * @param array $data
         * @return Edit
         */
        public function savePostUrl($url)
        {
            $this->_savePostUrl = Url::build($url);
        }

        /**
         * input和下拉选择组合
         * @param string $name
         * @param string $title
         * @param string $tips
         * @param array  $config
         * @param string $style
         * @return Edit
         */
        public function keyInputAndSelect($name, $title, $tips, $config, $style = 'width:400px;')
        {
            $name = is_array($name) ? $name : explode('|', $name);
            $key = [
                'name' => $name,
                'title' => $title,
                'tips' => $tips,
                'type' => 'ias',
                'config' => $config,
                'style' => $style,
            ];
            $this->_keyList[] = $key;

            return $this;
        }

        /**
         * keyMultiInput 输入组组件.
         * @param string $name
         * @param string $title
         * @param string $tips
         * @param array  $config
         * @param string $style
         * @return Edit
         */
        public function keyMultiInput($name, $title, $tips, $config, $style = 'width:400px;')
        {
            $name = is_array($name) ? $name : explode('|', $name);
            $key = [
                'name' => $name,
                'title' => $title,
                'tips' => $tips,
                'type' => 'multiInput',
                'config' => $config,
                'style' => $style,
            ];
            $this->_keyList[] = $key;

            return $this;
        }

        /**
         * 插入配置分组.
         * @param              $name 组名
         * @param array|string $list 组内字段列表
         * @return Edit
         */
        public function group($name, $list = [])
        {
            if ( is_array($name) ) {
                $this->_group = array_merge($this->_group, $name);
            } else {
                !is_array($list) && $list = explode(',', $list);
                $this->_group[$name] = $list;
            }

            return $this;
        }

        /**
         * 解析加载模版输出
         * @param string $template
         * @param array  $vars
         * @param array  $config
         * @return string
         * @throws \Exception
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/3/14 18:23
         */
        public function fetch($template = '', $vars = [], $config = [])
        {
            // 将数据融入到key中
            $this->_formatData();
            // 配置触发
            $this->_formatTrigger();
            // 编译按钮的html属性
            foreach ( $this->_buttonList as &$button ) {
                $button['attr'] = $this->compileHtmlAttr($button['attr']);
            }
            // 显示页面
            $this->assign('group', $this->_group);
            $this->assign('title', $this->_title);
            $this->assign('suggest', $this->_suggest);
            $this->assign('warning', $this->_warning);
            if ( count($this->_keyList) ) {
                $this->assign('keyList', $this->_keyList);
            }
            // 在有赋值的情况展示
            if ( count($this->_explaints) > 0 ) {
                $this->assign('explaints', $this->_explaints);
            }
            if ( count($this->_buttonList) ) {
                $this->assign('buttonList', $this->_buttonList);
            }
            $this->assign('savePostUrl', $this->_savePostUrl ?: Url::build());
            $this->assign('triggers', $this->_triggers);
            //            dump($this->_triggers);
            $this->assign('reload', $this->_reload);
            $this->assign('mask', $this->_mask);
            return parent::fetch('edit', $vars, $config);
        }

        /**
         * 将验证规范格式化为html形式
         * @param null $validate_rule
         * @return array|string|null
         * @throws Exception
         * @author  : 微尘 <yicmf@qq.com>
         * @datetime: 2019/3/14 12:35
         */
        private function _getValidaorRule($validate_rule = null)
        {
            if ( is_null($validate_rule) ) {
                return $validate_rule;
            } elseif ( is_array($validate_rule) ) {
                return $this->_compileHtmlAttr($validate_rule, 'data-');
            } elseif ( is_string($validate_rule) ) {
                return $this->_compileHtmlAttr(['data-rule' => $validate_rule]);
            } else {
                throw new Exception('ValidaorRule格式异常');
            }
        }

        /**
         * 规范触发数据格式.
         */
        private function _formatTrigger()
        {
            //TODO:考虑在没有配置当前字段默认值的情况，目前设计为不显示
            foreach ( $this->_triggers as $name => &$trigger ) {
                $field = explode(',', $trigger['field']);
                unset($trigger['field']);
                foreach ( $trigger as $key => $trigger_detail ) {
                    $jquery_show = '';
                    $shows = explode(',', $trigger_detail['show']);
                    foreach ( $shows as $show ) {
                        if ( strpos($show, '.') ) {
                            $show = md5(json_encode(explode('.', $show)));
                        } elseif ( strpos($show, '|') ) {
                            $show = md5(json_encode(explode('|', $show)));
                        } else {
                            $show = md5(is_array($show) ? implode(',', $show) : $show);
                        }
                        //md5(is_array($e['name'])?implode(',', $e['name']):$e['name']);
                        $jquery_show .= empty($jquery_show) ? ('#trigger_' . $show) : (',#trigger_' . $show);
                    }
                    $jquery_hide = '';
                    foreach ( array_diff($field, $shows) as $show ) {
                        if ( strpos($show, '.') ) {
                            $show = md5(json_encode(explode('.', $show)));
                        } elseif ( strpos($show, '|') ) {
                            $show = md5(json_encode(explode('|', $show)));
                        } else {
                            $show = md5(is_array($show) ? implode(',', $show) : $show);
                        }
                        $jquery_hide .= empty($jquery_hide) ? ('#trigger_' . $show) : (',#trigger_' . $show);
                    }
                    //                    dump($shows);
                    $trigger[$key]['value'] = json_encode(explode(',', $trigger_detail['value']));
                    $trigger[$key]['show'] = json_encode($shows);
                    $trigger[$key]['jquery_show'] = $jquery_show;
                    $trigger[$key]['jquery_hide'] = $jquery_hide;
                    foreach ( $this->_keyList as $data ) {
                        if ( $name == $data['name'] ) {
                            //当前值
                            if ( in_array($data['value'], explode(',', $trigger_detail['value'])) ) {
                                $trigger[$key]['is_show'] = 1;
                            } else {
                                $trigger[$key]['is_show'] = 0;
                            }
                        }
                    }
                }
            }
        }

        /**
         * 输入数据格式化.
         */
        private function _formatData()
        {
            if ( 'id' !== $this->_default_pk && is_object($this->_data) ) {
                $pk = $this->_data->getPk();
            } else {
                $pk = $this->_default_pk;
            }
            $flag = false;
            foreach ( $this->_keyList as &$e ) {
                if ( $pk == $e['name'] ) {
                    $flag = true;
                }
                $e['data'] = $this->_data;
                $e['jquery_id'] = md5(is_array($e['name']) ? implode(',', $e['name']) : $e['name']);

                if ( $e['type'] instanceof \Closure ) {
                    // 闭包
                    $e['value'] = $e['type']($this->_data);
                    $e['type'] = 'label';
                } elseif ( is_array($e['name']) ) {
                    $i = 0;
                    $n = count($e['name']);
                    while ( $n > 0 ) {
                        $e['value'][$i] = isset($this->_data[$e['name'][$i]]) ? $this->_data[$e['name'][$i]] : (isset($e['value'][$i]) ? $$e['value'][$i] : 0); //
                        $i++;
                        $n--;
                    }
                } elseif ( strpos($e['name'], '.') ) { // 支持点语法
                    $temp = explode('.', $e['name']);
                    $e['relation']['parent'] = $temp[0];
                    $e['relation']['child'] = $temp[1];
                    $e['name'] = $temp[0] . '[' . $temp[1] . ']';
                    $e['value'] = isset($this->_data[$temp[0]][$temp[1]]) ? $this->_data[$temp[0]][$temp[1]] : (isset($e['value']) ? $e['value'] : '');
                } elseif ( strpos($e['name'], '|') ) { // 使用‘|’代表同级字段
                    if ( isset($e['value']) ) {
                        $e['value'] = explode('|', $e['name']);
                    }
                    $e['name'] = explode('|', $e['name']);
                    foreach ( $e['name'] as $t_key => $t_value ) {
                        $e['value'][$t_key] = isset($this->_data[$t_value]) ? $this->_data[$t_value] : (isset($e[$t_key]) ? $e[$t_key] : '');
                    }
                } else {
                    $e['value'] = isset($this->_data[$e['name']]) ? $this->_data[$e['name']] : (isset($e['value']) ? $e['value'] : '');
                }
            }
            if ( !$flag && isset($this->_data[$pk]) ) {
                //自动增加隐藏表单用于编辑;
                $edit['name'] = $pk;
                $edit['type'] = 'hidden';
                $edit['value'] = $this->_data[$pk];
                $this->_keyList[] = $edit;
            }
        }

        private function readUserGroups($module)
        {
            return Db::name('AuthGroup')->where('status', 1)->where('module', $module)
                ->order('id ASC')
                ->column('id,title');
        }
    }
