<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Admin\Controller;

use Admin\Data\Common;
use Admin\Entity\Delivery;
use Admin\Entity\DeliveryIndividuation;
use Admin\Entity\Region;
use Admin\Form\DeliveryForm;
use Admin\Form\DeliveryIndividuationForm;
use Admin\Service\DeliveryIndividuationManager;
use Admin\Service\DeliveryManager;
use Doctrine\ORM\EntityManager;
use Goods\Entity\GoodsClass;
use Laminas\Mvc\I18n\Translator;
use Laminas\View\Model\JsonModel;

class DeliveryController extends AdminActionController
{
    private $translator;
    private $entityManager;
    private $deliveryManager;
    private $deliveryIndividuationManager;

    public function __construct(
        Translator      $translator,
        EntityManager   $entityManager,
        DeliveryManager $deliveryManager,
        DeliveryIndividuationManager $deliveryIndividuationManager
    )
    {
        $this->translator       = $translator;
        $this->entityManager    = $entityManager;
        $this->deliveryManager  = $deliveryManager;
        $this->deliveryIndividuationManager = $deliveryIndividuationManager;
    }

    /**
     * 配送列表
     * @return array|\Laminas\View\Model\ViewModel
     */
    public function indexAction()
    {
        $deliveryList = $this->entityManager->getRepository(Delivery::class)->findBy([], ['deliveryId' => 'ASC']);
        return ['deliveryList' => $deliveryList];
    }

    /**
     * 添加配送方式
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addAction()
    {
        $form = new DeliveryForm();
        $form->get('cashOnDelivery')->setValueOptions(Common::cashOnDeliveryState($this->translator));
        $form->get('deliveryNameCode')->setValueOptions($this->adminCommon()->deliveryNameCodeOptions());

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();

                $deliveryInfo = $this->deliveryManager->addDelivery($data);

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('配送方式 %s 添加成功!'), $data['deliveryName']), $this->translator->translate('配送设置'));

                return $this->redirect()->toRoute('delivery');
            }
        }

        return ['form' => $form];
    }

    /**
     * 编辑配送方式
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function editAction()
    {
        $deliveryId = (int) $this->params()->fromRoute('id', -1);
        $deliveryInfo = $this->entityManager->getRepository(Delivery::class)->findOneByDeliveryId($deliveryId);
        if($deliveryInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该配送方式不存在!'));
            return $this->redirect()->toRoute('delivery');
        }

        $form = new DeliveryForm();
        $form->get('cashOnDelivery')->setValueOptions(Common::cashOnDeliveryState($this->translator));
        $form->get('deliveryNameCode')->setValueOptions($this->adminCommon()->deliveryNameCodeOptions());

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();

                $this->deliveryManager->editDelivery($data, $deliveryInfo);

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('配送方式 %s 编辑成功!'), $data['deliveryName']), $this->translator->translate('配送设置'));

                return $this->redirect()->toRoute('delivery');
            }
        } else $form->setData($deliveryInfo->valuesArray());

        return [
            'form'          => $form,
            'delivery'      => $deliveryInfo
        ];
    }

    /**
     * 删除配送方式
     * @return \Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteAction()
    {
        if(!$this->adminCommon()->validatorCsrf()) return $this->redirect()->toRoute('delivery');

        $deliveryId = (int) $this->params()->fromRoute('id', -1);
        $deliveryInfo = $this->entityManager->getRepository(Delivery::class)->findOneByDeliveryId($deliveryId);
        if($deliveryInfo == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该配送方式不存在!'));
            return $this->redirect()->toRoute('delivery');
        }

        $this->deliveryManager->deleteDelivery($deliveryInfo);

        $this->adminCommon()->addOperLog(sprintf($this->translator->translate('配送方式 %s 删除成功!'), $deliveryInfo->getDeliveryName()), $this->translator->translate('配送设置'));

        return $this->adminCommon()->toReferer();
    }
}