<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Admin\Controller;

use Admin\Data\Common;
use Admin\Entity\PhoneAreaCode;
use Admin\Form\PhoneAreaCodeForm;
use Admin\Service\PhoneAreaCodeManager;
use Doctrine\ORM\EntityManager;
use Laminas\Mvc\I18n\Translator;

class PhoneAreaCodeController extends AdminActionController
{
    private $translator;
    private $entityManager;
    private $phoneAreaCodeManager;

    public function __construct(
        Translator      $translator,
        EntityManager   $entityManager,
        PhoneAreaCodeManager $phoneAreaCodeManager
    )
    {
        $this->translator       = $translator;
        $this->entityManager    = $entityManager;
        $this->phoneAreaCodeManager = $phoneAreaCodeManager;
    }

    /**
     * 国际电话区号列表
     * @return array|\Laminas\View\Model\ViewModel
     */
    public function indexAction()
    {
        $phoneAreaList = $this->entityManager->getRepository(PhoneAreaCode::class)->findBy([], ['areaSort' => 'ASC']);

        return ['phoneAreaList' => $phoneAreaList];
    }

    /**
     * 添加国际电话区号
     * @return PhoneAreaCodeForm[]|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addAction()
    {
        $form = new PhoneAreaCodeForm();
        $phoneAreaArray = $this->adminCommon()->phoneAreaArray();
        $form->get('areaCode')->setValueOptions($phoneAreaArray);

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $data['areaName']       = $phoneAreaArray[$data['areaCode']];
                $data['areaPhoneCode']  = Common::areaPhoneCode($data['areaCode']);
                $this->phoneAreaCodeManager->addPhoneArea($data);

                $this->createConfig()->createPhoneArea();

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('地区 %s 的区号添加成功!'), $data['areaName']), $this->translator->translate('国际电话区号'));

                return $this->redirect()->toRoute('phone-area');
            }
        }

        return ['form' => $form];
    }

    /**
     * 编辑国际电话区号
     * @return array|\Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function editAction()
    {
        $areaId = (int) $this->params()->fromRoute('id', 0);
        $phoneArea = $this->entityManager->getRepository(PhoneAreaCode::class)->findOneBy(['areaId' => $areaId]);
        if($phoneArea == null) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该区号不存在!'));
            return $this->redirect()->toRoute('phone-area');
        }

        $form = new PhoneAreaCodeForm();
        $phoneAreaArray = $this->adminCommon()->phoneAreaArray();
        $form->get('areaCode')->setValueOptions($phoneAreaArray);

        if($this->getRequest()->isPost()) {
            $data = $this->params()->fromPost();
            $form->setData($data);
            if ($form->isValid()) {
                $data = $form->getData();
                $data['areaName'] = $phoneAreaArray[$data['areaCode']];
                $data['areaPhoneCode']  = Common::areaPhoneCode($data['areaCode']);
                $this->phoneAreaCodeManager->editPhoneArea($data, $phoneArea);

                $this->createConfig()->createPhoneArea();

                $this->adminCommon()->addOperLog(sprintf($this->translator->translate('地区 %s 的区号编辑成功!'), $data['areaName']), $this->translator->translate('国际电话区号'));

                return $this->redirect()->toRoute('phone-area');
            }
        } else $form->setData($phoneArea->valuesArray());

        return ['form' => $form, 'phoneArea' => $phoneArea];
    }

    /**
     * 删除国际电话区号
     * @return \Laminas\Http\Response
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function deleteAction()
    {
        if(!$this->adminCommon()->validatorCsrf()) return $this->adminCommon()->toReferer();

        $areaId = (int) $this->params()->fromRoute('id', 0);
        $phoneArea = $this->entityManager->getRepository(PhoneAreaCode::class)->findOneBy(['areaId' => $areaId]);
        if($phoneArea == null || $areaId == 1) {
            $this->flashMessenger()->addWarningMessage($this->translator->translate('该区号不存在!'));
            return $this->redirect()->toRoute('phone-area');
        }

        $this->phoneAreaCodeManager->deletePhoneArea($phoneArea);

        $this->createConfig()->createPhoneArea();

        $this->adminCommon()->addOperLog(sprintf($this->translator->translate('地区 %s 的区号成功!'), $phoneArea->getAreaName()), $this->translator->translate('国际电话区号'));

        return $this->adminCommon()->toReferer();
    }
}