<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Admin\Plugin;

use Admin\Data\Common;
use Admin\Data\Config;
use Admin\Entity\AdminUserGroup;
use Admin\Entity\Delivery;
use Admin\Entity\Region;
use Admin\Service\OperLogManager;
use Doctrine\ORM\EntityManager;
use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use DoctrineORMModule\Paginator\Adapter\DoctrinePaginator;
use Giggsey\Locale\Locale;
use Laminas\Config\Factory;
use Laminas\Crypt\BlockCipher;
use Laminas\Http\Client;
use Laminas\Json\Json;
use Laminas\Mvc\Controller\Plugin\AbstractPlugin;
use Laminas\Mvc\I18n\Translator;
use Laminas\Session\Container;

class AdminCommonPlugin extends AbstractPlugin
{
    private $entityManager;
    private $translator;
    private $operlogManager;
    private $sessionAdmin;

    public function __construct(
        EntityManager   $entityManager,
        Translator      $translator,
        OperLogManager  $operlogManager
    )
    {
        $this->entityManager    = $entityManager;
        $this->translator       = $translator;
        $this->operlogManager   = $operlogManager;

        if($this->sessionAdmin == null) {
            $this->sessionAdmin     = new Container('dbshopAdmin');
        }
    }

    /**
     * 管理员信息
     * @param $adminField
     * @return mixed
     */
    public function admin($adminField)
    {
        return $this->sessionAdmin->offsetGet($adminField);
    }

    /**
     * 获取时区数组
     * @return mixed
     */
    public function timezoneArray()
    {
        $localLanguage = $this->translator->getTranslator()->getLocale();
        if(!file_exists('data/moduleData/System/TimezoneData/' . $localLanguage . '.php')) $localLanguage = 'zh_CN';

        return include 'data/moduleData/System/TimezoneData/' . $localLanguage . '.php';
    }

    /**
     * 获取货币数组
     * @return mixed
     */
    public function currencyArray()
    {
        $localLanguage = $this->translator->getTranslator()->getLocale();
        if(!file_exists('data/moduleData/System/CurrencyData/' . $localLanguage . '.php')) $localLanguage = 'zh_CN';
        $array = include 'data/moduleData/System/CurrencyData/' . $localLanguage . '.php';

        $currencyArray = [];
        foreach ($array['currency'] as $value) {
            $currencyArray[$value['type']] = $value['displayName'];
        }

        return $currencyArray;
    }

    /**
     * 支付列表数组，$onlinePayment 是否是在线支付，true 是，false 不是
     * @param string $onlinePayment
     * @param false $open 是否只显示已启用的在线支付
     * @return array
     */
    public function paymentList($onlinePayment = 'true', $open = false)
    {
        $paymentList = Common::paymentList($open, $onlinePayment);
        $paymentArray= [];
        foreach ($paymentList as $payValue) {
            $paymentArray[$payValue['editAction']] = $payValue['paymentName']['content'];
        }

        return $paymentArray;
    }

    /**
     * 国际电话区域数组，有地区名称的
     * @return string[]
     */
    public function phoneAreaArray()
    {
        return Locale::getAllCountriesForLocale('zh');
    }

    /**
     * 验证删除的CSRF Token
     * @return bool
     */
    public function validatorCsrf()
    {
        $csrfValue = $this->getController()->getRequest()->getQuery('qToken');
        if(empty($csrfValue) || ($csrfValue && $csrfValue != $this->sessionAdmin->offsetGet('queryToken'))) {
            $this->getController()->flashMessenger()->addErrorMessage($this->translator->translate('不正确的请求!'));
            return false;
        }
        $this->sessionAdmin->offsetSet('queryToken', '');

        return true;
    }

    /**
     * 后台公共分页方法
     * @param Query $query
     * @param int $pageNumber
     * @param int $itemCountPerPage
     * @param bool $fetchJoinCollection
     * @return \Laminas\Paginator\Paginator
     */
    public function shopPaginator(Query $query, int $pageNumber, $itemCountPerPage = 16, $fetchJoinCollection = false)
    {
        $adapter    = new DoctrinePaginator(new Paginator($query, $fetchJoinCollection));
        $paginator  = new \Laminas\Paginator\Paginator($adapter);
        $paginator->setItemCountPerPage($itemCountPerPage);
        $paginator->setCurrentPageNumber($pageNumber);

        return $paginator;
    }

    /**
     * 返回上一页
     * @return mixed
     */
    public function toReferer()
    {
        $referer = $this->getController()->params()->fromHeader('Referer');
        if($referer) {
            $refererUrl     = $referer->uri()->getPath().(!empty($referer->uri()->getQuery()) ? '?'.$referer->uri()->getQuery() : '').(!empty($referer->uri()->getFragment()) ? '#'.$referer->uri()->getFragment() : '');
            $refererHost    = $referer->uri()->getHost();
            $host           = $this->getController()->getRequest()->getUri()->getHost();
            if ($refererUrl && $refererHost == $host) {
                return $this->getController()->redirect()->toUrl($refererUrl);
            }
        }
        return $this->getController()->redirect()->toRoute('home');
    }

    /**
     * 添加操作日志 同时输出 flashMessenger
     * @param $logBody
     * @param $operClassName
     * @param bool $message
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    public function addOperLog($logBody, $operClassName, $message = true)
    {
        $this->operlogManager->addOperLog(
            [
                'logOperUser'       => $this->sessionAdmin->admin_name,
                'logOperUserGroup'  => $this->sessionAdmin->admin_group_name,
                'logTime'           => time(),
                'logIp'             => $this->getController()->getRequest()->getServer('REMOTE_ADDR'),
                'logBody'           => '['.$operClassName.'] '.$logBody
            ]
        );

        if($message) $this->getController()->flashMessenger()->addSuccessMessage($logBody);
    }

    /**
     * 删除图片
     * @param $image
     * @return bool
     */
    public function deleteImage($image)
    {
        if(empty($image)) return false;
        $imageArray = is_array($image) ? $image : [$image];

        foreach ($imageArray as $imageValue) {
            if (stripos($imageValue, 'http') !== false) continue;

            $imagePath = dirname($imageValue);
            $imagePath = empty($imagePath) ? '' : str_replace('.', '', $imagePath) . '/';
            $imageName = basename($imageValue);
            $imageFile = getcwd() . '/public/' . $imagePath . $imageName;
            if(file_exists($imageFile)) unlink($imageFile);
        }
    }

    /**
     * 获取地区下级
     * @param int $topId
     * @return array|object[]|null
     */
    public function getRegionSub($topId = 0)
    {
        $region = $this->entityManager->getRepository(Region::class)->findBy(['regionTopId' => $topId], ['regionSort' => 'ASC']);
        return $region ? $region : null;
    }

    /**
     * 快递信息数组
     * @param string $topName
     * @return string[]
     */
    public function deliveryNameOptions($topName = '')
    {
        $deliveryList       = $this->entityManager->getRepository(Delivery::class)->findBy(['deliveryState' => 1], ['deliverySort' => 'ASC']);
        $deliveryNameArray  = [0 => empty($topName) ? $this->translator->translate('请选择') : $topName];
        if($deliveryList) foreach ($deliveryList as $deValue) {
            $deliveryNameArray[$deValue->getDeliveryId()] = $deValue->getDeliveryName();
        }
        return $deliveryNameArray;
    }
    /**
     * 获取动态绑定列表
     * @param string $topName
     * @return array
     */
    public function deliveryNameCodeOptions($topName = '')
    {
        $deliveryNameCodeArray   = [0 => empty($topName) ? $this->translator->translate('请选择') : $topName];
        $deliveryConfig = Common::readConfigFile('delivery');
        if (isset($deliveryConfig['deliveryApiState']) && $deliveryConfig['deliveryApiState'] == 1 && isset($deliveryConfig['deliveryApiType'])) {
            $companyArray = Common::deliveryApiCompanyCode($deliveryConfig['deliveryApiType']);
            if (!empty($companyArray)) $deliveryNameCodeArray = array_merge($deliveryNameCodeArray, $companyArray);
        }

        return $deliveryNameCodeArray;
    }

    /**
     * 获取管理员组列表
     * @param string $topName
     * @return array
     */
    public function adminGroupOptions($topName = '')
    {
        $groupList  = [0 => empty($topName) ? $this->translator->translate('选择管理组') : $topName];
        $group      = $this->entityManager->getRepository(AdminUserGroup::class)->findBy([], ['adminGroupId' => 'ASC']);
        if($group) {
            foreach ($group as $value) {
                $groupList[$value->getAdminGroupId()] = $value->getAdminGroupName();
            }
        }
        return $groupList;
    }

    /**
     * 推送服务
     * @param $action
     * @param array $data
     * @param string $path
     * @return mixed|null
     */
    public function dbshopPushService($action, $data = [], $path = '')
    {
        $serviceConfig = Common::readConfigFile('dbshopService');
        if (!$serviceConfig['key'] || !$serviceConfig['code'] || !$serviceConfig['userName']) return null;

        $blockCipher = BlockCipher::factory('openssl', ['algo' => 'aes', 'mode' => 'gcm']);
        $blockCipher->setKey($serviceConfig['key']);
        $dataStr = $blockCipher->encrypt(Json::encode(array_merge($data,
            [
                'code'          => $serviceConfig['code'],
                'scheme'        => $this->getController()->getRequest()->getUri()->getScheme(),
                'host'          => $this->getController()->getRequest()->getUri()->getHost(),
                'baseUrl'       => $this->getController()->getRequest()->getBaseUrl(),
                'port'          => $this->getController()->getRequest()->getUri()->getPort(),
                'versionNumber' => DBSHOP_VERSION_NUMBER
            ]
        )));

        $client = new Client(Config::SERVICE_URL . $path, [
            'adapter' => 'Laminas\Http\Client\Adapter\Curl',
            'curloptions' => [
                CURLOPT_SSL_VERIFYPEER => false
            ]
        ]);
        $client->setHeaders(['Accept' => 'application/json']);
        $client->setMethod('POST');
        $client->setParameterPost([
            'userName'      => $serviceConfig['userName'],
            'systemType'    => 'dbshop',
            'action'        => $action,
            'dataStr'       => $dataStr
        ]);

        try {
            $response = $client->send();
            if ($response->isSuccess()) {
                return Json::decode($response->getBody(), Json::TYPE_ARRAY);
            }
        } catch (\Exception $e) {
            return null;
        }

        return null;
    }

    /**
     * 服务调用
     * @param $action
     * @param array $data
     * @param string $path
     * @return mixed|null
     */
    public function dbshopApiService($action, $data = [], $path = '')
    {
        $serviceConfig = Common::readConfigFile('dbshopService');
        if (empty($path)) {
            if (!$serviceConfig['key'] || !$serviceConfig['code'] || !$serviceConfig['userName']) return null;

            $blockCipher = BlockCipher::factory('openssl', ['algo' => 'aes', 'mode' => 'gcm']);
            $blockCipher->setKey($serviceConfig['key']);
            $dataStr = $blockCipher->encrypt(Json::encode(array_merge($data,
                [
                    'code'          => $serviceConfig['code'],
                    'scheme'        => $this->getController()->getRequest()->getUri()->getScheme(),
                    'host'          => $this->getController()->getRequest()->getUri()->getHost(),
                    'baseUrl'       => $this->getController()->getRequest()->getBaseUrl(),
                    'port'          => $this->getController()->getRequest()->getUri()->getPort(),
                    'versionNumber' => DBSHOP_VERSION_NUMBER
                ]
            )));
        } else {//单独调用不需要走加密接口
            $dataStr = serialize(array_merge($data,
            [
                'scheme'        => $this->getController()->getRequest()->getUri()->getScheme(),
                'host'          => $this->getController()->getRequest()->getUri()->getHost(),
                'baseUrl'       => $this->getController()->getRequest()->getBaseUrl(),
                'port'          => $this->getController()->getRequest()->getUri()->getPort(),
                'versionNumber' => DBSHOP_VERSION_NUMBER
            ]));
        }

        $client = new Client(Config::SERVICE_API_URL . $path, [
            'adapter' => 'Laminas\Http\Client\Adapter\Curl',
            'curloptions' => [
                CURLOPT_SSL_VERIFYPEER => false
            ]
        ]);
        $client->setHeaders(['Accept' => 'application/json']);
        $client->setMethod('POST');
        $client->setParameterPost([
            'userName'      => $serviceConfig['userName'],
            'systemType'    => 'dbshop',
            'action'        => $action,
            'dataStr'       => $dataStr
        ]);

        try {
            $response = $client->send();
            if ($response->isSuccess()) {
                return Json::decode($response->getBody(), Json::TYPE_ARRAY);
            }
        } catch (\Exception $e) {
            return null;
        }

        return null;
    }

    /**
     * 下载服务中的包文件
     * @param $action
     * @param $data
     * @param $localPackage
     * @return bool
     */
    public function dbshopDownloadPackage($action, $data, $localPackage)
    {
        $serviceConfig  = Common::readConfigFile('dbshopService');

        $updateKey = '';
        $clientGet = new Client(Config::SERVICE_API_URL . '/getUpdateKey', ['adapter' => 'Laminas\Http\Client\Adapter\Curl', 'curloptions' => [CURLOPT_SSL_VERIFYPEER => false]]);
        $clientGet->setMethod('POST');
        $clientGet->setParameterPost(['userName' => $serviceConfig['userName'], 'actionName' => $action, 'systemType' => 'dbshop', 'hostUrl' => $this->getController()->getRequest()->getUri()->getHost()]);
        $responseGet = $clientGet->send();
        if ($responseGet->isSuccess()) $updateKey = $responseGet->getBody();

        $blockCipher    = BlockCipher::factory('openssl', ['algo' => 'aes', 'mode' => 'gcm']);
        $blockCipher->setKey($serviceConfig['key']);
        $dataStr = $blockCipher->encrypt(Json::encode(array_merge($data,
            [
                'code'          => $serviceConfig['code'],
                'scheme'        => $this->getController()->getRequest()->getUri()->getScheme(),
                'host'          => $this->getController()->getRequest()->getUri()->getHost(),
                'baseUrl'       => $this->getController()->getRequest()->getBaseUrl(),
                'port'          => $this->getController()->getRequest()->getUri()->getPort(),
                'updateKey'     => $updateKey,
                'versionNumber' => DBSHOP_VERSION_NUMBER
            ]
        )));

        $updateKeyIni = Factory::fromFile(Config::PACKAGE_UPDATE_KEY_FILE);
        $updateKeyIni[$action] = $updateKey;
        Factory::toFile(Config::PACKAGE_UPDATE_KEY_FILE, $updateKeyIni);

        $client = new Client(Config::SERVICE_API_URL, [
            'adapter' => 'Laminas\Http\Client\Adapter\Curl',
            'curloptions' => [
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_HEADER => 0
            ],
            'timeout' => 300
        ]);
        $client->setMethod('POST');
        $client->setParameterPost([
            'userName'      => $serviceConfig['userName'],
            'systemType'    => 'dbshop',
            'action'        => $action,
            'dataStr'       => $dataStr
        ]);
        $response = $client->send();
        if ($response->isSuccess()) {
            if ($response->getBody() == 'updateKeyError') return 'keyError';
            else {
                $signedUrl = $response->getBody();
                if (!empty($signedUrl)) {
                    $downClient     = new Client($signedUrl, ['adapter' => 'Laminas\Http\Client\Adapter\Curl', 'curloptions' => [CURLOPT_SSL_VERIFYPEER => false, CURLOPT_HEADER => 0], 'timeout' => 300]);
                    $downClient->setMethod('GET');
                    $downResponse   = $downClient->send();
                    if ($downResponse->isSuccess()) {
                        file_put_contents($localPackage, $downResponse->getBody());
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * 平铺无限分类-暂时未使用
     * @param $items
     * @param string $id
     * @param string $pid
     * @param string $name
     * @param string $path
     * @param string $son
     * @return array
     */
    public function genTree(
        $items,
        $id='getGoodsCategoryId',
        $pid='getGoodsCategoryTopId',
        $name='getGoodsCategoryName',
        $path='getGoodsCategoryPath',
        $son = 'children'
    )
    {
        $tree = []; //格式化的树
        $tmpMap = [];  //临时扁平数据

        foreach ($items as $item) {
            $tmpMap[$item->$id()] = [
                'id'    => $item->$id(),
                'top_id'=> $item->$pid(),
                'name'  => $item->$name(),
                'path'  => $item->$path()
            ];
        }

        foreach ($items as $item) {
            if (isset($tmpMap[$item->$pid()])) {
                $tmpMap[$item->$pid()][$son][] = &$tmpMap[$item->$id()];
                //$tmpMap[$item->$pid()][$son][] = $tmpMap[$item->$id()];
            } else {
                $tree[] = &$tmpMap[$item->$id()];
                //$tree[] = $tmpMap[$item->$id()];
            }
        }
        unset($tmpMap);
        return $tree;
    }
}