<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Admin\Service;

use Admin\Controller\EditorController;
use Admin\Controller\HomeController;
use Admin\Entity\AdminUser;
use Laminas\Authentication\AuthenticationService;
use Laminas\Authentication\Result;
use Laminas\Session\Container;

class AuthManager
{
    const ADMIN_ALREADY_LOGIN    = 1;     //已经登录
    const ADMIN_NOT_LOGIN        = -1;    //未登录
    const ADMIN_NOT_PERMISSION   = -2;    //无权限

    private $authService;
    private $sessionManager;

    private $config;

    public function __construct(
        AuthenticationService $authService,
        $sessionManager,
        $config
    )
    {
        $this->authService      = $authService;
        $this->sessionManager   = $sessionManager;
        $this->config           = $config;
    }

    /**
     * 登录
     * @param $name
     * @param $password
     * @return Result
     */
    public function login($name, $password)
    {
        //不为空，说明已经登录
        if($this->authService->getIdentity() != null) return new Result(Result::SUCCESS, $this->authService->getIdentity(), ['Already logged in']);

        $authAdapter = $this->authService->getAdapter();
        $authAdapter->setName($name);
        $authAdapter->setPassword($password);
        $authAdapter->setAdminState(1);

        $result = $this->authService->authenticate();
        if($result->getCode() == Result::SUCCESS) {
            $userInfo   = $authAdapter->getUser();
            $userGroup  = $userInfo->getGroup();
            $session    = new Container('dbshopAdmin', $this->sessionManager);

            $session->offsetSet('admin_id', $userInfo->getAdminId());
            $session->offsetSet('admin_name', $userInfo->getAdminName());
            $session->offsetSet('admin_group_id', $userGroup->getAdminGroupId());
            $session->offsetSet('admin_group_name', $userGroup->getAdminGroupName());
            $session->offsetSet('session_id', md5($this->sessionManager->getId()));
        }

        return $result;
    }

    /**
     * 退出操作
     * @return false
     */
    public function logout()
    {
        if ($this->authService->getIdentity() == null) {
            return false;
        }

        $session = new Container('dbshopAdmin', $this->sessionManager);
        $session->getManager()->getStorage()->clear('dbshopAdmin');

        $this->authService->clearIdentity();
    }

    /**
     * 检查是否已经登录
     * @return bool
     */
    public function checkLogin()
    {
        return $this->authService->getIdentity();
    }

    /**
     * 权限验证
     * @param $controllerName
     * @param $actionName
     * @return int
     */
    public function filterAccess($controllerName, $actionName)
    {
        //未登录
        if(!$this->authService->hasIdentity()) {
            return self::ADMIN_NOT_LOGIN;
        }

        $adminInfo  = $this->authService->getAdapter()->checkAdmin($this->authService->getIdentity());
        $session    = new Container('dbshopAdmin', $this->sessionManager);
        //判断该管理员是否存在，如果不存在则进行session清理
        if(!$adminInfo) {
            $session->getManager()->getStorage()->clear('dbshopAdmin');
            $this->authService->clearIdentity();
            $this->sessionManager->destroy();

            return self::ADMIN_NOT_LOGIN;
        }

        //判断是否有权限(不检查管理员组)
        $adminGroupPurview = $adminInfo->getAdminGroupId() != 1 ? explode(',', $adminInfo->getGroup()->getAdminGroupPurview()) : [];
        if(
            !in_array($controllerName, [HomeController::class, EditorController::class])
            && $adminInfo->getAdminGroupId() != 1
            && !empty($adminGroupPurview)
            && !in_array(str_replace('\\', '_', $controllerName).'_'.$actionName, $adminGroupPurview)
        ) {
            if (
                stripos($actionName, 'ajax') === false
                && !in_array($actionName, [//不需要授权即可访问的
                    'uploadGoodsImage'
                ])
            ) return self::ADMIN_NOT_PERMISSION;
        }

        //判断管理员组是否已修改
        if ($adminInfo->getAdminGroupId() != $session->offsetGet('admin_group_id')) {
            $session->offsetSet('admin_group_id', $adminInfo->getAdminGroupId());
            $session->offsetSet('admin_group_name', $adminInfo->getGroup()->getAdminGroupName());
        }

        return self::ADMIN_ALREADY_LOGIN;
    }
}