<?php
/**
 * DBShop 电子商务系统
 *
 * ==========================================================================
 * @link      https://www.dbshop.net/
 * @copyright 北京珑大钜商科技有限公司，并保留所有权利。
 * @license   https://www.dbshop.net/license.html License
 * ==========================================================================
 *
 * @author    静静的风 <baron@loongdom.cn>
 *
 */

namespace Admin\Service\Payment;

use Admin\Data\Common;
use Laminas\Config\Writer\PhpArray;
use Yansongda\Pay\Exceptions\Exception;
use Yansongda\Pay\Pay;

class F2fPayService
{
    private $config         = [];
    private $alipayConfig   = [];

    public function __construct()
    {
        if(!$this->config) {
            $this->config = Common::getPaymentConfig('F2fPay');
        }

        $this->alipayConfig = [
            'app_id'        => $this->config['paymentAppId']['content'],
            'ali_public_key'=> $this->config['paymentPublicKey']['content'],
            'private_key'   => $this->config['paymentPrivateKey']['content'],
            'http' => [
                'timeout' => 10.0,
                'connect_timeout' => 8.0,
            ]
        ];
    }


    public function savePaymentConfig(array $data, $paymentConfig, $configFile)
    {
        $configArray = PaymentForm::setFormValue($paymentConfig, $data);

        $write = new PhpArray();
        $write->setUseBracketArraySyntax(true);
        $write->toFile($configFile, $configArray);

        Common::opcacheInvalidate($configFile);

        return $configArray;
    }

    public function getFormInput($paymentConfig)
    {
        return PaymentForm::createFormInput($paymentConfig);
    }

    /**
     * 订单支付
     * @param array $data
     * @return array
     */
    public function orderPaymentTo(array $data)
    {
        $orderInfo = $data['orderInfo'];
        $goodsArray= $orderInfo->getOrderGoods();

        $order = [
            'out_trade_no'  => $orderInfo->getOrderSn(),
            'total_amount'  => $orderInfo->getOrderAmount(),
            'subject'       => (count($goodsArray) == 1 ? $goodsArray[0]->getGoodsName() : '多商品合并购买'),
        ];

        try {
            $alipay = Pay::alipay(array_merge($this->alipayConfig, ['notify_url' => $data['notifyUrl']]));
            $result = $alipay->scan($order);

            $result->offsetSet('code_url', $result->offsetGet('qr_code'));
            $result->offsetSet('click_url_pay', true);
            $result->offsetSet('pay_name', '支付宝扫码支付');
            return [
                'type'  => 'scan',
                'result'=> $result
            ];
        } catch (Exception $e) {
            exit($e->getMessage());
        }
    }

    /**
     * 发货处理
     * @param array $data
     */
    public function orderToSend(array $data)
    {

    }

    /**
     * 支付返回验证
     * @param $orderInfo
     * @return array
     */
    public function orderPaymentReturn($orderInfo)
    {
        return ['paymentStatus' => ($orderInfo->getOrderStatus() >= Common::ORDER_WAIT_GOODS_ARRIVE), 'orderAmount' => $orderInfo->getOrderAmount()];
    }


    /**
     * 支付异步通知
     * @param $orderInfo
     * @return array
     */
    public function orderPaymentNotify($orderInfo)
    {
        $alipay = Pay::alipay($this->alipayConfig);
        try {
            $result = $alipay->verify();
            if(
                in_array($result->offsetGet('trade_status'), ['TRADE_SUCCESS', 'TRADE_FINISHED'])
                && $result->offsetGet('out_trade_no') == $orderInfo->getOrderSn()
                && $result->offsetGet('total_amount') == $orderInfo->getOrderAmount()
            )
            {
                //查看是否已经支付完毕
                if($orderInfo->getOrderStatus() >= Common::orderStatusCode('WAIT_GOODS_ARRIVE')) return ['finishPay' => true, 'orderAmount' => $result->offsetGet('total_amount'), 'sendMessage' => 'success'];

                return ['paymentStatus' => true, 'outTradeNo' => $result->offsetGet('out_trade_no'), 'orderAmount' => $result->offsetGet('total_amount'), 'sendMessage' => 'success'];
            }
        } catch (\Exception $e) {
            //$e->getMessage();
        }
        //return $alipay->success()->send();
        return ['paymentStatus' => false, 'sendMessage' => ''];
    }

    /**
     * 支付
     * @param array $data
     * @return array
     */
    public function paymentTo(array $data)
    {
        $info = [
            'out_trade_no'  => $data['sn'],
            'total_amount'  => $data['amount'],
            'subject'       => $data['subject']
        ];

        try {
            $alipay = Pay::alipay(array_merge($this->alipayConfig, ['notify_url' => $data['notifyUrl']]));
            $result = $alipay->scan($info);

            $result->offsetSet('code_url', $result->offsetGet('qr_code'));
            $result->offsetSet('click_url_pay', true);
            $result->offsetSet('pay_name', '支付宝扫码支付');
            return [
                'type'  => 'scan',
                'result'=> $result
            ];
        } catch (Exception $e) {
            exit($e->getMessage());
        }
    }

    /**
     * 支付返回
     * @param array $data
     * @return array
     */
    public function paymentReturn(array $data)
    {
        return ['paymentStatus' => ($data['status'] >= 20), 'amount' => $data['amount']];
    }

    /**
     * 支付异步
     * @param array $data
     * @return array
     */
    public function paymentNotify(array $data)
    {
        $alipay = Pay::alipay($this->alipayConfig);
        try {
            $result = $alipay->verify();
            if(
                in_array($result->offsetGet('trade_status'), ['TRADE_SUCCESS', 'TRADE_FINISHED'])
                && $result->offsetGet('out_trade_no') == $data['sn']
                && $result->offsetGet('total_amount') == $data['amount']
            )
            {
                //查看是否已经支付完毕
                if($data['status'] >= 20) return ['finishPay' => true, 'amount' => $result->offsetGet('total_amount'), 'sendMessage' => 'success'];

                return ['paymentStatus' => true, 'amount' => $result->offsetGet('total_amount'), 'sendMessage' => 'success'];
            }
        } catch (\Exception $e) {
            //$e->getMessage();
        }
        return ['paymentStatus' => false, 'sendMessage' => ''];
    }
}